-- TODO 'sticky note' mode (nice graphics with alpha?)
-- TODO chat print -> post text in any channel, define a delay

local CLEANUI_HOOKED = nil;

local waitTable = {};
local waitFrame = nil;

local NOTE_TEMPLATE = {title=NOTES_TEMPLATE_TITLE, category=nil, text="", type=1, received=nil};

local selectedIndex = nil;
local lastCursorPos = nil;

local NOTE_TYPE = {
	NOTES_TYPE_NOTODO,
	NOTES_TYPE_TODODONE,
	NOTES_TYPE_TODONOTDONE,
}

local NOTE_QUEST_INFO = {
	NOTES_CATEGORY_QUESTLINK,
	NOTES_CATEGORY_QUESTNAME,
	QUEST_OBJECTIVES,
	QUEST_REWARDS,
	PET.." ".. INFO,
}

local selectedAddIndex = 1;

-- saved variables
NotesData = {};
NotesData.notes = {};

local ldb = LibStub:GetLibrary("LibDataBroker-1.1");

local dataobj = ldb:NewDataObject("Notes", {
	type = "launcher",
	icon = "Interface\\Addons\\Notes\\images\\frame-icon-dataobject",
	OnClick = function(self, button)
		if button == "LeftButton" then
			NotesFrame_Toggle();
		end
	end,
	OnTooltipShow = function(tooltip)
		tooltip:AddLine(BINDING_NAME_NOTESTOGGLE);
	end
});

StaticPopupDialogs["NOTES_EDIT_TITLE_POPUP"] = {
    text = NOTES_CHANGETITLE,
    button1 = NOTES_OK,
    button2 = NOTES_CANCEL,
    OnShow = function (self, data)
        self.editBox:SetText(data or "nil")
    end,
    OnAccept = function (self, data, data2)
        local text = self.editBox:GetText()
        Notes_ChangeTitle(text)
    end,
    hasEditBox = true,
    timeout = 0,
    whileDead = true,
    hideOnEscape = true,
}

StaticPopupDialogs["NOTES_EDIT_CATEGORY_POPUP"] = {
    text = NOTES_CHANGECATEGORY,
    button1 = NOTES_OK,
    button2 = NOTES_CANCEL,
    OnShow = function (self, data)
        self.editBox:SetText(data or "nil")
    end,
    OnAccept = function (self, data, data2)
        local text = self.editBox:GetText()
        Notes_ChangeCategory(text)
    end,
    hasEditBox = true,
    timeout = 0,
    whileDead = true,
    hideOnEscape = true,
}

StaticPopupDialogs["NOTES_SEND_NOTE"] = {
    text = MAIL_TO_LABEL,
    button1 = NOTES_OK,
    button2 = NOTES_CANCEL,
    OnShow = function (self, data)
        self.editBox:SetText(data or "nil")
    end,
    OnAccept = function (self, data, data2)
        local name = self.editBox:GetText()
        Notes_SendNoteToPlayer("WHISPER", name)
    end,
    hasEditBox = true,
    timeout = 0,
    whileDead = true,
    hideOnEscape = true,
}

-- functions
function Notes_OnLoad(self)
	self:RegisterEvent("PLAYER_ENTERING_WORLD");
	self:RegisterEvent("ADDON_LOADED");
	
	self:RegisterEvent("CHAT_MSG_ADDON");
	
	TextBodyEditBox:RegisterForDrag("LeftButton");
	
    -- register for addon messages
    RegisterAddonMessagePrefix("NI_SendNote");
    RegisterAddonMessagePrefix("NI_SendNote+");
    
	tinsert(UISpecialFrames, self:GetName());
	
	Notes_InitSlashCommandHandler();
end

function Notes_InitSlashCommandHandler()

	SLASH_NOTES1 = "/note";
  	SLASH_NOTES2 = "/notes";
  	SlashCmdList["NOTES"] = function(msg)
		Notes_SlashCommandHandler(msg);
	end
end

local debug = false;
function notes_debug(value, name)
    if (not debug) then
        return;
    end

    if (name) then
 	    DEFAULT_CHAT_FRAME:AddMessage("debug "..name..": "..(value or "nil"), 1.0, 1.0, 1.0, 1, 10);
    else
 	    DEFAULT_CHAT_FRAME:AddMessage("debug: "..(value or "nil"), 1.0, 1.0, 1.0, 1, 10);
    end
end

function Notes_SlashCommandHandler(msg)	
	if (msg == "reload") then
	 	ReloadUI();
	else
	    NotesFrame_Toggle();
	end
end

function Notes_StartDelay(delay, func, ...)
    if(type(delay)~="number" or type(func)~="function") then
        return false;
    end
    
    if(waitFrame == nil) then
        waitFrame = CreateFrame("Frame","Notes_WaitFrame", UIParent);
        waitFrame:SetScript("onUpdate",function (self,elapse)
            local count = #waitTable;
            local i = 1;
            
            while(i<=count) do
                local waitRecord = tremove(waitTable,i);
                local d = tremove(waitRecord,1);
                local f = tremove(waitRecord,1);
                local p = tremove(waitRecord,1);
                
                if(d>elapse) then
                    tinsert(waitTable,i,{d-elapse,f,p});
                    i = i + 1;
                else
                    count = count - 1;
                    f(unpack(p));
                end
            end
        end);
    end
    
    tinsert(waitTable,{delay,func,{...}});
    
    return true;
end

function Notes_OnEvent(self, event, ...)
	if (event == "ADDON_LOADED") then
		local addonName = ...;
		if (addonName == "Notes") then
		    Notes_StartDelay(1, NotesFrame_Init);
        end
    end
    if (event == "CHAT_MSG_ADDON") then
        local prefix, message, type, sender = ...;
        Notes_ReceiveNote(prefix, message, type, sender);
    end
	
    if (event == "PLAYER_ENTERING_WORLD") then
		Notes_StartDelay(3, Notes_HookCleanUI);
    end  
end

function NotesFrame_Init()
    NotesData.notes = NotesData.notes or {};
    
    UIDropDownMenu_Initialize(Notes_AddInfoDropDown, Note_AddInfoDropDown_Initialize);
	UIDropDownMenu_SetSelectedValue(Notes_AddInfoDropDown, selectedAddIndex);
	
	-- save mail button
	local saveMailButton = _G["Notes_SaveMailButton"];
    
    if (not saveMailButton) then
        saveMailButton = CreateFrame("Button", "Notes_SaveMailButton", OpenMailFrame, "UIPanelButtonTemplate");
        
        saveMailButton:SetText("Notes+");
        
        saveMailButton:RegisterForClicks("AnyUp");

        saveMailButton:SetSize(78, 22);
        saveMailButton:ClearAllPoints();
        saveMailButton:SetPoint("RIGHT", OpenMailReplyButton, "LEFT", -4, 0);
        
        saveMailButton:HookScript("OnClick", Notes_SaveMail);
        saveMailButton:HookScript("OnEnter", Notes_SaveMailButtonShowTooltip);
        saveMailButton:HookScript("OnLeave", Notes_SaveMailButtonHideTooltip);
    end	
end  

function Notes_SaveMail()
    local data = {};
    
    local sender = OpenMailSender.Name:GetText();
    local sendSubject = OpenMailSubject:GetText();
    
	local bodyText, texture, isTakeable, isInvoice = GetInboxText(InboxFrame.openMailID);
    
    local newNote = CopyTable(NOTE_TEMPLATE);
    newNote.title = sendSubject.." ("..FROM.." "..sender..")";
    newNote.text = bodyText;
    newNote.type = tonumber(NOTE_TYPE[0]);
    newNote.received = 1;
    newNote.category = MAIL_LABEL;
    tinsert(NotesData.notes, newNote);
end  

function Notes_SaveMailButtonShowTooltip()
    GameTooltip_SetDefaultAnchor(GameTooltip, _G["Notes_SaveMailButton"]);    
    GameTooltip:SetOwner(_G["Notes_SaveMailButton"], "ANCHOR_RIGHT");

    GameTooltip:SetText(NOTES_SAVE_MAIL_TOOLTIP);  
    
    GameTooltip:Show();      
end  

function Notes_SaveMailButtonHideTooltip()
	GameTooltip:FadeOut();
end  

function Notes_HookCleanUI()
    if (CLEANUI_HOOKED or not IsAddOnLoaded("CleanUI")) then
        return;
    end
    
    Notes_InitCleanUILayout();
    
    CLEANUI_HOOKED = 1;
end

function Notes_InitCleanUILayout()
    CleanUI_RemoveTextures(NotesFrame);  
    CleanUI_SetBackdrop(NotesFrame); 
    
    CleanUI_ChangeCloseButton(NotesFrame);
    
    CleanUI_CreateHeaderFrame(NotesFrame, "CUI_NotesFrameHeaderBorderframe", 200, NotesFrameTitleText);
    
    local background = NotesFrame:CreateTexture("CUI_NotesFramePaperBackground", "BORDER");
    background:ClearAllPoints();
    background:SetPoint("TOPLEFT", NotesFrame, "TOPLEFT", 350, -65);
    background:SetPoint("BOTTOMRIGHT", NotesFrame, "BOTTOMRIGHT", -25, 32);
    
    background:SetTexture("interface\\Addons\\CleanUI\\skins\\frame\\paper-background");
    
    NotesFrameCancelButton:SetPoint("BOTTOMRIGHT", -7, 8);
    NotesFrameCreateNoteButton:SetPoint("BOTTOMLEFT", 18, 8);
    
    CleanUI_ChangeButton(NotesFrameCancelButton); 
    CleanUI_ChangeButton(NotesFrameCreateNoteButton); 
    CleanUI_ChangeButton(NotesFrameCopyNoteButton); 
    CleanUI_ChangeButton(NotesFrameDeleteNoteButton); 
    CleanUI_ChangeButton(NotesFrameSendNoteButton); 
    
    CleanUI_ChangeButton(Notes_AddInfo); 
    CleanUI_ChangeDropDown(Notes_AddInfoDropDown); 
    CleanUI_ChangeDropDown(Notes_TypeDropDown); 
    
    CleanUI_ChangeButton(_G["Notes_SaveMailButton"]); 
end

function NotesFrame_Toggle()
    if (NotesFrame:IsVisible()) then
		HideUIPanel(NotesFrame);
    else
		ShowUIPanel(NotesFrame);
    end
end

function Notes_OnUpdate()
end

function pairsByKeysAsc(t, f)
    local a = {}
        for n in pairs(t) do table.insert(a, n) end
        table.sort(a, function(a,b) return a<b end)
        local i = 0      -- iterator variable
        local iter = function ()   -- iterator function
            i = i + 1
            if a[i] == nil then return nil
            else return a[i], t[a[i]]
            end
        end
    return iter
end

function Notes_SortNotesByCategory()
    local note, header;
    
    local help = {};
    local actCategory;    
    
    -- sort
    for i=1, #NotesData.notes do
       note = {};
       
       note.title = NotesData.notes[i].title;
       note.text = NotesData.notes[i].text;
       note.type = NotesData.notes[i].type;
       note.received = NotesData.notes[i].received;
       note.index = i;
       note.isHeader = nil;
       
       actCategory = NotesData.notes[i].category or "";
       help[actCategory] = help[actCategory] or {};

       tinsert(help[actCategory], note);
    end
    
    
    local entries = {};
    
    for category, notes in pairsByKeysAsc(help) do
        if (category ~= "") then
            header = {};
           
            header.title = category;
            header.isHeader = 1;
        
            tinsert(entries, header);
        end
        
        for i=1, #notes do
            tinsert(entries, notes[i]);
        end
    end
    
    return entries;
end

function Notes_OnUpdate()
	if ( not NotesFrame:IsShown() ) then
		return;
	end

	local entries = Notes_SortNotesByCategory();
    
    local numEntries = #entries;
	
	if ( numEntries == 0 ) then
		EmptyNotesFrame:Show();
	else
		EmptyNotesFrame:Hide();
	end
	
	if ( selectedIndex ) then
		ConfigNotesFrame:Show();
		EditNotesFrame:Show();
	else
		ConfigNotesFrame:Hide();
		EditNotesFrame:Hide();
	end

	-- hide the highlight frame initially, it may be shown when we loop through the quest listing if a quest is selected
	NotesFrameHighlightFrame:Hide();
	
	-- Update the quest listing
	local buttons = NotesFrameScrollFrame.buttons;
	local numButtons = #buttons;
	local scrollOffset = HybridScrollFrame_GetOffset(NotesFrameScrollFrame);
	local buttonHeight = buttons[1]:GetHeight();
	local displayedHeight = 0;
	
	local title, text, type, received;
	for i=1, numButtons do
	
		local noteIndex = i + scrollOffset;
		local notesTitle = buttons[i];
		
		if ( noteIndex <= numEntries ) then
		    if (entries[noteIndex].isHeader) then
                notesTitle:Disable();
                notesTitle.normalText:SetPoint("LEFT", notesTitle, "LEFT", 5, 0);
                
                title = entries[noteIndex].title;
                notesTitle:Show();
                notesTitle:SetText(GRAY_FONT_COLOR_CODE..title..FONT_COLOR_CODE_CLOSE);
                
                notesTitle.ready:Hide();
    		    notesTitle.notready:Hide();
		    else
                notesTitle:Enable();
                notesTitle.normalText:SetPoint("LEFT", notesTitle, "LEFT", 20, 0);
                
    		    title = entries[noteIndex].title;
    		    text = entries[noteIndex].text;
    		    type = entries[noteIndex].type;
                received = entries[noteIndex].received;
    		    
    			notesTitle:Show();
    		    
    		    if (not received) then
    		        notesTitle:SetText(title);
    		    else
    		        notesTitle:SetText(ORANGE_FONT_COLOR_CODE..title..FONT_COLOR_CODE_CLOSE);
    		    end
    		    
    		    notesTitle.index = entries[noteIndex].index;
    		    
    		    notesTitle.ready:Hide();
    		    notesTitle.notready:Hide();
    		    
    		    if (type == 2) then
    	    	    notesTitle.ready:Show();
    		    elseif (type == 3) then
        		    notesTitle.notready:Show();
    		    end
    
                -- this isn't a header, hide the header textures
                notesTitle:SetNormalTexture("");
                notesTitle:SetHighlightTexture("");
                
                if (entries[noteIndex].index == selectedIndex) then
                    -- reposition highlight frames
                    NotesFrameHighlightFrame:SetParent(notesTitle);
                    NotesFrameHighlightFrame:SetPoint("TOPLEFT", notesTitle, "TOPLEFT", 0, 0);
                    NotesFrameHighlightFrame:SetPoint("BOTTOMRIGHT", notesTitle, "BOTTOMRIGHT", 0, 0);
    		        NotesFrameHighlightFrame:Show();
                end
            end
		else
			notesTitle:Hide();
		end
		
		displayedHeight = displayedHeight + buttonHeight;
	end
	
	HybridScrollFrame_Update(NotesFrameScrollFrame, numEntries * buttonHeight, displayedHeight);
end

function Notes_OnMouseDown(frame, button)
    if (button == "LeftButton") then
        NotesFrame:StartMoving();
    end
end

function Notes_OnMouseUp(frame, button)
    NotesFrame:StopMovingOrSizing();
end

function Notes_OnReceiveDrag()
    local infoType, info1, info2 = GetCursorInfo();
	    
    if (infoType == "item") then
        TextBodyEditBox:Insert(info2);
    end
	
    if (infoType == "spell") then
        local skillType, spellId = GetSpellBookItemInfo(info1, "player");
        local link = GetSpellLink(spellId);
        TextBodyEditBox:Insert(link);
    end
	
    if (infoType == "merchant") then
        local link = GetMerchantItemLink(info1);
        TextBodyEditBox:Insert(link);
    end
	    		
	ClearCursor();
end

function Notes_DetectLink()
    local actCursorPosition = TextBodyEditBox:GetCursorPosition();
    if (actCursorPosition == lastCursorPos) then
        return;
    end
    lastCursorPos = actCursorPosition;
    
    local text = TextBodyEditBox:GetText();
    local start, ende = Notes_DetectLinkPosition(text, actCursorPosition);
    
    if (start and ende) then
        local link = strsub(text, start, ende);
		GameTooltip:SetOwner(NotesFrame, "ANCHOR_RIGHT");
		
		if (string.find(link, "[", 1, true)) then
	        GameTooltip:SetHyperlink(link);
	    end
    end
end

function Notes_DetectLinkPosition(text, cursorPos)

    local pos = 1;

    local start = 0;
    local ende = 0;
    
    while (pos and pos < strlen(text)) do    
        start = string.find(text, "|c", pos, true);
        
        if (not start) then
            return nil, nil;
        end
        
        ende = string.find(text, "|r", start, true);
        
        if (start and start < cursorPos and ende and ende > cursorPos) then
            return start, ende+1;
        end
        
        pos = ende;
 	end

    return nil, nil;
end

function NotesFrame_CreateNote()
 	local newNote = CopyTable(NOTE_TEMPLATE);
	tinsert(NotesData.notes, newNote);
 	
 	Notes_OnUpdate();
end

function NotesFrame_CopyNote()
 	if (selectedIndex) then
 	    local copyNote = CopyTable(NotesData.notes[selectedIndex]);
	    selectedIndex = selectedIndex + 1;
	    tinsert(NotesData.notes, selectedIndex, copyNote);
	
        Notes_OnUpdate();
    end
end

function NotesFrame_DeleteNote()
 	if (selectedIndex) then
        tremove(NotesData.notes, selectedIndex);
        selectedIndex = nil;
        
        Notes_OnUpdate();
    end
end

function NotesFrame_SendNote()
 	if (selectedIndex) then
        NotesFrameSendToMenu:Show();
    end
end

function NotesFrameSendToMenu_OnLoad(self)
	
	UIMenu_Initialize(self);
	UIMenu_AddButton(self, CHAT_MSG_PARTY, nil, function(frame) Notes_SendNoteTo(1) end);
	UIMenu_AddButton(self, CHAT_MSG_RAID, nil, function(frame) Notes_SendNoteTo(2) end);
	UIMenu_AddButton(self, CHAT_MSG_GUILD, nil, function(frame) Notes_SendNoteTo(3) end);
	UIMenu_AddButton(self, CHAT_MSG_BATTLEGROUND, nil, function(frame) Notes_SendNoteTo(4) end);
	UIMenu_AddButton(self, CHAT_MSG_WHISPER_INFORM, nil, function(frame) Notes_SendNoteTo(5) end);
	UIMenu_AutoSize(self);
end

function Notes_SendNoteTo(sendto)
    NotesFrameSendToMenu:Hide();
    
    if (sendto == 1) then       -- group
        Notes_SendNoteToPlayer("PARTY");
    elseif (sendto == 2) then   -- raid
        Notes_SendNoteToPlayer("RAID");
    elseif (sendto == 3) then   -- guild
        Notes_SendNoteToPlayer("GUILD");
    elseif (sendto == 4) then   -- battleground
        Notes_SendNoteToPlayer("BATTLEGROUND");
    elseif (sendto == 5) then   -- whisper
        local dialog = StaticPopup_Show("NOTES_SEND_NOTE");
        local editbox = _G[dialog:GetName().."EditBox"];
        editbox:SetText("");
    end
end

function Notes_SendNoteToPlayer(channel, name)
    
    local title = NotesData.notes[selectedIndex].title;
    local text = NotesData.notes[selectedIndex].text;
    local type = NotesData.notes[selectedIndex].type;
    
    if (strlen(text) > 200) then
        local first = true;
        
        while (strlen(text) > 200) do            
            local part = string.sub(text, 1, 200);
            text = string.sub(text, 201);
            
            if (first) then
                local message = title..";"..part..";"..type;
                Notes_SendNote("NI_SendNote", message, channel, name);
                first = false;
            else
                local message = title..";"..part..";"..type;
                Notes_SendNote("NI_SendNote+", message, channel, name);
            end
        end
        
        local message = title..";"..text..";"..type;
        Notes_SendNote("NI_SendNote+", message, channel, name);
    else
        local message = title..";"..text..";"..type;
        Notes_SendNote("NI_SendNote", message, channel, name);
    end
end

function Notes_SendNote(prefix, message, channel, name)    
    if (channel ~= "WHISPER") then
        SendAddonMessage(prefix, message, channel);
    else
        SendAddonMessage(prefix, message, channel, name);
    end
end

function Notes_ReceiveNote(prefix, message, channel, sender)

    if (prefix == nil) then
        return;
    end
    
    local playername, _ = UnitName("player");

    if (sender == playername) then
        return;
    end    

    if (prefix == "NI_SendNote") then    
        local title, text, type = strsplit(";", message, 3);
        
        title = title.." ("..sender..")";
        
        local newNote = CopyTable(NOTE_TEMPLATE);
        newNote.title = title;
        newNote.text = text;
        newNote.type = tonumber(type);
        newNote.received = 1;
        tinsert(NotesData.notes, newNote);
    end

    if (prefix == "NI_SendNote+") then    
        local title, text, type = strsplit(";", message, 3);
        
        title = title.." ("..sender..")";
        
        local note = Notes_FindNote(title);
        note.title = title;
        note.text = note.text..text;
        note.type = tonumber(type);
        note.received = 1;
    end
end

function Notes_FindNote(title)
    for i=1, #NotesData.notes do    
        local foundtitle = NotesData.notes[i].title;
        
        if (foundtitle == title) then
            return NotesData.notes[i];
        end
    end

    return CopyTable(NOTE_TEMPLATE);
end

function NotesFrameScrollFrame_OnLoad(self)
	HybridScrollFrame_OnLoad(self);
	self.update = NotesFrame_Update;
	HybridScrollFrame_CreateButtons(self, "NotesFrameTitleButtonTemplate");
end

function NotesFrameTitleButton_OnLoad(self)
	self:RegisterForClicks("LeftButtonUp", "RightButtonUp");
	self:RegisterEvent("UNIT_QUEST_LOG_CHANGED");
	self:RegisterEvent("GROUP_ROSTER_UPDATE");
	self:RegisterEvent("PARTY_MEMBER_ENABLE");
	self:RegisterEvent("PARTY_MEMBER_DISABLE");
end

function NotesFrameTitleButton_OnEvent(self, event, ...)
	if ( GameTooltip:IsOwned(self) ) then
		GameTooltip:FadeOut();
	end
end

function NotesFrameTitleButton_OnClick(self, button)
    selectedIndex = self.index;	
    lastCursorPos = nil;
    
    local title = NotesData.notes[selectedIndex].title;
    local text = NotesData.notes[selectedIndex].text;
    local type = NotesData.notes[selectedIndex].type;
    local category = NotesData.notes[selectedIndex].category;
	
	if (button == "LeftButton") then
		
		if (not type) then
		    type = 1;
		    NotesData.notes[selectedIndex].type = 1;
		end
		
	    TextBodyEditBox:SetText(text);
        UIDropDownMenu_Initialize(Notes_TypeDropDown, Note_TypeDropDown_Initialize);
	    UIDropDownMenu_SetSelectedValue(Notes_TypeDropDown, type);
    elseif (button == "RightButton" and IsControlKeyDown()) then
        -- edit note title
        local dialog = StaticPopup_Show("NOTES_EDIT_CATEGORY_POPUP");
        local editbox = _G[dialog:GetName().."EditBox"];
        editbox:SetText(category or "");
    elseif (button == "RightButton") then
        -- edit note title
        local dialog = StaticPopup_Show("NOTES_EDIT_TITLE_POPUP");
        local editbox = _G[dialog:GetName().."EditBox"];
        editbox:SetText(title or "");
    end
end

function Notes_OnTextChanged()
    local text = TextBodyEditBox:GetText();
    
    local num = TextBodyEditBox:GetNumLetters();
    local max = TextBodyEditBox:GetMaxLetters();
    
    EditNotesFrame_max:SetText(string.format("(%d/%d)", num, max));
    
    Notes_ChangeText(text);
end

function Notes_ChangeTitle(newTitle)
    NotesData.notes[selectedIndex].title = newTitle;
    NotesData.notes[selectedIndex].received = nil;
end

function Notes_ChangeCategory(newCategory)
    NotesData.notes[selectedIndex].category = newCategory;
    NotesData.notes[selectedIndex].received = nil;
end

function Notes_ChangeText(newText)
    NotesData.notes[selectedIndex].text = newText;
    NotesData.notes[selectedIndex].received = nil;
end

function NotesFrameTitleButton_OnEnter(self)
end

function NotesFrameTitleButton_OnLeave(self)
	if ( self:GetID() ~= selectedIndex ) then
	end
	GameTooltip:FadeOut();
end

-- 
-- add quest
--

function NotesFrame_AddInfo()
	if (selectedAddIndex >= 1 and selectedAddIndex <= 4) then
		NotesFrame_AddQuestInfo();
    end
    
	if (selectedAddIndex == 5) then
		NotesFrame_AddPetInfo();
    end
end

function NotesFrame_AddQuestInfo()
    local index = GetQuestLogSelection();
    
    if (not index) then   
        return;
    end
    
    local questTitle, level, questTag, suggestedGroup, isHeader, isCollapsed, isComplete, isDaily, questID = GetQuestLogTitle(index);
    
    if (not questTitle or isHeader) then   
        return;
    end
    
    local questDescription, questObjectives = GetQuestLogQuestText();
    
    if (selectedAddIndex == 1) then
        local questLink = GetQuestLink(index);
        TextBodyEditBox:Insert(questLink);
    elseif (selectedAddIndex == 2) then
        TextBodyEditBox:Insert(questTitle);
    elseif (selectedAddIndex == 3) then
        TextBodyEditBox:Insert(questObjectives);
    elseif (selectedAddIndex == 4) then
        
        local numrewards = GetNumQuestLogRewards();
        if (numrewards and numrewards > 0) then
            for i = 1, numrewards do
                local itemLink = GetQuestLogItemLink("reward", i);
                TextBodyEditBox:Insert(itemLink.."\n");
            end
            
            TextBodyEditBox:Insert("\n");
        end
        
        local numchoices = GetNumQuestLogChoices(); 
        if (numchoices and numchoices > 0) then
            for i = 1, numchoices do
                local itemLink = GetQuestLogItemLink("choice", i);
                TextBodyEditBox:Insert(itemLink.."\n");
            end
            
            TextBodyEditBox:Insert("\n");
        end
        
        local money = GetQuestLogRewardMoney();
        if (money and money > 0) then
            TextBodyEditBox:Insert(GetCoinText(money,", "));
        end
    end
end

function NotesFrame_AddPetInfo()        
    if (not PetJournalPetCard) then   
        return;
    end
    
	local selected = PetJournalPetCard.petIndex;
    
    if (not selected) then   
        return;
    end
    
	local petID, speciesID, owned = C_PetJournal.GetPetInfoByIndex(selected);	
	local speciesName, speciesIcon, petType, companionID, tooltipSource, tooltipDescription, isWild, canBattle, isTradeable, isUnique, obtainable = C_PetJournal.GetPetInfoBySpeciesID(speciesID)
	
    TextBodyEditBox:Insert(GREEN_FONT_COLOR_CODE..speciesName..FONT_COLOR_CODE_CLOSE.."\n    - "..tooltipSource.."\n    - "..tooltipDescription.."\n\n");
end

-- 
-- Add info dropdown
--

function Note_AddInfoDropDown_Initialize(self)
	for i = 1, #NOTE_QUEST_INFO do
        local info = UIDropDownMenu_CreateInfo();
        info.text = NOTE_QUEST_INFO[i];
        info.value = i;
        info.func = function()
                Note_ToggleAddQuestInfo(i);
            end
        UIDropDownMenu_AddButton(info);
	end
end

function Note_ToggleAddQuestInfo(index)
    selectedAddIndex = index;
	UIDropDownMenu_SetSelectedValue(Notes_AddInfoDropDown, index);
end

-- 
-- Notes settings
--

function Note_TypeDropDown_Initialize(self)
	for i = 1, #NOTE_TYPE do
        local info = UIDropDownMenu_CreateInfo();
        info.text = NOTE_TYPE[i];
        info.value = i;
        info.func = function()
                Note_ToggleNoteType(i);
            end
        UIDropDownMenu_AddButton(info);
	end
end

function Note_ToggleNoteType(type)
    NotesData.notes[selectedIndex].type = type;
    UIDropDownMenu_SetSelectedValue(Notes_TypeDropDown, type);
end