local _, addon = ...
local L = addon.L

addon.defaults = {
  fontSize               = {0, 0, 32},  -- set to custom number like 10 to use fontsize 10 for the item icons, 0 means automatic
  hideEvents             = {},           -- will be populated with events to totally ignore from showing (use the config GUI to do this, preferably)
  hideJunk               = {0, 0, 1},   -- set to 1 to hide gray (junk) loot, set to 0 to show it
  hideSoloLoot           = {0, 0, 1},    -- set to 1 to hide solo loot, note that the threshold option below will show those quality items yet skip all others, so set the below value to 8 to totally hide all solo loots
  hideSoloLootThreshold  = {0, 0, 8},    -- 0: gray, 1: white, 2: green, 3: blue, 4: purple, 5: legendar, 6: artifact, 7: heirloom, 8: disable
  hidePartyLoot          = {0, 0, 1},    -- set to 1 to hide party loot, note that the threshold option below will show those quality items yet skip all others, so set the below value to 8 to totally hide all party loots
  hidePartyLootThreshold = {3, 0, 8},    -- 0: gray, 1: white, 2: green, 3: blue, 4: purple, 5: legendar, 6: artifact, 7: heirloom, 8: disable
  outputChat             = "ChatFrame1", -- first chat, but you can set the number to 2 (combat log), 3 (new chat you've made) and so forth
  showRollDecisions      = {1, 0, 1},   -- shows need/greed/pass/disenchant decisions real time
  hideLFRRollDecisions   = {0, 0, 1},   -- hides need/greed/pass/dissenchant decisions in the Raid Finder
  showRollSummary        = {1, 0, 1},   -- shows summary of the roll outcome with the rest of the report
  hideLFRRollSummary     = {0, 0, 1},   -- hides summary of the roll outcome in the Raid Finder
  showRollIcons          = {0, 0, 1},   -- shows roll icons instead of item links
  sleepAfterCombat       = {2, -1, 25},  -- after combat ends it waits X seconds before reporting (subdued to extension from above setting)
  sleepBetweenEvents     = {2, 0, 25},  -- each time looting occurs it adds extra X seconds to wait before reporting
  mouseoverLinks         = {0, 0, 1},   -- set to 1 to show mouseover links, otherwise set to 0 to disable this feature
  mouseoverLinksAnchor   = {0, 0, 1},   -- set to 1 to anchor on the mouse, set to 0 to anchor above the chat frame
  mouseoverLinksIcons    = {0, 0, 1},   -- set to 1 to show icons, set to 0 to hide them
  lootCount              = {0, 0, 1},   -- set to 1 to count total items in inventory, set to 0 to turn this off
  classColors            = {0, 0, 1}    -- set to 1 to color by classes, set to 0 to turn this off
}

addon.lootwin = LOOT
addon.lootwintypes = {
  "COMBAT_FACTION_CHANGE",
  "COMBAT_GUILD_XP_GAIN",
  "COMBAT_HONOR_GAIN",
  "COMBAT_MISC_INFO",
  "COMBAT_XP_GAIN",
  "CURRENCY",
  "LOOT",
  "MONEY",
  --"OPENING", -- alerts you if someone mined a node, opens a box, e.g. (anyone near you will be shown)
  "PET_INFO",
  "SKILL",
  --"TRADESKILLS", -- crafts are spammy (includes everything near you that crafts something)
}

local print = function(...) -- for loot messages (output in the right chat)
  if not ... then return end
  local str = ""
  for _, raw in pairs({...}) do
    str = str..tostring(raw).." "
  end
  if str:len() > 1 then
    _G[MiniLootDB.outputChat]:AddMessage(str:sub(1, str:len()-1), 1, 1, 0)
  end
end addon.print = print

local _print = function(...) -- addon related messages have a prefix and output in the default chat
  if not ... then return end
  local str = ""
  for _, raw in pairs({...}) do
    str = str..tostring(raw).." "
  end
  if str:len() > 1 then
    DEFAULT_CHAT_FRAME:AddMessage(format("|cff00FF99[%s]|r %s", addon.name3, str:sub(1, str:len()-1)))
  end
end addon._print = _print

addon.variablecheck = function(addon, speckey)
  local defaults, resets = addon.defaults, 0
  MiniLootDB = type(MiniLootDB) == "table" and MiniLootDB or {}
  for opt, val in pairs(defaults) do
    if not speckey or (speckey == opt) then
      local bk, mi, ma, tm
      if opt == "hideEvents" or opt == "outputChat" then
        bk = val
      else
        bk = select(2, next(val))
        mi, ma = select(2, unpack(val))
      end
      if opt == "outputChat" then
        tm = MiniLootDB[opt]
        if type(tm) ~= "string" or not _G[tm] or not _G[tm].AddMessage then
          MiniLootDB[opt] = bk
          resets = resets + 1
        end
      elseif type(MiniLootDB[opt]) ~= type(bk) then
        MiniLootDB[opt] = bk
        resets = resets + 1
      elseif mi and ma then
        tm = tonumber(tostring(MiniLootDB[opt]))
        if tm and (tm < mi or tm > ma) then
          MiniLootDB[opt] = bk
          resets = resets + 1
        end
      end
    end
  end
  return resets
end addon:variablecheck()

local bool = function(val)
  val = type(val) == "number" and val or tonumber(val)
  if type(val) ~= "number" then return nil end -- returns nil (aka. false) if not a number
  if val == 0 then
    return false
  end
  return true
end addon.bool = bool

addon.GetItemCount = function(addon, count, ...)
  local total = GetItemCount(...)
  if count == total then
    return ""
  end
  return total > 1 and format("|cff999999(%d)|r", total) or ""
end

addon.GetCurrencyCount = function(addon, count, currencyLink)
  currencyLink = currencyLink:match("|h%[(.+)%]|h")
  local currency = type(addon.currencies) == "table" and addon.currencies[currencyLink]
  local total = 0
  if type(currency) == "table" then
    total = select(2, GetCurrencyInfo(currency[1]))
    if count == total then
      return ""
    end
  end
  return total > 1 and format("|cff999999(%d)|r", total) or ""
end

addon.ClassColor = function(addon, plr)
  if not bool(MiniLootDB.classColors) then
    plr = nil
  end
  local class = type(plr) == "string" and select(2, UnitClass(plr))
  return class and type(RAID_CLASS_COLORS[class]) == "table" and format("%02x%02x%02x", RAID_CLASS_COLORS[class].r*255, RAID_CLASS_COLORS[class].g*255, RAID_CLASS_COLORS[class].b*255) or "FFFF00" -- fallback to system yellow (happens if player leaves group before message is printed)
end

do
  addon.itemQualityColorTags = {}
  addon.itemQualityColorNames = {ITEM_QUALITY0_DESC, ITEM_QUALITY1_DESC, ITEM_QUALITY2_DESC, ITEM_QUALITY3_DESC, ITEM_QUALITY4_DESC, ITEM_QUALITY5_DESC, ITEM_QUALITY6_DESC, ITEM_QUALITY7_DESC, L["Hide all"]}
  do for i = 0, 7 do addon.itemQualityColorTags[i] = "|c"..select(4, GetItemQualityColor(i)) end end -- patch 4.2 removed the prefix (had to manually reapply it for this addon)
  addon.grayColorTag = addon.itemQualityColorTags[0]
  addon.getColorRankByTag = function(self, str)
    for rank, tag in pairs(self.itemQualityColorTags) do
      if str:match(tag) then
        return rank
      end
    end
    return 0 -- fallback to gray item quality
  end
  addon.getItemQualityName = function(self, id)
    return self.itemQualityColorNames[(tonumber(id) or 0)+1] or self.itemQualityColorNames[1] -- fallback to first value (poor/gray/junk)
  end
  do
    for i = 0, 8 do
      addon.itemQualityListWithNames = (addon.itemQualityListWithNames or "")..format("%d = %s%s|r\n", i, addon.itemQualityColorTags[i] or "|cffFFFF00", addon.itemQualityColorNames[i+1])
    end
    addon.itemQualityListWithNames = addon.itemQualityListWithNames:sub(1, addon.itemQualityListWithNames:len()-1)
  end
  addon.tipScan = CreateFrame("GameTooltip", addon.name2.."TooltipScanner", UIParent, "GameTooltipTemplate")
  addon.isItemClassQuest = function(self, class) return class == select(10, GetAuctionItemClasses()) end
  addon.isItemQuestStarter = function(self, str, questingeneral)
    local itemid = tonumber(str)
    itemid = itemid or str:match("item:(%d+)")
    if itemid then
      self.tipScan:Show()
      self.tipScan:SetOwner(UIParent, "ANCHOR_NONE")
      self.tipScan:ClearAllPoints()
      self.tipScan:SetPoint("CENTER")
      self.tipScan:SetHyperlink("item:"..itemid)
      for i = 1, self.tipScan:NumLines() do
        if _G[self.tipScan:GetName().."TextLeft"..i]:GetText():match(ITEM_STARTS_QUEST) then
          self.tipScan:Hide()
          return 1
        end
        if questingeneral and _G[self.tipScan:GetName().."TextLeft"..i]:GetText():match(ITEM_BIND_QUEST) then
          self.tipScan:Hide()
          return 1
        end
      end
      self.tipScan:Hide()
    end
  end
end

do
  local tip = CreateFrame("GameTooltip", addon.name2.."HoveringTooltip", UIParent, "GameTooltipTemplate")
  local safeHyperlinks = {
    achievement = {1, 1, 0}, -- yellow
    currency = {0, .67, 0}, -- dark green
    enchant = {1, 1, 0}, -- yellow
    glyph = {.44, .84, 1}, -- light blue
    instancelock = {1, .5, 0}, -- orange
    item = {1, 1, 1}, -- white (dynamic)
    --journal = {.44, .84, 1}, -- light blue
    quest = {1, 1, 0}, -- yellow
    spell = {.44, .84, 1}, -- light blue
    talent = {.31, .59, .97}, -- dark blue
    unit = {.8, .8, .8}, -- gray
  }
  local glyphClassFilenames = {
    "DeathKnight",
    "Druid",
    "Hunter",
    "Mage",
    "Paladin",
    "Priest",
    "Rogue",
    "Shaman",
    "Warlock",
    "Warrior",
  }
  tip.GetGlyphType = function(tip)
    local obj, txt
    for i = 1, 10 do
      obj = _G[tip:GetName().."TextLeft"..i]
      if obj then
        txt = obj:GetText()
        if txt:find(PRIME_GLYPH) then
          return 1
        elseif txt:find(MAJOR_GLYPH) then
          return 2
        elseif txt:find(MINOR_GLYPH) then
          return 3
        end
      end
    end
  end
  tip.SetIconTexture = function(tip, texture, count)
    if not bool(MiniLootDB.mouseoverLinksIcons) then
      if tip.icon then
        tip.icon:Hide()
      end
      return
    end
    if type(texture) ~= "string" or texture:len() == 0 then
      --texture = "Interface\\Icons\\INV_Misc_QuestionMark"
      --texture = "Interface\\Icons\\TEMP"
      texture = ""
    end
    if not tip.icon then
      tip.icon = tip:CreateTexture(nil, "BACKGROUND")
      tip.icon:SetSize(42, 42)
      --tip.icon.count = tip:CreateFontString(nil, "ARTWORK", "GameFontNormal")
      --tip.icon.count:SetTextColor(1, 1, 1)
      --tip.icon.count:SetPoint("BOTTOMRIGHT", tip.icon, "BOTTOMRIGHT", -3, 3)
    end
    tip.icon:ClearAllPoints()
    if ((tip:GetLeft() or 46)-42-4) < 0 then -- show on the right side if it will be out of the screen (:GetLeft() is nil when ANCHOR_CURSOR -when cursored show always on left side no mater what)
      tip.icon:SetPoint("TOPLEFT", tip, "TOPRIGHT", 0, -2.5)
    else
      tip.icon:SetPoint("TOPRIGHT", tip, "TOPLEFT", 0, -2.5)
    end
    tip.icon:SetTexture(texture)
    --tip.icon.count:SetText(count or "")
    tip.icon:Show()
  end
  tip.Set = function(tip, linkData, link)
    if linkData then
      if tip.icon then
        tip.icon:Hide()
      end
      local chat = _G[MiniLootDB.outputChat]
      tip.elem = GetMouseFocus()
      if chat.buttonFrame and bool(MiniLootDB.mouseoverLinksAnchor) then
        tip:SetOwner(chat.buttonFrame, "ANCHOR_TOPLEFT", 1, 40)
      else
        tip:SetOwner(UIParent, "ANCHOR_CURSOR")
      end
      tip:ClearLines()
      local href = (":"):split(linkData, 2)
      local bcol = safeHyperlinks[href]
      if bcol then -- it's nil if the hyperlink is not supported by the widget API
        tip:SetHyperlink(linkData)
        if href == "item" then
          local count, _, _, quality, _, _, class, subclass, maxstack, _, texture = GetItemCount(link or 0) or 1, GetItemInfo(link or 0)
          tip:SetIconTexture(texture, count > 1 and count or "")
          if addon:isItemClassQuest(class) or addon:isItemClassQuest(subclass) then
            if addon:isItemQuestStarter(linkData) then
              bcol = {1, .2, .2} -- it starts a quest so instead of heirloom color it's now red!
            else
              bcol = {.9, .8, .5}
            end
          else
            bcol = {GetItemQualityColor(quality or 1)} -- fallback to white color
            bcol[4] = nil -- remove hex color -it's not alpha after all
          end
        elseif href == "enchant" or href == "spell" or href == "talent" then
          local spellID = tonumber(select(3, tip:GetSpell()) or 0) or 0
          local _, _, texture = GetSpellInfo(spellID)
          tip:SetIconTexture(texture or "", "")
        elseif href == "glyph" then
          local gtype = tip:GetGlyphType() or 3
          local cname = glyphClassFilenames[math.random(1, #glyphClassFilenames)]
          local file = format("INV_Glyph_%s%s", gtype == 1 and "Prime" or gtype == 2 and "Major" or "Minor", cname)
          tip:SetIconTexture("Interface\\Icons\\"..file, "") -- we don't know what class the glyph belongs to so we pick one at random
        elseif href == "achievement" then
          local achID = tonumber(linkData:match("^"..href..":(%d+)") or 0) or 0
          local _, _, _, _, _, _, _, _, _, texture = GetAchievementInfo(achID)
          tip:SetIconTexture(texture or "", "")
        elseif href == "currency" then
          local curID = tonumber(linkData:match("^"..href..":(%d+)") or 0) or 0
          local name, amount, texture, weekAmount, weekMax, totalMax, isHeader = GetCurrencyInfo(curID)
          tip:SetIconTexture("Interface\\Icons\\"..texture, "")
        end
        if not tip.backdropmod then
          tip.backdropmod = 1 -- only apply the backdrop once per tip creation (without this block the background insets are wrong)
          local bd, bdc = tip:GetBackdrop(), {tip:GetBackdropColor()}
          bd.insets = {left=2, right=2, top=2, bottom=2}
          tip:SetBackdrop(bd)
          tip:SetBackdropColor(unpack(bdc))
        end
        tip:SetBackdropBorderColor(unpack(bcol))
      else
        --local raw = ""
        --local tmp = {(":"):split(linkData)}
        --for i = 1, #tmp do tip:AddLine("["..i.."] "..tmp[i]) end
        local prefix, category, id = (":"):split(linkData)
        if prefix == addon.name2 then
          --[[
          if category == "currency" then
            local name, amount, texture, weekAmount, weekMax, totalMax, isHeader = GetCurrencyInfo(id)
            if (name or ""):len() == 0 then
              name, isHeader, _, _, _, amount, texture, _, _ = GetCurrencyListInfo(id)
              weekAmount, weekMax, totalMax = 0, 0, 0 -- this API does not really tell us, now does it?
            else
              texture = "Interface\\Icons\\"..(texture or "")
            end
            if (name or ""):len() > 0 then
              tip:SetIconTexture(texture or "", "")
              tip:AddLine(format("|cffFFFFFF%d|r |cffCCCCCCx|r %s", amount, name))
              if weekMax > 0 then
                tip:AddLine(format(CURRENCY_WEEKLY_CAP, "", weekAmount, weekMax))
              end
              if totalMax > 0 then
                tip:AddLine(format(CURRENCY_TOTAL_CAP, "", amount, totalMax))
              end
            end
          end
          --]]
        end
      end
      if tip:NumLines() > 0 then
        tip:Show()
      end
    else
      if tip.icon then
        tip.icon:Hide()
      end
      tip.elem = nil
      tip:Hide()
    end
  end
  tip.Enable = function(tip)
    local chat = _G[MiniLootDB.outputChat]
    if not tip.oH then
      tip.oH = 1
      tip.oHEnter = chat:GetScript("OnHyperlinkEnter")
      tip.oHLeave = chat:GetScript("OnHyperlinkLeave")
      tip.oUpdate = chat:GetScript("OnUpdate")
    end
    chat:SetScript("OnHyperlinkEnter", function(chat, linkData, link, ...)
      local tip = addon.hoverTip
      tip:Set(linkData, link)
      if tip.oHEnter then
        tip:oHEnter(linkData, link, ...)
      end
    end)
    chat:SetScript("OnHyperlinkLeave", function(chat, linkData, link, ...)
      local tip = addon.hoverTip
      tip:Set(nil)
      if tip.oHLeave then
        tip:oHLeave(linkData, link, ...)
      end
    end)
    chat:SetScript("OnUpdate", function(chat, elapsed, ...)
      tip.oUS = (tip.oUS or 0) + elapsed
      if tip.oUS > .1 then
        local tip = addon.hoverTip
        if tip.elem ~= GetMouseFocus() then
          tip:Set(nil)
        end
        tip.oUS = 0
      end
      if tip.oUpdate then
        tip:oUpdate(chat, elapsed, ...)
      end
    end)
  end
  tip.Disable = function(tip)
    local chat = _G[MiniLootDB.outputChat]
    if tip.oH then
      chat:SetScript("OnHyperlinkEnter", tip.oHEnter)
      chat:SetScript("OnHyperlinkLeave", tip.oHLeave)
      chat:SetScript("OnUpdate", tip.oUpdate)
    end
  end
  addon.hoverTip = tip
  addon.mouseoverLinks = function(addon)
    local tip = addon.hoverTip
    local on = bool(MiniLootDB.mouseoverLinks)
    if on then
      tip:Enable()
    else
      tip:Disable()
    end
  end
end

local function cfgAddCat(child)
  local panel = CreateFrame("Frame", addon.name2..child.key)
  panel.name = child.name
  panel.parent = child.parent
  -- optional functions
  for _, fname in pairs({"default", "refresh", "okay", "cancel"}) do
    panel[fname] = child[fname]
  end
  -- title
  local title = panel:CreateFontString()
  if child.title then
    title:SetFont(STANDARD_TEXT_FONT, 16, "")
    title:SetTextColor(1, .82, 0, 1)
    title:SetShadowColor(0, 0, 0, 1)
    title:SetShadowOffset(1, -1)
    title:SetJustifyH("LEFT")
    title:SetJustifyV("TOP")
    title:SetPoint("TOPLEFT", 16, -16)
    title:SetHeight(16)
    title:SetText(child.title)
  end
  -- description
  local desc = panel:CreateFontString()
  if child.desc then
    desc:SetFont(STANDARD_TEXT_FONT, 10, "")
    desc:SetTextColor(1, 1, 1, 1)
    desc:SetShadowColor(0, 0, 0, 1)
    desc:SetShadowOffset(1, -1)
    desc:SetJustifyH("LEFT")
    desc:SetJustifyV("TOP")
    desc:SetPoint("TOPLEFT", title, 0, title:GetText() and -24 or -16)
    desc:SetHeight(32)
    desc:SetText(child.desc)
  end
  -- create point where we can start adding elements to the panel
  local last = panel:CreateFontString()
  last:SetWidth(1)
  last:SetHeight(1)
  if desc:GetText() then
    last:SetPoint("TOPLEFT", desc, "TOPLEFT", 0, -desc:GetStringHeight()-8)
  elseif title:GetText() then
    last:SetPoint("TOPLEFT", title, "TOPLEFT", 0, -title:GetStringHeight()-8)
  else
    last:SetPoint("TOPLEFT", 16, -16)
  end
  -- add to category
  panel:HookScript("OnShow", addon.cfgrefresh)
  addon[child.key] = panel
  InterfaceOptions_AddCategory(panel)
  panel:Hide()
  return panel, last
end

local function cfgAddLabel(name, parent, relative, width, height, padx, pady, halig, valig)
  width = width or 355
  height = height or 16
  padx = padx or 0
  pady = pady or 0
  local label = parent:CreateFontString(name, "ARTWORK", "ChatFontNormal")
  local font = {label:GetFont()}
  font[2] = 12
  label:SetFont(unpack(font))
  label:SetPoint("TOPLEFT", relative, "TOPLEFT", 0+padx, -relative:GetHeight()-8+pady)
  label:SetWidth(width)
  label:SetHeight(height)
  label:SetJustifyH(halig or "CENTER")
  label:SetJustifyV(valig or "MIDDLE")
  return label
end

local function cfgAddInput(name, parent, relative, width, height, padx, pady)
  width = width or 355
  height = height or 16
  padx = padx or 0
  pady = pady or 0
  local edit = CreateFrame("EditBox", name, parent, "InputBoxTemplate")
  local font = {edit:GetFont()}
  font[2] = 12
  edit:SetFont(unpack(font))
  edit:SetPoint("LEFT", relative, "RIGHT", padx, pady)
  edit:SetWidth(width)
  edit:SetHeight(height)
  edit:SetAutoFocus(false)
  edit:SetScript("OnEditFocusGained", function(edit)
    if type(edit.OnEnter) == "function" then
      edit.text = edit.text or edit:GetText()
    end
  end)
  edit:SetScript("OnEditFocusLost", function(edit)
    EditBox_ClearFocus(edit)
    if type(edit.OnEnter) == "function" then
      if edit.text then
        edit:SetText(edit.text)
        edit.text = nil
      end
    end
  end)
  edit:SetScript("OnEnterPressed", function(edit)
    if type(edit.OnEnter) == "function" then
      edit:OnEnter()
    end
    edit:GetScript("OnEditFocusLost")(edit)
  end)
  edit:SetScript("OnTextChanged", edit:GetScript("OnEditFocusGained"))
  edit:SetScript("OnEscapePressed", edit:GetScript("OnEditFocusLost"))
  edit:HookScript("OnEnter", function(edit)
    if type(edit.tooltipText) == "string" then
      GameTooltip_AddNewbieTip(edit, edit.tooltipText, 1, 1, 1, nil)
    end
  end)
  edit:HookScript("OnLeave", GameTooltip_Hide)
  return edit
end

local function cfgAddCheck(name, parent, relative, width, height, padx, pady)
  width = width or 16
  height = height or 16
  padx = padx or 0
  pady = pady or 0
  local check = CreateFrame("CheckButton", name, parent, "UICheckButtonTemplate")
  check:SetPoint("LEFT", relative, "RIGHT", padx, pady)
  check:SetWidth(width)
  check:SetHeight(height)
  check:SetScript("OnClick", function(check)
    if type(check.OnEnter) == "function" then
      check:OnEnter()
    end
  end)
  check:HookScript("OnEnter", function(check)
    if type(check.tooltipText) == "string" then
      GameTooltip_AddNewbieTip(check, check.tooltipText, 1, 1, 1, nil)
    end
  end)
  check:HookScript("OnLeave", GameTooltip_Hide)
  return check
end

local function cfgAddRadio(name, parent, relative, width, height, padx, pady)
  width = width or 16
  height = height or 16
  padx = padx or 0
  pady = pady or 0
  local radio = CreateFrame("CheckButton", name, parent, "UIRadioButtonTemplate")
  radio:SetPoint("LEFT", relative, "RIGHT", padx, pady)
  radio:SetWidth(width)
  radio:SetHeight(height)
  radio:SetScript("OnClick", function(radio)
    if type(radio.OnEnter) == "function" then
      radio:OnEnter()
    end
  end)
  radio:HookScript("OnEnter", function(radio)
    if type(radio.tooltipText) == "string" then
      GameTooltip_AddNewbieTip(radio, radio.tooltipText, 1, 1, 1, nil)
    end
  end)
  radio:HookScript("OnLeave", GameTooltip_Hide)
  return radio
end

local function cfgAddBtn(name, parent, relative, width, height, padx, pady)
  width = width or 128
  height = height or 16
  padx = padx or 0
  pady = pady or 0
  local btn = CreateFrame("Button", name, parent, "UIPanelButtonTemplate2")
  btn:SetPoint("LEFT", relative, "RIGHT", padx, pady)
  btn:SetWidth(width)
  btn:SetHeight(height)
  btn:SetScript("OnClick", function(btn)
    if type(btn.OnEnter) == "function" then
      btn:OnEnter()
    end
  end)
  btn:HookScript("OnEnter", function(btn)
    if type(btn.tooltipText) == "string" then
      GameTooltip_AddNewbieTip(btn, btn.tooltipText, 1, 1, 1, nil)
    end
  end)
  btn:HookScript("OnLeave", GameTooltip_Hide)
  return btn
end

function addon:loadCfgUI()
  -- handle refresh call
  addon.cfgrefresh = function()
    if addon.sub0.refresh then addon.sub0:refresh() end
    if addon.sub1.refresh then addon.sub1:refresh() end
    if addon.sub2.refresh then addon.sub2:refresh() end
    if addon.sub3.refresh then addon.sub3:refresh() end
    if addon.sub4.refresh then addon.sub4:refresh() end
  end
  -- default and refresh
  local function default()
    MiniLootDB = nil
    addon:variablecheck()
    addon:mouseoverLinks()
  end
  local function refresh(self)
    addon:variablecheck()
    local children = {self:GetChildren()}
    local value
    for _, child in pairs(children) do
      if type(child) == "table" and type(child.key) == "string" then
        if child.key == "hideEvents" then
          value = MiniLootDB[child.key][child.val]
          child:SetChecked(value and true or false)
        elseif child.key == "hideJunk" or child.key == "showRollDecisions" or child.key == "hideLFRRollDecisions" or child.key == "showRollSummary" or child.key == "hideLFRRollSummary" or child.key == "showRollIcons" or child.key == "mouseoverLinksAnchor" or child.key == "mouseoverLinksIcons" or child.key == "lootCount" or child.key == "classColors" then
          value = bool(MiniLootDB[child.key])
          child:SetChecked(value)
        elseif child.key == "hidePartyLoot" or child.key == "hideSoloLoot" then
          value = bool(MiniLootDB[child.key])
          child:SetChecked(value)
          addon.sub1:refreshBoxes()
        elseif child.key == "hidePartyLootThreshold" or child.key == "hideSoloLootThreshold" then
          if MiniLootDB[child.key] == child:GetID() then
            child:OnEnter()
          end
        elseif child.key == "mouseoverLinks" then
          value = bool(MiniLootDB[child.key])
          child:SetChecked(value)
          addon:mouseoverLinks()
        else -- handles unspecified editboxes
          value = MiniLootDB[child.key]
          value = tostring(value) or ""
          child:SetText(value)
        end
      end
    end
  end
  -- sub0
  local sub0, point0 = cfgAddCat({key="sub0", name=addon.name2, title=addon.name2..L[" settings"], desc=L["Set custom font-size if you wish, or change what chat frame the loot is written to."], default=default, refresh=refresh})
  local label = cfgAddLabel("$parentLabel1", sub0, point0, 150, 16, 0, -8, "LEFT")
  label:SetText(L["Font size"])
  local value = cfgAddInput("$parentValue1", sub0, label, 200, 16)
  value.key = "fontSize"
  value.tooltipText = L["Enter \"0\" to let the game decide."]
  value:SetNumeric(true)
  value:SetMaxBytes(3) -- 2 digits + null
  value.OnEnter = function(value)
    MiniLootDB[value.key] = value:GetNumber()
    addon:variablecheck()
    value.text = MiniLootDB[value.key]
    print(format("%s: %sx1337 (%s)", UnitName("player"), addon.GetIconWithLink("|cffffffff|Hitem:6948:0:0:0:0:0:0:0:0|h[Hearthstone]|h|r"), L["did the new size fit?"]))
  end
  local label = cfgAddLabel("$parentLabel2", sub0, value, 150, 16, -label:GetWidth(), 0, "LEFT")
  label:SetText(L["Output chat"])
  local value = cfgAddInput("$parentValue2", sub0, label, 200, 16)
  value.key = "outputChat"
  value.tooltipText = L[([=[
The reports will be printed in this chat.
If what you enter is not valid the value will
reset to default.
"ChatFrame2" would print in the Combat log,
"ChatFrame3" would print in a custom chat
you made that is next to the Combat log,
and so forth. Use "/fstack" to find the
name of the frame you wish to use.
]=])]:trim()
  value.OnEnter = function(value)
    local tip = addon.hoverTip
    tip:Disable()
    MiniLootDB[value.key] = value:GetText()
    addon:variablecheck()
    value.text = MiniLootDB[value.key]
    addon:mouseoverLinks()
    print(format(L["This message should have been shown in %s."], MiniLootDB[value.key]))
  end
  -- sub1
  local sub1, point1 = cfgAddCat({key="sub1", parent=sub0.name, name=L["Filter"], title=L["Filter settings"], desc=L["Decide what you see and what is hidden and ignored."], default=default, refresh=refresh})
  sub1.boxes1 = {}
  sub1.boxes2 = {}
  sub1.refreshBoxes = function(sub1)
    local box
    for i = 1, #sub1.boxes1 do
      box = sub1.boxes1[i]
      if sub1.pararelBox1:GetChecked() then
        box:Enable()
        box:SetAlpha(1)
      else
        box:Disable()
        box:SetAlpha(.5)
      end
    end
    box = nil
    for i = 1, #sub1.boxes2 do
      box = sub1.boxes2[i]
      if sub1.pararelBox2:GetChecked() then
        box:Enable()
        box:SetAlpha(1)
      else
        box:Disable()
        box:SetAlpha(.5)
      end
    end
  end
  local label = cfgAddLabel("$parentLabel1", sub1, point1, 150, 16, 0, -8, "LEFT")
  label:SetText(L["Hide these events"])
  do
    local HideEvents = {
      CHAT_MSG_COMBAT_FACTION_CHANGE = REPUTATION,
      CHAT_MSG_COMBAT_HONOR_GAIN = HONOR,
      CHAT_MSG_COMBAT_XP_GAIN = COMBAT_XP_GAIN,
      CHAT_MSG_CURRENCY = format("%s + %s", CURRENCY, PROFESSIONS_ARCHAEOLOGY),
      CHAT_MSG_LOOT = LOOT,
      CHAT_MSG_MONEY = MONEY,
    }
    local OnEnter = function(value)
      MiniLootDB[value.key][value.val] = value:GetChecked() and 1 or nil
      addon:variablecheck()
      value:SetChecked(MiniLootDB[value.key][value.val] and true or false)
    end
    local LastHideEvent
    for event, tip in pairs(HideEvents) do
      local value = cfgAddCheck("$parentValue1_"..event, sub1, LastHideEvent or label, 16, 16, not LastHideEvent and -7 or 0, 0)
      value.key = "hideEvents"
      value.val = event
      value.OnEnter = OnEnter
      value.tooltipText = tip
      LastHideEvent = value
    end
  end
  local label = cfgAddLabel("$parentLabel2", sub1, value, 150, 16, -label:GetWidth(), 24, "LEFT")
  label:SetText(L["Hide all junk loot"])
  local value = cfgAddCheck("$parentValue2", sub1, label, 16, 16, -7, 0)
  value.key = "hideJunk"
  value.tooltipText = L["All items of poor quality will not be shown."]
  value.OnEnter = function(value)
    MiniLootDB[value.key] = value:GetChecked() and 1 or 0
    addon:variablecheck()
    value:SetChecked(bool(MiniLootDB[value.key]))
  end
  local label = cfgAddLabel("$parentLabel3", sub1, value, 150, 16, -label:GetWidth()+7, -16, "LEFT")
  label:SetText(L["Hide own loot"])
  local value = cfgAddCheck("$parentValue3", sub1, label, 16, 16, -7, 0)
  sub1.pararelBox1 = value
  value.key = "hideSoloLoot"
  value.tooltipText = L["Hide items below specific quality."]
  value.OnEnter = function(value)
    MiniLootDB[value.key] = value:GetChecked() and 1 or 0
    addon:variablecheck()
    value:SetChecked(bool(MiniLootDB[value.key]))
    sub1:refreshBoxes()
  end
  local label = cfgAddLabel("$parentLabel4", sub1, value, 150, 16, -label:GetWidth()+7, 0+8, "LEFT")
  label:SetText(L["Quality threshold"])
  do
    local rarities = addon.itemQualityColorNames
    local OnEnter = function(self, skip)
      local box
      for i = 1, #sub1.boxes1 do
        box = sub1.boxes1[i]
        if self == box then
          MiniLootDB[self.key] = self:GetID()
          box:SetChecked(true)
        else
          box:SetChecked(false)
        end
      end
      if not skip then
        addon:variablecheck()
        box = sub1.boxes1[MiniLootDB[self.key]+1]
        box:OnEnter(1)
      end
    end
    local rarity, rcolor
    local LastRarityElement
    for i = 1, #rarities do
      rarity = rarities[i]
      rcolor = addon.itemQualityColorTags[i-1] or "|cffFF5555"
      local value = cfgAddRadio("$parentValue4_Rarity"..i, sub1, LastRarityElement or label, 12, 12, not LastRarityElement and -6 or -12, not LastRarityElement and 0 or -12)
      value:SetID(i-1)
      value.key = "hideSoloLootThreshold"
      value.OnEnter = OnEnter
      _G[value:GetName().."Text"]:SetText(rcolor..rarity.."|r")
      LastRarityElement = value
      table.insert(sub1.boxes1, value)
    end
  end
  local label = cfgAddLabel("$parentLabel5", sub1, value, 150, 16, -label:GetWidth()+7, -16-106, "LEFT")
  label:SetText(L["Hide party loot"])
  local value = cfgAddCheck("$parentValue5", sub1, label, 16, 16, -7, 0)
  sub1.pararelBox2 = value
  value.key = "hidePartyLoot"
  value.tooltipText = L["Hide items below specific quality."]
  value.OnEnter = function(value)
    MiniLootDB[value.key] = value:GetChecked() and 1 or 0
    addon:variablecheck()
    value:SetChecked(bool(MiniLootDB[value.key]))
    sub1:refreshBoxes()
  end
  local label = cfgAddLabel("$parentLabel6", sub1, value, 150, 16, -label:GetWidth()+7, 0+8, "LEFT")
  label:SetText(L["Quality threshold"])
  do
    local rarities = addon.itemQualityColorNames
    local OnEnter = function(self, skip)
      local box
      for i = 1, #sub1.boxes2 do
        box = sub1.boxes2[i]
        if self == box then
          MiniLootDB[self.key] = self:GetID()
          box:SetChecked(true)
        else
          box:SetChecked(false)
        end
      end
      if not skip then
        addon:variablecheck()
        box = sub1.boxes2[MiniLootDB[self.key]+1]
        box:OnEnter(1)
      end
    end
    local rarity, rcolor
    local LastRarityElement
    for i = 1, #rarities do
      rarity = rarities[i]
      rcolor = addon.itemQualityColorTags[i-1] or "|cffFF5555"
      local value = cfgAddRadio("$parentValue6_Rarity"..i, sub1, LastRarityElement or label, 12, 12, not LastRarityElement and -6 or -12, not LastRarityElement and 0 or -12)
      value:SetID(i-1)
      value.key = "hidePartyLootThreshold"
      value.OnEnter = OnEnter
      _G[value:GetName().."Text"]:SetText(rcolor..rarity.."|r")
      LastRarityElement = value
      table.insert(sub1.boxes2, value)
    end
  end
  -- sub2
  local sub2, point2 = cfgAddCat({key="sub2", parent=sub0.name, name=L["Roll"], title=L["Roll settings"], desc=L["Change the roll handling behavior."], default=default, refresh=refresh})
  local label = cfgAddLabel("$parentLabel1", sub2, point2, 150, 16, 0, -8, "LEFT")
  label:SetText(L["Show roll decisions"])
  local value = cfgAddCheck("$parentValue1", sub2, label, 16, 16, -7, 0)
  value.key = "showRollDecisions"
  value.tooltipText = L["The need/greed decisions are instantly shown."]
  value.OnEnter = function(value)
    MiniLootDB[value.key] = value:GetChecked() and 1 or 0
    addon:variablecheck()
    value:SetChecked(bool(MiniLootDB[value.key]))
  end
  local label = cfgAddLabel("$parentLabel1B", sub2, value, 150, 16, -label:GetWidth()+7, 0, "LEFT")
  label:SetText(L["Hide roll decisions in LFR"])
  local value = cfgAddCheck("$parentValue1B", sub2, label, 16, 16, -7, 0)
  value.key = "hideLFRRollDecisions"
  value.tooltipText = L["Overrides above option and hides roll decisions in the Raid Finder."]
  value.OnEnter = function(value)
    MiniLootDB[value.key] = value:GetChecked() and 1 or 0
    addon:variablecheck()
    value:SetChecked(bool(MiniLootDB[value.key]))
  end
  local label = cfgAddLabel("$parentLabel2", sub2, value, 150, 16, -label:GetWidth()+7, 0, "LEFT")
  label:SetText(L["Show roll summary"])
  local value = cfgAddCheck("$parentValue2", sub2, label, 16, 16, -7, 0)
  value.key = "showRollSummary"
  value.tooltipText = L["Prints the roll results in one line together\nwith the rest of the report."]
  value.OnEnter = function(value)
    MiniLootDB[value.key] = value:GetChecked() and 1 or 0
    addon:variablecheck()
    value:SetChecked(bool(MiniLootDB[value.key]))
  end
  local label = cfgAddLabel("$parentLabel2B", sub2, value, 150, 16, -label:GetWidth()+7, 0, "LEFT")
  label:SetText(L["Hide roll summary in LFR"])
  local value = cfgAddCheck("$parentValue2B", sub2, label, 16, 16, -7, 0)
  value.key = "hideLFRRollSummary"
  value.tooltipText = L["Overrides above option and hides the roll results in the Raid Finder."]
  value.OnEnter = function(value)
    MiniLootDB[value.key] = value:GetChecked() and 1 or 0
    addon:variablecheck()
    value:SetChecked(bool(MiniLootDB[value.key]))
  end
  local label = cfgAddLabel("$parentLabel3", sub2, value, 150, 16, -label:GetWidth()+7, 0, "LEFT")
  label:SetText(L["Show rolls as icons"])
  local value = cfgAddCheck("$parentValue3", sub2, label, 16, 16, -7, 0)
  value.key = "showRollIcons"
  value.tooltipText = L["Instead of item links, will show icons at roll related events."]
  value.OnEnter = function(value)
    MiniLootDB[value.key] = value:GetChecked() and 1 or 0
    addon:variablecheck()
    value:SetChecked(bool(MiniLootDB[value.key]))
  end
  -- sub3
  local sub3, point3 = cfgAddCat({key="sub3", parent=sub0.name, name=L["Timer"], title=L["Timer settings"], desc=L["Change how quickly the summary is shown and interval that data is gathered."], default=default, refresh=refresh})
  local label = cfgAddLabel("$parentLabel1", sub3, point3, 150, 16, 0, -8, "LEFT")
  label:SetText(L["Sleep after combat"])
  local value = cfgAddInput("$parentValue1", sub3, label, 200, 16)
  value.key = "sleepAfterCombat"
  value.tooltipText = L[([=[
For example "1.5" would make the addon wait
1.5 seconds before printing the report in the
chat. This gives you time to finish looting
after the combat, it would merge the combat-
looters and the loot after combat ended.
Use "-1" to skip combat checking and only
use between events sleep timer. Using "0"
will only instantly display after combat,
not during.
]=])]:trim()
  value.OnEnter = function(value)
    MiniLootDB[value.key] = value:GetNumber()
    addon:variablecheck()
    value.text = MiniLootDB[value.key]
  end
  local label = cfgAddLabel("$parentLabel2", sub3, value, 150, 16, -label:GetWidth(), 0, "LEFT")
  label:SetText(L["Sleep between events"])
  local value = cfgAddInput("$parentValue2", sub3, label, 200, 16)
  value.key = "sleepBetweenEvents"
  value.tooltipText = L[([=[
Each time there is an event like item loot,
experience gain, honor gain and such, the addon
waits for a moment to make sure no more events
will come, then when it's "safe" it will report
the findings and await new information.
This timer will decide how long it waits between
an event and until the next one, if no next event
then it reports and continues as normal.
]=])]:trim()
  value.OnEnter = function(value)
    MiniLootDB[value.key] = value:GetNumber()
    addon:variablecheck()
    value.text = MiniLootDB[value.key]
  end
  -- sub4
  local sub4, point4 = cfgAddCat({key="sub4", parent=sub0.name, name=L["Extra"], title=L["Extra features"], desc=L["Not perhaps a major aspect of the addon, but may be helpful."], default=default, refresh=refresh})
  local label = cfgAddLabel("$parentLabel1", sub4, point4, 150, 16, 0, -8, "LEFT")
  label:SetText(L["Original loot chat"])
  local value = cfgAddBtn("$parentValue1", sub4, label, 128, 18, -7, 0)
  value.tooltipText = L["This will create an additional chat frame,\ndo not be afraid it will overwrite anything\nbecause it wont."]
  value:SetText(L["Create this window!"])
  value.OnEnter = function(value)
    addon.LootWindowCreate()
  end
  local label = cfgAddLabel("$parentLabel2", sub4, value, 150, 16, -label:GetWidth()+7, 0, "LEFT")
  label:SetText(L["Show mouseover links"])
  local value = cfgAddCheck("$parentValue2", sub4, label, 16, 16, -7, 0)
  value.key = "mouseoverLinks"
  value.tooltipText = L["For example when you mouseover an item\nthe tooltip is shown without clicking it."]
  value.OnEnter = function(value)
    MiniLootDB[value.key] = value:GetChecked() and 1 or 0
    addon:variablecheck()
    local val = bool(MiniLootDB[value.key])
    value:SetChecked(val)
    addon:mouseoverLinks()
  end
  local label = cfgAddLabel("$parentLabel3", sub4, value, 150, 16, -label:GetWidth()+7, 0, "LEFT")
  label:SetText(L["Anchor to the chat"])
  local value = cfgAddCheck("$parentValue3", sub4, label, 16, 16, -7, 0)
  value.key = "mouseoverLinksAnchor"
  value.tooltipText = L["Anchor the mouseover tooltip to the top\nof the chat frame."]
  value.OnEnter = function(value)
    MiniLootDB[value.key] = value:GetChecked() and 1 or 0
    addon:variablecheck()
    local val = bool(MiniLootDB[value.key])
    value:SetChecked(val)
  end
  local label = cfgAddLabel("$parentLabel4", sub4, value, 150, 16, -label:GetWidth()+7, 0, "LEFT")
  label:SetText(L["Show tooltip icon"])
  local value = cfgAddCheck("$parentValue4", sub4, label, 16, 16, -7, 0)
  value.key = "mouseoverLinksIcons"
  value.tooltipText = L["Shows an icon on the side of the tooltip."]
  value.OnEnter = function(value)
    MiniLootDB[value.key] = value:GetChecked() and 1 or 0
    addon:variablecheck()
    local val = bool(MiniLootDB[value.key])
    value:SetChecked(val)
  end
  local label = cfgAddLabel("$parentLabel5", sub4, value, 150, 16, -label:GetWidth()+7, 0, "LEFT")
  label:SetText(L["Show lootcount"])
  local value = cfgAddCheck("$parentValue5", sub4, label, 16, 16, -7, 0)
  value.key = "lootCount"
  value.tooltipText = L["Append number of items owned behind each looted item."]
  value.OnEnter = function(value)
    MiniLootDB[value.key] = value:GetChecked() and 1 or 0
    addon:variablecheck()
    local val = bool(MiniLootDB[value.key])
    value:SetChecked(val)
  end
  local label = cfgAddLabel("$parentLabel6", sub4, value, 150, 16, -label:GetWidth()+7, 0, "LEFT")
  label:SetText(L["Color names by class"])
  local value = cfgAddCheck("$parentValue6", sub4, label, 16, 16, -7, 0)
  value.key = "classColors"
  value.tooltipText = L["Color player names by their class."]
  value.OnEnter = function(value)
    MiniLootDB[value.key] = value:GetChecked() and 1 or 0
    addon:variablecheck()
    local val = bool(MiniLootDB[value.key])
    value:SetChecked(val)
  end
  -- refresh variables (and load the hooks we need at login)
  addon:cfgrefresh()
  -- debug
  --InterfaceOptionsFrame_OpenToCategory(addon.sub1)
end
