local _, addon = ...
local L, bool, print, _print = addon.L, addon.bool, addon.print, addon._print
local worker = CreateFrame("Frame")
local rollIndex = "AAA" -- fist when sorted

addon.loaded = {}
addon.events = {}
addon.queue = {}

addon.matches = {
  CHAT_MSG_COMBAT_FACTION_CHANGE = { -- name(faction), value(rep), type(1/nil for inc/dec)
    [1] = {FACTION_STANDING_DECREASED = {1, 2, nil}},
    [2] = {FACTION_STANDING_DECREASED_GENERIC = {1, nil, nil}},
    [3] = {FACTION_STANDING_INCREASED = {1, 2, 1}},
    [4] = {FACTION_STANDING_INCREASED_BONUS = {1, 2, 1}},
    [5] = {FACTION_STANDING_INCREASED_GENERIC = {1, nil, 1}},
  },
  CHAT_MSG_COMBAT_HONOR_GAIN = { -- points(honor), source(name), rank(int)
    [1] = {COMBATLOG_HONORAWARD = {1, nil, nil}},
    [2] = {COMBATLOG_HONORGAIN = {3, 1, 2}},
    [3] = {COMBATLOG_HONORGAIN_NO_RANK = {2, 1, nil}},
  },
  CHAT_MSG_COMBAT_XP_GAIN = { -- points(exp), source(name), isLost(1/nil for lost or gained)
    [1] = {COMBATLOG_XPGAIN_EXHAUSTION1 = {2, 1, nil}},
    [2] = {COMBATLOG_XPGAIN_EXHAUSTION1_GROUP = {2, 1, nil}},
    [3] = {COMBATLOG_XPGAIN_EXHAUSTION1_RAID = {2, 1, nil}},
    [4] = {COMBATLOG_XPGAIN_EXHAUSTION2 = {2, 1, nil}},
    [5] = {COMBATLOG_XPGAIN_EXHAUSTION2_GROUP = {2, 1, nil}},
    [6] = {COMBATLOG_XPGAIN_EXHAUSTION2_RAID = {2, 1, nil}},
    [7] = {COMBATLOG_XPGAIN_EXHAUSTION4 = {2, 1, nil}},
    [8] = {COMBATLOG_XPGAIN_EXHAUSTION4_GROUP = {2, 1, nil}},
    [9] = {COMBATLOG_XPGAIN_EXHAUSTION4_RAID = {2, 1, nil}},
    [10] = {COMBATLOG_XPGAIN_EXHAUSTION5 = {2, 1, nil}},
    [11] = {COMBATLOG_XPGAIN_EXHAUSTION5_GROUP = {2, 1, nil}},
    [12] = {COMBATLOG_XPGAIN_EXHAUSTION5_RAID = {2, 1, nil}},
    [13] = {COMBATLOG_XPGAIN_FIRSTPERSON = {2, 1, nil}},
    [14] = {COMBATLOG_XPGAIN_FIRSTPERSON_GROUP = {2, 1, nil}},
    [15] = {COMBATLOG_XPGAIN_FIRSTPERSON_RAID = {2, 1, nil}},
    [16] = {COMBATLOG_XPGAIN_FIRSTPERSON_UNNAMED = {1, nil, nil}},
    [17] = {COMBATLOG_XPGAIN_FIRSTPERSON_UNNAMED_GROUP = {1, nil, nil}},
    [18] = {COMBATLOG_XPGAIN_FIRSTPERSON_UNNAMED_RAID = {1, nil, nil}},
    [19] = {COMBATLOG_XPGAIN_QUEST = {1, nil, nil}},
    [20] = {COMBATLOG_XPLOSS_FIRSTPERSON_UNNAMED = {1, nil, 1}},
  },
  CHAT_MSG_CURRENCY = { -- name(item, no link), count(int), target(name)
    [1] = {CURRENCY_GAINED_MULTIPLE = {1, 2, nil}},
    [2] = {CURRENCY_GAINED = {1, nil, nil}},
  },
  CHAT_MSG_LOOT = { -- name(item), count(int), target(name)
    [1] = {LOOT_ITEM_CREATED_SELF_MULTIPLE = {1, 2, nil}},
    [2] = {LOOT_ITEM_CREATED_SELF = {1, nil, nil}},
    [3] = {LOOT_ITEM_PUSHED_SELF_MULTIPLE = {1, 2, nil}},
    [4] = {LOOT_ITEM_PUSHED_SELF = {1, nil, nil}},
    [5] = {LOOT_ITEM_SELF_MULTIPLE = {1, 2, nil}},
    [6] = {LOOT_ITEM_SELF = {1, nil, nil}},
    [7] = {LOOT_ITEM_MULTIPLE = {2, 3, 1}},
    [8] = {LOOT_ITEM = {2, nil, 1}},
    [9] = {CREATED_ITEM_MULTIPLE = {2, 3, 1}},
    [10] = {CREATED_ITEM = {2, nil, 1}},
    -- roll handling; item(name), player(name), roll(int), type(nil or 0xABC; A:disenchant, B:greed, C:need)
    [11] = {LOOT_ROLL_ROLLED_NEED_ROLE_BONUS = {2, 3, 1, 0x001}},
    [12] = {LOOT_ROLL_ROLLED_NEED = {2, 3, 1, 0x001}},
    [13] = {LOOT_ROLL_ROLLED_GREED = {2, 3, 1, 0x010}},
    [14] = {LOOT_ROLL_ROLLED_DE = {2, 3, 1, 0x100}},
    -- instant roll decision info; item(name), player(name), type(nil:pass, 1:disenchant, 2:greed, 3:need), dummy(nil), isInstant
    [15] = {LOOT_ROLL_NEED_SELF = {1, nil, 3, nil, 1}},
    [16] = {LOOT_ROLL_NEED = {2, 1, 3, nil, 1}},
    [17] = {LOOT_ROLL_GREED_SELF = {1, nil, 2, nil, 1}},
    [18] = {LOOT_ROLL_GREED = {2, 1, 2, nil, 1}},
    [19] = {LOOT_ROLL_DISENCHANT_SELF = {1, nil, 1, nil, 1}},
    [20] = {LOOT_ROLL_DISENCHANT = {2, 1, 1, nil, 1}},
    [21] = {LOOT_ROLL_PASSED_SELF = {1, nil, nil, nil, 1}},
    [22] = {LOOT_ROLL_PASSED = {2, 1, nil, nil, 1}},
  },
  CHAT_MSG_MONEY = { -- gold(money), target(name), unused(nil)
    [1] = {YOU_LOOT_MONEY_GUILD = {1, nil, nil}},
    [2] = {YOU_LOOT_MONEY = {1, nil, nil}},
    [3] = {LOOT_MONEY_SPLIT_GUILD = {1, nil, nil}},
    [4] = {LOOT_MONEY_SPLIT = {1, nil, nil}},
    [5] = {LOOT_MONEY = {2, 1, nil}},
  },
}

do -- grammatical differences in other languages
  if addon.locale == "ruRU" then
    -- roll handling; item(name), player(name), roll(int), type(nil or 0xABC; A:disenchant, B:greed, C:need)
    addon.matches.CHAT_MSG_LOOT[11].LOOT_ROLL_ROLLED_NEED_ROLE_BONUS = {2, 1, 3, 0x001}
    addon.matches.CHAT_MSG_LOOT[12].LOOT_ROLL_ROLLED_NEED = {2, 1, 3, 0x001}
    addon.matches.CHAT_MSG_LOOT[13].LOOT_ROLL_ROLLED_GREED = {2, 1, 3, 0x010}
    addon.matches.CHAT_MSG_LOOT[14].LOOT_ROLL_ROLLED_DE = {2, 1, 3, 0x100}
    -- instant roll decision info; item(name), player(name), type(nil:pass, 1:disenchant, 2:greed, 3:need), dummy(nil), isInstant
    addon.matches.CHAT_MSG_LOOT[15].LOOT_ROLL_NEED_SELF = {2, nil, 3, nil, 1}
    addon.matches.CHAT_MSG_LOOT[16].LOOT_ROLL_NEED = {1, 2, 3, nil, 1}
    addon.matches.CHAT_MSG_LOOT[17].LOOT_ROLL_GREED_SELF = {2, nil, 2, nil, 1}
    addon.matches.CHAT_MSG_LOOT[18].LOOT_ROLL_GREED = {1, 2, 2, nil, 1}
    addon.matches.CHAT_MSG_LOOT[20].LOOT_ROLL_DISENCHANT = {1, 2, 1, nil, 1}
  end
end

local whenInRF = function(expect, fallback)
  if GetLFGMode() and GetLFGModeType() == "raid" then
    return expect
  end
  if type(fallback) ~= "nil" then
    return fallback and true or false
  end
  return true
end

local validPlayerName = function(str)
  if addon.locale == "ruRU" then
    if str:match(" ") and (not str:match("%+")) then -- if missmatched player name with another string (representing "You" then player name is nil, notice that LFR bonus rolls contain an "+" so keep that in mind)
      str = nil
    end
  end
  return str
end

local stripNameRollBonus = function(str)
  local isBonus = str:match("%+")
  return isBonus and (str:gsub("%+.+$", ""):trim().."") or str, isBonus -- fixes " + Role Bonus" at the end of the name, roll bonuses are always 101-200 so there should not be a need for the description
end

local FormatToMatcher = function(str)
  if type(str) ~= "string" then return "" end
  str = str:gsub("%.", "%%%.") -- "." into "%."
  str = str:gsub("%(", "%%%(") -- "(" into "%("
  str = str:gsub("%)", "%%%)") -- ")" into "%)"
  str = str:gsub("%-", "%%%-") -- "-" into "%-"
  str = str:gsub("%%(%d)%$(%a)", "%%%2") -- "%3$s" into "%s" (works for s and d)
  str = str:gsub("%%%%%.(%d)f", "%%s") -- "%.#f" into "%s" (will have decimal)
  str = str:gsub("%%d", "%(%%d+%)") -- numbers
  str = str:gsub("%%s", "%(%.+%)") -- strings
  return str
end

local MoneyStringToCopper = function(str)
  if type(str) ~= "string" then return 0 end
  str = str:lower()
  local sum = 0
  local gm, sm, cm = FormatToMatcher(GOLD_AMOUNT):lower(), FormatToMatcher(SILVER_AMOUNT):lower(), FormatToMatcher(COPPER_AMOUNT):lower()
  local g, s, c = str:match(gm), str:match(sm), str:match(cm)
  if g then sum = sum + g * 10000 end
  if s then sum = sum + s * 100 end
  if c then sum = sum + c end
  return sum
end

StaticPopupDialogs["MINILOOT_LOOT_WINDOW_OVERWRITE"] = {
  text = format(L["The chat window \"%s\" already exists.\nAre you sure you wish to overwrite it?\n\nBy overwriting it will reset the messages it shows to default. (Messages like experience, honor, money, reputation, and many others.)"], addon.lootwin),
  button1 = YES,
  button2 = NO,
  OnAccept = function(self)
    self.LootWindowCreate(1)
  end,
  exclusive = 1,
  whileDead = 1,
  hideOnEscape = 1,
  showAlert = 1,
  timeout = 0,
}

local LootWindowExists = function(name)
  local f, t
  for i = 1, NUM_CHAT_WINDOWS do
    f = _G["ChatFrame"..i]
    t = _G["ChatFrame"..i.."Tab"]
    if f and t and (t:IsVisible() or f.minimized) and name == FCF_GetChatWindowInfo(i) then
      return _G["ChatFrame"..i]
    end
  end
end

local ClearLootWindow = function(name)
  for i = 1, NUM_CHAT_WINDOWS do
    if name == FCF_GetChatWindowInfo(i) then
      FCF_Close(_G["ChatFrame"..i])
    end
  end
end

local LootWindowCreate LootWindowCreate = function(confirmed)
  local LootWindow = LootWindowExists(addon.lootwin)
  if LootWindow and not confirmed then
    StaticPopup_Show("MINILOOT_LOOT_WINDOW_OVERWRITE").LootWindowCreate = LootWindowCreate
  else
    ClearLootWindow(addon.lootwin)
    FCF_OpenNewWindow(addon.lootwin)
    LootWindow = LootWindowExists(addon.lootwin)
    ChatFrame_RemoveAllMessageGroups(LootWindow)
    ChatFrame_RemoveAllChannels(LootWindow)
    for _, chatType in pairs(addon.lootwintypes) do
      ChatFrame_AddMessageGroup(LootWindow, chatType)
    end
  end
end addon.LootWindowCreate = LootWindowCreate

local CurrencyCaches = function()
  local temp, name, isHeader, icon, link = {}
  for i = 1, GetCurrencyListSize() do
    name, isHeader, _, _, _, _, icon = GetCurrencyListInfo(i)
    link = GetCurrencyListLink(i) or ""
    if name and (not isHeader) then
      temp[name] = {tonumber(link:match(":(%d+)")) or i, icon}
    end
  end
  addon.currencies = temp
end

local ArchaeologyCaches = function()
  local name, icon, link
  for _, i in pairs({384, 385, 393, 394, 397, 398, 399, 400, 401}) do -- CurrencyTypes.dbc
    name, _, icon = GetCurrencyInfo(i)
    link = GetCurrencyListLink(i) or ""
    if name and icon then
      addon.currencies[name] = {tonumber(link:match(":(%d+)")) or i, "Interface\\Icons\\"..icon} -- must add the folder path manually
    end
  end
end

local ConvertMatches = function()
  local temp = {}
  for event, data in pairs(addon.matches) do
    temp[event] = {}
    for i = 1, #data do
      temp[event][i] = {}
      for key, tbl in pairs(data[i]) do
        temp[event][i][FormatToMatcher(_G[key])] = tbl
      end
    end
  end
  addon.matches = temp
end

local UCFirst = function(str)
  local temp = ""
  temp = str:sub(1,1):upper()
  return temp..str:sub(2):lower()
end

local Abbrev = function(str)
  local abbr = ""
  local words = {(" "):split(tostring(str))}
  if #words > 1 then
    for _, part in pairs(words) do
      abbr = abbr..UCFirst(part:sub(1,3)) -- try with 3 letters first...
    end
    if abbr:len() > 12 then -- only shorten more if larger than 12 chars!
      abbr = ""
      for _, part in pairs(words) do
        abbr = abbr..UCFirst(part:sub(1,2)) -- using 2 letters as it exceeds the limit with 3!
      end
    end
    return abbr
  end
  return str
end

local GetSelectedWindow = function()
  return _G[MiniLootDB.outputChat] or DEFAULT_CHAT_FRAME
end

local GetFontSize = function()
  return type(MiniLootDB.fontSize) == "number" and MiniLootDB.fontSize > 0 and MiniLootDB.fontSize or (select(2, FCF_GetChatWindowInfo(GetSelectedWindow():GetID())) or 13)-1 -- either use custom fontsize specified or attempt to extract the default size, at worst fallback to 12px!
end

local GetIconWithLink = function(str, iscurrency)
  if type(str) ~= "string" then return "" end -- BUG: maybe fixed, not tested, have to uncomment this and find out!

  if iscurrency then
    str = str:match("|h%[(.+)%]|h")
    local currency = type(addon.currencies) == "table" and addon.currencies[str]
    if type(currency) == "table" then
      --return format("|cff%s|H%s:%s:%d|h[|T%s:%d|t]|h|r", "FFFFFF", addon.name2, "currency", currency[1], currency[2], GetFontSize())
      return format("|cff00AA00|Hcurrency:%d|h[|T%s:%d|t]|h|r", currency[1], currency[2], GetFontSize())
    end
  else
    local color, itemstr, name = str:match("|cff(.+)|Hitem:(.+)|h%[(.+)%]|h|r")
    if color then
      local _, _, _, _, _, class, subclass = GetItemInfo(str)
      if addon:isItemClassQuest(class) or addon:isItemClassQuest(subclass) then -- if quest item color and add additional noticable tags to the output
        if addon:isItemQuestStarter(str) then
          color = "FF3333" -- it starts a quest so instead of heirloom color it's now red!
        else
          color = "E6CC80"
        end
      end
      return format("|cff%s|Hitem:%s|h[|T%s:%d|t]|h|r", color, itemstr, GetItemIcon(str), GetFontSize()) -- font size corresponds to chat font size (uses currently set size)
    end
  end
  return str
end addon.GetIconWithLink = GetIconWithLink

local NewLine = function(str)
  if str:sub(str:len()) ~= "\n" then
    return str.."\n" -- add new line if does not exist from before (at the end)
  end
  return str
end

local UpdRepLoot = function(tbl, data)
  if not data[1] then return end -- no faction name, no entry!
  tbl[data[1]] = (tbl[data[1]] or 0) + (math.abs(data[2]) * data[3]) -- insert or update faction rep value (data[3] modifies if it's gained or lost)
end

local UpdHonorLoot = function(tbl, data)
  if not data[1] then return end -- no honor value, no entry!
  tbl[1] = (tbl[1] or 0) + data[1]
end

local UpdXPLoot = function(tbl, data)
  if not data[1] then return end -- no xp value, no entry!
  tbl[1] = (tbl[1] or 0) + (math.abs(data[1]) * data[3]) -- insert or update xp value (data[3] modifies if it's gained or lost)
end

local UpdCurrencyLoot = function(tbl, data)
  if not data[1] then return end -- no currency name, no currency entry!
  if type(tbl[data[3]]) ~= "table" then tbl[data[3]] = {} end
  if type(tbl[data[3]][data[1]]) ~= "table" then -- new currency, insert it
    tbl[data[3]][data[1]] = data
  else -- update currency count for person
    tbl[data[3]][data[1]][2] = tbl[data[3]][data[1]][2] + data[2]
  end
end

local UpdItemLoot = function(tbl, data)
  if not data[1] then return end -- no item name, no item entry!
  if bool(MiniLootDB.hideJunk) and data[1]:match(addon.grayColorTag) then return end -- if we are supposed to hide junk, hide it!
  if bool(MiniLootDB.hidePartyLoot) and data[3] ~= UnitName("player") then -- if we are supposed to hide party loot (under the threshold, also can't be our own loot), this is where we handle this function...
    if MiniLootDB.hidePartyLootThreshold == 8 or MiniLootDB.hidePartyLootThreshold > addon:getColorRankByTag(data[1]) then -- if threshold is disabled, or if it's higher than the item, hide it!
      if not addon:isItemQuestStarter(data[1], 1) then -- do not hide quest (starting) items (in case you forget to loot the mob that dropped it in the first place, think of it as a "hey dude, loot the corpse for a quest!" kind of nudge)
        return
      end
    end
  elseif bool(MiniLootDB.hideSoloLoot) and data[3] == UnitName("player") then -- if we are supposed to hide some of our own loot (under the threshold, also can't be someone elses loot), this is where we handle this function...
    if MiniLootDB.hideSoloLootThreshold == 8 or MiniLootDB.hideSoloLootThreshold > addon:getColorRankByTag(data[1]) then -- if threshold is disabled, or if it's higher than the item, hide it!
      if not addon:isItemQuestStarter(data[1], 1) then -- do not hide quest (starting) items (in case you forget to loot the mob that dropped it in the first place, think of it as a "hey dude, loot the corpse for a quest!" kind of nudge)
        return
      end
    end
  end
  local itemid = data[1]:match("item:(%d+)") or data[1] -- use this as key for the table (so stacks of the same item always end up at the same entry, not duplicates!) NOTE: only issue here is if there are several "of the" items, same itemid but various attributes, but luckily this isn't often met in the game anyway!
  if type(tbl[data[3]]) ~= "table" then tbl[data[3]] = {} end -- character doesn't exist, create a empty table for it
  if type(tbl[data[3]][itemid]) ~= "table" then -- new item, insert it
    tbl[data[3]][itemid] = data
  else -- existing item, add additional number of items to the counter field
    tbl[data[3]][itemid][2] = tbl[data[3]][itemid][2] + data[2]
  end
end

local UpdMoneyLoot = function(tbl, data)
  if not data[1] then return end -- no gold looted, no money entry!
  tbl[data[2]] = (tbl[data[2]] or 0) + MoneyStringToCopper(data[1]) -- insert or update gold value
end

local UpdRollData = function(tbl, data)
  if not data[1] then return end -- no item rolled for, no roll entry!
  if type(tbl[rollIndex]) ~= "table" then tbl[rollIndex] = {} end
  if type(tbl[rollIndex][data[1]]) ~= "table" then tbl[rollIndex][data[1]] = {} end
  local dise, greed, need = bit.band(data[4], 0x100)>0, bit.band(data[4], 0x010)>0, bit.band(data[4], 0x001)>0
  if dise then
    tbl[rollIndex][data[1]][data[2]] = {3, data[3]}
  elseif greed then
    tbl[rollIndex][data[1]][data[2]] = {2, data[3]}
  elseif need then
    tbl[rollIndex][data[1]][data[2]] = {1, data[3]}
  else
    tbl[rollIndex][data[1]][data[2]] = {0, 0}
  end
end

local OnEvent = function(worker, event, message, ...)
  -- make the interval skip until we have waited some seconds between the events
  addon.lastevent = GetTime()
  -- patch event, ptbl contains metadata while found contains the match return
  local temp, found, ptbl, matches, breakloop
  matches = addon.matches[event]
  for i = 1, #matches do
    if breakloop then
      break
    end
    for matcher, tbl in pairs(matches[i]) do
      temp = {message:match(matcher)}
      if #temp > 0 then
        found = temp
        ptbl = tbl
        breakloop = 1
      end
    end
  end
  -- abort if no pattern matched the message (message is not important, I guess...)
  if not found or not ptbl then
    return --_print("[DEBUG]", event, message)
  end
  -- abort if we are supposed to hide this kind of events
  if MiniLootDB.hideEvents[event] then
    return --_print("[DEBUG]", event, message, "(matched filter, will not be shown!)")
  end
  -- create the nessecary table structure
  if type(addon.queue[event]) ~= "table" then
    addon.queue[event] = {}
  end
  -- appropriately store the information we got
  if event == "CHAT_MSG_COMBAT_FACTION_CHANGE" then
    UpdRepLoot(addon.queue[event], {
      ptbl[1] and found[ptbl[1]],
      ptbl[2] and tonumber(found[ptbl[2]]) or 0,
      ptbl[3] and 1 or -1,
    })
  elseif event == "CHAT_MSG_COMBAT_HONOR_GAIN" then
    UpdHonorLoot(addon.queue[event], {
      ptbl[1] and tonumber(found[ptbl[1]]) or 0,
      ptbl[2] and found[ptbl[2]] or "",
      ptbl[3] and tonumber(found[ptbl[3]]) or 0,
    })
  elseif event == "CHAT_MSG_COMBAT_XP_GAIN" then
    UpdXPLoot(addon.queue[event], {
      ptbl[1] and tonumber(found[ptbl[1]]) or 0,
      ptbl[2] and found[ptbl[2]] or "",
      ptbl[3] and -1 or 1,
    })
  elseif event == "CHAT_MSG_CURRENCY" then
    UpdCurrencyLoot(addon.queue[event], {
      ptbl[1] and found[ptbl[1]],
      ptbl[2] and tonumber(found[ptbl[2]]) or 1,
      ptbl[3] and found[ptbl[3]] or UnitName("player"),
    })
  elseif event == "CHAT_MSG_LOOT" then
    if ptbl[5] and (ptbl[1] and found[ptbl[1]]) then
      if bool(MiniLootDB.showRollDecisions) and whenInRF(not bool(MiniLootDB.hideLFRRollDecisions)) then
        local item = ptbl[1] and found[ptbl[1]]
        local plr = ptbl[2] and validPlayerName(found[ptbl[2]]) or UnitName("player")
        if bool(MiniLootDB.showRollIcons) then
          item = GetIconWithLink(item)
        end
        if bool(MiniLootDB.classColors) then
          print(format("%s |cff%s%s|r|cff%s: %s|r ", item, addon:ClassColor(plr), plr, (ptbl[3] == 1 and "EE3333") or (ptbl[3] == 2 and "CCCC33") or (ptbl[3] == 3 and "33EE33") or "CCCCCC", (ptbl[3] == 1 and ROLL_DISENCHANT) or (ptbl[3] == 2 and GREED) or (ptbl[3] == 3 and NEED) or PASS))
        else
          print(format("%s |cff%s%s: %s|r ", item, (ptbl[3] == 1 and "EE3333") or (ptbl[3] == 2 and "CCCC33") or (ptbl[3] == 3 and "33EE33") or "CCCCCC", plr, (ptbl[3] == 1 and ROLL_DISENCHANT) or (ptbl[3] == 2 and GREED) or (ptbl[3] == 3 and NEED) or PASS))
        end
      end
    elseif ptbl[4] then
      if bool(MiniLootDB.showRollSummary) and whenInRF(not bool(MiniLootDB.hideLFRRollSummary)) then
        UpdRollData(addon.queue[event], {
          ptbl[1] and found[ptbl[1]],
          ptbl[2] and validPlayerName(found[ptbl[2]]) or UnitName("player"),
          ptbl[3] and tonumber(found[ptbl[3]]) or 0,
          tonumber(ptbl[4]) or 0,
        })
      end
    else
      UpdItemLoot(addon.queue[event], {
        ptbl[1] and found[ptbl[1]],
        ptbl[2] and tonumber(found[ptbl[2]]) or 1,
        ptbl[3] and found[ptbl[3]] or UnitName("player"),
      })
    end
  elseif event == "CHAT_MSG_MONEY" then
    UpdMoneyLoot(addon.queue[event], {
      ptbl[1] and found[ptbl[1]],
      ptbl[2] and found[ptbl[2]] or UnitName("player"),
    })
  --else
  --  _print("[DEBUG]", event, message)
  end
end

local HookChatEvents = function()
  -- register our events and apply filtering and parsing of messages
  for event, _ in pairs(addon.matches) do
    ChatFrame_AddMessageEventFilter(event, function(chatFrame, event, ...)
      if chatFrame == GetSelectedWindow() then return true end -- block all events in ChatFrame1 (addon handles that window)
      return false, ...
    end)
    addon.events[event] = OnEvent
    worker:RegisterEvent(event)
  end
  -- block and parse certain special system messages
  ChatFrame_AddMessageEventFilter("CHAT_MSG_SYSTEM", function(chatFrame, event, msg, ...)
    if chatFrame == GetSelectedWindow() and msg:find(FormatToMatcher(ERR_QUEST_REWARD_MONEY_S)) then OnEvent(worker, "CHAT_MSG_MONEY", format(YOU_LOOT_MONEY, msg), ...) return true end -- simulate gold loot event (and stop the message from showing up)
    if chatFrame == GetSelectedWindow() and msg:find(FormatToMatcher(ERR_QUEST_REWARD_EXP_I)) then return true end -- stop "Experience gained: %d." messages from showing
    local itemlink, byname = msg:match(FormatToMatcher(LOOT_DISENCHANT_CREDIT))
    --if chatFrame == GetSelectedWindow() and itemlink then return true end -- uncomment this line to hide all "X was disenchanted for loot by Y" messages, otherwise replaces item link with icon and keeps it as minimal as possible
    if chatFrame == GetSelectedWindow() and itemlink then return false, format(LOOT_DISENCHANT_CREDIT, GetIconWithLink(itemlink), byname), ... end
    return false, msg, ...
  end)
end

local GenerateChatReport = function()
  local str, lastname, temp = "", ""
  table.sort(addon.queue, function(a, b) return a < b end) -- asc. sort (but not on key, on value... needs to be fixed!)
  for event, entries in pairs(addon.queue) do
    -- reputation
    if event == "CHAT_MSG_COMBAT_FACTION_CHANGE" then
      for k, v in pairs(entries) do
        if v > 0 then
          str = str..format("|cff9696FF%s|r|cff%s+%d|r ", Abbrev(k), "00FF00", v)
        elseif v < 0 then
          str = str..format("|cff9696FF%s|r|cff%s%d|r ", Abbrev(k), "FF0000", v) -- removed "-" as prefix as the %d automatically adds it when negative value!
        end
      end
    -- honor
    elseif event == "CHAT_MSG_COMBAT_HONOR_GAIN" then
      temp = format("%.2f", select(2, next(entries)))
      if (tonumber(temp) or 0) > 0 then -- do not show when you gain 0 honor (or 0.001...)
        str = str..format("|cff9696FF%s: %s|r ", HONOR, temp)
      end
    -- experience
    elseif event == "CHAT_MSG_COMBAT_XP_GAIN" then
      str = str..format("|cffFFFF00%s: %d|r ", XP, select(2, next(entries)))
    -- currency
    elseif event == "CHAT_MSG_CURRENCY" then
      for plr, currencies in pairs(entries) do
        if lastname ~= plr then
          str = str..format("|cff%s%s|r|cffFFFF00:|r ", addon:ClassColor(plr), plr)
          lastname = plr
        end
        temp = 0
        if type(currencies) == "table" then
          for _, curr in pairs(currencies) do
            if type(curr) == "table" then
              curr[2] = tonumber(curr[2]) or 1 -- TODO: WIP
              if bool(MiniLootDB.lootCount) then
                str = str..format("%s%s%s ", GetIconWithLink(curr[1], 1), curr[2] > 1 and "x"..curr[2] or "", addon:GetCurrencyCount(curr[2], curr[1]))
              else
                str = str..format("%s%s ", GetIconWithLink(curr[1], 1), curr[2] > 1 and "x"..curr[2] or "")
              end
              temp = temp + 1
            else -- no table, thus a "_" entry with only a item and nothing else (NYI)
              if bool(MiniLootDB.lootCount) then
                str = str..format("%s%s ", GetIconWithLink(curr, 1), addon:GetCurrencyCount(1, curr))
              else
                str = str..format("%s ", GetIconWithLink(curr, 1))
              end
              temp = temp + 1
            end
          end
        end
        if temp == 0 then
          str = str..format("|cff999999(%s)|r ", L["no loot"])
        end
      end
    -- items (and rolls)
    elseif event == "CHAT_MSG_LOOT" then
      local sortedentries = {}
      for plr, items in pairs(entries) do
        table.insert(sortedentries, {plr, items})
      end
      table.sort(sortedentries, function(a, b) return a[1] < b[1] end)
      for i = 1, #sortedentries do
        local plr, items = unpack(sortedentries[i])
        if plr == rollIndex then -- special table containing roll data (safe because no player can use this as a name)
          if type(items) == "table" then
            str = NewLine(str) -- before rolls start we add a new line to separate them
            for item, rolldata in pairs(items) do
              if bool(MiniLootDB.showRollIcons) then
                item = GetIconWithLink(item)
              end
              str = str..format("|cff00AA00%s|r %s|cff00AA00:|r ", L["Roll"], item) -- BUG: "Wurf un" but this string is used by options for the word "Roll"
              local hasNeed, hasBonus, hadBonus -- "hasNeed" kind of obscolete I reckon... but I am so lazy right now!
              local sortedrolls = {}
              for plr, roll in pairs(rolldata) do
                if roll[1] == 1 then hasNeed = 1 end
                roll[3], hadBonus = stripNameRollBonus(plr) -- use this to store playername (strip away the " + Roll Bonus" junk in LFR)
                if hadBonus then hasBonus = 1 end
                table.insert(sortedrolls, roll)
              end
              table.sort(sortedrolls, function(a, b) return tonumber(a[2]) > tonumber(b[2]) end)
              local color, suffix
              for _, roll in pairs(sortedrolls) do
                if roll[3] == UnitName("player") or ((not hasNeed) or (hasNeed and roll[1] == 1)) and ((not hasBonus) or (hasBonus and roll[2] > 100)) then -- always show our own rolls, or if the filtering is okay with it...
                  if roll[1] == 1 then
                    color = "33EE33"
                    suffix = L["N"] -- short for "Need"
                  elseif roll[1] == 2 then
                    color = "CCCC33"
                    suffix = L["G"] -- short for "Greed"
                  elseif roll[1] == 3 then
                    color = "EE3333"
                    suffix = L["D"] -- short for "Dissenchant"
                  else
                    color = "CCCCCC"
                    suffix = L["P"] -- short for "Pass"
                  end
                  if bool(MiniLootDB.classColors) then
                    str = str..format("|cff%s%s|r|cff%s:%s:%d|r ", addon:ClassColor(roll[3]), roll[3], color, suffix, roll[2])
                  else
                    str = str..format("|cff%s%s:%s:%d|r ", color, roll[3], suffix, roll[2])
                  end
                end
              end
              str = NewLine(str) -- each roll has it's own line
            end
          end
        else -- handle item data since it's not a roll entry...
          if lastname ~= plr then
            str = str..format("|cff%s%s|r|cffFFFF00:|r ", addon:ClassColor(plr), plr)
            lastname = plr
          end
          temp = 0
          if type(items) == "table" then
            for _, item in pairs(items) do
              if type(item) == "table" then
                item[2] = tonumber(item[2]) or 1 -- TODO: WIP
                if plr == UnitName("player") and bool(MiniLootDB.lootCount) then
                  str = str..format("%s%s%s ", GetIconWithLink(item[1]), item[2] > 1 and "x"..item[2] or "", addon:GetItemCount(item[2], item[1], true, true))
                else
                  str = str..format("%s%s ", GetIconWithLink(item[1]), item[2] > 1 and "x"..item[2] or "")
                end
              else -- no table, thus a "_" entry with only a item and nothing else (TODO: WIP)
                if plr == UnitName("player") and bool(MiniLootDB.lootCount) then
                  str = str..format("%s%s ", GetIconWithLink(item), addon:GetItemCount(1, item, true, true))
                else
                  str = str..format("%s ", GetIconWithLink(item))
                end
              end
              temp = temp + 1
            end
          end
          if temp == 0 then
            str = str..format("|cff999999(%s)|r ", L["no loot"])
          end
        end
      end
    -- gold
    elseif event == "CHAT_MSG_MONEY" then
      for plr, copper in pairs(entries) do
        if tonumber(copper or 0) > 0 then -- weird bug, causes "0 copper" to be printed sometimes!
          if lastname ~= plr then
            str = str..format("|cff%s%s|r|cffFFFF00:|r ", addon:ClassColor(plr), plr)
            lastname = plr
          end
          str = str..format("|cffFFFF00%s|r ", GetCoinTextureString(copper))
        end
      end
    end
  end
  if str:len() > 3 then -- only output the string if it contains something proper, tough it should never contain only 1 or 2 chars...
    local lines = {("\n"):split(str)} -- instead of having many lines in 1 message, split them up (icons bug otherwise!)
    for _, line in pairs(lines) do
      line = line:trim() -- trim spacing on the left and right edges
      if line:len() > 3 then -- avoid empty strings like empty line at the end...
        print(line) -- output the summary
      end
    end
  end
end

local OnUpdate = function(worker, elapsed)
  if InCombatLockdown() and MiniLootDB.sleepAfterCombat ~= -1 then return end -- wait for combat to end (ignore if sleep is -1)
  if (not addon.lastevent) then return end -- nothing to report? wait...
  worker.elapsed = (worker.elapsed or 0) + elapsed -- keep counting now!
  if GetTime()-addon.lastevent < MiniLootDB.sleepBetweenEvents then return end -- wait for last event timer to expire
  if worker.elapsed >= MiniLootDB.sleepAfterCombat then
    GenerateChatReport()
    table.wipe(addon.queue)
    worker.elapsed = 0
    addon.lastevent = nil
  end
end

addon.init = function(addon, force)
  if not force and (type(addon.loaded) ~= "table" or #addon.loaded < 2) then return end -- already init.
  addon.loaded = nil
  local reset = addon:variablecheck()
  if reset ~= 0 then
    _print(format(L["Note that %s had to reset %d setting(s) due to various reasons. This may happen when you update the addon, in that case please check the addon configurations as some may have been reset to default value."], addon.name2, reset))
  end
  addon.events.LOOT_OPENED = function(worker, event) addon.lastevent = GetTime() end
  addon.events.LOOT_CLOSED = function(worker, event) addon.lastevent = GetTime() end
  worker:RegisterEvent("LOOT_OPENED")
  worker:RegisterEvent("LOOT_CLOSED")
  ConvertMatches()
  HookChatEvents()
  worker:SetScript("OnUpdate", OnUpdate)
  addon:loadCfgUI()
  do
    local orig1 = ItemRefTooltip.SetHyperlink -- we have to hook this function since the default ChatFrame code assumes that all links except for player and channel links are valid arguments for this function
    function ItemRefTooltip:SetHyperlink(link, ...)
      if link:match("^"..addon.name2) then return end
      return orig1(self, link, ...)
    end
    local orig2 = SetItemRef -- we have to hook this function to make currency from the chat linkable
    function SetItemRef(link, text, button, chatFrame, ...)
      local curID = tonumber(link:match("currency:(%d+)"))
      if curID and curID > 0 and IsModifiedClick() then
        text = GetCurrencyLink(curID) or text
      end
      return orig2(link, text, button, chatFrame, ...)
    end
  end
end

addon.events.ADDON_LOADED = function(worker, event, name)
  if addon.name == name then
    worker:UnregisterEvent("ADDON_LOADED")
    if type(addon.loaded) == "table" then table.insert(addon.loaded, event) end
  end
  addon:init()
end

addon.events.CURRENCY_DISPLAY_UPDATE = function(worker, event, ...)
  CurrencyCaches()
  ArchaeologyCaches() -- run last as this appends to the currency cache
end

addon.events.VARIABLES_LOADED = function(worker, event)
  worker:UnregisterEvent("VARIABLES_LOADED")
  if type(addon.loaded) == "table" then table.insert(addon.loaded, event) end
  addon:init()
end

worker:SetScript("OnEvent", function(worker, event, ...)
  addon.events[event](worker, event, ...)
end)

for k, _ in pairs(addon.events) do
  worker:RegisterEvent(k)
end

if addon.name ~= addon.name2 then
  addon:init(1) -- force init() in case addon is loaded as a child-addon
end
