local Stack = {}
local libBaggage = LibStub:GetLibrary("LibBaggage-1.0")

-- Stack Objects --
local GetStack, DepositStack = Luggage:GetCompost()
local stackCache = setmetatable({}, {__mode='k'})

function Stack:New(item)
   local newStack = GetStack(item)
   if newStack then 
      newStack:SetItem(item)
      return newStack 
   end
   newStack = {
     ["itemID"] = item.itemID,
     ["currentSlot"] = 0,
     ["slots"] = {},
     ["stackCount"] = 0,
     ["icon"] = item.icon
   }
   setmetatable(newStack, {__index = self})
   newStack:AddItemStack(item)
   return newStack
end

function Stack:Release()
   self:SetItem(nil)
   self.stackCount = 0
   DepositStack(self)
end

function Stack:SetItem(item)
   if not item then
      self.currentSlot = 0
      for bagID, bag in pairs(self.slots) do
         for slotID, locked in pairs(bag) do
            bag[slotID] = nil
         end
      end
      return
   end
   self.itemID = item.itemID
   self.icon = item.icon
   self:AddItemStack(item)
end

function Stack:AddItemStack(item)
   local bagID, slotID = item.bagID, item.slotID
   local slots = self.slots
   if slots[bagID] and type(slots[bagID][slotID]) ~= "nil" then
      return
   end
   self.stackCount = self.stackCount + item.stackCount
   self:AddSlot(item.bagID, item.slotID)
end

function Stack:RemoveItemStack(item)
   self.stackCount = self.stackCount - item.stackCount
   self:RemoveSlot(item.bagID, item.slotID)
   return self.stackCount ~= 0
end

function Stack:AddSlot(bagID, slotID)
   local slots = self.slots
   if not slots[bagID] then
      slots[bagID] = {}
   end
   local item = libBaggage:GetItem(bagID, slotID)
   slots[bagID][slotID] = item.locked or false
end

function Stack:RemoveSlot(bagID, slotID)
   local bag = self.slots[bagID]
   if not bag then return end
   bag[slotID] = nil
end

function Stack:GetNextSlot()
   local maxStackCount = 0
   local maxBagID, maxSlotID
   local allLocked
   for bagID, bag in pairs(self.slots) do
      for slotID in pairs(bag) do
         local item = libBaggage:GetItem(bagID, slotID)
         if item then
            if not (maxBagID and maxSlotID) then
               maxBagID, maxSlotID = bagID, slotID
               if item.locked then
                  allLocked = true
               else
                  maxStackCount = item.stackCount
               end
            elseif not item.locked and item.stackCount >= maxStackCount then
               maxBagID, maxSlotID = bagID, slotID
               maxStackCount = item.stackCount
               allLocked = false
            end
         end
      end
   end
   if maxBagID then
      return maxBagID, maxSlotID, allLocked
   end
end

function Stack:GetItemTable()
   local item = self.item
   if not item then
      self.item = {}
      item = self.item
   end
   local bagID, slotID, locked = self:GetNextSlot()
   item.bagID = bagID
   item.slotID = slotID
   item.icon = self.icon
   local libItem = libBaggage:GetItem(bagID, slotID)
   item.name = libItem.name
   item.link = libItem.link
   item.stackCount = self.stackCount
   item.locked = locked
   item.itemID = self.itemID
   return item
end

Luggage.Stack = Stack