local Bag = {}
local libBaggage = LibStub:GetLibrary("LibBaggage-1.0")
local L = LibStub("AceLocale-3.0"):GetLocale("Luggage")

-- Bag cache
local GetBag, DepositBag = Luggage:GetCompost()
local GetBagButton, DepositBagButton = Luggage:GetCompost()
local GetBagFrame, DepositBagFrame = Luggage:GetCompost()
local GetBagTitleFrame, DepositBagTitleFrame = Luggage:GetCompost()

function Bag:New(savedVars)
   local newBag = GetBag()
   if not newBag then
      newBag = {
        itemStacks = {},
        buttons = {},
        containedItems = {},
      }
      setmetatable(newBag, {__index = self})
   end
   newBag.filters = savedVars.filters
   newBag.savedVars = savedVars
   newBag:CreateBagButton()
   newBag:CreateBagFrame()
   newBag:CreateTitleFrame()
   newBag:SetSavedVariables(savedVars)
   return newBag
end

function Bag:Release()
   Luggage:UnregisterBagForEvents(self)
   self:TrimItemButtons(0)
   -- Bag frame
   local frame = self.frame
   frame:Hide()
   DepositBagFrame(frame)
   self.frame = nil
   -- Bag button
   local bagButton = self.bagButton
   bagButton:Hide()
   DepositBagButton(bagButton)
   self.bagButton = nil
   -- Title frame
   local titleFrame = self.titleFrame
   titleFrame:SetParent(UIParent)
   titleFrame:Hide()
   DepositBagTitleFrame(titleFrame)
   self.titleFrame = nil
   -- Self
   self.savedVars = nil
   self.options = nil
   DepositBag(self)
end

local bagButtonScripts = {
   "OnClick",
   "OnEnter",
   "OnLeave",
}

local nBagButtons = 0
function Bag:CreateBagButton()
   local bagButton = GetBagButton()
   if not bagButton then
      bagButton = CreateFrame("CheckButton", "Luggage_BagButton"..nBagButtons, UIParent, "Luggage_BagSlotButtonTemplate")
      nBagButtons = nBagButtons + 1
      bagButton.countText = getglobal(bagButton:GetName().."Count")
      bagButton.stockText = getglobal(bagButton:GetName().."Stock")
      bagButton.iconTexture = getglobal(bagButton:GetName().."IconTexture")
      for i, scriptName in ipairs(bagButtonScripts) do
         bagButton:SetScript(scriptName, Luggage.Bag[scriptName])
      end
      bagButton:RegisterForClicks("LeftButtonUp", "RightButtonUp")
   end
   bagButton.bag = self
   self.bagButton = bagButton
   bagButton:ClearAllPoints()
end

local nBagFrames = 0
function Bag:CreateBagFrame()
   local frame = GetBagFrame()
   if not frame then
      frame = CreateFrame("Frame", "Luggage_BagFrame"..nBagFrames, UIParent)
      table.insert(UISpecialFrames, frame:GetName())
      nBagFrames = nBagFrames + 1
      frame:SetBackdrop({
         bgFile   = "Interface/Tooltips/UI-Tooltip-Background", 
         edgeFile = "Interface/Tooltips/UI-Tooltip-Border",
         tile     = true, 
         tileSize = 16, 
         edgeSize = 16, 
         insets   = { left = 4, right = 4, top = 4, bottom = 4 }
      })
      frame:EnableMouse(true)
      frame:SetMovable(true)
      frame:SetClampedToScreen(true)
      frame:SetFrameStrata("HIGH")
   end
   frame:SetBackdropColor(self:GetColor())
   frame:SetScript("OnShow",
      function(this)
         self:Show()
      end
   )
   frame:SetScript("OnMouseUp",
      function(this, mouseButton)
         self:SavePosition(this)
      end
   )
   frame:SetScript("OnMouseDown",
      function(this, mouseButton)
         if mouseButton == "LeftButton" and not this.isMoving and self:IsUserPlaced() then
            this:StartMoving()
            this.isMoving = true
         end
      end
   )
   frame:SetScript( "OnHide",
      function(this)
         self:SavePosition(this)
         self:Hide()
      end
   )
   self.frame = frame
end

function Bag:SetSavedVariables(savedVars)
   self.savedVars = savedVars
   self:SetIcon(savedVars.icon)
   self:SetName(savedVars.name)
end

function Bag:SavePosition(this)
   if this.isMoving then
      this:StopMovingOrSizing()
      this.isMoving = false
      local sv = self.savedVars
      local point, relativeTo, relativePoint, xOffs, yOffs = self.frame:GetPoint(1)
      sv.point = point or "BOTTOMLEFT"
      sv.x = xOffs or frame:GetLeft()
      sv.y = yOffs or frame:GetBottom()
   end
end

local BAG_TITLE_HEIGHT = 30
function Bag:CreateTitleFrame()
   local frame = self.frame
   local titleFrame = GetBagTitleFrame()
   if not titleFrame then
      -- Frame
      titleFrame = CreateFrame("Frame", frame:GetName().."_Title", UIParent)
      titleFrame:SetHeight(BAG_TITLE_HEIGHT)
      titleFrame:EnableMouse(true)
      -- Close Button
      local closeButton = CreateFrame("Button", titleFrame:GetName().."CloseButton", titleFrame, "UIPanelCloseButton")
      titleFrame.closeButton = closeButton
      closeButton:SetPoint("RIGHT", titleFrame, "RIGHT")
      -- Text
      local fontString = titleFrame:CreateFontString(titleFrame:GetName().."Text", nil, "GameFontNormal")
      titleFrame.text = fontString
      fontString:SetPoint("LEFT", titleFrame, "LEFT", 7, 0)
      fontString:SetPoint("RIGHT", titleFrame, "RIGHT", - closeButton:GetWidth(), 0)
      fontString:SetJustifyH("LEFT")
   end
   titleFrame:SetParent(frame)
   titleFrame:SetPoint("TOPLEFT", frame, "TOPLEFT", 0, 0)
   titleFrame:SetPoint("TOPRIGHT", frame, "TOPRIGHT", 0, 0)
   titleFrame:Show()
   titleFrame.closeButton:SetScript("OnClick", function() self:Hide() end)
   self.closeButton = titleFrame.closeButton
   self.titleText = titleFrame.text
   titleFrame.text:SetText(self.savedVars.name)
   titleFrame:SetScript( "OnMouseUp",
      function(this, mouseButton)
         if mouseButton == "LeftButton" then
            self:SavePosition(frame)
         elseif mouseButton == "RightButton" then
            self:ShowTitleDropdown()
         end
      end
   )
   titleFrame:SetScript( "OnMouseDown",
      function(this, mouseButton)
         if mouseButton == "LeftButton" and not frame.isMoving and self:IsUserPlaced() then
            frame:StartMoving()
            frame.isMoving = true
         end
      end
   )
   titleFrame:SetScript( "OnHide",
      function(this)
         self:SavePosition(frame)
      end
   )
   
   self.titleFrame = titleFrame
end

function Bag.OnClick(frame, mouseButton)
   local self = frame.bag
   if mouseButton == "LeftButton" then
      if IsModifierKeyDown("shift") then
         local infoType, itemID = GetCursorInfo()
         if infoType == "item" then
            local nextState = Luggage:GetNextState(self.savedVars.itemIDs[itemID])
            local str
            if nextState == true then
               str = "true"
            elseif nextState == false then
               str = "false"
            elseif nextState == nil then
               str = "nil"
            end
            self:SetFilterForItem(itemID, nextState)
            Bag.OnEnter(frame)
         end
      else
         if self:IsShown() then
            self:Hide()
         else
            self:Show(true)
         end
      end
   elseif mouseButton == "RightButton" then
      self.bagButton:SetChecked(not self.bagButton:GetChecked()) -- Looks weird, but leads to not changing the checked status
      self:ShowDropdown()
   end
end

function Bag.OnEnter(frame)
   local self = frame.bag
   GameTooltip:SetOwner(frame, "ANCHOR_LEFT");
   if CursorHasItem() then
      local infoType, itemID = GetCursorInfo()
      local r,g,b = Luggage:GetFilterColor(self.savedVars.itemIDs[itemID])
      GameTooltip:SetText(self.savedVars.name, r,g,b)
      GameTooltip:AddLine(L["Shift-Click to cycle through the filter modes for this particular item"], NORMAL_FONT_COLOR.r, NORMAL_FONT_COLOR.g, NORMAL_FONT_COLOR.b)
   else
      GameTooltip:SetText(self.savedVars.name)
   end
   GameTooltip:Show()
end

function Bag.OnLeave(frame)
   local self = frame.bag
   GameTooltip:Hide()
   ResetCursor()
end

function Bag:ShowDropdown()
   Luggage:ShowBagDropdown(self)
end

function Bag:ShowTitleDropdown()
   if not self.titleDropdownMenu then
      self.titleDropdownMenu = {
         {
            text = L["Place Manually"],
            func = function(value)
               self.savedVars.isUserPlaced = not self.savedVars.isUserPlaced
               self:UpdatePosition()
               Luggage:ArrangeBags()
            end,
            checked = function() return self:IsUserPlaced() end,
            keepShownOnClick = true,
         }
      }
      self.titleDropdownFrame = CreateFrame("Frame", self.savedVars.name.."TitleDropdownFrame", UIParent, "UIDropDownMenuTemplate")
   end
   EasyMenu(self.titleDropdownMenu, self.titleDropdownFrame, self.titleFrame, nil, nil, "MENU")
end

function Bag:ToggleCombineStacks()
   local combineStacks = self.savedVars.combineStacks
   self:SetCombineStacks(not combineStacks)
end

function Bag:ToggleEmptySlot()
   local showEmptySlot = self.savedVars.showEmptySlot
   self:SetShowEmptySlot(not showEmptySlot)
end

function Bag:ToggleShowNumberOfContainedItems()
   self.savedVars.showNumberOfContainedItems = not self.savedVars.showNumberOfContainedItems
   self:UpdateBagSpace()
end

function Bag:ToggleShowFreeSpace(bagType)
   self.savedVars.showFreeSpace = bit.bxor(self.savedVars.showFreeSpace or 0, bagType)
   self:UpdateBagSpace()
end

function Bag:SetShowEmptySlot(showEmptySlot)
   local svShowEmptySlot = self.savedVars.showEmptySlot
   if (not showEmptySlot and not svShowEmptySlot) or (showEmptySlot and svShowEmptySlot) then
      return
   end
   self:Clear()
   self.savedVars.showEmptySlot = showEmptySlot
   self:Update()
end

function Bag:UpdateEmptySlot()
   if not self:IsShown() then return end
   if not self.savedVars.showEmptySlot then return end
   if #self.buttons == 0 then return end
   local emptyButton = self.buttons[#self.buttons]
   local emptyStack = Luggage:GetEmptySlotTable(self)
   emptyButton:SetItem(emptyStack)
end

local normalTextureScale = 64/37
function Bag:SetBagButtonSize(size)
   local bagButton = self.bagButton
   local normalTexture = getglobal(bagButton:GetName().."NormalTexture")
   bagButton:SetHeight(size)
   bagButton:SetWidth(size)
   normalTexture:SetHeight(size * normalTextureScale)
   normalTexture:SetWidth(size * normalTextureScale)
end

function Bag:SetCombineStacks(combineStacks)
   local savedCombineStacks = self.savedVars.combineStacks
   if (not savedCombineStacks and not combineStacks) or (savedCombineStacks and combineStacks) then
      return
   end
   self:Clear()
   self.savedVars.combineStacks = combineStacks
   self:Update()
end

function Bag:SetIcon(icon)
   self.savedVars.icon = icon
   self.bagButton.iconTexture:SetTexture(icon)
   self.bagButton:Show()
end

function Bag:SetName(name)
   self.savedVars.name = name
   self.titleText:SetText(name)
end

function Bag:SetFilterForItem(itemID, value)
   self.savedVars.itemIDs[itemID] = value
   Luggage:FullUpdate()
end

function Bag:SetShowAt(key, value)
   self.savedVars[key] = value
end

function Bag:GetShowAt(key)
   return self.savedVars[key]
end

function Bag:Update()
   for item in libBaggage:AllItems() do
      if self:ContainsItem(item) then
         self:AddItem(item)
      end
   end
   self:UpdateItemButtons()
end

function Bag:SortItems()
   local buttons
   if self.savedVars.combineStacks then
      buttons = self.itemStacks
   else
      buttons = self.itemButtons
   end
end

function Bag:GetItemList()
   local list = self.itemList
   if list then
      for i, item in pairs(list) do
         list[i] = nil
      end
   else
      self.itemList = {}
      list = self.itemList
   end
   local n = 1
   if self.savedVars.combineStacks then
      for itemID, stack in pairs(self.itemStacks) do
         list[n] = stack:GetItemTable()
         n = n + 1
      end
   else
      for item in pairs(self.containedItems) do
         list[n] = item
         n = n + 1
      end
   end
   if self.savedVars.showEmptySlot then
      list[n] = Luggage:GetEmptySlotTable(self)
      n = n + 1
   end
   list = self:SortItemList(list)
   return list
end

local function sortByName(item1, item2)
   local name1 = item1.name
   local name2 = item2.name
   if not name1 then 
      return false
   end
   if not name2 then
      return true
   end
   return name1 < name2
end

function Bag:SortItemList(list)
   table.sort(list, sortByName)
   return list
end

function Bag:TrimItemButtons(count)
   local buttons = self.buttons
   while count < #buttons do
      local itemButton = table.remove(buttons)
      itemButton:Release()
   end
   local prototype = Luggage.ItemButton
   while count > #buttons do
      table.insert(buttons, prototype:New())
   end
end

function Bag:UpdateItemButtons()
   if not self:IsShown() then return end
   local itemList = self:GetItemList()
   self:TrimItemButtons(#itemList)
   for i, button in ipairs(self.buttons) do
      button:SetItem(itemList[i])
   end
   self:UpdatePosition() 
end

function Bag:UpdateItemLock(bagID, slotID)
   if not self:IsShown() then return end
   local item = libBaggage:GetItem(bagID, slotID)
   if self.savedVars.combineStacks then
      local stack = self.itemStacks[item.itemID]
      if not stack then return end
      local new_item = stack:GetItemTable()
      for i, button in ipairs(self.buttons) do
         if button.itemID == item.itemID then
            button:SetItem(new_item)
            break
         end
      end
   else
      for i, button in ipairs(self.buttons) do
         if button.bagID == bagID and button.slotID == slotID then
            button:SetItem(item)
         end
      end
   end
   
end

local MIN_HEIGHT = 50
local MIN_WIDTH = 50
function Bag:UpdatePosition()
   local frame = self.frame
   frame:ClearAllPoints()
   frame:SetPoint(self.savedVars.point, UIParent, self.savedVars.point, self.savedVars.x, self.savedVars.y)
   local columns = math.min(self:GetNumberOfColumns(), #self.buttons)
   if columns == 0 then columns = 1 end
   local itemSize = self.savedVars.itemSize
   local inset = self.savedVars.inset
   local padding = self.savedVars.padding
   local width = math.max(inset + columns*(itemSize + padding) - padding + inset, self.titleText:GetStringWidth() + self.closeButton:GetWidth() + 10)
   local rows = math.ceil(#self.buttons / columns)
   local height = math.max(inset + rows*(itemSize + padding) - padding + BAG_TITLE_HEIGHT, MIN_HEIGHT)
   frame:SetHeight(height)
   frame:SetWidth(width)   
   for i, button in ipairs(self.buttons) do
      button.frame:ClearAllPoints()
      local column = (i-1) % columns
      local row = math.floor((i-1)/columns)
      local x,y = self:GetButtonOffset(row, column, width, height)
      button.frame:SetPoint("TOPLEFT", frame, "TOPLEFT", x, y)
      i = i + 1
      button:SetSize(itemSize)
   end
   if not self:IsUserPlaced() then
      Luggage:ArrangeBags()
   end
end

function Bag:Locations()
   local locationFilter = self.savedVars.filters.Location
   local bankFilter
   local mainBagFilter
   if locationFilter then
      bankFilter = locationFilter[1]
      mainBagFilter = locationFilter[2]
   end
   local isBank = bankFilter or bankFilter == nil
   local isMainBags = mainBagFilter or mainBagFilter == nil
   return isMainBags, isBank
end

function Bag:UpdateBagSpace()
   local freeSpace
   local isMainBags, isBank = self:Locations()
   local bagType = self.savedVars.showFreeSpace
   local count = getglobal(self.bagButton:GetName().."Count")
   if bagType ~= 0 then
      if isBank then
         freeSpace = (freeSpace or 0) + libBaggage:GetTotalFreeBankSpace(bagType, true)
      end
      if isMainBags then
         freeSpace = (freeSpace or 0) + libBaggage:GetTotalFreeMainBagSpace(bagType, true)
      end
      count:SetText(freeSpace)
      count:Show()
   else
      count:Hide()
   end
   local stock = getglobal(self.bagButton:GetName().."Stock")
   if self.savedVars.showNumberOfContainedItems then
      local nItems = 0
      for item in pairs(self.containedItems) do
         nItems = nItems + 1
      end
      stock:SetText(nItems)
      stock:Show()
   else
      stock:Hide()
   end
end

function Bag:GetNumberOfColumns()
   return self.savedVars.columns
end

function Bag:GetButtonOffset(row, column, width, height)
   local itemSize = self.savedVars.itemSize
   local padding = self.savedVars.padding
   local inset = self.savedVars.inset
   local x,y
   if self.savedVars.growUpwards then
      y = row*(itemSize + padding) - height + itemSize + inset - BAG_TITLE_HEIGHT
   else
      y = -( row*(itemSize + padding)) - BAG_TITLE_HEIGHT
   end
   if self.savedVars.growLeft then
      x = -inset - column*(itemSize+padding) + width - itemSize
   else
      x = inset + column*(itemSize+padding)
   end
   return x,y
end

function Bag:ContainsItem(itemTable, excludeFilters)
   local sv = self.savedVars
   local byItemID = sv.itemIDs[itemTable.itemID]
   if byItemID ~= nil then
      return byItemID 
   end
   local contains = false
   local filters = sv.filters
   for id, values in pairs(filters) do
      if Luggage.filters[id] and (not excludeFilters or not excludeFilters[id]) then
         local filterFunc = Luggage.filters[id].func
         if type(values) == "boolean" then
            if values then
               contains = contains or filterFunc(self, itemTable)
            else
               if filterFunc(self, itemTable) then
                  return false
               end
            end
         elseif type(values) == "table" then
            for value, isContained in pairs(values) do
               if isContained == true then
                  contains = contains or filterFunc(self, itemTable, value)
               elseif isContained == false then
                  if filterFunc(self, itemTable, value) then
                     return false
                  end
               end            
            end
         end
      end
   end
   return contains
end

function Bag:AddItem(itemTable)
   self.containedItems[itemTable] = true
   if self.savedVars.combineStacks then
      local itemID = itemTable.itemID
      local stack = self.itemStacks[itemID] 
      if not stack then 
         self.itemStacks[itemID] = Luggage.Stack:New(itemTable)
      else
         stack:AddItemStack(itemTable)
      end
   end
   if(not self.savedVars.hideAnimation) then
      self:PlayAnim(itemTable.icon)
   end
end

function Bag:PlayAnim(icon)
   local anim = getglobal(self.bagButton:GetName().."ItemAnim")
   if not icon then
      anim:Hide()
      return
   end
   anim:ReplaceIconTexture(icon);
   anim:SetSequence(0);
   anim:SetSequenceTime(0, 0);
   anim:Show();
end


function Bag:RemoveItem(itemTable)
   local containedItems = self.containedItems
   if not containedItems[itemTable] then return end
   containedItems[itemTable] = nil
   if self.savedVars.combineStacks then
      local itemID = itemTable.itemID
      local stack = self.itemStacks[itemID]
      if not stack then return end
      local stackShouldRemain = stack:RemoveItemStack(itemTable)
      if stackShouldRemain then
         self:UpdateSingleItemButton(itemTable.bagID, itemTable.slotID, stack:GetItemTable())
      else
         self:HideItemButton(itemTable.bagID, itemTable.slotID)
         self.itemStacks[itemID] = nil
         stack:Release()
      end
   else
      self:HideItemButton(itemTable.bagID, itemTable.slotID)
   end
end

function Bag:Clear()
   local containedItems = self.containedItems
   for item in pairs(containedItems) do
      containedItems[item] = nil
   end
   local itemStacks = self.itemStacks
   for itemID, stack in pairs(self.itemStacks) do
      -- local stack = table.remove(itemStacks, itemID)
      stack:Release()
      self.itemStacks[itemID] = nil
   end
   local itemList = self.itemList
   if itemList then
      for i, item in ipairs(itemList) do
         itemList[i] = nil
      end
   end
end

function Bag:HideItemButton(bagID, slotID)
   if type(bagID) == "string" and bagID == "bank" then bagID = -1 end
   for i, button in ipairs(self.buttons) do
      if button.bagID == bagID and button.slotID == slotID then
         button:SetItem(nil)
         return
      end
   end
end

function Bag:UpdateSingleItemButton(oldBagID, oldSlotID, new_item)
   if not self:IsShown() then return end
   local combineStacks = self.savedVars.combineStacks
   for i, button in ipairs(self.buttons) do
      if combineStacks then
         if button.itemID == new_item.itemID then
            button:SetItem(new_item)
         end
      else
         if button.bagID == oldBagID and button.slotID == oldSlotID then
            button:SetItem(new_item)
            return
         end
      end
   end
end

function Bag:Show(force)
   local itemList = self:GetItemList()
   if #itemList == 0 and self.savedVars.hideWhenEmpty and not force then
      return
   end
   self.show = true
   self.frame:Show()
   self.bagButton:SetChecked(1)
   self:UpdateItemButtons()
end

function Bag:Hide()
   self.show = nil
   self.frame:Hide()
   self.bagButton:SetChecked(0)
   for i, button in ipairs(self.buttons) do
      button.frame:Hide()
   end
   if not self:IsUserPlaced() then
      Luggage:ArrangeBags()
   end
--   self:TrimItemButtons(0) -- I think this line is doing more harm than good
end

function Bag:IsShown()
   return self.show
end

function Bag:SetItemButtonSize(size)
   self.savedVars.itemSize = size
   self:UpdatePosition()
end

function Bag:GetItemButtonSize()
   return self.savedVars.itemSize
end

function Bag:SetInset(inset)
   self.savedVars.inset = inset
   self:UpdatePosition()
end

function Bag:GetInset()
   return self.savedVars.inset
end

function Bag:SetPadding(padding)
   self.savedVars.padding = padding
   self:UpdatePosition()
end

function Bag:GetPadding()
   return self.savedVars.padding
end

function Bag:SetColumns(nr)
   self.savedVars.columns = nr
   self:UpdatePosition()
end

function Bag:GetColumns()
   return self.savedVars.columns
end

function Bag:SetGrowUpwards(growUpwards)
   self.savedVars.growUpwards = growUpwards
   self:UpdatePosition()
end

function Bag:GetGrowUpwards()
   return self.savedVars.growUpwards
end

function Bag:SetGrowLeft(growLeft)
   self.savedVars.growLeft = growLeft
   self:UpdatePosition()
end

function Bag:GetGrowLeft()
   return self.savedVars.growLeft
end

function Bag:SetColor(r,g,b,a)
   local color = self.savedVars.color
   color.r = r
   color.g = g
   color.b = b
   color.a = a
   self.frame:SetBackdropColor(r, g, b, a)
end

function Bag:GetColor()
   local color = self.savedVars.color
   return color.r, color.g, color.b, color.a
end

function Bag:IsUserPlaced()
   return self.savedVars.isUserPlaced
end

function Bag:GetHeight()
   return self.frame:GetHeight() / self.frame:GetEffectiveScale()
end

function Bag:GetWidth()
   return self.frame:GetWidth() / self.frame:GetEffectiveScale()
end

function Bag:SetPoint(point, frame, relativePoint, xOffs, yOffs)
   if string.sub(point, 1, 3) == "TOP" then
      yOffs = yOffs - self.titleFrame:GetHeight()
   end
   self.frame:SetPoint(point, frame, relativePoint, xOffs, yOffs)
end

function Bag:ClearAllPoints()
   self.frame:ClearAllPoints()
end

function Bag:SetHideWhenEmpty(value)
   self.savedVars.hideWhenEmpty = value
end

function Bag:GetHideWhenEmpty()
   return self.savedVars.hideWhenEmpty
end

function Bag:SetHideAnimation(value)
   self.savedVars.hideAnimation = value
end

function Bag:GetHideAnimation()
   return self.savedVars.hideAnimation
end

local optionsTemplate = {
   type = "group",
   args = {
      layoutGroup = {
         type = "group",
         name = L["Layout"],
         inline = true,
         order = 10,
         args = {
            itemSize = {
               type = "range",
               name = L["Item Button Size"],
               desc = L["Size of the items inside the bag"],
               order = 20,
               min = 20,
               max = 50,
               step = 1,
               handler = optionsBag,
               set = function(info, value) info.handler:SetItemButtonSize(value) end,
               get = function(info) return info.handler:GetItemButtonSize() end,
            },
            inset = {
               type = "range",
               name = L["Item Button Inset"],
               desc = L["The distance between the item buttons and the edge of the bag frame."],
               order = 30,
               min = 0,
               max = 20,
               step = 1,
               handler = optionsBag,
               set = function(info, value) info.handler:SetInset(value) end,
               get = function(info) return info.handler:GetInset() end,
            },
            padding = {
               type = "range",
               name = L["Item Button Padding"],
               desc = L["The distance between two item buttons in this bag."],
               order = 40,
               min = 0,
               max = 20,
               step = 1,
               handler = optionsBag,
               set = function(info, value) info.handler:SetPadding(value) end,
               get = function(info) return info.handler:GetPadding() end,
            },
            columns = {
               type = "range",
               name = L["Number of columns"],
               desc = L["The maximum number of item buttons in one row."],
               order = 50,
               min = 1,
               max = 20,
               step = 1,
               handler = optionsBag,
               set = function(info, value) info.handler:SetColumns(value) end,
               get = function(info) return info.handler:GetColumns() end,
            },
            growUpwards = {
               type = "toggle",
               name = L["Expand upwards"],
               desc = L["Begin placing item buttons to the bottom, then stack them up as more items are added."],
               order = 60,
               handler = optionsBag,
               set = function(info, value) info.handler:SetGrowUpwards(value) end,
               get = function(info) return info.handler:GetGrowUpwards() end,
            },
            growLeft = {
               type = "toggle",
               name = L["Expand to the left"],
               desc = L["Begin placing item buttons to the right, then expand to the left as more items are added."],
               order = 70,
               handler = optionsBag,
               set = function(info, value) info.handler:SetGrowLeft(value) end,
               get = function(info) return info.handler:GetGrowLeft() end,
            },
            color = {
               type = "color",
               name = L["Color"],
               desc = L["Set the color of the bag's background."],
               order = 80,
               handler = optionsBag,
               set = function(info, r,g,b,a) info.handler:SetColor(r, g, b, a) end,
               get = function(info) return info.handler:GetColor() end,
            },
         },
      },
      showAt = {
         type = "multiselect",
         name = L["Open:"],
         desc = L["Set when this bag should be automatically opened."],
         order = 20,
         values = {
            showAtBank = L["At Bank"],
            showAtMailbox = L["At Mailbox"],
            showAtVendor = L["At Vendor"],
            showAtAuctionHouse = L["At Auction House"],
            showAtGuildbank = L["At Guildbank"],
            showOnToggleAll = L["On Toggle All Bags"],
            showOnToggleSet1 = L["On Toggle Set %d"]:format(1),
            showOnToggleSet2 = L["On Toggle Set %d"]:format(2),
            showOnToggleSet3 = L["On Toggle Set %d"]:format(3),
            showOnToggleSet4 = L["On Toggle Set %d"]:format(4),
            showOnToggleSet5 = L["On Toggle Set %d"]:format(5),
         },
         handler = optionsBag,
         set = function(info, key, value) info.handler:SetShowAt(key, value) end,
         get = function(info, key) return info.handler:GetShowAt(key) end,
      },
      hideWhenEmpty = {
         type = "toggle",
         name = L["Hide when empty"],
         desc = L["Do not open this bag automatically when it contains no items"],
         order = 21,
         handler = optionsBag,
         set = function(info, value) info.handler:SetHideWhenEmpty(value) end,
         get = function(info) return info.handler:GetHideWhenEmpty() end,
      },
      hideAnim = {
         type = "toggle",
         name = L["Hide item animation"],
         desc = L["Do not play animation when an item is added to this bag"],
         order = 100,
         handler = optionsBag,
         set = function(info, value) info.handler:SetHideAnimation(value) end,
         get = function(info) return info.handler:GetHideAnimation() end,
      },
   },
}

local function getOptionsTemplate(optionsBag)
   for key, option in pairs(optionsTemplate.args) do
      option.handler = optionsBag
   end
   return optionsTemplate
end

local function deepCopy(src, dest)
   for key, value in pairs(src) do
      if type(value) == "table" and key ~= "handler" then
         local newDest = dest[key]
         if not newDest then
            newDest = {}
            dest[key] = newDest
         end
         deepCopy(value, newDest)
      else
         dest[key] = value
      end
   end
end

function Bag:GetOptions()
   local options = self.options
   if not options then
      options = {}
      deepCopy(getOptionsTemplate(self), options)
      options.name = self.savedVars.name
      self.options = options
   end
   return self.options
end

Luggage.Bag = Bag
