local MAJOR_VERSION = "LibBaggage-1.0"
local MINOR_VERSION = tonumber(("$Revision: 31 $"):match("(%d+)"))

if not LibStub then error(MAJOR_VERSION .. " requires LibStub") end

local libBaggage, oldLib = LibStub:NewLibrary(MAJOR_VERSION, MINOR_VERSION)
local self = libBaggage
if not libBaggage then
	return
end

local debug = false

local function print(message)
    DEFAULT_CHAT_FRAME:AddMessage(message)
end

libBaggage.events = libBaggage.events or 
  LibStub("CallbackHandler-1.0"):New(libBaggage)

-- Set up frame for event handling. Code blatantly copied from LibMobHealth-4.0
-- (sorry, ckknight!) and modified to better suit our purpose
local frame = CreateFrame("Frame", MAJOR_VERSION .. "_Frame")
--frame:RegisterEvent("ADDON_LOADED")
frame:RegisterEvent("PLAYER_ENTERING_WORLD")
frame:RegisterEvent("BAG_UPDATE")
frame:RegisterEvent("BANKFRAME_OPENED")
frame:RegisterEvent("PLAYERBANKSLOTS_CHANGED")
frame:RegisterEvent("ITEM_LOCK_CHANGED")
frame:RegisterEvent("GUILDBANKFRAME_OPENED")
frame:RegisterEvent("GUILDBANKBAGSLOTS_CHANGED")

frame:SetScript("OnEvent", function(this, event, ...)
	this[event](libBaggage, ...)
end)

local listenToBag_Update = false
frame:SetScript("OnUpdate", function(this, dTime) 
   listenToBag_Update = true 
end)


local baggage_inventory = {bank = {}, equipment = {}, keyring = {}, guildbank = {}}
for bagID=0, NUM_BAG_SLOTS+NUM_BANKBAGSLOTS do
    baggage_inventory[bagID] = {}
end
local GUILDBANK_OFFSET = 1000
for tab=1,GetNumGuildBankTabs() do
   baggage_inventory[tab + GUILDBANK_OFFSET] = {}
end

local equipmentSlots = {
"HeadSlot",
"NeckSlot",
"ShoulderSlot",
"BackSlot",
"ChestSlot",
"ShirtSlot",
"TabardSlot",
"WristSlot",
"HandsSlot",
"WaistSlot",
"LegsSlot",
"FeetSlot",
"Finger0Slot",
"Finger1Slot",
"Trinket0Slot",
"Trinket1Slot",
"MainHandSlot",
"SecondaryHandSlot",
"RangedSlot",
"AmmoSlot",
"Bag0Slot",
"Bag1Slot",
"Bag2Slot",
"Bag3Slot"
}

------------------------------------------------------------------------------------------------------------------
-- Tooltip scanning for binding status
------------------------------------------------------------------------------------------------------------------
-- Code from wowwiki.com
local scanTooltip = CreateFrame( "GameTooltip", MAJOR_VERSION.."_ScanningTooltip" );
scanTooltip:SetOwner( WorldFrame, "ANCHOR_NONE" );
scanTooltip:AddFontStrings(
    scanTooltip:CreateFontString( "$parentTextLeft1", nil, "GameTooltipText" ),
    scanTooltip:CreateFontString( "$parentTextRight1", nil, "GameTooltipText" ) )
-- We will actually only use this line
local scanline = scanTooltip:CreateFontString( "$parentTextLeft2", nil, "GameTooltipText" )
scanline:SetFontObject(GameFontNormal)
scanTooltip:AddFontStrings(scanline, scanTooltip:CreateFontString( "$parentTextRight2", nil, "GameTooltipText" ))

-- The binding status (soulbound, BoE, BoU, Quest Item or not binding) can only be extracted from the tooltip
local function GetBindingStatus(bagID, slotID)
    scanTooltip:ClearLines()
    if bagID > GUILDBANK_OFFSET then
        scanTooltip:SetGuildBankItem(bagID,slotID)
    elseif slotID then
        scanTooltip:SetBagItem(bagID,slotID)
    else
        scanTooltip:SetInventoryItem("player",bagID)
    end
    local bindingText = scanline:GetText()
    if not bindingText then return nil end
    if string.find(bindingText, ITEM_SOULBOUND) then
        return ITEM_SOULBOUND
    elseif string.find(bindingText, ITEM_BIND_QUEST) then
        return ITEM_BIND_QUEST
    elseif string.find(bindingText, ITEM_BIND_ON_EQUIP) then
        return ITEM_BIND_ON_EQUIP
    elseif string.find(bindingText, ITEM_BIND_ON_USE) then
        return ITEM_BIND_ON_USE
    else 
      return nil
    end
end

------------------------------------------------------------------------------------------------------------------
-- Item table creation and recycling
------------------------------------------------------------------------------------------------------------------
local cache = setmetatable({}, {__mode='k'})

local function DestroyItemTable(itemTable)
    cache[itemTable] = true
end

-- Regex from wowwiki.com
local function ExtractItemStringFromLink(link)
    local found, _, itemString = string.find(link, "^|c%x+|H(.+)|h%[.*%]")
    if(found) then
        return itemString
    end
end

-- Combining GetContainerItemLink and GetInventoryItemLink so we can handle
-- bags and bank / equipment uniformly
local function GetItemLink(bagID, slotID)
   if type(bagID) == "number" and bagID > GUILDBANK_OFFSET then
      return GetGuildBankItemLink(bagID - GUILDBANK_OFFSET, slotID)
   end
   if slotID then
      return GetContainerItemLink(bagID, slotID);
   else
      return GetInventoryItemLink("player", bagID)
   end
end

-- Mimicking  the functionality of GetContainerItemInfo with GetInventoryItemTexture
-- and GetInventoryItemCount so we can handle bags and bank / equipment uniformly
local function GetIconAndCount(bagID, slotID)
   if bagID > GUILDBANK_OFFSET then
      return GetGuildBankItemInfo(bagID - GUILDBANK_OFFSET, slotID)
   elseif slotID then
      return GetContainerItemInfo(bagID, slotID)
   else
      local icon = GetInventoryItemTexture("player", bagID)
      local stackCount = GetInventoryItemCount("player", bagID)
      return icon, stackCount
   end
end

local meta = {
   __eq = function(item1, item2)
      return item1.itemID == item2.itemID and 
             item1.uniqueID == item2.uniqueID and 
             item1.stackCount == item2.stackCount
   end
}

local function GetItemTable(bagID, slotID)
   local link = GetItemLink(bagID, slotID)
   if not link then 
       return 
   end
   local name, _, rarity, level, minLevel, type, subType, maxStackCount, equipLocation = GetItemInfo(link);
   local icon, stackCount, locked = GetIconAndCount(bagID, slotID)
   local binding = GetBindingStatus(bagID, slotID)
   local itemString = ExtractItemStringFromLink(link)
   local _, itemID, enchantID, jewelID1, jewelID2, jewelID3, jewelID4, suffixID, uniqueID = strsplit(":", itemString)
   if slotID then
      local oldItemTable = baggage_inventory[bagID][slotID]
   end
   uniqueID = tonumber(uniqueID)
   -- Let's check if this item has really changed before creating a nasty table
   if oldItemTable and
      itemID == oldItemTable.itemID and 
      uniqueID == oldItemTable.uniqueID and 
      stackCount == oldItemTable.stackCount then
     return oldItemTable
   end

   jewelID1 = tonumber(jewelID1)
   jewelID2 = tonumber(jewelID2)
   jewelID3 = tonumber(jewelID3)
   jewelID4 = tonumber(jewelID4)
   suffixID = tonumber(suffixID)
   enchantID = tonumber(enchantID)

   local itemTable = next(cache)
   if itemTable then
       cache[itemTable] = nil
   else
       itemTable = {}
       setmetatable(itemTable, meta)
   end
   itemTable.itemID = tonumber(itemID)
   itemTable.name = name
   itemTable.link = link
   itemTable.rarity = rarity
   itemTable.level = level
   itemTable.minLevel = minLevel
   itemTable.type = type
   itemTable.subType = subType
   itemTable.maxStackCount = maxStackCount
   itemTable.stackCount = stackCount
   itemTable.icon = icon
   itemTable.locked = locked
   itemTable.bagID = bagID
   itemTable.slotID = slotID
   itemTable.binding = binding
   itemTable.jewelID1 = jewelID1
   itemTable.jewelID2 = jewelID2
   itemTable.jewelID3 = jewelID3
   itemTable.jewelID4 = jewelID4
   itemTable.suffixID = suffixID
   itemTable.uniqueID = uniqueID
   itemTable.enchantID = enchantID
   itemTable.equipLocation = equipLocation
   return itemTable
end

local function AddItem(itemTable)
   baggage_inventory[itemTable.bagID][itemTable.slotID] = itemTable
end

local function GetBagType(bagID)
    if bagID ~= "bank" and bagID == 0 then return 0 end
    if bagID == "bank" then return 0 end
    if bagID <= NUM_BAG_SLOTS then
        local itemLink = GetItemLink(bagID + 19)
        return GetItemFamily(itemLink)
    end
    if bagID > NUM_BAG_SLOTS then
        local itemLink = GetItemLink(bagID - NUM_BAG_SLOTS + 67)
        return GetItemFamily(itemLink)
    end
end

local function RescanBag(bagID, initialScan)
   local bag = baggage_inventory[bagID]
   local size = GetContainerNumSlots(bagID)
   if bagID >= NUM_BAG_SLOTS + 1 and not BankFrame:IsShown() then
      return
   end
   bag.size = size
   bag.freeSpace = bag.freeSpace or size
   bag.type = GetBagType(bagID)
   for slotID=1,size do
      local itemTable = GetItemTable(bagID, slotID)
      local oldItemTable = bag[slotID]
      if itemTable ~=  oldItemTable then
         if oldItemTable and not initialScan then
            bag.freeSpace = bag.freeSpace + 1
            self:FireItemRemoved(oldItemTable)
         end
         if itemTable then
            bag.freeSpace = bag.freeSpace - 1
            AddItem(itemTable)
            if not initialScan then
               self:FireItemAdded(bagID, slotID)
            end
         else
            baggage_inventory[oldItemTable.bagID][oldItemTable.slotID] = nil
         end
         if oldItemTable then DestroyItemTable(oldItemTable) end
      end
   end
   -- At login, the bag sizes of the bagIDs 1-4 are all zero, regardless
   -- of their content.
   if bag.size <= 0 then
      bag.freeSpace = nil
   end
end

local function RescanMainBags(initialScan)
   for bagID=0,NUM_BAG_SLOTS do
      RescanBag(bagID, initialScan)
   end
   if initialScan then
      libBaggage.events:Fire("LibBaggage_MainBagItemsAvailable")
   end
end


-- We set this to true when rescanning several bags (i.e. on login). When a full rescan is
-- in progress, we don't fire several ItemAdded events, but one single "RescanDone" event
-- so as not to flood addons with event requests
local fullRescanInProgress

local function RescanBank()
   local bank = baggage_inventory.bank
   bank.size = NUM_BANKGENERIC_SLOTS
   bank.type = 0
   bank.freeSpace = bank.freeSpace or NUM_BANKGENERIC_SLOTS
   for slotID=1,NUM_BANKGENERIC_SLOTS do
      local itemTable = GetItemTable(BankButtonIDToInvSlotID(slotID))
      local oldItemTable = bank[slotID]
      if itemTable ~=  oldItemTable then
         if oldItemTable then
            bank.freeSpace = bank.freeSpace + 1
            self:FireItemRemoved(oldItemTable)
         end
         if itemTable then
            itemTable.bagID = "bank"
            itemTable.slotID = slotID
            if slotID > 28 then
               DEFAULT_CHAT_FRAME:AddMessage("Argh")
            end
            AddItem(itemTable)
            bank.freeSpace = bank.freeSpace - 1
            self:FireItemAdded(bagID, slotID)
         else
            baggage_inventory[oldItemTable.bagID][oldItemTable.slotID] = nil
         end
         if oldItemTable then DestroyItemTable(oldItemTable) end
      end
   end
   if bank.freeSpace < 0 then
      bank.freeSpace = nil
   end
   for bagID=NUM_BAG_SLOTS+1,NUM_BAG_SLOTS+NUM_BANKBAGSLOTS do
--~       ClearBag(bagID)
      RescanBag(bagID)
   end
end


local function RescanGuildBank()
   for tab=1,GetNumGuildBankTabs() do
      local guildbankTab = baggage_inventory[tab + GUILDBANK_OFFSET]
      guildbankTab.size = MAX_GUILDBANK_SLOTS_PER_TAB
      guildbankTab.freeSpace = guildbankTab.freeSpace or MAX_GUILDBANK_SLOTS_PER_TAB
      guildbankTab.type = 0
      for slot=1,MAX_GUILDBANK_SLOTS_PER_TAB do
         local itemTable = GetItemTable(tab + GUILDBANK_OFFSET, slot)
         local oldItemTable = guildbankTab[slot]
         if itemTable ~=  oldItemTable then
            if oldItemTable then
               guildbankTab.freeSpace = guildbankTab.freeSpace + 1
               self:FireItemRemoved(oldItemTable)
            end
            if itemTable then
               guildbankTab[slot] = itemTable
               guildbankTab.freeSpace = guildbankTab.freeSpace - 1
               self:FireItemAdded(tab + GUILDBANK_OFFSET, slot)
            else
               baggage_inventory[oldItemTable.bagID][oldItemTable.slotID] = nil
            end
            if oldItemTable then DestroyItemTable(oldItemTable) end
         end
      end
      if guildbankTab.freeSpace < 0 then
         guildbankTab.freeSpace = nil
      end
   end
end

local function ClearEquipment()
    for i,slotName in ipairs(equipmentSlots) do
        if baggage_inventory.equipment[slotName] then
            DestroyItemTable(baggage_inventory.equipment[slotName])
            baggage_inventory.equipment[slotName] = nil
        end
    end
end

local function RescanEquipment()
   ClearEquipment()
   for i, slotName in ipairs(equipmentSlots) do
      local invSlot = GetInventorySlotInfo(slotName)
      local itemTable = GetItemTable(invSlot)
      local oldItemTable = baggage_inventory.equipment[slotName]
      if itemTable ~=  oldItemTable then
         if oldItemTable then
            self:FireItemRemoved(oldItemTable)
         end
         if itemTable then
            itemTable.bagID = "equipment"
            itemTable.slotID = slotName
            AddItem(itemTable)
         end
         self:FireItemAdded("equipment", slotName)
         if oldItemTable then DestroyItemTable(oldItemTable) end
      end
   end
end

local function ClearKeyring()
    for slotID = 1,GetKeyRingSize() do
        if baggage_inventory.keyring[slotID] then
            DestroyItemTable(baggage_inventory.keyring[slotID])
            baggage_inventory.keyring[slotID] = nil
        end
    end
end

local function RescanKeyring()
   ClearKeyring()
   for slotID = 1,GetKeyRingSize() do
      local invSlot = KeyRingButtonIDToInvSlotID(slotID)
      local itemTable = GetItemTable(invSlot)
      local oldItemTable = baggage_inventory.bank[slotID]
      if itemTable ~=  oldItemTable then
         if oldItemTable then
            self:FireItemRemoved(oldItemTable)
         end
         if itemTable then
            itemTable.bagID = "keyring"
            itemTable.slotID = slotID
            AddItem(itemTable)
         end
         self:FireItemAdded("keyring", slotID)
         if oldItemTable then DestroyItemTable(oldItemTable) end
      end
   end
end

local function RescanAll()
   fullRescanInProgress = true
   RescanMainBags()
   RescanEquipment()
   RescanKeyring()
   fullRescanInProgress = nil
   self:FireFullRescanDone("bags")
end

function frame:BANKFRAME_OPENED()
   fullRescanInProgress = true
   RescanBank()
   fullRescanInProgress = nil
   self:FireFullRescanDone("bank")
end

function frame:PLAYER_ENTERING_WORLD()
   fullRescanInProgress = true
   RescanMainBags()
   RescanEquipment()
   RescanKeyring()
   fullRescanInProgress = nil
   self:FireFullRescanDone("bags")
   listenToBag_Update = true
end

-- We need this one so we don't dismiss a BAG_UPDATE when something changes in several bags at once
local lastBag_Update_BagID = -100
function frame:BAG_UPDATE(bagID, slotID)
   if bagID < -1 or (not listenToBag_Update and bagID == lastBag_Update_BagID) then return end
   listenToBag_Update = false
   lastBag_Update_BagID = bagID
   RescanBag(bagID)
end

local defragCoroutine  -- I hate to declare this here, but ITEM_LOCK_CHANGED needs it
function frame:ITEM_LOCK_CHANGED(bagID, slotID)
    if not slotID then 
        slotID = bagID
        bagID = "equipment"
        for i, slotName in ipairs(equipmentSlots) do
            local invSlot = GetInventorySlotInfo(slotName)
            if invSlot == slotID then
                slotID = slotName
                break
            end
        end
    elseif bagID == -1 then 
        bagID = "bank"
    elseif bagID == -2 then
      bagID = "keyring"
    end
    local item = baggage_inventory[bagID][slotID]
    if item then
        item.locked = not item.locked
        libBaggage.events:Fire("LibBaggage_ItemLockChanged", bagID, slotID)
    end
    if defragCoroutine and coroutine.status(defragCoroutine) == "suspended" and not item.locked then
      coroutine.resume(defragCoroutine)
    end
end

function frame:PLAYERBANKSLOTS_CHANGED(slotID)
   if slotID > NUM_BANKGENERIC_SLOTS then
      RescanBank()
      return
   end
   local bank = baggage_inventory.bank
   local oldItem = bank[slotID]
   if oldItem then
      bank.freeSpace = bank.freeSpace + 1
      self:FireItemRemoved(oldItem)
      DestroyItemTable(oldItem)
   end
   bank[slotID] = nil
   local itemTable = GetItemTable(BankButtonIDToInvSlotID(slotID))
   if itemTable then
      itemTable.bagID = "bank"
      itemTable.slotID = slotID
      AddItem(itemTable)
      bank.freeSpace = bank.freeSpace - 1
      self:FireItemAdded("bank", slotID)
   end
end

function frame:GUILDBANKFRAME_OPENED()
   for i=1,GetNumGuildBankTabs() do
      QueryGuildBankTab(i)
   end
end

function frame:GUILDBANKBAGSLOTS_CHANGED()
   --fullRescanInProgress = true
   RescanGuildBank()
   --fullRescanInProgress = nil
   --self:FireFullRescanDone("guildbank")
end

function libBaggage:FireItemAdded(bagID, slotID, misc)
   if not fullRescanInProgress then
      libBaggage.events:Fire("LibBaggage_ItemAdded", bagID, slotID, misc)
   end
end

function libBaggage:FireItemRemoved(oldItem)
   if not fullRescanInProgress then
      libBaggage.events:Fire("LibBaggage_ItemRemoved", oldItem)
   end
end

function libBaggage:FireFullRescanDone(location)
   libBaggage.events:Fire("LibBaggage_FullRescanDone", location)
end

function libBaggage:GetRawData()
    return baggage_inventory
end

function libBaggage:GetItem(bagID, slotID, misc)
   if bagID == "guildbank" then
      return baggage_inventory.guildbank[slotID][misc]
   end
   if bagID and slotID and baggage_inventory[bagID] then
      return baggage_inventory[bagID][slotID]
   end
end

function libBaggage:GetFreeBagSpace(bagID)
   local bag = baggage_inventory[bagID]
   return bag.freeSpace or 0
end

function libBaggage:GetTotalFreeMainBagSpace(bagType, useAlternateBitFlags)
   if not bagType then bagType = 0 end
   if useAlternateBitFlags then
      bagType = bit.rshift(bagType, 1)
   end
   local space = 0
   for bagID=0,NUM_BAG_SLOTS do
      if bagType == 0 or bit.band(baggage_inventory[bagID].type, bagType) > 0 then
         space = space + self:GetFreeBagSpace(bagID)
      end
   end
   return space
end

function libBaggage:GetTotalFreeBankSpace(bagType, useAlternateBitFlags)
   if not bagType then bagType = 0 end
   if useAlternateBitFlags then
      bagType = bit.rshift(bagType, 1)
   end
   local space = 0
   if bagType == 0 then
      space = self:GetFreeBagSpace("bank") or 0
   end
   for bagID=NUM_BAG_SLOTS+1,NUM_BAG_SLOTS+NUM_BANKBAGSLOTS do
      if bagType == 0 or bit.band(baggage_inventory[bagID].type or 0, bagType) > 0 then
         space = space + (self:GetFreeBagSpace(bagID) or 0)
      end
   end
   return space
end

function libBaggage:GetFreeSlot(bank)
   if bank then
      local bankBag = baggage_inventory.bank
      for slotID=1,NUM_BANKGENERIC_SLOTS do
         if not bankBag[slotID] then
            return -1, slotID
         end
      end
      for bagID=NUM_BAG_SLOTS+1,NUM_BAG_SLOTS+NUM_BANKBAGSLOTS do
         local bag = baggage_inventory[bagID]
         for slotID = 1,GetContainerNumSlots(bagID) do
            if not bag[slotID] then
               return bagID, slotID
            end
         end
      end
   else
      for bagID=0,NUM_BAG_SLOTS do
         local bag = baggage_inventory[bagID]
         for slotID = 1,GetContainerNumSlots(bagID) do
            if not bag[slotID] then
               return bagID, slotID
            end
         end
      end
   end
end

local stackCache = setmetatable({}, {mode='k'})
partialStacks = {}
local function UpdatePartialStacks(bank)
   local iterator, state, initial
   for itemID, stacks in pairs(partialStacks) do
      for i, item in pairs(stacks) do
         stackCache[stacks[i]] = true
         stacks[i] = nil
      end
      partialStacks[itemID] = nil
   end
   if bank then
      iterator, state, initial = libBaggage:BankItems()
   else
      iterator, state, initial = libBaggage:MainBagItems()
   end
   for item in iterator, state, initial do
      if item.stackCount < item.maxStackCount then
         local itemID = item.itemID
         if not partialStacks[itemID] then
            partialStacks[itemID] = {}
         end
         local stackCoord = next(stackCache) or {}
         stackCache[stackCoord] = nil
         stackCoord.bagID = item.bagID
         stackCoord.slotID = item.slotID
         stackCoord.stackCount = item.stackCount
         table.insert(partialStacks[itemID], stackCoord)
      end
   end
end

local defragging = false
local function Defrag()
   for itemID, stacks in pairs(partialStacks) do
      Luggage_Stacks = stacks
      while #stacks > 1 do
         local stack1Coord = stacks[1]
         local stack2Coord = stacks[2]
         local stack1, stack2
         while true do
            stack1 = baggage_inventory[stack1Coord.bagID][stack1Coord.slotID]
            stack2 = baggage_inventory[stack2Coord.bagID][stack2Coord.slotID]
            if stack1.locked or stack2.locked then
               coroutine.yield()
            else
               break
            end
         end
         if stack2.bagID == "bank" then
            PickupContainerItem(-1, stack2.slotID)
         else
            PickupContainerItem(stack2.bagID, stack2.slotID)
         end
         if stack1.bagID == "bank" then
            PickupContainerItem(-1, stack1.slotID)
         else
            PickupContainerItem(stack1.bagID, stack1.slotID)
         end
         local combinedCount = stack1.stackCount + stack2.stackCount
         if combinedCount <= stack1.maxStackCount then
            table.remove(stacks, 2)
         elseif combinedCount > stack1.maxStackCount then
            table.remove(stacks, 1)
         end
      end
   end
   defragging = false
end

function libBaggage:DefragmentInventory()
   if defragging then
      --coroutine.status(defragCoroutine)
      return
   end
   defragging = true
   UpdatePartialStacks()
   if type(defragCoroutine) ~= "thread" or coroutine.status(defragCoroutine) == "dead" then
      defragCoroutine = coroutine.create(Defrag)
      coroutine.resume(defragCoroutine)
   end
end

function libBaggage:DefragmentBank()
   if not BankFrame:IsShown() or defragging then
      --coroutine.status(defragCoroutine)
      return
   end
   defragging = true
   UpdatePartialStacks(true)
   if type(defragCoroutine) ~= "thread" or coroutine.status(defragCoroutine) == "dead" then
      defragCoroutine = coroutine.create(Defrag)
      coroutine.resume(defragCoroutine)
   end
end

----------------------- Iterators -------------------------------

local function mainBagsIterator(state, current)
    local curr_bagID = current.bagID or 0
    local curr_slotID = current.slotID or 0
    for bagID = curr_bagID,NUM_BAG_SLOTS do
        local bag = baggage_inventory[bagID]
        for slotID=curr_slotID+1,(bag.size or 0) do
            local item = bag[slotID]
            if item then return item end
        end
        curr_slotID = 0
    end
end

function libBaggage:MainBagItems()
    return mainBagsIterator,1,{}
end

local function filteredMainBagsIterator(filter, current)
    local curr_bagID = current.bagID or 0
    local curr_slotID = current.slotID or 1
    for bagID = curr_bagID,NUM_BAG_SLOTS do
        local bag = baggage_inventory[bagID]
        for slotID=curr_slotID+1,bag.size do
            local item = bag[slotID]
            if item and filter(item) then return item end
        end
        curr_slotID = 0
    end
end

function libBaggage:FilteredMainBagItems(filterFunction)
    if not filterFunction then 
        filterFunction = function(item) return true end
    end
    return filteredMainBagsIterator,filterFunction,{}
end

local function bankBagsIterator(state, current)
    local curr_bagID = current.bagID or "bank"
    local curr_slotID = current.slotID or 0
    if curr_bagID == "bank" then
        local bag = baggage_inventory[curr_bagID]
        for slotID=curr_slotID+1,NUM_BANKGENERIC_SLOTS do
            local item = bag[slotID]
            if item then return item end
        end
        curr_bagID = NUM_BAG_SLOTS+1
        curr_slotID = 0
    end
    for bagID = curr_bagID,NUM_BAG_SLOTS+NUM_BANKBAGSLOTS do
        local bag = baggage_inventory[bagID]
        for slotID=curr_slotID+1,bag.size do
            local item = bag[slotID]
            if item then return item end
        end
        curr_slotID = 0
    end
end

function libBaggage:BankItems()
    return bankBagsIterator,nil,{}
end

local function filteredBankBagsIterator(filter, current)
    local curr_bagID = current.bagID or "bank"
    local curr_slotID = current.slotID or 0
    if curr_bagID == "bank" then
        local bag = baggage_inventory[curr_bagID]
        for slotID=curr_slotID+1,NUM_BANKGENERIC_SLOTS do
            local item = bag[slotID]
            if item and filter(item) then return item end
        end
        curr_bagID = NUM_BAG_SLOTS+1
        curr_slotID = 0
    end
    for bagID = curr_bagID,NUM_BAG_SLOTS+NUM_BANKBAGSLOTS do
        local bag = baggage_inventory[bagID]
        for slotID=curr_slotID+1,bag.size do
            local item = bag[slotID]
            if item and filter(item) then return item end
        end
    end
end

function libBaggage:FilteredBankItems(filterFunction)
    return filteredBankBagsIterator,filterFunction,{}
end

local function allItemsIterator(state, current)
   local curr_bagID = current.bagID or "bank"
   local curr_slotID = current.slotID or 0
   if curr_bagID == "bank" then
      local bag = baggage_inventory[curr_bagID]
      for slotID=curr_slotID+1,NUM_BANKGENERIC_SLOTS do
         local item = bag[slotID]
         if item then return item end
      end
      curr_bagID = GUILDBANK_OFFSET+1
      curr_slotID = 0      
   end
   if type(curr_bagID) == "number" and curr_bagID > GUILDBANK_OFFSET then
      for bagID = curr_bagID,GUILDBANK_OFFSET+GetNumGuildBankTabs() do
         local bag = baggage_inventory[bagID]
         for slotID=curr_slotID+1,(bag.size or 0) do
            local item = bag[slotID]
            if item then return item end
         end
         curr_slotID = 0
      end
      curr_bagID = "keyring"
      curr_slotID = 0      
   end
   if curr_bagID == "keyring" then
      local bag = baggage_inventory[curr_bagID]
      for slotID=curr_slotID+1,GetKeyRingSize() do
         local item = bag[slotID]
         if item then return item end
      end
      curr_bagID = 0
      curr_slotID = 0      
   end
   for bagID = curr_bagID,NUM_BAG_SLOTS+NUM_BANKBAGSLOTS do
      local bag = baggage_inventory[bagID]
      for slotID=curr_slotID+1,(bag.size or 0) do
         local item = bag[slotID]
         if item then return item end
      end
      curr_slotID = 0
   end
end

function libBaggage:AllItems()
   return allItemsIterator, nil, {}
end
