################################################################################
# Description: Extracts all WoW icon names to a Lua table
# Requires: Blizzard Interface Art installed
# Usage:
#   python generateIconTable.py PATH
#   - PATH: Path to the icon directory, may be omitted if the script is executed 
#           from within an addon directory (i.e. WoW/Interface/Addons/SomeAddon)
# Author: Nimbal
# Date: $Date$
# Revision: $Revision$
# Legal blurp:
#   You may use, modify and redistribute this script freely as long as you
#   give credit to all previous authors and don't charge money for it
################################################################################
import os
import re
import sys

# See constructIconCategoryTree for more information about DEFAULT_INDEX
DEFAULT_INDEX = "Default"
# Each level of the category tree will be indented with this much whitespace
# relative to the previous level
TAB_WIDTH = 3
# Change this if you want to use this script for another addon:
ADDON_NAME = "Luggage"
# The Lua table will be written to this file in the current working directory
OUTPUT_FILE = "icons.lua"

def getIconDirectoryListing():
    """
    Returns a list of files from the directory given at the command line.
    If no argument is given, it is assumed that the current working directory
    is WoW/Interface/Addons/SomeAddon
    """
    currentDir = os.getcwd()
    if len(sys.argv) > 1:
        os.chdir(sys.argv[1])
    else:
        os.chdir('../../../Blizzard Interface Art (enGB)/Icons')
    files = os.listdir('.')
    os.chdir(currentDir)
    return files
    
def extractIconNames(fileList):
    """
    Takes a list of files from getIconDirectoyListing() and returns the
    name (i.e. stripped of extension) of all files ending with .blp or .BLP
    """
    regex = re.compile(r'^(.*)\.(blp|BLP)$')
    icon_list = []
    for filename in files:
        m = regex.match(filename.replace('-', '_'))
        if m:
            icon_list.append(m.group(1))
    return icon_list

def setValueAtPath(tree, path, value):
    """
    Takes a tree (a nested dictionary), a path (a list of valid dictionary
    indices) and a value and sets the value at path. If the path doesn't exist,
    it is created. If the path itself already contains a subtree, the value
    will be inserted in this subtree at the DEFAULT_INDEX.
    See constructIconCategoryTree for more information about DEFAULT_INDEX
    """
    currentSubtree = tree
    for i in range(len(path)):
        index = path[i]
        # sometimes the icons are named in all uppercase. To prevent
        # duplicate tree entries with different cases, capitalize them
        if index == index.upper():
            index = index.capitalize()
        
        if currentSubtree.has_key(index):
            if i == len(path) - 1:
                currentSubtree[DEFAULT_INDEX] = value
            else:
                currentSubtree = currentSubtree[index]
        elif i == len(path) - 1:
            currentSubtree[index] = value
        else:
            currentSubtree[index] = {}
            currentSubtree = currentSubtree[index]

def constructIconCategoryTree(iconNames):
    """
    Takes a list of icon names and splits them at their underscores. The
    resulting splits are then used as a path in a tree, with the iconName
    itself as leaf. The iconNames are traversed in reverse order to catch
    those cases where there is an icon named 'Path' and one or more named
    'Path_XYZ'. The latter are added first to the tree so that setValueAtPath
    can insert the former at the DEFAULT_INDEX
    """
    icon_table = {}
    for j in reversed(range(len(icon_list))):
        icon_name = icon_list[j]
        path = icon_name.split('_')
        setValueAtPath(icon_table, path, icon_name)
    return icon_table

def merge(list1, list2):
    """
        Merges two lists *without* removing duplicates
    """
    merged = []
    for e in list1:
        merged.append(e)
    for e in list2:
        merged.append(e)
    return merged

def isNumber(s):
    """
    Whether the given string is a positive integer
    """
    if re.match("^[0-9]+$", s):
        return True
    return False

def compare(x, y):
    """
    If both arguments are convertible to numbers, compare those. If not,
    return the usual lexical comparison. Using this comparator sorts
    sequences like [11, 2, 1] to [1, 2, 11] instead of [1, 11, 2]
    """
    if isNumber(x) and isNumber(y):
        return int(x) - int(y)
    elif x > y:
        return 1
    elif x < y:
        return -1
    else:
        return 0

def constructLuaTable(tree, tabs):
    """
    Traverses the tree in prefix order and returns a list of Lua code lines that
    represent a Lua table of the tree. The tabs argument is eyecandy.
    """
    n_icons_added = 0
    lines = []
    leadingWhitespace = ' '*tabs*TAB_WIDTH
    keys = tree.keys()
    keys.sort(compare)
    for key in keys:
        if type(tree[key]) == str:
            # leaf node reached
            lines.append(leadingWhitespace + '["' + key +
                         r'"] = "Interface\\Icons\\' + tree[key] + '",\n')
            n_icons_added += 1
        else:
            # we need a subtree
            subTreeLines, n_SubTree_icons = constructLuaTable(tree[key], tabs+1)
            n_icons_added += n_SubTree_icons
            lines.append(leadingWhitespace + '["' + key + '"] = {\n')
            lines = merge(lines, subTreeLines)
            lines.append(leadingWhitespace + '},\n')
    return lines, n_icons_added

##def constructFlatLuaTable(iconList):
##    """
##    Writes a non-tree list of all icons as lua table
##    """
##    lines = []
##    leadingWhitespace = ' '*2*TAB_WIDTH
##    for icon in iconList:
##        lines.append(leadingWhitespace + r'"Interface\\Icons\\' + icon + '",\n')
##    return lines

files = getIconDirectoryListing()
icon_list = extractIconNames(files)
if len(icon_list) == 0:
    print 'No *.blp files found in path "%s" Aborting.' % os.getcwd()
else:
    print "Processing %i files..." % len(icon_list)
    icon_table = constructIconCategoryTree(icon_list)
    f = open(OUTPUT_FILE, 'w')
    f.write('Luggage.CreateIconTable = function(self)\n')
    f.write(' '*TAB_WIDTH + 'local icons = self.icons or {\n')
    luaTable, n_icon_entries = constructLuaTable(icon_table, 2)
    f.write(''.join(luaTable))
    f.write(' '*TAB_WIDTH + '}\n')
    f.write(' '*TAB_WIDTH + 'self.icons = icons\n')
##    f.write(' '*TAB_WIDTH + 'local flatIcons = self.flatIcons or {\n')
##    f.write(''.join(constructFlatLuaTable(icon_list)))
##    f.write(' '*TAB_WIDTH + '}\n')
    f.write(' '*TAB_WIDTH + 'return icons, flatIcons\n')
    f.write('end')
    f.close()
    print "Done. %i icon entries written to %s." % (n_icon_entries, OUTPUT_FILE)
