local L = LibStub("AceLocale-3.0"):GetLocale("Luggage")
local B = LibStub("LibBabble-Inventory-3.0")
local BL = B:GetLookupTable()

Luggage:RegisterFilter("Rarity", {
   name = L["Rarity"],
   tooltipText = L["Filter items by rarity"],
   values = {
      {
         name = L["Poor"],
         tooltipTitle = L["Poor"],
         tooltipText = L["Poor (grey) items"],
      },
      {
         name = L["Common"],
         tooltipTitle = L["Common"],
         tooltipText = L["Common (white) items"],
      },
      {
         name = L["Uncommon"],
         tooltipTitle = L["Uncommon"],
         tooltipText = L["Uncommon (gree) items"],
      },
      {
         name = L["Rare"],
         tooltipTitle = L["Rare"],
         tooltipText = L["Rare (blue) items"],
      },
      {
         name = L["Epic"],
         tooltipTitle = L["Epic"],
         tooltipText = L["Epic (purple) items"],
      },
   },
   func = function(bag, itemTable, rarity)
      return itemTable.rarity == rarity - 1
   end,
})

Luggage:RegisterFilter("Location", {
   name = L["Location"],
   tooltipText = L["Filter items depending on where they are in your inventory"],
   values = {
      {
         name = L["Bank"],
         tooltipTitle = L["Bank"],
         tooltipText = L["All items in your bank"],
      },
      {
         name = L["Main Bags"],
         tooltipTitle = L["Main Bags"],
         tooltipText = L["All items in your main bags"],
      },
      {
         name = L["Keyring"],
         tooltipTitle = L["Keyring"],
         tooltipText = L["All your current keys"],
      },
      {
         name = L["Guild Bank"],
         tooltipTitle = L["Guild Bank"],
         tooltipText = L["All items in the guild bank"],
      },
   },
   func = function(bag, itemTable, position)
      local bagID = itemTable.bagID
      if position == 1 then
         -- Bank
         if type(bagID) == "string" then 
            return bagID == "bank"
         else
            return bagID > NUM_BAG_SLOTS and bagID <= NUM_BAG_SLOTS + NUM_BANKBAGSLOTS
         end
      elseif position == 2 then
         -- Inventory
         return type(bagID) == "number" and bagID <= NUM_BAG_SLOTS
      elseif position == 3 then
         -- Keyring
         return bagID == "keyring"
      elseif position == 4 then
         -- Guild Bank
         return bagID > 1000
      end
   end,
})

local bagsCheckingUnsorted = {}
local excludeUnsortedFilter = {Unsorted = true}
Luggage:RegisterFilter("Unsorted", {
   name = L["Unsorted"],
   tooltipText = L["Filter items that aren't contained by other bags. Beware! The almighty Unsorted filter doesn't like being set to red. If you do it anyways, the Unsorted filter will not strike you down, as he is a benevolent filter, but he will cause your virtual bag to appear always empty."],
   func = function(currentBag, itemTable)
      bagsCheckingUnsorted[currentBag] = true
      for bag in pairs(Luggage.bags) do
         if     not bagsCheckingUnsorted[bag]
            and     bag:ContainsItem(itemTable, excludeUnsortedFilter) then
            bagsCheckingUnsorted[currentBag] = nil
            return false
         end
      end
      bagsCheckingUnsorted[currentBag] = nil
      return true
   end,
})

Luggage:RegisterFilter("Quest_Item", {
   name = L["Quest Items"],
   tooltipText = L["Filter quest items. Note: This only includes items marked as \'Quest Item\' in their tooltips."],
   func = function(bag, itemTable)
      if itemTable.type == BL["Quest"] and itemTable.subType == BL["Quest"] then
         return true
      end
      return itemTable.binding == ITEM_BIND_QUEST
   end,
})

Luggage:RegisterFilter("Binding", {
   name = L["Binding"],
   tooltipText = L["Filter items by their binding status."],
   values = {
      BoE = {
         name = L["BoE"],
         tooltipText = string.format(L["Filters all items that have \'%s\' in their tooltip."], ITEM_BIND_ON_EQUIP),
      },
      BoP = {
         name = L["BoP"],
         tooltipText = string.format(L["Filters all items that have \'%s\' in their tooltip."], ITEM_BIND_ON_PICKUP),
      },
      BoU = {
         name = L["BoU"],
         tooltipText = string.format(L["Filters all items that have \'%s\' in their tooltip."], ITEM_BIND_ON_USE),
      },
      Soulbound = {
         name = ITEM_SOULBOUND,
         tooltipText = string.format(L["Filters all items that have \'%s\' in their tooltip."], ITEM_SOULBOUND),
      },
   },
   func = function(bag, itemTable, binding)
      if binding == "BoE" then
         return itemTable.binding == ITEM_BIND_ON_EQUIP
      elseif binding == "BoP" then
         return itemTable.binding == ITEM_BIND_ON_PICKUP
      elseif binding == "BoU" then
         return itemTable.binding == ITEM_BIND_ON_USE
      elseif binding == "Soulbound" then
         return itemTable.binding == ITEM_SOULBOUND
      end
   end,
})

local types = {
   BL["Armor"],
   BL["Projectile"],
   BL["Gem"],
   BL["Weapon"],
   BL["Mount"],
   BL["Pet"],
   BL["Reagent"],
}

Luggage:RegisterFilter("Type", {
   name = L["Type"],
   tooltipText = L["Filter by the type of the item."],
   values = {
      {
         name = L["Armor"],
         tooltipText = L["Armor of all types"],
      },
      {
         name = L["Gems"],
         tooltipText = L["Gems of all types"]
      },
      {
         name = L["Ammunition"],
         tooltipText = L["Things that hurt from afar"],
      },
      {
         name = L["Weapon"],
         tooltipText = L["Things that hurt when close"],
      },
      {
         name = L["Mount"],
         tooltipText = L["Bumpety bump, rider"],
      },
      {
         name = L["Pet"],
         tooltipText = L["Cute, cuddly, adorable"],
      },
      {
         name = L["Reagent"],
         tooltipText = L["What? You can't do magic without voodoo dolls?"],
      },
   },
   func = function(bag, item, value)
      return item.type == types[value] or item.subType == types[value]
   end,
})

local consumables = {
   BL["Food & Drink"],
   BL["Potion"],
   BL["Flask"],
   BL["Elixir"],
   BL["Bandage"],
   BL["Item Enhancement"],
   BL["Scroll"],
   BL["Other"],
}

Luggage:RegisterFilter("Consumable", {
   name = L["Consumable"],
   tooltipText = L["Filter items you can eat, drink or otherwise consume. Or make others consume. Possibly with force."],
   values = {
      {
         name = L["Food & Drink"],
         tooltipText = L["Yummy."],
      },
      {
         name = L["Potion"],
         tooltipText = L["Potions of all kinds. (Those with 2 minute cooldown)"],
      },
      {
         name = L["Flasks"],
         tooltipText = L["Colorful drinks with really long term effects. Keep away from small children."],
      },
      {
         name = L["Elixirs"],
         tooltipText = L["Colorful drinks with mostly long term effects. Keep away from small children."],
      },
      {
         name = L["Bandage"],
         tooltipText = L["For when you cut yourself"],
      },
      {
         name = L["Item Enhancement"],
         tooltipText = L["Sharpening stones, poisons, etc."],
      },
      {
         name = L["Scroll"],
         tooltipText = L["For when you forget your text"],
      },
      {
         name = L["Other"],
         tooltipText = L["The others are not coming. They are right in your bag."],
      },
   },
   func = function(bag, item, subTypeID)
      if item.type ~= BL["Consumable"] then
         return false
      end
      return item.subType == consumables[subTypeID]
   end,
})

tradeskills = {
   BL["Cloth"],
   BL["Leather"],
   BL["Metal & Stone"],
   BL["Meat"],
   BL["Herb"],
   BL["Enchanting"],
   BL["Jewelcrafting"],
   BL["Parts"],
   BL["Devices"],
   BL["Explosives"],
   BL["Elemental"],
}

Luggage:RegisterFilter("Tradeskill", {
   name = L["Tradeskill"],
   tooltipText = L["Things for your craft"],
   values = {
--~       Elemental = {
--~          name = L["Elemental"],
--~          tooltipText = ???
--~       },
      {
         name = L["Cloth"],
         tooltipText = L["Rags and fine fabrics"],
      },
      {
         name = L["Leather"],
         tooltipText = L["Dead animal's exterior"],
      },
      {
         name = L["Metal & Stone"],
         tooltipText = L["Ripped from the rocks"],
      },
      {
         name = L["Meat"],
         tooltipText = L["Dead animal's interior"],
      },
      {
         name = L["Herb"],
         tooltipText = L["Ripped from the earth"],
      },
      {
         name = L["Enchanting"],
         tooltipText = L["Making things glow"],
      },
      {
         name = L["Jewelcrafting"],
         tooltipText = L["Making shiny things shinier"],
      },
      {
         name = L["Parts"],
         tooltipText = L["Why is that screw left over?"],
      },
      {
         name = L["Device"],
         tooltipText = L["I wonder what that big red button does..."],
      },
      {
         name = L["Explosives"],
         tooltipText = L["Special effects"],
      },
      {
         name = L["Elemental"],
         tooltipText = L["Primals, motes, essences..."],
      },
--~       Materials = {
--~          name = L["Materials"],
--~          tooltipText = L["Rags and fine fabrics"]
--~       },
--~       Other = {
--~          name = L["Cloth"],
--~          tooltipText = L["Rags and fine fabrics"]
--~       },
   },
   func = function(bag, item, subTypeID)
      if item.type ~= BL["Trade Goods"] then
         return
      end
      return item.subType == tradeskills[subTypeID]
   end,
})

Luggage:RegisterFilter("EquipLocation", {
   name = L["Equipment Location"],
   tooltipText = L["Where do you put this thing?"],
   values = {
      ["INVTYPE_AMMO"] = {
         name = _G["INVTYPE_AMMO"],
         tooltipText = L["No pew, pew without these"],
      },
      ["INVTYPE_HEAD"] = {
         name = _G["INVTYPE_HEAD"],
         tooltipText = L["Don't lose it!"],
      },
      ["INVTYPE_NECK"] = {
         name = _G["INVTYPE_NECK"],
         tooltipText = L["Mah Bling"],
      },
      ["INVTYPE_SHOULDER"] = {
         name = _G["INVTYPE_SHOULDER"],
         tooltipText = L["Shoulderpads are so 80's"],
      },
      ["INVTYPE_BODY"] = {
         name = _G["INVTYPE_BODY"],
         tooltipText = L["If you are allergic to your mithril armor, put a shirt on"],
      },
      ["Chest"] = {
         name = _G["INVTYPE_CHEST"],
         tooltipText = L["Chest pain prevention (in case of sharp pointy things)"],
      },
      ["INVTYPE_WAIST"] = {
         name = _G["INVTYPE_WAIST"],
         tooltipText = L["Unnnggghh! I need to lose weight."],
      },
      ["INVTYPE_LEGS"] = {
         name = _G["INVTYPE_LEGS"],
         tooltipText = L["Jeans, leggings and spray paint"],
      },
      ["INVTYPE_FEET"] = {
         name = _G["INVTYPE_FEET"],
         tooltipText = L["May they carry you long"],
      },
      ["INVTYPE_WRIST"] = {
         name = _G["INVTYPE_WRIST"],
         tooltipText = L["No, officer, those ain\'t no handcuffs."],
      },
      ["INVTYPE_HAND"] = {
         name = _G["INVTYPE_HAND"],
         tooltipText = L["Cold fingers?"],
      },
      ["INVTYPE_FINGER"] = {
         name = _G["INVTYPE_FINGER"],
         tooltipText = L["Ten fingers, but only two rings. Ever wondered why?"],
      },
      ["INVTYPE_TRINKET"] = {
         name = _G["INVTYPE_TRINKET"],
         tooltipText = L["Earrings, Talismans, Bunny feet"],
      },
      ["INVTYPE_CLOAK"] = {
         name = _G["INVTYPE_CLOAK"],
         tooltipText = L["Ever wondered why draenei cloaks seem to flutter constantly, even if there is now wind?"],
      },
      ["Weapon"] = {
         name = L["Weapon"],
         tooltipText = L["Things that hurt when close"],
      },
      ["INVTYPE_SHIELD"] = {
         name = L["Shield"],
         tooltipText = L["Yay, frisbees!"],
      },
      ["INVTYPE_HOLDABLE"] = {
         name = _G["INVTYPE_HOLDABLE"],
         tooltipText = L["What is this? Lollipop? Q-Tip? Some kind of modern art form?"],
      },
      ["Ranged"] = {
         name = _G["INVTYPE_RANGED"],
         tooltipText = L["Pew, pew!"],
      },
      ["INVTYPE_RELIC"] = {
         name = _G["INVTYPE_RELIC"],
         tooltipText = L["No pew, pew, but close"],
      },
      ["INVTYPE_TABARD"] = {
         name = _G["INVTYPE_TABARD"],
         tooltipText = L["Shirt, chest armor, tabard. For those really cold days."],
      },
   },
   func = function(bag, item, slot)
      local location = item.equipLocation
      if slot == "Chest" then
         return location == "INVTYPE_ROBE" or location == "INVTYPE_CHEST"
      end
      if slot == "Ranged" then
         return location == "INVTYPE_RANGED" or location == "INVTYPE_THROWN" or location == "INVTYPE_RANGEDRIGHT"
      end
      if slot == "Weapon" then
         return location == "INVTYPE_WEAPON" or location == "INVTYPE_2HWEAPON" or location == "INVTYPE_WEAPONMAINHAND" or location == "INVTYPE_WEAPONOFFHAND"
      end
      return item.equipLocation == slot
   end,
})