local TabbedFrame = {}
local SELECTED_ICON_SIZE = 25
local DESELECTED_ICON_SIZE = 0.8*SELECTED_ICON_SIZE
local NAME_PREFIX = "Luggage_TabbedFrame"

local compostHeaps = {}
local GetCompost = function()
   table.insert(compostHeaps, setmetatable({}, {__mode='k'}))
   local compostNumber = #compostHeaps
   local thisCompost = compostHeaps[compostNumber]
   local get = function()
      local object = next(thisCompost)
      if object then
         thisCompost[object] = nil
      end
      return object
   end
   local deposit = function(object)
      assert(object, "Can't compost nil object")
      thisCompost[object] = true
   end
   return get, deposit
end

-- Objects (the tables)
local GetObject, DepositObject = GetCompost()

-- Frames (the backdropped ones)
local GetFrame, DepositFrame = GetCompost()

function TabbedFrame:New()
   local newTabbedFrame = GetObject()
   if not newTabbedFrame then
      newTabbedFrame = {
         tabButtons = {},
         tabFrames = {},
         tabAlignment = "LEFT",
         content = {},
      }
      setmetatable(newTabbedFrame, {__index = self})
   end
   newTabbedFrame:CreateFrame()
   return newTabbedFrame
end

function TabbedFrame:Release()
   self:TrimTabs(0)
   self.frame:ClearAllPoints()
   self.frame:Hide()
   DepositFrame(self.frame)
   self.frame = nil
   DepositObject(self)
end

--[[
The main frame is created here, nothing more.
]]--
local nFrames = 0
function TabbedFrame:CreateFrame()
   local frame = GetFrame()
   if not frame then
      frame = CreateFrame("Frame", NAME_PREFIX..nFrames, UIParent)
      nFrames = nFrames + 1
      frame:SetBackdrop({
         bgFile   = "Interface/Tooltips/UI-Tooltip-Background", 
         edgeFile = "Interface/Tooltips/UI-Tooltip-Border",
         tile     = true, 
         tileSize = 16, 
         edgeSize = 16, 
         insets   = { left = 4, right = 4, top = 4, bottom = 4 }
      })
      frame:SetBackdropColor(0,0,0,0.5);
      frame:EnableMouse(true)
      frame:SetMovable(true)
      frame:SetClampedToScreen(true)
      frame:SetFrameStrata("High")
      frame:SetScript( "OnMouseDown",
         function(this, mouseButton)
            if mouseButton == "LeftButton" and not this.isMoving and frame:IsMovable() then
               this:StartMoving()
               this.isMoving = true
            end
         end
      )
   end
   frame:SetScript( "OnMouseUp",
      function(this, mouseButton)
         if this.isMoving then
            this:StopMovingOrSizing()
            this.isMoving = false
            self:FrameMovedCallback()
         end
      end
   )
   frame:SetScript( "OnHide",
      function(this)
         if this.isMoving then
            this:StopMovingOrSizing()
            this.isMoving = false
            self:FrameMovedCallback()
         end
      end
   )
   -- Without this, the tab button borders are not right when showing / hiding the interface twice
   frame:SetScript( "OnShow",
      function(this)
         self:SelectTab(self.activeTabID)
      end
   )
   self.frame = frame
end

--[[
Sets the background color
]]--
function TabbedFrame:SetColor(r, g, b, a)
   self.frame:SetBackdropColor(r, g, b, a);
end

local GetTabFrame, DepositTabFrame = GetCompost()
local GetTabButton, DepositTabButton = GetCompost()
local globalTabNumber = 0
local function GetFrameAndButton()
   -- Frame
   local tabFrame = GetTabFrame()
   if not tabFrame then
      tabFrame = CreateFrame("Frame", "TabbedFrame_TabFrame"..globalTabNumber, nil)
   end
   tabFrame.totalWidth = 0
   -- Button
   local tabButton = GetTabButton() 
   if not tabButton then
      tabButton = CreateFrame("Button", "TabbedFrame_TabButton"..globalTabNumber, nil, "OptionsFrameTabButtonTemplate")
      tabButton:SetFrameStrata("HIGH")
      tabButton:RegisterForClicks("LeftButtonUp", "RightButtonUp")
      tabButton.icon = tabButton:CreateTexture()
      -- Make the highlight texture a bit larger
      local highlight = getglobal(tabButton:GetName().."HighlightTexture")
      highlight:ClearAllPoints()
      highlight:SetPoint("TOPLEFT", tabButton, "TOPLEFT", 10, -4)
      highlight:SetPoint("BOTTOMRIGHT", tabButton, "BOTTOMRIGHT", -10, -4)
   end
   globalTabNumber = globalTabNumber + 1
   return tabFrame, tabButton
end

--[[
Creates a new tab button. If you ever want to use another template than
OptionsFrameTabButtonTemplate, be prepared to change some numbers in
ResizeTabButton, too
]]--
function TabbedFrame:AddTab(icon, name)
   local mainFrame = self.frame
   local numTabs = #self.tabButtons + 1
   local tabFrame, tabButton = GetFrameAndButton()
   -- Button --
   tabButton:SetParent(mainFrame)
   -- We need the overhang for anchoring later
   if not self.tabButtonOverhang then
      self.tabButtonOverhang = getglobal(tabButton:GetName().."Left"):GetWidth();
   end
   tabButton:SetID(numTabs)
   tabButton:SetScript("OnMouseDown",
      function(buttonFrame, mouseButton)
         local frame = self.frame
         if mouseButton == "LeftButton" and not frame.isMoving and frame:IsMovable() then
            frame:StartMoving()
            frame.isMoving = true
         end
      end
   )
   tabButton:SetScript("OnMouseUp",
      function(buttonFrame, mouseButton)
         local frame = self.frame
         if frame.isMoving then
            frame:StopMovingOrSizing()
            frame.isMoving = false
            self:FrameMovedCallback()
         end
      end
   )
   tabButton:SetScript("OnClick", 
      function(buttonFrame, mouseButton)
         return TabbedFrame.Button_OnClick(self, buttonFrame, mouseButton)
      end
   )
   tabButton:SetScript("OnEnter", 
      function(buttonFrame)
         return TabbedFrame.Button_OnEnter(self, buttonFrame)
      end
   )
   tabButton:SetScript("OnLeave", 
      function(buttonFrame)
         return TabbedFrame.Button_OnLeave(self, buttonFrame)
      end
   )
   tabButton.tabName = name
   -- Button Icon
   local buttonIcon = tabButton.icon
   buttonIcon:SetTexture(icon)
   -- Frame --
   tabFrame:SetParent(mainFrame)
   tabFrame:SetAllPoints(mainFrame)
   tabFrame.requiredHeight = tabFrame:GetHeight()
   
   table.insert(self.tabButtons, tabButton)
   table.insert(self.tabFrames, tabFrame)
   tabButton:Show()
   self:ReanchorTabButtons()
   self:SelectTab(self.activeTabID or 1)
end

function TabbedFrame:RemoveTab(tabNumber)
   local activeTabID = self.activeTabID
   if activeTabID == tabNumber and activeTabID > 1 then
      self.activeTabID = activeTabID - 1
   end
   -- Button
   local tabButton = table.remove(self.tabButtons, tabNumber)
   tabButton:ClearAllPoints()
   tabButton:Hide()
   DepositTabButton(tabButton)
   -- Frame
   local tabFrame = table.remove(self.tabFrames, tabNumber)
   tabFrame:ClearAllPoints()
   tabFrame:Hide()
   DepositTabFrame(tabFrame)
   for i, tabButton in ipairs(self.tabButtons) do
      tabButton:SetID(i)
   end
   -- Content
   local tabContent = self.content[tabNumber]
   for i, frame in ipairs(tabContent) do
      frame:ClearAllPoints()
      tabContent[i] = nil
   end
   self:ReanchorTabButtons()
   self:SelectTab(self.activeTabID)
   self:AdjustSize()
end

function TabbedFrame:TrimTabs(number)
   while number > #self.tabButtons do
      self:AddTab("Interface\\Icons\\INV_Misc_QuestionMark", "?")
   end
   while number < #self.tabButtons do
      self:RemoveTab(#self.tabButtons)
   end
end

function TabbedFrame:Button_OnEnter(tabButton)
   GameTooltip:SetOwner(tabButton, "ANCHOR_LEFT");
   GameTooltip:SetText(tabButton.tabName or "I don't know my name")
   GameTooltip:Show()
end

function TabbedFrame:Button_OnLeave(frame)
   local self = frame.bag
   GameTooltip:Hide()
   ResetCursor()
end

function TabbedFrame:Button_OnClick(tabButton, mouseButton)
   if mouseButton == "LeftButton" then
      self:SelectTab(tabButton:GetID())
   else
      self:ShowDropdown(tabButton)
   end
end

function TabbedFrame:FrameMovedCallback()
end

--[[
If you want to use this outside Luggage, you'll have to change this function
]]--
function TabbedFrame:GetDropdownTable()
   return {}
end

local dropdown = CreateFrame("Frame", NAME_PREFIX.."Dropdown", nil, "UIDropDownMenuTemplate")
function TabbedFrame:ShowDropdown(tabButton)
   local menu = self:GetDropdownTable(tabButton)
   EasyMenu(menu, dropdown, tabButton, nil,nil, "MENU")
end

function TabbedFrame:SelectTab(tabNumber)
   for i, tabFrame in ipairs(self.tabFrames) do
      local tabButton = assert(self.tabButtons[i], "No tab button for tab frame "..tabFrame:GetName().." found!")
      self.activeTabID = tabNumber
      if i == tabNumber then
         PanelTemplates_SelectTab(tabButton)
         tabButton:Enable()
         self:ResizeTabButton(tabButton, SELECTED_ICON_SIZE)
         SetDesaturation(tabButton.icon, nil)
         tabFrame:Show()
      else
         PanelTemplates_DeselectTab(tabButton)
         self:ResizeTabButton(tabButton, DESELECTED_ICON_SIZE)
         SetDesaturation(tabButton.icon, 1)
         tabFrame:Hide()
      end
   end
   self:AdjustSize()
--   self:ReanchorTabButtons()
end

function TabbedFrame:ResizeTabButton(tabButton, iconSize, width, height)
   -- icon
   local buttonIcon = tabButton.icon
   buttonIcon:SetHeight(iconSize)
   buttonIcon:SetWidth(iconSize)
   
   -- If you want to use a tab button type other than OptionsFrameTabButtonTemplate,
   -- you will need to adjust these numbers here.
   if tabButton:GetID() == self.activeTabID then
      width = width or (iconSize + 25)
      height = height or (iconSize * 1.1 + 4)
      buttonIcon:SetPoint("BOTTOM", tabButton, "BOTTOM", 0, 0)
   else
      width = width or (iconSize + 24)
      height = height or (iconSize * 1.5 - 1)
      -- Don't ask me why the icon is one pixel off with the disabled texture
      buttonIcon:SetPoint("BOTTOM", tabButton, "BOTTOM", 1, 0)
   end
   
   -- Adjust width. Short and painless.
   PanelTemplates_TabResize(tabButton, 0, width)

   -- Adjust height. Iterate through the textures of the button
   local tabName = tabButton:GetName()
   local buttonElements = {
      left = getglobal(tabName.."Left"),
      right = getglobal(tabName.."Right"),
      middle = getglobal(tabName.."Middle"),
      disabled_left = getglobal(tabName.."LeftDisabled"),
      disabled_right = getglobal(tabName.."RightDisabled"),
      disabled_middle = getglobal(tabName.."MiddleDisabled"),
   }
   local oldHeight = tabButton:GetHeight()
   local scale = height/oldHeight
   for name, element in pairs(buttonElements) do
      element:SetHeight(scale * element:GetHeight())
   end
   -- only pro-forma. Visually, this doesn't do anything, but it
   -- gives us the correct value for the next GetHeight()
   tabButton:SetHeight(height)
end

function TabbedFrame:ReanchorTabButtons()
   local tabButtons = self.tabButtons
   if #tabButtons == 0 then return end

   -- The "Left" and "Right" elements of tab buttons are largely invisible and
   -- cause a too large padding if not corrected for
   local interTabPadding = -getglobal(tabButtons[1]:GetName().."Left"):GetWidth()/2
   local alignment = self.tabAlignment
   local oppositeAlignment
   local firstTabXOffset = 0
   if alignment == "LEFT" then
      oppositeAlignment = "RIGHT"
   elseif alignment == "RIGHT" then
      oppositeAlignment = "LEFT"
      interTabPadding = -interTabPadding
      firstTabXOffset = -firstTabXOffset
   else
      assert(false, string.format("Invalid alignment for tabbed frame: %s", alignment or "nil"))
   end
   local firstAnchor = "BOTTOM"..alignment
   local secondAnchor = "BOTTOM"..oppositeAlignment
   
   for i, button in ipairs(tabButtons) do
      button:ClearAllPoints()
      if i == 1 then
         button:SetPoint(firstAnchor, self.frame, "TOP"..alignment, interTabPadding/2 + firstTabXOffset, -3)
      else
         button:SetPoint(firstAnchor, tabButtons[i-1], secondAnchor, interTabPadding, 0)
      end
   end
end

function TabbedFrame:SetAlignment(newAlignment)
   assert(
      type(newAlignment) == "string" and (string.upper(newAlignment) == "LEFT" or string.upper(newAlignment) == "RIGHT"), 
      "Invalid argument newAlignment for "..self.name..".SetAlignment: "..(newAlignment or "nil")
   )
   self.tabAlignment = string.upper(newAlignment)
   self:ReanchorTabButtons()
end

function TabbedFrame:SetPoint(...)
   self.frame:SetPoint(...)
end

function TabbedFrame:Show()
   self.frame:Show()
end

function TabbedFrame:Hide()
   self.frame:Hide()
end

--[[
Returns the frame of the tab with the number 'id'. If 'id' is nil, the current
active tab's frame is returned. Use this for anchoring and parenting the tab's
content.
]]--
function TabbedFrame:GetFrame(id)
   id = id or self.activeTabID
   return self.tabFrames[id]
end

function TabbedFrame:SetMovable(movable)
   local frame = self.frame
   if not movable then
      if frame.isMoving then
         frame:StopMovingOrSizing()
         frame.isMoving = false
      end
   end
   frame:SetMovable(movable)
end

function TabbedFrame:AddContent(frame, tabNumber)
   if not self.content[tabNumber] then
      self.content[tabNumber] = {}
   end
   table.insert(self.content[tabNumber], frame)
end

function TabbedFrame:RemoveContent(frame)
   for tabNumber, tabContent in pairs(self.content) do
      for i, myFrame in ipairs(tabContent) do
         if myFrame == frame then
            return table.remove(tabContent, i)
         end
      end
   end
end

function TabbedFrame:ClearTabContent(tabNumber)
   local contentToClear = self.content[tabNumber]
   self.content[tabNumber] = {}
   return contentToClear
end

function TabbedFrame:LayoutContent(inset, padding)
   local alignment = self.tabAlignment
   local anchor = "TOP"..alignment
   for i, tabFrame in ipairs(self.tabFrames) do
      local maxHeight = 0
      local totalWidth = inset or 10
      local content = self.content[i] or {}
      for j, frame in ipairs(content) do
         frame:SetParent(tabFrame)
         frame:ClearAllPoints()
         if alignment == "LEFT" then
            frame:SetPoint(anchor, tabFrame, anchor, totalWidth, -inset)
         elseif alignment == "RIGHT" then
            frame:SetPoint(anchor, tabFrame, anchor, -totalWidth, -inset)
         end
         maxHeight = math.max(maxHeight, frame:GetHeight())
         totalWidth = totalWidth + frame:GetWidth() + padding
      end
      -- One padding too much, but we need a second inset
      tabFrame.totalWidth = totalWidth - padding + inset
      tabFrame.requiredHeight = maxHeight + 2*inset
   end
   self:AdjustSize()
end

function TabbedFrame:AdjustSize()
   local tabFrame = self.tabFrames[self.activeTabID]
   if not tabFrame then
      self.frame:SetWidth(1)
      self.frame:SetHeight(1)
   else
      local width = math.max(tabFrame.totalWidth, self:GetMinimumWidth())
      self.frame:SetWidth(width)
      self.frame:SetHeight(tabFrame.requiredHeight)
   end
end

function TabbedFrame:GetMinimumWidth()
   local firstButton = self.tabButtons[1]
   if not firstButton then return 0 end
   if #self.tabButtons == 1 then 
      local right = getglobal(firstButton:GetName().."Right")
      local left = getglobal(firstButton:GetName().."Left")
      local width = right:GetRight() - left:GetLeft()
      width = width - right:GetWidth()/2
      return width
   end
   local lastButton = self.tabButtons[#self.tabButtons]
   if self.tabAlignment == "LEFT" then
      local right = getglobal(lastButton:GetName().."Right")
      local left = getglobal(firstButton:GetName().."Left")
      local parent = right
      local width = right:GetRight() - left:GetLeft()
      width = width - right:GetWidth()/2
      return width

--~       local leftBorder = getglobal(firstButton:GetName().."Left"):GetLeft()
--~       local rightBorder = getglobal(lastButton:GetName().."Right"):GetRight()
--~       return rightBorder - leftBorder
   elseif self.tabAlignment == "RIGHT" then
      local right = getglobal(firstButton:GetName().."Right")
      local left = getglobal(lastButton:GetName().."Left")
      local width = right:GetRight() - left:GetLeft()
      width = width - right:GetWidth()/2
      return width
--~       local leftBorder = getglobal(lastButton:GetName().."Left"):GetLeft()
--~       local rightBorder = getglobal(firstButton:GetName().."Right"):GetRight()
--~       return rightBorder - leftBorder
   else
      assert(false, "Invalid tab alignment: "..(self.tabAlignment or "nil"))
   end
end

function TabbedFrame:SetIcon(tabNumber, icon)
   local tabButton = self.tabButtons[tabNumber]
   tabButton.icon:SetTexture(icon)
   if tabNumber == self.activeTabID then
      SetDesaturation(tabButton.icon, nil)
   else
      SetDesaturation(tabButton.icon, 1)
   end
end

function TabbedFrame:SetName(tabNumber, name)
   self.tabButtons[tabNumber].tabName = name
end

Luggage.TabbedFrame = TabbedFrame