local IconTree = {}
local L = LibStub("AceLocale-3.0"):GetLocale("Luggage")

function IconTree:Create()
   self.iconCategoryTree = Luggage:CreateIconTable()
end

--[[
For use with ButtonMatrix
]]--
function IconTree:GetButtonTableForPath(path)
   local subTree = self.iconCategoryTree
   for i, category in ipairs(path) do
      subTree = subTree[category]
   end
   local buttonTable = {}
   for name, iconPath in pairs(subTree) do
      if type(iconPath) == "string" then
         table.insert(buttonTable, {
            texture = iconPath,
            tooltipTitle = name,
            tooltipText = iconPath,
            value = iconPath,
         })
      end
   end
   return buttonTable
end

local function CreateTreeViewTable(icons)
   local iconTree
   for category, subTreeOrIconPath in pairs(icons) do
      if type(subTreeOrIconPath) == "table" then
         iconTree = iconTree or {}
         local subTree = CreateTreeViewTable(subTreeOrIconPath)
         table.insert(iconTree, {
            value = category, 
            text = category, 
            children = subTree
         })
      end
   end
   return iconTree
end

--[[
For use with TreeGroup
]]--
function IconTree:GetTableForTreeView()
   return CreateTreeViewTable(self.iconCategoryTree)
end

function IconTree:GetSubCategoryTree(path)
   local subTree = self.iconCategoryTree
   for i, category in ipairs(path) do
      subTree = subTree[category]
      if not subTree then return {} end
   end
   return subTree
end

--[[
Traverses the category tree leaf for leaf and returns all leafs that match the 
pattern. Surprisingly, even with 4000 icons, most of them nested 3 levels down, 
it is decently fast (about 30ms on my machine)
]]--
local function FindInTree(pattern, tree)
   local results = {}
   for key, subTree in pairs(tree) do
      if type(subTree) == "string" then
         -- leaf node reached
         if string.find(subTree:lower(), pattern:lower()) then
            table.insert(results, subTree)
         end
      elseif type(subTree) == "table" then
         -- search subtree
         local subTreeResults = FindInTree(pattern, subTree)
         if subTreeResults then
            for i, result in ipairs(subTreeResults) do
               table.insert(results, result)
            end
         end
      else
         assert(false, "Invalid argument for FindInTree: "..type(subTree))
      end
   end
   return results
end

function IconTree:Search(pattern, path)
   local treeToSearch
   if path then
      treeToSearch = self:GetSubCategoryTree(path)
   else
      treeToSearch = self.iconCategoryTree
   end
   return FindInTree(pattern, treeToSearch)
-- Faster search, but needs another (flat) table with all icons in it. 
-- Too memory intensive.
--~    local results = {}
--~    local n = 1
--~    for i, icon in ipairs(self.flatIconList) do
--~       if string.find(icon:lower(), pattern:lower()) then
--~          results[n] = icon
--~          n = n + 1
--~       end
--~    end
--~    return results
end

--[[
This one's eating memory. Lots of it. With 3000+ search results, memory
consumption jumps by more than 2 MB, so we limit it to 300 results max.
On user request, this limit may be changed some day, but almost certainly
never removed.
]]--
local MAX_SEARCH_RESULTS = 300
function IconTree:SearchAndConvertToButtonTable(pattern, path)
   local results = self:Search(pattern, path)
   local buttonTable = {}
   for i, result in ipairs(results) do
      if i > MAX_SEARCH_RESULTS then
         DEFAULT_CHAT_FRAME:AddMessage(string.format(L["More than %i search results. Please refine your search."], MAX_SEARCH_RESULTS))
         break
      end
      table.insert(buttonTable, {
         texture = result,
         tooltipTitle = L["Search Result"],
         tooltipText = result,
         value = result,
      })
   end
   return buttonTable
end

Luggage.IconTree = IconTree