local Bar = {}
local L = LibStub("AceLocale-3.0"):GetLocale("Luggage")

local GetBagBar, DepositBagBar = Luggage:GetCompost()

function Bar:New(name, savedVars)
   local newBagBar = GetBagBar()
   if not newBagBar then
      newBagBar = {
         bags = {},
      }
      setmetatable(newBagBar, {__index = self})
   end
   newBagBar.name = name
   newBagBar.savedVars = savedVars
   newBagBar:Create()
   return newBagBar
end

function Bar:Release()
   -- TabbedFrame
   self.tabbedFrame:Release()
   self.tabbedFrame = nil
   -- Bags
   for bag in pairs(self.bags) do
      bag:Release()
      self.bags[bag] = nil
   end
   self.globalBag:Release()
   self.globalBag = nil
   -- Self
   self.name = nil
   self.savedVars = nil
   self.options = nil
   DepositBagBar(self)
end

function Bar:Create()
   self.tabbedFrame = Luggage.TabbedFrame:New()
   local tabbedFrame = self.tabbedFrame
   tabbedFrame.FrameMovedCallback = function() self:SavePosition() end
   tabbedFrame:AdjustSize()
   tabbedFrame.GetDropdownTable = function(tabbedFrame, tabButton)
      local menu = {
         {
            text = L["Add Tab"],
            func = function()
               self:AddTab()
            end,
            tooltipTitle = L["Add Tab"],
            tooltipText = L["Add one tab to this bag bar"],
         },
         {
            text = L["Edit Tab"],
            tooltipTitle = L["Edit Tab"],
            tooltipText = L["Change icon and name"],
            func = function()
               self:EditTab(tabButton:GetID())
            end
         },
         {
            text = L["Remove Tab"],
            func = function()
               self:RemoveTab(tabButton:GetID())
            end,
            tooltipTitle = L["Remove Tab"],
            tooltipText = L["Removes this tab and all bags belonging to it"],
         },
         {
            text = L["Add Bag"],
            func = function()
               self:NewBagDialog(tabButton)
            end,
            tooltipTitle = L["Add Bag"],
            tooltipText = L["Add a bag to this tab"],
         },
      }
      return menu
   end
   self:UpdateTabbedFrame()
end

function Bar:UpdateTabbedFrame()
   local tabbedFrame = self.tabbedFrame
   local sv = self.savedVars
   tabbedFrame:SetPoint(sv.point, UIParent, sv.point, sv.x, sv.y)
   local holdLayout = true
   tabbedFrame:TrimTabs(#sv.tabs)
   for i, tab in ipairs(sv.tabs) do
      tabbedFrame:SetIcon(i, tab.icon)
      tabbedFrame:SetName(i, tab.name)
      for id, bagSavedVars in pairs(tab.bags) do
         self:AddBag(i, bagSavedVars, holdLayout)
      end
   end
   tabbedFrame:LayoutContent(sv.bagInset, sv.bagPadding)
   if sv.showAlways then
      tabbedFrame:Show()
   else
      tabbedFrame:Hide()
   end
end

function Bar:AddTab()
   local callback = function(icon, tabName)
      local tabSVs = self.savedVars.tabs
      local newTabSV = tabSVs[#tabSVs+1]
      newTabSV.icon = icon
      newTabSV.name = tabName
      self.tabbedFrame:AddTab(icon, tabName)
      self:AddBag(#tabSVs, newTabSV.bags[1])
   end
   Luggage:AskUserForIconAndName(callback)
end

function Bar:EditTab(tabID)
   local callback = function(icon, name)
      tabbedFrame:SetIcon(tabID, icon)
      tabbedFrame:SetName(tabID, name)
      self.savedVars.tabs[tabID].icon = icon
      self.savedVars.tabs[tabID].name = name
   end
   Luggage:AskUserForIconAndName(callback)
end

function Bar:RemoveTab(tabID)
   if #self.savedVars.tabs == 1 then
      DEFAULT_CHAT_FRAME:AddMessage(L["Can't remove the last tab. Edit this one or add a new tab before deleting it."])
      return
   end
   StaticPopupDialogs["Luggage_DeleteTab"] = {
      text = L["Warning: This will delete this tab and all bags belonging to it! Are you sure?"],
      button1 = L["Remove Tab"],
      button2 = L["Cancel"],
      OnAccept = function()
         local bags = self.savedVars.tabs[tabID].bags
         for i, bagSVs in ipairs(bags) do
            for bag in pairs(self.bags) do
               if bag.savedVars == bagSVs then
                  self:RemoveBag(bag)
               end
            end
         end
         table.remove(self.savedVars.tabs, tabID)
         self.tabbedFrame:RemoveTab(tabID)
      end,
      timeout = 0,
      whileDead = 1,
      hideOnEscape = 1
   }
   StaticPopup_Show ("Luggage_DeleteTab")
end


function Bar:SavePosition()
   local sv = self.savedVars
   local frame = self.tabbedFrame.frame
   local point, relativeTo, relativePoint, xOffs, yOffs = frame:GetPoint(1)
   sv.point = point or "BOTTOMLEFT"
   sv.x = xOffs or frame:GetLeft()
   sv.y = yOffs or frame:GetBottom()
end

function Bar:NewBagDialog(tabButton)
   local tabbedFrame = self.tabbedFrame
   local callback = function(icon, bagName)
      local bagList = self.savedVars.tabs[tabButton:GetID()].bags
      local bagOptions = bagList[#bagList+1]
      bagOptions.icon = icon
      bagOptions.name = bagName
      self:AddBag(tabButton:GetID(), bagOptions)
   end
   Luggage:AskUserForIconAndName(callback)
end

function Bar:AddBag(tabNumber, bagSavedVars, holdLayout)
   local bag = Luggage.Bag:New(bagSavedVars)
   self.bags[bag] = true
   bag:SetBagButtonSize(self.savedVars.bagButtonSize)
   bag.bagbar = self
   Luggage:RegisterBagForEvents(bag)
   local tabbedFrame = self.tabbedFrame
   tabbedFrame:AddContent(bag.bagButton, tabNumber)
   if not holdLayout then
      tabbedFrame:LayoutContent(self.savedVars.bagInset, self.savedVars.bagPadding)
   end
   if self.options then
      local bagOptions = self.options.args
      i = 1
      for _,_ in pairs(bagOptions) do
         i = i + 1
      end
      bagOptions[string.format("%i", i+1)] = bag:GetOptions()
      Luggage:UpdateOptions()
   end
end

function Bar:RemoveBag(bag)
   self.tabbedFrame:RemoveContent(bag.bagButton)
   self.tabbedFrame:LayoutContent(self.savedVars.bagInset, self.savedVars.bagPadding)
   for i, tab in pairs(self.savedVars.tabs) do
      for j, bagSV in ipairs(tab.bags) do
         if bagSV == bag.savedVars then
            table.remove(tab.bags, j)
            break
         end
      end
   end
   if self.options then
      local bagOptions = self.options.args
      local thisBagsOptions = bag:GetOptions()
      for bagIndex, options in pairs(bagOptions) do
         if options == thisBagsOptions then
            bagOptions[bagIndex] = nil
            Luggage:UpdateOptions()
            break
         end
      end
   end
   self.bags[bag] = nil
   bag:Release()
end

function Bar:Show()
   if self.disabled then return end
   self.tabbedFrame:Show()
end

function Bar:Hide()
   self.tabbedFrame:Hide()
   for bag in pairs(self.bags) do
      bag:Hide()
   end
end

function Bar:Disable()
   self:Hide()
   self.disabled = true
end

function Bar:Enable()
   self.disabled = nil
end

function Bar:SetBagInset(inset)
   if inset ~= self.savedVars.bagInset then
      self.savedVars.bagInset = inset
      self.tabbedFrame:LayoutContent(inset, self.savedVars.bagPadding)
   end
end

function Bar:SetBagPadding(padding)
   if padding ~= self.savedVars.bagPadding then
      self.savedVars.bagPadding = padding
      self.tabbedFrame:LayoutContent(self.savedVars.bagInset, padding)
   end
end

function Bar:SetBagButtonSize(size)
   if size ~= self.savedVars.bagButtonSize then
      self.savedVars.bagButtonSize = size
      for bag in pairs(self.bags) do
         bag:SetBagButtonSize(size)
      end
      self.tabbedFrame:LayoutContent(self.savedVars.bagInset, self.savedVars.bagPadding)
   end
end


function Bar:ResetPosition()
   self.savedVars.x = 500
   self.savedVars.y = 500
   self.savedVars.point = "BOTTOMLEFT"
   self.tabbedFrame:SetPoint("BOTTOMLEFT", UIParent, "BOTTOMLEFT", self.savedVars.x, self.savedVars.y)
end

function Bar:Delete()
   for bag in pairs(self.bags) do
      self:RemoveBag(bag)
      self.bags[bag] = nil
   end
   for i, bagbar in ipairs(Luggage.bagbars) do
      if bagbar == self then
         table.remove(Luggage.bagbars, i)
         break
      end
   end
   for index, sv in pairs(Luggage.db.profile.bagbars) do
      if sv == self.savedVars then
         Luggage.db.profile.bagbars[index] = nil
      end
   end
   self:Hide()
   Luggage:UpdateOptions()
end

function Bar:SetShowAt(key, value)
   local sv = self.savedVars
   sv[key] = value
   if sv.showAlways then
      self:Show()
      return
   end
   if sv.showAtBank and getglobal("BankFrame"):IsVisible() then
      self:Show()
      return
   end
   self:Hide()
end

local optionsTemplate = {
   type = "group",
   childGroups = "tree",
   args = {
      layoutGroup = {
         type = "group",
         name = L["Layout"],
         order = 10,
         inline = true,
         args = {
            inset = {
               type = "range",
               name = L["Bag Button Inset"],
               desc = L["The distance between the bag buttons and the edge of the bag bar frame."],
               min = 0,
               max = 20,
               step = 1,
               get = function(info) return info.handler.savedVars.bagInset end,
               set = function(info, value) info.handler:SetBagInset(value) end,
            },
            padding = {
               type = "range",
               name = L["Bag Button Padding"],
               desc = L["The distance between two bag buttons in this bar."],
               min = 0,
               max = 20,
               step = 1,
               get = function(info) return info.handler.savedVars.bagPadding end,
               set = function(info, value) info.handler:SetBagPadding(value) end,
            },
            size = {
               type = "range",
               name = L["Bag Button Size"],
               desc = L["The size of this bar's bag buttons."],
               min = 10,
               max = 50,
               step = 1,
               get = function(info) return info.handler.savedVars.bagButtonSize end,
               set = function(info, value) info.handler:SetBagButtonSize(value) end,
            },
            resetPos = {
               type = "execute",
               name = L["Reset Position"],
               desc = L["Reset the bar's position to (roughly) the center of the screen."],
               func = function(info) info.handler:ResetPosition() end,
            },
         },
      },
      showAt = {
         type = "multiselect",
         name = L["Show at:"],
         desc = L["Set when this bag bar should be displayed."],
         order = 20,
         values = {
            showAlways = L["Always"],
            showAtBank = L["At Bank"],
         },
         get = function(info, key) return info.handler.savedVars[key] end,
         set = function(info, key, value) info.handler:SetShowAt(key, value) end,
      },
      delete = {
         type = "execute",
         name = L["Delete"],
         desc = L["Delete Bag Bar"],
         order = 30,
         confirm = true,
         confirmText = L["This will delete the bag bar and all its bags. Are you sure?"],
         func = function(info) 
            info.handler:Delete()
--~             local optionsTable = info.options
--~             for i=1,#info-2 do
--~                optionsTable = optionsTable[info[i]]
--~             end
--~             table.remove(optionsTable, info[#info-1])
         end,
      },
      globalBagsHeader = {
         type = "header",
         name = L["Global Bag Options"],
         order = 99,
      },
--~       bags = {
--~          type = "group",
--~          name = L["Bags"],
--~          desc = L["Bag specific settings"],
--~          args = {
--~          },
--~       },
   },
}
local function getOptionsTemplate(optionsBar)
   for key, option in pairs(optionsTemplate.args) do
      option.handler = optionsBar
   end
   return optionsTemplate
end

local function deepCopy(src, dest)
   for key, value in pairs(src) do
      if type(value) == "table" and key ~= "handler" then
         local newDest = dest[key]
         if not newDest then
            newDest = {}
            dest[key] = newDest
         end
         deepCopy(value, newDest)
      else
         dest[key] = value
      end
   end
end

function Bar:GetOptions(target)
   local options = self.options
   if not options then
      options = {}
      deepCopy(getOptionsTemplate(self), options)
      options.name = self.name
      self.options = options
      local bagOptions = options.args
      i = 1
      for bag in pairs(self.bags) do
         bagOptions[string.format("%i", i)] = bag:GetOptions()
         i = i + 1
      end
      self.globalBag = Luggage.MultiBag:New(self.savedVars.global.bags, self.bags)
      local globalOptions = self.globalBag:GetOptions()
      globalOptions.name = ""
      globalOptions.inline = true
      globalOptions.order = 100
      options.args.globalBags = globalOptions
   end
   return self.options
end

Luggage.BagBar = Bar