local AceGUI = LibStub("AceGUI-3.0")

--------------------------
-- Button Matrix        --
--------------------------
--[[
   Row after row of buttons, nothing more.
   Provides:
      SetButtonTable(table): 
      Takes a table that has one entry for every button that should be 
      displayed. The entries should look like this:
      {
         texture = "SomeTexturePath (NOT a texture object)",
         tooltipTitle = "Text that is shown in white and large in the tooltip",
         tooltipText = "Text that is shown in yellow and small in the tooltip",
         value = The value that is put into status.selectedButtonValue,
      }
      Optionally, the table can contain the entries
         padding = Distance between buttons
         minIconSize = The minimum size of icons
         maxIconSize = The maximum size of icons
      
]]--
do
	local Type = "ButtonMatrix"
	local Version = 1
		
	local function OnAcquire(self)
      self.localstatus.buttonTable = {}
		self.localstatus.buttons = {}      
      self.status = self.localstatus
	end
	
   local buttonCache = {}
   local function GetButton()
      local button = next(buttonCache)
      if button then
         buttonCache[button] = nil
         return button
      end
      button = CreateFrame("CheckButton")
      button:SetHighlightTexture("Interface\\Buttons\\ButtonHilight-Square")
      button:SetCheckedTexture("Interface\\Buttons\\CheckButtonHilight")
      return button
   end
   local function DepositButton(button)
      button:ClearAllPoints()
      button:Hide()
      button:SetChecked(nil)
      buttonCache[button] = true
   end

	local function OnRelease(self)
		self.frame:ClearAllPoints()
		self.frame:Hide()
		self.status = nil
      self.buttonPane:Hide()
      for i, button in ipairs(self.localstatus.buttons) do
         DepositButton(button)
      end
		for k in pairs(self.localstatus) do
			self.localstatus[k] = nil
		end
	end

	local function LayoutFinished(self,width,height)
		self:BuildButtonMatrix()
	end
	
	-- called to set an external table to store status in
	local function SetStatusTable(self, status)
		assert(type(status) == "table")
		self.status = status
		if not status.buttonTable then
			status.buttonTable = {}
		end
	end
	
   local function OnWidthSet(self, width)
      self:BuildButtonMatrix()
	end
   
   local function SetButtonTable(self, buttonTable)
      assert(type(buttonTable) == "table", "SetButtonTable failed. Expected table, got "..type(buttonTable))
      self.status.buttonTable = buttonTable
      self:SelectButton(0)
      self:BuildButtonMatrix()
   end
   
   local function SelectButton(self, number)
--~       DEFAULT_CHAT_FRAME:AddMessage("Selected: "..number)
      for i, button in ipairs(self.status.buttons) do
         if i == number then
--~             DEFAULT_CHAT_FRAME:AddMessage("Setting status: "..number)
            self.status.selectedButtonValue = self.status.buttonTable[i].value
         else
            button:SetChecked(nil)
         end
      end
   end
   
   local DEFAULT_PADDING = 3
   local DEFAULT_MIN_ICON_SIZE = 37
   local DEFAULT_MAX_ICON_SIZE = 50
   local function BuildButtonMatrix(self)
      local status = self.status
      local buttonTable = status.buttonTable
      local padding = buttonTable.padding or DEFAULT_PADDING
      local minIconSize = buttonTable.minIconSize or DEFAULT_MIN_ICON_SIZE
      local maxIconSize = buttonTable.maxIconSize or DEFAULT_MAX_ICON_SIZE
      local buttonPane = self.buttonPane
      local availableWidth = self.frame:GetWidth()
--      DEFAULT_CHAT_FRAME:AddMessage("Available Width: "..availableWidth)
      local nButtonsPerRow = math.min(math.floor( (availableWidth) / (minIconSize + padding) ), #buttonTable)
--      DEFAULT_CHAT_FRAME:AddMessage("Buttons per Row: "..nButtonsPerRow)
      local actualIconSize = math.min((availableWidth - padding * nButtonsPerRow ) / nButtonsPerRow, maxIconSize)
--      DEFAULT_CHAT_FRAME:AddMessage("Actual Size: "..actualIconSize)
      local buttons = status.buttons
      -- trim number of buttons
      while #buttons < #buttonTable do
         local button = GetButton()
         table.insert(buttons, button)
      end
      while #buttons > #buttonTable do
         local button = table.remove(buttons)
         DepositButton(button)
      end
      for i, button in ipairs(buttons) do
         local texture = buttonTable[i].texture
         local tooltipTitle = buttonTable[i].tooltipTitle
         local tooltipText = buttonTable[i].tooltipText
         local value = buttonTable[i].value
         
         -- Set up button
         button:SetNormalTexture(texture)
         button:SetScript("OnEnter", 
            function()
               GameTooltip:SetText(tooltipTitle, 1,1,1)
               GameTooltip:AddLine(tooltipText, NORMAL_FONT_COLOR.r, NORMAL_FONT_COLOR.g, NORMAL_FONT_COLOR.b)
            end
         )
         button:SetScript("OnLeave", function() GameTooltip:Hide() end)
         button:SetScript("OnClick", function() self:SelectButton(i) end)
         
         button:ClearAllPoints()
         button:SetHeight(actualIconSize)
         button:SetWidth(actualIconSize)
         if (i % nButtonsPerRow) == 1 or nButtonsPerRow == 1 then
            local row = (i - 1) / nButtonsPerRow
            button:SetPoint("TOPLEFT", buttonPane, "TOPLEFT", padding, -(row*(actualIconSize + padding) + padding) )
         else
            button:SetPoint("LEFT", buttons[i-1], "RIGHT", padding, 0)
         end
         button:SetFrameStrata("TOOLTIP")
         button:SetParent(buttonPane)
         button:Show()
      end
      buttonPane:Show()
   end
   
	local createdcount = 0
	
   local function Constructor()
		local frame = CreateFrame("Frame",nil,UIParent)
		local self = {}
		self.type = Type
	
		self.OnRelease = OnRelease
		self.OnAcquire = OnAcquire
		
		self.LayoutFinished = LayoutFinished
		self.SetStatusTable = SetStatusTable
		self.OnWidthSet = OnWidthSet
      self.BuildButtonMatrix = BuildButtonMatrix
      self.SetButtonTable = SetButtonTable
      self.SelectButton = SelectButton
		
		self.localstatus = {}
      self.status = self.localstatus
		self.frame = frame
		frame.obj = self
      
      local buttonPane = CreateFrame("Frame", nil, frame)
      self.buttonPane = buttonPane
      buttonPane.obj = self
      buttonPane:SetAllPoints(frame)
      buttonPane:SetScript("OnSizeChanged", OnSizeChanged)
      
		self.localstatus.buttonTable = {}
      self.localstatus.selectedButtonValue = nil
		self.localstatus.buttons = {}
		AceGUI:RegisterAsWidget(self)
		return self
	end
	
	AceGUI:RegisterWidgetType(Type,Constructor,Version)
end
