--[[
   Dropdown menus of Luggage
]]--
local L = LibStub("AceLocale-3.0"):GetLocale("Luggage")

-- Return green for true, red for false and white for nil
function Luggage:GetFilterFontObject(value)
   if value == nil then
      return GameFontWhiteSmall
   elseif value == true then
      return GameFontGreenSmall
   elseif value == false then
      return GameFontRedSmall
   end
end

-- Return green for true, red for false and white for nil
function Luggage:GetFilterColor(value)
   if value == nil then
      return 1,1,1
   elseif value == true then
      return 0,1,0
   elseif value == false then
      return 1,0,0
   end
end

-- Cycle nil, true, false, nil, true, false, ...
function Luggage:GetNextState(current)
   if current == nil then
      return true
   elseif current == true then
      return false
   elseif current == false then
      return nil
   end
end

-- Update the color of the dropdown button and the bag's filter
local function CycleBagFilter(frame, id, value)
   local bag = Luggage.dropdownBag
   local bagFilters = bag.filters
   local bagFilterEntry = bagFilters[id]
   if not value then
      -- divalent filter
      local nextState = Luggage:GetNextState(bagFilterEntry)
      bagFilters[id] = nextState
      frame:SetNormalFontObject(Luggage:GetFilterFontObject(nextState));
      frame:SetHighlightFontObject(Luggage:GetFilterFontObject(nextState));
   else
      -- polyvalent filter
      if not bagFilterEntry then
         bag.filters[id] = {}
         bagFilterEntry = bag.filters[id]
      end
      local nextState = Luggage:GetNextState(bagFilterEntry[value])
      bagFilterEntry[value] = nextState
      frame:SetNormalFontObject(Luggage:GetFilterFontObject(nextState));
      frame:SetHighlightFontObject(Luggage:GetFilterFontObject(nextState));
   end
   getglobal(this:GetName().."Check"):Hide();
   Luggage:FullUpdate()
end

-- the notCheckable option of Blizzard's dropdown API removes the extra
-- spacing, but not the check itself, so we need to hide it all the time
local function HideCheck(button)
   getglobal(button:GetName().."Check"):Hide()
end

-- Sets up a callback for changing a bag's icon and name
local function EditBag(button, bag)
   local callback = function(icon, name)
      bag:SetIcon(icon)
      bag:SetName(name)
   end
   if bag.savedVars and bag.savedVars.name then
      Luggage:AskUserForIconAndName(callback, bag.savedVars.name, bag.savedVars.icon)
   else
     Luggage:AskUserForIconAndName(callback)
   end
end

-- Confirmation Dialog
local function RemoveBag(button, bag)
   StaticPopupDialogs["Luggage_DeleteBag"] = {
      text = L["Warning: This will delete this bag with all its settings! Are you sure?"],
      button1 = L["Remove Bag"],
      button2 = L["Cancel"],
      OnAccept = function()
         bag.bagbar:RemoveBag(bag)
         Luggage:UpdateOptions()
      end,
      timeout = 0,
      whileDead = 1,
      hideOnEscape = 1
   }
   StaticPopup_Show ("Luggage_DeleteBag")
end

local function DefragBag(button, bag)
   local isMainBags, isBank = bag:Locations()
   if isMainBags then
      LibStub("LibBaggage-1.0"):DefragmentInventory()
   end
   if isBank then
      LibStub("LibBaggage-1.0"):DefragmentBank()
   end
end

local SHOW_FREE_BAG_SPACE_VALUE = "showFreeBagSpace"
local BAG_TYPES_BIT_VALUES = {
   ["All"] = 1,                  -- L["All"]
   ["Quiver"] = 2,               -- L["Quiver"]
   ["Ammo Pouch"] = 4,           -- L["Ammo Pouch"]
   ["Soul Bag"] = 8,             -- L["Soul Bag"]
   ["Leatherworking Bag"] = 16,   -- L["Leatherworking Bag"]
   ["Herb Bag"] = 64,            -- L["Herb Bag"]
   ["Enchanting Bag"] = 128,      -- L["Enchanting Bag"]
   ["Engineering Bag" ] = 256,    -- L["Engineering Bag"]
   ["Gem Bag"] = 1024,            -- L["Gem Bag"]
   ["Mining Bag"] = 2048,        -- L["Mining Bag"]
}
local optionalDropdownParams = {
   "isTitle",
   "func",
   "textR",
   "textG",
   "textB",
}

local function callWrapper(button, func, arg2)
   Luggage.dropdownBag[func](Luggage.dropdownBag, arg2)
end
-- Always fill the same table with the dropdown options
local dropdownItem = {}
Luggage.InitBagDropdown = function(frame, level, menu)
   dropdownItem.FontObject = Luggage:GetFilterFontObject(nil)
   for i, key in ipairs(optionalDropdownParams) do
      dropdownItem[key] = nil
   end
   if level == 1 then
      -- First level of dropdown
      -- Edit Bag --
      dropdownItem.text = L["Edit Bag"]
      dropdownItem.tooltipTitle = L["Edit Bag"]
      dropdownItem.tooltipText = L["Change icon and name"]
      dropdownItem.isTitle = false
      dropdownItem.notCheckable = true
      dropdownItem.keepShownOnClick = false
      dropdownItem.hasArrow = false
      dropdownItem.func = EditBag
      dropdownItem.arg1 = Luggage.dropdownBag
      UIDropDownMenu_AddButton(dropdownItem)
      -- Remove Bag --
      dropdownItem.text = L["Remove Bag"]
      dropdownItem.tooltipTitle = L["Remove Bag"]
      dropdownItem.tooltipText = L["Remove this bag from its bag bar"]
      dropdownItem.isTitle = false
      dropdownItem.notCheckable = true
      dropdownItem.keepShownOnClick = false
      dropdownItem.hasArrow = false
      dropdownItem.func = RemoveBag
      dropdownItem.arg1 = Luggage.dropdownBag
      UIDropDownMenu_AddButton(dropdownItem)
      -- Defrag Inventory --
      dropdownItem.text = L["Defragment Inventory"]
      dropdownItem.tooltipTitle = L["Defragment Inventory"]
      dropdownItem.tooltipText = L["Physically compresses your partial stacks to minimize bag space usage."]
      dropdownItem.isTitle = false
      dropdownItem.notCheckable = true
      dropdownItem.keepShownOnClick = false
      dropdownItem.hasArrow = false
      dropdownItem.func = DefragBag
      dropdownItem.arg1 = Luggage.dropdownBag
      UIDropDownMenu_AddButton(dropdownItem)
      -- Show Bag Space --
      dropdownItem.text = L["Show Free Space"]
      dropdownItem.tooltipTitle = L["Show Free Space"]
      dropdownItem.tooltipText = L["Show the available space on this bag."]
      dropdownItem.isTitle = false
      dropdownItem.notCheckable = true
      dropdownItem.keepShownOnClick = true
      dropdownItem.hasArrow = true
      dropdownItem.value = SHOW_FREE_BAG_SPACE_VALUE
      UIDropDownMenu_AddButton(dropdownItem)
      -- Show number of contained items --
      dropdownItem.text = L["Show No. Of Contained Items"]
      dropdownItem.tooltipTitle = L["Show No. Of Contained Items"]
      dropdownItem.tooltipText = L["Show the number of items this bag is currently holding."]
      dropdownItem.isTitle = false
      dropdownItem.notCheckable = false
      dropdownItem.keepShownOnClick = true
      dropdownItem.hasArrow = false
      dropdownItem.checked = Luggage.dropdownBag.savedVars.showNumberOfContainedItems
      dropdownItem.func = callWrapper
      dropdownItem.arg1 = "ToggleShowNumberOfContainedItems"
      UIDropDownMenu_AddButton(dropdownItem)
      -- Combine Stacks --
      dropdownItem.text = L["Combine Stacks"]
      dropdownItem.tooltipTitle = L["Combine Stacks"]
      dropdownItem.tooltipText = L["Combine all stacks of one item type into one. Note: This does not mean compressing the stack, it is merely a display matter."]
      dropdownItem.isTitle = false
      dropdownItem.notCheckable = false
      dropdownItem.keepShownOnClick = true
      dropdownItem.hasArrow = false
      dropdownItem.checked = Luggage.dropdownBag.savedVars.combineStacks
      dropdownItem.func = callWrapper
      dropdownItem.arg1 = "ToggleCombineStacks"
      UIDropDownMenu_AddButton(dropdownItem)
      dropdownItem.checked = false
      -- Show empty slot --
      dropdownItem.text = L["Show Empty Slot"]
      dropdownItem.tooltipTitle = L["Show Empty Slot"]
      dropdownItem.tooltipText = L["Show an empty inventory slot in this bag (if there is one, that is)."]
      dropdownItem.isTitle = false
      dropdownItem.notCheckable = false
      dropdownItem.keepShownOnClick = true
      dropdownItem.hasArrow = false
      dropdownItem.checked = Luggage.dropdownBag.savedVars.showEmptySlot
      dropdownItem.func = callWrapper
      dropdownItem.arg1 = "ToggleEmptySlot"
      UIDropDownMenu_AddButton(dropdownItem)
      dropdownItem.checked = false
      -- Separator --
      dropdownItem.text = ""
      dropdownItem.isTitle = true
      dropdownItem.notCheckable = true
      UIDropDownMenu_AddButton(dropdownItem)
      dropdownItem.text = L["Filters"]
      dropdownItem.isTitle = true
      dropdownItem.hasArrow = false
      dropdownItem.notCheckable = true
      UIDropDownMenu_AddButton(dropdownItem)
      
      -- Filters --
      dropdownItem.isTitle = false
      dropdownItem.disabled = false
      dropdownItem.notCheckable = true
      dropdownItem.hasArrow = false
      dropdownItem.keepShownOnClick = 1
      for id, filterTable in pairs(Luggage.filters) do
         local tooltipTitle
         if not filterTable.tooltipTitle then
            filterTable.tooltipTitle = filterTable.name
         end
         if filterTable.author then
            tooltipTitle = string.format(L["%s by %s"], filterTable.tooltipTitle, filterTable.author)
         else
            tooltipTitle = filterTable.tooltipTitle
         end
         dropdownItem.text = filterTable.name
         dropdownItem.tooltipTitle = tooltipTitle
         dropdownItem.tooltipText = filterTable.tooltipText
         if filterTable.values then
            -- Filter is polyvalent
            dropdownItem.hasArrow = true
            dropdownItem.value = id
            dropdownItem.func = false -- set to false, not nil, to avoid allocating this index later on
--~             local r,g,b = Luggage:GetFilterColor(nil)
            dropdownItem.fontObject = Luggage:GetFilterFontObject(nextState)
            dropdownItem.func = HideCheck
         else
            -- Filter is divalent, so let's make the button do something
            dropdownItem.hasArrow = false
            dropdownItem.func = CycleBagFilter
            dropdownItem.arg1 = id
            dropdownItem.arg2 = nil
            dropdownItem.fontObject = Luggage:GetFilterFontObject(Luggage.dropdownBag.filters[id])
         end
         UIDropDownMenu_AddButton(dropdownItem);
      end
   elseif level == 2 and UIDROPDOWNMENU_MENU_VALUE == SHOW_FREE_BAG_SPACE_VALUE then
      -- Show bag types
      dropdownItem.hasArrow = false
      dropdownItem.notCheckable = false
      dropdownItem.func = callWrapper
      dropdownItem.arg1 = "ToggleShowFreeSpace"
--~       local availableBagTypes = {}
      for bagType, bitFlag in pairs(BAG_TYPES_BIT_VALUES) do
         dropdownItem.text = L[bagType]
         dropdownItem.arg2 = bitFlag
         local bagTypeFlags = Luggage.dropdownBag.savedVars.showFreeSpace
         dropdownItem.checked = (bit.band(bagTypeFlags, bitFlag) > 0)
         UIDropDownMenu_AddButton(dropdownItem, 2);         
      end
   elseif level == 2 then
      -- Some polyvalent filter
      local filterValues = Luggage.filters[UIDROPDOWNMENU_MENU_VALUE].values
      if type(filterValues) == "function" then
         filterValues = filterValues()
      end
      dropdownItem.hasArrow = false
      dropdownItem.func = CycleBagFilter
      dropdownItem.arg1 = UIDROPDOWNMENU_MENU_VALUE
      local bagFilterEntry = Luggage.dropdownBag.filters[UIDROPDOWNMENU_MENU_VALUE]
      for value, dropdownEntry in pairs(filterValues) do
         dropdownItem.text = dropdownEntry.name
         dropdownItem.tooltipTitle = dropdownEntry.tooltipTitle or dropdownEntry.name
         dropdownItem.tooltipText = dropdownEntry.tooltipText
         dropdownItem.arg2 = value
         dropdownItem.fontObject = Luggage:GetFilterFontObject(bagFilterEntry and bagFilterEntry[value])
         UIDropDownMenu_AddButton(dropdownItem, 2);         
      end
   end
end

-- The frame in which the dropdown is rendered
Luggage.dropdown = CreateFrame("Frame", "Luggage_Dropdown", nil, "UIDropDownMenuTemplate")
UIDropDownMenu_Initialize(Luggage.dropdown, Luggage.InitBagDropdown, "MENU")
function Luggage:ShowBagDropdown(bag)
   Luggage.dropdownBag = bag
   ToggleDropDownMenu(1, nil, Luggage.dropdown, bag.bagButton, 0, 0)
end
