local L = LibStub("AceLocale-3.0"):GetLocale("Luggage")
local backdrop = {
   bgFile   = "Interface/Tooltips/UI-Tooltip-Background", 
   edgeFile = "",
   tile     = true, 
   tileSize = 16, 
   edgeSize = 0, 
   insets   = { left = 0, right = 0, top = 0, bottom = 0 }
}

local function GetBackdroppedFrame(name, parent)
   local frame = CreateFrame("Frame", string.format("Luggage_Placement_%s", name), parent)
   frame:SetBackdrop(backdrop)
   frame:EnableMouse(true)
   return frame
end

local edgeWidth = 20
local expansionButtonSize = 40
local expansionButtonInset = 10
local textSize = 20
local iconSize = 20
local positionFrame

local topSizingHandle
local bottomSizingHandle 
local rightSizingHandle 
local leftSizingHandle

local topLeftCorner
local topRightCorner
local bottomLeftCorner
local bottomRightCorner

local topLeftExpansion
local topRightExpansion
local bottomLeftExpansion
local bottomRightExpansion

local topLeftText
local topRightText
local bottomLeftText
local bottomRightText

local topLeftIcon
local topRightIcon
local bottomLeftIcon
local bottomRightIcon


local function SavePositionAndSize()
   local anchor = Luggage.db.profile.placement.anchor
   local xOffs, yOffs
   if anchor == "TOPLEFT" then
      xOffs = leftSizingHandle:GetLeft()
      yOffs = -(UIParent:GetHeight() - topSizingHandle:GetTop())
   elseif anchor == "TOPRIGHT" then
      xOffs = -(UIParent:GetWidth() - rightSizingHandle:GetRight())
      yOffs = -(UIParent:GetHeight() - topSizingHandle:GetTop())
   elseif anchor == "BOTTOMRIGHT" then
      xOffs = -(UIParent:GetWidth() - rightSizingHandle:GetRight())
      yOffs = bottomSizingHandle:GetBottom()
   elseif anchor == "BOTTOMLEFT" then
      xOffs = leftSizingHandle:GetLeft()
      yOffs = bottomSizingHandle:GetBottom()
   end
   Luggage.db.profile.placement.xOffs = xOffs / UIParent:GetEffectiveScale()
   Luggage.db.profile.placement.yOffs = yOffs / UIParent:GetEffectiveScale()
   Luggage.db.profile.placement.width = (rightSizingHandle:GetRight() - leftSizingHandle:GetLeft()) / UIParent:GetEffectiveScale()
   Luggage.db.profile.placement.height = (topSizingHandle:GetTop() - bottomSizingHandle:GetBottom()) / UIParent:GetEffectiveScale()
   Luggage:ArrangeBags()
end

local function CreatePositionFrame()
   positionFrame = GetBackdroppedFrame("Position", UIParent)
   positionFrame:SetMovable(true)
   positionFrame:SetResizable(true)
   local minSize = expansionButtonSize*2 + 3*expansionButtonInset
   positionFrame:SetMinResize(minSize, minSize)
   positionFrame:SetScript( "OnMouseUp",
      function(this, mouseButton)
         if this.isMoving then
            this:StopMovingOrSizing()
            this.isMoving = false
            SavePositionAndSize()
         end
         if mouseButton == "RightButton" then
            positionFrame:Hide()
         end
      end
   )
   positionFrame:SetScript( "OnMouseDown",
      function(this, mouseButton)
         if mouseButton == "LeftButton" and not this.isMoving and positionFrame:IsMovable() then
            this:StartMoving()
            this.isMoving = true
         end
      end
   )
   positionFrame:SetScript( "OnHide",
      function(this)
         if this.isMoving then
            this:StopMovingOrSizing()
            this.isMoving = false
            SavePositionAndSize()
         end
      end
   )
   local tipText = positionFrame:CreateFontString(nil, "ARTWORK", "GameFontNormalSmall")
   tipText:SetAllPoints(positionFrame)
   tipText:SetText(L["Mouse over the colored regions to learn about their function.\n Right-Click to close."])
   return positionFrame
end

local function SizingHandles_OnEnter()
   GameTooltip:SetOwner(this, "ANCHOR_LEFT");
   GameTooltip:SetText(L["Resize Bag Area"], 1,1,1)
   GameTooltip:AddLine(L["Drag to resize the bag area. Note that only the green edge will be taken into account."], NORMAL_FONT_COLOR.r, NORMAL_FONT_COLOR.g, NORMAL_FONT_COLOR.b)
   GameTooltip:Show()
end
local function SizingHandles_OnLeave()
   GameTooltip:Hide()
end
local function CreateSizingHandles()
   -- Top
   topSizingHandle = GetBackdroppedFrame("TopSizingHandle", positionFrame)
   topSizingHandle:SetPoint("BOTTOMLEFT", positionFrame, "TOPLEFT")
   topSizingHandle:SetPoint("BOTTOMRIGHT", positionFrame, "TOPRIGHT")
   topSizingHandle:SetHeight(edgeWidth)
   topSizingHandle:SetScript( "OnMouseUp",
      function(this, mouseButton)
         if positionFrame.isSizing then
            positionFrame:StopMovingOrSizing()
            positionFrame.isSizing = false
            SavePositionAndSize()
         end
      end
   )
   topSizingHandle:SetScript( "OnMouseDown",
      function(this, mouseButton)
         if mouseButton == "LeftButton" and not this.isSizing and positionFrame:IsResizable() then
            positionFrame:StartSizing("TOP")
            positionFrame.isSizing = true
         end
      end
   )
   topSizingHandle:SetScript( "OnHide",
      function(this)
         if positionFrame.isSizing then
            positionFrame:StopMovingOrSizing()
            positionFrame.isSizing = false
            SavePositionAndSize()
         end
      end
   )
   topSizingHandle:SetScript( "OnEnter", SizingHandles_OnEnter)
   topSizingHandle:SetScript( "OnLeave", SizingHandles_OnLeave)
   
   -- Bottom
   bottomSizingHandle = GetBackdroppedFrame("BottomSizingHandle", positionFrame)
   bottomSizingHandle:SetPoint("TOPLEFT", positionFrame, "BOTTOMLEFT")
   bottomSizingHandle:SetPoint("TOPRIGHT", positionFrame, "BOTTOMRIGHT")
   bottomSizingHandle:SetHeight(edgeWidth)
   bottomSizingHandle:SetScript( "OnMouseUp",
      function(this, mouseButton)
         if positionFrame.isSizing then
            positionFrame:StopMovingOrSizing()
            positionFrame.isSizing = false
            SavePositionAndSize()
         end
      end
   )
   bottomSizingHandle:SetScript( "OnMouseDown",
      function(this, mouseButton)
         if mouseButton == "LeftButton" and not this.isSizing and positionFrame:IsResizable() then
            positionFrame:StartSizing("BOTTOM")
            positionFrame.isSizing = true
         end
      end
   )
   bottomSizingHandle:SetScript( "OnHide",
      function(this)
         if positionFrame.isSizing then
            positionFrame:StopMovingOrSizing()
            positionFrame.isSizing = false
            SavePositionAndSize()
         end
      end
   )
   bottomSizingHandle:SetScript( "OnEnter", SizingHandles_OnEnter)
   bottomSizingHandle:SetScript( "OnLeave", SizingHandles_OnLeave)
   -- Right
   rightSizingHandle = GetBackdroppedFrame("RightSizingHandle", positionFrame)
   rightSizingHandle:SetPoint("BOTTOMLEFT", positionFrame, "BOTTOMRIGHT")
   rightSizingHandle:SetPoint("TOPLEFT", positionFrame, "TOPRIGHT")
   rightSizingHandle:SetWidth(edgeWidth)
   rightSizingHandle:SetScript( "OnMouseUp",
      function(this, mouseButton)
         if positionFrame.isSizing then
            positionFrame:StopMovingOrSizing()
            positionFrame.isSizing = false
            SavePositionAndSize()
         end
      end
   )
   rightSizingHandle:SetScript( "OnMouseDown",
      function(this, mouseButton)
         if mouseButton == "LeftButton" and not this.isSizing and positionFrame:IsResizable() then
            positionFrame:StartSizing("RIGHT")
            positionFrame.isSizing = true
         end
      end
   )
   rightSizingHandle:SetScript( "OnHide",
      function(this)
         if positionFrame.isSizing then
            positionFrame:StopMovingOrSizing()
            positionFrame.isSizing = false
            SavePositionAndSize()
         end
      end
   )
   rightSizingHandle:SetScript( "OnEnter", SizingHandles_OnEnter)
   rightSizingHandle:SetScript( "OnLeave", SizingHandles_OnLeave)
   -- Left
   leftSizingHandle = GetBackdroppedFrame("LeftSizingHandle", positionFrame)
   leftSizingHandle:SetPoint("BOTTOMRIGHT", positionFrame, "BOTTOMLEFT")
   leftSizingHandle:SetPoint("TOPRIGHT", positionFrame, "TOPLEFT")
   leftSizingHandle:SetWidth(edgeWidth)
   leftSizingHandle:SetScript( "OnMouseUp",
      function(this, mouseButton)
         if positionFrame.isSizing then
            positionFrame:StopMovingOrSizing()
            positionFrame.isSizing = false
            SavePositionAndSize()
         end
      end
   )
   leftSizingHandle:SetScript( "OnMouseDown",
      function(this, mouseButton)
         if mouseButton == "LeftButton" and not this.isSizing and positionFrame:IsResizable() then
            positionFrame:StartSizing("LEFT")
            positionFrame.isSizing = true
         end
      end
   )
   leftSizingHandle:SetScript( "OnHide",
      function(this)
         if positionFrame.isSizing then
            positionFrame:StopMovingOrSizing()
            positionFrame.isSizing = false
            SavePositionAndSize()
         end
      end
   )
   leftSizingHandle:SetScript( "OnEnter", SizingHandles_OnEnter)
   leftSizingHandle:SetScript( "OnLeave", SizingHandles_OnLeave)
end



local function Corners_OnEnter()
   GameTooltip:SetOwner(this, "ANCHOR_LEFT");
   GameTooltip:SetText(L["Select Anchor For First Bag"], 1,1,1)
   GameTooltip:AddLine(L["Click to select in which corner the first bag should be placed."], NORMAL_FONT_COLOR.r, NORMAL_FONT_COLOR.g, NORMAL_FONT_COLOR.b)
   GameTooltip:Show()
end

local function Corners_OnLeave()
   GameTooltip:Hide()
end
local function CreateCorners()
   local savedVars = Luggage.db.profile.placement
   -- Top Left
   topLeftCorner = GetBackdroppedFrame("TopLeftCorner", positionFrame)
   topLeftCorner:SetPoint("BOTTOMRIGHT", positionFrame, "TOPLEFT")
   topLeftCorner:SetPoint("BOTTOMLEFT", leftSizingHandle, "TOPLEFT")
   topLeftCorner:SetPoint("TOPRIGHT", topSizingHandle, "TOPLEFT")
   topLeftCorner:SetScript("OnMouseUp",
      function()
         savedVars.anchor = "TOPLEFT"
         savedVars.primaryExpansion = "BOTTOM"
         savedVars.secondaryExpansion = "RIGHT"
         SavePositionAndSize()
         Luggage:UpdatePlacementFrame()
      end
   )
   topLeftCorner:SetScript("OnEnter", Corners_OnEnter)
   topLeftCorner:SetScript("OnLeave", Corners_OnLeave)
   -- Top Right
   topRightCorner = GetBackdroppedFrame("TopRightCorner", positionFrame)
   topRightCorner:SetPoint("BOTTOMLEFT", positionFrame, "TOPRIGHT")
   topRightCorner:SetPoint("BOTTOMRIGHT", rightSizingHandle, "TOPRIGHT")
   topRightCorner:SetPoint("TOPLEFT", topSizingHandle, "TOPRIGHT")
   topRightCorner:SetScript("OnMouseUp",
      function()
         savedVars.anchor = "TOPRIGHT"
         savedVars.primaryExpansion = "BOTTOM"
         savedVars.secondaryExpansion = "LEFT"
         SavePositionAndSize()
         Luggage:UpdatePlacementFrame()
      end
   )
   topRightCorner:SetScript("OnEnter", Corners_OnEnter)
   topRightCorner:SetScript("OnLeave", Corners_OnLeave)
   -- Bottom Left
   bottomLeftCorner = GetBackdroppedFrame("BottomLeftCorner", positionFrame)
   bottomLeftCorner:SetPoint("TOPRIGHT", positionFrame, "BOTTOMLEFT")
   bottomLeftCorner:SetPoint("TOPLEFT", leftSizingHandle, "BOTTOMLEFT")
   bottomLeftCorner:SetPoint("BOTTOMRIGHT", bottomSizingHandle, "BOTTOMLEFT")
   bottomLeftCorner:SetScript("OnMouseUp",
      function()
         savedVars.anchor = "BOTTOMLEFT"
         savedVars.primaryExpansion = "TOP"
         savedVars.secondaryExpansion = "RIGHT"
         SavePositionAndSize()
         Luggage:UpdatePlacementFrame()
      end
   )
   bottomLeftCorner:SetScript("OnEnter", Corners_OnEnter)
   bottomLeftCorner:SetScript("OnLeave", Corners_OnLeave)
   -- Bottom Right
   bottomRightCorner = GetBackdroppedFrame("BottomRightCorner", positionFrame)
   bottomRightCorner:SetPoint("TOPLEFT", positionFrame, "BOTTOMRIGHT")
   bottomRightCorner:SetPoint("TOPRIGHT", rightSizingHandle, "BOTTOMRIGHT")
   bottomRightCorner:SetPoint("BOTTOMLEFT", bottomSizingHandle, "BOTTOMRIGHT")
   bottomRightCorner:SetScript("OnMouseUp",
      function()
         savedVars.anchor = "BOTTOMRIGHT"
         savedVars.primaryExpansion = "TOP"
         savedVars.secondaryExpansion = "LEFT"
         SavePositionAndSize()
         Luggage:UpdatePlacementFrame()
      end
   )
   bottomRightCorner:SetScript("OnEnter", Corners_OnEnter)
   bottomRightCorner:SetScript("OnLeave", Corners_OnLeave)
end

local function ExpansionButtons_OnEnter()
   GameTooltip:SetOwner(this, "ANCHOR_LEFT");
   GameTooltip:SetText(L["Toggle primary expansion direction"], 1,1,1)
   GameTooltip:AddLine(L["Click to toggle in which direction newly opened bags should be placed first."], NORMAL_FONT_COLOR.r, NORMAL_FONT_COLOR.g, NORMAL_FONT_COLOR.b)
   GameTooltip:Show()
end

local function ExpansionButtons_OnLeave()
   GameTooltip:Hide()
end
local function CreateExpansionButtons()
   local savedVars = Luggage.db.profile.placement
   -- Top Left
   topLeftExpansion = GetBackdroppedFrame("TopLeftExpansion", positionFrame)
   topLeftExpansion:SetPoint("TOPLEFT", positionFrame, "TOPLEFT", expansionButtonInset, -expansionButtonInset)
   topLeftExpansion:SetWidth(expansionButtonSize)
   topLeftExpansion:SetHeight(expansionButtonSize)
   topLeftExpansion:SetScript("OnMouseUp",
      function()
         local buffer = savedVars.secondaryExpansion
         savedVars.secondaryExpansion = savedVars.primaryExpansion
         savedVars.primaryExpansion = buffer
         Luggage:UpdatePlacementFrame()
      end
   )
   topLeftExpansion:SetScript("OnEnter", ExpansionButtons_OnEnter)
   topLeftExpansion:SetScript("OnLeave", ExpansionButtons_OnLeave)
   topLeftText = topLeftExpansion:CreateFontString(nil, "ARTWORK", "GameFontNormal")
   topLeftText:SetPoint("TOPLEFT")
   topLeftText:SetWidth(textSize)
   topLeftText:SetHeight(textSize)
   topLeftIcon = topLeftExpansion:CreateTexture(nil, "ARTWORK")
   topLeftIcon:SetPoint("BOTTOMRIGHT")
   topLeftIcon:SetWidth(iconSize)
   topLeftIcon:SetHeight(iconSize)
   topLeftIcon:SetTexture("Interface\\ChatFrame\\UI-ChatIcon-ScrollUp-Up")
   -- Top Right
   topRightExpansion = GetBackdroppedFrame("TopRightExpansion", positionFrame)
   topRightExpansion:SetPoint("TOPRIGHT", positionFrame, "TOPRIGHT", -expansionButtonInset, -expansionButtonInset)
   topRightExpansion:SetWidth(expansionButtonSize)
   topRightExpansion:SetHeight(expansionButtonSize)
   topRightExpansion:SetScript("OnMouseUp",
      function()
         local buffer = savedVars.secondaryExpansion
         savedVars.secondaryExpansion = savedVars.primaryExpansion
         savedVars.primaryExpansion = buffer
         Luggage:UpdatePlacementFrame()
      end
   )
   topRightExpansion:SetScript("OnEnter", ExpansionButtons_OnEnter)
   topRightExpansion:SetScript("OnLeave", ExpansionButtons_OnLeave)
   topRightText = topRightExpansion:CreateFontString(nil, "ARTWORK", "GameFontNormal")
   topRightText:SetPoint("TOPRIGHT")
   topRightText:SetWidth(textSize)
   topRightText:SetHeight(textSize)
   topRightIcon = topRightExpansion:CreateTexture(nil, "ARTWORK")
   topRightIcon:SetPoint("BOTTOMLEFT")
   topRightIcon:SetWidth(iconSize)
   topRightIcon:SetHeight(iconSize)
   topRightIcon:SetTexture("Interface\\ChatFrame\\UI-ChatIcon-ScrollUp-Up")
   -- Bottom Left
   bottomLeftExpansion = GetBackdroppedFrame("BottomLeftExpansion", positionFrame)
   bottomLeftExpansion:SetPoint("BOTTOMLEFT", positionFrame, "BOTTOMLEFT", expansionButtonInset, expansionButtonInset)
   bottomLeftExpansion:SetWidth(expansionButtonSize)
   bottomLeftExpansion:SetHeight(expansionButtonSize)
   bottomLeftExpansion:SetScript("OnMouseUp",
      function()
         local buffer = savedVars.secondaryExpansion
         savedVars.secondaryExpansion = savedVars.primaryExpansion
         savedVars.primaryExpansion = buffer
         Luggage:UpdatePlacementFrame()
      end
   )
   bottomLeftExpansion:SetScript("OnEnter", ExpansionButtons_OnEnter)
   bottomLeftExpansion:SetScript("OnLeave", ExpansionButtons_OnLeave)
   bottomLeftText = bottomLeftExpansion:CreateFontString(nil, "ARTWORK", "GameFontNormal")
   bottomLeftText:SetPoint("BOTTOMLEFT")
   bottomLeftText:SetWidth(textSize)
   bottomLeftText:SetHeight(textSize)
   bottomLeftIcon = bottomLeftExpansion:CreateTexture(nil, "ARTWORK")
   bottomLeftIcon:SetPoint("TOPRIGHT")
   bottomLeftIcon:SetWidth(iconSize)
   bottomLeftIcon:SetHeight(iconSize)
   bottomLeftIcon:SetTexture("Interface\\ChatFrame\\UI-ChatIcon-ScrollUp-Up")
   -- Bottom Right
   bottomRightExpansion = GetBackdroppedFrame("BottomRightExpansion", positionFrame)
   bottomRightExpansion:SetPoint("BOTTOMRIGHT", positionFrame, "BOTTOMRIGHT", -expansionButtonInset, expansionButtonInset)
   bottomRightExpansion:SetWidth(expansionButtonSize)
   bottomRightExpansion:SetHeight(expansionButtonSize)
   bottomRightExpansion:SetScript("OnMouseUp",
      function()
         local buffer = savedVars.secondaryExpansion
         savedVars.secondaryExpansion = savedVars.primaryExpansion
         savedVars.primaryExpansion = buffer
         Luggage:UpdatePlacementFrame()
      end
   )
   bottomRightExpansion:SetScript("OnEnter", ExpansionButtons_OnEnter)
   bottomRightExpansion:SetScript("OnLeave", ExpansionButtons_OnLeave)
   bottomRightText = bottomRightExpansion:CreateFontString(nil, "ARTWORK", "GameFontNormal")
   bottomRightText:SetPoint("BOTTOMRIGHT")
   bottomRightText:SetWidth(textSize)
   bottomRightText:SetHeight(textSize)
   bottomRightIcon = bottomRightExpansion:CreateTexture(nil, "ARTWORK")
   bottomRightIcon:SetPoint("TOPLEFT")
   bottomRightIcon:SetWidth(iconSize)
   bottomRightIcon:SetHeight(iconSize)
   bottomRightIcon:SetTexture("Interface\\ChatFrame\\UI-ChatIcon-ScrollUp-Up")
end


local mainColor = {r = 0.0, g = 0.0, b = 0.7, a = 0.5}
local selectedCornerColor = {r = 0.0, g = 0.7, b = 0.0, a = 0.5}
local deselectedCornerColor = {r = 0.7, g = 0.0, b = 0.0, a = 0.5}
local selectedSizingHandleColor = {r = 0.0, g = 0.5, b = 0.0, a = 0.5}
local deselectedSizingHandleColor = {r = 0.5, g = 0.0, b = 0.0, a = 0.5}

function Luggage:CreatePlacementFrame()
   if self.placementFrame then return end
   local positionFrame = CreatePositionFrame()
   CreateSizingHandles()
   CreateCorners()
   CreateExpansionButtons()
   self.placementFrame = positionFrame
end

local function SetFrameColor(frame, color)
   frame:SetBackdropColor(color.r, color.g, color.b)
end

local s2 = sqrt(2);
local cos, sin, rad = math.cos, math.sin, math.rad;
local function CalculateCorner(angle)
	local r = rad(angle);
	return 0.5 + cos(r) / s2, 0.5 + sin(r) / s2;
end
local function RotateIcon(texture, angle)
	local LRx, LRy = CalculateCorner(angle + 45);
	local LLx, LLy = CalculateCorner(angle + 135);
	local ULx, ULy = CalculateCorner(angle + 225);
	local URx, URy = CalculateCorner(angle - 45);
	
	texture:SetTexCoord(ULx, ULy, LLx, LLy, URx, URy, LRx, LRy);
end

--~ local function RotateIcon(icon, degrees)
--~ 	local angle = math.rad(degrees)
--~ 	local cos, sin = math.cos(angle), math.sin(angle)
--~ 	icon:SetTexCoord(sin - cos, -(cos + sin), -cos, -sin, sin, -cos, 0, 0)
--~ end


local function UpdateExpansionButtons(selected, primary, secondary)
   if selected == "TOPLEFT" then
      topLeftExpansion:Hide()
      bottomRightExpansion:Hide()
      topRightExpansion:Show()
      bottomLeftExpansion:Show()
      RotateIcon(topRightIcon, -90)
      RotateIcon(bottomLeftIcon, 180)
      if primary == "RIGHT" then
         topRightText:SetText("1")
         bottomLeftText:SetText("2")
      elseif primary == "BOTTOM" then
         topRightText:SetText("2")
         bottomLeftText:SetText("1")
      end
   elseif selected == "TOPRIGHT" then
      topLeftExpansion:Show()
      bottomRightExpansion:Show()
      topRightExpansion:Hide()
      bottomLeftExpansion:Hide()
      RotateIcon(topLeftIcon, 90)
      RotateIcon(bottomRightIcon, 180)
      if primary == "LEFT" then
         topLeftText:SetText("1")
         bottomRightText:SetText("2")
      elseif primary == "BOTTOM" then
         topLeftText:SetText("2")
         bottomRightText:SetText("1")
      end
   elseif selected == "BOTTOMRIGHT" then
      topLeftExpansion:Hide()
      bottomRightExpansion:Hide()
      topRightExpansion:Show()
      bottomLeftExpansion:Show()
      RotateIcon(topRightIcon, 0)
      RotateIcon(bottomLeftIcon, 90)
      if primary == "TOP" then
         topRightText:SetText("1")
         bottomLeftText:SetText("2")
      elseif primary == "LEFT" then
         topRightText:SetText("2")
         bottomLeftText:SetText("1")
      end
   elseif selected == "BOTTOMLEFT" then
      topLeftExpansion:Show()
      bottomRightExpansion:Show()
      topRightExpansion:Hide()
      bottomLeftExpansion:Hide()
      RotateIcon(topLeftIcon, 0)
      RotateIcon(bottomRightIcon, -90)
      if primary == "RIGHT" then
         bottomRightText:SetText("1")
         topLeftText:SetText("2")
      elseif primary == "TOP" then
         bottomRightText:SetText("2")
         topLeftText:SetText("1")
      end
   end
end

local function UpdateCorners(selected)
   if selected == "TOPLEFT" then
      SetFrameColor(topLeftCorner, selectedCornerColor)
      SetFrameColor(topRightCorner, deselectedCornerColor)
      SetFrameColor(bottomLeftCorner, deselectedCornerColor)
      SetFrameColor(bottomRightCorner, deselectedCornerColor)
   elseif selected == "TOPRIGHT" then
      SetFrameColor(topLeftCorner, deselectedCornerColor)
      SetFrameColor(topRightCorner, selectedCornerColor)
      SetFrameColor(bottomLeftCorner, deselectedCornerColor)
      SetFrameColor(bottomRightCorner, deselectedCornerColor)
   elseif selected == "BOTTOMLEFT" then
      SetFrameColor(topLeftCorner, deselectedCornerColor)
      SetFrameColor(topRightCorner, deselectedCornerColor)
      SetFrameColor(bottomLeftCorner, selectedCornerColor)
      SetFrameColor(bottomRightCorner, deselectedCornerColor)
   elseif selected == "BOTTOMRIGHT" then
      SetFrameColor(topLeftCorner, deselectedCornerColor)
      SetFrameColor(topRightCorner, deselectedCornerColor)
      SetFrameColor(bottomLeftCorner, deselectedCornerColor)
      SetFrameColor(bottomRightCorner, selectedCornerColor)
   end
end

local function UpdateSizingHandles(primaryExpansion)
   if primaryExpansion == "TOP" then
      SetFrameColor(topSizingHandle, selectedSizingHandleColor)
      SetFrameColor(bottomSizingHandle, deselectedSizingHandleColor)
      SetFrameColor(rightSizingHandle, deselectedSizingHandleColor)
      SetFrameColor(leftSizingHandle, deselectedSizingHandleColor)
   elseif primaryExpansion == "BOTTOM" then
      SetFrameColor(topSizingHandle, deselectedSizingHandleColor)
      SetFrameColor(bottomSizingHandle, selectedSizingHandleColor)
      SetFrameColor(rightSizingHandle, deselectedSizingHandleColor)
      SetFrameColor(leftSizingHandle, deselectedSizingHandleColor)
   elseif primaryExpansion == "RIGHT" then
      SetFrameColor(topSizingHandle, deselectedSizingHandleColor)
      SetFrameColor(bottomSizingHandle, deselectedSizingHandleColor)
      SetFrameColor(rightSizingHandle, selectedSizingHandleColor)
      SetFrameColor(leftSizingHandle, deselectedSizingHandleColor)
   elseif primaryExpansion == "LEFT" then
      SetFrameColor(topSizingHandle, deselectedSizingHandleColor)
      SetFrameColor(bottomSizingHandle, deselectedSizingHandleColor)
      SetFrameColor(rightSizingHandle, deselectedSizingHandleColor)
      SetFrameColor(leftSizingHandle, selectedSizingHandleColor)
   end
end

function Luggage:UpdatePlacementFrame()
   local savedVars = self.db.profile.placement
   local scale =  UIParent:GetEffectiveScale()
   local xOffs = savedVars.xOffs * scale
   local yOffs = savedVars.yOffs * scale
   local width = savedVars.width * scale
   local height = savedVars.height * scale
   local anchor = savedVars.anchor
   local primaryExpansion = savedVars.primaryExpansion
   local secondaryExpansion = savedVars.secondaryExpansion
   if anchor == "TOPLEFT" then
      xOffs = xOffs + edgeWidth
      yOffs = yOffs - edgeWidth
   elseif anchor == "TOPRIGHT" then
      xOffs = xOffs - edgeWidth
      yOffs = yOffs - edgeWidth
   elseif anchor == "BOTTOMRIGHT" then
      xOffs = xOffs - edgeWidth
      yOffs = yOffs + edgeWidth
   elseif anchor == "BOTTOMLEFT" then
      xOffs = xOffs + edgeWidth
      yOffs = yOffs + edgeWidth
   end
   positionFrame:ClearAllPoints()
   positionFrame:SetPoint(anchor, UIParent, anchor, xOffs, yOffs)
   positionFrame:SetWidth(width - 2*edgeWidth)
   positionFrame:SetHeight(height - 2*edgeWidth)
   SetFrameColor(positionFrame, mainColor)
   UpdateCorners(anchor)
   UpdateSizingHandles(primaryExpansion)
   UpdateExpansionButtons(anchor, primaryExpansion, secondaryExpansion)
   Luggage:ArrangeBags()
end

local usedPrimaryLength = 0
local nWraps = 0
local nBagsInCurrentLine = 0
local usedSecondaryLength = 0
local previousUsedSecondaryLength = 0
local bagPadding
local function AvailableLengthSuffices(bag)
   local sv = Luggage.db.profile.placement
   if sv.primaryExpansion == "LEFT" or sv.primaryExpansion == "RIGHT" then
      return usedPrimaryLength + bag:GetWidth() + (nBagsInCurrentLine-1)*bagPadding < sv.width
   else
      return usedPrimaryLength + bag:GetHeight() + (nBagsInCurrentLine-1)*bagPadding < sv.height
   end
end

local function GetXOffset()
   local sv = Luggage.db.profile.placement
   local primaryExpansion = sv.primaryExpansion
   local secondaryExpansion = sv.secondaryExpansion
   local xOffs
   if primaryExpansion == "LEFT" then
      xOffs = sv.xOffs - (usedPrimaryLength + bagPadding*nBagsInCurrentLine)
   elseif primaryExpansion == "RIGHT" then
      xOffs = sv.xOffs + (usedPrimaryLength + bagPadding*nBagsInCurrentLine)
   elseif secondaryExpansion == "LEFT" then
      xOffs = sv.xOffs - (previousUsedSecondaryLength + bagPadding*nWraps)
   elseif secondaryExpansion == "RIGHT" then
      xOffs = sv.xOffs + (previousUsedSecondaryLength + bagPadding*nWraps)
   end
   return xOffs*UIParent:GetEffectiveScale()
end

local function GetYOffset()
   local sv = Luggage.db.profile.placement
   local primaryExpansion = sv.primaryExpansion
   local secondaryExpansion = sv.secondaryExpansion
   local yOffs
   if primaryExpansion == "TOP" then
      yOffs = sv.yOffs + (usedPrimaryLength + bagPadding*nBagsInCurrentLine)
   elseif primaryExpansion == "BOTTOM" then
      yOffs = sv.yOffs - (usedPrimaryLength + bagPadding*nBagsInCurrentLine)
   elseif secondaryExpansion == "TOP" then
      yOffs = sv.yOffs + (previousUsedSecondaryLength + bagPadding*nWraps)
   elseif secondaryExpansion == "BOTTOM" then
      yOffs = sv.yOffs - (previousUsedSecondaryLength + bagPadding*nWraps)
   end
   return yOffs*UIParent:GetEffectiveScale()
end

local function UpdateUsedLengths(bag)
   local sv = Luggage.db.profile.placement
   if sv.primaryExpansion == "TOP" or sv.primaryExpansion == "BOTTOM" then
      usedPrimaryLength = usedPrimaryLength + bag:GetHeight()
      usedSecondaryLength  = math.max(usedSecondaryLength, bag:GetWidth())
   else
      usedPrimaryLength = usedPrimaryLength + bag:GetWidth()
      usedSecondaryLength  = math.max(usedSecondaryLength, bag:GetHeight())
   end
end
local function GetNextBagAnchor(bag)
   local sv = Luggage.db.profile.placement
   if usedPrimaryLength ~= 0 and not AvailableLengthSuffices(bag) then
      -- We need to wrap arround
      nWraps = nWraps + 1
      usedPrimaryLength = 0
      previousUsedSecondaryLength = previousUsedSecondaryLength + usedSecondaryLength
      usedSecondaryLength = 0
      nBagsInCurrentLine = 0
   end
   local xOffs,yOffs = GetXOffset(), GetYOffset()
   UpdateUsedLengths(bag)
   return sv.anchor, UIParent, sv.anchor, xOffs, yOffs
end

function Luggage:ArrangeBags()
   usedPrimaryLength = 0
   nWraps = 0
   nBagsInCurrentLine = 0
   usedSecondaryLength = 0
   previousUsedSecondaryLength = 0
   bagPadding = 10 * UIParent:GetEffectiveScale()

   for bag in pairs(self.bags) do
      if bag:IsShown() and not bag:IsUserPlaced() then
         local point, parent, relativePoint, xOffs, yOffs = GetNextBagAnchor(bag)
         bag:ClearAllPoints()
         bag:SetPoint(point, parent, relativePoint, xOffs, yOffs)
         nBagsInCurrentLine = nBagsInCurrentLine + 1
      end
   end
end

function Luggage:ShowPlacementFrame()
   self:CreatePlacementFrame()
   self:UpdatePlacementFrame()
   self.placementFrame:Show()
end
