--[[
	main.lua
		Calculate and display item scores on tooltips.
--]]

ItemScore = LibStub("AceAddon-3.0"):NewAddon("ItemScore", 'AceConsole-3.0')

-- Libraries --
local TipHooker = LibStub("LibTipHooker-1.1")
local StatLogic = LibStub("LibStatLogic-1.1")

-- Local data --

--[[ ItemScore functions ]]--

function ItemScore:CheckDB()
	if not self.db.char.scoreSets then
		-- create default score set
		self.db.char.scoreSets = {}
		
		self.db.char.scoreSets["Default"] = self:CreateEmptyScoreSet("Default")
	end
end

function ItemScore:CreateEmptyScoreSet(name)
	local ret = {}
	
	ret.name = name
	ret.display = 1
	ret.r = 0
	ret.g = 0
	ret.b = 1 -- default to blue
	ret.pattern = "%n: %a" -- Default
	ret.precision = 2
	ret.info = {}
	
	return ret
end

function ItemScore:InitStats()
	-- store list of StatLogic stat names to provide scores for
	self.supportedStats = 
	{
		["EMPTY_SOCKET_RED"] = true,
		["EMPTY_SOCKET_YELLOW"] = true,
		["EMPTY_SOCKET_BLUE"] = true,
		["EMPTY_SOCKET_META"] = true,

		["STR"] = true,
		["AGI"] = true,
		["STA"] = true,
		["INT"] = true,
		["SPI"] = true,
		["ARMOR"] = true,
		["ARMOR_BONUS"] = true,

		["FIRE_RES"] = true,
		["NATURE_RES"] = true,
		["FROST_RES"] = true,
		["SHADOW_RES"] = true,
		["ARCANE_RES"] = true,

		["BLOCK_VALUE"] = true,

		["AP"] = true,
		["RANGED_AP"] = true,
		["FERAL_AP"] = true,
		
		["HEAL"] = true,

		["SPELL_DMG"] = true,
		
		["SPELLPEN"] = true,

		["HEALTH"] = true,
		["MANA"] = true,
		["HEALTH_REG"] = true,
		["MANA_REG"] = true,

		["MAX_DAMAGE"] = true,
		["DPS"] = true,

		["DEFENSE_RATING"] = true,
		["DODGE_RATING"] = true,
		["PARRY_RATING"] = true,
		["BLOCK_RATING"] = true,
		["MELEE_HIT_RATING"] = true,
		--["RANGED_HIT_RATING"] = true,
		--["SPELL_HIT_RATING"] = true,
		["MELEE_CRIT_RATING"] = true,
		--["RANGED_CRIT_RATING"] = true,
		--["SPELL_CRIT_RATING"] = true,
		["RESILIENCE_RATING"] = true,
		["MELEE_HASTE_RATING"] = true,
		--["RANGED_HASTE_RATING"] = true,
		--["SPELL_HASTE_RATING"] = true,
		["EXPERTISE_RATING"] = true,
		["ARMOR_PENETRATION_RATING"] = true
	}
	
	-- now set the values of each index to readable values from stat logic
	for i in pairs(self.supportedStats) do
		self.supportedStats[i] = StatLogic:GetStatNameFromID(i)
	end
end

function ItemScore:OnInitialize()
	self.db = LibStub("AceDB-3.0"):New("ItemScoreDB")
	self:InitStats()
	self:CheckDB()
	
	self.options = {}
	self:InitOptions()
	self:RegisterSlashCommands()
	
	LibStub("AceConfig-3.0"):RegisterOptionsTable("ItemScore", self.options)
	self.optionsFrame = LibStub("AceConfigDialog-3.0"):AddToBlizOptions("ItemScore", "ItemScore")
end

function ItemScore:OnEnable()
    -- Called when the addon is enabled
    
    -- Hook item tooltips
	TipHooker:Hook(self.ProcessTooltip, "item")
end

function ItemScore:OnDisable()
    -- Called when the addon is disabled
    
    -- Unhook item tooltips
	TipHooker:Unhook(self.ProcessTooltip, "item")
end

--[[ Local Functions ]]--

--[[
	The Code
--]]

function ItemScore:GetTotalCurrentStats(currentStats, stat)
	-- calculate total stats of currently worn equipment
	if currentStats[stat] then return end
	
	currentStats[stat] = 0
	
	for i = 0, 19 do
		local link = GetInventoryItemLink("player", i)
		local itemTable = {}
		StatLogic:GetSum(link, itemTable)
		if itemTable[stat] then
			currentStats[stat] = currentStats[stat] + itemTable[stat]
		end
	end
end

local function getCappedValue(cap, currentTotal, value)
	-- returns the amount of value counted such that currentTotal + value <= cap
	local statLeft = math.max(0, cap - currentTotal)
	
	return math.min(value, statLeft)
end

function ItemScore:CalculateItemScore(itemTable, diff1, diff2, scoreSet, currentStats)
	local score = 0
	local score2 = 0
	
	local equippedScore = 0
	local equippedScore2 = 0
	
	for i in pairs(self.supportedStats) do
		local thisScore = 0
		local thisCap = -1
		
		if scoreSet.info[i] then
			thisScore = scoreSet.info[i].score
			thisCap = scoreSet.info[i].cap
		elseif scoreSet.parent and ItemScore.db.char.scoreSets[scoreSet.parent] and ItemScore.db.char.scoreSets[scoreSet.parent].info[i] then
			thisScore = ItemScore.db.char.scoreSets[scoreSet.parent].info[i].score
			thisCap = ItemScore.db.char.scoreSets[scoreSet.parent].info[i].cap
		end 
		
		if thisScore ~= 0 then
			if not itemTable[i] then
				itemTable[i] = 0
			end
			-- calculate current stats for those with caps
			
			-- calculate for current equipment too
			local current1Val = 0
			local current2Val = 0
			
			if diff1 then
				if not diff1[i] then
					-- no difference
					diff1[i] = 0
				end
				
				current1Val = itemTable[i] - diff1[i]
			else
				current1Val = 0	
			end
			
			current2Val = current1Val
			
			if diff2 then
				if not diff2[i] then
					-- no difference
					diff2[i] = 0
				end
				
				current2Val = itemTable[i] - diff2[i]
			end
			
			if thisCap ~= -1 then
				-- calculate capped value
				-- get current stat (as needed)
				self:GetTotalCurrentStats(currentStats, i)
				
				-- determine current value(s) of currently equipped items
				local add = 0
				local equipAdd = 0
				
				-- get capped value
				add = getCappedValue(thisCap, currentStats[i] - current1Val, itemTable[i]) * thisScore
				score = score + add
				
				-- do for equipped item as well
				equipAdd = getCappedValue(thisCap, currentStats[i] - current1Val, current1Val) * thisScore
				equippedScore = equippedScore + equipAdd
			
				if current2Val ~= current1Val then
					add = getCappedValue(thisCap, currentStats[i] - current2Val, itemTable[i]) * thisScore
					
					-- do for equipped item as well
					equipAdd = getCappedValue(thisCap, currentStats[i] - current2Val, current2Val) * thisScore
				end
				
				score2 = score2 + add
				equippedScore2 = equippedScore2 + equipAdd

			else
				local add = thisScore * itemTable[i]
				score = score + add 
				score2 = score2 + add
				equippedScore = equippedScore + (thisScore * current1Val)
				equippedScore2 = equippedScore2 + (thisScore * current2Val)
			end
		end   
	end
	
	return score, score2, equippedScore, equippedScore2
end

function ItemScore:MakeTooltipText(scoreSet, score, score2, equippedScore, equippedScore2)
	
	local scoreText = string.format("%." .. scoreSet.precision .. "f", score)
			
	if score ~= score2 then
		scoreText = scoreText .. "/" .. string.format("%." .. scoreSet.precision .. "f", score2)
	end
	
	local diff = score - equippedScore
	local sign = ""
	local signColor = "|cFFFF0000"
	
	if diff >= 0 then
		sign = "+"
		signColor = "|cFF00FF00"
	end
	
	local equippedDiffText = signColor .. sign .. string.format("%." .. scoreSet.precision .. "f", diff) .. "|r"
	
	if score ~= score2 or equippedScore ~= equippedScore2 then
		diff = score2 - equippedScore2
		if diff >= 0 then
			sign = "+"
			signColor = "|cFF00FF00"
		else
			sign = ""
			signColor = "|cFFFF0000"
		end
			
		equippedDiffText = equippedDiffText .. "/" .. signColor ..  sign .. string.format("%." .. scoreSet.precision .. "f", diff) .. "|r"
	end
			
	local ret = scoreSet.pattern
	
	ret = string.gsub(ret, "%%n", scoreSet.name)
	ret = string.gsub(ret, "%%a", scoreText)
	ret = string.gsub(ret, "%%d", equippedDiffText)
	
	return ret
end

function ItemScore.ProcessTooltip(tooltip, name, link, ...)
	local currentStats = {}
	local itemTable = {}	
	
	-- get the item stats
	StatLogic:GetSum(tooltip, itemTable)
	
	-- check if we're a gem
	local isGem = nil
	if (not itemTable.itemType or itemTable.itemType == "") and itemTable.link then
		local _, _, _, _, _, itemType = GetItemInfo(itemTable.link)
		
		if itemType == "Gem" then
			isGem = true
		else
			return
		end
	end
	
	-- if we're not equippable or we are tabard, bag or quiver, then don't show scores
	if not isGem and itemTable and (itemTable.itemType == "" or itemTable.itemType == "INVTYPE_TABARD" or itemTable.itemType == "INVTYPE_BAG" or
		itemTable.itemType == "INVTYPE_QUIVER") then
		return
	end
	
	if itemTable.itemType and itemTable.itemType ~= "" then
		-- by default, we'll score and compare items in their unenchanted state
		local unchantedLink = StatLogic:RemoveEnchant(itemTable.link)
		
		if unchantedLink ~= itemTable.link then
			itemTable.link = unchantedLink
			StatLogic:GetSum(itemTable.link, itemTable)
		end
	end
	
	-- get difference in stat between equipped and this equipment
	local equipDiff1, equipDiff2 = StatLogic:GetDiff(tooltip, nil, nil, true)
	
	for i in pairs(ItemScore.db.char.scoreSets) do
		if ItemScore.db.char.scoreSets[i].display then
			local setName = ItemScore.db.char.scoreSets[i].name
			
			-- calculate the score
			local score, score2, equippedScore, equippedScore2 = ItemScore:CalculateItemScore(itemTable, equipDiff1, equipDiff2, ItemScore.db.char.scoreSets[i], currentStats)
			
			-- display the score
			tooltip:AddLine(ItemScore:MakeTooltipText(ItemScore.db.char.scoreSets[i], score, score2, equippedScore, equippedScore2), ItemScore.db.char.scoreSets[i].r, ItemScore.db.char.scoreSets[i].g, ItemScore.db.char.scoreSets[i].b)
			tooltip:Show()
		end
	end
end

function ItemScore:PrintTotalScores()
	local currentStats = {}
	self:Print("Total scores:")
	
	for j in pairs(ItemScore.db.char.scoreSets) do
		local totalScore = 0
		local scoreSet = ItemScore.db.char.scoreSets[j]
		
		for i in pairs(self.supportedStats) do
			local thisScore = 0
			local thisCap = -1
			
			if scoreSet.info[i] then
				thisScore = scoreSet.info[i].score
				thisCap = scoreSet.info[i].cap
			elseif scoreSet.parent and ItemScore.db.char.scoreSets[scoreSet.parent] and ItemScore.db.char.scoreSets[scoreSet.parent].info[i] then
				thisScore = ItemScore.db.char.scoreSets[scoreSet.parent].info[i].score
				thisCap = ItemScore.db.char.scoreSets[scoreSet.parent].info[i].cap
			end 
			if thisScore ~= 0 then
				self:GetTotalCurrentStats(currentStats, i)
				
				if thisCap ~= -1 then
					totalScore = totalScore + (thisScore * getCappedValue(thisCap, 0, currentStats[i]))
				else
					totalScore = totalScore + (thisScore * currentStats[i])
				end
			end
		end
		
		self:Print(scoreSet.name .. ": " .. string.format("%." .. scoreSet.precision .. "f", totalScore))
	end
end

function ItemScore:OnCmd(args)
	local cmd = string.split(' ', args):lower() or args:lower()
	local restOfString = string.gsub(args, cmd .. ' ', '')

	self:PrintTotalScores()
end
