
--[[ Configuration functions ]]--

function ItemScore:InitOptions()
	
	self.selectedSet = nil
	
	for i in pairs(self.db.char.scoreSets) do
		self.selectedSet = self.db.char.scoreSets[i]
		break
	end
	
	self.options = {
		desc = "ItemScore",
		type = "group",
		handler = ItemScore,
		
		args = {
			scoreSets = {
				type = "select",
				style = "dropdown",
				name = "ItemScore set",
				order = 10,
				values = {},
				get = "GetSelectedSet",
				set = "SetSelectedSet",
			},
			newName = {
			    type = "input",
			    name = "Name",
			    order = 20,
			    get = "GetNewName",
			    set = "SetNewName",
			},
			newSet = {
			    type = "execute",
			    name = "New",
			    order = 30,
			    func = "CreateNewSet",
			    width = "half",
			},
			copySet = {
			    type = "execute",
			    name = "Duplicate",
			    order = 40,
			    func = "CopySet",
			    width = "half",
			},
			renameSet = {
				type = "execute",
			    name = "Rename",
			    order = 50,
			    func = "RenameSet",
			    width = "half",
			},
			deleteSet = {
			    type = "execute",
			    name = "Delete",
			    order = 60,
			    func = "DeleteSet",
			    width = "half",
			},
			setEnabled = {
				type = "toggle",
				name = "Display",
				order = 70,
				get = "GetEnabled",
				set = "SetEnabled",
			},
			setParent = {
				type = "select",
				style = "dropdown",
				name = "Parent set",
				order = 75,
				values = {},
				get = "GetParent",
				set = "SetParent",
			},
			setPrecision = {
				type = "input",
			    name = "Precision",
			    order = 80,
			    get = "GetPrecision",
			    set = "SetPrecision",
			    width = "half",
			},
			setColour = {
				type = "color",
				name = "Colour",
				order = 90,
				get = "GetColour",
				set = "SetColour",
			},
			setPattern = {
				type = "input",
			    name = "Display text",
			    order = 100,
			    get = "GetPattern",
			    set = "SetPattern",
			},
			exampleText = {
				type = "description",
				name = "",
				order = 110,
			},
			statTable = {
				name = "Stat scores",
				type = "group",
				order = 120,
				args = {},
				width = "half",
			},
			availableStats = {
				name = "Available stats",
				type = "group",
				order = 130,
				args = {},
				width = "half",
			}
		},
	}
	
	self:UpdateStatTable(self.options.args, self.selectedSet)
	self:UpdateSetList(self.options.args.scoreSets.values, self.db.char.scoreSets, nil, false)
	self:UpdateSetList(self.options.args.setParent.values, self.db.char.scoreSets, self.selectedSet, true)
end

function ItemScore:RegisterSlashCommands()
	self:RegisterChatCommand('itemscore', 'OnCmd')
end

function ItemScore:UpdateStatTable(optionList, statList)
	-- if nothing selected, clear out the table
	if not statList then
		for i in pairs(optionList.statTable.args) do
			if optionList.statTable.args[i].type == "group" then
				optionList.statTable.args[i] = nil
			end
		end
		for i in pairs(optionList.availableStats.args) do
			if optionList.availableStats.args[i].type == "group" then
				optionList.availableStats.args[i] = nil
			end
		end
	else
		for i in pairs(statList.info) do
			local capVal = ""
			
			if statList.info[i].cap and statList.info[i].cap ~= -1 then
				capVal = " (<=" .. statList.info[i].cap .. ")"
			end
			
			if not optionList.statTable.args[i] then
			 
				local options = {
					score = {
						name = "Score",
						type = "input",
						order = 110,
						set = "SetScore",
						get = "GetScore",
						arg = i					
					},
					cap = {
						name = "Cap",
						type = "input",
						order = 200,
						set = "SetCap",
						get = "GetCap",
						arg = i
					},
					remove = {
						type = "execute",
					    name = "Remove",
					    order = 210,
					    func = "RemoveStat",
					    arg = i,
					}
				}
							
				-- quietly remove stat if its not supported
				if not self.supportedStats[i] then
					statList.info[i] = nil
				else
					optionList.statTable.args[i] = {
						name = self.supportedStats[i] .. " - " .. statList.info[i].score .. capVal,
						type = "group",
						width = "half",
						args = options
					}
				end
			else
				-- just update the name
				optionList.statTable.args[i].name = self.supportedStats[i] .. " - " .. statList.info[i].score .. capVal
			end
		end
		
		-- remove stats that are no longer in the list
		for i in pairs(optionList.statTable.args) do
			if not statList.info[i] then
				optionList.statTable.args[i] = nil
			end
		end
		
		-- now populate another list with stats that aren't in the set
		for i in pairs(self.supportedStats) do
			if not statList.info[i] then
				local options = {
					add = {
						type = "execute",
					    name = "Add",
					    order = 210,
					    func = "AddStat",
					    arg = i,
					}
				}
						
				optionList.availableStats.args[i] = {
					name = self.supportedStats[i],
					type = "group",
					width = "half",
					args = options
				}
			end
		end
		
		-- remove stats that are in the set
		for i in pairs(optionList.availableStats.args) do
			if statList.info[i] then
				optionList.availableStats.args[i] = nil
			end
		end
	end
end

function ItemScore:UpdateSetList(optionList, setList, ignoreSet, addEmpty)
	-- add new sets that aren't in optionList
	for i in pairs(setList) do
		if not optionList[i] then
			optionList[i] = i
		end
	end
	
	-- remove from optionList spells that aren't in spellList
	for i in pairs(optionList) do
		if not setList[i] then
			optionList[i] = nil
		end
	end
	
	if ignoreSet then
	    optionList[ignoreSet.name] = nil
	end
	
	self:UpdateExampleText(self.options.args.exampleText, self.selectedSet)
	
	if addEmpty then
	    optionList[""] = ""
	end
end

function ItemScore:UpdateExampleText(exampleText, selectedSet)
	if not selectedSet then
		exampleText.name = ""
		return
	end
	
	local score = 123.456789
	local equippedScore = 125.567891
	
	exampleText.name = ItemScore:MakeTooltipText(selectedSet, score, score, equippedScore, equippedScore)
end

-- Utility functions

local function deepcopy(object)
    local lookup_table = {}
    local function _copy(object)
        if type(object) ~= "table" then
            return object
        elseif lookup_table[object] then
            return lookup_table[object]
        end
        local new_table = {}
        lookup_table[object] = new_table
        for index, value in pairs(object) do
            new_table[_copy(index)] = _copy(value)
        end
        return setmetatable(new_table, getmetatable(object))
    end
    return _copy(object)
end

-- Handlers

function ItemScore:GetSelectedSet(info, value)
	-- return the selected set name
	if self.selectedSet then
		return self.selectedSet.name
	end
	
	return nil
end

function ItemScore:SetSelectedSet(info, value)
	if not value then
		self.selectedSet = nil
	else
		self.selectedSet = self.db.char.scoreSets[value]
	end
	
	self:UpdateStatTable(self.options.args, self.selectedSet)
	self:UpdateExampleText(self.options.args.exampleText, self.selectedSet)
	self:UpdateSetList(self.options.args.setParent.values, self.db.char.scoreSets, self.selectedSet, true)
end

function ItemScore:GetNewName(info, value)
	return self.newName
end

function ItemScore:SetNewName(info, value)
	self.newName = value
end

function ItemScore:CreateNewSet()
	if not self.newName or self.newName == '' then
		-- display error here
	else
		self.db.char.scoreSets[self.newName] = self:CreateEmptyScoreSet(self.newName)
		self:SetSelectedSet(nil, self.newName)
		self:UpdateSetList(self.options.args.scoreSets.values, self.db.char.scoreSets, nil, false)
	end 
end

function ItemScore:CopySet()
	if not self.newName or self.newName == '' then
		-- display error here
	elseif not self.selectedSet then
		-- display error here
	else
		self.db.char.scoreSets[self.newName] = deepcopy(self.selectedSet)
		self.db.char.scoreSets[self.newName].name = self.newName 
		self:SetSelectedSet(nil, self.newName)
		self:UpdateSetList(self.options.args.scoreSets.values, self.db.char.scoreSets, nil, false)
	end 
end

function ItemScore:RenameSet()
	if not self.newName or self.newName == '' then
		-- display error here
	elseif not self.selectedSet then
		-- display error here
	else
		self.db.char.scoreSets[self.newName] = self.selectedSet
		self.db.char.scoreSets[self.selectedSet.name] = nil
		
		-- need to check all sets and ensure that any that use it as a parent have the parent name changed
		for i in pairs(self.db.char.scoreSets) do
			if self.db.char.scoreSets[i].parent == self.selectedSet.name then
				self.db.char.scoreSets[i].parent = self.newName
			end
		end
		
		self.selectedSet.name = self.newName
		self:UpdateSetList(self.options.args.scoreSets.values, self.db.char.scoreSets, nil, false)
	end 
end

function ItemScore:DeleteSet()
	if not self.selectedSet then
		-- display error here
	else
		self.db.char.scoreSets[self.selectedSet.name] = nil
		
		-- need to check all sets and ensure that any that use it as a parent have the parent blanked
		for i in pairs(self.db.char.scoreSets) do
			if self.db.char.scoreSets[i].parent == self.selectedSet.name then
				self.db.char.scoreSets[i].parent = nil
			end
		end
		
		self:SetSelectedSet(nil, nil)
		self:UpdateSetList(self.options.args.scoreSets.values, self.db.char.scoreSets, nil, false)
	end 
end

function ItemScore:SetScore(info, value)
	if not self.selectedSet then
		DEFAULT_CHAT_FRAME:AddMessage("Cannot set value with no selected set")
	else
		self.selectedSet.info[info.arg].score = value
		self:UpdateStatTable(self.options.args, self.selectedSet)
	end 
end

function ItemScore:GetScore(info, value)
	if not self.selectedSet or not self.selectedSet.info[info.arg] then
		-- display error here
		return 0
	else
		return self.selectedSet.info[info.arg].score
	end
end

function ItemScore:SetCap(info, value)
	if not self.selectedSet then
		DEFAULT_CHAT_FRAME:AddMessage("Cannot set cap value with no selected set")
	else
		if value == "" then
			value = -1
		end
		self.selectedSet.info[info.arg].cap = value
		self:UpdateStatTable(self.options.args, self.selectedSet)
	end 
end

function ItemScore:GetCap(info, value)
	if not self.selectedSet or not self.selectedSet.info[info.arg] then
		-- display error here
		return ""
	else
		if self.selectedSet.info[info.arg].cap == -1 then
			return ""
		else
			return self.selectedSet.info[info.arg].cap
		end
	end
end

function ItemScore:AddStat(info, value)
	if not self.selectedSet then
		DEFAULT_CHAT_FRAME:AddMessage("Cannot add stat with no selected set")
	else
		self.selectedSet.info[info.arg] = {}
		self.selectedSet.info[info.arg].score = 0
		self.selectedSet.info[info.arg].cap = -1
		
		self:UpdateStatTable(self.options.args, self.selectedSet)
	end 
end

function ItemScore:RemoveStat(info, value)
	if not self.selectedSet then
		DEFAULT_CHAT_FRAME:AddMessage("Cannot remove stat with no selected set")
	else
		self.selectedSet.info[info.arg] = nil
		self:UpdateStatTable(self.options.args, self.selectedSet)
	end 
end

function ItemScore:GetParent(info, value)
	-- return the selected set name
	if self.selectedSet and self.selectedSet.parent then
		return self.selectedSet.parent
	end
	
	return nil
end

function ItemScore:SetParent(info, value)
	if not self.selectedSet then
		DEFAULT_CHAT_FRAME:AddMessage("Cannot set parent with no selected set")
	else
	    if value and value ~= "" then
		    self.selectedSet.parent = value
		else
		    self.selectedSet.parent = nil
		end
	end 
end

function ItemScore:SetEnabled(info, value)
	if not self.selectedSet then
		DEFAULT_CHAT_FRAME:AddMessage("Cannot set display value with no selected set")
	else
		self.selectedSet.display = value
	end 
end

function ItemScore:GetEnabled(info, value)
	if not self.selectedSet then
		-- display error here
		return nil
	else
		return self.selectedSet.display
	end
end

function ItemScore:SetPrecision(info, value)
	if not self.selectedSet then
		DEFAULT_CHAT_FRAME:AddMessage("Cannot set precision value with no selected set")
	else
		self.selectedSet.precision = value
		self:UpdateExampleText(self.options.args.exampleText, self.selectedSet)
	end 
end

function ItemScore:GetPrecision(info, value)
	if not self.selectedSet then
		-- display error here
		return 0
	else
		return self.selectedSet.precision
	end
end

function ItemScore:SetColour(info, r, g, b, a)
	if not self.selectedSet then
		DEFAULT_CHAT_FRAME:AddMessage("Cannot set colour with no selected set")
	else
		self.selectedSet.r = r
		self.selectedSet.g = g
		self.selectedSet.b = b
		--self:UpdateExampleText(self.options.args.exampleText, self.selectedSet)
	end 
end

function ItemScore:GetColour(info, value)
	if not self.selectedSet then
		-- display error here
		return 0,0,0,1
	else
		return self.selectedSet.r, self.selectedSet.g, self.selectedSet.b, 1.0 
	end
end

function ItemScore:SetPattern(info, value)
	if not self.selectedSet then
		DEFAULT_CHAT_FRAME:AddMessage("Cannot set text with no selected set")
	else
		self.selectedSet.pattern = value
		self:UpdateExampleText(self.options.args.exampleText, self.selectedSet)
	end 
end

function ItemScore:GetPattern(info, value)
	if not self.selectedSet then
		-- display error here
		return ""
	else
		return self.selectedSet.pattern
	end
end
