--[[

Copyright 2008-∞ Daniel Ceregatti (daniel@ceregatti.org)

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License as
published by the Free Software Foundation; either version 2 of the
License, or (at your option) any later version.

This program is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
02111-1307, USA.

Some code borrowed from other addons, namely:

TomTom
FuBar_SpeedFu
HandyNotes
Omen

And surely others that I can't remember at the moment. Most, if not all,
are GPL.

]]--

--[===[@debug@
local eventID = 0
local INFO = 1
local WARN = 2
local DEBUG1 = 4
local DEBUG2 = 8
--@end-debug@]===]

-- Locale variable
local L = LibStub("AceLocale-3.0"):GetLocale("FlightHUD", false)

-- Astrolabe for waypoints and horizontal speed
local Astrolabe = DongleStub("Astrolabe-0.4")

-- Keybindings
BINDING_HEADER_FLIGHTHUD = "FlightHUD"
BINDING_NAME_FLIGHTHUD_TOGGLE = L["Toggle FlightHUD"]

-- Sizes, ratios, and offsets
local MainFrameWidth = 512
local MainFrameHeight = 256
local MainFrameWidthDiv = MainFrameWidth / 2
local MainFrameHeightDiv = MainFrameHeight / 2

local MainFrameWidthLess = MainFrameWidthDiv - 32
local MainFrameHeightLess = MainFrameHeightDiv - 32

local fov = 90 -- 90 degree field of view
local bxscale = MainFrameWidthDiv / (fov / 2) -- 5.688 = (512 / 2) / (90 / 2)
local byscale = MainFrameHeight / fov -- 2.844 = 256/90
local yardsdiv = 100 / 7

local headingOffset = .268

-- Some math
local math_sqrt = math.sqrt
local math_floor = math.floor
local math_pi = math.pi
local math_abs = math.abs
local math_cos = math.cos
local math_sin = math.sin
local math_deg = math.deg
local math_rad = math.rad
local rad90 = math_rad (90)

-- Local variables
local OnUpdateElapsed = 0
local DummyOnUpdateElapsed = 0
local DummyUpdateFrequency = 0.25 -- Make this configurable
local SpeedFrequency = 0.5
local ForceToggle = 0
local oldWorldMapScript

-- Local aliases for expensive API calls
local GetUnitSpeed = GetUnitSpeed
local GetUnitPitch = GetUnitPitch

local nonCompanionGroundMountValues = {
  783,   -- Druid Travel Form
  25859, -- Reindeer
  42680, -- Magic Broom 60
  42692, -- Magic Broom walking? (Slow and steady)
  42693, -- Magic Broom walking? (Slow and steady)
  42683, -- Magic Broom 100
  -- These are companions, but don't work because they don't set "active"!!
  51621, -- Headless Horseman's Mount 60
  48024, -- Headless Horseman's Mount 100
  75619, -- Celestial Steed 60
  75620, -- Celestial Steed 100
  71343, -- Big Love Rocket walking? (Slow and steady)
  71344, -- Big Love Rocket 60
  71345, -- Big Love Rocket 100
  72281, -- Invinsible 60
  72282, -- Invinsible 100
}

local nonCompanionFlyingMountValues = {
  40120, -- Druid Swift Flight Form
  33943, -- Druid Flight Form
  42667, -- Flying Broom 150
  42668, -- Swift Flying Broom 280
  42673, -- Flying Broom
  42679, -- Swift Flying Broom
  44825, -- Flying Reindeer
  40212, -- Dragonmaw Nether Drake
  -- These are companions, but don't work because they don't set "active"!!
  51617, -- Headless Horseman's Mount 100,
  48023, -- Headless Horseman's Mount 280
  75617, -- Celestial Steed 150
  75618, -- Celestial Steed 280
  76153, -- Celestial Steed 310
  71346, -- Big Love Rocket 150
  71347, -- Big Love Rocket 310
  72283, -- Invinsible 150
  72284, -- Invinsible 310
  75957, -- X-53 Touring Rocket 150
  75972, -- X-53 Touring Rocket 280
  76154, -- X-53 Touring Rocket 310
}

local nonCompanionGroundMounts = {}
for i = 1, #nonCompanionGroundMountValues do
  nonCompanionGroundMounts[i] = GetSpellInfo(nonCompanionGroundMountValues[i])
end

local nonCompanionFlyingMounts = {}
for i = 1, #nonCompanionFlyingMountValues do
  nonCompanionFlyingMounts[i] = GetSpellInfo(nonCompanionFlyingMountValues[i])
end

-- All local functions
local function round (num, places)
  local num = num or 0 -- In case num is nil?
  local mult = 10^(places or 0)
  return math_floor (num * mult + 0.5) / mult
end
	
local function GetCurrentCursorPosition()
  local x, y = GetCursorPosition()
  local left, top = WorldMapDetailFrame:GetLeft(), WorldMapDetailFrame:GetTop()
  local width = WorldMapDetailFrame:GetWidth()
  local height = WorldMapDetailFrame:GetHeight()
  local scale = WorldMapDetailFrame:GetEffectiveScale()
  local cx = (x/scale - left) / width
  local cy = (top - y/scale) / height
  if cx < 0 or cx > 1 or cy < 0 or cy > 1 then
    return nil, nil
  end
  return cx, cy
end

local function GetHorizontalSpeed (C, Z, x, y, arg)
  -- Horizontal speed text, code borrowed from FuBar_SpeedFu
  FlightHUD.vars.Elapsed = FlightHUD.vars.Elapsed + arg;
  if (FlightHUD.vars.Elapsed > SpeedFrequency) then
    if not FlightHUD.vars.lastPos then
      if C and Z and x and y then
        FlightHUD.vars.lastTime = GetTime()
        FlightHUD.vars.lastPos = { C = C, Z = Z, x = x, y = y }
      else
        return 0
      end
    end

    local CurrTime = GetTime()

    local dist = Astrolabe:ComputeDistance(
      FlightHUD.vars.lastPos.C,
      FlightHUD.vars.lastPos.Z,
      FlightHUD.vars.lastPos.x,
      FlightHUD.vars.lastPos.y, C, Z, x, y) or 0
    local delta = CurrTime - FlightHUD.vars.lastTime
		
    if dist > 0 then
      FlightHUD.vars.speed = floor ((dist / delta / 7 * 100)) + 0.497
    else
      FlightHUD.vars.speed = 0
    end
    FlightHUD.vars.lastTime = CurrTime
    FlightHUD.vars.lastPos.C = C
    FlightHUD.vars.lastPos.Z = Z
    FlightHUD.vars.lastPos.x = x
    FlightHUD.vars.lastPos.y = y
    FlightHUD.vars.Elapsed = 0
  end
  return FlightHUD.vars.speed
end

local function FindWaypointFrame ()
  -- Search the pool for an unused frame
  for frameid, waypointframe in pairs (FlightHUD.waypointframes) do
    if not waypointframe.inuse then
      waypointframe.inuse = true
      return frameid
    end
  end
  -- None were found. Create a new one
  local frame = {}
  frame.inuse = true
  local frameid = #FlightHUD.waypointframes + 1

  if not FlightHUDMapOverlay then
    local overlay = CreateFrame("Frame", "FlightHUDMapOverlay", WorldMapButton)
    overlay:SetAllPoints(true)
  end

  -- Minimap frame
  frame.minimap = CreateFrame("Button", nil, Minimap)
  frame.minimap:SetHeight(12)
  frame.minimap:SetWidth(12)
  frame.minimap:RegisterForClicks("RightButtonUp")
  frame.minimap.frame = frame
  -- Minimap icon
  frame.minimap.icon = frame.minimap:CreateTexture("BACKGROUND")
  frame.minimap.icon:SetTexture("Interface\\TargetingFrame\\UI-RaidTargetingIcons")
  frame.minimap.icon:SetAllPoints()
  -- Worldmap frame
  frame.worldmap = CreateFrame("Button", nil, FlightHUDMapOverlay)
  frame.worldmap:SetHeight(12)
  frame.worldmap:SetWidth(12)
  frame.worldmap:RegisterForClicks("RightButtonUp")
  frame.worldmap.frame = frame
  -- Worldmap icon
  frame.worldmap.icon = frame.worldmap:CreateTexture("ARTWORK")
  frame.worldmap.icon:SetAllPoints()
  frame.worldmap.icon:SetTexture("Interface\\TargetingFrame\\UI-RaidTargetingIcons")
  -- Waypoint graphic
  frame.WaypointFrame = CreateFrame("Frame", "WaypointFrame", FlightHUD.FlightHUDFrame)
  frame.WaypointFrame:Hide()
  frame.WaypointFrame:SetScale(FlightHUD.db.profile.FrameScale)
  frame.WaypointFrame:SetWidth(16)
  frame.WaypointFrame:SetHeight(16)
  frame.WaypointFrame:SetPoint("CENTER", "FlightHUDFrame", "CENTER", 0, 0)
  -- Raid UI graphic
  frame.icon = frame.WaypointFrame:CreateTexture(nil, "OVERLAY")
  frame.icon:Show()
  frame.icon:SetBlendMode("BLEND")
  frame.icon:SetTexture("Interface\\TargetingFrame\\UI-RaidTargetingIcons")
  frame.icon:SetAlpha(FlightHUD.db.profile.HUDAlpha)
  frame.icon:SetWidth(16)
  frame.icon:SetHeight(16)
  frame.icon:SetPoint("TOP", frame.WaypointFrame)
  -- Waypoint text
  frame.TitleText = frame.WaypointFrame:CreateFontString(nil, "OVERLAY", "GameFontNormalSmall")
  frame.TitleText:Show()
  frame.TitleText:SetWidth(1000)
  frame.TitleText:SetHeight(300)
  frame.TitleText:SetJustifyV("TOP")
  frame.TitleText:SetPoint("TOP", frame.icon, "TOP", 0, -17)
  frame.TitleText:SetTextColor(unpack(FlightHUD.db.profile.WaypointTextColor))
  frame.TitleText:SetShadowColor (unpack(FlightHUD.db.profile.WaypointTextShadowColor))
  frame.TitleText:SetAlpha(FlightHUD.db.profile.HUDAlpha)
  frame.TitleText:SetFont(FlightHUD.HeadingText:GetFont(), FlightHUD.db.profile.FontScale)
  -- Add it to the frame pool
  FlightHUD.waypointframes[frameid] = frame
  return frameid
end

local function strip_colors(x)
  local ass = (x:gsub("|r", ""):gsub("|c%x%x%x%x%x%x%x%x", ""))
  return ass
end

local function MakeWaypointID (data)
  return format ("%d-%d-%f-%f", data.c, data.z, data.x, data.y)
end

local function AssignWaypointToFrame (waypointid)
  --[===[@debug@
  FlightHUD:Debug(DEBUG1, "AssignWaypointToFrame")
  --@end-debug@]===]
  local frameid = FindWaypointFrame ()
  local waypoint = FlightHUD.db.profile.waypoints[waypointid]
  local waypointframe = FlightHUD.waypointframes[frameid]
  waypointframe.waypoint = waypoint
  SetRaidTargetIconTexture (waypointframe.minimap.icon, waypoint.icon)
  SetRaidTargetIconTexture (waypointframe.worldmap.icon, waypoint.icon)
  SetRaidTargetIconTexture (waypointframe.icon, waypoint.icon)
  waypointframe.TitleText:SetText(waypoint.title)
  return frameid
end

local function AddWaypoint (data)
  --[===[@debug@
  FlightHUD:Debug(DEBUG1, "AddWaypoint")
  --@end-debug@]===]
  local waypointid = MakeWaypointID (data)
  data.lastsound = 0
  data.waypointid = waypointid
  data.titlebg = strip_colors (data.title)
  FlightHUD.db.profile.waypoints[waypointid] = data
  local frameid = AssignWaypointToFrame (waypointid)
  FlightHUD.db.profile.waypoints[waypointid].frameid = frameid
  local waypointframe = FlightHUD.waypointframes[frameid]
  Astrolabe:PlaceIconOnWorldMap (
    FlightHUDMapOverlay,
    waypointframe.worldmap,
    data.c,
    data.z,
    data.x,
    data.y
  )
  if data.worldmap then
    waypointframe.worldmap:Show()
    waypointframe.worldmap.icon:Show()
    waypointframe.worldmap:SetScript("OnClick", FlightHUD.mapicon_OnClick)
    waypointframe.worldmap:SetScript("OnEnter", FlightHUD.mapicon_OnEnter)
    waypointframe.worldmap:SetScript("OnLeave", FlightHUD.mapicon_OnLeave)
  else
    waypointframe.worldmap:Hide()
    waypointframe.worldmap.icon:Hide()
    waypointframe.worldmap:SetScript("OnClick", nil)
    waypointframe.worldmap:SetScript("OnEnter", nil)
    waypointframe.worldmap:SetScript("OnLeave", nil)
  end
  Astrolabe:PlaceIconOnMinimap (
    waypointframe.minimap,
    data.c,
    data.z,
    data.x,
    data.y
  )
  if data.minimap then
    waypointframe.minimap:Show()
    waypointframe.minimap.icon:Show()
    waypointframe.minimap:SetScript("OnClick", FlightHUD.mapicon_OnClick)
    waypointframe.minimap:SetScript("OnEnter", FlightHUD.mapicon_OnEnter)
    waypointframe.minimap:SetScript("OnLeave", FlightHUD.mapicon_OnLeave)
  else
    waypointframe.minimap:Hide()
    waypointframe.minimap.icon:Hide()
    waypointframe.minimap:SetScript("OnClick", nil)
    waypointframe.minimap:SetScript("OnEnter", nil)
    waypointframe.minimap:SetScript("OnLeave", nil)
  end
  return frameid
end

local click_verify = {
  ["A"] = function() return IsAltKeyDown() end,
  ["C"] = function() return IsControlKeyDown() end,
  ["S"] = function() return IsShiftKeyDown() end,
}

-- Borrowed mostly verbatim from TomTom
local function POIAnchorToCoord(poiframe)
  local point, relto, relpoint, x, y = poiframe:GetPoint()
  local frame = WorldMapDetailFrame
  local width = frame:GetWidth()
  local height = frame:GetHeight()
  local scale = frame:GetScale() / poiframe:GetScale()
  local cx = (x / scale) / width
  local cy = (-y / scale) / height

  if cx < 0 or cx > 1 or cy < 0 or cy > 1 then
    return nil, nil
  end

  return cx, cy
end

local function poi_OnClick(self, button)
  if not FlightHUD.db.profile.WorldMapPOI then
    return
  end
  if button == "RightButton" then
    for mod in FlightHUD.db.profile.WorldmapPOIModifier:gmatch("[ACS]") do
      if not click_verify[mod] or not click_verify[mod]() then
       	return self
      end
    end
  else
    return
  end
  if self.parentName == "WatchFrameLines" then
    local questFrame = _G["WorldMapQuestFrame"..self.questLogIndex];
    local selected = WorldMapQuestScrollChildFrame.selected
    local poiIcon = selected.poiIcon;
    self = poiIcon
  end
  local c, z = GetCurrentMapContinent(), GetCurrentMapZone();
  local x, y = POIAnchorToCoord(self)
  local qid = self.questId
  local title;
  if self.quest.questLogIndex then
    title = GetQuestLogTitle(self.quest.questLogIndex)
  else
    title = "Quest #" .. qid .. " POI"
  end
  FlightHUD:AddTWaypoint (c, z, x, y, title, L["Blizzard Quest Waypoint"], qid)
end

local hooked = {}
hooksecurefunc("QuestPOI_DisplayButton", function(parentName, buttonType, buttonIndex, questId)
  local buttonName = "poi"..tostring(parentName)..tostring(buttonType).."_"..tostring(buttonIndex);
  local poiButton = _G[buttonName];
  if not hooked[buttonName] then
    poiButton:HookScript("OnClick", poi_OnClick)
    poiButton:RegisterForClicks("AnyUp")
    hooked[buttonName] = true
  end
end)
-- End of mostly verbatim TomTom borrowed code

local ModifierValues = {
  ["A"] = L["Alt"],
  ["C"] = L["Ctrl"],
  ["S"] = L["Shift"],
  ["AC"] = L["Alt-Ctrl"],
  ["AS"] = L["Alt-Shift"],
  ["CS"] = L["Ctrl-Shift"],
  ["ACS"] = L["Alt-Ctrl-Shift"],
}

local orig_WorldMapButton_OnClick = WorldMapButton_OnClick

WorldMapButton_OnClick = function(self, ...)
  local mouseButton, button = ...
  if mouseButton == "RightButton" then
    -- Check for all the modifiers that are currently set
    for mod in FlightHUD.db.profile.WorldmapModifier:gmatch("[ACS]") do
      if not click_verify[mod] or not click_verify[mod]() then
        return orig_WorldMapButton_OnClick and orig_WorldMapButton_OnClick(self, ...) or true
      end
    end
    -- Actually try to add a note
    FlightHUD.EditFrame.c,FlightHUD.EditFrame.z = GetCurrentMapContinent (), GetCurrentMapZone ()
    FlightHUD.EditFrame.x, FlightHUD.EditFrame.y = GetCurrentCursorPosition ()
    local dx = FlightHUD.EditFrame.x * 100
    local dy = FlightHUD.EditFrame.y * 100
    if FlightHUD.EditFrame.z == 0 then
      return orig_WorldMapButton_OnClick and orig_WorldMapButton_OnClick(self, ...) or true
    end
    FlightHUD.EditFrame:Hide()
    FlightHUD.EditFrame:Show()
  else
    return orig_WorldMapButton_OnClick and orig_WorldMapButton_OnClick(self, ...) or true
  end
end

-- Add our localized text to the map help text
local origMapText

local orig_WorldMapButton_OnShow = WorldMapButton_OnShow

WorldMapButton_OnShow = function(self, ...)
  origMapText = WorldMapMagnifyingGlassButton:GetText()
  WorldMapMagnifyingGlassButton:SetText (origMapText .. "\n" .. 
    ModifierValues[FlightHUD.db.profile.WorldmapModifier] .. L["+"] .. L["Right Click To Add a FlightHUD Waypoint"])
end

local orig_WorldMapButton_OnHide = WorldMapButton_OnHide

WorldMapButton_OnHide = function(self, ...)
  WorldMapMagnifyingGlassButton:SetText(origMapText)
end

FlightHUD = LibStub("AceAddon-3.0"):NewAddon("FlightHUD", "AceConsole-3.0", "AceEvent-3.0")

function FlightHUD:OnInitialize()
  self.enabled = false
  self.vars = {}
  self.vars.lastTime = 0
  self.vars.speed = 0
  self.vars.Elapsed = 0
  self.waypointframes = {}
  self.ShowCompassGraphic = false
  self.ShowPitchImage = false
  self.GetFrom = "player"
  self.flying = false
  self.mounted = false
  self.swimming = false
  self.LastSwimming = false
  self.IsShowing = false
  self.CBid = {}
  self.Tid = {}
  self.PitchFactor = 1.1575

  StaticPopupDialogs["FLIGHTHUD_CONFIRM_WAYPOINT_DELETE"] = {
    text = L["Really delete waypoint '%s'?"],
    button1 = L["Yes"],
    button2 = L["Cancel"],
    OnAccept = function()
      FlightHUD:RemoveWaypoint (nil, true)
    end,
    timeout = 0,
    whileDead = true,
    hideOnEscape = true,
  }

  self.db = LibStub("AceDB-3.0"):New("FlightHUDDB", FlightHUDDefaultConfigDB, "Default")
  self.db.RegisterCallback(self, "OnDatabaseShutdown", "DatabaseShutdown")

  LibStub("AceConfigRegistry-3.0"):RegisterOptionsTable("FlightHUD", FlightHUDConfig)

  self.optionsFrames = {}
  local ACD3 = LibStub("AceConfigDialog-3.0")
  self.optionsFrames.FlightHUD = LibStub("LibAboutPanel").new (nil, "FlightHUD")

  -- Thanks, Omen!
  self.optionsFrames.ShowEvents = ACD3:AddToBlizOptions("FlightHUD", L["General"], "FlightHUD", "General")
  self.optionsFrames.ShowEvents = ACD3:AddToBlizOptions("FlightHUD", L["Show Events"], "FlightHUD", "ShowEvents")
  self.optionsFrames.SpeedVars = ACD3:AddToBlizOptions("FlightHUD", L["Speed Controls"], "FlightHUD", "SpeedVars")
  self.optionsFrames.Scaling = ACD3:AddToBlizOptions("FlightHUD", L["Scaling"], "FlightHUD", "Scaling")
  self.optionsFrames.TextOptions = ACD3:AddToBlizOptions("FlightHUD", L["Text Options"], "FlightHUD", "TextOptions")
  self.optionsFrames.WaypointOptions = ACD3:AddToBlizOptions("FlightHUD", L["Waypoint Options"], "FlightHUD", "WaypointOptions")
  self.optionsFrames.ResetOptions = ACD3:AddToBlizOptions("FlightHUD", L["Reset"], "FlightHUD", "ResetOptions")
  --[===[@debug@
  self.optionsFrames.ResetOptions = ACD3:AddToBlizOptions("FlightHUD", L["Debug"], "FlightHUD", "DebugOptions")
  --@end-debug@]===]

  self:RegisterChatCommand("flighthud", function()
    InterfaceOptionsFrame_OpenToCategory(self.optionsFrames.ResetOptions)
    InterfaceOptionsFrame_OpenToCategory(self.optionsFrames.FlightHUD)
  end)
  self:RegisterChatCommand("fh", function()
    InterfaceOptionsFrame_OpenToCategory(self.optionsFrames.ResetOptions)
    InterfaceOptionsFrame_OpenToCategory(self.optionsFrames.FlightHUD)
  end)
  self:RegisterChatCommand("fht", function()
    FlightHUD.db.profile.DisableAddon = not FlightHUD.db.profile.DisableAddon
    FlightHUD.EnableAddon ()
  end)

  -- Dummy frame to monitor if we're swimming
  self.DummyFrame = CreateFrame("Frame", "FlightHUDDummyFrame", UIParent)
  self.DummyFrame:SetPoint("CENTER")
  self.DummyFrame:SetWidth(32)
  self.DummyFrame:SetHeight(32)

  -- The parent frame
  self.FlightHUDFrame = CreateFrame("ScrollFrame", "FlightHUDFrame", UIParent)
  self.FlightHUDFrame:Hide()
  self.FlightHUDFrame:SetScale(self.db.profile.FrameScale)
  if (self.db.profile.FramePoint_point ~= false) then
    local scale = self.FlightHUDFrame:GetScale()
    local x = self.db.profile.FramePoint_xOfs / scale
    local y = self.db.profile.FramePoint_yOfs / scale
    self.FlightHUDFrame:SetPoint(
     self.db.profile.FramePoint_point,
      UIParent,
      self.db.profile.FramePoint_relativePoint,
      x,
      self.db.profile.FramePoint_yOfs
    )
  else
    self.FlightHUDFrame:SetPoint("CENTER")
  end
  self.FlightHUDFrame:SetWidth(MainFrameWidth)
  self.FlightHUDFrame:SetHeight(MainFrameHeight)
  -- HUD background image (always shown)
  self.timage = self.FlightHUDFrame:CreateTexture(nil, "BACKGROUND")
  self.timage:Show()
  self.timage:SetBlendMode("BLEND")
  self.timage:SetTexture("Interface\\AddOns\\FlightHUD\\transparent")
  self.timage:SetAlpha(self.db.profile.HUDBGAlpha)
  self.timage:SetWidth(512)
  self.timage:SetHeight(256)
  self.timage:SetPoint("CENTER", self.FlightHUDFrame)
  -- Heading text
  self.HeadingText = self.FlightHUDFrame:CreateFontString(nil, "OVERLAY", "GameFontNormalSmall")
  if (FlightHUD.db.profile.ShowHeadingText) then
    self.HeadingText:Show()
  else
    self.HeadingText:Hide()
  end
  self.HeadingText:SetWidth(512)
  self.HeadingText:SetHeight(16)
  self.HeadingText:SetTextColor(unpack (self.db.profile.HeadingTextColor))
  self.HeadingText:SetPoint("CENTER", self.FlightHUDFrame)
  self.HeadingText:SetJustifyV("TOP")
  self.HeadingText:SetJustifyH("RIGHT")
  self.HeadingText:SetAlpha(self.db.profile.HUDAlpha)
  self.HeadingText:SetFont(self.HeadingText:GetFont(), self.db.profile.FontScale)
  -- Pitch text
  self.PitchText = self.FlightHUDFrame:CreateFontString(nil, "OVERLAY", "GameFontNormalSmall")
  if (self.db.profile.ShowPitchText) then
    self.PitchText:Show()
  else
    self.PitchText:Hide()
  end
  self.PitchText:SetWidth(512)
  self.PitchText:SetHeight(16)
  self.PitchText:SetTextColor(unpack (self.db.profile.PitchTextColor))
  self.PitchText:SetPoint("CENTER", self.FlightHUDFrame)
  self.PitchText:SetJustifyV("TOP")
  self.PitchText:SetJustifyH("LEFT")
  self.PitchText:SetAlpha(self.db.profile.HUDAlpha)
  self.PitchText:SetFont(self.PitchText:GetFont(), self.db.profile.FontScale)
  -- Coordinates text
  self.CoordinatesText = self.FlightHUDFrame:CreateFontString(nil, "OVERLAY", "GameFontNormalSmall")
  if (self.db.profile.ShowCoordinatesText) then
    self.CoordinatesText:Show()
  else
    self.CoordinatesText:Hide()
  end
  self.CoordinatesText:SetWidth(512)
  self.CoordinatesText:SetHeight(16)
  self.CoordinatesText:SetTextColor(unpack (self.db.profile.CoordinatesTextColor))
  self.CoordinatesText:SetPoint("TOP", self.FlightHUDFrame)
  self.CoordinatesText:SetJustifyV("TOP")
  self.CoordinatesText:SetJustifyH("LEFT")
  self.CoordinatesText:SetAlpha(self.db.profile.HUDAlpha)
  self.CoordinatesText:SetFont(self.CoordinatesText:GetFont(), self.db.profile.FontScale)
  -- Speed text
  self.SpeedText = self.FlightHUDFrame:CreateFontString(nil, "OVERLAY", "GameFontNormalSmall")
  if (self.db.profile.ShowSpeedText) then
    self.SpeedText:Show()
  else
    self.SpeedText:Hide()
  end
  self.SpeedText:SetWidth(512)
  self.SpeedText:SetHeight(16)
  self.SpeedText:SetTextColor(unpack (self.db.profile.SpeedTextColor))
  self.SpeedText:SetPoint("TOP", self.FlightHUDFrame)
  self.SpeedText:SetJustifyV("TOP")
  self.SpeedText:SetJustifyH("RIGHT")
  self.SpeedText:SetAlpha(self.db.profile.HUDAlpha)
  self.SpeedText:SetFont(self.SpeedText:GetFont(), self.db.profile.FontScale)
  -- First compass heading image (shown/hidden in the OnUpdate)
  self.CompassHeadingImage1 = self.FlightHUDFrame:CreateTexture(nil, "OVERLAY")
  self.CompassHeadingImage1:Hide()
  self.CompassHeadingImage1:SetBlendMode("BLEND")
  self.CompassHeadingImage1:SetTexture("Interface\\AddOns\\FlightHUD\\FlightHUD1")
  self.CompassHeadingImage1:SetAlpha(self.db.profile.HUDAlpha)
  self.CompassHeadingImage1:SetWidth(512)
  self.CompassHeadingImage1:SetHeight(32)
  self.CompassHeadingImage1:SetPoint("BOTTOM", self.FlightHUDFrame)
  -- Second compass heading image (shown/hidden in the OnUpdate)
  self.CompassHeadingImage2 = self.FlightHUDFrame:CreateTexture(nil, "OVERLAY")
  self.CompassHeadingImage2:Hide()
  self.CompassHeadingImage2:SetBlendMode("BLEND")
  self.CompassHeadingImage2:SetTexture("Interface\\AddOns\\FlightHUD\\FlightHUD2")
  self.CompassHeadingImage2:SetAlpha(self.db.profile.HUDAlpha)
  self.CompassHeadingImage2:SetWidth(512)
  self.CompassHeadingImage2:SetHeight(32)
  self.CompassHeadingImage2:SetPoint("BOTTOM", self.FlightHUDFrame)
  -- Heading center graphic
  self.HeadingCenterImage = self.FlightHUDFrame:CreateTexture(nil, "OVERLAY")
  self.HeadingCenterImage:SetBlendMode("BLEND")
  self.HeadingCenterImage:SetTexture("Interface\\AddOns\\FlightHUD\\FlightHUD5")
  self.HeadingCenterImage:SetAlpha(self.db.profile.HUDAlpha)
  self.HeadingCenterImage:SetWidth(16)
  self.HeadingCenterImage:SetHeight(256)
  self.HeadingCenterImage:SetPoint("CENTER", self.FlightHUDFrame)
  -- Pitch graphic 1
  self.PitchImage1 = self.FlightHUDFrame:CreateTexture(nil, "OVERLAY")
  self.PitchImage1:SetBlendMode("BLEND")
  self.PitchImage1:SetTexture("Interface\\AddOns\\FlightHUD\\FlightHUD3")
  self.PitchImage1:SetAlpha(self.db.profile.HUDAlpha)
  self.PitchImage1:SetWidth(512)
  self.PitchImage1:SetHeight(190)
  self.PitchImage1:SetPoint("CENTER", self.FlightHUDFrame)
  -- Pitch graphic 2, for overlap
  self.PitchImage2 = self.FlightHUDFrame:CreateTexture(nil, "OVERLAY")
  self.PitchImage2:SetBlendMode("BLEND")
  self.PitchImage2:SetTexture("Interface\\AddOns\\FlightHUD\\FlightHUD3")
  self.PitchImage2:SetAlpha(self.db.profile.HUDAlpha)
  self.PitchImage2:SetWidth(512)
  self.PitchImage2:SetHeight(190)
  self.PitchImage2:SetPoint("CENTER", self.FlightHUDFrame)
  -- Artificial horizon graphic
  self.ArtificialHorizonImage = self.FlightHUDFrame:CreateTexture(nil, "OVERLAY")
  self.ArtificialHorizonImage:SetBlendMode("BLEND")
  self.ArtificialHorizonImage:SetTexture("Interface\\AddOns\\FlightHUD\\FlightHUD4")
  self.ArtificialHorizonImage:SetAlpha(self.db.profile.HUDAlpha)
  self.ArtificialHorizonImage:SetWidth(128)
  self.ArtificialHorizonImage:SetHeight(16)
  self.ArtificialHorizonImage:SetPoint("CENTER", self.FlightHUDFrame)
  -- "Add Waypoint" dialog
  -----------------------------------------------------------------------------------
  -- BEGIN -- Code taken mostly verbatim from HandyNotes
  -----------------------------------------------------------------------------------
  local info = {}
  local icons = {
    [1] = UnitPopupButtons.RAID_TARGET_1,
    [2] = UnitPopupButtons.RAID_TARGET_2,
    [3] = UnitPopupButtons.RAID_TARGET_3,
    [4] = UnitPopupButtons.RAID_TARGET_4,
    [5] = UnitPopupButtons.RAID_TARGET_5,
    [6] = UnitPopupButtons.RAID_TARGET_6,
    [7] = UnitPopupButtons.RAID_TARGET_7,
    [8] = UnitPopupButtons.RAID_TARGET_8,
  }
  local backdrop2 = {
    bgFile = nil,
    edgeFile = "Interface\\Tooltips\\UI-Tooltip-Border",
    tile = true, tileSize = 16, edgeSize = 16,
    insets = { left = 0, right = 0, top = 0, bottom = 0 },
  }
  self.EditFrame = CreateFrame("Frame", nil, WorldMapButton)
  self.EditFrame:Hide()
  self.EditFrame:SetWidth(350)
  self.EditFrame:SetHeight(230)
  self.EditFrame:SetPoint("BOTTOM", 0, 90)
  self.EditFrame:SetBackdrop({
    bgFile = "Interface\\DialogFrame\\UI-DialogBox-Background",
    edgeFile = "Interface\\DialogFrame\\UI-DialogBox-Border",
    tile = true, tileSize = 32, edgeSize = 32,
    insets = { left = 11, right = 12, top = 12, bottom = 11 },
  })
  self.EditFrame:SetBackdropColor(0,0,0,1)
  self.EditFrame:EnableMouse(true)
  self.EditFrame:SetToplevel(true)
  self.EditFrame:SetClampedToScreen(true)
  self.EditFrame:SetMovable(true)
  self.EditFrame:SetFrameStrata(WorldMapButton:GetFrameStrata())
  self.EditFrame.titleTexture = self.EditFrame:CreateTexture(nil, "ARTWORK")
  self.EditFrame.titleTexture:SetTexture("Interface\\DialogFrame\\UI-DialogBox-Header")
  self.EditFrame.titleTexture:SetWidth(375)
  self.EditFrame.titleTexture:SetHeight(64)
  self.EditFrame.titleTexture:SetPoint("TOP", 0, 12)
  self.EditFrame.titleTexture = temp
  self.EditFrame.title = self.EditFrame:CreateFontString(nil, "ARTWORK", "GameFontNormal")
  self.EditFrame.title:SetPoint("TOP", 0, -3)
  self.EditFrame.title:SetText(L["Set FlightHUD Waypoint"])
  -- This creates a transparent textureless draggable frame to move self.EditFrame
  -- It overlaps the above title text and texture (more or less) exactly.
  temp = CreateFrame("Frame", nil, self.EditFrame)
  temp:SetWidth(150)
  temp:SetHeight(30)
  temp:SetPoint("TOP", 0, 8)
  temp:EnableMouse(true)
  temp:RegisterForDrag("LeftButton")
  temp:SetScript("OnDragStart", function(self)
    self:GetParent():StartMoving()
  end)
  temp:SetScript("OnDragStop", function(self)
    self:GetParent():StopMovingOrSizing()
  end)
  -- Create the Close button
  self.EditFrame.CloseButton = CreateFrame("Button", nil, self.EditFrame, "UIPanelCloseButton")
  self.EditFrame.CloseButton:SetPoint("TOPRIGHT", -2, -1)
  self.EditFrame.CloseButton:SetHitRectInsets(5, 5, 5, 5)
  -- Create and position the Title text string
  self.EditFrame.titletext = self.EditFrame:CreateFontString(nil, "ARTWORK", "GameFontNormalSmall")
  self.EditFrame.titletext:SetPoint("TOPLEFT", 25, -28)
  self.EditFrame.titletext:SetText(L["Title"])
-- Create the Title Input Box and position it below the text
  self.EditFrame.titleinputframe = CreateFrame("Frame", nil, self.EditFrame)
  self.EditFrame.titleinputframe:SetWidth(300)
  self.EditFrame.titleinputframe:SetHeight(24)
  self.EditFrame.titleinputframe:SetBackdrop(backdrop2)
  self.EditFrame.titleinputframe:SetPoint("TOPLEFT", self.EditFrame.titletext, "BOTTOMLEFT", 0, 0)
  self.EditFrame.titleinputbox = CreateFrame("EditBox", nil, self.EditFrame.titleinputframe)
  self.EditFrame.titleinputbox:SetWidth(290)
  self.EditFrame.titleinputbox:SetHeight(24)
  self.EditFrame.titleinputbox:SetMaxLetters(100)
  self.EditFrame.titleinputbox:SetNumeric(false)
  self.EditFrame.titleinputbox:SetAutoFocus(false)
  self.EditFrame.titleinputbox:SetFontObject("GameFontHighlightSmall")
  self.EditFrame.titleinputbox:SetPoint("TOPLEFT", 5, 1)
  self.EditFrame.titleinputbox:SetScript("OnShow", self.EditFrame.titleinputbox.SetFocus)
  self.EditFrame.titleinputbox:SetScript("OnEscapePressed", self.EditFrame.titleinputbox.ClearFocus)
  -- Create and position the Description text string
  self.EditFrame.desctext = self.EditFrame:CreateFontString(nil, "ARTWORK", "GameFontNormalSmall")
  self.EditFrame.desctext:SetPoint("TOPLEFT", self.EditFrame.titleinputframe, "BOTTOMLEFT", 0, 0)
  self.EditFrame.desctext:SetText(L["Description/Notes:"])
  -- Create the ScrollFrame for the Description Edit Box
  self.EditFrame.descframe = CreateFrame("Frame", nil, self.EditFrame)
  self.EditFrame.descframe:SetWidth(300)
  self.EditFrame.descframe:SetHeight(67)
  self.EditFrame.descframe:SetBackdrop(backdrop2)
  self.EditFrame.descframe:SetPoint("TOPLEFT", self.EditFrame.desctext, "BOTTOMLEFT", 0, 0)
  self.EditFrame.descscrollframe = CreateFrame("ScrollFrame", "FlightHUD_EditScrollFrame",
    self.EditFrame.descframe, "UIPanelScrollFrameTemplate")
  self.EditFrame.descscrollframe:SetWidth(269)
  self.EditFrame.descscrollframe:SetHeight(59)
  self.EditFrame.descscrollframe:SetPoint("TOPLEFT", 5, -4)
  -- Create the Description Input Box and position it below the text
  self.EditFrame.descinputbox = CreateFrame("EditBox", nil, self.EditFrame)
  self.EditFrame.descinputbox:SetWidth(269) -- Height is auto set in a multiline editbox
  self.EditFrame.descinputbox:SetMaxLetters(512)
  self.EditFrame.descinputbox:SetNumeric(false)
  self.EditFrame.descinputbox:SetAutoFocus(false)
  self.EditFrame.descinputbox:SetFontObject("GameFontHighlightSmall")
  self.EditFrame.descinputbox:SetMultiLine(true)
  self.EditFrame.descinputbox:SetScript("OnCursorChanged", function(self, x, y, w, h)
    local scrollFrame = self:GetParent()
    local height = scrollFrame:GetHeight()
    local range = scrollFrame:GetVerticalScrollRange()
    local scroll = scrollFrame:GetVerticalScroll()
    local size = height + range
    local cursorOffset = -y
    while ( cursorOffset < scroll ) do
      scroll = (scroll - (height / 2))
      if ( scroll < 0 ) then
        scroll = 0
      end
      scrollFrame:SetVerticalScroll(scroll)
    end
    while ( (cursorOffset + h) > (scroll + height) and scroll < range ) do
      scroll = (scroll + (height / 2))
      if ( scroll > range ) then
        scroll = range
      end
      scrollFrame:SetVerticalScroll(scroll)
    end
  end)
  self.EditFrame.descinputbox:SetScript("OnEscapePressed", self.EditFrame.descinputbox.ClearFocus)
  -- Attach the ScrollChild to the ScrollFrame
  self.EditFrame.descscrollframe:SetScrollChild(self.EditFrame.descinputbox)
  -- Create the Icon Dropdown
  self.EditFrame.icondropdown = CreateFrame("Frame", "FlightHUD_IconDropDown", self.EditFrame, "UIDropDownMenuTemplate")
  self.EditFrame.icondropdown:SetPoint("TOPLEFT", self.EditFrame.descframe, "BOTTOMLEFT", -17, 0)
  self.EditFrame.icondropdown:SetHitRectInsets(16, 16, 0, 0)
  UIDropDownMenu_SetWidth(self.EditFrame.icondropdown,100)
  UIDropDownMenu_EnableDropDown(self.EditFrame.icondropdown)
  self.EditFrame.icondropdown.displayMode = "MENU"
  self.EditFrame.icondropdown.texture = self.EditFrame.icondropdown:CreateTexture(nil, "OVERLAY")
  self.EditFrame.icondropdown.texture:SetWidth(12)
  self.EditFrame.icondropdown.texture:SetHeight(12)
  self.EditFrame.icondropdown.texture:SetPoint("RIGHT", self.EditFrame.icondropdown, -41, 2)
  self.EditFrame.icondropdown.text = FlightHUD_IconDropDownText
  self.EditFrame.icondropdown.text:SetPoint("RIGHT", self.EditFrame.icondropdown.texture, "LEFT", -3, 0)
  self.EditFrame.icondropdown.OnClick = function(button,value)
    if type(button) ~= "table" then
      value = button
    end
    local t = icons[value]
    self.EditFrame.icondropdown.selectedValue = value
    self.EditFrame.icondropdown.texture:SetTexture(t.icon)
    if t.tCoordLeft then
      self.EditFrame.icondropdown.texture:SetTexCoord(t.tCoordLeft, t.tCoordRight, t.tCoordTop, t.tCoordBottom)
    else
      self.EditFrame.icondropdown.texture:SetTexCoord(0, 1, 0, 1)
    end
    self.EditFrame.icondropdown.text:SetText(t.text)
    local color = t.color
    if color then
      self.EditFrame.icondropdown.text:SetTextColor(color.r, color.g, color.b, color.a or 1)
    else
      self.EditFrame.icondropdown.text:SetTextColor(1, 1, 1, 1)
    end
  end
  self.EditFrame.icondropdown.initialize = function(level)
    for i = 1, #icons do
      local t = icons[i]
      info.text = t.text
      info.icon = t.icon
      local color = t.color
      info.textR = color and color.r or nil
      info.textG = color and color.g or nil
      info.textB = color and color.b or nil
      info.tCoordLeft = t.tCoordLeft or nil
      info.tCoordRight = t.tCoordRight or nil
      info.tCoordTop = t.tCoordTop or nil
      info.tCoordBottom = t.tCoordBottom or nil
      info.arg1 = i
      info.func = self.EditFrame.icondropdown.OnClick
      info.checked = self.EditFrame.icondropdown.selectedValue == i
      info.keepShownOnClick = nil
      UIDropDownMenu_AddButton(info)
    end
  end
  -- Create the  sticky checkbox
  self.EditFrame.stickycheckbox = CreateFrame("CheckButton", nil, self.EditFrame, "UICheckButtonTemplate")
  self.EditFrame.stickycheckbox:SetWidth(24)
  self.EditFrame.stickycheckbox:SetHeight(24)
  self.EditFrame.stickycheckbox:SetPoint("LEFT", self.EditFrame.icondropdown, "RIGHT", -10, 2)
  self.EditFrame.stickycheckbox.string = self.EditFrame.stickycheckbox:CreateFontString()
  self.EditFrame.stickycheckbox.string:SetWidth(200)
  self.EditFrame.stickycheckbox.string:SetJustifyH("LEFT")
  self.EditFrame.stickycheckbox.string:SetPoint("LEFT", 24, 1)
  self.EditFrame.stickycheckbox:SetFontString(self.EditFrame.stickycheckbox.string)
  self.EditFrame.stickycheckbox:SetNormalFontObject("GameFontNormalSmall")
  self.EditFrame.stickycheckbox:SetHighlightFontObject("GameFontHighlightSmall")
  self.EditFrame.stickycheckbox:SetDisabledFontObject("GameFontDisableSmall")
  self.EditFrame.stickycheckbox:SetText(L["Sticky"])
  self.EditFrame.stickycheckbox:SetChecked(true)
  self.EditFrame.stickycheckbox:SetHitRectInsets(0, -self.EditFrame.stickycheckbox.string:GetStringWidth(), 0, 0)
  self.EditFrame.stickycheckbox:SetPushedTextOffset(0, 0)
  -- Create the  zone checkbox
  self.EditFrame.zoneonlycheckbox = CreateFrame("CheckButton", nil, self.EditFrame, "UICheckButtonTemplate")
  self.EditFrame.zoneonlycheckbox:SetWidth(24)
  self.EditFrame.zoneonlycheckbox:SetHeight(24)
  self.EditFrame.zoneonlycheckbox:SetPoint("LEFT", self.EditFrame.icondropdown, "RIGHT", 70, 2)
  self.EditFrame.zoneonlycheckbox.string = self.EditFrame.zoneonlycheckbox:CreateFontString()
  self.EditFrame.zoneonlycheckbox.string:SetWidth(200)
  self.EditFrame.zoneonlycheckbox.string:SetJustifyH("LEFT")
  self.EditFrame.zoneonlycheckbox.string:SetPoint("LEFT", 24, 1)
  self.EditFrame.zoneonlycheckbox:SetFontString(self.EditFrame.zoneonlycheckbox.string)
  self.EditFrame.zoneonlycheckbox:SetNormalFontObject("GameFontNormalSmall")
  self.EditFrame.zoneonlycheckbox:SetHighlightFontObject("GameFontHighlightSmall")
  self.EditFrame.zoneonlycheckbox:SetDisabledFontObject("GameFontDisableSmall")
  self.EditFrame.zoneonlycheckbox:SetText(L["Zone Only"])
  self.EditFrame.zoneonlycheckbox:SetChecked(true)
  self.EditFrame.zoneonlycheckbox:SetHitRectInsets(0, -self.EditFrame.zoneonlycheckbox.string:GetStringWidth(), 0, 0)
  self.EditFrame.zoneonlycheckbox:SetPushedTextOffset(0, 0)
  -- Create the  minimap checkbox
  self.EditFrame.minimapcheckbox = CreateFrame("CheckButton", nil, self.EditFrame, "UICheckButtonTemplate")
  self.EditFrame.minimapcheckbox:SetWidth(24)
  self.EditFrame.minimapcheckbox:SetHeight(24)
  self.EditFrame.minimapcheckbox:SetPoint("TOPRIGHT", self.EditFrame.icondropdown, "BOTTOMLEFT", 40, 0)
  self.EditFrame.minimapcheckbox.string = self.EditFrame.minimapcheckbox:CreateFontString()
  self.EditFrame.minimapcheckbox.string:SetWidth(200)
  self.EditFrame.minimapcheckbox.string:SetJustifyH("LEFT")
  self.EditFrame.minimapcheckbox.string:SetPoint("LEFT", 24, 1)
  self.EditFrame.minimapcheckbox:SetFontString(self.EditFrame.minimapcheckbox.string)
  self.EditFrame.minimapcheckbox:SetNormalFontObject("GameFontNormalSmall")
  self.EditFrame.minimapcheckbox:SetHighlightFontObject("GameFontHighlightSmall")
  self.EditFrame.minimapcheckbox:SetDisabledFontObject("GameFontDisableSmall")
  self.EditFrame.minimapcheckbox:SetText(L["Show on minimap"])
  self.EditFrame.minimapcheckbox:SetChecked(true)
  self.EditFrame.minimapcheckbox:SetHitRectInsets(0, -self.EditFrame.minimapcheckbox.string:GetStringWidth(), 0, 0)
  self.EditFrame.minimapcheckbox:SetPushedTextOffset(0, 0)
  -- Create the  wolrdmap checkbox
  self.EditFrame.worldmapcheckbox = CreateFrame("CheckButton", nil, self.EditFrame, "UICheckButtonTemplate")
  self.EditFrame.worldmapcheckbox:SetWidth(24)
  self.EditFrame.worldmapcheckbox:SetHeight(24)
  self.EditFrame.worldmapcheckbox:SetPoint("LEFT", self.EditFrame.minimapcheckbox, "RIGHT", 130, 0)
  self.EditFrame.worldmapcheckbox.string = self.EditFrame.worldmapcheckbox:CreateFontString()
  self.EditFrame.worldmapcheckbox.string:SetWidth(200)
  self.EditFrame.worldmapcheckbox.string:SetJustifyH("LEFT")
  self.EditFrame.worldmapcheckbox.string:SetPoint("LEFT", 24, 1)
  self.EditFrame.worldmapcheckbox:SetFontString(self.EditFrame.worldmapcheckbox.string)
  self.EditFrame.worldmapcheckbox:SetNormalFontObject("GameFontNormalSmall")
  self.EditFrame.worldmapcheckbox:SetHighlightFontObject("GameFontHighlightSmall")
  self.EditFrame.worldmapcheckbox:SetDisabledFontObject("GameFontDisableSmall")
  self.EditFrame.worldmapcheckbox:SetText(L["Show on worldmap"])
  self.EditFrame.worldmapcheckbox:SetChecked(true)
  self.EditFrame.worldmapcheckbox:SetHitRectInsets(0, -self.EditFrame.worldmapcheckbox.string:GetStringWidth(), 0, 0)
  self.EditFrame.worldmapcheckbox:SetPushedTextOffset(0, 0)
  -- Create the OK button
  self.EditFrame.okbutton = CreateFrame("Button", nil, self.EditFrame, "OptionsButtonTemplate")
  self.EditFrame.okbutton:SetWidth(150)
  self.EditFrame.okbutton:SetHeight(22)
  self.EditFrame.okbutton:SetPoint("TOPRIGHT", self.EditFrame.stickycheckbox, "BOTTOMRIGHT", 0, -30)
  self.EditFrame.okbutton:SetText(OKAY)
  -- Create the Cancel button
  self.EditFrame.cancelbutton = CreateFrame("Button", nil, self.EditFrame, "OptionsButtonTemplate")
  self.EditFrame.cancelbutton:SetWidth(150)
  self.EditFrame.cancelbutton:SetHeight(22)
  self.EditFrame.cancelbutton:SetPoint("LEFT", self.EditFrame.okbutton, "RIGHT", 3, 0)
  self.EditFrame.cancelbutton:SetText(CANCEL)
  self.EditFrame.cancelbutton:SetScript("OnClick", self.EditFrame.CloseButton:GetScript("OnClick"))
  -- Additional Behavior functions
  self.EditFrame:SetScript("OnMouseDown", function(self, button)
    if MouseIsOver(FlightHUD.EditFrame.descframe) and button == "LeftButton" then
      FlightHUD.EditFrame.descinputbox:SetFocus()
    end
  end)
  self.EditFrame.titleinputbox:SetScript("OnTabPressed", function(self)
    FlightHUD.EditFrame.descinputbox:SetFocus()
  end)
  self.EditFrame.descinputbox:SetScript("OnTabPressed", function(self)
    FlightHUD.EditFrame.titleinputbox:SetFocus()
  end)
  ---------------------------------------------------------
  -- OnShow function to show a note for adding or editing
  self.EditFrame:SetScript("OnShow", function(self)
    FlightHUD.EditFrame.title:SetText(L["Add FlightHUD Waypoint"])
    FlightHUD.EditFrame.titleinputbox:SetText("")
    FlightHUD.EditFrame.descinputbox:SetText("")
    FlightHUD.EditFrame.icondropdown.OnClick(1)
    FlightHUD.EditFrame.stickycheckbox:SetChecked(nil)
    FlightHUD.EditFrame.zoneonlycheckbox:SetChecked(nil)
    FlightHUD.EditFrame.minimapcheckbox:SetChecked(true)
    FlightHUD.EditFrame.worldmapcheckbox:SetChecked(true)
  end)
  ---------------------------------------------------------
  -- OnClick function to accept the changes for a new/edited note
  self.EditFrame.okbutton:SetScript("OnClick", function(self)
    FlightHUD.EditFrame:Hide()
    local data = {}
    data.t = false
    data.c = FlightHUD.EditFrame.c
    data.x = FlightHUD.EditFrame.x
    data.y = FlightHUD.EditFrame.y
    data.z = FlightHUD.EditFrame.z 
    data.title = FlightHUD.EditFrame.titleinputbox:GetText()
    data.desc = FlightHUD.EditFrame.descinputbox:GetText()
    data.icon = FlightHUD.EditFrame.icondropdown.selectedValue
    data.sticky = FlightHUD.EditFrame.stickycheckbox:GetChecked() or false
    data.zoneonly = FlightHUD.EditFrame.zoneonlycheckbox:GetChecked() or false
    data.minimap = FlightHUD.EditFrame.minimapcheckbox:GetChecked() or false
    data.worldmap = FlightHUD.EditFrame.worldmapcheckbox:GetChecked() or false
    AddWaypoint (data)
  end)
  -----------------------------------------------------------------------------------
  -- END -- (mostly) Verbatim HandyNotes code
  -----------------------------------------------------------------------------------
  self.FlightHUDFrame:RegisterForDrag("LeftButton")
  self.FlightHUDFrame:EnableMouse(false)
  self.FlightHUDFrame:SetMovable(false)
  FlightHUD:EnableAddon ()
end

function FlightHUD:EnableAddon ()
  --[===[@debug@
  FlightHUD:Debug(DEBUG1, "DisableAddon: " .. (FlightHUD.db.profile.DisableAddon and "true" or "false"))
  --@end-debug@]===]
  if not FlightHUD.db.profile.DisableAddon then
    FlightHUD.FlightHUDFrame:SetScript("OnDragStart", FlightHUD.OnDragStart)
    FlightHUD.FlightHUDFrame:SetScript("OnDragStop", FlightHUD.OnDragStop)
    FlightHUD.FlightHUDFrame:SetScript("OnUpdate", FlightHUD.OnUpdate)
    FlightHUD.FlightHUDFrame:SetScript("OnShow", FlightHUD.OnShow)
    FlightHUD.FlightHUDFrame:SetScript("OnHide", FlightHUD.OnHide)
    FlightHUD.FlightHUDFrame:SetScript("OnEvent", function(frame,event,...) FlightHUD[event](FlightHUD,...) end)
    FlightHUD.FlightHUDFrame:RegisterEvent("WORLD_MAP_UPDATE")
    FlightHUD.FlightHUDFrame:RegisterEvent("ZONE_CHANGED_NEW_AREA")
    FlightHUD.FlightHUDFrame:RegisterEvent("UNIT_ENTERED_VEHICLE")
    FlightHUD.FlightHUDFrame:RegisterEvent("UNIT_EXITED_VEHICLE")
    FlightHUD.FlightHUDFrame:RegisterEvent("MODIFIER_STATE_CHANGED")
    FlightHUD.FlightHUDFrame:RegisterEvent("PLAYER_ENTERING_WORLD")
    FlightHUD.FlightHUDFrame:RegisterEvent("UNIT_AURA")
    if self.db.profile.ShowWhenSwimming then
      FlightHUD.DummyFrame:SetScript("OnUpdate", self.DummyOnUpdate)
    end
    FlightHUD:SetupQuestHelperCallback ()
    WorldMapButton:SetScript("OnMouseUp", WorldMapButton_OnClick)
    WorldMapButton:SetScript("OnShow", WorldMapButton_OnShow)
    WorldMapButton:SetScript("OnHide", WorldMapButton_OnHide)
    FlightHUD.enabled = true
    FlightHUD:UNIT_AURA ()
  else
    FlightHUD.FlightHUDFrame:SetScript("OnDragStart", nil)
    FlightHUD.FlightHUDFrame:SetScript("OnDragStop", nil)
    FlightHUD.FlightHUDFrame:SetScript("OnUpdate", nil)
    FlightHUD.FlightHUDFrame:SetScript("OnShow", nil)
    FlightHUD.FlightHUDFrame:SetScript("OnHide", nil)
    FlightHUD.FlightHUDFrame:SetScript("OnEvent", nil)
    FlightHUD.FlightHUDFrame:UnregisterEvent("WORLD_MAP_UPDATE")
    FlightHUD.FlightHUDFrame:UnregisterEvent("ZONE_CHANGED_NEW_AREA")
    FlightHUD.FlightHUDFrame:UnregisterEvent("UNIT_ENTERED_VEHICLE")
    FlightHUD.FlightHUDFrame:UnregisterEvent("UNIT_EXITED_VEHICLE")
    FlightHUD.FlightHUDFrame:UnregisterEvent("MODIFIER_STATE_CHANGED")
    FlightHUD.FlightHUDFrame:UnregisterEvent("PLAYER_ENTERING_WORLD")
    FlightHUD.FlightHUDFrame:UnregisterEvent("UNIT_AURA")
    WorldMapButton:SetScript("OnMouseUp", orig_WorldMapButton_OnClick)
    WorldMapButton:SetScript("OnShow", orig_WorldMapButton_OnShow)
    WorldMapButton:SetScript("OnHide", orig_WorldMapButton_OnHide)
    FlightHUD.DummyFrame:SetScript("OnUpdate", nil)
    FlightHUD:RemoveQuestHelperCallback ()
    FlightHUD.FlightHUDFrame:Hide()
    FlightHUD.enabled = false
  end
end

-- Event methods
function FlightHUD:MODIFIER_STATE_CHANGED(key, state)
  if not FlightHUD.db.profile.LockPosition then
    if (key == 'LCTRL' or key == 'RCTRL') and state == 1 then
      self.FlightHUDFrame:EnableMouse(true)
      self.FlightHUDFrame:SetMovable(true)
    else
      if (key == 'LCTRL' or key == 'RCTRL') and state == 0 then
        self.FlightHUDFrame:EnableMouse(false)
        self.FlightHUDFrame:SetMovable(false)
        self.FlightHUDFrame:StopMovingOrSizing()
      end
    end
  end
end

function FlightHUD:OnDragStart()
  self:StartMoving()
end

function FlightHUD:OnDragStop()
  self:StopMovingOrSizing()
  FlightHUD.db.profile.FramePoint_point,
  FlightHUD.db.profile.FramePoint_relativeTo,
  FlightHUD.db.profile.FramePoint_relativePoint,
  FlightHUD.db.profile.FramePoint_xOfs,
  FlightHUD.db.profile.FramePoint_yOfs = self:GetPoint()
end

function FlightHUD:OnEnable()
  self.enabled = true
end

function FlightHUD:OnDisable()
  self.enabled = false
end

-- Other object methods
function FlightHUD:ResetPosition()
  FlightHUD.db.profile.FramePoint_point = false
  FlightHUD.db.profile.FramePoint_relativeTo = false
  FlightHUD.db.profile.FramePoint_relativePoint = false
  FlightHUD.db.profile.FramePoint_xOfs = false
  FlightHUD.db.profile.FramePoint_yOfs = false
  FlightHUD.FlightHUDFrame:ClearAllPoints()
  FlightHUD.FlightHUDFrame:SetPoint("CENTER")
end

function FlightHUD:ResetAll()
  FlightHUD.db:ResetProfile()
  local font = FlightHUD.PitchText:GetFont()
  FlightHUD.PitchText:SetFont(font, FlightHUD.db.profile.FontScale)
  FlightHUD.HeadingText:SetFont(font, FlightHUD.db.profile.FontScale)
  FlightHUD.CoordinatesText:SetFont(font, FlightHUD.db.profile.FontScale)
  FlightHUD.SpeedText:SetFont(font, FlightHUD.db.profile.FontScale)
  FlightHUD.CompassHeadingImage1:SetAlpha(FlightHUD.db.profile.HUDAlpha)
  FlightHUD.CompassHeadingImage2:SetAlpha(FlightHUD.db.profile.HUDAlpha)
  FlightHUD.PitchImage1:SetAlpha(FlightHUD.db.profile.HUDAlpha)
  FlightHUD.PitchImage2:SetAlpha(FlightHUD.db.profile.HUDAlpha)
  FlightHUD.ArtificialHorizonImage:SetAlpha(FlightHUD.db.profile.HUDAlpha)
  FlightHUD.HeadingCenterImage:SetAlpha(FlightHUD.db.profile.HUDAlpha)
  FlightHUD.HeadingText:SetAlpha(FlightHUD.db.profile.HUDAlpha)
  FlightHUD.PitchText:SetAlpha(FlightHUD.db.profile.HUDAlpha)
  FlightHUD.CoordinatesText:SetAlpha(FlightHUD.db.profile.HUDAlpha)
  FlightHUD.SpeedText:SetAlpha(FlightHUD.db.profile.HUDAlpha)
  FlightHUD.timage:SetAlpha(FlightHUD.db.profile.HUDBGAlpha)
  FlightHUD.FlightHUDFrame:SetScale(FlightHUD.db.profile.FrameScale)
  FlightHUD.FlightHUDFrame:ClearAllPoints()
  FlightHUD.FlightHUDFrame:SetPoint("CENTER")
end

function FlightHUD:DummyOnUpdate(arg)
  DummyOnUpdateElapsed = DummyOnUpdateElapsed + arg
  if (DummyOnUpdateElapsed > DummyUpdateFrequency) then
    if (IsSwimming()) then
      FlightHUD.swimming = true;
    else
      FlightHUD.swimming = false;
    end
    if (FlightHUD.swimming ~= FlightHUD.LastSwimming) then
      FlightHUD:EventCheck ()
      FlightHUD.LastSwimming = FlightHUD.swimming
    end
    DummyOnUpdateElapsed = 0
  end
end
 
function FlightHUD:OnUpdate (arg)
  OnUpdateElapsed = OnUpdateElapsed + arg
  if (OnUpdateElapsed > FlightHUD.db.profile.RefreshRate) then
    -- Used by both compass graphic and heading text
    local facingRadians = GetPlayerFacing()
    local facingDegrees = math_deg (-facingRadians) % 360
    -- Compass graphic
    if (FlightHUD.ShowCompassImage) then
      if facingDegrees > 85 and facingDegrees < 268 then
        local rx = (facingDegrees / 360) + headingOffset
        local lx = rx - 0.5
        FlightHUD.CompassHeadingImage1:SetTexCoord(lx, 0, lx, 1, rx, 0, rx, 1)
        FlightHUD.CompassHeadingImage1:Show()
        FlightHUD.CompassHeadingImage2:Hide()
      else
        local newdegrees = 0
        if facingDegrees > 0 and facingDegrees < 180 then
          newdegrees = facingDegrees + 180
        else
          newdegrees = facingDegrees - 180
        end
        local rx = (newdegrees / 360) + headingOffset
        local lx = rx - 0.5
        FlightHUD.CompassHeadingImage2:SetTexCoord(lx, 0, lx, 1, rx, 0, rx, 1)
        FlightHUD.CompassHeadingImage1:Hide()
        FlightHUD.CompassHeadingImage2:Show()
      end
    end
    -- Heading text
    if (FlightHUD.db.profile.ShowHeadingText) then
      FlightHUD.HeadingText:SetText(format("%d", facingDegrees))
    end
    -- Used by both pitch graphic and pitch text
    local angle = math_deg (GetUnitPitch(FlightHUD.GetFrom))
    -- Pitch graphic
    local mult = 1
    if angle < 0 then
      mult = -1
    end
    if (FlightHUD.ShowPitchImage) then
      local newangle = -(angle % (mult * 180)) * 0.005 
      local pgs = FlightHUD.db.profile.PitchGraphicsScale / 100
      local ax = newangle + 0.25 + pgs
      local ay = newangle + 0.75 - pgs
      local bx = newangle + (mult * FlightHUD.PitchFactor) + pgs
      local by = newangle + (mult * (FlightHUD.PitchFactor + .5)) - pgs
      FlightHUD.PitchImage1:SetTexCoord(0, ax, 0, ay, 1, ax, 1, ay)
      FlightHUD.PitchImage2:SetTexCoord(0, bx, 0, by, 1, bx, 1, by)
    end
    -- Pitch text
    if (FlightHUD.db.profile.ShowPitchText) then
      local displayAngle = angle
      if angle > 90 then
        displayAngle = -(180 - angle)
      end
      if angle < -90 then
        displayAngle = -(-180 - angle)
      end
      FlightHUD.PitchText:SetText(format("%d", round (displayAngle)))
    end
    -- Used by ShowCoordinatesText, and (ShowSpeedText or waypoint ETA)
    local C, Z, px, py = Astrolabe:GetCurrentPlayerPosition()
    if not C or not Z or not px or not py then
      return
    end
    -- Coordinates text
    if (FlightHUD.db.profile.ShowCoordinatesText) then
      FlightHUD.CoordinatesText:SetText(format("%d, %d", px * 100, py * 100))
    end
    -- Used in ETA calculation. If not set in the speed conditional, it must be obtained in the ETA calculator
    local etahorizspeed = 0
    local etahorizspeedobtained = false
    -- Speed text
    if (FlightHUD.db.profile.ShowSpeedText) then
      local speedtext = ""
      if (FlightHUD.db.profile.WhichSpeedType == "LEFT" or FlightHUD.db.profile.WhichSpeedType == "RIGHT") then
        local horizspeed = GetHorizontalSpeed (C, Z, px, py, arg)
        etahorizspeed = horizspeed / yardsdiv
        etahorizspeedobtained = true
        if (FlightHUD.db.profile.SpeedType == "BOTTOM") then
          horizspeed = round (horizspeed, FlightHUD.db.profile.SpeedPrecision) .. "%"
        end
        if (FlightHUD.db.profile.SpeedType == "LEFT") then
          horizspeed = round (horizspeed / yardsdiv, FlightHUD.db.profile.SpeedPrecision) .. L["y/s"]
        end
        if (FlightHUD.db.profile.SpeedType == "RIGHT") then
          horizspeed = round (horizspeed, FlightHUD.db.profile.SpeedPrecision) .. "%" .. "," ..
            round (horizspeed / yardsdiv, FlightHUD.db.profile.SpeedPrecision) .. L["y/s"]
        end
        if (FlightHUD.db.profile.ShowSpeedPrefix) then
          speedtext = L["H: "] .. horizspeed
        else
          speedtext = horizspeed
        end
      end
      -- True speed
      if (FlightHUD.db.profile.WhichSpeedType == "BOTTOM" or FlightHUD.db.profile.WhichSpeedType == "RIGHT") then
        local realspeed = GetUnitSpeed(FlightHUD.GetFrom)
        if (FlightHUD.db.profile.SpeedType == "BOTTOM") then
          realspeed = round ((realspeed * yardsdiv), FlightHUD.db.profile.SpeedPrecision) .. "%"
        end
        if (FlightHUD.db.profile.SpeedType == "LEFT") then
          realspeed = round (realspeed, FlightHUD.db.profile.SpeedPrecision) .. L["y/s"]
        end
        if (FlightHUD.db.profile.SpeedType == "RIGHT") then
          realspeed = round ((realspeed * yardsdiv), FlightHUD.db.profile.SpeedPrecision) ..
            "%" .. "," .. round (realspeed, FlightHUD.db.profile.SpeedPrecision) .. L["y/s"]
        end
        if (speedtext ~= "") then
          speedtext = speedtext .. " - "
        end
        if (FlightHUD.db.profile.ShowSpeedPrefix) then
          speedtext = speedtext .. L["T: "] .. realspeed
        else
          speedtext = speedtext .. realspeed
        end
      end
      FlightHUD.SpeedText:SetText(speedtext)
    end
    -- Waypoints
    for frameid, waypointframe in pairs(FlightHUD.waypointframes) do
      if (waypointframe.inuse == true) then
        local waypoint = waypointframe.waypoint
        local dist, x, y = Astrolabe:GetDistanceToIcon (waypointframe.minimap)
        if (dist and dist < 6500) then
          -- Remove the non sticky, non-transient waypoint, as we're too close
          if (
            dist < FlightHUD.db.profile.RemoveWaypointDistance
          ) then
            if not waypoint.sticky and not waypoint.t then
              FlightHUD:RemoveWaypoint (frameid, true)
            end
            -- Prevent sound spam if the waypoint is sticky
            local currtime = GetTime()
            if FlightHUD.db.profile.WaypointNotify and currtime - waypoint.lastsound > 3 then
              waypoint.lastsound = currtime
              PlaySoundFile(FlightHUD.db.profile.WaypointNotifySound)
            end 
          else
            waypointframe.WaypointFrame:Show()
            -- This returns the direction of the icon relative to the player's
            -- map position in radians
            local directionRadians = Astrolabe:GetDirectionToIcon (waypointframe.minimap)
            -- Convert to degrees for text display
            local directionDegrees = math_deg (directionRadians)
            -- bearingRadians is used for math
            local bearingRadians = facingRadians - directionRadians 
            -- bearingDegrees is for text representation
            local bearingDegrees = math_deg (bearingRadians) % 360
            -- Create a distance scaling factor
            local distscale = 0.55
            if (dist > 0 and dist < 50) then
              distscale = 1
            end
            if (dist > 50 and dist < 150) then
              distscale = 0.95
            end
            if (dist > 150 and dist < 300) then
              distscale = 0.9
            end
            if (dist > 300 and dist < 450) then
              distscale = 0.85
            end
            if (dist > 450 and dist < 600) then
              distscale = 0.8
            end
            if (dist > 600 and dist < 750) then
              distscale = 0.75
            end
            if (dist > 750 and dist < 900) then
              distscale = 0.7
            end
            if (dist > 900 and dist < 1050) then
              distscale = 0.65
            end
            if (dist > 1050 and dist < 1200) then
              distscale = 0.6
            end
            local bxmax = MainFrameWidthDiv / distscale
            local bx
            local by
            -- Display as an ellipse
            if (FlightHUD.db.profile.WaypointDisplay == "BOTTOM") then
              -- Semimajor axis
              local a = MainFrameWidthLess / distscale
              -- Semiminor axis
              local b = MainFrameHeightLess / distscale
              local temp = -bearingRadians + rad90
              bx = a * math_cos (temp)
              by = b * math_sin (temp)
            else
              -- Display along the parent frame (minus some arbitrary value) 
              bx = bearingDegrees
              -- Calculate "by" to make it go up and down when the waypoint is in:
              -- The "fore" quadrant
              by = MainFrameHeightDiv
              -- The "port" quadrant
              if (bearingDegrees > 45 and bearingDegrees < 135) then
                -- This number needs to go from 0 to 90
                by = (90 - bearingDegrees) * byscale
              end
              -- The "starboard"s quadrant
              if (bearingDegrees < -45 and bearingDegrees > -135) then
                by = (90 - math_abs (bearingDegrees)) * byscale
              end
              -- The "aft" quadrant
              if (bearingDegrees > 135 or bearingDegrees < -135) then
                by = -MainFrameHeightDiv
                local mod = 180
                if (bearingDegrees < 0) then 
                  mod = -180
                end
                -- bx (bearingDegrees still at this point) needs to go in reverse when in the aft quadrant
                bx = mod - bx
              end
              by = by / distscale
              -- Calculate bx
              local bx = (-bx * bxscale) / distscale
              if (bx > bxmax) then bx = bxmax end
              if (bx < -bxmax) then bx = -bxmax end
            end
            waypointframe.WaypointFrame:ClearAllPoints()
            waypointframe.WaypointFrame:SetScale (distscale)
            waypointframe.WaypointFrame:SetPoint ("CENTER", "FlightHUDFrame", "CENTER", bx, by)

            local etatext = ' '
            -- The user may have selected to not display horizontal speed. Get the horizontal speed
            -- in yards per second here.
            if not etahorizspeedobtained then
              etahorizspeed = GetHorizontalSpeed (C, Z, px, py, arg) / yardsdiv
            end
            -- If we have an ETA to display, calculate it, but only if we're facing it in a 90° arc
            -- Make this configurable?
            if
              etahorizspeed > 0
              and (bearingDegrees > FlightHUD.db.profile.etaArcL or
              bearingDegrees < FlightHUD.db.profile.etaArcR)
            then
              local totalseconds = math_floor (dist / etahorizspeed)
              local minutes = math_floor (totalseconds / 60)
              if (minutes > 0) then
                if (math_floor (minutes) == 1) then
                  etatext = format ("%d %s ", minutes, L["minute"])
                else
                  etatext = format ("%d %s ", minutes, L["minutes"])
                end
              end
              local seconds = totalseconds % 60
              if (math_floor (seconds) == 1) then
                etatext = format ("%s%d %s", etatext, seconds, L["second"])
              else
                etatext = format ("%s%d %s", etatext, seconds, L["seconds"])
              end
              if (etatext) then
                etatext = format ("%s %s", L["ETA: "], etatext) 
              end
            end

            local DegreesText = L["degrees"]
            if (bearingDegrees >= 1 and bearingDegrees < 2) then
              DegreesText = L["degree"]
            end

            waypointframe.TitleText:SetText(
              format(
                "%s\n%d %s %d %s\n%s",
                waypoint.title,
                dist, L["yards"], bearingDegrees, DegreesText,
                etatext
              )
            )

          end
        else
          waypointframe.WaypointFrame:Hide()
        end
      end
    end
  end
end

function FlightHUD:UNIT_ENTERED_VEHICLE ()
  FlightHUD.GetFrom = "vehicle"
  FlightHUD:UNIT_AURA ()
end

function FlightHUD:UNIT_EXITED_VEHICLE ()
  FlightHUD.GetFrom = "player"
  FlightHUD:UNIT_AURA ()
end

function FlightHUD:ResetWaypoints ()
  local c = GetCurrentMapContinent ()
  local z = GetCurrentMapZone()
  FlightHUD:RemoveAllWaypoints ()
  for waypointid, waypoint in pairs(FlightHUD.db.profile.waypoints) do
    if waypoint.c == c then
      if not waypoint.zoneonly or (waypoint.zoneonly and waypoint.z == z) then
        AddWaypoint (waypoint)
      end
    end
  end
end

function FlightHUD:PLAYER_ENTERING_WORLD ()
  FlightHUD.GetFrom = "player"
  FlightHUD:ResetWaypoints()
  FlightHUD:UNIT_AURA ()
end

function FlightHUD:DatabaseShutdown ()
  for waypointid, waypoint in pairs(FlightHUD.db.profile.waypoints) do
    if not waypoint.sticky then
      FlightHUD.db.profile.waypoints[waypointid] = nil
    end
  end
end

function FlightHUD:UNIT_AURA (arg)
  if arg == "player" then
    --[===[@debug@
    eventID = eventID + 1
    FlightHUD:Debug(DEBUG1, "UNIT_AURA")
    --@end-debug@]===]
    FlightHUD.flying = false
    FlightHUD.mounted = false
    -- Check non-companion mounts
    for i = 1, #nonCompanionGroundMounts do
      if select (11, UnitBuff ("player", nonCompanionGroundMounts[i])) == nonCompanionGroundMountValues[i] then
        --[===[@debug@
        FlightHUD:Debug(DEBUG2, "Matched non-companion ground mount " .. nonCompanionGroundMounts[i] .. ", id: " .. nonCompanionGroundMountValues[i])
        --@end-debug@]===]
        FlightHUD.mounted = true
      end
    end
    for i = 1, #nonCompanionFlyingMounts do
      if select (11, UnitBuff ("player", nonCompanionFlyingMounts[i])) == nonCompanionFlyingMountValues[i] then
        --[===[@debug@
        FlightHUD:Debug(DEBUG2, "Matched non-companion flying mount " .. nonCompanionFlyingMounts[i] .. ", id: " .. nonCompanionFlyingMountValues[i])
        --@end-debug@]===]
        FlightHUD.flying = true
      end
    end
    -- Normal companion mounts, but only if the above failed
    if not FlightHUD.flying and not FlightHUD.mounted then
      for i = 1, GetNumCompanions("MOUNT") do
        local cid, cname, spellid, icon, active = GetCompanionInfo("MOUNT", i)
        if (active) then
          if (FlightHUDFlyingMounts[spellid]) then
            --[===[@debug@
            FlightHUD:Debug(DEBUG2, "Matched companion flying mount id: " .. spellid)
            --@end-debug@]===]
            FlightHUD.flying = true
          else
            --[===[@debug@
            FlightHUD:Debug(DEBUG1, "Matched companion ground mount id: " .. spellid)
            --@end-debug@]===]
            FlightHUD.mounted = true
          end
        end
      end
    end
    FlightHUD:EventCheck ()
  end
end

function FlightHUD:EventCheck ()
  local ShowFrame = false
  if (
    (FlightHUD.db.profile.ShowCompassWhenFlying and FlightHUD.flying)
    or (FlightHUD.db.profile.ShowCompassWhenMounted and FlightHUD.mounted)
    or (FlightHUD.db.profile.ShowCompassWhenSwimming and FlightHUD.swimming)
    or FlightHUD.db.profile.ShowCompassAlways
  ) then
    FlightHUD.HeadingCenterImage:Show()
    FlightHUD.ShowCompassImage = true
  else
    FlightHUD.HeadingCenterImage:Hide()
    FlightHUD.CompassHeadingImage1:Hide()
    FlightHUD.CompassHeadingImage2:Hide()
    FlightHUD.ShowCompassImage = false
  end
  if (
    (FlightHUD.db.profile.ShowPitchWhenFlying and FlightHUD.flying)
    or (FlightHUD.db.profile.ShowPitchWhenMounted and FlightHUD.mounted)
    or (FlightHUD.db.profile.ShowPitchWhenSwimming and FlightHUD.swimming)
    or FlightHUD.db.profile.ShowPitchAlways
  ) then
    FlightHUD.PitchImage1:Show()
    FlightHUD.PitchImage2:Show()
    FlightHUD.ArtificialHorizonImage:Show()
    FlightHUD.ShowPitchImage = true
  else
    FlightHUD.PitchImage1:Hide()
    FlightHUD.PitchImage2:Hide()
    FlightHUD.ArtificialHorizonImage:Hide()
    FlightHUD.ShowPitchImage = false
  end
  if (
  (FlightHUD.db.profile.ShowWhenFlying and FlightHUD.flying)
    or (FlightHUD.db.profile.ShowWhenMounted and FlightHUD.mounted)
    or (FlightHUD.db.profile.ShowWhenSwimming and FlightHUD.swimming)
    or FlightHUD.db.profile.ShowAlways
  ) then
    ShowFrame = true
  end
  if ForceToggle == 1 then
    ShowFrame = true
  end
  if ForceToggle == 2 then
    ShowFrame = false
  end
  if ShowFrame and not FlightHUD.IsShowing then
    FlightHUD.FlightHUDFrame:Show()
    return
  end
  if not ShowFrame and FlightHUD.IsShowing then
    FlightHUD.FlightHUDFrame:Hide()
  end
end

function FlightHUD:ToggleHUD()
  --[===[@debug@
  FlightHUD:Debug (DEBUG1, "ToggleHUD")
  FlightHUD:Debug (DEBUG2, "  current FT: " .. ForceToggle)
  --@end-debug@]===]
  if not FlightHUD.enabled then
    --[===[@debug@
    FlightHUD:Debug (DEBUG2, "  Addon was not enabled")
    --@end-debug@]===]
    return
  end
  if ForceToggle == 1 or ForceToggle == 2 then
    --[===[@debug@
    FlightHUD:Debug (DEBUG2, "  Setting to 0")
    --@end-debug@]===]
    ForceToggle = 0
    return FlightHUD:EventCheck()
  end
  if not FlightHUD.IsShowing and ForceToggle == 0 then
    ForceToggle = 1
  end
  if FlightHUD.IsShowing and ForceToggle == 0 then
    ForceToggle = 2
  end
  --[===[@debug@
  FlightHUD:Debug (DEBUG2, "  Setting to " .. ForceToggle)
  --@end-debug@]===]
  FlightHUD:EventCheck()
end

function FlightHUD:OnShow()
  --[===[@debug@
  eventID = eventID + 1
  FlightHUD:Debug (DEBUG1, "OnShow")
  --@end-debug@]===]
  FlightHUD.IsShowing = true
  --[===[@debug@
  FlightHUD:Debug (DEBUG2, "IsShowing set to true")
  --@end-debug@]===]
end

function FlightHUD:OnHide()
  --[===[@debug@
  eventID = eventID + 1
  FlightHUD:Debug (DEBUG1, "OnHide")
  --@end-debug@]===]
  FlightHUD.IsShowing = false
  --[===[@debug@
  FlightHUD:Debug (DEBUG2, "IsShowing set to false")
  --@end-debug@]===]
end

function FlightHUD:ZONE_CHANGED_NEW_AREA()
  --[===[@debug@
  eventID = eventID + 1
  FlightHUD:Debug (DEBUG1, "ZONE_CHANGED_NEW_AREA")
  --@end-debug@]===]
  if not WorldMapFrame:IsShown() then
    SetMapToCurrentZone()
    FlightHUD.vars.speed = 0.0
    FlightHUD.vars.lastPos = {}
    FlightHUD.vars.lastTime = 0
  end
  FlightHUD:ResetWaypoints()
  FlightHUD:WORLD_MAP_UPDATE ()
end

function FlightHUD:mapicon_OnClick ()
  FlightHUD.deleteWaypoint = self.frame.waypoint.frameid
  StaticPopup_Show ("FLIGHTHUD_CONFIRM_WAYPOINT_DELETE", self.frame.waypoint.title);
end

function FlightHUD:mapicon_OnEnter ()
  --[===[@debug@
  eventID = eventID + 1
  FlightHUD:Debug(DEBUG1, "mapicon_OnEnter")
  --@end-debug@]===]
  GameTooltip_SetDefaultAnchor (GameTooltip, UIParent)
  GameTooltip:SetText (self.frame.waypoint.title)
  GameTooltip:AddLine (self.frame.waypoint.desc)
  GameTooltip:Show ()
end

function FlightHUD:mapicon_OnLeave ()
  --[===[@debug@
  eventID = eventID + 1
  FlightHUD:Debug(DEBUG1, "mapicon_OnLeave")
  --@end-debug@]===]
  GameTooltip:Hide ()
end

function FlightHUD:WORLD_MAP_UPDATE ()
  --[===[@debug@
  eventID = eventID + 1
  FlightHUD:Debug(DEBUG1, "WORLD_MAP_UPDATE")
  --@end-debug@]===]
  if (WorldMapFrame:IsShown()) then
    local z = GetCurrentMapZone()
    local c = GetCurrentMapContinent()
    for frameid, waypoint in pairs (FlightHUD.waypointframes) do
      if (waypoint.inuse) then
        if (waypoint.waypoint.c == c and waypoint.waypoint.z == z) then
          waypoint.worldmap:Show()
        else
          waypoint.worldmap:Hide()
        end        
      end
    end
  end
end

function FlightHUD:AddQHWaypoint (c, z, x, y, desc)
  --[===[@debug@
  eventID = eventID + 1
  FlightHUD:Debug(DEBUG1, "AddQHWaypoint")
  --@end-debug@]===]
  -- Somehow QuestHelper calls this with x being nil...
  if c and z and x and y then
    FlightHUD:AddTWaypoint (c, z, x, y, desc, L["QuestHelper Waypoint"], c .. z .. x .. y)
  end
end

function FlightHUD:AddTWaypoint (c, z, x, y, desc, t, pid)
  --[===[@debug@
  eventID = eventID + 1
  FlightHUD:Debug(DEBUG1, "AddTWaypoint")
  FlightHUD:Debug(DEBUG2, "  pid: " .. pid)
  --@end-debug@]===]
  if FlightHUD.Tid[t] then
    -- A waypoint of this type already exists. Replace it, but first
    -- check to see if it's the same waypoint as what we already have.
    if
      FlightHUD.waypointframes[FlightHUD.Tid[t]].waypoint
      and FlightHUD.waypointframes[FlightHUD.Tid[t]].waypoint.pid
      and FlightHUD.waypointframes[FlightHUD.Tid[t]].waypoint.pid == pid
    then
      return
    end
    FlightHUD:RemoveWaypoint (FlightHUD.Tid[t], true)
    FlightHUD.Tid[t] = nil
  end
  if not c then
    return
  end
  local data = {}
  data.pid = pid
  data.t = true
  data.c = c
  data.x = x
  data.y = y
  data.z = z
  data.title = desc 
  data.desc = t
  data.icon = 1
  data.sticky = false
  data.zoneonly = false
  data.minimap = false
  data.worldmap = false
  FlightHUD.Tid[t] = AddWaypoint (data)
end

function FlightHUD:RemoveWaypoint (frameid, delete)
  --[===[@debug@
  FlightHUD:Debug(DEBUG1, "RemoveWaypoint")
  --@end-debug@]===]
  if frameid == nil then
    frameid = FlightHUD.deleteWaypoint
    if frameid == nil then
      return
    end
    FlightHUD.deleteWaypoint = nil
  end
  local waypointframe = FlightHUD.waypointframes[frameid]
  local waypoint = waypointframe.waypoint
  if not waypoint then
    return
  end
  Astrolabe:RemoveIconFromMinimap(waypoint.minimap)
  waypointframe.worldmap:Hide()
  waypointframe.minimap:Hide()
  waypointframe.WaypointFrame:Hide()
  waypointframe.inuse = false
  if delete then
    FlightHUD.db.profile.waypoints[waypoint.waypointid] = nil
  end
  waypointframe.waypoint = nil
end

function FlightHUD:RemoveAllWaypoints (all)
  --[===[@debug@
  FlightHUD:Debug(DEBUG1, "RemoveAllWaypoints")
  --@end-debug@]===]
  for frameid, waypointframe in pairs (FlightHUD.waypointframes) do
    FlightHUD:RemoveWaypoint (frameid, all)
  end
end

function FlightHUD:SetupQuestHelperCallback ()
  --[===[@debug@
  eventID = eventID + 1
  FlightHUD:Debug(DEBUG1, "SetupQuestHelperCallback")
  --@end-debug@]===]
  if QuestHelper and not FlightHUD.CBid["qh"] and FlightHUD.db.profile.QuestHelper then
    FlightHUD.CBid["qh"] = QuestHelper:AddWaypointCallback (FlightHUD.AddQHWaypoint, FlightHUD)
  end
end

function FlightHUD:RemoveQuestHelperCallback ()
  --[===[@debug@
  eventID = eventID + 1
  FlightHUD:Debug(DEBUG1, "RemoveQuestHelperCallback")
  --@end-debug@]===]
  if FlightHUD.CBid["qh"] then
    QuestHelper:RemoveWaypointCallback (FlightHUD.CBid["qh"])
    -- Explicitly remove the existing waypoint, since removing the callback doesn't seem to do it
    FlightHUD:AddQHWaypoint ()
    FlightHUD.CBid["qh"] = nil
  end
end

--[===[@debug@
function FlightHUD:Debug (level, arg)
  if FlightHUD.db.profile.EnableDebug then
    BugGrabber:StoreError(arg)
    print(level .. " - " .. eventID .. " : " .. arg)
  end
end
--@end-debug@]===]
