--[[
Name: EnchantingSell-Util-1.0
Author(s): Shamino
Website: 
Documentation: 
SVN: 
Description: A utility library for Enchanting
Dependencies: AceLibrary
]]

local MAJOR_VERSION = "EnchantingSell-Util-1.0"
local MINOR_VERSION = 1000

if not AceLibrary then error(MAJOR_VERSION .. " requires AceLibrary") end
if not AceLibrary:HasInstance("AceLocale-2.2") then error(MAJOR_VERSION .. " requires AceLocale-2.2") end

--local _, x = AceLibrary("AceLocale-2.2"):GetLibraryVersion()
--MINOR_VERSION = MINOR_VERSION * 100000 + x

if not AceLibrary:IsNewVersion(MAJOR_VERSION, MINOR_VERSION) then return end

local L = AceLibrary("AceLocale-2.2"):new("EnchantingSell")
local ESD = AceLibrary("EnchantingSell-Debug-1.0")

local EnchantingSellUtil = {}

local ES_ArmorCarac = {
	[1] = {"Boots", "Boots"};
	[2] = {"Bracer", "Bracer"};
	[3] = {"Chest", "Chest"};
	[4] = {"Cloak", "Cloak"};
	[5] = {"Gloves", "Gloves"};
	[6] = {"Oil", "Oil"};
	[7] = {"Ring", "Ring"};
	[8] = {"Rod", "Rod"};
	[9] = {"Shield", "Shield"};
	[10] = {"Trinket", "Trinket"};
	[11] = {"Wand", "Wand"};
	[12] = {"2H Weapon", "2H Weap"};
	[13] = {"Weapon", "Weapon"};
	[14] = {"Other", "Other"};
};

--currently returns the name of the current player
--will update to return the name of the selected enchanter, when the feature is working.
function EnchantingSellUtil:GetCurrentSelectedEnchanterName()
	local player = UnitName("player");
	return player;
end

function EnchantingSellUtil:IsEnchanter()
	local isEnchanter = false;
	local i=1;
	while true do
		local spellName, spellRank = GetSpellName(i, "spell" ) 
		if (not spellName) then
			break;
		end
		if (spellName == L["Enchanting"]) then
			isEnchanter=true;
			break;
		end
		i = i + 1;
	end
	return isEnchanter;
end

function EnchantingSellUtil:LaunchFunctionInCraftSpellFrame(arg1Function)
	local isEnchanter = false;
	local i=1;
	while true do
		local spellName, spellRank = GetSpellName(i, "spell" ) 
		if (not spellName) then
			break;
		end
		if (spellName == L["Enchanting"]) then
			--added back since it is no longer protected
			CastSpellByName(spellName);
			arg1Function();
			isEnchanter = true;
			break;
		end
		i = i + 1;
	end
	if (ESeller:isCraftFrameOpen() and ESeller:shouldCloseCraftFrame()) then CloseCraft() end;
	return isEnchanter;
end

function EnchantingSellUtil:RetrieveCraftList(enchantList, reagentList)
	self:LaunchFunctionInCraftSpellFrame(
		function()
			local reagentIndex={};

			-- iterate all the enchantements to find their information
			for i=1, GetNumCrafts(), 1 do
				local name, craftSubSpellName, craftType, numAvailable, isExpanded, trainingPointCost, requiredLevel = GetCraftInfo(i);
				local reagents = {};
				local nameOnly, bonusTexte, onThis, bonus, bonusNb;
				
				-- if the craftType=="header", then we need to skip this one.  Header is used for defining groups of enchants. 
				-- ie.  Boots, Bracer, Gloves are "headers"
				if (craftType~="header") then
					-- Seek all Reagents of enchants and place into tmp table (reagents)
					local numReagents = GetCraftNumReagents(i);
					for j=1, numReagents, 1 do
						local reagentName, reagentTexture, reagentCount = GetCraftReagentInfo(i, j);
						local reagentlink = GetCraftReagentItemLink(i,j);

						--do some data checking 
						if (reagentName~=nil) then
							tinsert(reagents, {Name = reagentName, Count = reagentCount});

							if (reagentIndex[reagentName]) then
							else
								local temp = {};
								temp.Texture = reagentTexture;
								temp.Link = reagentlink;
								temp.Name = reagentName;
								temp.IsUse = true;
								reagentList[#reagentList+1]=temp;

								reagentIndex[reagentName]=true;
							end
						end
					end
					tinsert(enchantList, {
						["IdOriginal"] = i ,
						["EnchantName"] = name,
						["Description"] = GetCraftDescription(i),
						["Icon"] = GetCraftIcon(i),
						["Reagents"] = reagents,
						["Required"] = GetCraftSpellFocus(i),
						["Link"] = GetCraftItemLink(i),
						["IsKnow"] = true,
					});					

				end
			end
		end
	);
end

function EnchantingSellUtil:UpdateKnownEnchants(data, value)
	for i, enchants in ipairs(data) do
		enchants["IsKnow"] = false;
	end
end

function EnchantingSellUtil:MergeEnchantTables(database, new)
	for i, newEnchants in ipairs(new) do
		local newEnchant = true;
		local name=newEnchants.EnchantName;
		-- Update the table of the enchantements
		for j, dbEnchants in ipairs(database) do
			if (name == dbEnchants.EnchantName) then
				dbEnchants.IdOriginal = newEnchants.IdOriginal;
				dbEnchants.IsKnow = true;
				newEnchant = false;
				break;
			end
		end
		if newEnchant then
			local index=#database+1;
			database[index].IdOriginal = newEnchants.IdOriginal;
			database[index].EnchantName = newEnchants.EnchantName;
			database[index].Description = newEnchants.Description;
			database[index].Icon = newEnchants.Icon;
			database[index].Reagents = newEnchants.Reagents;
			database[index].Required = newEnchants.Required;
			database[index].Link = newEnchants.Link;
			database[index].IsKnow = newEnchants.IsKnow;
		end
	end
end
function EnchantingSellUtil:MergeReagentTables(database, new)
	for i, newReagents in ipairs(new) do
		local newReagent = true;
		local name=newReagents.Name;

		-- Update the table of the reagents
		for j, dbReagents in ipairs(database) do
			if (name == dbReagents.Name) then
				dbReagents.IsUse = true;
				newReagent = false;
				break;
			end
		end
		if newReagent then
			local index=#database+1;
			database[index].Texture = newReagents.Texture;
			database[index].Link = newReagents.Link;
			database[index].Name = newReagents.Name;
			database[index].IsUse = newReagents.IsUse or false;

		end
	end

end
function EnchantingSellUtil:AddBonusInfo(enchantTable)
--	ESD:addLine("EnchantingSellUtil:AddBonusInfo called");

	for k,v in pairs(enchantTable) do
		local nameOnly, bonusTexte, onThis, bonus, bonusNb;

		-- Seperate the name and the bonus from the craft name --ie.. name = Enchant Boots - Lesser Stamina
		--ie.. nameOnly = Enchant Boots
		_,_,nameOnly = string.find(enchantTable[k].EnchantName, L["PATTERN_EnchantName"]);

		--ie.. bonusTexte = Lesser Stamina
		_,_,bonusTexte = string.find(enchantTable[k].EnchantName, L["PATTERN_Quality"]);
		
		--for crafts that do not fit the above patterns, then nameOnly = name.  ie...Runed Arcanite Rod
		if not nameOnly then nameOnly = enchantTable[k].EnchantName end
			
		-- Seek on which equipment one can put enchantment
		for j, Armor in ipairs(L["ARMOR_TYPES"]) do
			if string.find(nameOnly, Armor[1]) then
				onThis = Armor[2];
				break;
			end
		end

		-- if onThis has not been determined, it may be a rod or wand
		if not onThis  then 
			for j, nameObj in pairs(L["OBJECT_TYPES"]) do
--				ESD:addLine("nameOnly="..nameOnly..", nameObj="..nameObj[1])
				if nameObj[1] == nameOnly then
					onThis = nameObj[3];
					bonus = nameObj[2];
					break;
				end
			end
		end

		-- if not able to determine onThis, then make it other.
		if not onThis then onThis = L["ARMOR_TYPES"].Other end

		-- Seek Type and Bonus value for each enchantment
		if (bonusTexte) then
			-- Seek the Type of the no-claims bonus in the EnchantingSell_BonusCarac.Type table
			-- In order to indicate the more readable no-claims bonus and more room in the list
			for j, bonusCarac in pairs(L["BONUS_TYPES"]) do
				-- Seek the Type of no-claims bonus
				-- Also test, if particular case or case normal
				if ((string.find(bonusTexte, bonusCarac[1])) and ((not bonusCarac[4]) or (bonusCarac[4] == onThis))) then
					bonus = bonusCarac[2];
					-- Si une table de Bonus existe pour le Type trouvé recherche du bonus dans la table EnchantingSell_Quality referencé dans EnchantingSell_BonusCarac["Type"][j][2]
					if (bonusCarac[3]) then
						local bonustextetmp = bonusTexte;
						-- Si bonusTexte = bonuscarac sans supplement, alors le bonusNb = bonus standard soit la valeur 3 dans les tables EnchantingSell_Quality						
						if (string.len(bonusTexte) == string.len(bonusCarac[1])) then
							bonustextetmp = "None";
						end				
						-- Si supplement alors recherche dans la table EnchantingSell_Quality referencé dans EnchantingSell_BonusCarac[j][3] le bonus 
						for k, bonusAdd in pairs(bonusCarac[3]) do
							-- Si bonus trouvé alors 'bonusNb' est mis a jour avec la valeur bonusAdd de la table EnchantingSell_Quality
							if string.find(bonustextetmp, bonusAdd[1]) then
								bonusNb = bonusAdd[2];
								break;
							end
						end
					end
					break;
				end
			end

			-- Si aucune table de type de bonus ne correspond mettre directement la descrition tel quel dans 'bonus'
			if not bonus then
				bonus = bonusTexte;
			end
		else
			bonusTexte=nameOnly;
		end

		-- if bonus has not been determined, it may be an Oil
		if not bonus  then 
			for j, nameObj in pairs(L["OBJECT_TYPES"]) do
				if nameObj[1] == nameOnly then
					bonus = nameObj[2];
					break;
				end
			end
		end

		enchantTable[k].Name=nameOnly;
		enchantTable[k].OnThis=onThis;
		enchantTable[k].Bonus=bonus;
		enchantTable[k].BonusNb=bonusNb;
		enchantTable[k].Quality=bonusTexte;
	end

end

function EnchantingSellUtil:UpdateReagentNumberInBank(edata, rdata, playerName)
	if (ESeller:isBankFrameOpen()) then
		self:ResetBankCount(rdata, playerName);

		--as of 2.00.
		--container -1 is main bank
		--containers 0-4 are normal inventory bags
		--containers 5-11 are bank bags

		--there are now 7 bank bag slots
		--there are 28 generic bank slots
		--GetNumBankSlots()   - Returns total purchased bank bag slots, and a flag indicating if it's full.
		
		for container=-1, (GetNumBankSlots()+4), 1 do
			if (container >= 0 and container<5) then 
				--do nothing, just skip these since they are not bank bags.
			else
				for slot=1, GetContainerNumSlots(container), 1 do
					local itemName = self:NameFromLink(GetContainerItemLink(container,slot)) ;
					for reagent=1, #rdata, 1 do
						if (itemName == rdata[reagent].Name) then
							local texture, itemCount, locked, quality, readable = GetContainerItemInfo(container,slot);
							rdata[reagent].ByPlayer[playerName].NbInBank=
								(itemCount + rdata[reagent].ByPlayer[playerName].NbInBank);
						end
					end
				end
			end
		end
	end
end

function EnchantingSellUtil:UpdateReagentNumberInBag(edata, rdata, playerName)
	self:ResetBagCount(rdata, playerName);
	for container=0, 4, 1 do
		for slot=1, GetContainerNumSlots(container), 1 do
			local itemName = self:NameFromLink(GetContainerItemLink(container,slot)) ;
			for reagent=1, #rdata, 1 do
				if (itemName == rdata[reagent].Name) then
					local texture, itemCount = GetContainerItemInfo(container,slot);
					rdata[reagent].ByPlayer[playerName].NbInBag = 
						itemCount + rdata[reagent].ByPlayer[playerName].NbInBag;
				end
			end
--			if EnchantingSell_Config and ESell_Player_IsEq(EnchantingSell_CourantPlayer, EnchantingSell_Config.EnchantorPlayerSelected) and
--				EnchantingSell_ListComponant["Required"] then
--				for required=1, getn(EnchantingSell_ListComponant["Required"]), 1 do
--					if (itemName == EnchantingSell_ListComponant["Required"][required]["Name"]) then
--						local texture, itemCount = GetContainerItemInfo(container,slot);
--						EnchantingSell_ListComponant["Required"][required]["NbInBag"] = itemCount + EnchantingSell_ListComponant["Required"][required]["NbInBag"];
--					end
--				end
--			end
		end
	end	
	self:UpdateReagentNumberInBank(edata, rdata, playerName);
end


function EnchantingSellUtil:UpdateFeasibility(edata, rdata, playerName, feasibilityTable)
	for enchantIndex=1, #edata, 1 do
		local feasibility = self:RodCheck(enchantIndex);

		if edata[enchantIndex].IsKnow then
			for reagentId=1, #edata[enchantIndex].Reagents, 1 do
				local reagentStats = self:GetReagentStats(edata, rdata, enchantIndex, reagentId, playerName);

				if (reagentStats > feasibility) then
					feasibility = reagentStats;
				end
			end
		else
			feasibility = -2;
		end
		local enchantName=edata[enchantIndex].EnchantName;
		feasibilityTable[enchantName] = feasibility;
	end
end

function EnchantingSellUtil:RodCheck()
	--hard coded for now
	return 1;
end

function EnchantingSellUtil:GetReagentStats(edata, rdata, enchantIndex, reagentId, playerName)
	local reagentStats = 1;

	local reagentName = edata[enchantIndex].Reagents[reagentId].Name;
	local reagentCount = edata[enchantIndex].Reagents[reagentId].Count;
	local reagentIndex = self:GetReagentId(rdata, reagentName);

	local nbInBag, nbInBank, nbInReroll = self:GetReagentCount(rdata, reagentIndex, playerName);
	if not nbInBag then return 4 end
	if (nbInBag < reagentCount) then
		reagentStats = 2;
		if ((nbInBag + nbInBank) < reagentCount) then
			reagentStats = 3;
			if ((nbInBag + nbInBank + nbInReroll) < reagentCount) then
				reagentStats = 4;
			end
		end
	end
	return reagentStats;
end

function EnchantingSellUtil:GetReagentCount(rdata, reagentId, player)
--	if (not player) then player = EnchantingSell_CourantPlayer; end
	local tableNbArg = rdata[reagentId].ByPlayer;
	local nbInBag, nbInBank, nbInReroll = self:GetReagentCountFromTable(tableNbArg, player);
	return nbInBag, nbInBank, nbInReroll;
end

function EnchantingSellUtil:GetReagentAltCounts(tableNbArg, namePlayer)
	local _,_,nbInReroll = self:GetReagentCountFromTable(tableNbArg, namePlayer);
	return nbInReroll;
end

function EnchantingSellUtil:GetReagentCountFromTable(tableNbArg, namePlayer)	
	local nbInBank = 0;
	local nbInReroll =0;
	local nbInBag =0;	
	if (tableNbArg) then
		for name,bagTable in pairs(tableNbArg) do
			if (name == namePlayer) then
				nbInBag = bagTable["NbInBag"];
				nbInBank = bagTable["NbInBank"];
			else
				nbInReroll = (bagTable["NbInBank"] + bagTable["NbInBag"] + nbInReroll);
			end
		end
	end
	return nbInBag, nbInBank, nbInReroll;
end

function EnchantingSellUtil:ResetBagCount(data, playerName)
	for k,v in pairs(data) do
		data[k].ByPlayer[playerName].NbInBag=0;
	end
end

function EnchantingSellUtil:ResetBankCount(data, playerName)
	for k,v in pairs(data) do
		data[k].ByPlayer[playerName].NbInBank=0;
	end
end

function EnchantingSellUtil:NameFromLink(link)
	if( not link ) then
		return nil;
	end
	local itemName = GetItemInfo(link);
	return itemName;
end

function EnchantingSellUtil:UpdateEnchantPrice()

end

function EnchantingSellUtil:GetReagentId(data, reagentName)
	for index=1, #data, 1 do
		local name = data[index].Name;
		if (name == reagentName) then
			return index;
		end
	end
	return nil;
end
function EnchantingSellUtil:GetReagentLink(data, reagentName)
	for index=1, #data, 1 do
		local name = data[index].Name;
		if (name == reagentName) then
			return data[index].Link;
		end
	end
	return nil;
end

function EnchantingSellUtil:isNullOrFalse(value)
	if (value) then 
		return true;
	else 
		return false;
	end
end

function EnchantingSellUtil:isNullOrValue(value)
	if (value) then 
		return value;
	else 
		return "nil";
	end
end

function EnchantingSellUtil:getTrueOrFalse(value)
	if (value) then 
		return "true";
	else 
		return "false";
	end
end

function EnchantingSellUtil:copyTable(to, from)
	setmetatable(to, nil)
	for k,v in pairs(from) do
		if type(k) == "table" then
			k = self:copyTable({}, k)
		end
		if type(v) == "table" then
			v = self:copyTable({}, v)
		end
		to[k] = v
	end
	setmetatable(to, from)
	return to
end


AceLibrary:Register(EnchantingSellUtil, MAJOR_VERSION, MINOR_VERSION)
EnchantingSellUtil=nil;
