﻿-- Author      : Kurapica
-- Create Date : 06/16/2010
-- ChangeLog   :

----------------------------------------------------------------------------------------------------------------------------------------
--- A Logger is a table to contains 
-- @name Logger
-- @class table
-- @field LogLevel the logging system's loglevel, if log message's loglevel is lower than Logger.LogLevel, that message will be discarded.
-- @field MaxLog the log message will be stored in the Logger temporarily , you can using Logger[1], Logger[2] to access it, the less index the newest log message.When the message number is passed Logger.MaxLog, the oldest messages would be discarded, the default MaxLog is 1.
-- @field TimeFormat if the timeformat is setted, the log message will add a timestamp at the header.<br><br>
-- Time Format:<br>
-- %a	abbreviated weekday name (e.g., Wed)<br>
-- %A	full weekday name (e.g., Wednesday)<br>
-- %b	abbreviated month name (e.g., Sep)<br>
-- %B	full month name (e.g., September)<br>
-- %c	date and time (e.g., 09/16/98 23:48:10)<br>
-- %d	day of the month (16) [01-31]<br>
-- %H	hour, using a 24-hour clock (23) [00-23]<br>
-- %I	hour, using a 12-hour clock (11) [01-12]<br>
-- %M	minute (48) [00-59]<br>
-- %m	month (09) [01-12]<br>
-- %p	either "am" or "pm" (pm)<br>
-- %S	second (10) [00-61]<br>
-- %w	weekday (3) [0-6 = Sunday-Saturday]<br>
-- %x	date (e.g., 09/16/98)<br>
-- %X	time (e.g., 23:48:10)<br>
-- %Y	full year (1998)<br>
-- %y	two-digit year (98) [00-99]<br>
----------------------------------------------------------------------------------------------------------------------------------------
do
	-- Check Version
	local version = 1
	
	------------------------------------------------------
	-- Version Check
	------------------------------------------------------
	local _G = _G
	local rawset = rawset
	local rawget = rawget
	local setfenv = setfenv
	local floor = math.floor
	local tinsert = tinsert
	local tremove = tremove
	local getn = getn
	
	IGAS.__LOGSYS_ENV = IGAS.__LOGSYS_ENV or setmetatable({}, {
		__index = function(self, key)
			if _G[key] then
				rawset(self, key, _G[key])
				
				return rawget(self, key)
			end
		end,
	})	
	setfenv(1, IGAS.__LOGSYS_ENV)
	
	if IGAS_LOGSYS_VERSION and IGAS_LOGSYS_VERSION >= version then
		return
	end
	IGAS_LOGSYS_VERSION = version
	
	------------------------------------------------------
	-- Main Trunk
	------------------------------------------------------	
	_Loggers = _Loggers or {}
	_LogLevels = _LogLevels or {}
	_LogMaxLogs = _LogMaxLogs or {}
	_LoggerPools = _LoggerPools or {}
	_LoggerPrefixs = _LoggerPrefixs or {}
	_LoggerHandlers = _LoggerHandlers or {}
	_LoggerTimeFormats = _LoggerTimeFormats or {}
	
	-- Error Handler
	local function errorhandler(err)
		return geterrorhandler()(err)
	end
	
	-- Metatable for methods
	local _MetaFunc = {
		------------------------------------
	    --- Logout message
		-- @name Logger:Log
		-- @class function
		-- @param loglevel the output message's loglevel, if lower than Logger.LogLevel, the message will be discarded.
		-- @param message the output message
		-- @return nil
		-- @usage Logger:Log(1, "Something wrong")
		------------------------------------

		-- Log
		["Log"] = function(self, logLvl, msg)
			if logLvl and type(logLvl) == "number" and logLvl >= self.LogLevel and msg and type(msg) == "string" then
				-- Prefix and TimeStamp
				local prefix = self.TimeFormat and date(self.TimeFormat)
				
				if not prefix or type(prefix) ~= "string" then
					prefix = ""
				end
				
				if prefix ~= "" and not strmatch(prefix, "^%[.*%]$") then 
					prefix = "["..prefix.."]" 
				end
				
				prefix = prefix..(_LoggerPrefixs[self][logLvl] or "")
				
				msg = prefix..msg
				
				-- Save message to pool
				_LoggerPools[self][_LoggerPools[self].EndLog] = msg
				_LoggerPools[self].EndLog = _LoggerPools[self].EndLog + 1
				
				-- Remove old message
				while _LoggerPools[self].EndLog - _LoggerPools[self].StartLog - 1 > self.MaxLog do
					_LoggerPools[self].StartLog = _LoggerPools[self].StartLog + 1
					_LoggerPools[self][_LoggerPools[self].StartLog] = nil
				end
				
				-- Send message to handlers
				local chk, err
				
				for handler in pairs(_LoggerHandlers[self]) do
					chk, err = pcall(handler, msg)
					if not chk then
						errorhandler(err)
					end
				end
			end
		end,
		
		------------------------------------
	    --- Add a log handler, the handler must be a function and receive <b>message</b> as arg.
		-- @name Logger:AddHandler
		-- @class function
		-- @param handler a function to handle the log message
		-- @return nil
		-- @usage Logger:AddHandler(print) -- this would print the message out to the ChatFrame
		------------------------------------

		-- AddHandler
		["AddHandler"] = function(self, handler)
			if handler and type(handler) == "function" then
				_LoggerHandlers[self][handler] = true
			end
		end,
		
		------------------------------------
	    --- Remove a log handler
		-- @name Logger:RemoveHandler
		-- @class function
		-- @param handler a function to handle the log message
		-- @return nil
		-- @usage Logger:RemoveHandler(print) -- this would print the message out to the ChatFrame
		------------------------------------

		-- RemoveHandler
		["RemoveHandler"] = function(self, handler)
			if handler and type(handler) == "function" then
				_LoggerHandlers[self][handler] = nil
			end
		end,
	
		------------------------------------
	    --- Set a prefix for a log level, the prefix will be added to the message when the message is that log level.
		-- @name Logger:SetPrefix
		-- @class function
		-- @param logLevel the log message's level
		-- @param prefix the prefix string
		-- @return nil
		-- @usage Logger:SetPrefix(1, "[DEBUG]") -- this would print the message out to the ChatFrame
		------------------------------------

		-- SetPrefix
		["SetPrefix"] = function(self, loglvl, prefix)
			if prefix and type(prefix) == "string" then
				if not strmatch(prefix, "%W+$") then
					prefix = prefix.." "
				end
			else
				prefix = nil
			end
			_LoggerPrefixs[self][loglvl] = prefix
		end,
	}
	
	-- Metatable for properties
	local _MetaProp = {
		-- LogLevel
		["LogLevel"] = {
			["Set"] = function(self, lvl)
				if not lvl or type(lvl) ~= "number" then
					error("The LogLevel must be a number", 2)
				end
				
				if lvl < 0 then lvl = 0 end
				
				_LogLevels[self] = floor(lvl)
			end,
			["Get"] = function(self)
				return _LogLevels[self] or 0
			end,
		},
		-- MaxLog
		["MaxLog"] = {
			["Set"] = function(self, maxv)
				if not maxv or type(maxv) ~= "number" or maxv < 1 then
					error("The MaxLog must be no less than 1.", 2)
				end
				
				if maxv < 1 then maxv = 1 end
				
				_LogMaxLogs[self] = floor(maxv)
				
				while _LoggerPools[self].EndLog - _LoggerPools[self].StartLog - 1 > _LogMaxLogs[self] do
					_LoggerPools[self].StartLog = _LoggerPools[self].StartLog + 1
					_LoggerPools[self][_LoggerPools[self].StartLog] = nil
				end
			end,
			["Get"] = function(self)
				return _LogMaxLogs[self] or 1
			end,
		},
		-- TimeFormat
		["TimeFormat"] = {
			["Set"] = function(self, timeFormat)
				if timeFormat and type(timeFormat) == "string" and timeFormat ~= "*t" then
					_LoggerTimeFormats[self] = timeFormat
				else
					_LoggerTimeFormats[self] = nil
				end
			end,
			["Get"] = function(self)
				return _LoggerTimeFormats[self]
			end,
		},
	}
	
	-- IGAS Logger, the base proxy
	_Loggers["IGAS"] = _Loggers["IGAS"] or newproxy(true)
	
	_Meta = _Meta or getmetatable(_Loggers["IGAS"])
	
	_Meta.__index = function(logger, key)
		-- Property Get
		if _MetaProp[key] then
			if _MetaProp[key]["Get"] then
				return _MetaProp[key]["Get"](logger)
			else
				error(key.." is write-only.",2)
			end
		end

		if _MetaFunc[key] then
			return _MetaFunc[key]
		end
		
		if type(key) == "number" and key >= 1 then
			key = floor(key)
			
			return _LoggerPools[logger][_LoggerPools[logger].EndLog - key]
		end
	end	
	
	_Meta.__newindex = function(logger, key, value)
		-- Property Set
		if _MetaProp[key] then
			if _MetaProp[key]["Set"] then
				return _MetaProp[key]["Set"](logger, value)
			else
				error(key.." is read-only.", 2)
			end
		end

		if _MetaFunc[key] then
			error(key.." is read-only.", 2)
		end
		
		-- no need to rawset data to logger, a Logger is just a proxy.
	end
	
	_Meta.__len = function(logger)
		return _LoggerPools[logger].EndLog - _LoggerPools[logger].StartLog - 1
	end
	
	_Meta.__call = function(logger, loglvl, msg)
		return logger:Log(loglvl, msg)
	end
	
	-- Protect metatable, making it not easy to be overwrited, since this lib is embed-able, so there are always ways to access the _Meta.
	_Meta.__metatable = {}
	
	------------------------------------
	--- Create or get the logger for the given log name
	-- @name IGAS:NewLogger
	-- @class function
	-- @param LogName always be the addon's name, using to manage an addon's message
	-- @return Logger used for the log name
	-- @usage IGAS:NewLogger("IGAS")
	------------------------------------
	IGAS:RegisterFunction("NewLogger", function(self, LogName)
		local logger = _Loggers[LogName] or newproxy(_Loggers["IGAS"])
		
		_Loggers[LogName] = logger
		
		_LoggerPools[logger] = _LoggerPools[logger] or {["StartLog"] = 0, ["EndLog"] = 1}
		_LoggerHandlers[logger] = _LoggerHandlers[logger] or {}
		_LoggerPrefixs[logger] = _LoggerPrefixs[logger] or {}
		
		return logger
	end)
	
	------------------------------------------------------
	-- Special Settings for IGAS
	------------------------------------------------------
	local Log = IGAS:NewLogger("IGAS")
	
	Log.TimeFormat = "%X"
	Log:SetPrefix(1, "[IGAS][Trace]")
	Log:SetPrefix(2, "[IGAS][Debug]")
	Log:SetPrefix(3, "[IGAS][Info]")
	Log:SetPrefix(4, "[IGAS][Warn]")
	Log:SetPrefix(5, "[IGAS][Error]")
	Log:SetPrefix(6, "[IGAS][Fatal]")
end