﻿-- Author       : Kurapica
-- Create Date  : 9/6/2008
-- Change Log   :
--					2009.12.29	Add New ScriptType: OnClick, OnDoubleClick
--					2010.01.23 Add Some function to TreeNode

---------------------------------------------------------------------------------------------------------------------------------------
--- TreeView is used to display hierarchical data.
-- <br><br>inherit <a href=".\ScrollForm.html">ScrollForm</a> For all methods, properties and scriptTypes
-- @name TreeView
-- @class table
-- @field ChildNodeNum the child-node's count, readonly
-- @field TogglePosition the toggle button's position, LEFT or RIGHT
---------------------------------------------------------------------------------------------------------------------------------------

--[[
		example tree

		Alpha
		Bravo
		  -Charlie
		  -Delta
			-Echo
		Foxtrot

		tree = {
			{
				["Value"] = "A",
				["Text"] = "Alpha",
                ["Menu"] = DropDownList1,
			},
			{
				["Value"] = "B",
				["Text"] = "Bravo",
                ["Menu"] = DropDownList2,
				["Childs"] = {
					{
						["Value"] = "C",
						["Text"] = "Charlie",
					},
					{
						["Value"] = "D",
						["Text"] = "Delta",
						["Childs"] = {
							{
								["Value"] = "E",
								["Text"] = "Echo",
                                ["Menu"] = DropDownList3,
							}
						}
					}
				}
			},
			{
				["Value"] = "F",
				["Text"] = "Foxtrot"
			},
		}
--]]
do
	-- Check Version
	local version = 5
	if not IGAS:NewAddon("IGAS.GUI.TreeView", version) then
		return
	end

    local type = type
    local error = error
	local gsub = gsub
    local CreateFrame = IGAS.GUI.CreateFrame

    -- Style
    local TEMPLATE_LEFT = "LEFT"
    local TEMPLATE_RIGHT = "RIGHT"

    -- Define Block
	local _ToggleEnum = IGAS:NewEnum("TreeViewTogglePosition", {
        TEMPLATE_LEFT,
		TEMPLATE_RIGHT,
    })

	local _ToggleType = IGAS:NewPropertyType("TreeViewTogglePosition", {
		["Type"] = "enum",
		["EnumType"] = _ToggleEnum,
	})
	
	---------------------------------------------------------------------------------------------------------------------------------------
	--- TreeNode is using as node in the TreeView, only can be created by TreeView
	-- <br><br>inherit <a href="..\Base\Frame.html">Frame</a> For all methods, properties and scriptTypes
	-- @name TreeNode
	-- @class table
	-- @field Text The text that displayed on the TreeNode
	-- @field ToggleState Whether the TreeNode is toggled or not
	-- @field Menu the menu of the TreeNode
	-- @field CanToggle Whether the treenode has child-treenode
	-- @field ChildNodeNum the child-treenode's count
	---------------------------------------------------------------------------------------------------------------------------------------
    local TreeNode = (IGAS.GUI.Widget.TreeView and IGAS.GUI.Widget.TreeView.TreeNode) or nil

    do
        local _WidgetName = "TreeNode"
        local _Base = "Frame"

        local nodeHeight = 18
        local nodeIndent = 10

        -- Scripts
		local function FixHeight(self)
			local height = nodeHeight

			if self:GetChild("TreeNode1") then
				if self.__NodeNum > 0 then
					for i = 1, self.__NodeNum do
						if self:GetChild("TreeNode"..i) and self:GetChild("TreeNode"..i).Visible then
							height = height + self:GetChild("TreeNode"..i).Height
						end
					end
				end
			end

			self.Height = height
		end

		local function OnSizeChanged(self)
			FixHeight(self.Parent)
		end

        local function OnClick(self, button, down)
            local root = self.Parent.__Root

            if button == "RightButton" then
                -- Open Menu
                if self.Parent.__DropDownList then
                    self.Parent.__DropDownList.Visible = false
                    self.Parent.__DropDownList.Parent = self.Parent
                    self.Parent.__DropDownList.Visible = true
                end
                return
            end

			self.Parent:Select()

			return self.Parent:Fire("OnClick", button, down)
        end

		local function OnDoubleClick(self, ...)
			return self.Parent:Fire("OnDoubleClick", ...)
		end

		local function ToggleBtn_OnClick(self)
			self.Parent.ToggleState = not self.Parent.ToggleState
		end

        -- ScriptType
        local _ScriptType = {
			------------------------------------
			--- ScriptType, Run when the treenode is toggled
			-- @name TreeNode:OnToggle
			-- @class function
			-- @usage function TreeNode:OnToggle()<br>
			--    -- do someting<br>
			-- end
			------------------------------------
            ["OnToggle"] = true,

			------------------------------------
			--- ScriptType, Run when the treenode is dbl-clicked
			-- @name TreeNode:OnDoubleClick
			-- @class function
			-- @usage function TreeNode:OnDoubleClick()<br>
			--    -- do someting<br>
			-- end
			------------------------------------
			["OnDoubleClick"] = true,

			------------------------------------
			--- ScriptType, Run when the treenode is clicked
			-- @name TreeNode:OnClick
			-- @class function
			-- @usage function TreeNode:OnClick()<br>
			--    -- do someting<br>
			-- end
			------------------------------------
			["OnClick"] = true,
        }

        -- FuncProxy
        local _FuncProxy = {
			------------------------------------
			--- Gets the text that displayed on the TreeNode
			-- @name TreeNode:GetText
			-- @class function
			-- @return the displayed text
			-- @usage TreeNode:GetText()
			------------------------------------

			-- GetText
			["GetText"] = function(self)
				return self:GetChild("TextBtn"):GetChild("Text").Text
			end,

			------------------------------------
			--- Gets the value that mark the treenode, ex: return "Main", "Second"
			-- @name TreeNode:GetValue
			-- @class function
			-- @return text treenode's value
			-- @usage TreeNode:GetValue()
			------------------------------------

			-- GetValue
			["GetValue"] = function(self)
                return strsplit("\001", self.__Value or "")
			end,

			------------------------------------
			--- Add child-node to the treenode
			-- @name TreeNode:AddNode
			-- @class function
			-- @param node a table contains node informations
			-- @usage TreeNode:AddNode{["Text"] = "Show Text", ["Value"] = "Second",}
			------------------------------------

			-- AddNode
			["AddNode"] = function(self, node)
				local root = self.__Root
				local value

				if node and type(node) == "table" and
					(node.Text and type(node.Text) == "string" and node.Text ~= "") and
					(node.Value and type(node.Value) == "string" and node.Value ~= "") then

					-- Value Check
					value = self.__Value.."\001"..node.Value
					if root.__Nodes[value] then
						error("This node["..(gsub(value, "\001", ",")).."] is exist", 2)
					end

					local frame = CreateFrame(TreeNode, nil, self)
					frame:SetNode(node)

					return frame
				end
			end,

			------------------------------------
			--- Gets a child-node by index
			-- @name TreeNode:GetNode
			-- @class function
			-- @param index the index of the child-node
			-- @return the child-node
			-- @usage TreeNode:GetNode(1)
			------------------------------------

			-- GetNode
            ["GetNode"] = function(self, index)
				return (index and type(index) == "number" and self:GetChild("TreeNode"..index)) or nil
            end,

			------------------------------------
			--- Removes a child-node by index
			-- @name TreeNode:RemoveNode
			-- @class function
			-- @param index the index of the child-node
			-- @usage TreeNode:RemoveNode(1)
			------------------------------------

			-- RemoveNode
			["RemoveNode"] = function(self, index)
				if index and type(index) == "number" and self:GetChild("TreeNode"..index) then
					self:GetChild("TreeNode"..index):Dispose()
				end
			end,

			------------------------------------
			--- Gets a child-node by value
			-- @name TreeNode:GetNodeByValue
			-- @class function
			-- @param ... the value list of a child-node
			-- @return the child node
			-- @usage TreeNode:GetNodeByValue("Second")
			------------------------------------

			-- GetNodeByValue
			["GetNodeByValue"] = function(self, ...)
				local value, e

				if select('#', ...) == 0 then
					return nil
				end

				value = self.__Value or ""

				for i = 1, select('#', ...) do
					e = select(i, ...)
					if e and type(e) == "string" then
						if value == "" then
							value = e
						else
							value = value.."\001"..e
						end
					end
				end

				if value ~= "" and self.__Root.__Nodes[value] then
					return self.__Root.__Nodes[value]
				end
			end,

			------------------------------------
			--- Removes a child-node by value
			-- @name TreeNode:RemoveNodeByValue
			-- @class function
			-- @param ... the value list of a child-node
			-- @usage TreeNode:RemoveNodeByValue("Second")
			------------------------------------

			-- RemoveNodeByValue
			["RemoveNodeByValue"] = function(self, ...)
				local value, e

				if select('#', ...) == 0 then
					return nil
				end

				value = self.__Value or ""
				for i = 1, select('#', ...) do
					e = select(i, ...)
					if e and type(e) == "string" then
						if value == "" then
							value = e
						else
							value = value.."\001"..e
						end
					end
				end

				if value ~= "" and self.__Root.__Nodes[value] then
					self.__Root.__Nodes[value]:Dispose()
				end
			end,

			------------------------------------
			--- Whether the TreeNode is selected
			-- @name TreeNode:IsSelected
			-- @class function
			-- @return true if the treenode is selected
			-- @usage TreeNode:IsSelected()
			------------------------------------

			-- IsSelected
			["IsSelected"] = function(self)
				return (self.__Root.__SelectedItem == self)
			end,

			------------------------------------
			--- Select the TreeNode
			-- @name TreeNode:Select
			-- @class function
			-- @usage TreeNode:Select()
			------------------------------------

			-- Select
			["Select"] = function(self)
				local root = self.__Root
				local parent = self.Parent

				-- Toggle
				while parent and parent ~= root do
					parent.ToggleState = true
					parent = parent.Parent
				end

				-- Select
				if root.__SelectedItem ~= self then
					if root.__SelectedItem then
						root.__SelectedItem:GetChild("TextBtn"):UnlockHighlight()
					end
					self:GetChild("TextBtn"):LockHighlight()
					root.__SelectedItem = self
					root:Fire("OnNodeClk")
				end
			end,

			------------------------------------
			--- Reset the TreeNode's information
			-- @name TreeNode:SetNode
			-- @class function
			-- @usage TreeNode:SetNode{["Text"] = "Show Text", ["Value"] = "Main",}
			------------------------------------

			-- SetNode
			["SetNode"] = function(self, node)
				local root = self.__Root
				local value
				local index
				local idx, cht
				local childNode

				if node and type(node) == "table" and
					(node.Text and type(node.Text) == "string" and node.Text ~= "") and
					(node.Value and type(node.Value) == "string" and node.Value ~= "") then

					-- Text
					self:GetChild("TextBtn"):GetChild("Text").Text = node.Text

					-- Value
					value = ""
					if self.Parent.__Value and self.Parent.__Value ~= "" then
						value = self.Parent.__Value.."\001"
					end
					value = value..node.Value

					if self.__Value and root.__Nodes[self.__Value] == self then
						root.__Nodes[self.__Value] = nil
					end
					self.__Value = value
					root.__Nodes[self.__Value] = self

					-- Menu
					if node.Menu and type(node.Menu) == "table" and node.Menu["IsObjectType"] and node.Menu:IsObjectType("DropDownList") then
						self.__DropDownList = node.Menu
					end

					-- Childs
					index = 0
					--- clear child's key
					if self:GetChild("TreeNode1") then
						if self.__NodeNum > 0 then
							for i = 1, self.__NodeNum do
								childNode = self:GetChild("TreeNode"..i)
								if childNode.__Value and root.__Nodes[childNode.__Value] == childNode then
									root.__Nodes[childNode.__Value] = nil
								end
								childNode.__Value = nil
							end
						end
					end
					--- Set node's child
					if node.Childs and type(node.Childs) == "table" then
						idx, cht = next(node.Childs)
						while(idx) do
							if cht and type(cht) == "table" and
								(cht.Text and type(cht.Text) == "string" and cht.Text ~= "") and
								(cht.Value and type(cht.Value) == "string" and cht.Value ~= "") then

								index = index + 1

								if self:GetChild("TreeNode"..index) then
									self:GetChild("TreeNode"..index):SetNode(cht)
								else
									self:AddNode(cht)
								end
							end
							idx, cht = next(node.Childs, idx)
						end
					end
					--- remove useless child-node.
					if self.__NodeNum > index then
						for i = self.__NodeNum, index+1, -1 do
							self:RemoveNode(i)
						end
					end
				end
			end,

			------------------------------------
			--- Gets the TreeNode's toggle state
			-- @name TreeNode:GetToggleState
			-- @class function
			-- @return true if the TreeNode is toggled
			-- @usage TreeNode:GetToggleState()
			------------------------------------

			-- GetToggleState
            ["GetToggleState"] = function(self)
                return self.__ToggleState
            end,

			------------------------------------
			--- Sets the TreeNode's toggle state
			-- @name TreeNode:SetToggleState
			-- @class function
			-- @param enable True if toggle the TreeNode
			-- @usage TreeNode:SetToggleState(true)
			------------------------------------

			-- SetToggleState
			["SetToggleState"] = function(self, flag)
				flag = (flag and true) or false

                if self.__ToggleState == flag then
                    return
                end

				if flag == false then
					self.__ToggleState = false
					self:GetChild("ToggleBtn"):SetNormalTexture("Interface\\Buttons\\UI-PlusButton-UP")
					self:GetChild("ToggleBtn"):SetPushedTexture("Interface\\Buttons\\UI-PlusButton-DOWN")

					if self.__NodeNum > 0 then
						for i = self.__NodeNum,1,-1 do
							self:GetChild("TreeNode"..i):Hide()
						end
					end
				else
					self.__ToggleState = true
					self:GetChild("ToggleBtn"):SetNormalTexture("Interface\\Buttons\\UI-MinusButton-UP")
					self:GetChild("ToggleBtn"):SetPushedTexture("Interface\\Buttons\\UI-MinusButton-DOWN")

					if self.__NodeNum > 0 then
						local frame
						for i = 1, self.__NodeNum do
							frame = self:GetChild("TreeNode"..i)

							frame:Show()

							-- SetPoint
                            if i == 1 then
                                frame:SetPoint("TOPRIGHT", self, "TOPRIGHT", 0, - nodeHeight)
                                frame:SetPoint("TOPLEFT", self, "TOPLEFT", nodeIndent, - nodeHeight)
                            else
                                frame:SetPoint("TOPRIGHT", self:GetChild("TreeNode"..(i-1)), "BOTTOMRIGHT", 0, 0)
                                frame:SetPoint("TOPLEFT", self:GetChild("TreeNode"..(i-1)), "BOTTOMLEFT", 0, 0)
                            end
						end
					end
				end

				FixHeight(self)

                self:Fire("OnToggle")
			end,

			-- Dispose, release resource
	        ["Dispose"] = function(self)
				local index = self.ID
                local parent = self.Parent

                self:SetToggleState(false)

				if self.__DropDownList and self.__DropDownList.Parent == self then
					self.__DropDownList.Visible = false
					self.__DropDownList.Parent = nil
					self.__DropDownList = nil
				end

                -- Dispose child node first
                if self.__NodeNum > 0 then
                    for i = self.__NodeNum, 1, -1 do
                        self:GetChild("TreeNode"..i):Dispose()
                    end
                end

                self.OnSizeChanged = nil
				self:StackScript("OnSizeChanged", nil, true)

				if parent.ToggleState or self.__Root == parent then
	                if parent:GetChild("TreeNode"..(index+1)) then
                        if index == 1 then
                            if self.__Root == parent then
                                parent:GetChild("TreeNode"..(index+1)):SetPoint("TOPRIGHT", parent, "TOPRIGHT", 0, -6)
                                parent:GetChild("TreeNode"..(index+1)):SetPoint("TOPLEFT", parent, "TOPLEFT", 0, -6)
                            else
                                parent:GetChild("TreeNode"..(index+1)):SetPoint("TOPRIGHT", parent, "TOPRIGHT", 0, - nodeHeight)
                                parent:GetChild("TreeNode"..(index+1)):SetPoint("TOPLEFT", parent, "TOPLEFT", nodeIndent, - nodeHeight)
                            end
                        else
                            parent:GetChild("TreeNode"..(index+1)):SetPoint("TOPRIGHT", parent:GetChild("TreeNode"..(index-1)), "BOTTOMRIGHT", 0, 0)
                            parent:GetChild("TreeNode"..(index+1)):SetPoint("TOPLEFT", parent:GetChild("TreeNode"..(index-1)), "BOTTOMLEFT", 0, 0)
                        end
	                end
				end

                self:ClearAllPoints()

                self:Hide()

                parent:RemoveChild(self)

				if parent.__NodeNum > index then
	                for i = index + 1, parent.__NodeNum do
						parent:GetChild("TreeNode"..i).ID = i - 1
	                    parent:GetChild("TreeNode"..i).Name = "TreeNode"..(i - 1)
	                end
				end

				parent.__NodeNum = parent.__NodeNum - 1

                FixHeight(parent)

				if parent.__NodeNum == 0 then
                    if parent:GetChild("ToggleBtn") then
					    parent:GetChild("ToggleBtn").Visible = false
                    end
				end

				if self.__Value and self.__Root.__Nodes[self.__Value] == self then
					self.__Root.__Nodes[self.__Value] = nil
				end
				self.__Value = nil

                if self.__Root.__SelectedItem == self then
                    self.__Root.__SelectedItem = nil
                end

	            -- Call super's dispose
				self:CallMethod(_Base, "Dispose")
	        end,
        }

        -- Property
        local _Property = {
			-- Text
			["Text"] = {
				["Get"] = function(self)
					return self:GetText()
				end,

				["Type"] = "string",
			},

			-- ToggleState
			["ToggleState"] = {
				["Set"] = function(self, flag)
					self:SetToggleState(flag)
				end,

				["Get"] = function(self)
					return self:GetToggleState()
				end,

				["Type"] = "boolean",
			},

			-- Menu
            ["Menu"] = {
                ["Set"] = function(self, list)
                    self.__DropDownList = list
                end,

                ["Get"] = function(self)
                    return self.__DropDownList
                end,

                ["Type"] = "DropDownList",
            },

			-- CanToggle
			["CanToggle"] = {
				["Get"] = function(self)
					if self.__NodeNum > 0 then
						return true
					else
						return false
					end
				end,

				["Type"] = "boolean",
			},

			-- ChildNodeNum
			["ChildNodeNum"] = {
				["Get"] = function(self)
					return self.__NodeNum or 0
				end,

				["Type"] = "number",
			},
			
			-- TogglePosition
			["TogglePosition"] = {
				["Set"] = function(self, position)
					if position == self.__TogglePosition then
						return
					end
					
					self.__TogglePosition = position
					
					local toggleBtn = CreateFrame("Button","ToggleBtn",frame)
					local textBtn = CreateFrame("Button", "TextBtn", frame)
					
					if position == TEMPLATE_RIGHT then
						toggleBtn:ClearAllPoints()
						toggleBtn:SetPoint("TOPRIGHT", frame, "TOPRIGHT", -6, -1)

						textBtn:SetPoint("RIGHT", frame, "RIGHT", -14, 0)
						textBtn:SetPoint("LEFT", frame, "LEFT", 4, 0)
					else					
						toggleBtn:ClearAllPoints()
						toggleBtn:SetPoint("TOPLEFT", frame, "TOPLEFT", 4, - 1)

						textBtn:SetPoint("RIGHT", frame, "RIGHT", 0, 0)
						textBtn:SetPoint("LEFT", frame, "LEFT", 14, 0)
					end
					
					for i =1, self.ChildNodeNum do
						self:GetChild("TreeNode"..i).TogglePosition = position
					end
				end,
				
				["Get"] = function(self)
					return self.Container.__TogglePosition or TEMPLATE_LEFT
				end,
				
				["Type"] = _ToggleType,
			},
        }

        -- New
        local _New = function(parent)
            local index = parent.__NodeNum or 0
            index = index + 1

			-- Base Frame
            local frame = CreateFrame("Frame", nil, parent)

			-- Init
			frame.__NodeNum = 0
			frame.ID = index
			frame.__ToggleState = false
            frame.__Root = parent.__Root or parent
			if frame.__Root ~= parent then
				frame.Visible = parent.ToggleState
			else
				frame.Visible = true
			end
            if parent:GetChild("ToggleBtn") then
                parent:GetChild("ToggleBtn").Visible = true
            end

			frame.Height = nodeHeight
			if frame.Visible then
				parent.Height = parent.Height + nodeHeight
				if index == 1 then
					if frame.__Root == parent then
						frame:SetPoint("TOPRIGHT", parent, "TOPRIGHT", 0, -6)
						frame:SetPoint("TOPLEFT", parent, "TOPLEFT", 0, -6)
					else
						frame:SetPoint("TOPRIGHT", parent, "TOPRIGHT", 0, - nodeHeight)
						frame:SetPoint("TOPLEFT", parent, "TOPLEFT", nodeIndent, - nodeHeight)
					end
				else
					frame:SetPoint("TOPRIGHT", parent:GetChild("TreeNode"..(index-1)), "BOTTOMRIGHT", 0, 0)
					frame:SetPoint("TOPLEFT", parent:GetChild("TreeNode"..(index-1)), "BOTTOMLEFT", 0, 0)
	            end
			end
			parent.__NodeNum = index

			-- Toggle Button
            local toggleBtn = CreateFrame("Button","ToggleBtn",frame)
			toggleBtn.Visible = false
            toggleBtn.Width = 14
            toggleBtn.Height = 14

			toggleBtn:SetNormalTexture("Interface\\Buttons\\UI-PlusButton-UP")
			toggleBtn:SetPushedTexture("Interface\\Buttons\\UI-PlusButton-DOWN")
            toggleBtn:SetHighlightTexture("Interface\\Buttons\\UI-PlusButton-Hilight")
            local t = toggleBtn:GetHighlightTexture()
            t:SetBlendMode("ADD")

			-- Text Button
			local textBtn = CreateFrame("Button", "TextBtn", frame)
			textBtn.Height = nodeHeight
			textBtn:SetPoint("TOP", frame, "TOP", 0, 0)
			textBtn:RegisterForClicks("AnyUp")

			frame.__TogglePosition = parent.__TogglePosition
			
			if frame.__TogglePosition == TEMPLATE_RIGHT then
				toggleBtn:ClearAllPoints()
				toggleBtn:SetPoint("TOPRIGHT", frame, "TOPRIGHT", -6, -1)

				textBtn:SetPoint("RIGHT", frame, "RIGHT", -14, 0)
				textBtn:SetPoint("LEFT", frame, "LEFT", 4, 0)
			else					
				toggleBtn:ClearAllPoints()
				toggleBtn:SetPoint("TOPLEFT", frame, "TOPLEFT", 4, - 1)

				textBtn:SetPoint("RIGHT", frame, "RIGHT", 0, 0)
				textBtn:SetPoint("LEFT", frame, "LEFT", 14, 0)
			end
			
            t = CreateFrame("FontString", "Text", textBtn, "OVERLAY")
			t:SetFontObject(GameFontNormal)
			t:SetJustifyH("LEFT")
            t:SetAllPoints(textBtn)

			textBtn:SetHighlightTexture("Interface\\QuestFrame\\UI-QuestLogTitleHighlight")
            t = textBtn:GetHighlightTexture()
            t:SetBlendMode("ADD")
			t:SetPoint("TOPLEFT", textBtn, "TOPLEFT", 0, 1)
			t:SetPoint("BOTTOMRIGHT", textBtn, "BOTTOMRIGHT", 0, 1)

			-- scripts
			toggleBtn.OnClick = ToggleBtn_OnClick
			textBtn.OnClick = OnClick
			textBtn.OnDoubleClick = OnDoubleClick
			frame:StackScript("OnSizeChanged", OnSizeChanged)

            return frame
        end

        local _WidgetInfo = {
            ["WidgetName"] = _WidgetName,
            ["Base"] = _Base,
            ["FuncProxy"] = _FuncProxy,
            ["Property"] = _Property,
            ["ScriptType"] = _ScriptType,
            ["New"] = _New,
        }

        TreeNode = IGAS:NewWidget(_WidgetInfo, TreeNode)
    end

    ------------------------------------------------------
	--------------------- TreeView ----------------------
	------------------------------------------------------
	local _WidgetName = "TreeView"
    local _Base = "ScrollForm"

    -- Scripts
	local function OnNodeClk(self)
		self.Parent:Fire("OnNodeSelected", self.Parent:GetValue())
	end

	-- ScriptType
	local _ScriptType = {
		------------------------------------
		--- ScriptType, Run when an treenode in the treeview is selected
		-- @name TreeView:OnNodeSelected
		-- @class function
		-- @usage function TreeView:OnNodeSelected()<br>
		--    -- do someting<br>
		-- end
		------------------------------------
        ["OnNodeSelected"] = true,
	}

	-- FuncProxy
	local _FuncProxy = {
		------------------------------------
		--- Gets a Treenode from the TreeView by value
		-- @name TreeView:GetNodeByValue
		-- @class function
		-- @param ... the value list of the TreeNode
		-- @return the treenode with that value list
		-- @usage TreeView:GetNodeByValue("Main", "Second")
		------------------------------------

        -- GetNodeByValue
		["GetNodeByValue"] = function(self, ...)
			local value, e

			if select('#', ...) == 0 then
	            return nil
	        end

			value = ""

	        for i = 1, select('#', ...) do
	            e = select(i, ...)
	            if e and type(e) == "string" then
                    if value == "" then
                        value = e
                    else
	                    value = value.."\001"..e
                    end
	            end
	        end

			if value ~= "" and self.Container.__Nodes[value] then
				return self.Container.__Nodes[value]
			end
		end,

		------------------------------------
		--- Removes a Treenode from the TreeView by value
		-- @name TreeView:RemoveNodeByValue
		-- @class function
		-- @param ... the value list of the TreeNode
		-- @usage TreeView:RemoveNodeByValue("Main", "Second")
		------------------------------------

        -- RemoveNodeByValue
		["RemoveNodeByValue"] = function(self, ...)
			local value, e

			if select('#', ...) == 0 then
	            return nil
	        end

			value = ""
	        for i = 1, select('#', ...) do
	            e = select(i, ...)
	            if e and type(e) == "string" then
	                if value == "" then
                        value = e
                    else
	                    value = value.."\001"..e
                    end
	            end
	        end

			if value ~= "" and self.Container.__Nodes[value] then
				self.Container.__Nodes[value]:Dispose()
			end
		end,

		------------------------------------
		--- Selects a Treenode from the TreeView by value
		-- @name TreeView:SelectNodeByValue
		-- @class function
		-- @param ... the value list of the TreeNode
		-- @usage TreeView:SelectNodeByValue("Main", "Second")
		------------------------------------

        -- SelectNodeByValue
		["SelectNodeByValue"] = function(self, ...)
			local value, e
			local container = self.Container

			if select('#', ...) == 0 then
	            return nil
	        end

			value = ""
			-- Toggle
	        for i = 1, select('#', ...) do
	            e = select(i, ...)
	            if e and type(e) == "string" then
	                if value == "" then
                        value = e
                    else
	                    value = value.."\001"..e
                    end

					if value ~= "" and container.__Nodes[value] then
						if container.__Nodes[value]:GetChild("TreeNode1") then
							container.__Nodes[value].ToggleState = true
						end
					end
	            end
	        end

			-- Select
			if value ~= "" and container.__Nodes[value] then
				container.__Nodes[value]:Select()
			end
		end,

		------------------------------------
		--- Gets the selected TreeNode of the TreeView
		-- @name TreeView:GetSelectNode
		-- @class function
		-- @return the selected TreeView
		-- @usage TreeView:GetSelectNode()
		------------------------------------

        -- GetSelectNode
        ["GetSelectNode"] = function(self)
            if self.Container.__SelectedItem then
				return self.Container.__SelectedItem
			end
        end,

		------------------------------------
		--- Gets the selected TreeNode's value
		-- @name TreeView:GetValue
		-- @class function
		-- @return the selected TreeView's value list
		-- @usage TreeView:GetValue()
		------------------------------------

        -- GetValue
		["GetValue"] = function(self)
			if self.Container.__SelectedItem then
				return self.Container.__SelectedItem:GetValue()
			end
		end,

		------------------------------------
		--- Adds child-nodes by the given information
		-- @name TreeView:AddNode
		-- @class function
		-- @param node a table contains the node's information
		-- @return the child-node
		-- @usage TreeView:AddNode{["Text"] = "Show Text", ["Value"] = "Main",}
		------------------------------------

        -- AddNode
        ["AddNode"] = function(self, node)
            local container = self.Container
            local value

			if node and type(node) == "table" and
				(node.Text and type(node.Text) == "string" and node.Text ~= "") and
				(node.Value and type(node.Value) == "string" and node.Value ~= "") then

				-- Value Check
				value = ""..node.Value
                if container.__Nodes[value] then
                    error("This node["..value.."] is exist", 2)
                end

				local frame = CreateFrame(TreeNode, nil, container)
				frame:SetNode(node)
				return frame
			end
        end,

		------------------------------------
		--- Gets a child-node by index
		-- @name TreeView:GetNode
		-- @class function
		-- @param index the index of a child-node
		-- @return the child-node with that index
		-- @usage TreeView:GetNode(1)
		------------------------------------

        -- GetNode
        ["GetNode"] = function(self, index)
			return (index and type(index) == "number" and self.Container:GetChild("TreeNode"..index)) or nil
        end,

		------------------------------------
		--- Removes a child-node by index
		-- @name TreeView:RemoveNode
		-- @class function
		-- @param index the index of a child-node
		-- @usage TreeView:RemoveNode(1)
		------------------------------------

        -- RemoveNode
		["RemoveNode"] = function(self, index)
            if index and type(index) == "number" and self.Container:GetChild("TreeNode"..index) then
				self.Container:GetChild("TreeNode"..index):Dispose()
            end
		end,

		------------------------------------
		--- ReBudild the treeView with a data table
		-- @name TreeView:SetTree
		-- @class function
		-- @param tree a table contains a data table
		-- @usage TreeView:SetTree{<br>
		--	{<br>
		--		["Value"] = "A",<br>
		--		["Text"] = "Alpha",<br>
        --        ["Menu"] = DropDownList1,<br>
		--	},<br>
		--	{<br>
		--		["Value"] = "B",<br>
		--		["Text"] = "Bravo",<br>
        --        ["Menu"] = DropDownList2,<br>
		--		["Childs"] = {<br>
		--			{<br>
		--				["Value"] = "C",<br>
		--				["Text"] = "Charlie",<br>
		--			},<br>
		--			{<br>
		--				["Value"] = "D",<br>
		--				["Text"] = "Delta",<br>
		--				["Childs"] = {<br>
		--					{<br>
		--						["Value"] = "E",<br>
		--						["Text"] = "Echo",<br>
        --                        ["Menu"] = DropDownList3,<br>
		--					}<br>
		--				}<br>
		--			}<br>
		--		}<br>
		--	},<br>
		--	{<br>
		--		["Value"] = "F",<br>
		--		["Text"] = "Foxtrot"<br>
		--	},<br>
		--}<br>
		------------------------------------

        -- SetTree
		["SetTree"] = function(self, tree)
			local container = self.Container
			local index = 0
			local idx, cht
			local frame

            if not tree then
                if container.__NodeNum and container.__NodeNum > index then
                    for i = container.__NodeNum, index+1, -1 do
                        container:GetChild("TreeNode"..i):Dispose()
                    end
                end
                container.__SelectedItem = nil
                return
            end

			wipe(container.__Nodes)

            idx, cht = next(tree)
			while(idx) do
				if cht and type(cht) == "table" and
					(cht.Text and type(cht.Text) == "string" and cht.Text ~= "") and
					(cht.Value and type(cht.Value) == "string" and cht.Value ~= "") then

					index = index + 1

					if container:GetChild("TreeNode"..index) then
						container:GetChild("TreeNode"..index):SetNode(cht)
					else
						node = CreateFrame(TreeNode, nil, container)
						node:SetNode(cht)
					end
				end
				idx, cht = next(tree, idx)
			end
			if container.__NodeNum and container.__NodeNum > index then
				for i = container.__NodeNum, index+1, -1 do
					container:GetChild("TreeNode"..i):Dispose()
				end
			end
		end,

        -- Dispose, release resource
        ["Dispose"] = function(self)
            self:SetTree(nil)

            -- Call super's dispose
			self:CallMethod(_Base, "Dispose")
        end,
    }

	-- Property
	local _Property = {
        ["ChildNodeNum"] = {
			["Get"] = function(self)
				return self.Container.__NodeNum or 0
			end,

			["Type"] = "number",
		},
		-- TogglePosition
		["TogglePosition"] = {
			["Set"] = function(self, position)
				if position == self.Container.__TogglePosition then
					return
				end				
				
				self.Container.__TogglePosition = position
				
				for i = 1, self.ChildNodeNum do
					self.Container:GetChild("TreeNode"..i).TogglePosition = position
				end
			end,
			
			["Get"] = function(self)
				return self.Container.__TogglePosition or TEMPLATE_LEFT
			end,
			
			["Type"] = _ToggleType,
		},
	}

    -- New
    local _New = function(parent)
		local frame = CreateFrame("ScrollForm",nil,parent)
        frame.ValueStep = 10
		local container = frame.Container

		container.Height = 30
		container.__Nodes = {}
		container.__TogglePosition = TEMPLATE_LEFT

		container.OnNodeClk = OnNodeClk

        return frame
    end

    local _WidgetInfo = {
        ["WidgetName"] = _WidgetName,
        ["Base"] = _Base,
        ["FuncProxy"] = _FuncProxy,
        ["Property"] = _Property,
        ["ScriptType"] = _ScriptType,
        ["New"] = _New,
		-- Special Item
		["TreeNode"] = TreeNode,
    }

    IGAS:NewWidget(_WidgetInfo, true)
end