﻿-- Author      : Kurapica
-- Create Date : 8/03/2008 17:14

---------------------------------------------------------------------------------------------------------------------------------------
--- MultiLineTextBox is using to contains multi-text
-- <br><br>inherit <a href="..\Base\MultiLineTextBox.html">MultiLineTextBox</a> For all methods, properties and scriptTypes
-- @name MultiLineTextBox
-- @class table
-- @field NumericOnly true if the edit box only accepts numeric input
-- @field Password true if the text entered in the edit box is masked
-- @field AutoFocus true if the edit box automatically acquires keyboard input focus
-- @field HistoryLines the maximum number of history lines stored by the edit box
-- @field Focused true if the edit box is currently focused
-- @field AltArrowKeyMode true if the arrow keys are ignored by the edit box unless the Alt key is held
-- @field BlinkSpeed the rate at which the text insertion blinks when the edit box is focused
-- @field CursorPosition the current cursor position inside edit box
-- @field MaxBytes the maximum number of bytes of text allowed in the edit box, default is 0(Infinite)
-- @field MaxLetters the maximum number of text characters allowed in the edit box
-- @field Number the contents of the edit box as a number
-- @field Text the edit box's text contents
-- @field TextInsets the insets from the edit box's edges which determine its interactive text area
-- @field Editable true if the edit box is editable
---------------------------------------------------------------------------------------------------------------------------------------
do
	-- Check Version
	local version = 3

	if not IGAS:NewAddon("IGAS.GUI.MultiLineTextBox", version) then
		return
	end

    local CreateFrame = IGAS.GUI.CreateFrame

	local _WidgetName = "MultiLineTextBox"
    local _Base = "ScrollForm"

    -- Scripts
    local function OnMouseUp(self)
        self.__Text:SetFocus()
    end

    local function OnCursorChanged(self, x, y, w, h)
		if y and h then
			y = -y

			if self.__Container.Value > y then
				self.__Container.Value = y
			elseif self.__Container.Value + self.__Container.Height < y + h then
				self.__Container.Value = y + h - self.__Container.Height
			end
		end

		self.__Container:Fire("OnCursorChanged", x, y, w, h)
    end

    local function OnEscapePressed(self, ...)
        self:ClearFocus()
		self.__Container:Fire("OnEscapePressed", ...)
    end

    local function OnTextChanged(self, ...)
		self.__Container:Fire("OnTextChanged", ...)
	end

	local function OnSizeChanged(self)
		self.__Container:FixHeight()
	end

    local function OnEditFocusGained(self, ...)
		self.__Container:Fire("OnEditFocusGained", ...)
	end

    local function OnEditFocusLost(self, ...)
		self.__Container:Fire("OnEditFocusLost", ...)
	end

    local function OnEnterPressed(self, ...)
		self:Insert("\n")	-- Added for 3.3
		self.__Container:Fire("OnEnterPressed", ...)
	end

    local function OnInputLanguageChanged(self, ...)
		self.__Container:Fire("OnInputLanguageChanged", ...)
	end

    local function OnSpacePressed(self, ...)
		self.__Container:Fire("OnSpacePressed", ...)
	end

    local function OnTabPressed(self, ...)
		self:Insert("    ")
		self.__Container:Fire("OnTabPressed", ...)
	end

    local function OnTextSet(self, ...)
		self.__Container:Fire("OnTextSet", ...)
	end

	local function OnChar(self, ...)
		self.__Container:Fire("OnChar", ...)
	end

	local function OnCharComposition(self, ...)
		self.__Container:Fire("OnCharComposition", ...)
	end

	-- ScriptType
	local _ScriptType = {
		------------------------------------
		--- ScriptType, Run when the edit box's text is changed
		-- @name MultiLineTextBox:OnTextChanged
		-- @class function
		-- @param isUserInput
		-- @usage function MultiLineTextBox:OnTextChanged(isUserInput)<br>
		--    -- do someting<br>
		-- end
		------------------------------------
        ["OnTextChanged"] = true,

		------------------------------------
		--- ScriptType, Run when the position of the text insertion cursor in the edit box changes
		-- @name MultiLineTextBox:OnCursorChanged
		-- @class function
		-- @param x Horizontal position of the cursor relative to the top left corner of the edit box (in pixels)
		-- @param y Vertical position of the cursor relative to the top left corner of the edit box (in pixels)
		-- @param width Width of the cursor graphic (in pixels)
		-- @param height Height of the cursor graphic (in pixels); matches the height of a line of text in the edit box
		-- @usage function MultiLineTextBox:OnCursorChanged(x, y, width, height)<br>
		--    -- do someting<br>
		-- end
		------------------------------------
		["OnCursorChanged"] = true,

		------------------------------------
		--- ScriptType, Run when the edit box becomes focused for keyboard input
		-- @name MultiLineTextBox:OnEditFocusGained
		-- @class function
		-- @usage function MultiLineTextBox:OnEditFocusGained()<br>
		--    -- do someting<br>
		-- end
		------------------------------------
		["OnEditFocusGained"] = true,

		------------------------------------
		--- ScriptType, Run when the edit box loses keyboard input focus
		-- @name MultiLineTextBox:OnEditFocusLost
		-- @class function
		-- @usage function MultiLineTextBox:OnEditFocusLost()<br>
		--    -- do someting<br>
		-- end
		------------------------------------
		["OnEditFocusLost"] = true,

		------------------------------------
		--- ScriptType, Run when the Enter (or Return) key is pressed while the edit box has keyboard focus
		-- @name MultiLineTextBox:OnEnterPressed
		-- @class function
		-- @usage function MultiLineTextBox:OnEnterPressed()<br>
		--    -- do someting<br>
		-- end
		------------------------------------
		["OnEnterPressed"] = true,

		------------------------------------
		--- ScriptType, Run when the Escape key is pressed while the edit box has keyboard focus
		-- @name MultiLineTextBox:OnEscapePressed
		-- @class function
		-- @usage function MultiLineTextBox:OnEscapePressed()<br>
		--    -- do someting<br>
		-- end
		------------------------------------
		["OnEscapePressed"] = true,

		------------------------------------
		--- ScriptType, Run when the edit box's language input mode changes
		-- @name MultiLineTextBox:OnInputLanguageChanged
		-- @class function
		-- @param language Name of the new input language
		-- @usage function MultiLineTextBox:OnInputLanguageChanged("language")<br>
		--    -- do someting<br>
		-- end
		------------------------------------
		["OnInputLanguageChanged"] = true,

		------------------------------------
		--- ScriptType, Run when the space bar is pressed while the edit box has keyboard focus
		-- @name MultiLineTextBox:OnSpacePressed
		-- @class function
		-- @usage function MultiLineTextBox:OnSpacePressed()<br>
		--    -- do someting<br>
		-- end
		------------------------------------
		["OnSpacePressed"] = true,

		------------------------------------
		--- ScriptType, Run when the Tab key is pressed while the edit box has keyboard focus
		-- @name MultiLineTextBox:OnTabPressed
		-- @class function
		-- @usage function MultiLineTextBox:OnTabPressed()<br>
		--    -- do someting<br>
		-- end
		------------------------------------
		["OnTabPressed"] = true,

		------------------------------------
		--- ScriptType, Run when the edit box's text is set programmatically
		-- @name MultiLineTextBox:OnTextSet
		-- @class function
		-- @usage function MultiLineTextBox:OnTextSet()<br>
		--    -- do someting<br>
		-- end
		------------------------------------
		["OnTextSet"] = true,

		------------------------------------
		--- ScriptType, Run for each text character typed in the MultiLineTextBox
		-- @name MultiLineTextBox:OnChar
		-- @class function
		-- @param text The text entered
		-- @usage function MultiLineTextBox:OnChar(text)<br>
		--    -- do someting<br>
		-- end
		------------------------------------
		["OnChar"] = true,

		------------------------------------
		--- ScriptType, Run when the edit box's input composition mode changes
		-- @name MultiLineTextBox:OnCharComposition
		-- @class function
		-- @param text The text entered
		-- @usage function MultiLineTextBox:OnCharComposition(text)<br>
		--    -- do someting<br>
		-- end
		------------------------------------
		["OnCharComposition"] = true,
    }

	--	FuncProxy
	local _FuncProxy = {
		------------------------------------
		--- Add text to the edit history.
		-- @name  MultiLineTextBox:AddHistoryLine
		-- @class function
		-- @param text Text to be added to the edit box's list of history lines
		-- @usage  MultiLineTextBox:AddHistoryLine("This is a line")
		------------------------------------

		-- AddHistoryLine
		["AddHistoryLine"] = function(self, ...)
			return self.__Text:AddHistoryLine(...)
		end,

		------------------------------------
		--- Releases keyboard input focus from the edit box
		-- @name  MultiLineTextBox:ClearFocus
		-- @class function
		-- @usage  MultiLineTextBox:ClearFocus()
		------------------------------------

		-- ClearFocus
		["ClearFocus"] = function(self, ...)
			return self.__Text:ClearFocus(...)
		end,

		------------------------------------
		--- Return whether only alt+arrow keys work for navigating the edit box, not arrow keys alone.
		-- @name  MultiLineTextBox:GetAltArrowKeyMode
		-- @class function
		-- @return true if only alt+arrow keys work for navigating the edit box, not arrow keys alone
		-- @usage  MultiLineTextBox:GetAltArrowKeyMode()
		------------------------------------

		-- GetAltArrowKeyMode
		["GetAltArrowKeyMode"] = function(self, ...)
			return self.__Text:GetAltArrowKeyMode(...)
		end,

		------------------------------------
		--- Gets the blink speed of the EditBox in seconds
		-- @name  MultiLineTextBox:GetBlinkSpeed
		-- @class function
		-- @return the blink speed of the EditBox in seconds
		-- @usage  MultiLineTextBox:GetBlinkSpeed()
		------------------------------------

		-- GetBlinkSpeed
		["GetBlinkSpeed"] = function(self, ...)
			return self.__Text:GetBlinkSpeed(...)
		end,

		------------------------------------
		--- Gets the position of the cursor inside the EditBox
		-- @name  MultiLineTextBox:GetCursorPosition
		-- @class function
		-- @return the position of the cursor inside the EditBox
		-- @usage  MultiLineTextBox:GetCursorPosition()
		------------------------------------

		-- GetCursorPosition
		["GetCursorPosition"] = function(self, ...)
			return self.__Text:GetCursorPosition(...)
		end,

		------------------------------------
		--- Get the number of history lines for this edit box
		-- @name  MultiLineTextBox:GetHistoryLines
		-- @class function
		-- @return the number of history lines for this edit box
		-- @usage  MultiLineTextBox:GetHistoryLines()
		------------------------------------

		-- GetHistoryLines
		["GetHistoryLines"] = function(self, ...)
			return self.__Text:GetHistoryLines(...)
		end,

		------------------------------------
		--- Get the input language (locale based not in-game)
		-- @name  MultiLineTextBox:GetInputLanguage
		-- @class function
		-- @return the input language (locale based not in-game)
		-- @usage  MultiLineTextBox:GetInputLanguage()
		------------------------------------

		-- GetInputLanguage
		["GetInputLanguage"] = function(self, ...)
			return self.__Text:GetInputLanguage(...)
		end,

		------------------------------------
		--- Gets the maximum number bytes allowed in the EditBox
		-- @name  MultiLineTextBox:GetMaxBytes
		-- @class function
		-- @return the maximum number bytes allowed in the EditBox
		-- @usage  MultiLineTextBox:GetMaxBytes()
		------------------------------------

		-- GetMaxBytes
		["GetMaxBytes"] = function(self, ...)
			return self.__Text:GetMaxBytes(...)
		end,

		------------------------------------
		--- Gets the maximum number of letters allowed in the EditBox
		-- @name  MultiLineTextBox:GetMaxLetters
		-- @class function
		-- @return the maximum number of letters allowed in the EditBox
		-- @usage  MultiLineTextBox:GetMaxLetters()
		------------------------------------

		-- GetMaxLetters
		["GetMaxLetters"] = function(self, ...)
			return self.__Text:GetMaxLetters(...)
		end,

		------------------------------------
		--- Gets the number of letters in the box.
		-- @name  MultiLineTextBox:GetNumLetters
		-- @class function
		-- @return he number of letters in the box
		-- @usage  MultiLineTextBox:GetNumLetters()
		------------------------------------

		-- GetNumLetters
		["GetNumLetters"] = function(self, ...)
			return self.__Text:GetNumLetters(...)
		end,

		------------------------------------
		--- Get the contents of the edit box as a number
		-- @name  MultiLineTextBox:GetNumber
		-- @class function
		-- @return the contents of the edit box as a number
		-- @usage  MultiLineTextBox:GetNumber()
		------------------------------------

		-- GetNumber
		["GetNumber"] = function(self, ...)
			return self.__Text:GetNumber(...)
		end,

		------------------------------------
		--- Get the current text contained in the edit box.
		-- @name  MultiLineTextBox:GetText
		-- @class function
		-- @return the current text contained in the edit box
		-- @usage  MultiLineTextBox:GetText()
		------------------------------------

		-- GetText
		["GetText"] = function(self, ...)
			return self.__Text:GetText(...)
		end,

		------------------------------------
		--- Gets the insets from the edit box's edges which determine its interactive text area
		-- @name  MultiLineTextBox:GetTextInsets
		-- @class function
		-- @return left - Distance from the left edge of the edit box to the left edge of its interactive text area (in pixels)
		-- @return right - Distance from the right edge of the edit box to the right edge of its interactive text area (in pixels)
		-- @return top - Distance from the top edge of the edit box to the top edge of its interactive text area (in pixels)
		-- @return bottom - Distance from the bottom edge of the edit box to the bottom edge of its interactive text area (in pixels)
		-- @usage  MultiLineTextBox:GetTextInsets()
		------------------------------------

		-- GetTextInsets
		["GetTextInsets"] = function(self, ...)
			return self.__Text:GetTextInsets(...)
		end,

		------------------------------------
		--- Selects all or a portion of the text in the edit box
		-- @name  MultiLineTextBox:HighlightText
		-- @class function
		-- @param start Optional,Character position at which to begin the selection (between 0, for the position before the first character, and  MultiLineTextBox:GetNumLetters(), for the position after the last character); defaults to 0 if not specified
		-- @param end Optional,Character position at which to end the selection; if not specified or if less than start, selects all characters after the start  position; if equal to start, selects nothing and positions the cursor at the start position
		-- @usage  MultiLineTextBox:HighlightText(1, 10)
		------------------------------------

		-- HighlightText
		["HighlightText"] = function(self, ...)
			return self.__Text:HighlightText(...)
		end,

		------------------------------------
		--- Insert text into the edit box.
		-- @name  MultiLineTextBox:Insert
		-- @class function
		-- @param text Text to be inserted
		-- @usage  MultiLineTextBox:Insert("    ")
		------------------------------------

		-- Insert
		["Insert"] = function(self, ...)
			return self.__Text:Insert(...)
		end,

		------------------------------------
		--- Determine if the EditBox has autofocus enabled
		-- @name  MultiLineTextBox:IsAutoFocus
		-- @class function
		-- @return true if the EditBox has autofocus enabled
		-- @usage  MultiLineTextBox:IsAutoFocus()
		------------------------------------

		-- IsAutoFocus
		["IsAutoFocus"] = function(self, ...)
			return self.__Text:IsAutoFocus(...)
		end,

		------------------------------------
		--- Determine if the EditBox accepts multiple lines
		-- @name  MultiLineTextBox:IsMultiLine
		-- @class function
		-- @return true if the EditBox accepts multiple lines
		-- @usage  MultiLineTextBox:IsMultiLine()
		------------------------------------

		-- IsMultiLine
		["IsMultiLine"] = function(self, ...)
			return self.__Text:IsMultiLine(...)
		end,

		------------------------------------
		--- Determine if the EditBox only accepts numeric input
		-- @name  MultiLineTextBox:IsNumeric
		-- @class function
		-- @return true if the EditBox only accepts numeric input
		-- @usage  MultiLineTextBox:IsNumeric()
		------------------------------------

		-- IsNumeric
		["IsNumeric"] = function(self, ...)
			return self.__Text:IsNumeric(...)
		end,

		------------------------------------
		--- Determine if the EditBox performs password masking
		-- @name  MultiLineTextBox:IsPassword
		-- @class function
		-- @return true if the EditBox performs password masking
		-- @usage  MultiLineTextBox:IsPassword()
		------------------------------------

		-- IsPassword
		["IsPassword"] = function(self, ...)
			return self.__Text:IsPassword(...)
		end,

		------------------------------------
		--- Make only alt+arrow keys work for navigating the edit box, not arrow keys alone.
		-- @name  MultiLineTextBox:SetAltArrowKeyMode
		-- @class function
		-- @param enable True to cause the edit box to ignore arrow key presses unless the Alt key is held; false to allow unmodified arrow key presses for cursor movement
		-- @usage  MultiLineTextBox:SetAltArrowKeyMode(true)
		------------------------------------

		-- SetAltArrowKeyMode
		["SetAltArrowKeyMode"] = function(self, ...)
			return self.__Text:SetAltArrowKeyMode(...)
		end,

		------------------------------------
		--- Set whether or not the editbox will attempt to get input focus when it gets shown (default: true)
		-- @name  MultiLineTextBox:SetAutoFocus
		-- @class function
		-- @param enable True to enable the edit box to automatically acquire keyboard input focus; false to disable
		-- @usage  MultiLineTextBox:SetAutoFocus()
		------------------------------------

		-- SetAutoFocus
		["SetAutoFocus"] = function(self, ...)
			return self.__Text:SetAutoFocus(...)
		end,

		------------------------------------
		--- Sets the rate at which the text insertion blinks when the edit box is focused
		-- @name  MultiLineTextBox:SetBlinkSpeed
		-- @class function
		-- @param duration Amount of time for which the cursor is visible during each "blink" (in seconds)
		-- @usage  MultiLineTextBox:SetBlinkSpeed(2)
		------------------------------------

		-- SetBlinkSpeed
		["SetBlinkSpeed"] = function(self, ...)
			return self.__Text:SetBlinkSpeed(...)
		end,

		------------------------------------
		--- Set the position of the cursor within the EditBox
		-- @name  MultiLineTextBox:SetCursorPosition
		-- @class function
		-- @param position New position for the keyboard input cursor (between 0, for the position before the first character, and  MultiLineTextBox:GetNumLetters(), for the position after the last character)
		-- @usage  MultiLineTextBox:SetCursorPosition(123)
		------------------------------------

		-- SetCursorPosition
		["SetCursorPosition"] = function(self, ...)
			return self.__Text:SetCursorPosition(...)
		end,

		------------------------------------
		--- Move input focus (the cursor) to this editbox
		-- @name  MultiLineTextBox:SetFocus
		-- @class function
		-- @usage  MultiLineTextBox:SetFocus()
		------------------------------------

		-- SetFocus
		["SetFocus"] = function(self, ...)
			return self.__Text:SetFocus(...)
		end,

		------------------------------------
		--- Sets the font to use for text, returns 1 if the path was valid, nil otherwise(no change occurs).
		-- @name  MultiLineTextBox:SetFont
		-- @class function
		-- @param filename Path to a font file
		-- @param fontHeight Height (point size) of the font to be displayed (in pixels)
		-- @param flags Additional properties for the font specified by one or more (separated by commas) of the following tokens:MONOCHROME,OUTLINE,THICKOUTLINE
		-- @return 1 if filename refers to a valid font file; otherwise nil
		-- @usage  MultiLineTextBox:SetFont("Fonts\\ARIALN.TTF", 10)
		------------------------------------

		-- SetFont
		["SetFont"] = function(self, ...)
			return self.__Text:SetFont(...)
		end,

		------------------------------------
		--- Sets the maximum number of history lines stored by the edit box
		-- @name  MultiLineTextBox:SetHistoryLines
		-- @class function
		-- @param count Maximum number of history lines to be stored by the edit box
		-- @usage  MultiLineTextBox:SetHistoryLines(12)
		------------------------------------

		-- SetHistoryLines
		["SetHistoryLines"] = function(self, ...)
			return self.__Text:SetHistoryLines(...)
		end,

		------------------------------------
		--- Sets the maximum number of bytes of text allowed in the edit box
		-- @name  MultiLineTextBox:SetMaxBytes
		-- @class function
		-- @param maxBytes Maximum number of text bytes allowed in the edit box, or 0 for no limit
		-- @usage  MultiLineTextBox:SetMaxBytes(1024)
		------------------------------------

		-- SetMaxBytes
		["SetMaxBytes"] = function(self, ...)
			return self.__Text:SetMaxBytes(...)
		end,

		------------------------------------
		--- Sets the maximum number of text characters allowed in the edit box
		-- @name  MultiLineTextBox:SetMaxLetters
		-- @class function
		-- @param maxLetters Maximum number of text characters allowed in the edit box, or 0 for no limit
		-- @usage  MultiLineTextBox:SetMaxLetters(1024)
		------------------------------------

		-- SetMaxLetters
		["SetMaxLetters"] = function(self, ...)
			return self.__Text:SetMaxLetters(...)
		end,

		------------------------------------
		--- Sets whether the edit box shows more than one line of text, no use
		-- @name  MultiLineTextBox:SetMultiLine
		-- @class function
		-- @param multiLine True to allow the edit box to display more than one line of text; false for single-line display
		-- @usage  MultiLineTextBox:SetMultiLine(true)
		------------------------------------

		-- SetMultiLine
		["SetMultiLine"] = function(self, ...)
			error("This editbox must be multi-line", 2)
		end,

		------------------------------------
		--- Sets the contents of the edit box to a number
		-- @name  MultiLineTextBox:SetNumber
		-- @class function
		-- @param num New numeric content for the edit box
		-- @usage  MultiLineTextBox:SetNumber(1234)
		------------------------------------

		-- SetNumber
		["SetNumber"] = function(self, ...)
			return self.__Text:SetNumber(...)
		end,

		------------------------------------
		--- Sets whether the edit box only accepts numeric input
		-- @name  MultiLineTextBox:SetNumeric
		-- @class function
		-- @param enable True to allow only numeric input; false to allow any text
		-- @usage  MultiLineTextBox:SetNumeric(false)
		------------------------------------

		-- SetNumeric
		["SetNumeric"] = function(self, ...)
			return self.__Text:SetNumeric(...)
		end,

		------------------------------------
		--- Sets whether the text entered in the edit box is masked
		-- @name  MultiLineTextBox:SetPassword
		-- @class function
		-- @param enable True to mask text entered in the edit box with asterisk characters (*); false to show the actual text entered
		-- @usage  MultiLineTextBox:SetPassword(false)
		------------------------------------

		-- SetPassword
		["SetPassword"] = function(self, ...)
			return self.__Text:SetPassword(...)
		end,

		------------------------------------
		--- Sets the edit box's text contents
		-- @name  MultiLineTextBox:SetText
		-- @class function
		-- @param text Text to be placed in the edit box
		-- @usage  MultiLineTextBox:SetText("Hello World")
		------------------------------------

		-- SetText
		["SetText"] = function(self, text)
			self.__Text.Text = text
		end,

		------------------------------------
		--- Sets the insets from the edit box's edges which determine its interactive text area
		-- @name  MultiLineTextBox:SetTextInsets
		-- @class function
		-- @param left Distance from the left edge of the edit box to the left edge of its interactive text area (in pixels)
		-- @param right Distance from the right edge of the edit box to the right edge of its interactive text area (in pixels)
		-- @param top Distance from the top edge of the edit box to the top edge of its interactive text area (in pixels)
		-- @param bottom Distance from the bottom edge of the edit box to the bottom edge of its interactive text area (in pixels)
		-- @usage  MultiLineTextBox:SetTextInsets(4, 4, 6, 6)
		------------------------------------

		-- SetTextInsets
		["SetTextInsets"] = function(self, ...)
			return self.__Text:SetTextInsets(...)
		end,

		------------------------------------
		--- Switches the edit box's language input mode
		-- @name  MultiLineTextBox:ToggleInputLanguage
		-- @class function
		-- @usage  MultiLineTextBox:ToggleInputLanguage()
		------------------------------------

		-- ToggleInputLanguage
		["ToggleInputLanguage"] = function(self, ...)
			return self.__Text:ToggleInputLanguage(...)
		end,

		------------------------------------
		--- Returns whether the edit box is currently focused for keyboard input
		-- @name  MultiLineTextBox:HasFocus
		-- @class function
		-- @return True if the edit box is currently focused for keyboard input
		-- @usage  MultiLineTextBox:HasFocus()
		------------------------------------

		-- HasFocus
		["HasFocus"] = function(self, ...)
			return self.__Text:HasFocus(...)
		end,
    }

	--	Property
	local _Property = {
        -- NumericOnly
        ["NumericOnly"] = {
			["Set"] = function(self, state)
				self:SetNumeric(state)
			end,

			["Get"] = function(self)
				return (self:IsNumeric() and true) or false
			end,

			["Type"] = "boolean",
		},

        -- Password
        ["Password"] = {
			["Set"] = function(self, state)
				self:SetPassword(state)
			end,

			["Get"] = function(self)
				return (self:IsPassword() and true) or false
			end,

			["Type"] = "boolean",
		},

        -- AutoFocus
        ["AutoFocus"] = {
			["Set"] = function(self, state)
				self:SetAutoFocus(state)
			end,

			["Get"] = function(self)
				return (self:IsAutoFocus() and true) or false
			end,

			["Type"] = "boolean",
		},

        -- HistoryLines
        ["HistoryLines"] = {
			["Set"] = function(self, num)
				self:SetHistoryLines(num)
			end,

			["Get"] = function(self)
				return self:GetHistoryLines()
			end,

			["Type"] = "number",
		},

        -- Focus
        ["Focused"] = {
			["Set"] = function(self, focus)
				if focus then
					self:SetFocus()
				else
					self:ClearFocus()
				end
			end,

			["Get"] = function(self)
				return (self:HasFocus() and true) or false
			end,

			["Type"] = "boolean",
		},

        -- AltArrowKeyMode
        ["AltArrowKeyMode"] = {
			["Set"] = function(self, enable)
				self:SetAltArrowKeyMode(enable)
			end,

			["Get"] = function(self)
				return (self:GetAltArrowKeyMode() and true) or false
			end,

			["Type"] = "boolean",
		},

        -- BlinkSpeed
        ["BlinkSpeed"] = {
			["Set"] = function(self, speed)
				self:SetBlinkSpeed(speed)
			end,

			["Get"] = function(self)
				return self:GetBlinkSpeed()
			end,

			["Type"] = "number",
		},

        -- CursorPosition
        ["CursorPosition"] = {
			["Set"] = function(self, position)
				self:SetCursorPosition(position)
			end,

			["Get"] = function(self)
				return self:GetCursorPosition()
			end,

			["Type"] = "number",
		},

        -- MaxBytes
        ["MaxBytes"] = {
			["Set"] = function(self, maxBytes)
				self:SetMaxBytes(maxBytes)
			end,

			["Get"] = function(self)
				return self:GetMaxBytes()
			end,

			["Type"] = "number",
		},

        -- MaxLetters
        ["MaxLetters"] = {
			["Set"] = function(self, maxLetters)
				self:SetMaxLetters(maxLetters)
			end,

			["Get"] = function(self)
				return self:GetMaxLetters()
			end,

			["Type"] = "number",
		},

        -- Number
        ["Number"] = {
			["Set"] = function(self, number)
				self:SetNumber(number)
			end,

			["Get"] = function(self)
				return self:GetNumber()
			end,

			["Type"] = "number",
		},

        -- Text
        ["Text"] = {
			["Set"] = function(self, text)
				self:SetText(text)
			end,

			["Get"] = function(self)
				return self:GetText()
			end,

			["Type"] = "string",
		},

		-- TextInsets
		["TextInsets"] = {
			["Set"] = function(self, RectInset)
				self:SetTextInsets(RectInset.left, RectInset.right, RectInset.top, RectInset.bottom)
			end,

			["Get"] = function(self)
				local RectInset = {}
				RectInset.left, RectInset.right, RectInset.top, RectInset.bottom = self:GetTextInsets()
				return RectInset
			end,

			["Type"] = "RectInset",
		},

        -- Editable
        ["Editable"] = {
			["Set"] = function(self, flag)
				self.MouseEnabled = flag
				self.__Text.MouseEnabled = flag
				if not flag then
					self.__Text.AutoFocus = false
					if self.__Text:HasFocus() then
						self.__Text:ClearFocus()
					end
				end
			end,

			["Get"] = function(self)
				return self.__Text.MouseEnabled
			end,

			["Type"] = "boolean",
        },
    }

    -- Scripts
    local function _New(parent)
		local frame = CreateFrame(_Base, nil, parent)

		local container = frame.Container

		local editbox = CreateFrame("EditBox", "Text", container)
		editbox:SetPoint("TOPLEFT", container, "TOPLEFT", 0, 0)
		editbox:SetPoint("TOPRIGHT", container, "TOPRIGHT", 0, 0)
		editbox:SetMultiLine(true)
		editbox:SetTextInsets(5, 5, 3, 3)
		editbox:EnableMouse(true)
		editbox:SetAutoFocus(false)
		editbox:SetFontObject("GameFontNormal")
		editbox.Text = ""

        frame.__Text = editbox
        editbox.__Container = frame

		frame:StackScript("OnMouseUp", OnMouseUp)

		editbox:StackScript("OnEscapePressed", OnEscapePressed)
		editbox:StackScript("OnTextChanged", OnTextChanged)
	    editbox:StackScript("OnCursorChanged", OnCursorChanged)
	    editbox:StackScript("OnEditFocusGained", OnEditFocusGained)
	    editbox:StackScript("OnEditFocusLost", OnEditFocusLost)
	    editbox:StackScript("OnEnterPressed", OnEnterPressed)
		editbox:StackScript("OnSizeChanged", OnSizeChanged)
	    editbox:StackScript("OnInputLanguageChanged", OnInputLanguageChanged)
	    editbox:StackScript("OnSpacePressed", OnSpacePressed)
	    editbox:StackScript("OnTabPressed", OnTabPressed)
	    editbox:StackScript("OnTextSet", OnTextSet)
		editbox:StackScript("OnCharComposition", OnCharComposition)
		editbox:StackScript("OnChar", OnChar)

		frame:FixHeight()

		return frame
	end

	-- Register
    local _WidgetInfo = {
        ["WidgetName"] = _WidgetName,
        ["Base"] = _Base,
        ["FuncProxy"] = _FuncProxy,
        ["Property"] = _Property,
        ["ScriptType"] = _ScriptType,
        ["New"] = _New,
    }

    IGAS:NewWidget(_WidgetInfo, true)
end
