﻿-- Author      : Kurapica
-- ChangreLog  :
--				2010.01.13	Change the ItemList's Parent to WorldFrame

---------------------------------------------------------------------------------------------------------------------------------------
--- ItemButton is used to display items
-- <br><br>inherit <a href="..\Base\Button.html">Button</a> For all methods, properties and scriptTypes
-- @name ItemButton
-- @class table
-- @field ShowItemCount Whether show item count or not
-- @field BankItemIncluded Whether count an item include the bank items
-- @field ItemCount Set the item's count
-- @field ItemStock the stock of the item
-- @field ItemTexture the item's texture
-- @field ShowItemName Whether show the item's name or not
-- @field ItemName the item's name
-- @field Item the item link
-- @field ListDirection the item list's direction
-- @field ListUnit the item list's unit
-- @field ListShowNilButton Whether show the nil button in item list
---------------------------------------------------------------------------------------------------------------------------------------
do
	-- Check Version
	local version = 5

	if not IGAS:NewAddon("IGAS.GUI.ItemButton", version) then
		return
	end

	local CreateFrame = IGAS.GUI.CreateFrame

	local GameTooltip = IGAS:GetWrapper(_G.GameTooltip)
	local WorldFrame = IGAS:GetWrapper(_G.WorldFrame)

	-- ItemListDirection
	local _EnumDir = IGAS:NewEnum("ItemListDirection", {
		"TOP",
		"BOTTOM",
		"LEFT",
		"RIGHT",
	})

	local _PropDir = IGAS:NewPropertyType("ItemListDirection", {
		["Type"] = "enum",
		["EnumType"] = _EnumDir,
	})

	---------------------------------------------------------------------------------------------------------------------------------------
	--- ItemList is used to display a list of items, can only be created by ItemButton
	-- <br><br>inherit <a href="..\Base\VirtualUIObject.html">VirtualUIObject</a> For all methods, properties and scriptTypes
	-- @name ItemList
	-- @class table
	-- @field Alpha Set or get the frame's transparency value(0-1)
	-- @field Filter a function that used by the itemlist to check the container and inventory, if true then display it<br>Sample Filter: function(self, item, bag, slot)<br>    -- self: the itemlist's parent(itembutton)<br>    -- item: the item link<br>    -- bag: if -1 means inventory, others the bagid<br>    -- slot: the slot id in inventory of bag<br>    if bag > -1 then return true end<br>end
	-- @field Direction the diplay direction, can be :TOP, BOTTOM, LEFT, RIGHT
	-- @field Unit who's inventory to be checked, if Unit is "player", then the bag will also be checked.
	-- @field ShowNilButton whether show nil button or not
	-- @field AllowSameItem whether allow same item to be displayed
	-- @field Visible the visible of the item list
	-- @field Backdrop the backdrop of the item list
	-- @field BackdropBorderColor the shading color for the frame's border graphic
	-- @field BackdropColor the shading color for the frame's background graphic
	-- @field ClampRectInsets offsets from the frame's edges used when limiting user movement or resizing of the frame
	-- @field HitRectInsets the insets from the frame's edges which determine its mouse-interactable area
	-- @field Depth the 3D depth of the frame (for stereoscopic 3D setups)
	-- @field DepthIgnored whether the frame's depth property is ignored (for stereoscopic 3D setups)
	---------------------------------------------------------------------------------------------------------------------------------------
	local ItemList = (IGAS.GUI.Widget.ItemButton and IGAS.GUI.Widget.ItemButton.ItemList) or nil
	local ItemSize = 37
	do
		local _WidgetName = "ItemList"
		local _Base = "VirtualUIObject"
		local _ItemListContainer = CreateFrame("Frame", "IGAS_GUI_ListContainer", WorldFrame)

		_ItemListContainer.__ShowList = _ItemListContainer.__ShowList or nil
		if not _ItemListContainer.__Timer then
			local timer = CreateFrame("Timer", "Timer", _ItemListContainer)
			timer.Interval = 0
			timer.OnTimer = function(self)
				self.Interval = 0
				if self.Parent.__ShowList then
					self.Parent.__ShowList.Visible = false
				end
			end
			_ItemListContainer.__Timer = timer
		end

		-- Scripts
		local function OnShow(self)
			if _ItemListContainer.__ShowList and _ItemListContainer.__ShowList ~= self then
				_ItemListContainer.__ShowList.Visible = false
			end

			_ItemListContainer.__Timer.Interval = 2

			_ItemListContainer.__ShowList = self
		end

		local function OnHide(self)
            if _ItemListContainer.__ShowList and _ItemListContainer.__ShowList == self then
				_ItemListContainer.__ShowList = nil
			end

			_ItemListContainer.__Timer.Interval = 0
		end

		local function OnEnter(self)
			_ItemListContainer.__Timer.Interval = 0
		end

		local function OnLeave(self)
			_ItemListContainer.__Timer.Interval = 2
		end

		local function Item_OnEnter(self)
			self.Parent:Fire("OnEnter")
		end

		local function Item_OnLeave(self)
			self.Parent:Fire("OnLeave")
		end

		local function Item_OnClick(self)
            self.Parent.__Mask:Fire("OnClick", self.Item)
			self.Parent.Visible = false
		end

		local function makePosition(self, start, count)
			local btn, locX, locY
			local i, maxC = start, start + (count or 0)

			btn = self:GetChild("ItemButton"..i)

			while (not count or i < maxC) and btn do
				btn = self:GetChild("ItemButton"..i)
				if btn then
					if self.__Direction == "TOP" then
						locX = 0
						locY = ItemSize * i
					elseif self.__Direction == "BOTTOM" then
						locX = 0
						locY = - ItemSize * i
					elseif self.__Direction == "LEFT" then
						locX = - ItemSize * i
						locY = 0
					elseif self.__Direction == "RIGHT" then
						locX = ItemSize * i
						locY = 0
					end
					btn:ClearAllPoints()
					btn:SetPoint(self.__Direction, self, self.__Direction, locX, locY)
				end
				i = i + 1
				btn = self:GetChild("ItemButton"..i)
			end
		end

		local function setItem(self, index, link)
			local btn
			btn = self:GetChild("ItemButton"..index)

			if not btn then
				btn = CreateFrame("ItemButton", "ItemButton"..index, self)

				btn.ShowItemCount = false
				btn.ShowItemName = false

				btn.OnEnter = Item_OnEnter
				btn.OnLeave = Item_OnLeave
				btn.OnClick = Item_OnClick

				makePosition(self, index, 1)
			end
			btn.Item = link
			btn.Visible = true
		end

		local function makeList(self)
			local index = 1
			local item

			self.__Mask:Clear()

			if self.__FilterFunc then
				-- Set List
				--- Check Inventory Item
				for slot = 0, 19 do
					item = GetInventoryItemLink(self.__Unit, slot)
					if item and (self.__AllowSameItem or (not self.__Mask:HasItem(item))) and self.__FilterFunc(self.__Mask.Parent, item, -1, slot) then
						setItem(self, index, item)
						index = index + 1
					end
				end

				-- Check Container Item
				if self.__Unit == "player" then
					for i=0,NUM_BAG_FRAMES do
						for j=1,GetContainerNumSlots(i) do
							item = GetContainerItemLink(i,j)
							if item and (self.__AllowSameItem or (not self.__Mask:HasItem(item))) and self.__FilterFunc(self.__Mask.Parent, item, -1, slot) then
								setItem(self, index, item)
								index = index + 1
							end
						end
					end
				end

				-- A nil ItemButton
				if self.__ShowNilButton then
					setItem(self, index, nil)
					index = index + 1
				end

				-- Hide no-use ItemButton
				while self:GetChild("ItemButton"..index) do
					self:GetChild("ItemButton"..index).Item = nil
					self:GetChild("ItemButton"..index).Visible = false
					index = index + 1
				end
			else
				-- Hide List
				if self.__ShowNilButton then
					setItem(self, index, nil)
					index = index + 1
				end

				-- Hide no-use ItemButton
				while self:GetChild("ItemButton"..index) do
					self:GetChild("ItemButton"..index).Item = nil
					self:GetChild("ItemButton"..index).Visible = false
					index = index + 1
				end
			end
		end

		-- ScriptType
		local _ScriptType = {
			------------------------------------
			--- ScriptType, Run when the button is clicked
			-- @name ItemList:OnClick
			-- @class function
			-- @param button Name of the mouse button responsible for the click action:Button4, Button5, LeftButton, MiddleButton, RightButton
			-- @param down True for a mouse button down action; false for button up or other actions
			-- @usage function ItemList:OnClick(button, down)<br>
			--    -- do someting<br>
			-- end
			------------------------------------
            ["OnClick"] = true,
		}

		-- FuncProxy
		local _FuncProxy = {
			------------------------------------
			--- Refresh the item list.
			-- @name ItemList:Refresh
			-- @class function
			-- @usage ItemList:Refresh()
			------------------------------------

			-- Refresh
			["Refresh"] = function(self)
				self = self.__ItemList
				makeList(self)
			end,

			------------------------------------
			--- Add an item to the list
			-- @name ItemList:Add
			-- @class function
			-- @param item the item to be added
			-- @usage ItemList:Add([XXX' blade])
			------------------------------------

			-- Add
			["Add"] = function(self, item)
				local index = 1
				local btn

				self = self.__ItemList
				btn = self:GetChild("ItemButton"..index)
				while btn and btn.Visible and btn.Item do
					index = index + 1
					btn = self:GetChild("ItemButton"..index)
				end

				setItem(self, index, item)

				-- A nil ItemButton
				if self.__ShowNilButton then
					setItem(self, index, nil)
					index = index + 1
				end
			end,

			------------------------------------
			--- Remove an item from the list
			-- @name ItemList:Remove
			-- @class function
			-- @param item the item to be removed
			-- @usage ItemList:Remove([XXX' blade])
			------------------------------------

			-- Remove
			["Remove"] = function(self, item)
				local index = 1
				local btn, nextBtn

				self = self.__ItemList
				if not item then
					return
				end

				btn = self:GetChild("ItemButton"..index)
				while btn and btn.Visible and btn.Item ~= item do
					index = index + 1
					btn = self:GetChild("ItemButton"..index)
				end

				if btn and btn.Item == item then
					index = index + 1
					nextBtn = self:GetChild("ItemButton"..index)
					while nextBtn and nextBtn.Visible do
						btn.Item = nextBtn.Item
						btn = nextBtn
						index = index + 1
						nextBtn = self:GetChild("ItemButton"..index)
					end
					btn.Item = nil
					btn.Visible = false
				end
			end,

			------------------------------------
			--- Clear the list
			-- @name ItemList:Clear
			-- @class function
			-- @usage ItemList:Clear()
			------------------------------------

			-- Clear
			["Clear"] = function(self)
				local index = 1
				local btn

				self = self.__ItemList
				btn = self:GetChild("ItemButton"..index)
				while btn and btn.Visible do
					btn.Item = nil
					btn.Visible = false

					index = index + 1
					btn = self:GetChild("ItemButton"..index)
				end

				-- A nil ItemButton
				if self.__ShowNilButton then
					setItem(self, 1, nil)
				end
			end,

			------------------------------------
			--- Check if the item list has an item
			-- @name ItemList:HasItem
			-- @class function
			-- @param item the item to be checked
			-- @usage ItemList:HasItem([XXX' blade])
			------------------------------------

			-- HasItem
			["HasItem"] = function(self, item)
				local index = 1
				local btn, nextBtn

				self = self.__ItemList
				if not item then
					return
				end

				btn = self:GetChild("ItemButton"..index)
				while btn and btn.Visible and btn.Item ~= item do
					index = index + 1
					btn = self:GetChild("ItemButton"..index)
				end

				if btn and btn.Item == item then
					return true
				else
					return false
				end
			end,

			------------------------------------
			--- Return this object's alpha (transparency) value.
			-- @name ItemList:GetAlpha
			-- @class function
			-- @return this object's alpha (transparency) value
			-- @usage ItemList:GetAlpha()
			------------------------------------

			-- GetAlpha
			["GetAlpha"] = function(self)
				return self.__ItemList:GetAlpha() or 1
			end,

			------------------------------------
			--- Set the object's alpha (transparency) value.
			-- @name ItemList:SetAlpha
			-- @class function
			-- @param alpha this object's alpha (transparency) value
			-- @usage ItemList:SetAlpha(1)
			------------------------------------

			-- SetAlpha
			["SetAlpha"] = function(self, alpha)
				self.__ItemList:SetAlpha(alpha)
			end,

			------------------------------------
			--- Clear all attachment points for this object.
			-- @name ItemList:ClearAllPoints
			-- @class function
			-- @usage ItemList:ClearAllPoints()
			------------------------------------

			-- ClearAllPoints
			["ClearAllPoints"] = function(self)
				self.__ItemList:ClearAllPoints()
			end,

			------------------------------------
			--- Get the y location of the bottom edge of this frame
			-- @name ItemList:GetBottom
			-- @class function
			-- @return the y location of the bottom edge of this frame
			-- @usage ItemList:GetBottom()
			------------------------------------

			-- GetBottom
			["GetBottom"] = function(self)
				return self.__ItemList:GetBottom()
			end,

			------------------------------------
			--- Get the coordinates of the center of this frame
			-- @name ItemList:GetCenter
			-- @class function
			-- @return x - Distance from the left edge of the screen to the center of the region (in pixels)
			-- @return y - Distance from the bottom edge of the screen to the center of the region (in pixels)
			-- @usage ItemList:GetCenter()
			------------------------------------

			-- GetCenter
			["GetCenter"] = function(self)
				return self.__ItemList:GetCenter()
			end,

			------------------------------------
			--- Get the height of this object.
			-- @name ItemList:GetHeight
			-- @class function
			-- @return the height of this object
			-- @usage ItemList:GetHeight()
			------------------------------------

			-- GetHeight
			["GetHeight"] = function(self)
				return self.__ItemList:GetHeight()
			end,

			------------------------------------
			--- Get the x location of the left edge of this frame
			-- @name ItemList:GetLeft
			-- @class function
			-- @return the x location of the left edge of this frame
			-- @usage ItemList:GetLeft()
			------------------------------------

			-- GetLeft
			["GetLeft"] = function(self)
				return self.__ItemList:GetLeft()
			end,

			------------------------------------
			--- Get the number of anchor points for this frame
			-- @name ItemList:GetNumPoints
			-- @class function
			-- @return the number of anchor points for this frame
			-- @usage ItemList:GetNumPoints()
			------------------------------------

			-- GetNumPoints
			["GetNumPoints"] = function(self)
				return self.__ItemList:GetNumPoints()
			end,

			------------------------------------
			--- Returns information about one of the region's anchor points
			-- @name ItemList:GetPoint
			-- @class function
			-- @param index Index of an anchor point defined for the region (between 1 and ItemList:GetNumPoints())
			-- @return point - Point on this region at which it is anchored to another (string, anchorPoint)
			-- @return relativeTo - Reference to the other region to which this region is anchored (region)
			-- @return relativePoint - Point on the other region to which this region is anchored (string, anchorPoint)
			-- @return xOffset - Horizontal distance between point and relativePoint (in pixels; positive values put point  to the right of relativePoint) (number)
			-- @return yOffset - Vertical distance between point and relativePoint (in pixels; positive values put point  below relativePoint) (number)
			-- @usage ItemList:GetPoint(1)
			------------------------------------

			-- GetPoint
			["GetPoint"] = function(self, pointNum)
				return self.__ItemList:GetPoint(pointNum)
			end,

			------------------------------------
			--- Get the x location of the right edge of this frame
			-- @name ItemList:GetRight
			-- @class function
			-- @return the x location of the right edge of this frame
			-- @usage ItemList:GetRight()
			------------------------------------

			-- GetRight
			["GetRight"] = function(self)
				return self.__ItemList:GetRight()
			end,

			------------------------------------
			--- Get the y location of the top edge of this frame
			-- @name ItemList:GetTop
			-- @class function
			-- @return the y location of the top edge of this frame
			-- @usage ItemList:GetTop()
			------------------------------------

			-- GetTop
			["GetTop"] = function(self)
				return self.__ItemList:GetTop()
			end,

			------------------------------------
			--- Get the width of this object
			-- @name ItemList:GetWidth
			-- @class function
			-- @return the width of this object
			-- @usage ItemList:GetWidth()
			------------------------------------

			-- GetWidth
			["GetWidth"] = function(self)
				return self.__ItemList:GetWidth()
			end,

			------------------------------------
			--- Set this object to hidden (it and all of its children will disappear).
			-- @name ItemList:Hide
			-- @class function
			-- @usage ItemList:Hide()
			------------------------------------

			-- Hide
			["Hide"] = function(self)
				return self.__ItemList:Hide()
			end,

			------------------------------------
			--- Set this object to shown (it will appear if its parent is visible).
			-- @name ItemList:Show
			-- @class function
			-- @usage ItemList:Show()
			------------------------------------

			-- Show
			["Show"] = function(self)
				self.__ItemList:Show()
			end,

			------------------------------------
			--- Determine if this object is shown (would be visible if its parent was visible).
			-- @name ItemList:IsShown
			-- @class function
			-- @return true if the object is shown
			-- @usage ItemList:IsShown()
			------------------------------------

			-- IsShown
			["IsShown"] = function(self)
				return self.__ItemList:IsShown()
			end,

			------------------------------------
			--- Get whether the object is visible on screen (logically (IsShown() and GetParent():IsVisible()));
			-- @name ItemList:IsVisible
			-- @class function
			-- @return true if the object is visible
			-- @usage ItemList:IsVisible()
			------------------------------------

			-- IsVisible
			["IsVisible"] = function(self)
				return self.__ItemList:IsVisible()
			end,

			------------------------------------
			--- Sets all anchor points of the region to match those of another region. If no region is specified, the region's anchor points are set to those of its parent.
			-- @name ItemList:SetAllPoints
			-- @class function
			-- @param region Reference to a region or a global name of a region
			-- @usage ItemList:SetAllPoints(UIParent)
			------------------------------------

			-- SetAllPoints
			["SetAllPoints"] = function(self, frame)
				if frame and type(frame) == "string" then
					frame = _G[frame]
				end

				if not frame or type(frame) ~= "table" then
					frame = self.Parent
				end

				self.__ItemList:SetAllPoints(frame)
			end,

			------------------------------------
			--- Sets an anchor point for the region
			-- @name ItemList:SetPoint
			-- @class function
			-- @param point Point on this region at which it is to be anchored to another (string, anchorPoint)
			-- @param relativeTo Reference to the other region to which this region is to be anchored; if nil or omitted, anchors the region relative to its parent (or to the screen dimensions if the region has no parent) (region)
			-- @param relativePoint Point on the other region to which this region is to be anchored; if nil or omitted, defaults to the same value as point (string, anchorPoint)
			-- @param xOffset Horizontal distance between point and relativePoint (in pixels; positive values put point to the right of relativePoint); if nil or omitted, defaults to 0 (number)
			-- @param yOffset Vertical distance between point and relativePoint (in pixels; positive values put point below relativePoint); if nil or omitted, defaults to 0 (number)
			-- @usage ItemList:SetPoint("CENTER", UIParent, "CENTER", 50, 40)
			------------------------------------

			-- SetPoint
			["SetPoint"] = function(self, point, relativeObject, relativePoint, xOfs, yOfs)
				self.__ItemList:SetPoint(point, relativeObject, relativePoint, xOfs, yOfs)
			end,

			-- Dispose, release resource
			["Dispose"] = function(self)
				self.__ItemList:Dispose()

				-- Call super's dispose
				self:CallMethod(_Base, "Dispose")
			end,

			------------------------------------
			--- Returns the position and dimensions of the region
			-- @name ItemList:GetRect
			-- @class function
			-- @return left - Distance from the left edge of the screen to the left edge of the region (in pixels) (number)
			-- @return bottom - Distance from the bottom edge of the screen to the bottom of the region (in pixels) (number)
			-- @return width - Width of the region (in pixels) (number)
			-- @return height - Height of the region (in pixels)
			-- @usage ItemList:GetRect()
			------------------------------------

			-- GetRect
			["GetRect"] = function(self)
				return self.__ItemList:GetRect()
			end,

			------------------------------------
			--- Returns whether the region is currently being dragged
			-- @name ItemList:IsDragging
			-- @class function
			-- @return isDragging - 1 if the region (or its parent or ancestor) is currently being dragged; otherwise nil
			-- @usage ItemList:IsDragging()
			------------------------------------

			-- IsDragging
			["IsDragging"] = function(self)
				return self.__ItemList:IsDragging()
			end,

			------------------------------------
			--- Disable rendering of "regions" (fontstrings, textures) in the specified draw layer.
			-- @name ItemList:DisableDrawLayer
			-- @class function
			-- @param layer Name of a graphics layer
			-- @usage ItemList:DisableDrawLayer("ARTWORK")
			------------------------------------

			-- DisableDrawLayer
			["DisableDrawLayer"] = function(self, ...)
				return self.__ItemList:DisableDrawLayer(...)
			end,

			------------------------------------
			--- Enable rendering of "regions" (fontstrings, textures) in the specified draw layer.
			-- @name ItemList:EnableDrawLayer
			-- @class function
			-- @param layer Name of a graphics layer
			-- @usage ItemList:EnableDrawLayer("ARTWORK")
			------------------------------------

			-- EnableDrawLayer
			["EnableDrawLayer"] = function(self, ...)
				return self.__ItemList:EnableDrawLayer(...)
			end,

			------------------------------------
			--- Creates and returns a backdrop table suitable for use in SetBackdrop
			-- @name ItemList:GetBackdrop
			-- @class function
			-- @return A table containing the backdrop settings, or nil if the frame has no backdrop
			-- @usage ItemList:GetBackdrop()
			------------------------------------

			-- GetBackdrop
			["GetBackdrop"] = function(self, ...)
				return self.__ItemList:GetBackdrop(...)
			end,

			------------------------------------
			--- Set the backdrop of the frame according to the specification provided.
			-- @name ItemList:SetBackdrop
			-- @class function
			-- @param backdropTable Optional,A table containing the backdrop settings, or nil to remove the frame's backdrop
			-- @usage ItemList:SetBackdrop{<br>
			--    bgFile = "Interface\\DialogFrame\\UI-DialogBox-Gold-Background",<br>
			--    edgeFile = "Interface\\DialogFrame\\UI-DialogBox-Gold-Border",<br>
			--    tile = true,<br>
			--    tileSize = 32,<br>
			--    edgeSize = 32,<br>
			--    insets = {<br>
			--         left = 11,<br>
			--         right = 12,<br>
			--         top = 12,<br>
			--         bottom = 11,<br>
			--    },<br>
			-- }<br>
			------------------------------------

			-- SetBackdrop
			["SetBackdrop"] = function(self, backdropTable)
				return self.__ItemList:SetBackdrop(backdropTable or nil)
			end,

			------------------------------------
			--- Gets the frame's backdrop border color (r, g, b, a)
			-- @name ItemList:GetBackdropBorderColor
			-- @class function
			-- @return red Red component of the color (0.0 - 1.0)
			-- @return green Green component of the color (0.0 - 1.0)
			-- @return blue Blue component of the color (0.0 - 1.0)
			-- @return alpha Alpha (opacity) for the graphic (0.0 = fully transparent, 1.0 = fully opaque)
			-- @usage ItemList:GetBackdropBorderColor()
			------------------------------------

			-- GetBackdropBorderColor
			["GetBackdropBorderColor"] = function(self, ...)
				return self.__ItemList:GetBackdropBorderColor(...)
			end,

			------------------------------------
			--- Set the frame's backdrop's border's color.
			-- @name ItemList:SetBackdropBorderColor
			-- @class function
			-- @param red Red component of the color (0.0 - 1.0)
			-- @param green Green component of the color (0.0 - 1.0)
			-- @param blue Blue component of the color (0.0 - 1.0)
			-- @param alpha Alpha (opacity) for the graphic (0.0 = fully transparent, 1.0 = fully opaque)
			-- @usage ItemList:SetBackdropBorderColor(1, 1, 0.8, 0.4)
			------------------------------------

			-- SetBackdropBorderColor
			["SetBackdropBorderColor"] = function(self, ...)
				return self.__ItemList:SetBackdropBorderColor(...)
			end,

			------------------------------------
			--- Gets the frame's backdrop color (r, g, b, a)
			-- @name ItemList:GetBackdropColor
			-- @class function
			-- @return red Red component of the color (0.0 - 1.0)
			-- @return green Green component of the color (0.0 - 1.0)
			-- @return blue Blue component of the color (0.0 - 1.0)
			-- @return alpha Alpha (opacity) for the graphic (0.0 = fully transparent, 1.0 = fully opaque)
			-- @usage ItemList:GetBackdropColor()
			------------------------------------

			-- GetBackdropColor
			["GetBackdropColor"] = function(self, ...)
				return self.__ItemList:GetBackdropColor(...)
			end,


			------------------------------------
			--- Set the frame's backdrop color.
			-- @name ItemList:SetBackdropColor
			-- @class function
			-- @param red Red component of the color (0.0 - 1.0)
			-- @param green Green component of the color (0.0 - 1.0)
			-- @param blue Blue component of the color (0.0 - 1.0)
			-- @param alpha Alpha (opacity) for the graphic (0.0 = fully transparent, 1.0 = fully opaque)
			-- @usage ItemList:SetBackdropColor(1, 1, 0.8, 0.4)
			------------------------------------

			-- SetBackdropColor
			["SetBackdropColor"] = function(self, ...)
				return self.__ItemList:SetBackdropColor(...)
			end,

			------------------------------------
			--- Gets the modifiers to the frame's rectangle used for clamping the frame to screen.
			-- @name ItemList:GetClampRectInsets
			-- @class function
			-- @return left Offset from the left edge of the frame to the left edge of its clamping area (in pixels)
			-- @return right Offset from the right edge of the frame's clamping area to the right edge of the frame (in pixels)
			-- @return top Offset from the top edge of the frame's clamping area to the top edge of the frame (in pixels)
			-- @return bottom Offset from the bottom edge of the frame to the bottom edge of its clamping area (in pixels)
			-- @usage ItemList:GetClampRectInsets()
			------------------------------------

			-- GetClampRectInsets
			["GetClampRectInsets"] = function(self, ...)
				return self.__ItemList:GetClampRectInsets(...)
			end,

			------------------------------------
			--- Modify the frame's rectangle used to prevent dragging offscreen.
			-- @name ItemList:SetClampRectInsets
			-- @class function
			-- @param left Offset from the left edge of the frame to the left edge of its clamping area (in pixels)
			-- @param right Offset from the right edge of the frame's clamping area to the right edge of the frame (in pixels)
			-- @param top Offset from the top edge of the frame's clamping area to the top edge of the frame (in pixels)
			-- @param bottom Offset from the bottom edge of the frame to the bottom edge of its clamping area (in pixels)
			-- @usage ItemList:SetClampRectInsets(50, -50, -50, 50)
			------------------------------------

			-- SetClampRectInsets
			["SetClampRectInsets"] = function(self, ...)
				return self.__ItemList:SetClampRectInsets(...)
			end,

			------------------------------------
			--- Returns the 3D depth of the frame (for stereoscopic 3D setups)
			-- @name ItemList:GetDepth
			-- @class function
			-- @return Apparent 3D depth of this frame relative to that of its parent frame
			-- @usage ItemList:GetDepth()
			------------------------------------

			-- GetDepth
			["GetDepth"] = function(self, ...)
				return self.__ItemList:GetDepth(...)
			end,

			------------------------------------
			--- Sets the 3D depth of the frame (for stereoscopic 3D configurations)
			-- @name ItemList:SetDepth
			-- @class function
			-- @param depth Apparent 3D depth of this frame relative to that of its parent frame
			-- @usage ItemList:SetDepth(10)
			------------------------------------

			-- SetDepth
			["SetDepth"] = function(self, ...)
				return self.__ItemList:SetDepth(...)
			end,

			------------------------------------
			--- Returns the effective alpha of a frame.
			-- @name ItemList:GetEffectiveAlpha
			-- @class function
			-- @return Effective alpha (opacity) of the region (0.0 = fully transparent, 1.0 = fully opaque)
			-- @usage ItemList:GetEffectiveAlpha()
			------------------------------------

			-- GetEffectiveAlpha
			["GetEffectiveAlpha"] = function(self, ...)
				return self.__ItemList:GetEffectiveAlpha(...)
			end,

			------------------------------------
			--- Returns the overall 3D depth of the frame (for stereoscopic 3D configurations).
			-- @name ItemList:GetEffectiveDepth
			-- @class function
			-- @return Apparent 3D depth of this frame relative to the screen
			-- @usage ItemList:GetEffectiveDepth()
			------------------------------------

			-- GetEffectiveDepth
			["GetEffectiveDepth"] = function(self, ...)
				return self.__ItemList:GetEffectiveDepth(...)
			end,

			------------------------------------
			--- Get the scale factor of this object relative to the root window.
			-- @name ItemList:GetEffectiveScale
			-- @class function
			-- @return Scale factor for the frame relative to its parent
			-- @usage ItemList:GetEffectiveScale()
			------------------------------------

			-- GetEffectiveScale
			["GetEffectiveScale"] = function(self, ...)
				return self.__ItemList:GetEffectiveScale(...)
			end,

			------------------------------------
			--- Gets the frame's hit rectangle inset distances (l, r, t, b)
			-- @name ItemList:GetHitRectInsets
			-- @class function
			-- @return left Distance from the left edge of the frame to the left edge of its mouse-interactive area (in pixels)
			-- @return right Distance from the right edge of the frame to the right edge of its mouse-interactive area (in pixels)
			-- @return top Distance from the top edge of the frame to the top edge of its mouse-interactive area (in pixels)
			-- @return bottom Distance from the bottom edge of the frame to the bottom edge of its mouse-interactive area (in pixels)
			-- @usage ItemList:GetHitRectInsets()
			------------------------------------

			-- GetHitRectInsets
			["GetHitRectInsets"] = function(self, ...)
				return self.__ItemList:GetHitRectInsets(...)
			end,

			------------------------------------
			--- Set the inset distances for the frame's hit rectangle
			-- @name ItemList:SetHitRectInsets
			-- @class function
			-- @param left Distance from the left edge of the frame to the left edge of its mouse-interactive area (in pixels)
			-- @param right Distance from the right edge of the frame to the right edge of its mouse-interactive area (in pixels)
			-- @param top Distance from the top edge of the frame to the top edge of its mouse-interactive area (in pixels)
			-- @param bottom Distance from the bottom edge of the frame to the bottom edge of its mouse-interactive area (in pixels)
			-- @usage ItemList:SetHitRectInsets(10, -10, -10, 10)
			------------------------------------

			-- SetHitRectInsets
			["SetHitRectInsets"] = function(self, ...)
				return self.__ItemList:SetHitRectInsets(...)
			end,

			------------------------------------
			--- Get the ID of this frame.
			-- @name ItemList:GetID
			-- @class function
			-- @return the ID of this frame
			-- @usage ItemList:GetID()
			------------------------------------

			-- GetID
			["GetID"] = function(self, ...)
				return self.__ItemList:GetID(...)
			end,

			------------------------------------
			--- Set the ID of this frame.
			-- @name ItemList:SetID
			-- @class function
			-- @param id A numeric identifier for the frame
			-- @usage ItemList:SetID(1)
			------------------------------------

			-- SetID
			["SetID"] = function(self, ...)
				return self.__ItemList:SetID(...)
			end,

			------------------------------------
			--- Sets whether the frame's depth property is ignored (for stereoscopic 3D setups).
			-- @name ItemList:IgnoreDepth
			-- @class function
			-- @param enable True to ignore the frame's depth property; false to disable
			-- @usage ItemList:IgnoreDepth(false)
			------------------------------------

			-- IgnoreDepth
			["IgnoreDepth"] = function(self, ...)
				return self.__ItemList:IgnoreDepth(...)
			end,

			------------------------------------
			--- Returns whether the frame's depth property is ignored (for stereoscopic 3D setups)
			-- @name ItemList:IsIgnoringDepth
			-- @class function
			-- @return 1 if the frame's depth property is ignored; otherwise nil
			-- @usage ItemList:IsIgnoringDepth()
			------------------------------------

			-- IsIgnoringDepth
			["IsIgnoringDepth"] = function(self, ...)
				return self.__ItemList:IsIgnoringDepth(...)
			end,

			------------------------------------
			--- Get the list of "children" (frames and things derived from frames) of this frame.
			-- @name ItemList:GetChilds
			-- @class function
			-- @return a list of child-frames
			-- @usage ItemList:GetChilds()
			------------------------------------

			-- GetChilds
			["GetChilds"] = function(self, ...)
				return self.__ItemList:GetChilds()
			end,

			------------------------------------
			--- Get the child of the given name
			-- @name ItemList:GetChild
			-- @class function
			-- @param name the child's name or itself
			-- @return the child
			-- @usage ItemList:GetChild("BtnOkay")
			------------------------------------

			-- GetChild
			["GetChild"] = function(self, ob)
				return self.__ItemList:GetChild(ob)
			end,
	    }

		-- Property
		local _Property = {
			-- Filter
			["Filter"] = {
				["Set"] = function(self, func)
					self = self.__ItemList
                    if func and type(func) ~= "function" then
						error("The parameter must be a function.", 2)
					end
					self.__FilterFunc = func
                end,

                ["Get"] = function(self)
					self = self.__ItemList
                    return self.__FilterFunc
                end,

                ["Type"] = "function",
			},
			-- Direction
			["Direction"] = {
				["Set"] = function(self, dir)
					self = self.__ItemList

					if not _EnumDir[dir] then
						error("The Direction is incorrect."..dir, 2)
					end
					self.__Direction = dir

					makePosition(self, 1)
                end,

                ["Get"] = function(self)
					self = self.__ItemList
                    return self.__Direction
                end,

                ["Type"] = _PropDir,
			},
			-- Unit
			["Unit"] = {
				["Set"] = function(self, unit)
					self = self.__ItemList
					self.__Unit = unit
                end,

                ["Get"] = function(self)
					self = self.__ItemList
					return self.__Unit
                end,

                ["Type"] = "string",
			},
			-- ShowNilButton
			["ShowNilButton"] = {
				["Set"] = function(self, flag)
					self = self.__ItemList
					self.__ShowNilButton = (flag and true) or false
                end,

                ["Get"] = function(self)
					self = self.__ItemList
					return (self.__ShowNilButton and true) or false
                end,

                ["Type"] = "boolean",
			},
			-- AllowSameItem
			["AllowSameItem"] = {
				["Set"] = function(self, flag)
					self = self.__ItemList
					self.__AllowSameItem = (flag and true) or false
                end,

                ["Get"] = function(self)
					self = self.__ItemList
					return (self.__AllowSameItem and true) or false
                end,

                ["Type"] = "boolean",
			},
			--- Visible
			["Visible"] = {
				["Set"] = function(self, visible)
					self.__ItemList.Visible = visible
				end,

				["Get"] = function(self)
					return self.__ItemList.Visible
				end,

				["Type"] = "boolean",
			},

			["Backdrop"] = {
				["Set"] = function(self, backdropTable)
					self:SetBackdrop(backdropTable)
				end,

				["Get"] = function(self)
					return self:GetBackdrop()
				end,

				["Type"] = "Backdrop",
			},

			["BackdropBorderColor"] = {
				["Set"] = function(self, colorTable)
					self:SetBackdropBorderColor(colorTable.red, colorTable.green, colorTable.blue, colorTable.alpha)
				end,

				["Get"] = function(self)
					local colorTable = {}
					colorTable.red, colorTable.green, colorTable.blue, colorTable.alpha = self:GetBackdropBorderColor()
					return colorTable
				end,

				["Type"] = "Color",
			},

			["BackdropColor"] = {
				["Set"] = function(self, colorTable)
					self:SetBackdropColor(colorTable.red, colorTable.green, colorTable.blue, colorTable.alpha)
				end,

				["Get"] = function(self)
					local colorTable = {}
					colorTable.red, colorTable.green, colorTable.blue, colorTable.alpha = self:GetBackdropColor()
					return colorTable
				end,

				["Type"] = "Color",
			},

			["ClampRectInsets"] = {
				["Set"] = function(self, RectInset)
					self:SetClampRectInsets(RectInset.left, RectInset.right, RectInset.top, RectInset.bottom)
				end,

				["Get"] = function(self)
					local RectInset = {}
					RectInset.left, RectInset.right, RectInset.top, RectInset.bottom = self:GetClampRectInsets()
					return RectInset
				end,

				["Type"] = "RectInset",
			},

			["HitRectInsets"] = {
				["Set"] = function(self, RectInset)
					self:SetHitRectInsets(RectInset.left, RectInset.right, RectInset.top, RectInset.bottom)
				end,

				["Get"] = function(self)
					local RectInset = {}
					RectInset.left, RectInset.right, RectInset.top, RectInset.bottom = self:GetHitRectInsets()
					return RectInset
				end,

				["Type"] = "RectInset",
			},

			["Depth"] = {
				["Set"] = function(self, depth)
					self:SetDepth(depth)
				end,

				["Get"] = function(self)
					return self:GetDepth()
				end,

				["Type"] = "number",
			},

			["DepthIgnored"] = {
				["Set"] = function(self, enabled)
					self:IgnoreDepth(enabled)
				end,

				["Get"] = function(self)
					return (self:IsIgnoringDepth() and true) or false
				end,

				["Type"] = "boolean",
			},
		}

	    -- New
	    local _New = function(parent)
			local mask = CreateFrame(_Base, nil, parent)

			local frame = CreateFrame("Frame", nil, _ItemListContainer)
			mask.__ItemList = frame
			frame.__Mask = mask

			frame.FrameStrata = "TOOLTIP"
			frame.Height = ItemSize
			frame.Width = ItemSize
            frame.Visible = false

			frame.__Direction = "BOTTOM"
			frame.__Unit = "player"
			frame.__ShowNilButton = true

			frame.OnShow = OnShow
			frame.OnHide = OnHide
			frame.OnEnter = OnEnter
			frame.OnLeave = OnLeave

	        return mask
	    end

	    local _WidgetInfo = {
	        ["WidgetName"] = _WidgetName,
	        ["Base"] = _Base,
	        ["FuncProxy"] = _FuncProxy,
	        ["Property"] = _Property,
	        ["ScriptType"] = _ScriptType,
	        ["New"] = _New,
	    }

		ItemList = IGAS:NewWidget(_WidgetInfo, ItemList)
	end

	------------------------------------------------------
	---------------       ItemButton          ------------
	------------------------------------------------------
	local _WidgetName = "ItemButton"
	local _Base = "Button"

	-- Script
	-- OnClick Script
	local function OnClick(self, ...)
		if self.__ItemLink then
			if ( IsModifiedClick() ) then
				return HandleModifiedItemClick(self.__ItemLink)
			end
		end

		if self:GetChild("ItemList") then
			self:GetChild("ItemList").Visible = not self:GetChild("ItemList").Visible
			return
		end
	end

	local function OnEnter(self, ...)
		if self.__ItemLink then
			GameTooltip:SetOwner(self, "ANCHOR_RIGHT")
			GameTooltip:SetHyperlink(self.__ItemLink)
			GameTooltip:Show()
		end
	end

	local function OnLeave(self, ...)
		GameTooltip:ClearLines()
		GameTooltip:Hide()
	end

	local function OnHide(self, ...)
		if self:GetChild("ItemList") and self:GetChild("ItemList").Visible then
			self:GetChild("ItemList"):Hide()
		end
	end

	local function SetItemTexture(self, texture)
		if ( texture ) then
			self:GetChild("Icon"):Show()
		else
			self:GetChild("Icon"):Hide()
		end
		self:GetChild("Icon"):SetTexture(texture)
	end

	local function SetItemName(self, name)
		self:GetChild("ItemName"):SetText(name)
	end

	local function SetItemCount(self, count)
		if ( not count ) then
			count = 0
		end

		self.__Count = count
		if ( count > 1  ) then
			if ( count > 999 ) then
				count = "*"
			end
			self:GetChild("Count"):SetText(count)
			if self.__ShowItemCount then
				self:GetChild("Count"):Show()
			end
		else
			self:GetChild("Count"):Hide()
		end
	end

    local function list_OnClick(self, item)
        self.Parent.Item = item
    end

	local function GetItemList(self)
		if self:GetChild("ItemList") then
			return self:GetChild("ItemList")
		end
		local itemL = CreateFrame(ItemList, "ItemList", self)
		itemL:SetPoint("CENTER", self, "CENTER")
        itemL.OnClick = list_OnClick
        return itemL
	end

	-- ScriptType
	local _ScriptType = {
		------------------------------------
		--- ScriptType, Run when the Item is changed
		-- @name ItemButton:OnItemChanged
		-- @class function
		-- @usage function ItemButton:OnItemChanged()<br>
		--    -- do someting<br>
		-- end
		------------------------------------
		["OnItemChanged"] = true,
	}

	--	FuncProxy
	local _FuncProxy = {
		------------------------------------
		--- Sets the item count that diplayed on the item button
		-- @name ItemButton:SetItemCount
		-- @class function
		-- @param count the item's count
		-- @usage ItemButton:SetItemCount(10)
		------------------------------------

        -- SetItemCount
		["SetItemCount"] = function(self, count)
			SetItemCount(self, count)
		end,

		------------------------------------
		--- Sets the item stock that diplayed on the item button(show in the bottomleft corner)
		-- @name ItemButton:SetItemStock
		-- @class function
		-- @param count the item's stock
		-- @usage ItemButton:SetItemStock(10)
		------------------------------------

        -- SetItemStock
		["SetItemStock"] = function(self, numInStock)
			if ( not numInStock ) then
				numInStock = 0
			end

			self.__Stock = numInStock;
			if ( numInStock > 0 ) then
				self:GetChild("Stock"):SetFormattedText(MERCHANT_STOCK, numInStock);
				self:GetChild("Stock"):Show()
			else
				self:GetChild("Stock"):Hide()
			end
		end,

		------------------------------------
		--- Sets the item texture
		-- @name ItemButton:SetItemTexture
		-- @class function
		-- @param texture the item's texture
		-- @usage ItemButton:SetItemTexture([[Interface\Icon\Amubush]])
		------------------------------------

        -- SetItemTexture
		["SetItemTexture"] = function(self, texture)
			SetItemTexture(self, texture)
		end,

		------------------------------------
		--- Sets a color shading for the Item's graphics. The effect of changing this property differs by the type of region
		-- @name ItemButton:SetItemTextureVertexColor
		-- @class function
		-- @param red Red component of the color (0.0 - 1.0)
		-- @param green Green component of the color (0.0 - 1.0)
		-- @param blue Blue component of the color (0.0 - 1.0)
		-- @param alpha Alpha (opacity) for the graphic (0.0 = fully transparent, 1.0 = fully opaque)
		-- @usage ItemButton:SetItemTextureVertexColor(1, 1, 1, 1)
		------------------------------------

        -- SetItemTextureVertexColor
		["SetItemTextureVertexColor"] = function(self, r, g, b, a)
			self:GetChild("Icon"):SetVertexColor(r, g, b, a)
		end,

		------------------------------------
		--- Sets whether the texture image should be displayed with zero saturation (i.e. converted to grayscale).
		-- @name ItemButton:SetItemDesaturated
		-- @class function
		-- @param desaturate True to display the texture in grayscale; false to display original texture colors
		-- @param red Optional,Red component of the color (0.0 - 1.0)
		-- @param green Optional,Green component of the color (0.0 - 1.0)
		-- @param blue Optional,Blue component of the color (0.0 - 1.0)
		-- @usage ItemButton:SetItemDesaturated(true)
		------------------------------------

		-- SetItemDesaturated
		["SetItemDesaturated"] = function(self, desaturated, r, g, b)
			local icon = self:GetChild("Icon")
			if ( not icon ) then
				return
			end
			local shaderSupported = icon:SetDesaturated(desaturated);

			if ( not desaturated ) then
				r = 1.0;
				g = 1.0;
				b = 1.0;
			elseif ( not r or not shaderSupported ) then
				r = 0.5;
				g = 0.5;
				b = 0.5;
			end

			icon:SetVertexColor(r, g, b)
		end,

		------------------------------------
		--- Sets a color shading for the itembutton's graphics. The effect of changing this property differs by the type of region
		-- @name ItemButton:SetItemNormalTextureVertexColor
		-- @class function
		-- @param desaturate True to display the texture in grayscale; false to display original texture colors
		-- @param red Optional,Red component of the color (0.0 - 1.0)
		-- @param green Optional,Green component of the color (0.0 - 1.0)
		-- @param blue Optional,Blue component of the color (0.0 - 1.0)
		-- @param alpha Alpha (opacity) for the graphic (0.0 = fully transparent, 1.0 = fully opaque)
		-- @usage ItemButton:SetItemNormalTextureVertexColor(1, 1, 1)
		------------------------------------

		-- SetItemNormalTextureVertexColor
		["SetItemNormalTextureVertexColor"] = function(self, r, g, b, a)
			self:GetChild("NormalTexture"):SetVertexColor(r, g, b, a);
		end,

		------------------------------------
		--- Sets an item name to the item button
		-- @name ItemButton:SetItemName
		-- @class function
		-- @param name the item's name
		-- @usage ItemButton:SetItemName([[XXX' sword]])
		------------------------------------

		-- SetItemName
		["SetItemName"] = function(self, name)
			SetItemName(self, name)
		end,

		------------------------------------
		--- Sets an item link to the item button
		-- @name ItemButton:SetItem
		-- @class function
		-- @param link the item's link
		-- @usage ItemButton:SetItem([XXX' sword])
		------------------------------------

		-- SetItem
		["SetItem"] = function(self, link)
			local itemName, itemTexture, itemCount
			if not link then
				-- Clear
				self.__ItemLink = nil
				SetItemName(self, "")
				SetItemTexture(self, nil)
				if self.__ShowItemCount then
					SetItemCount(self, 0)
				end
				self:Fire("OnItemChanged")
				return
			end
			-- Set
			itemName, _, _, _, _, _, _, _, _, itemTexture = GetItemInfo(link)
			if itemName then
				if link ~= self.__ItemLink then
					self.__ItemLink = link
					SetItemName(self, itemName)
					SetItemTexture(self, itemTexture)
					if self.__ShowItemCount then
						itemCount = GetItemCount(link, self.__BankItemIncluded)
						SetItemCount(self, itemCount)
					end
					self:Fire("OnItemChanged")
				end
			else
				error("The parameter you passed in is not an item.", 2)
			end
		end,

		------------------------------------
		--- Gets the item link from the item button
		-- @name ItemButton:GetItem
		-- @class function
		-- @return link the item's link
		-- @usage ItemButton:GetItem()
		------------------------------------

		-- GetItem
		["GetItem"] = function(self)
			return self.__ItemLink
		end,

		------------------------------------
		--- Refresh the button's item list
		-- @name ItemButton:RefreshList
		-- @class function
		-- @param filterFunc Optional,the filter function
		-- @usage ItemButton:RefreshList()
		------------------------------------

		-- RefreshList
		["RefreshList"] = function(self, filterFunc)
			local list = GetItemList(self)

			if filterFunc and type(filterFunc) == "function" then
				list.Filter = filterFunc
			end
			list:Refresh()
		end,

		------------------------------------
		--- Add an item to the button's item list
		-- @name ItemButton:AddListItem
		-- @class function
		-- @param item the item link
		-- @usage ItemButton:AddListItem([XXX's sword])
		------------------------------------

		-- AddListItem
		["AddListItem"] = function(self, item)
			local list = GetItemList(self)

			list:Add(item)
		end,

		------------------------------------
		--- Remove an item from the button's item list
		-- @name ItemButton:RemoveListItem
		-- @class function
		-- @param item the item link
		-- @usage ItemButton:RemoveListItem([XXX's sword])
		------------------------------------

		-- RemoveListItem
		["RemoveListItem"] = function(self, item)
			local list = GetItemList(self)

			list:Remove(item)
		end,

		------------------------------------
		--- Clear the button's item list
		-- @name ItemButton:ClearList
		-- @class function
		-- @usage ItemButton:ClearList()
		------------------------------------

		-- ClearList
		["ClearList"] = function(self)
			local list = GetItemList(self)

			list:Clear()
		end,

		------------------------------------
		--- Remove the button's item list and dipose the item list
		-- @name ItemButton:RemoveList
		-- @class function
		-- @usage ItemButton:RemoveList()
		------------------------------------

		-- RemoveList
		["RemoveList"] = function(self)
			if self:GetChild("ItemList") then
				self:GetChild("ItemList"):Dispose()
			end
		end,
	}

	--	Property
	local _Property = {
        -- Style
        ["Style"] = {
			["Get"] = function(self)
				return nil
			end,
		},
		-- ShowItemCount
		["ShowItemCount"] = {
			["Set"] = function(self, flag)
				self.__ShowItemCount = (flag and true) or false
			end,

			["Get"] = function(self)
				return (self.__ShowItemCount and true) or false
			end,

			["Type"] = "boolean",
		},
		-- BankItemIncluded
		["BankItemIncluded"] = {
			["Set"] = function(self, flag)
				self.__BankItemIncluded = (flag and true) or false
			end,

			["Get"] = function(self)
				return (self.__BankItemIncluded and true) or false
			end,

			["Type"] = "boolean",
		},
		-- ItemCount
		["ItemCount"] = {
			["Set"] = function(self, count)
				self:SetItemCount(count)
			end,

			["Get"] = function(self)
				return self.__Count or 0
			end,

			["Type"] = "number",
		},
		-- ItemStock
		["ItemStock"] = {
			["Set"] = function(self, numInStock)
				self:SetItemStock(numInStock)
			end,

			["Get"] = function(self)
				return self.__Stock or 0
			end,

			["Type"] = "number",
		},
		-- ItemTexture
		["ItemTexture"] = {
			["Set"] = function(self, texture)
				self:SetItemTexture(texture)
			end,

			["Type"] = "TexturePath",
		},
		-- ShowItemName
		["ShowItemName"] = {
			["Set"] = function(self, flag)
				self.__ShowItemName = (flag and true) or false
				self:GetChild("ItemName").Visible = self.__ShowItemName
			end,

			["Get"] = function(self)
				return (self.__ShowItemName and true) or false
			end,

			["Type"] = "boolean",
		},
		-- ItemName
		["ItemName"] = {
			["Set"] = function(self, name)
				self:SetItemName(name)
			end,

			["Type"] = "string",
		},
		-- Item
		["Item"] = {
			["Set"] = function(self, link)
				self:SetItem(link)
			end,

			["Get"] = function(self)
				return self:GetItem()
			end,

			["Type"] = "string",
		},
		--- Item List
		-- Direction
		["ListDirection"] = {
			["Set"] = function(self, dir)
				local list = GetItemList(self)

				list.Direction = dir
			end,

			["Get"] = function(self)
				local list = GetItemList(self)

				return list.Direction
			end,

			["Type"] = _PropDir,
		},
		-- Unit
		["ListUnit"] = {
			["Set"] = function(self, unit)
				local list = GetItemList(self)

				list.Unit = unit
			end,

			["Get"] = function(self)
				local list = GetItemList(self)
				return list.Unit
			end,

			["Type"] = "string",
		},
		-- ListShowNilButton
		["ListShowNilButton"] = {
			["Set"] = function(self, flag)
				local list = GetItemList(self)
				list.ShowNilButton = flag
			end,

			["Get"] = function(self)
				local list = GetItemList(self)
				return list.ShowNilButton
			end,

			["Type"] = "boolean",
		},
	}

	local function _New(parent)
        local btn = CreateFrame("Button", nil, parent)

        btn.Width = 37
        btn.Height = 37

		-- Texture
        local t = CreateFrame("Texture", "NormalTexture", btn)
        t:SetTexture("Interface\\Buttons\\UI-Quickslot2")
        t.Height = 64
        t.Width = 64
        t:SetPoint("CENTER", btn, "CENTER", 0, -1)
        btn:SetNormalTexture(t)
        btn:SetPushedTexture("Interface\\Buttons\\UI-Quickslot-Depress")
        btn:SetHighlightTexture("Interface\\Buttons\\ButtonHilight-Square")
        t = btn:GetHighlightTexture()
        t:SetBlendMode("ADD")

		-- Icon
        local icon = CreateFrame("Texture", "Icon", btn, "BORDER")
        icon:SetAllPoints(btn)

		-- Count
        local count = CreateFrame("FontString", "Count", btn, "BORDER", "NumberFontNormal")
        count.JustifyH = "RIGHT"
        count.Visible = false
        count:SetPoint("BOTTOMRIGHT", btn, "BOTTOMRIGHT", -5, 2)

        local stock = CreateFrame("FontString", "Stock", btn, "BORDER", "NumberFontNormalYellow")
        stock.JustifyH = "LEFT"
        stock.Visible = false
        stock:SetPoint("TOPLEFT", btn, "TOPLEFT", 0, -2)

		-- ItemName
		local itemName = CreateFrame("FontString", "ItemName", btn, "OVERLAY", "GameFontHighlightSmallOutline")
		itemName.Width = 36
		itemName.Height = 10
		itemName.Visible = false
        itemName:SetPoint("BOTTOM", btn, "BOTTOM", 0, 2)

		btn.__ShowItemCount = true
		btn.__ShowItemName = false

		btn:StackScript("OnClick", OnClick)
		btn:StackScript("OnEnter", OnEnter)
		btn:StackScript("OnLeave", OnLeave)
        btn:StackScript("OnHide", OnHide)

		return btn
	end

	-- Register
	local _WidgetInfo = {
		["WidgetName"] = _WidgetName,
		["Base"] = _Base,
		["FuncProxy"] = _FuncProxy,
		["Property"] = _Property,
		["ScriptType"] = _ScriptType,
		["New"] = _New,
		-- Special Item
		["ItemList"] = ItemList,
	}

	IGAS:NewWidget(_WidgetInfo, true)
end
