-- Author      : Kurapica

------------------------------------------------------------------------------------------------------------------------------------------------------------------
--- ColorPicker is using to pick color for special using.
-- <br><br>inherit <a href="..\Base\ColorSelect.html">ColorSelect</a> For all methods, properties and scriptTypes
-- @name ColorPicker
-- @class table
-- @field CaptionAlign the caption's align:LEFT, RIGHT, CENTER
-- @field TitleBarColor the title bar's color, default alpha is 0, so make it can't be see
-- @field Caption The text to be displayed at the top of the ColorPicker.
-- @field Color The color that the colorPicker is picked or to be displayed
-- @field OkayButtonText the text that displayed on the okay button
-- @field CancelButtonText the text that displayed on the cancel button
-- @field Style the colorPicker's style: COMMON, LIGHT
-- @field HasOpacity Whether the opacity should be used
----------------------------------------------------------------------------------------------------------------------------------------
do
	-- Check Version
	local version = 1
	if not IGAS:NewAddon("IGAS.GUI.ColorPicker", version) then
		return
	end

	-- API Localization
	local pairs = pairs
    local ipairs = ipairs
    local type = type
    local error = error
	local wipe = wipe
    local CreateFrame = IGAS.GUI.CreateFrame

	-- Widget Name & Inherit
	local _WidgetName = "ColorPicker"
	local _Base = "ColorSelect"

	-- Define Style
    -- Style
    local TEMPLATE_COMMON = "COMMON"
    local TEMPLATE_LIGHT = "LIGHT"

    -- Define Block
	local _StyleEnum = IGAS:NewEnum("ColorPickerStyle", {
        TEMPLATE_COMMON,
		TEMPLATE_LIGHT,
    })

	local _StyleType = IGAS:NewPropertyType("ColorPickerStyle", {
		["Type"] = "enum",
		["EnumType"] = _StyleEnum,
	})

	-- Backdrop Handlers
	local _FrameBackdrop = {
		bgFile = "Interface\\Tooltips\\UI-Tooltip-Background",
		edgeFile = "Interface\\Tooltips\\UI-Tooltip-Border",
		tile = true, tileSize = 16, edgeSize = 16,
		insets = { left = 3, right = 3, top = 3, bottom = 3 }
	}
	local _FrameBackdropTitle = {
		bgFile = "Interface\\Tooltips\\UI-Tooltip-Background",
		edgeFile = "",
		tile = true, tileSize = 16, edgeSize = 0,
		insets = { left = 3, right = 3, top = 3, bottom = 0 }
	}
	local _FrameBackdropCommon = {
		bgFile = "Interface\\DialogFrame\\UI-DialogBox-Background",
		edgeFile = "Interface\\DialogFrame\\UI-DialogBox-Border",
		tile = true, tileSize = 32, edgeSize = 32,
		insets = { left = 11, right = 12, top = 12, bottom = 11 }
	}
	local _FrameBackdropSlider = {
		bgFile = "Interface\\Buttons\\UI-SliderBar-Background",
		edgeFile = "Interface\\Buttons\\UI-SliderBar-Border",
		tile = true, tileSize = 8, edgeSize = 8,
		insets = { left = 3, right = 3, top = 6, bottom = 6 }
	}

	-- Event Handlers
	local function frameOnMouseDown(self)
		if not self.Parent.InDesignMode then
			self.Parent:StartMoving()
		end
	end

	local function frameOnMouseUp(self)
		self.Parent:StopMovingOrSizing()
	end

	local function Format(value)
		return tonumber(format("%.2f", value))
	end

	local function Slider_OnValueChanged(self)
		self.Text.Text = format("%.2f", 1 - self.Value)
		if self.Visible and self.Enabled then
			return self.Parent:Fire("OnColorPicked", self.Parent:GetColor())
		end
	end

	local function OnColorSelect(self)
		self.ColorSwatch:SetTexture(self:GetColorRGB())
		return self:Fire("OnColorPicked", self:GetColor())
	end

	local function Okay_OnClick(self)
		local parent = self.Parent
		parent:Fire("OnColorPicked", parent:GetColor())
		parent.Visible = false
	end

	local function Cancel_OnClick(self)
		local parent = self.Parent
		parent:Fire("OnColorPicked", parent.__DefaultValue.red, parent.__DefaultValue.green, parent.__DefaultValue.blue, parent.__DefaultValue.alpha)
		parent.Visible = false
	end

	-- ScriptType
	local _ScriptType = {
		------------------------------------
		--- ScriptType, Run when the color is selected
		-- @name ColorPicker:OnColorPicked
		-- @class function
		-- @usage function ColorPicker:OnColorPicked(red, green, blue, alpha)<br>
		--    -- do someting<br>
		-- end
		------------------------------------
		["OnColorPicked"] = true,
	}

	--	FuncProxy
	local _FuncProxy = {
		------------------------------------
		--- Sets the ColorPicker's style
		-- @name ColorPicker:SetStyle
		-- @class function
		-- @param style the style of the ColorPicker : COMMON, LIGHT
		-- @usage ColorPicker:SetStyle("LIGHT")
		------------------------------------

        -- SetStyle
        ["SetStyle"] = function(self, style)
            local t

            -- Check Style
            if not style or type(style) ~= "string" or style == self.__Style then
                return
            end

            -- Change Style
            if style == TEMPLATE_COMMON then
				self:SetBackdrop(_FrameBackdropCommon)

				self.ColorPicker_Caption:SetBackdrop(nil)
				self.ColorPicker_Caption:ClearAllPoints()
				self.ColorPicker_Caption:SetPoint("TOP", self, "TOP", 0, 12)
				self.ColorPicker_Caption.Width = 136
				self.ColorPicker_Caption.Height = 36
				self.ColorPicker_Caption.Text.JustifyV = "MIDDLE"
				self.ColorPicker_Caption.Text.JustifyH = "CENTER"

				local backTexture = CreateFrame("Texture", "HeaderBack", self.ColorPicker_Caption, "BACKGROUND")

				backTexture.TexturePath = [[Interface\DialogFrame\UI-DialogBox-Header]]
				backTexture:SetAllPoints(self.ColorPicker_Caption)
				backTexture:SetTexCoord(0.24, 0.76, 0, 0.56)
				backTexture.Visible = true

				if self:GetChild("ColorPicker_Caption"):GetChild("Text").Width + 36 > 136 then
					self:GetChild("ColorPicker_Caption").Width = self:GetChild("ColorPicker_Caption"):GetChild("Text").Width + 36
				else
					self:GetChild("ColorPicker_Caption").Width = 136
				end
            elseif style == TEMPLATE_LIGHT then
				self:SetBackdrop(_FrameBackdrop)
				self:SetBackdropColor(0, 0, 0)
				self:SetBackdropBorderColor(0.4, 0.4, 0.4)

				local title = CreateFrame("Frame", "ColorPicker_Caption", self)
				title:ClearAllPoints()
				title:SetPoint("TOPLEFT", self, "TOPLEFT", 6, 0)
				title:SetPoint("RIGHT", self, "RIGHT")
				title:SetBackdrop(_FrameBackdropTitle)
				title:SetBackdropColor(1, 0, 0, 0)
				title.Height = 24

				if title.HeaderBack then
					title.HeaderBack.Visible = false
				end
            end

			self.__Style = style
        end,

		------------------------------------
		--- Gets the ColorPicker's style
		-- @name ColorPicker:GetStyle
		-- @class function
		-- @return the style of the ColorPicker : COMMON, LIGHT
		-- @usage ColorPicker:GetStyle()
		------------------------------------

        -- GetStyle
        ["GetStyle"] = function(self)
            return self.__Style or TEMPLATE_LIGHT
        end,

		------------------------------------
		--- Sets the ColorPicker's default color
		-- @name ColorPicker:SetColor
		-- @class function
		-- @param red component of the color (0.0 - 1.0)
		-- @param green component of the color (0.0 - 1.0)
		-- @param blue component of the color (0.0 - 1.0)
		-- @param alpha Optional,opacity for the graphic (0.0 = fully transparent, 1.0 = fully opaque)
		-- @usage ColorPicker:SetColor(1, 0.7, 0, 0.6)
		------------------------------------

        -- SetColor
		["SetColor"] = function(self, red, green, blue, alpha)
			alpha = (alpha and type(alpha) == "number" and alpha >= 0 and alpha <= 1 and alpha) or 1

			self.OpacitySlider.Value = 1 - alpha

			self:SetColorRGB(red, green, blue)

			red, green, blue = self:GetColorRGB()

			alpha = 1 - self.OpacitySlider.Value

			-- Keep default
			self.__DefaultValue.red, self.__DefaultValue.green, self.__DefaultValue.blue, self.__DefaultValue.alpha = Format(red), Format(green), Format(blue), Format(alpha)
		end,

		------------------------------------
		--- Gets the ColorPicker's default color
		-- @name ColorPicker:GetColor
		-- @class function
		-- @return red component of the color (0.0 - 1.0)
		-- @return green component of the color (0.0 - 1.0)
		-- @return blue component of the color (0.0 - 1.0)
		-- @return alpha Optional,opacity for the graphic (0.0 = fully transparent, 1.0 = fully opaque)
		-- @usage ColorPicker:GetColor()
		------------------------------------

        -- GetColor
		["GetColor"] = function(self)
			local red, green, blue = self:GetColorRGB()

			if self.OpacitySlider.Visible and self.OpacitySlider.Enabled then
				return Format(red), Format(green), Format(blue), Format(1 - self.OpacitySlider.Value)
			else
				return Format(red), Format(green), Format(blue)
			end
		end,
	}

	--	Property
	local _Property = {
		-- Color
		["Color"] = {
			["Set"] = function(self, color)
				self:SetColor(color)
			end,

			["Get"] = function(self)
				local colorTable = {}
				colorTable.red, colorTable.green, colorTable.blue, colorTable.alpha = self:GetColor()
				return colorTable
			end,

			["Type"] = "Color",
		},
		-- Style
        ["Style"] = {
			["Set"] = function(self, style)
                self:SetStyle(style)
			end,

			["Get"] = function(self)
				return self:GetStyle()
			end,

			["Type"] = "ColorPickerStyle",
		},
		-- CaptionAlign
		["CaptionAlign"]={
			["Get"] = function(self)
				return self:GetChild("ColorPicker_Caption"):GetChild("Text").JustifyH
			end,
			["Set"] = function(self, value)
				if self.Style ~= TEMPLATE_COMMON then
					self:GetChild("ColorPicker_Caption"):GetChild("Text").JustifyH = value
				end
			end,
			["Type"] = "JustifyH",
		},
		-- TitleBarColor
		["TitleBarColor"]={
			["Get"] = function(self)
				return self:GetChild("ColorPicker_Caption").BackdropColor
			end,
			["Set"] = function(self, value)
				self:GetChild("ColorPicker_Caption").BackdropColor = value
			end,
			["Type"] = "Color",
		},
		-- Caption
		["Caption"] = {
			["Set"] = function(self, title)
				self:GetChild("ColorPicker_Caption"):GetChild("Text").Text = title

				if self.Style == TEMPLATE_COMMON then
					if self:GetChild("ColorPicker_Caption"):GetChild("Text").Width + 36 > 136 then
						self:GetChild("ColorPicker_Caption").Width =  self:GetChild("ColorPicker_Caption"):GetChild("Text").Width + 36
					else
						self:GetChild("ColorPicker_Caption").Width = 136
					end
				end
			end,
			["Get"] = function(self)
				return self:GetChild("ColorPicker_Caption"):GetChild("Text").Text
			end,
			["Type"] = "string",
		},
		-- OkayText
		["OkayButtonText"] = {
			["Set"] = function(self, text)
				self:GetChild("OkayBtn").Text = text or "Okay"
			end,
			["Get"] = function(self)
				return self:GetChild("OkayBtn").Text
			end,
			["Type"] = "string",
		},
		-- CancelText
		["CancelButtonText"] = {
			["Set"] = function(self, text)
				self:GetChild("CancelBtn").Text = text or "Cancel"
			end,
			["Get"] = function(self)
				return self:GetChild("CancelBtn").Text
			end,
			["Type"] = "string",
		},
		-- HasOpacity
		["HasOpacity"] = {
			["Set"] = function(self, flag)
				self:GetChild("OpacitySlider").Enabled = flag
			end,
			["Get"] = function(self)
				return self:GetChild("OpacitySlider").Enabled
			end,
			["Type"] = "boolean",
		},
	}

	local function _New(parent)
		-- New Frame
		local frame = CreateFrame("ColorSelect",nil,parent)

		frame.Width = 360
		frame.Height = 220
		frame.Movable = true
		frame.Resizable = true
		frame.FrameStrata = "FULLSCREEN_DIALOG"
		frame.Toplevel = true
		frame.MouseEnabled = true
		frame.KeyboardEnabled = true

		frame:SetPoint("CENTER",parent,"CENTER",0,0)
		frame:SetMinResize(300,200)
        frame:SetBackdrop(_FrameBackdrop)
		frame:SetBackdropColor(0, 0, 0)
		frame:SetBackdropBorderColor(0.4, 0.4, 0.4)

		frame.__DefaultValue = {}

		frame:StackScript("OnColorSelect", OnColorSelect)

		-- Caption
		local title = CreateFrame("Frame", "ColorPicker_Caption", frame)
		title.MouseEnabled = true
		title.Height = 24
		title:SetPoint("TOPLEFT", frame, "TOPLEFT", 6, 0)
		title:SetPoint("RIGHT", frame, "RIGHT")
		title:SetBackdrop(_FrameBackdropTitle)
		title:SetBackdropColor(1, 0, 0, 0)
		title.OnMouseDown = frameOnMouseDown
		title.OnMouseUp = frameOnMouseUp

		local titleText = CreateFrame("FontString", "Text", title, "OVERLAY", "GameFontNormal")
		titleText:SetPoint("LEFT", title, "LEFT")
		titleText:SetPoint("RIGHT", title, "RIGHT")
		titleText:SetPoint("CENTER", title, "CENTER")
		titleText.Height = 24
		titleText.Text = "ColorPicker"
		titleText.JustifyV = "MIDDLE"
		titleText.JustifyH = "CENTER"

		-- ColorWheelTexture
		local colorWheel = CreateFrame("Texture", "ColorWheel", frame)
		colorWheel.Width = 128
		colorWheel.Height = 128
		colorWheel:SetPoint("TOPLEFT", frame, "TOPLEFT", 32, -32)
		frame:SetColorWheelTexture(colorWheel)

		-- ColorWheelThumbTexture
		local colorWheelThumb = CreateFrame("Texture", "ColorWheelThumb", frame)
		colorWheelThumb.Width = 10
		colorWheelThumb.Height = 10
		colorWheelThumb.TexturePath = [[Interface\Buttons\UI-ColorPicker-Buttons]]
		colorWheelThumb:SetTexCoord(0, 0.15625, 0, 0.625)
		frame:SetColorWheelThumbTexture(colorWheelThumb)

		-- ColorValueTexture
		local colorValue = CreateFrame("Texture", "ColorValue", frame)
		colorValue.Width = 32
		colorValue.Height = 128
		colorValue:SetPoint("TOPLEFT", colorWheel, "TOPRIGHT", 24, 0)
		frame:SetColorValueTexture(colorValue)

		-- ColorValueThumbTexture
		local colorValueThumb = CreateFrame("Texture", "ColorValueThumb", frame)
		colorValueThumb.Width = 48
		colorValueThumb.Height = 14
		colorValueThumb.TexturePath = [[Interface\Buttons\UI-ColorPicker-Buttons]]
		colorValueThumb:SetTexCoord(0.25, 1.0, 0, 0.875)
		frame:SetColorValueThumbTexture(colorValueThumb)

		-- ColorSwatch
		local watch = CreateFrame("Texture", "ColorSwatch", frame, "ARTWORK")
		watch.Width = 32
		watch.Height = 32
		watch:SetPoint("TOPLEFT", colorValue, "TOPRIGHT", 24, 0)
		watch:SetTexture(1, 1, 1, 1)

		-- OpacitySlider
		local sliderOpacity = CreateFrame("Slider", "OpacitySlider", frame)
		sliderOpacity:SetMinMaxValues(0, 1)
		sliderOpacity:SetPoint("TOPLEFT", watch, "TOPRIGHT", 24, 0)
		sliderOpacity.Orientation = "VERTICAL"
		sliderOpacity.ValueStep = 0.01
		sliderOpacity.Value = 0
		sliderOpacity.Width = 16
		sliderOpacity.Height = 128
		sliderOpacity:SetBackdrop(_FrameBackdropSlider)
		sliderOpacity:SetThumbTexture("Interface\\Buttons\\UI-SliderBar-Button-Vertical")

		local thumb = sliderOpacity:GetThumbTexture()
		thumb.Height = 32
		thumb.Width = 32

		local sliderText = CreateFrame("FontString", "Text", sliderOpacity, "ARTWORK", "GameFontNormalSmall")
		sliderText:SetPoint("BOTTOM", sliderOpacity, "TOP")
		sliderText.Text = "1.00"

		local subText = CreateFrame("FontString", "SubText", sliderOpacity, "ARTWORK", "GameFontNormalHuge")
		subText.Text = "-"
		subText:SetPoint("BOTTOMLEFT", sliderOpacity, "BOTTOMRIGHT", 8, 3)
		subText:SetTextColor(1, 1, 1)

		local addText = CreateFrame("FontString", "AddText", sliderOpacity, "ARTWORK", "GameFontNormalHuge")
		addText.Text = "+"
		addText:SetPoint("TOPLEFT", sliderOpacity, "TOPRIGHT", 6, -3)
		addText:SetTextColor(1, 1, 1)

		sliderOpacity.OnValueChanged = Slider_OnValueChanged

		-- Okay Button
		local btnOkay = CreateFrame("Button", "OkayBtn", frame)
		btnOkay.Style = "NORMAL"
		btnOkay.Height = 24
		btnOkay.Text = "Okay"
		btnOkay:SetPoint("BOTTOMLEFT", frame, "BOTTOMLEFT", 6, 12)
		btnOkay:SetPoint("RIGHT", frame, "CENTER")
		btnOkay.OnClick = Okay_OnClick

		-- Cancel Button
		local btnCancel = CreateFrame("Button", "CancelBtn", frame)
		btnCancel.Style = "NORMAL"
		btnCancel.Height = 24
		btnCancel.Text = "Cancel"
		btnCancel:SetPoint("BOTTOMRIGHT", frame, "BOTTOMRIGHT", -6, 12)
		btnCancel:SetPoint("LEFT", frame, "CENTER")
		btnCancel.OnClick = Cancel_OnClick

		return frame
	end

	-- Register
	local _WidgetInfo = {
		["WidgetName"] = _WidgetName,
		["Base"] = _Base,
		["FuncProxy"] = _FuncProxy,
		["Property"] = _Property,
		["ScriptType"] = _ScriptType,
		["New"] = _New,
	}

	IGAS:NewWidget(_WidgetInfo, true)
end
