﻿-- Author      : Kurapica
-- ChangeLog
--				 2010/01/30
--				 2010/01/30 Change IsObjectType method

----------------------------------------------------------------------------------------------------------------------------------------
--- VirtualUIObject is an abstract UI object type that is used to group together methods that are common to all Virtual interface types
-- @name VirtualUIObject
-- @class table
-- @field Name Set or get the frame's name, it's an unique identify in it's parent's childs
-- @field InDesignMode Check if the frame is under design
-- @field Parent Set or Get the frame's parent
----------------------------------------------------------------------------------------------------------------------------------------
do
    -- Check Version
	local version = 5

	if not IGAS:NewAddon("IGAS.GUI.VirtualUIObject", version) then
		return
	end

    local type = type
    local setmetatable = setmetatable
	local error = error
	local getn = table.getn
	local tinsert = table.insert
	local tremove = table.remove
	local twipe = table.wipe
	local geterrorhandler = geterrorhandler
	local pcall = pcall
	local CreateFrame = IGAS.GUI.CreateFrame
	local wipe = wipe

	local _Widget = IGAS.GUI.Widget

	local _WidgetName = "VirtualUIObject"

	-- Script
	local function errorhandler(err)
		return geterrorhandler()(err)
	end

	-- ScriptType
	local _ScriptType = {
	}

	--	FuncProxy
	local _FuncProxy = {
		------------------------------------
		--- Release resource, will dipose it's childs the same time
		-- @name VirtualUIObject:Dispose
		-- @class function
		-- @usage VirtualUIObject:Dispose()
		------------------------------------

        -- Dispose
		["Dispose"] = function(self)
			-- Remove self from parent
			if self.__Name and self.__Parent then
				if self.__Parent.__Childs and type(self.__Parent.__Childs) == "table" then
					self.__Parent.__Childs[self.__Name] = nil
				end
			end

			self.__Parent = nil

			-- Dipose self
			setmetatable(self, nil)
			self.__StackScripts = nil
			return wipe(self)
		end,

        ------------------------------------
		--- Get the parent of this frame (The object, not just the name)
		-- @name VirtualUIObject:GetParent
		-- @class function
		-- @return the parent of the frame, if is not a igas frame, automaticaly convert
		-- @usage VirtualUIObject:GetParent()
		------------------------------------

        -- GetParent
        ["GetParent"] = function(self)
            return self.__Parent
        end,

        ------------------------------------
		--- Set the parent for this frame
		-- @name VirtualUIObject:SetParent
		-- @class function
		-- @param parent the parent of the frame, if is not a igas frame, automaticaly convert
		-- @usage VirtualUIObject:SetParent()
		------------------------------------

        -- SetParent
        ["SetParent"] = function(self, parent)
			if parent and type(parent) == "string" then
				parent = _G[parent]

				if not parent then
					error("No such frame", 2)
				end
			end

			parent = IGAS:GetWrapper(parent)

			if parent and (type(parent) ~= "table" or not parent["IsIGASFrame"] or not parent["AddChild"]) then
				error("This object can't add child.", 2)
			end

			if parent then
				if self.__Name then
	                -- Check
	                if parent.__Childs[self.__Name] then
	                    if parent.__Childs[self.__Name] ~= self then
	                        error("The parent has a child with the same name")
	                    else
	                        return
	                    end
	                end

	                -- Remove ob from it's old parent
					self:SetParent(nil)

	                -- Add ob to new parent
	                self.__Parent = parent
	                parent.__Childs[self.__Name] = self
	            else
	                error("can't add this child", 2)
	            end
			else
				if self.__Name and self.__Parent then
					self.__Parent.__Childs[self.__Name] = nil
				end
				self.__Parent = nil
			end
        end,

		------------------------------------
		--- Return the name of the object.
		-- @name VirtualUIObject:GetName
		-- @class function
		-- @return the name of the object.
		-- @usage VirtualUIObject:GetName()
		------------------------------------

		-- GetName
        ["GetName"] = function(self)
            return self.__Name
        end,

        ------------------------------------
		--- Set the name of the object.
		-- @name VirtualUIObject:SetName
		-- @class function
		-- @param name the name of the object.
		-- @usage VirtualUIObject:SetName("SkillPanel")
		------------------------------------

        -- SetName
        ["SetName"] = function(self, name)
            if self.__Parent then
                if name == self.__Name then
                    return
                end

                if self.__Parent.__Childs and type(self.__Parent.__Childs) == "table" then
                    if self.__Parent.__Childs[name] then
                        error("this name is used.",2)
                    end

                    if self.__Name then
                        self.__Parent.__Childs[self.__Name] = nil
                    end
                end

                if not (self.__Parent.__Childs and type(self.__Parent.__Childs) == "table") then
                    self.__Parent.__Childs = {}
                end
                self.__Parent.__Childs[name] = self
            end

            self.__Name = name
        end,

        ------------------------------------
		--- Get the type of this object.
		-- @name VirtualUIObject:GetObjectType
		-- @class function
		-- @return the object type of the frame(the type is defined in IGAS)
		-- @usage VirtualUIObject:GetObjectType()
		------------------------------------

        -- GetObjectType
        ["GetObjectType"] = function(self)
            return self.__ObjectType
        end,

        ------------------------------------
		--- Determine if this object is of the specified type, or a subclass of that type.
		-- @name VirtualUIObject:IsObjectType
		-- @class function
		-- @param type the object type to determined
		-- @return true if the frame's type is the given type or the given type's sub-type.
		-- @usage VirtualUIObject:IsObjectType("Form")
		------------------------------------

        -- IsObjectType
        ["IsObjectType"] = function(self, objType)
			if objType then
				if type(objType) == "string" then
					objType = _Widget[objType]
				end
				if not objType or type(objType) ~= "table" then
					return false
				end
				local tType = self.__ObjectType
				while tType and type(tType) == "table" and tType ~= objType do
					tType = tType.Base
				end
				if tType and tType == objType then
					return true
				end
			end

			return false
        end,

		------------------------------------
		--- Determine if this frame is created from igas
		-- @name VirtualUIObject:IsIGASFrame
		-- @class function
		-- @return always true
		-- @usage VirtualUIObject:IsIGASFrame()
		------------------------------------

        -- IsIGASFrame
		["IsIGASFrame"] = function(self)
			return true
		end,

		------------------------------------
		--- Fire script, to trigger the frame's script handlers
		-- @name VirtualUIObject:Fire
		-- @class function
		-- @param scriptType the script type to be triggered
		-- @param ... the parameters
		-- @usage VirtualUIObject:Fire("OnChoose", 1)
		------------------------------------

        -- Fire
		["Fire"] = function(self, sc, ...)
			local chk = false
			local ret = false

			if not sc then return end

			if self.__StackScripts[sc] then
				for i = 1, getn(self.__StackScripts[sc]) do
					chk, ret = pcall(self.__StackScripts[sc][i], self, ...)
					if not chk then
						return errorhandler(ret)
					end
					if not self.__StackScripts then
						-- means it's disposed
						ret = true
					end
					if ret then
						break
					end
				end

				if not ret and self.__StackScripts[sc][0] then
					chk, ret = pcall(self.__StackScripts[sc][0], self, ...)
					if not chk then
						return errorhandler(ret)
					end
				end
			elseif self[sc] and type(self[sc]) == "function" then
				chk, ret = pcall(self[sc], self, ...)
				if not chk then
					return errorhandler(ret)
				end
			end
		end,

		------------------------------------
		--- Using for call sup-type methods for this frame
		-- @name VirtualUIObject:CallMethod
		-- @class function
		-- @param superType the super type(name or object)
		-- @param method the method's name that to be called
		-- @param ... the parameters
		-- @usage VirtualUIObject:CallMethod("Frame", "Dispose")
		------------------------------------

        -- CallMethod
		["CallMethod"] = function(self, superType, method, ...)
			if not superType or not method or type(method) ~= "string" then return end

			if type(superType) == "string" then
				if _Widget[superType] then
					superType = _Widget[superType]
				else
					return
				end
			elseif type(superType) == "table" then
				if not superType["FuncProxy"] then
					return
				end
			else
				return
			end

	        if superType["FuncProxy"][method] and type(superType["FuncProxy"][method]) == "function" then
	        	return superType["FuncProxy"][method](self, ...)
	        end
		end,

		------------------------------------
		--- Check if the script type is supported by the frame
		-- @name VirtualUIObject:HasScript
		-- @class function
		-- @param type the script type's name
		-- @return true if the frame has that script type
		-- @usage VirtualUIObject:HasScript("OnEnter")
		------------------------------------

        -- HasScript
		["HasScript"] = function(self, handler)
			if not handler or type(handler) ~= "string" then
				return false
			end
			return (self.__ObjectType.ScriptType[handler] and true) or false
		end,

		------------------------------------
		--- Return the script handler of the given name
		-- @name VirtualUIObject:GetScript
		-- @class function
		-- @param type the script type's name
		-- @return the script handler of the given name if exists
		-- @usage VirtualUIObject:GetScript("OnEnter")
		------------------------------------

		-- GetScript
		["GetScript"] = function(self, handler)
			if not handler or type(handler) ~= "string" then
				return nil
			end
			return (self.__StackScripts[handler] and self.__StackScripts[handler][0]) or nil
		end,

		------------------------------------
		--- Set the script handler of the given name
		-- @name VirtualUIObject:SetScript
		-- @class function
		-- @param type the script type's name
		-- @param func the script handler
		-- @usage VirtualUIObject:SetScript("OnEnter", function() end)
		------------------------------------

		-- SetScript
		["SetScript"] = function(self, handler, func)
			if not handler or type(handler) ~= "string" then
				error("The handler can't be use", 2)
			end
			if not self:HasScript(handler) then
				error("The handler is not exist.", 2)
			end
			if func and type(func) ~= "function" then
				error("This is a script handler, must be a function", 2)
			end

			if not self.__StackScripts[handler] or type(self.__StackScripts[handler]) ~= "table" then
				self.__StackScripts[handler] = {}
			end

			rawset(self.__StackScripts[handler], 0, func)
		end,

		------------------------------------
		--- StackScript<br>
		-- This is a special method. It use a table __StackScripts to store the object's script handlers.<br>
		-- For Exp: OnClick handler<br>
		-- object.__StackScripts["OnClick"] = {<br>
		-- 	[0] == "function:XXXX"		-- This is the user's handler function.It will be called the last time.And can be replaced when you use this code object.OnClick = XXX.<br>
		--	[1] == "function:XXXX"		-- This will be called the first time when OnClick is triggered.If this return true, all left handlers will not be called. Can only be set by object:StackScript(handler, func, flag)<br>
		--	[2] == "function:XXXX"		-- This will be called the second time.If this return true, all left handlers will not be called.<br>
		--	......<br>
		-- }
		-- @name VirtualUIObject:StackScript
		-- @class function
		-- @param type the script type's name
		-- @param func the script handler
		-- @param flag Optional,clear the prev-defined scripts if true
		-- @usage function OnEnter() end<br>
		-- VirtualUIObject:StackScript("OnEnter", OnEnter)
		------------------------------------

		-- StackScript
		["StackScript"] = function(self, handler, func, flag)
			if type(handler) ~= "string" then
				error("The handler's name must be a string", 2)
			end
			if not self:HasScript(handler) then
				error("The handler is not exist.", 2)
			end
			if func and type(func) ~= "function" then
				error("The handler must be a function", 2)
			end

			if not self.__StackScripts[handler] or type(self.__StackScripts[handler]) ~= "table" then
				self.__StackScripts[handler] = {}
			end

			if flag then
				-- Clear this handler's stack
				for i = getn(self.__StackScripts[handler]), 1, -1 do
					tremove(self.__StackScripts[handler], i)
				end
			end

			tinsert(self.__StackScripts[handler], func)
		end,

		------------------------------------
		--- Remove a stacked script handler of the given name
		-- @name VirtualUIObject:UnStackScript
		-- @class function
		-- @param type the script type's name
		-- @param func the script handler
		-- @usage VirtualUIObject:UnStackScript("OnEnter", OnEnter)
		------------------------------------

		-- UnStackScript
		["UnStackScript"] = function(self, handler, func)
			if type(handler) ~= "string" then
				error("The handler's name must be a string", 2)
			end
			if not self:HasScript(handler) then
				error("The handler is not exist.", 2)
			end
			if not func then
				error("Don't use nil for unstack function.", 2)
			elseif func and type(func) ~= "function" then
				error("The unstack function must be a function", 2)
			end

			if not self.__StackScripts[handler] or type(self.__StackScripts[handler]) ~= "table" then
				self.__StackScripts[handler] = {}
			end

			-- Clear this handler's stack
			for i = getn(self.__StackScripts[handler]), 1, -1 do
				if self.__StackScripts[handler][i] == func then
					tremove(self.__StackScripts[handler], i)
				end
			end
		end,
	}

	--	Property
	local _Property = {
		--- Parent
		["Parent"] = {
			["Set"] = function(self, parent)
				self:SetParent(parent)
			end,

			["Get"] = function(self)
				return self:GetParent()
			end,

			["Type"] = "UIObject",
		},
		--- Name
		["Name"] = {
			["Set"] = function(self, name)
				self:SetName(name)
			end,

			["Get"] = function(self)
				return self:GetName()
			end,

			["Type"] = "string",
		},
        -- InDesignMode
        ["InDesignMode"] = {
            ["Get"] = function(self)
				return (self.__InDesignMode and true) or false
			end,
			["Type"] = "boolean",
        },
	}

	local function _New(parent)
		local VirtualUIObject = {}

        -- Set this, make it can work in gui system.
        VirtualUIObject.__Wrapper = VirtualUIObject
        VirtualUIObject.__UI = VirtualUIObject

		return VirtualUIObject
	end

	-- Register
	local _WidgetInfo = {
		["WidgetName"] = _WidgetName,
		["FuncProxy"] = _FuncProxy,
		["Property"] = _Property,
		["ScriptType"] = _ScriptType,
		["New"] = _New,
	}

	IGAS:NewWidget(_WidgetInfo, true)
end