﻿-- Author      : Kurapica
-- Create Date : 6/13/2008 9:27:45 PM

----------------------------------------------------------------------------------------------------------------------------------------
--- Texture is used to display pic or color.
-- <br><br>inherit <a href=".\LayeredRegion.html">LayeredRegion</a> For all methods, properties and scriptTypes
-- @name Texture
-- @class table
-- @field BlendMode the blend mode of the texture
-- @field Desaturated whether the texture image should be displayed with zero saturation (i.e. converted to grayscale)
-- @field TexturePath the texture object's image file path
-- @field PortraitUnit the unit be de displayed as a portrait, such as "player", "target"
-- @field PortraitTexture the texture to be displayed from a file applying circular opacity mask making it look round like portraits.
-- @field Color the texture's color
-- @field VertexColor the color shading for the region's graphics
----------------------------------------------------------------------------------------------------------------------------------------
do
	-- Check Version
	local version = 3
	if not IGAS:NewAddon("IGAS.GUI.Texture", version) then
		return
	end

    local CreateFrame = CreateFrame
	local select = select
	local SetPortraitTexture = SetPortraitTexture
	local UnitExists = UnitExists
	local strmatch = strmatch

	local _WidgetName = "Texture"
    local _Base = "LayeredRegion"

	-- FuncProxy
	local _FuncProxy = {

        -- SetParent(parent or "parentName") - Set the parent for this frame
        ["SetParent"] = function(self, parent)
			-- Get frame from name
			if parent and type(parent) == "string" then
				parent = _G[parent]

				if not parent then
					error("No such frame", 2)
				end
			end

			parent = IGAS:GetWrapper(parent)

			-- Check parent
			if parent and (type(parent) ~= "table" or not parent["IsIGASFrame"] or not parent["AddChild"]) then
				error("This object can't add child.", 2)
			end

			-- Set parent
			if parent then
				if self.__Name then
	                -- Check
	                if parent.__Childs[self.__Name] then
	                    if parent.__Childs[self.__Name] ~= self then
	                        error("The parent has a child with the same name")
	                    else
	                        return
	                    end
	                end

	                -- Remove ob from it's old parent
					self:SetParent(nil)

	                -- Add ob to new parent
	                self.__Parent = parent
	                parent.__Childs[self.__Name] = self

					-- SetParent
					if IGAS:GetUI(self) ~= self and IGAS:GetUI(self)["SetParent"] then
						IGAS:GetUI(self):SetParent(IGAS:GetUI(parent))
					end
	            else
	                error("can't add this child", 2)
	            end
			else
				if self.__Name and self.__Parent then
					self.__Parent.__Childs[self.__Name] = nil
				end
				self.__Parent = nil
			end
        end,

		------------------------------------
		--- Returns the blend mode of the texture
		-- @name Texture:GetBlendMode
		-- @class function
		-- @return mode - Blend mode of the texture: ADD, ALPHAKEY, BLEND, DISABLE, MOD
		-- @usage Texture:GetBlendMode()
		------------------------------------

		-- GetBlendMode
		["GetBlendMode"] = function(self, ...)
			return self.__UI:GetBlendMode(...)
		end,

		------------------------------------
		--- Returns corner coordinates for scaling or cropping the texture image.
		-- @name Texture:GetTexCoord
		-- @class function
		-- @return ULx - Upper left corner X position, as a fraction of the image's width from the left (number)
		-- @return ULy - Upper left corner Y position, as a fraction of the image's height from the top (number)
		-- @return LLx - Lower left corner X position, as a fraction of the image's width from the left (number)
		-- @return LLy - Lower left corner Y position, as a fraction of the image's height from the top (number)
		-- @return URx - Upper right corner X position, as a fraction of the image's width from the left (number)
		-- @return URy - Upper right corner Y position, as a fraction of the image's height from the top (number)
		-- @return LRx - Lower right corner X position, as a fraction of the image's width from the left (number)
		-- @return LRy - Lower right corner Y position, as a fraction of the image's height from the top (number)
		-- @usage Texture:GetTexCoord()
		------------------------------------

		-- GetTexCoord
		["GetTexCoord"] = function(self, ...)
			return self.__UI:GetTexCoord(...)
		end,

		------------------------------------
		--- Returns the path to the texture's image file
		-- @name Texture:GetTexture
		-- @class function
		-- @return texture - Path to the texture image file, or one of the following values: Portrait1, SolidTexture
		-- @usage Texture:GetTexture()
		------------------------------------

		-- GetTexture
		["GetTexture"] = function(self, ...)
			local value = self.__UI:GetTexture(...)
			
			if strmatch(value, "^Portrait%d*") then
				return self.__Unit
			elseif value == "SolidTexture" then
				if self.__Color then
					return self.__Color.red, self.__Color.green, self.__Color.blue, self.__Color.alpha
				else
					return nil
				end
			end
			
			return self.__UI:GetTexture(...)
		end,

		------------------------------------
		--- Returns the shading color of the texture.
		-- @name Texture:GetVertexColor
		-- @class function
		-- @return red - Red component of the color (0.0 - 1.0) (number)
		-- @return green - Green component of the color (0.0 - 1.0) (number)
		-- @return blue - Blue component of the color (0.0 - 1.0) (number)
		-- @return alpha - Alpha (opacity) for the texture (0.0 = fully transparent, 1.0 = fully opaque) (number)
		-- @usage Texture:GetVertexColor()
		------------------------------------

		-- GetVertexColor
		["GetVertexColor"] = function(self, ...)
			return self.__UI:GetVertexColor(...)
		end,

		------------------------------------
		--- Returns whether the texture image should be displayed with zero saturation (i.e. converted to grayscale). The texture may not actually be displayed in grayscale if the current display hardware doesn't support that feature;
		-- @name Texture:IsDesaturated
		-- @class function
		-- @return desaturated - 1 if the texture should be displayed in grayscale; otherwise nil
		-- @usage Texture:IsDesaturated()
		------------------------------------

		-- IsDesaturated
		["IsDesaturated"] = function(self, ...)
			return self.__UI:IsDesaturated(...)
		end,

		------------------------------------
		--- Sets the blend mode of the texture
		-- @name Texture:SetBlendMode
		-- @class function
		-- @param mode Blend mode of the texture:ADD, ALPHAKEY, BLEND, DISABLE, MOD
		-- @usage Texture:SetBlendMode("MOD")
		------------------------------------

		-- SetBlendMode
		["SetBlendMode"] = function(self, ...)
			return self.__UI:SetBlendMode(...)
		end,

		------------------------------------
		--- Sets whether the texture image should be displayed with zero saturation (i.e. converted to grayscale).
		-- @name Texture:SetDesaturated
		-- @class function
		-- @param desaturate True to display the texture in grayscale; false to display original texture colors
		-- @return supported - 1 if the current system supports texture desaturation; otherwise nil
		-- @usage Texture:SetDesaturated(true)
		------------------------------------

		-- SetDesaturated
	    ["SetDesaturated"] = function(self, ...)
			return self.__UI:SetDesaturated(...)
		end,

		------------------------------------
		--- Sets a gradient color shading for the texture. Gradient color shading does not change the underlying color of the texture image, but acts as a filter
		-- @name Texture:SetGradient
		-- @class function
		-- @param orientation Token identifying the direction of the gradient: HORIZONTAL, VERTICAL
		-- @param startR Red component of the start color (0.0 - 1.0) (number)
		-- @param startG Green component of the start color (0.0 - 1.0) (number)
		-- @param startB Blue component of the start color (0.0 - 1.0) (number)
		-- @param endR Red component of the end color (0.0 - 1.0) (number)
		-- @param endG Green component of the end color (0.0 - 1.0) (number)
		-- @param endB Blue component of the end color (0.0 - 1.0) (number)
		-- @usage Texture:SetGradient("VERTICAL", 0, 0, 0, 1, 1, 1)
		------------------------------------

		-- SetGradient
		["SetGradient"] = function(self, ...)
			return self.__UI:SetGradient(...)
		end,

		------------------------------------
		--- Sets a gradient color shading for the texture (including opacity in the gradient). Gradient color shading does not change the underlying color of the texture image, but acts as a filter
		-- @name Texture:SetGradientAlpha
		-- @class function
		-- @param orientation Token identifying the direction of the gradient: HORIZONTAL, VERTICAL
		-- @param startR Red component of the start color (0.0 - 1.0) (number)
		-- @param startG Green component of the start color (0.0 - 1.0) (number)
		-- @param startB Blue component of the start color (0.0 - 1.0) (number)
		-- @param startAlpha - Alpha (opacity) for the start side of the gradient (0.0 = fully transparent, 1.0 = fully opaque) (number)
		-- @param endR Red component of the end color (0.0 - 1.0) (number)
		-- @param endG Green component of the end color (0.0 - 1.0) (number)
		-- @param endB Blue component of the end color (0.0 - 1.0) (number)
		-- @param endAlpha - Alpha (opacity) for the end side of the gradient (0.0 = fully transparent, 1.0 = fully opaque) (number)
		-- @usage Texture:SetGradientAlpha("VERTICAL", 0, 0, 0, 1, 1, 1, 1, 1)
		------------------------------------

		-- SetGradientAlpha
		["SetGradientAlpha"] = function(self, ...)
			return self.__UI:SetGradientAlpha(...)
		end,

		------------------------------------
		--- Rotates the texture image.
		-- @name Texture:SetRotation
		-- @class function
		-- @param radians mount by which the texture image should be rotated (in radians; positive values for counter-clockwise rotation, negative for clockwise) (number)
		-- @usage Texture:SetRotation(43)
		------------------------------------

		-- SetRotation
	    ["SetRotation"] = function(self, ...)
			return self.__UI:SetRotation(...)
		end,

		------------------------------------
		--- Sets corner coordinates for scaling or cropping the texture image.
		-- @name Texture:SetTexCoord
		-- @class function
		-- @param left Left edge of the scaled/cropped image, as a fraction of the image's width from the left (number)
		-- @param right Right edge of the scaled/cropped image, as a fraction of the image's width from the left (number)
		-- @param top Top edge of the scaled/cropped image, as a fraction of the image's height from the top (number)
		-- @param bottom Bottom edge of the scaled/cropped image, as a fraction of the image's height from the top (number)
		-- @param ULx - Upper left corner X position, as a fraction of the image's width from the left (number)
		-- @param ULy - Upper left corner Y position, as a fraction of the image's height from the top (number)
		-- @param LLx - Lower left corner X position, as a fraction of the image's width from the left (number)
		-- @param LLy - Lower left corner Y position, as a fraction of the image's height from the top (number)
		-- @param URx - Upper right corner X position, as a fraction of the image's width from the left (number)
		-- @param URy - Upper right corner Y position, as a fraction of the image's height from the top (number)
		-- @param LRx - Lower right corner X position, as a fraction of the image's width from the left (number)
		-- @param LRy - Lower right corner Y position, as a fraction of the image's height from the top (number)
		-- @usage Texture:SetTexCoord(0, 1, 0, 1)
		-- @usage Texture:SetTexCoord(0.1, 0.2, 0.3, 0.3, 0.4, 0.5, 0.5, 0.7)
		------------------------------------

		-- SetTexCoord
	    ["SetTexCoord"] = function(self, ...)
			return self.__UI:SetTexCoord(...)
		end,

		------------------------------------
		--- Sets the texture object's image or color. Returns nil if the texture could not be set
		-- @name Texture:SetTexture
		-- @class function
		-- @param texture - Path to a texture image (string)
		-- @param red - Red component of the color (0.0 - 1.0) (number)
		-- @param green - Green component of the color (0.0 - 1.0) (number)
		-- @param blue - Blue component of the color (0.0 - 1.0) (number)
		-- @param alpha Optional,- Alpha (opacity) for the color (0.0 = fully transparent, 1.0 = fully opaque) (number)
		-- @usage Texture:SetTexture("Interface\\Icon\\Misc_1")
		-- @usage Texture:SetTexture(1, 1, 1)
		------------------------------------

		-- SetTexture
	    ["SetTexture"] = function(self, ...)
			self.__Color = nil
			self.__Unit = nil
			
			if select("#", ...) > 0 then
				if type(select(1, ...)) == "string" then
					return self.__UI:SetTexture(select(1, ...))
				elseif type(select(1, ...)) == "number" then
					local r, g, b, a = ...
					
					r = (r and type(r) == "number" and ((r < 0 and 0) or (r > 1 and 1) or r)) or 0
					g = (g and type(g) == "number" and ((g < 0 and 0) or (g > 1 and 1) or g)) or 0
					b = (b and type(b) == "number" and ((b < 0 and 0) or (b > 1 and 1) or b)) or 0
					a = (a and type(a) == "number" and ((a < 0 and 0) or (a > 1 and 1) or a)) or 1
					
					self.__Color = {red = r, green = g, blue = b, alpha = a}
					return self.__UI:SetTexture(r, g, b, a)
				end
			end
			return self.__UI:SetTexture(nil)
		end,

		------------------------------------
		--- Paint a Texture object with the specified unit's portrait.
		-- @name Texture:SetPortraitTexture
		-- @class function
		-- @param unit the unit to be painted
		-- @usage Texture:SetPortraitTexture("player")
		------------------------------------

		-- SetPortraitTexture
		["SetPortraitTexture"] = function(self, ...)
			self.__Color = nil
			self.__Unit = nil
			
			local unit = select(1, ...)
			
			if unit and type(unit) == "string" and UnitExists(unit) then
				self.__Unit = unit
				return SetPortraitTexture(self.__UI, unit)
			else
				return self.__UI:SetTexture(nil)
			end
		end,

		------------------------------------
		--- Sets the texture to be displayed from a file applying circular opacity mask making it look round like portraits.
		-- @name Texture:SetPortraitToTexture
		-- @class function
		-- @param texture the texture to be displayed
		-- @usage Texture:SetPortraitToTexture([[Interface\Texture\Test]])
		------------------------------------

		-- SetPortraitToTexture
		["SetPortraitToTexture"] = function(self, ...)
			local path = select(1, ...)
			
			self.__Portrait = nil
			
			if path and type(path) == "string" then
				self.__Portrait = path
				return SetPortraitToTexture(self.__UI, path)
			end
			
			return SetPortraitToTexture(self.__UI, nil)
		end,
	}

	-- Property
	local _Property = {
		--- BlendMode
		["BlendMode"] = {
			["Set"] = function(self, mode)
				self.__UI:SetBlendMode(mode)
			end,

			["Get"] = function(self)
				return self.__UI:GetBlendMode()
			end,

			["Type"] = "AlphaMode",
		},
		--- Desaturated
		["Desaturated"] = {
			["Set"] = function(self, desaturation)
				local shaderSupported = self.__UI:SetDesaturated(desaturation);
				if ( not shaderSupported ) then
					if ( desaturation ) then
						self.__UI:SetVertexColor(0.5, 0.5, 0.5);
					else
						self.__UI:SetVertexColor(1.0, 1.0, 1.0);
					end
				end
			end,

			["Get"] = function(self)
				return (self.__UI:IsDesaturated() and true) or false
			end,

			["Type"] = "boolean",
		},
		-- TexturePath
		["TexturePath"] = {
			["Set"] = function(self, path)
				return self:SetTexture(path)
			end,

			["Get"] = function(self)
				local path = self.__UI:GetTexture()
				
				if path and type(path) == "string" and path ~= "SolidTexture" and not strmatch(path, "^Portrait%d*") then
					return path
				end
			end,

			["Type"] = "TexturePath",
		},
		-- PortraitUnit
		["PortraitUnit"] = {
			["Set"] = function(self, unit)
				return self:SetPortraitTexture(unit)
			end,

			["Get"] = function(self)
				local texture = self.__UI:GetTexture()
				
				if texture and type(texture) == "string" and strmatch(texture, "^Portrait%d*") then
					return self.__Unit
				end
			end,

			["Type"] = "Unit",
		},
		-- PortraitTexture
		["PortraitTexture"] = {
			["Set"] = function(self, texture)
				return self:SetPortraitToTexture(texture)
			end,

			["Get"] = function(self)
				return self.__Portrait
			end,

			["Type"] = "string",
		},
		-- Color
		["Color"] = {
			["Set"] = function(self, color)
				if type(color) == "table" then
					self:SetTexture(color.red,color.green,color.blue,color.alpha)
				end
			end,

			["Get"] = function(self)
				local texture = self.__UI:GetTexture()
				
				if texture == "SolidTexture" then
					return self.__Color
				end
			end,

			["Type"] = "Color",
		},
		--- VertexColor
		["VertexColor"] = {
			["Set"] = function(self, color)
				self:SetVertexColor(color.red, color,green, color.blue, color.alpha)
			end,

			["Get"] = function(self)
				local colorTable = {}
				colorTable.red, colorTable.green, colorTable.blue, colorTable.alpha = self.__UI:GetVertexColor()

				return colorTable
			end,

			["Type"] = "Color",
		},
	}

    local _New = function(_parent, layer, inheritsFrom)
		if IGAS:GetUI(_parent) and IGAS:GetUI(_parent)["CreateTexture"] then
			return IGAS:GetUI(_parent):CreateTexture(nil, layer, inheritsFrom)
		else
			Log(3, ""..tostring(_parent).." has no CreateTexture methods")
		end
    end

    local _WidgetInfo = {
        ["WidgetName"] = _WidgetName,
        ["Base"] = _Base,
        ["FuncProxy"] = _FuncProxy,
        ["Property"] = _Property,
        ["New"] = _New,
    }

    IGAS:NewWidget(_WidgetInfo, true, true)
end