﻿-- Author      : Kurapica
-- Create Date : 7/16/2008 14:51

----------------------------------------------------------------------------------------------------------------------------------------
--- StatusBars are similar to Sliders, but they are generally used for display as they don't offer any tools to receive user input.
-- <br><br>inherit <a href=".\Frame.html">Frame</a> For all methods, properties and scriptTypes
-- @name StatusBar
-- @class table
-- @field MinMaxValue the minimum and maximum values of the status bar
-- @field Orientation the orientation of the status bar
-- @field StatusBarColor the color shading for the status bar's texture
-- @field StatusBarTexture the texture used for drawing the filled-in portion of the status bar
-- @field StatusBarTexturePath the texture file used for drawing the filled-in portion of the status bar 
-- @field Layer the layer used for drawing the filled-in portion of the status bar
-- @field Value the value of the status bar
----------------------------------------------------------------------------------------------------------------------------------------
do
	-- Check Version
	local version = 3
	if not IGAS:NewAddon("IGAS.GUI.StatusBar", version) then
		return
	end

    local CreateFrame = CreateFrame

	local _WidgetName = "StatusBar"
    local _Base = "Frame"

	-- ScriptType
	local _ScriptType = {
		------------------------------------
		--- ScriptType, Run when the slider's or status bar's minimum and maximum values change
		-- @name StatusBar:OnMinMaxChanged
		-- @class function
		-- @param min New minimum value of the slider or the status bar
		-- @param max New maximum value of the slider or the status bar
		-- @usage function StatusBar:OnMinMaxChanged(min, max)<br>
		--    -- do someting<br>
		-- end
		------------------------------------
		["OnMinMaxChanged"] = true,

		------------------------------------
		--- ScriptType, Run when the slider's or status bar's value changes
		-- @name StatusBar:OnValueChanged
		-- @class function
		-- @param value New value of the slider or the status bar
		-- @usage function StatusBar:OnValueChanged(value)<br>
		--    -- do someting<br>
		-- end
		------------------------------------
		["OnValueChanged"] = true,
	}

	-- FuncProxy
	local _FuncProxy = {
		------------------------------------
		--- Returns the minimum and maximum values of the status bar
		-- @name StatusBar:GetMinMaxValues
		-- @class function
		-- @return minValue - Lower boundary for values represented on the status bar (number)
		-- @return maxValue - Upper boundary for values represented on the status bar (number)
		-- @usage StatusBar:GetMinMaxValues()
		------------------------------------

		-- GetMinMaxValues
		["GetMinMaxValues"] = function(self, ...)
			return self.__UI:GetMinMaxValues(...)
		end,

		------------------------------------
		--- Returns the orientation of the status bar
		-- @name StatusBar:GetOrientation
		-- @class function
		-- @return orientation - Token describing the orientation and direction of the status bar: HORIZONTAL, VERTICAL
		-- @usage StatusBar:GetOrientation()
		------------------------------------

		-- GetOrientation
		["GetOrientation"] = function(self, ...)
			return self.__UI:GetOrientation(...)
		end,

		------------------------------------
		--- Returns the color shading used for the status bar's texture
		-- @name StatusBar:GetStatusBarColor
		-- @class function
		-- @return red - Red component of the color (0.0 - 1.0) (number)
		-- @return green - Green component of the color (0.0 - 1.0) (number)
		-- @return blue - Blue component of the color (0.0 - 1.0) (number)
		-- @return alpha - Alpha (opacity) for the graphic (0.0 = fully transparent, 1.0 = fully opaque) (number)
		-- @usage StatusBar:GetStatusBarColor()
		------------------------------------

		-- GetStatusBarColor
		["GetStatusBarColor"] = function(self, ...)
			return self.__UI:GetStatusBarColor(...)
		end,

		------------------------------------
		--- Returns the Texture object used for drawing the filled-in portion of the status bar
		-- @name StatusBar:GetStatusBarTexture
		-- @class function
		-- @return texture - Reference to the Texture object used for drawing the filled-in portion of the status bar
		-- @usage StatusBar:GetStatusBarTexture()
		------------------------------------

		-- GetStatusBarTexture
		["GetStatusBarTexture"] = function(self, ...)
			return IGAS:GetWrapper(self.__UI:GetStatusBarTexture(...))
		end,

		------------------------------------
		--- Returns the current value of the status bar
		-- @name StatusBar:GetValue
		-- @class function
		-- @return value - Value indicating the amount of the status bar's area to be filled in (between minValue and maxValue, where minValue, maxValue = StatusBar:GetMinMaxValues())
		-- @usage StatusBar:GetValue()
		------------------------------------

		-- GetValue
		["GetValue"] = function(self, ...)
			return self.__UI:GetValue(...)
		end,

		------------------------------------
		--- Sets the minimum and maximum values of the status bar
		-- @name StatusBar:SetMinMaxValues
		-- @class function
		-- @param minValue Lower boundary for values represented on the status bar (number)
		-- @param maxValue Upper boundary for values represented on the status bar (number)
		-- @usage StatusBar:SetMinMaxValues(1, 100)
		------------------------------------

		-- SetMinMaxValues
		["SetMinMaxValues"] = function(self, ...)
			return self.__UI:SetMinMaxValues(...)
		end,

		------------------------------------
		--- Sets the orientation of the status bar
		-- @name StatusBar:SetOrientation
		-- @class function
		-- @param orientation - Token describing the orientation and direction of the status bar: "HORIZONTAL" or "VERTICAL".
		-- @usage StatusBar:SetOrientation("HORIZONTAL")
		------------------------------------

		-- SetOrientation
		["SetOrientation"] = function(self, ...)
			return self.__UI:SetOrientation(...)
		end,

		------------------------------------
		--- Sets the color shading for the status bar's texture. As with :SetVertexColor(), this color is a shading applied to the texture image.
		-- @name StatusBar:SetStatusBarColor
		-- @class function
		-- @param red Red component of the color (0.0 - 1.0) (number)
		-- @param green Green component of the color (0.0 - 1.0) (number)
		-- @param blue Blue component of the color (0.0 - 1.0) (number)
		-- @param alpha Optional,Alpha (opacity) for the graphic (0.0 = fully transparent, 1.0 = fully opaque) (number)
		-- @usage StatusBar:SetStatusBarColor(1, 1, 1, 1)
		------------------------------------

		-- SetStatusBarColor
		["SetStatusBarColor"] = function(self, ...)
			return self.__UI:SetStatusBarColor(...)
		end,

		------------------------------------
		--- Sets the texture used for drawing the filled-in portion of the status bar. The texture image is stretched to fill the dimensions of the entire status bar, then cropped to show only a portion corresponding to the status bar's current value.
		-- @name StatusBar:SetStatusBarTexture
		-- @class function
		-- @param texture Reference to an existing Texture object Or Path to a texture file
		-- @param layer Optional,Graphics layer in which the texture should be drawn; defaults to ARTWORK if not specified
		-- @usage StatusBar:SetStatusBarTexture([[Interface\Icon\Icon_1]])
		------------------------------------

		-- SetStatusBarTexture
		["SetStatusBarTexture"] = function(self, texture, layer)
			self.__Layer = layer
			return self.__UI:SetStatusBarTexture(IGAS:GetUI(texture), layer)
		end,

		------------------------------------
		--- Sets the value of the status bar
		-- @name StatusBar:SetValue
		-- @class function
		-- @param value Value indicating the amount of the status bar's area to be filled in (between minValue and maxValue, where minValue, maxValue = StatusBar:GetMinMaxValues())
		-- @usage StatusBar:SetValue(30)
		------------------------------------

		-- SetValue
		["SetValue"] = function(self, ...)
			return self.__UI:SetValue(...)
		end,
    }

	-- Property
	local _Property = {
        -- MinMaxValue
        ["MinMaxValue"] = {
			["Set"] = function(self, set)
				self.__UI:SetMinMaxValues(set.min, set.max)
			end,

			["Get"] = function(self)
				local set = {}
				set.min, set.max = self.__UI:GetMinMaxValues()
				return set
			end,

			["Type"] = "MinMaxValue",
		},

        -- Orientation
        ["Orientation"] = {
			["Set"] = function(self, orientation)
				self.__UI:SetOrientation(orientation)
			end,

			["Get"] = function(self)
				return self.__UI:GetOrientation()
			end,

			["Type"] = "Orientation",
		},

		-- StatusBarColor
		["StatusBarColor"] = {
			["Set"] = function(self, colorTable)
				self.__UI:SetStatusBarColor(colorTable.red, colorTable.green, colorTable.blue, colorTable.alpha)
			end,

			["Get"] = function(self)
				local colorTable = {}
				colorTable.red, colorTable.green, colorTable.blue, colorTable.alpha = self.__UI:GetStatusBarColor()
				return colorTable
			end,

			["Type"] = "Color",
		},

        -- StatusBarTexture
        ["StatusBarTexture"] = {
			["Set"] = function(self, texture)
				self:SetStatusBarTexture(texture, self.Layer)
			end,

			["Get"] = function(self)
				return self:GetStatusBarTexture()
			end,

			["Type"] = "Texture",
		},
		
		-- StatusBarTexturePath
		["StatusBarTexturePath"] = {
			["Set"] = function(self, texture)
				self:SetStatusBarTexture(texture, self.Layer)
			end,

			["Get"] = function(self)
				return self:GetStatusBarTexture() and self:GetStatusBarTexture().TexturePath
			end,

			["Type"] = "TexturePath",
		},

		-- Layer
		["Layer"] = {
			["Set"] = function(self, layer)
				self:SetStatusBarTexture(self:GetStatusBarTexture(), layer)
			end,

			["Get"] = function(self)
				return self.__Layer or "ARTWORK"
			end,

			["Type"] = "DrawLayer",
		},

        -- Value
        ["Value"] = {
			["Set"] = function(self, value)
				self.__UI:SetValue(value)
			end,

			["Get"] = function(self)
				return self.__UI:GetValue()
			end,

			["Type"] = "number",
		},
	}

    -- New
    local _New = function(_parent,inheritFrame)
        -- New Frame
		return CreateFrame(_WidgetName,nil,IGAS:GetUI(_parent),inheritFrame)
    end

    local _WidgetInfo = {
        ["WidgetName"] = _WidgetName,
        ["Base"] = _Base,
        ["FuncProxy"] = _FuncProxy,
        ["Property"] = _Property,
        ["ScriptType"] = _ScriptType,
        ["New"] = _New,
    }

    IGAS:NewWidget(_WidgetInfo, true, true)
end