﻿-- Author      : Kurapica
-- Create Date : 7/16/2008 14:39

----------------------------------------------------------------------------------------------------------------------------------------
--- Sliders are elements intended to display or allow the user to choose a value in a range.
-- <br><br>inherit <a href=".\Frame.html">Frame</a> For all methods, properties and scriptTypes
-- @name Slider
-- @class table
-- @field Orientation the orientation of the slider
-- @field ThumbTexture the texture for the slider thumb
-- @field ThumbTexturePath the texture file for the slider thumb
-- @field Layer the layer used for drawing the filled-in portion of the slider
-- @field Value the value representing the current position of the slider thumb
-- @field ValueStep the minimum increment between allowed slider values
-- @field Enabled whether user interaction with the slider is allowed
----------------------------------------------------------------------------------------------------------------------------------------
do
	-- Check Version
	local version = 2
	if not IGAS:NewAddon("IGAS.GUI.Slider", version) then
		return
	end

    local CreateFrame = CreateFrame

	local _WidgetName = "Slider"
    local _Base = "Frame"

	-- ScriptType
	local _ScriptType = {
		------------------------------------
		--- ScriptType, Run when the slider's or status bar's minimum and maximum values change
		-- @name Slider:OnMinMaxChanged
		-- @class function
		-- @param min New minimum value of the slider or the status bar
		-- @param max New maximum value of the slider or the status bar
		-- @usage function Slider:OnMinMaxChanged(min, max)<br>
		--    -- do someting<br>
		-- end
		------------------------------------
		["OnMinMaxChanged"] = true,

		------------------------------------
		--- ScriptType, Run when the slider's or status bar's value changes
		-- @name Slider:OnValueChanged
		-- @class function
		-- @param value New value of the slider or the status bar
		-- @usage function Slider:OnValueChanged(value)<br>
		--    -- do someting<br>
		-- end
		------------------------------------
		["OnValueChanged"] = true,
	}

	-- FuncProxy
	local _FuncProxy = {
		------------------------------------
		--- Disables the slider
		-- @name Slider:Disable
		-- @class function
		-- @usage Slider:Disable()
		------------------------------------

		-- Disable
		["Disable"] = function(self, ...)
			return self.__UI:Disable(...)
		end,

		------------------------------------
		--- Enables the slider
		-- @name Slider:Enable
		-- @class function
		-- @usage Slider:Enable()
		------------------------------------

		-- Enable
		["Enable"] = function(self, ...)
			return self.__UI:Enable(...)
		end,

		------------------------------------
		--- Returns the minimum and maximum values for the slider
		-- @name Slider:GetMinMaxValues
		-- @class function
		-- @return minValue - Lower boundary for values represented by the slider position
		-- @return maxValue - Upper boundary for values represented by the slider position
		-- @usage Slider:GetMinMaxValues()
		------------------------------------

		-- GetMinMaxValues
		["GetMinMaxValues"] = function(self, ...)
			return self.__UI:GetMinMaxValues(...)
		end,

		------------------------------------
		--- Returns the orientation of the slider
		-- @name Slider:GetOrientation
		-- @class function
		-- @return orientation - Token describing the orientation and direction of the slider "HORIZONTAL" or "VERTICAL".
		-- @usage Slider:GetOrientation()
		------------------------------------

		-- GetOrientation
		["GetOrientation"] = function(self, ...)
			return self.__UI:GetOrientation(...)
		end,

		------------------------------------
		--- Returns the texture for the slider thumb
		-- @name Slider:GetThumbTexture
		-- @class function
		-- @return texture - Reference to the Texture object used for the slider thumb
		-- @usage Slider:GetThumbTexture()
		------------------------------------

		-- GetThumbTexture
		["GetThumbTexture"] = function(self, ...)
			return IGAS:GetWrapper(self.__UI:GetThumbTexture(...))
		end,

		------------------------------------
		--- Returns the value representing the current position of the slider thumb
		-- @name Slider:GetValue
		-- @class function
		-- @return value - Value representing the current position of the slider thumb (between minValue and maxValue, where minValue, maxValue = slider:GetMinMaxValues())
		-- @usage Slider:GetValue()
		------------------------------------

		-- GetValue
		["GetValue"] = function(self, ...)
			return self.__UI:GetValue(...)
		end,

		------------------------------------
		--- Returns the minimum increment between allowed slider values
		-- @name Slider:GetValueStep
		-- @class function
		-- @return step - Minimum increment between allowed slider values
		-- @usage Slider:GetValueStep()
		------------------------------------

		-- GetValueStep
		["GetValueStep"] = function(self, ...)
			return self.__UI:GetValueStep(...)
		end,

		------------------------------------
		--- Returns whether user interaction with the slider is allowed
		-- @name Slider:IsEnabled
		-- @class function
		-- @return enabled - 1 if user interaction with the slider is allowed; otherwise nil
		-- @usage Slider:IsEnabled()
		------------------------------------

		-- IsEnabled
		["IsEnabled"] = function(self, ...)
			return self.__UI:IsEnabled(...)
		end,

		------------------------------------
		--- Sets the minimum and maximum values for the slider
		-- @name Slider:SetMinMaxValues
		-- @class function
		-- @param minValue - Lower boundary for values represented by the slider position (number)
		-- @param maxValue - Upper boundary for values represented by the slider position (number)
		-- @usage Slider:SetMinMaxValues(1, 100)
		------------------------------------

		-- SetMinMaxValues
		["SetMinMaxValues"] = function(self, ...)
			return self.__UI:SetMinMaxValues(...)
		end,

		------------------------------------
		--- Sets the orientation of the slider
		-- @name Slider:SetOrientation
		-- @class function
		-- @param orientation Token describing the orientation and direction of the slider : "HORIZONTAL" or "VERTICAL".
		-- @usage Slider:SetOrientation("HORIZONTAL")
		------------------------------------

		-- SetOrientation
		["SetOrientation"] = function(self, ...)
			return self.__UI:SetOrientation(...)
		end,

		------------------------------------
		--- Sets the texture for the slider thumb
		-- @name Slider:SetThumbTexture
		-- @class function
		-- @param texture Reference to an existing Texture  object (texture) Or Path to a texture image file
		-- @param layer Optional,Graphics layer in which the texture should be drawn; defaults to ARTWORK if not specified (string, layer)
		-- @usage Slider:SetThumbTexture([[Interface\Icon\Amubush]], "ARTWORK")
		------------------------------------

		-- SetThumbTexture
		["SetThumbTexture"] = function(self, texture, layer)
			self.__Layer = layer
			return self.__UI:SetThumbTexture(IGAS:GetUI(texture), layer)
		end,

		------------------------------------
		--- Sets the value representing the position of the slider thumb
		-- @name Slider:SetValue
		-- @class function
		-- @param value Value representing the new position of the slider thumb (between minValue and maxValue, where minValue, maxValue = slider:GetMinMaxValues())
		-- @usage Slider:SetValue(10)
		------------------------------------

		-- SetValue
		["SetValue"] = function(self, ...)
			return self.__UI:SetValue(...)
		end,

		------------------------------------
		--- Sets the minimum increment between allowed slider values.
		-- @name Slider:SetValueStep
		-- @class function
		-- @param step Minimum increment between allowed slider values
		-- @usage Slider:SetValueStep(10)
		------------------------------------

		-- SetValueStep
		["SetValueStep"] = function(self, ...)
			return self.__UI:SetValueStep(...)
		end,
    }

	-- Property
	local _Property = {
        -- Orientation
        ["Orientation"] = {
			["Set"] = function(self, orientation)
				self.__UI:SetOrientation(orientation)
			end,

			["Get"] = function(self)
				return self.__UI:GetOrientation()
			end,

			["Type"] = "Orientation",
		},

        -- ThumbTexture
        ["ThumbTexture"] = {
			["Set"] = function(self, texture)
				self:SetThumbTexture(texture, self.Layer)
			end,

			["Get"] = function(self)
				return self:GetThumbTexture()
			end,

			["Type"] = "Texture",
		},
		
		-- ThumbTexturePath
		["ThumbTexturePath"] = {
			["Set"] = function(self, texture)
				self:SetThumbTexture(texture, self.Layer)
			end,

			["Get"] = function(self)
				return self:GetThumbTexture() and self:GetThumbTexture().TexturePath
			end,

			["Type"] = "TexturePath",
		},

		-- Layer
		["Layer"] = {
			["Set"] = function(self, layer)
				self:SetThumbTexture(self:GetThumbTexture(), layer)
			end,

			["Get"] = function(self)
				return self.__Layer or "ARTWORK"
			end,

			["Type"] = "DrawLayer",
		},
		
        -- Value
        ["Value"] = {
			["Set"] = function(self, value)
				self.__UI:SetValue(value)
			end,

			["Get"] = function(self)
				return self.__UI:GetValue()
			end,

			["Type"] = "number",
		},

        -- ValueStep
        ["ValueStep"] = {
			["Set"] = function(self, value)
				self.__UI:SetValueStep(value)
			end,

			["Get"] = function(self)
				return self.__UI:GetValueStep()
			end,

			["Type"] = "number",
		},

		-- Enabled
        ["Enabled"] = {
			["Set"] = function(self, enabled)
                if enabled then
				    self:Enable()
                else
				    self:Disable()
                end
			end,

			["Get"] = function(self)
				return (self:IsEnabled() and true) or false
			end,

			["Type"] = "boolean",
		},
	}

    -- New
    local _New = function(_parent,inheritFrame)
        -- New Frame
		return CreateFrame(_WidgetName,nil,IGAS:GetUI(_parent),inheritFrame)
    end

    local _WidgetInfo = {
        ["WidgetName"] = _WidgetName,
        ["Base"] = _Base,
        ["FuncProxy"] = _FuncProxy,
        ["Property"] = _Property,
        ["ScriptType"] = _ScriptType,
        ["New"] = _New,
    }

    IGAS:NewWidget(_WidgetInfo, true, true)
end