﻿-- Author      : Kurapica
-- Create Date : 7/16/2008 12:29

----------------------------------------------------------------------------------------------------------------------------------------
--- Model provide a rendering environment which is drawn into the backdrop of their frame, allowing you to display the contents of an .m2 file and set facing, scale, light and fog information, or run motions associated
-- <br><br>inherit <a href=".\Frame.html">Frame</a> For all methods, properties and scriptTypes
-- @name Model
-- @class table
-- @field FogColor the model's current fog color
-- @field FogFar the far clipping distance for the model's fog
-- @field FogNear the near clipping distance for the model's fog
-- @field ModelScale the scale factor determining the size at which the 3D model appears
-- @field Model the model file to be displayed
-- @field Position the position of the 3D model within the frame
-- @field Light properties of the light sources used when rendering the model
----------------------------------------------------------------------------------------------------------------------------------------
do
	-- Check Version
	local version = 2
	if not IGAS:NewAddon("IGAS.GUI.Model", version) then
		return
	end

    local CreateFrame = CreateFrame

	local _WidgetName = "Model"
    local _Base = "Frame"

	-- ScriptType
	local _ScriptType = {
		------------------------------------
		--- ScriptType, Run when the model's animation finishes
		-- @name Model:OnAnimFinished
		-- @class function
		-- @usage function Model:OnAnimFinished()<br>
		--    -- do someting<br>
		-- end
		------------------------------------
		["OnAnimFinished"] = true,

		------------------------------------
		--- ScriptType, Run when a model changes or animates
		-- @name Model:OnUpdateModel
		-- @class function
		-- @usage function Model:OnUpdateModel()<br>
		--    -- do someting<br>
		-- end
		------------------------------------
		["OnUpdateModel"] = true,
	}

	-- FuncProxy
	local _FuncProxy = {
		------------------------------------
		--- Advances to the model's next animation frame
		-- @name Model:AdvanceTime
		-- @class function
		-- @usage Model:AdvanceTime()
		------------------------------------

		-- AdvanceTime
		["AdvanceTime"] = function(self, ...)
			return self.__UI:AdvanceTime(...)
		end,

		------------------------------------
		--- Removes all fogging effects currently active in rendering.
		-- @name Model:ClearFog
		-- @class function
		-- @usage Model:ClearFog()
		------------------------------------

		-- ClearFog
		["ClearFog"] = function(self, ...)
			return self.__UI:ClearFog(...)
		end,

		------------------------------------
		--- Removes all geometry from the Model (i.e. makes it empty)
		-- @name Model:ClearModel
		-- @class function
		-- @usage Model:ClearModel()
		------------------------------------

		-- ClearModel
		["ClearModel"] = function(self, ...)
			return self.__UI:ClearModel(...)
		end,

		------------------------------------
		--- Returns the direction the model is facing.
		-- @name Model:GetFacing
		-- @class function
		-- @return facing - Current rotation angle of the model (in radians)
		-- @usage Model:GetFacing()
		------------------------------------

		-- GetFacing
		["GetFacing"] = function(self, ...)
			return self.__UI:GetFacing(...)
		end,

		------------------------------------
		--- Returns the model's current fog color. Does not indicate whether fog display is enabled.
		-- @name Model:GetFogColor
		-- @class function
		-- @return red - Red component of the color (0.0 - 1.0)
		-- @return green - Green component of the color (0.0 - 1.0)
		-- @return blue - Blue component of the color (0.0 - 1.0)
		-- @usage Model:GetFogColor()
		------------------------------------

		-- GetFogColor
		["GetFogColor"] = function(self, ...)
			return self.__UI:GetFogColor(...)
		end,

		------------------------------------
		--- Returns the far clipping distance for the model's fog.. This determines how far from the camera the fog ends.
		-- @name Model:GetFogFar
		-- @class function
		-- @return distance - The distance to the fog far clipping plane
		-- @usage Model:GetFogFar()
		------------------------------------

		-- GetFogFar
		["GetFogFar"] = function(self, ...)
			return self.__UIGetFogFar(...)
		end,

		------------------------------------
		--- Returns the near clipping distance for the model's fog.. This determines how close to the camera the fog begins.
		-- @name Model:GetFogNear
		-- @class function
		-- @return distance - The distance to the fog near clipping plane
		-- @usage Model:GetFogNear()
		------------------------------------

		-- GetFogNear
		["GetFogNear"] = function(self, ...)
			return self.__UI:GetFogNear(...)
		end,

		------------------------------------
		--- Returns properties of the light sources used when rendering the model
		-- @name Model:GetLight
		-- @class function
		-- @return enabled - 1 if lighting is enabled; otherwise nil
		-- @return omni - 1 if omnidirectional lighting is enabled; otherwise 0
		-- @return dirX - Coordinate of the directional light in the axis perpendicular to the screen (negative values place the light in front of the model, positive values behind)
		-- @return dirY - Coordinate of the directional light in the horizontal axis (negative values place the light to the left of the model, positive values to the right)
		-- @return dirZ - Coordinate of the directional light in the vertical axis (negative values place the light below the model, positive values above
		-- @return ambIntensity - Intensity of the ambient light (0.0 - 1.0)
		-- @return ambR - Red component of the ambient light color (0.0 - 1.0); omitted if ambIntensity is 0
		-- @return ambG - Green component of the ambient light color (0.0 - 1.0); omitted if ambIntensity is 0
		-- @return ambB - Blue component of the ambient light color (0.0 - 1.0); omitted if ambIntensity is 0
		-- @return dirIntensity - Intensity of the directional light (0.0 - 1.0)
		-- @return dirR - Red component of the directional light color (0.0 - 1.0); omitted if dirIntensity is 0
		-- @return dirG - Green component of the directional light color (0.0 - 1.0); omitted if dirIntensity is 0
		-- @return dirB - Blue component of the directional light color (0.0 - 1.0); omitted if dirIntensity is 0
		-- @usage Model:GetLight()
		------------------------------------

		-- GetLight
	    ["GetLight"] = function(self, ...)
			return self.__UI:GetLight(...)
		end,

		------------------------------------
		--- Returns the model file currently displayed. May instead return a reference to the Model object itself if a filename is not available.
		-- @name Model:GetModel
		-- @class function
		-- @return filename - Path to the model file currently displayed
		-- @usage Model:GetModel()
		------------------------------------

		-- GetModel
		["GetModel"] = function(self, ...)
			return self.__UI:GetModel(...)
		end,

		------------------------------------
		--- Returns the scale factor determining the size at which the 3D model appears
		-- @name Model:GetModelScale
		-- @class function
		-- @return scale - Scale factor determining the size at which the 3D model appears
		-- @usage Model:GetModelScale()
		------------------------------------

		-- GetModelScale
		["GetModelScale"] = function(self, ...)
			return self.__UI:GetModelScale(...)
		end,

		------------------------------------
		--- Returns the position of the 3D model within the frame
		-- @name Model:GetPosition
		-- @class function
		-- @return x - Position of the model on the axis perpendicular to the plane of the screen (positive values make the model appear closer to the viewer; negative values place it further away)
		-- @return y - Position of the model on the horizontal axis (positive values place the model to the right of its default position; negative values place it to the left)
		-- @return z - Position of the model on the vertical axis (positive values place the model above its default position; negative values place it below)
		-- @usage Model:GetPosition()
		------------------------------------

		-- GetPosition
		["GetPosition"] = function(self, ...)
			return self.__UI:GetPosition(...)
		end,

		------------------------------------
		--- Sets the icon texture used by the model. Only affects models that use icons (e.g. the model producing the default UI's animation which appears when an item goes into a bag).
		-- @name Model:ReplaceIconTexture
		-- @class function
		-- @param filename Path to an icon texture for use in the model
		-- @usage Model:ReplaceIconTexture("texutre")
		------------------------------------

		-- ReplaceIconTexture
		["ReplaceIconTexture"] = function(self, texture, ...)
			return self.__UI:ReplaceIconTexture(IGAS:GetUI(texture), ...)
		end,

		------------------------------------
		--- Sets the view angle on the model to a pre-defined camera location.Camera view angles are defined within the model files and not otherwise available to the scripting system. Some camera indices are standard across most models:<br>
		-- 0 - Non-movable camera, focused on the unit's face (if applicable); used by the game engine when rendering portrait textures<br>
		-- 1 - Movable camera, showing the entire body of the unit<br>
		-- 2 or higher - Movable camera in default position
		-- @name Model:SetCamera
		-- @class function
		-- @param index Index of a camera view defined by the model file
		-- @usage Model:SetCamera(1)
		------------------------------------

		-- SetCamera
		["SetCamera"] = function(self, ...)
			return self.__UI:SetCamera(...)
		end,

		------------------------------------
		--- Sets the model's current rotation. The 3D model displayed by the model object can be rotated about its vertical axis. <br>
		-- For example, if the model faces towards the viewer when its facing is set to 0, setting facing to math.pi faces it away from the viewer.
		-- @name Model:SetFacing
		-- @class function
		-- @param facing Rotation angle for the model (in radians)
		-- @usage Model:SetFacing(1)
		------------------------------------

		-- SetFacing
		["SetFacing"] = function(self, ...)
			return self.__UI:SetFacing(...)
		end,

		------------------------------------
		--- Sets the model's fog color, enabling fog display if disabled
		-- @name Model:SetFogColor
		-- @class function
		-- @param red Red component of the color (0.0 - 1.0)
		-- @param green Green component of the color (0.0 - 1.0)
		-- @param blue Blue component of the color (0.0 - 1.0)
		-- @usage Model:SetFogColor(1, 1, 1)
		------------------------------------

		-- SetFogColor
		["SetFogColor"] = function(self, ...)
			return self.__UI:SetFogColor(...)
		end,

		------------------------------------
		--- Sets the far clipping distance for the model's fog.. This sets how far from the camera the fog ends.
		-- @name Model:SetFogFar
		-- @class function
		-- @param distance The distance to the fog far clipping plane
		-- @usage Model:SetFogFar(12)
		------------------------------------

		-- SetFogFar
		["SetFogFar"] = function(self, ...)
			return self.__UI:SetFogFar(...)
		end,

		------------------------------------
		--- Sets the near clipping distance for the model's fog.. This sets how close to the camera the fog begins.
		-- @name Model:SetFogNear
		-- @class function
		-- @param distance The distance to the fog near clipping plane
		-- @usage Model:SetFogNear(12)
		------------------------------------

		-- SetFogNear
		["SetFogNear"] = function(self, ...)
			return self.__UI:SetFogNear(...)
		end,

		------------------------------------
		--- Sets the model's glow amount
		-- @name Model:SetGlow
		-- @class function
		-- @param amount Glow amount for the model
		-- @usage Model:SetGlow(12)
		------------------------------------

		-- SetGlow
		["SetGlow"] = function(self, ...)
			return self.__UI:SetGlow(...)
		end,

		------------------------------------
		--- Sets properties of the light sources used when rendering the model
		-- @name Model:SetLight
		-- @class function
		-- @param enabled 1 if lighting is enabled; otherwise nil (1nil)
		-- @param omni 1 if omnidirectional lighting is enabled; otherwise 0
		-- @param dirX Coordinate of the directional light in the axis perpendicular to the screen (negative values place the light in front of the model, positive values behind)
		-- @param dirY Coordinate of the directional light in the horizontal axis (negative values place the light to the left of the model, positive values to the right)
		-- @param dirZ Coordinate of the directional light in the vertical axis (negative values place the light below the model, positive values above
		-- @param ambIntensity Intensity of the ambient light (0.0 - 1.0)
		-- @param ambR Optional,Red component of the ambient light color (0.0 - 1.0); omitted if ambIntensity is 0
		-- @param ambG Optional,Green component of the ambient light color (0.0 - 1.0); omitted if ambIntensity is 0
		-- @param ambB Optional,Blue component of the ambient light color (0.0 - 1.0); omitted if ambIntensity is 0
		-- @param dirIntensity Optional,Intensity of the directional light (0.0 - 1.0)
		-- @param dirR Optional,Red component of the directional light color (0.0 - 1.0); omitted if dirIntensity is 0
		-- @param dirG Optional,Green component of the directional light color (0.0 - 1.0); omitted if dirIntensity is 0
		-- @param dirB Optional,Blue component of the directional light color (0.0 - 1.0); omitted if dirIntensity is 0
		-- @usage Model:SetLight(1, 1, 12, 13, 14, 1)
		------------------------------------

		-- SetLight
	    ["SetLight"] = function(self, ...)
			return self.__UI:SetLight(...)
		end,

		------------------------------------
		--- Sets the model file to be displayed
		-- @name Model:SetModel
		-- @class function
		-- @param filename Path to the model file to be displayed
		-- @usage Model:SetModel("filename")
		------------------------------------

		-- SetModel
		["SetModel"] = function(self, ...)
			return self.__UI:SetModel(...)
		end,

		------------------------------------
		--- Sets the scale factor determining the size at which the 3D model appears
		-- @name Model:SetModelScale
		-- @class function
		-- @param scale Scale factor determining the size at which the 3D model appears
		-- @usage Model:SetModelScale(1.5)
		------------------------------------

		-- SetModelScale
		["SetModelScale"] = function(self, ...)
			return self.__UI:SetModelScale(...)
		end,

		------------------------------------
		--- Set the position of the 3D model within the frame
		-- @name Model:SetPosition
		-- @class function
		-- @param x Position of the model on the axis perpendicular to the plane of the screen (positive values make the model appear closer to the viewer; negative values place it further away)
		-- @param y Position of the model on the horizontal axis (positive values place the model to the right of its default position; negative values place it to the left)
		-- @param z Position of the model on the vertical axis (positive values place the model above its default position; negative values place it below)
		-- @usage Model:SetPosition(12,12,14)
		------------------------------------

		-- SetPosition
		["SetPosition"] = function(self, ...)
			return self.__UI:SetPosition(...)
		end,

		------------------------------------
		--- Sets the animation sequence to be used by the model. The number of available sequences and behavior of each are defined within the model files and not available to the scripting system.
		-- @name Model:SetSequence
		-- @class function
		-- @param sequence Index of an animation sequence defined by the model file
		-- @usage Model:SetSequence(1)
		------------------------------------

		-- SetSequence
		["SetSequence"] = function(self, ...)
			return self.__UI:SetSequence(...)
		end,

		------------------------------------
		--- Sets the animation sequence and time index to be used by the model. The number of available sequences and behavior of each are defined within the model files and not available to the scripting system.
		-- @name Model:SetSequenceTime
		-- @class function
		-- @param sequence Index of an animation sequence defined by the model file
		-- @param time Time index within the sequence
		-- @usage Model:SetSequenceTime(1, 10)
		------------------------------------

		-- SetSequenceTime
		["SetSequenceTime"] = function(self, ...)
			return self.__UI:SetSequenceTime(...)
		end,
    }

	-- Property
	local _Property = {
    	-- FogColor
		["FogColor"] = {
			["Set"] = function(self, colorTable)
				self.__UI:SetFogColor(colorTable.red, colorTable.green, colorTable.blue, colorTable.alpha)
			end,

			["Get"] = function(self)
				local colorTable = {}
				colorTable.red, colorTable.green, colorTable.blue, colorTable.alpha = self.__UI:GetFogColor()
				return colorTable
			end,

			["Type"] = "Color",
		},

    	-- FogFar
		["FogFar"] = {
			["Set"] = function(self, value)
				self.__UI:SetFogFar(value)
			end,

			["Get"] = function(self)
				return self.__UI:GetFogFar()
			end,

			["Type"] = "number",
		},

    	-- FogNear
		["FogNear"] = {
			["Set"] = function(self, value)
				self.__UI:SetFogNear(value)
			end,

			["Get"] = function(self)
				return self.__UI:GetFogNear()
			end,

			["Type"] = "number",
		},

    	-- ModelScale
		["ModelScale"] = {
			["Set"] = function(self, scale)
				self.__UI:SetModelScale(scale)
			end,

			["Get"] = function(self)
				return self.__UI:GetModelScale()
			end,

			["Type"] = "number",
		},

    	-- Model
		["Model"] = {
			["Set"] = function(self, file)
				if file and type(file) == "string" and file ~= "" then
					self.__UI:SetModel(file)
				else
					self.__UI:ClearModel()
				end
			end,

			["Get"] = function(self)
				return self.__UI:GetModel()
			end,

			["Type"] = "string",
		},

    	-- Position
		["Position"] = {
			["Set"] = function(self, table)
				self.__UI:SetPosition(table.x, table.y, table.z)
			end,

			["Get"] = function(self)
				local table = {}
				table.x, table.y, table.z = self.__UI:GetPosition()
				return table
			end,

			["Type"] = "Position",
		},

    	-- Light
		["Light"] = {
			["Set"] = function(self, set)
				self.__UI:SetLight(set.enabled, set.omni, set.dirX, set.dirY, set.dirZ, set.ambIntensity, set.ambR, set.ambG, set.ambB, set.dirIntensity, set.dirR, set.dirG, set.dirB)
			end,

			["Get"] = function(self)
				local set = {}
				set.enabled, set.omni, set.dirX, set.dirY, set.dirZ, set.ambIntensity, set.ambR, set.ambG, set.ambB, set.dirIntensity, set.dirR, set.dirG, set.dirB = self.__UI:GetLight()
				return set
			end,

			["Type"] = "Light",
		},
   }

    -- New
    local _New = function(_parent,inheritFrame)
        -- New Frame
		return CreateFrame(_WidgetName,nil,IGAS:GetUI(_parent),inheritFrame)
    end

    local _WidgetInfo = {
        ["WidgetName"] = _WidgetName,
        ["Base"] = _Base,
        ["FuncProxy"] = _FuncProxy,
        ["Property"] = _Property,
        ["ScriptType"] = _ScriptType,
        ["New"] = _New,
    }

    IGAS:NewWidget(_WidgetInfo, true, true)
end