﻿-- Author      : Kurapica
-- Create Date : 6/12/2008 1:14:03 AM

----------------------------------------------------------------------------------------------------------------------------------------
--- Frame is in many ways the most fundamental widget object. Other types of widget derivatives such as FontStrings, Textures and Animations can only be created attached to a Frame or other derivative of a Frame. <br>
-- Frames provide the basis for interaction with the user, and registering and responding to game events.
-- <br><br>inherit <a href=".\Region.html">Region</a> For all methods, properties and scriptTypes
-- @name Frame
-- @class table
-- @field KeyboardEnabled whether keyboard interactivity is enabled for the frame
-- @field MouseEnabled whether mouse interactivity is enabled for the frame
-- @field MouseWheelEnabled whether mouse wheel interactivity is enabled for the frame
-- @field Backdrop the backdrop graphic for the frame
-- @field BackdropBorderColor the shading color for the frame's border graphic
-- @field BackdropColor the shading color for the frame's background graphic
-- @field ClampedToScreen whether the frame's boundaries are limited to those of the screen
-- @field ClampRectInsets offsets from the frame's edges used when limiting user movement or resizing of the frame
-- @field FrameLevel the level at which the frame is layered relative to others in its strata
-- @field FrameStrata the general layering strata of the frame
-- @field HitRectInsets the insets from the frame's edges which determine its mouse-interactable area
-- @field ID a numeric identifier for the frame
-- @field MaxResize the maximum size of the frame for user resizing
-- @field MinResize the minimum size of the frame for user resizing
-- @field Scale the frame's scale factor
-- @field Movable whether the frame can be moved by the user
-- @field Resizable whether the frame can be resized by the user
-- @field Toplevel whether the frame should automatically come to the front when clicked
-- @field Depth the 3D depth of the frame (for stereoscopic 3D setups)
-- @field DepthIgnored whether the frame's depth property is ignored (for stereoscopic 3D setups)
----------------------------------------------------------------------------------------------------------------------------------------
do
	-- Check Version
	local version = 5
	if not IGAS:NewAddon("IGAS.GUI.Frame", version) then
		return
	end

    local CreateFrame = CreateFrame
	local pcall = pcall
	local tinsert = tinsert
	local tremove = tremove

	local _WidgetName = "Frame"
    local _Base = "Region"

	-- Event Checker
	_SystemEventChkr = _SystemEventChkr or CreateFrame("Frame")
	_SystemEventChkr:RegisterAllEvents()

	-- ScriptType
	local _ScriptType = {
		------------------------------------
		--- ScriptType, Run when a frame attribute is changed
		-- @name Frame:OnAttributeChanged
		-- @class function
		-- @param name Name of the changed attribute, always lower case
		-- @param value New value of the attribute
		-- @usage function Frame:OnAttributeChanged(name, value)<br>
		--    -- do someting<br>
		-- end
		------------------------------------
		["OnAttributeChanged"] = true,

		------------------------------------
		--- ScriptType, Run for each text character typed in the frame
		-- @name Frame:OnChar
		-- @class function
		-- @param text The text entered
		-- @usage function Frame:OnChar(text)<br>
		--    -- do someting<br>
		-- end
		------------------------------------
		["OnChar"] = true,

		------------------------------------
		--- ScriptType, Run when the frame is disabled
		-- @name Frame:OnDisable
		-- @class function
		-- @usage function Frame:OnDisable()<br>
		--    -- do someting<br>
		-- end
		------------------------------------
		["OnDisable"] = true,

		------------------------------------
		--- ScriptType, Run when the mouse is dragged starting in the frame
		-- @name Frame:OnDragStart
		-- @class function
		-- @param button Name of the mouse button responsible for the click action:Button4, Button5, LeftButton, MiddleButton, RightButton
		-- @usage function Frame:OnDragStart(button)<br>
		--    -- do someting<br>
		-- end
		------------------------------------
		["OnDragStart"] = true,

		------------------------------------
		--- ScriptType, Run when the mouse button is released after a drag started in the frame
		-- @name Frame:OnDragStop
		-- @class function
		-- @usage function Frame:OnDragStop()<br>
		--    -- do someting<br>
		-- end
		------------------------------------
		["OnDragStop"] = true,

		------------------------------------
		--- ScriptType, Run when the frame is enabled
		-- @name Frame:OnEnable
		-- @class function
		-- @usage function Frame:OnEnable()<br>
		--    -- do someting<br>
		-- end
		------------------------------------
		["OnEnable"] = true,

		------------------------------------
		--- ScriptType, Run when the mouse cursor enters the frame's interactive area
		-- @name Frame:OnEnter
		-- @class function
		-- @param motion True if the handler is being run due to actual mouse movement; false if the cursor entered the frame due to other circumstances (such as the frame being created underneath the cursor)
		-- @usage function Frame:OnEnter(motion)<br>
		--    -- do someting<br>
		-- end
		------------------------------------
		["OnEnter"] = true,

		------------------------------------
		--- ScriptType, Run whenever an event fires for which the frame is registered
		-- @name Frame:OnEvent
		-- @class function
		-- @param event the event's name
		-- @param ... the event's parameters
		-- @usage function Frame:OnEvent("event", ...)<br>
		--    -- do someting<br>
		-- end
		------------------------------------
		["OnEvent"] = true,

		------------------------------------
		--- ScriptType, Run when the frame's visbility changes to hidden
		-- @name Frame:OnHide
		-- @class function
		-- @usage function Frame:OnHide()<br>
		--    -- do someting<br>
		-- end
		------------------------------------
		["OnHide"] = true,

		------------------------------------
		--- ScriptType, Run when a keyboard key is pressed if the frame is keyboard enabled
		-- @name Frame:OnKeyDown
		-- @class function
		-- @param key Name of the key pressed
		-- @usage function Frame:OnKeyDown(key)<br>
		--    -- do someting<br>
		-- end
		------------------------------------
		["OnKeyDown"] = true,

		------------------------------------
		--- ScriptType, Run when a keyboard key is released if the frame is keyboard enabled
		-- @name Frame:OnKeyUp
		-- @class function
		-- @param key Name of the key pressed
		-- @usage function Frame:OnKeyUp(key)<br>
		--    -- do someting<br>
		-- end
		------------------------------------
		["OnKeyUp"] = true,

		------------------------------------
		--- ScriptType, Run when the mouse cursor leaves the frame's interactive area
		-- @name Frame:OnLeave
		-- @class function
		-- @param motion True if the handler is being run due to actual mouse movement; false if the cursor left the frame due to other circumstances (such as the frame being created underneath the cursor)
		-- @usage function Frame:OnLeave(motion)<br>
		--    -- do someting<br>
		-- end
		------------------------------------
		["OnLeave"] = true,

		------------------------------------
		--- ScriptType, Run when the frame is created
		-- @name Frame:OnLoad
		-- @class function
		-- @usage function Frame:OnLoad()<br>
		--    -- do someting<br>
		-- end
		------------------------------------
		["OnLoad"] = true,

		------------------------------------
		--- ScriptType, Run when a mouse button is pressed while the cursor is over the frame
		-- @name Frame:OnMouseDown
		-- @class function
		-- @param button Name of the mouse button responsible for the click action:Button4, Button5, LeftButton, MiddleButton, RightButton
		-- @usage function Frame:OnMouseDown(button)<br>
		--    -- do someting<br>
		-- end
		------------------------------------
		["OnMouseDown"] = true,

		------------------------------------
		--- ScriptType, Run when the mouse button is released following a mouse down action in the frame
		-- @name Frame:OnMouseUp
		-- @class function
		-- @param button Name of the mouse button responsible for the click action:Button4, Button5, LeftButton, MiddleButton, RightButton
		-- @usage function Frame:OnMouseUp(button)<br>
		--    -- do someting<br>
		-- end
		------------------------------------
		["OnMouseUp"] = true,

		------------------------------------
		--- ScriptType, Run when the frame receives a mouse wheel scrolling action
		-- @name Frame:OnMouseWheel
		-- @class function
		-- @param delta 1 for a scroll-up action, -1 for a scroll-down action
		-- @usage function Frame:OnMouseWheel(delta)<br>
		--    -- do someting<br>
		-- end
		------------------------------------
		["OnMouseWheel"] = true,

		------------------------------------
		--- ScriptType, Run when the mouse button is released after dragging into the frame
		-- @name Frame:OnReceiveDrag
		-- @class function
		-- @usage function Frame:OnReceiveDrag()<br>
		--    -- do someting<br>
		-- end
		------------------------------------
		["OnReceiveDrag"] = true,

		------------------------------------
		--- ScriptType, Run when the frame becomes visible
		-- @name Frame:OnShow
		-- @class function
		-- @usage function Frame:OnShow()<br>
		--    -- do someting<br>
		-- end
		------------------------------------
		["OnShow"] = true,

		------------------------------------
		--- ScriptType, Run when a frame's size changes
		-- @name Frame:OnSizeChanged
		-- @class function
		-- @param width New width of the frame
		-- @param height New height of the frame
		-- @usage function Frame:OnSizeChanged(width, height)<br>
		--    -- do someting<br>
		-- end
		------------------------------------
		["OnSizeChanged"] = true,

		------------------------------------
		--- ScriptType, Run each time the screen is drawn by the game engine
		-- @name Frame:OnUpdate
		-- @class function
		-- @param elapsed Number of seconds since the OnUpdate handlers were last run (likely a fraction of a second)
		-- @usage function Frame:OnUpdate(elapsed)<br>
		--    -- do someting<br>
		-- end
		------------------------------------
		["OnUpdate"] = true,
	}

	-- FuncProxy
	local _FuncProxy = {
		-- Dispose, release resource
        ["Dispose"] = function(self)
            self:UnregisterAllEvents()

            -- Call super's dispose
			self:CallMethod(_Base, "Dispose")
        end,

		------------------------------------
		--- Create and return a new FontString as a child of this Frame
		-- @name Frame:CreateFontString
		-- @class function
		-- @param name name for the new font string
		-- @param layer Optional,Graphic layer on which to create the font string; defaults to ARTWORK if not specified
		-- @param inherits Optional,Name of a template from which the new front string should inherit
		-- @return Reference to the new FontString object
		-- @usage Frame:CreateFontString("Label1")
		------------------------------------

		-- CreateFontString
		["CreateFontString"] = function(self, name, layer, inheritsFrom)
			return IGAS:NewFrame("FontString", name, self, layer, inheritsFrom)
        end,

		------------------------------------
		--- Create and return a new Texture as a child of this Frame.
		-- @name Frame:CreateTexture
		-- @class function
		-- @param name name for the new texture
		-- @param layer Graphic layer on which to create the texture; defaults to ARTWORK if not specified
		-- @param inherits Name of a template from which the new texture should inherit
		-- @return Reference to the new Texture object
		-- @usage Frame:CreateTexture("Icon1")
		------------------------------------

		-- CreateTexture
	    ["CreateTexture"] = function(self, name, layer, inheritsFrom)
			return IGAS:NewFrame("Texture", name, self, layer, inheritsFrom)
        end,

		------------------------------------
		--- Create a title region for the frame if it does not have one.
		-- @name Frame:CreateTitleRegion
		-- @class function
		-- @return Reference to the new Region object
		-- @usage Frame:CreateTitleRegion()
		------------------------------------

		-- CreateTitleRegion
        ["CreateTitleRegion"] = function(self, ...)
            return IGAS:GetWrapper(self.__UI:CreateTitleRegion(...))
        end,

		------------------------------------
		--- Disable rendering of "regions" (fontstrings, textures) in the specified draw layer.
		-- @name Frame:DisableDrawLayer
		-- @class function
		-- @param layer Name of a graphics layer
		-- @usage Frame:DisableDrawLayer("ARTWORK")
		------------------------------------

		-- DisableDrawLayer
        ["DisableDrawLayer"] = function(self, ...)
            return self.__UI:DisableDrawLayer(...)
        end,

		------------------------------------
		--- Enable rendering of "regions" (fontstrings, textures) in the specified draw layer.
		-- @name Frame:EnableDrawLayer
		-- @class function
		-- @param layer Name of a graphics layer
		-- @usage Frame:EnableDrawLayer("ARTWORK")
		------------------------------------

		-- EnableDrawLayer
        ["EnableDrawLayer"] = function(self, ...)
            return self.__UI:EnableDrawLayer(...)
        end,

		------------------------------------
		--- Set whether this frame will get keyboard input.
		-- @name Frame:EnableKeyboard
		-- @class function
		-- @param enable True to enable keyboard interactivity; false to disable
		-- @usage Frame:EnableKeyboard(true)
		------------------------------------

		-- EnableKeyboard
        ["EnableKeyboard"] = function(self, enabled)
			if not self.InDesignMode then
				self.__UI:EnableKeyboard(enabled)
			else
				self.__KeyboardEnabled = (enabled and true) or false
			end
        end,

		------------------------------------
		--- Set whether this frame will get mouse input.
		-- @name Frame:EnableMouse
		-- @class function
		-- @param enable True to enable mouse interactivity; false to disable
		-- @usage Frame:EnableMouse(true)
		------------------------------------

		-- EnableMouse
        ["EnableMouse"] = function(self, enabled)
			if not self.InDesignMode then
				self.__UI:EnableMouse(enabled)
			else
				self.__MouseEnabled = (enabled and true) or false
			end
        end,

		------------------------------------
		--- Set whether this frame will get mouse wheel notifications.
		-- @name Frame:EnableMouseWheel
		-- @class function
		-- @param enable True to enable mouse wheel interactivity; false to disable
		-- @usage Frame:EnableMouseWheel(true)
		------------------------------------

		-- EnableMouseWheel
        ["EnableMouseWheel"] = function(self, ...)
            return self.__UI:EnableMouseWheel(...)
        end,

		------------------------------------
		--- Returns the first existing attribute of (prefix..name..suffix), ("*"..name..suffix), (prefix..name.."*"), ("*"..name.."*"), (name).
		-- @name Frame:GetAttribute
		-- @class function
		-- @param prefix modifier prefix string: a concatenation of alt-, ctrl-, and shift-, whichever apply, in that order.
		-- @param name Attribute name. If passed as the only argument, this may be a fully qualified value; otherwise, it is the unmodified name of the attribute being retrieved.
		-- @param suffix button suffix string: either a numeric string or a dash followed by an arbitrary button name.
		-- @return Value of the named attribute
		-- @usage Frame:GetAttribute("Order")
		------------------------------------

		-- GetAttribute
        ["GetAttribute"] = function(self, ...)
            return self.__UI:GetAttribute(...)
        end,

		------------------------------------
		--- Creates and returns a backdrop table suitable for use in SetBackdrop
		-- @name Frame:GetBackdrop
		-- @class function
		-- @return A table containing the backdrop settings, or nil if the frame has no backdrop
		-- @usage Frame:GetBackdrop()
		------------------------------------

		-- GetBackdrop
        ["GetBackdrop"] = function(self, ...)
            return self.__UI:GetBackdrop(...)
        end,

		------------------------------------
		--- Gets the frame's backdrop border color (r, g, b, a)
		-- @name Frame:GetBackdropBorderColor
		-- @class function
		-- @return red Red component of the color (0.0 - 1.0)
		-- @return green Green component of the color (0.0 - 1.0)
		-- @return blue Blue component of the color (0.0 - 1.0)
		-- @return alpha Alpha (opacity) for the graphic (0.0 = fully transparent, 1.0 = fully opaque)
		-- @usage Frame:GetBackdropBorderColor()
		------------------------------------

		-- GetBackdropBorderColor
        ["GetBackdropBorderColor"] = function(self, ...)
            return self.__UI:GetBackdropBorderColor(...)
        end,

		------------------------------------
		--- Gets the frame's backdrop color (r, g, b, a)
		-- @name Frame:GetBackdropColor
		-- @class function
		-- @return red Red component of the color (0.0 - 1.0)
		-- @return green Green component of the color (0.0 - 1.0)
		-- @return blue Blue component of the color (0.0 - 1.0)
		-- @return alpha Alpha (opacity) for the graphic (0.0 = fully transparent, 1.0 = fully opaque)
		-- @usage Frame:GetBackdropColor()
		------------------------------------

		-- GetBackdropColor
        ["GetBackdropColor"] = function(self, ...)
            return self.__UI:GetBackdropColor(...)
        end,

		------------------------------------
		--- Get the list of "children" (frames and things derived from frames) of this frame.
		-- @name Frame:GetChildren
		-- @class function
		-- @return a list of child-frames
		-- @usage Frame:GetChildren()
		------------------------------------

		-- GetChildren
        ["GetChildren"] = function(self, ...)
            return self:GetChilds()
        end,

		------------------------------------
		--- Gets the modifiers to the frame's rectangle used for clamping the frame to screen.
		-- @name Frame:GetClampRectInsets
		-- @class function
		-- @return left Offset from the left edge of the frame to the left edge of its clamping area (in pixels)
		-- @return right Offset from the right edge of the frame's clamping area to the right edge of the frame (in pixels)
		-- @return top Offset from the top edge of the frame's clamping area to the top edge of the frame (in pixels)
		-- @return bottom Offset from the bottom edge of the frame to the bottom edge of its clamping area (in pixels)
		-- @usage Frame:GetClampRectInsets()
		------------------------------------

		-- GetClampRectInsets
        ["GetClampRectInsets"] = function(self, ...)
            return self.__UI:GetClampRectInsets(...)
        end,

		------------------------------------
		--- Returns the 3D depth of the frame (for stereoscopic 3D setups)
		-- @name Frame:GetDepth
		-- @class function
		-- @return Apparent 3D depth of this frame relative to that of its parent frame
		-- @usage Frame:GetDepth()
		------------------------------------

		-- GetDepth
        ["GetDepth"] = function(self, ...)
            return self.__UI:GetDepth(...)
        end,

		------------------------------------
		--- Returns the effective alpha of a frame.
		-- @name Frame:GetEffectiveAlpha
		-- @class function
		-- @return Effective alpha (opacity) of the region (0.0 = fully transparent, 1.0 = fully opaque)
		-- @usage Frame:GetEffectiveAlpha()
		------------------------------------

		-- GetEffectiveAlpha
        ["GetEffectiveAlpha"] = function(self, ...)
            return self.__UI:GetEffectiveAlpha(...)
        end,

		------------------------------------
		--- Returns the overall 3D depth of the frame (for stereoscopic 3D configurations).
		-- @name Frame:GetEffectiveDepth
		-- @class function
		-- @return Apparent 3D depth of this frame relative to the screen
		-- @usage Frame:GetEffectiveDepth()
		------------------------------------

		-- GetEffectiveDepth
        ["GetEffectiveDepth"] = function(self, ...)
            return self.__UI:GetEffectiveDepth(...)
        end,

		------------------------------------
		--- Get the scale factor of this object relative to the root window.
		-- @name Frame:GetEffectiveScale
		-- @class function
		-- @return Scale factor for the frame relative to its parent
		-- @usage Frame:GetEffectiveScale()
		------------------------------------

		-- GetEffectiveScale
        ["GetEffectiveScale"] = function(self, ...)
            return self.__UI:GetEffectiveScale(...)
        end,

		------------------------------------
		--- Get the level of this frame.
		-- @name Frame:GetFrameLevel
		-- @class function
		-- @return the level of this frame
		-- @usage Frame:GetFrameLevel()
		------------------------------------

		-- GetFrameLevel
        ["GetFrameLevel"] = function(self, ...)
            return self.__UI:GetFrameLevel(...)
        end,

		------------------------------------
		--- Get the strata of this frame.
		-- @name Frame:GetFrameStrata
		-- @class function
		-- @return the strata of this frame
		-- @usage Frame:GetFrameStrata()
		------------------------------------

		-- GetFrameStrata
        ["GetFrameStrata"] = function(self, ...)
            return self.__UI:GetFrameStrata(...)
        end,

		------------------------------------
		--- Get the type of this frame.
		-- @name Frame:GetFrameType
		-- @class function
		-- @return the name of the frame's type
		-- @usage Frame:GetFrameType()
		------------------------------------

		-- GetFrameType
        ["GetFrameType"] = function(self, ...)
            return self:GetObjectType().WidgetName
        end,

		------------------------------------
		--- Gets the frame's hit rectangle inset distances (l, r, t, b)
		-- @name Frame:GetHitRectInsets
		-- @class function
		-- @return left Distance from the left edge of the frame to the left edge of its mouse-interactive area (in pixels)
		-- @return right Distance from the right edge of the frame to the right edge of its mouse-interactive area (in pixels)
		-- @return top Distance from the top edge of the frame to the top edge of its mouse-interactive area (in pixels)
		-- @return bottom Distance from the bottom edge of the frame to the bottom edge of its mouse-interactive area (in pixels)
		-- @usage Frame:GetHitRectInsets()
		------------------------------------

		-- GetHitRectInsets
        ["GetHitRectInsets"] = function(self, ...)
            return self.__UI:GetHitRectInsets(...)
        end,

		------------------------------------
		--- Get the ID of this frame.
		-- @name Frame:GetID
		-- @class function
		-- @return the ID of this frame
		-- @usage Frame:GetID()
		------------------------------------

		-- GetID
        ["GetID"] = function(self, ...)
            return self.__UI:GetID(...)
        end,

		------------------------------------
		--- Gets the frame's maximum allowed resize bounds (w, h)
		-- @name Frame:GetMaxResize
		-- @class function
		-- @return maxWidth Maximum width of the frame (in pixels), or 0  for no limit
		-- @return maxHeight Maximum height of the frame (in pixels), or 0 for no limit
		-- @usage Frame:GetMaxResize()
		------------------------------------

		-- GetMaxResize
        ["GetMaxResize"] = function(self, ...)
            return self.__UI:GetMaxResize(...)
        end,

		------------------------------------
		--- Gets the frame's minimum allowed resize bounds (w, h)
		-- @name Frame:GetMinResize
		-- @class function
		-- @return minWidth Minimum width of the frame (in pixels), or 0  for no limit
		-- @return minHeight Minimum height of the frame (in pixels), or 0 for no limit
		-- @usage Frame:GetMinResize()
		------------------------------------

		-- GetMinResize
        ["GetMinResize"] = function(self, ...)
            return self.__UI:GetMinResize(...)
        end,

		------------------------------------
		--- Get the number of "children" (frames and things derived from frames) this frame has.
		-- @name Frame:GetNumChildren
		-- @class function
		-- @return the number of the childs
		-- @usage Frame:GetNumChildren()
		------------------------------------

		-- GetNumChildren
        ["GetNumChildren"] = function(self, ...)
			local cnt = 0
			for i in pairs(self.__Childs) do
				cnt = cnt + 1
			end
			return cnt
            --return self.__UI:GetNumChildren(...)
        end,

		------------------------------------
		--- Return the number of "regions" (fontstrings, textures) belonging to this frame.
		-- @name Frame:GetNumRegions
		-- @class function
		-- @return the number of the regions
		-- @usage Frame:GetNumRegions()
		------------------------------------

		-- GetNumRegions
        ["GetNumRegions"] = function(self, ...)
            return self.__UI:GetNumRegions(...)
        end,

		------------------------------------
		--- Returns a list of non-Frame child regions belonging to the frame
		-- @name Frame:GetRegions
		-- @class function
		-- @return ... A list of each non-Frame child region (FontString or Texture) belonging to the frame
		-- @usage Frame:GetRegions()
		------------------------------------

		-- GetRegions
	    --["GetRegions"] = function(self, ...)
        --    return self.__UI:GetRegions(...)
        --end,

		------------------------------------
		--- Get the scale factor of this object relative to its parent.
		-- @name Frame:GetScale
		-- @class function
		-- @return Scale factor for the frame relative to its parent
		-- @usage Frame:GetScale()
		------------------------------------

		-- GetScale
        ["GetScale"] = function(self, ...)
            return self.__UI:GetScale(...)
        end,

		------------------------------------
		--- Return the frame's title region
		-- @name Frame:GetTitleRegion
		-- @class function
		-- @return the frame's title region
		-- @usage Frame:GetTitleRegion()
		------------------------------------

		-- GetTitleRegion
        ["GetTitleRegion"] = function(self, ...)
            return IGAS:GetWrapper(self.__UI:GetTitleRegion(...))
        end,

	    -- Frame:HookScript("handler", function) - Hook a secure frame script. - Since 2.1.
        ["HookScript"] = function(self, ...)
			error("Please use StackScript instead of HookScript", 2)
            --return self.__UI:HookScript(...)
        end,

		------------------------------------
		--- Sets whether the frame's depth property is ignored (for stereoscopic 3D setups).
		-- @name Frame:IgnoreDepth
		-- @class function
		-- @param enable True to ignore the frame's depth property; false to disable
		-- @usage Frame:IgnoreDepth(false)
		------------------------------------

		-- IgnoreDepth
        ["IgnoreDepth"] = function(self, ...)
            return self.__UI:IgnoreDepth(...)
        end,

		------------------------------------
		--- Gets whether the frame is prohibited from being dragged off screen
		-- @name Frame:IsClampedToScreen
		-- @class function
		-- @return 1 if the frame's boundaries are limited to those of the screen when user moving/resizing; otherwise nil
		-- @usage Frame:IsClampedToScreen()
		------------------------------------

		-- IsClampedToScreen
        ["IsClampedToScreen"] = function(self, ...)
            return self.__UI:IsClampedToScreen(...)
        end,

		------------------------------------
		--- Returns whether the frame's depth property is ignored (for stereoscopic 3D setups)
		-- @name Frame:IsIgnoringDepth
		-- @class function
		-- @return 1 if the frame's depth property is ignored; otherwise nil
		-- @usage Frame:IsIgnoringDepth()
		------------------------------------

		-- IsIgnoringDepth
        ["IsIgnoringDepth"] = function(self, ...)
            return self.__UI:IsIgnoringDepth(...)
        end,

		------------------------------------
		--- Get whether this frame will get keyboard input.
		-- @name Frame:IsKeyboardEnabled
		-- @class function
		-- @return 1 if keyboard interactivity is enabled for the frame; otherwise nil
		-- @usage Frame:IsKeyboardEnabled()
		------------------------------------

		-- IsKeyboardEnabled
        ["IsKeyboardEnabled"] = function(self, ...)
			if not self.InDesignMode then
				return (self.__UI:IsKeyboardEnabled() and true) or false
			else
				return (self.__KeyboardEnabled and true) or false
			end
        end,

		------------------------------------
		--- Get whether this frame will get mouse input.
		-- @name Frame:IsMouseEnabled
		-- @class function
		-- @return 1 if mouse interactivity is enabled for the frame; otherwise nil
		-- @usage Frame:IsMouseEnabled()
		------------------------------------

		-- IsMouseEnabled
        ["IsMouseEnabled"] = function(self, ...)
			if not self.InDesignMode then
				return (self.__UI:IsMouseEnabled() and true) or false
			else
				return (self.__MouseEnabled and true) or false
			end
        end,

		------------------------------------
		--- Get whether this frame will get mouse wheel notifications.
		-- @name Frame:IsMouseWheelEnabled
		-- @class function
		-- @return 1 if mouse wheel interactivity is enabled for the frame; otherwise nil
		-- @usage Frame:IsMouseWheelEnabled()
		------------------------------------

		-- IsMouseWheelEnabled
        ["IsMouseWheelEnabled"] = function(self, ...)
            return self.__UI:IsMouseWheelEnabled(...)
        end,

		------------------------------------
		--- Determine if the frame can be moved.
		-- @name Frame:IsMovable
		-- @class function
		-- @return 1 if the frame can be moved by the user; otherwise nil
		-- @usage Frame:IsMovable()
		------------------------------------

		-- IsMovable
        ["IsMovable"] = function(self, ...)
			if not self.InDesignMode then
				return (self.__UI:IsMovable() and true) or false
			else
				return (self.__Movable and true) or false
			end
        end,

		------------------------------------
		--- Determine if the frame can be resized.
		-- @name Frame:IsResizable
		-- @class function
		-- @return 1 if the frame can be resized by the user; otherwise nil
		-- @usage Frame:IsResizable()
		------------------------------------

		-- IsResizable
        ["IsResizable"] = function(self, ...)
			if not self.InDesignMode then
				return (self.__UI:IsResizable() and true) or false
			else
				return (self.__Resizable and true) or false
			end
        end,

		------------------------------------
		--- Get whether the frame is set as toplevel
		-- @name Frame:IsToplevel
		-- @class function
		-- @return 1 if the frame is set as toplevel; otherwise nil
		-- @usage Frame:IsToplevel()
		------------------------------------

		-- IsToplevel
        ["IsToplevel"] = function(self, ...)
            return self.__UI:IsToplevel(...)
        end,

		------------------------------------
		--- Lower this frame behind other frames.
		-- @name Frame:Lower
		-- @class function
		-- @usage Frame:Lower()
		------------------------------------

		-- Lower
        ["Lower"] = function(self, ...)
            return self.__UI:Lower(...)
        end,

		------------------------------------
		--- Raise this frame above other frames.
		-- @name Frame:Raise
		-- @class function
		-- @usage Frame:Raise()
		------------------------------------

		-- Raise
        ["Raise"] = function(self, ...)
            return self.__UI:Raise(...)
        end,

		------------------------------------
		--- Inidicate that this frame should be notified of drag events for the specified buttons.
		-- @name Frame:RegisterForDrag
		-- @class function
		-- @param ... A list of strings, each the name of a mouse button:Button4,Button5,LeftButton,MiddleButton,RightButton
		-- @usage Frame:RegisterForDrag("LeftButton")
		------------------------------------

		-- RegisterForDrag
        ["RegisterForDrag"] = function(self, ...)
            return self.__UI:RegisterForDrag(...)
        end,

		------------------------------------
		--- Set the backdrop of the frame according to the specification provided.
		-- @name Frame:SetBackdrop
		-- @class function
		-- @param backdropTable Optional,A table containing the backdrop settings, or nil to remove the frame's backdrop
		-- @usage Frame:SetBackdrop{<br>
		--    bgFile = "Interface\\DialogFrame\\UI-DialogBox-Gold-Background",<br>
		--    edgeFile = "Interface\\DialogFrame\\UI-DialogBox-Gold-Border",<br>
		--    tile = true,<br>
		--    tileSize = 32,<br>
		--    edgeSize = 32,<br>
		--    insets = {<br>
		--         left = 11,<br>
		--         right = 12,<br>
		--         top = 12,<br>
		--         bottom = 11,<br>
		--    },<br>
		-- }<br>
		------------------------------------

		-- SetBackdrop
		["SetBackdrop"] = function(self, backdropTable)
			return self.__UI:SetBackdrop(backdropTable or nil)
		end,

		------------------------------------
		--- Set the frame's backdrop's border's color.
		-- @name Frame:SetBackdropBorderColor
		-- @class function
		-- @param red Red component of the color (0.0 - 1.0)
		-- @param green Green component of the color (0.0 - 1.0)
		-- @param blue Blue component of the color (0.0 - 1.0)
		-- @param alpha Alpha (opacity) for the graphic (0.0 = fully transparent, 1.0 = fully opaque)
		-- @usage Frame:SetBackdropBorderColor(1, 1, 0.8, 0.4)
		------------------------------------

		-- SetBackdropBorderColor
        ["SetBackdropBorderColor"] = function(self, ...)
            return self.__UI:SetBackdropBorderColor(...)
        end,

		------------------------------------
		--- Set the frame's backdrop color.
		-- @name Frame:SetBackdropColor
		-- @class function
		-- @param red Red component of the color (0.0 - 1.0)
		-- @param green Green component of the color (0.0 - 1.0)
		-- @param blue Blue component of the color (0.0 - 1.0)
		-- @param alpha Alpha (opacity) for the graphic (0.0 = fully transparent, 1.0 = fully opaque)
		-- @usage Frame:SetBackdropColor(1, 1, 0.8, 0.4)
		------------------------------------

		-- SetBackdropColor
        ["SetBackdropColor"] = function(self, ...)
            return self.__UI:SetBackdropColor(...)
        end,

		------------------------------------
		--- Set whether the frame is prohibited from being dragged off screen
		-- @name Frame:SetClampedToScreen
		-- @class function
		-- @param enable True to limit the frame's boundaries to those of the screen; false to allow the frame to be moved/resized without such limits
		-- @usage Frame:SetClampedToScreen(true)
		------------------------------------

		-- SetClampedToScreen
        ["SetClampedToScreen"] = function(self, ...)
            return self.__UI:SetClampedToScreen(...)
        end,

		------------------------------------
		--- Modify the frame's rectangle used to prevent dragging offscreen.
		-- @name Frame:SetClampRectInsets
		-- @class function
		-- @param left Offset from the left edge of the frame to the left edge of its clamping area (in pixels)
		-- @param right Offset from the right edge of the frame's clamping area to the right edge of the frame (in pixels)
		-- @param top Offset from the top edge of the frame's clamping area to the top edge of the frame (in pixels)
		-- @param bottom Offset from the bottom edge of the frame to the bottom edge of its clamping area (in pixels)
		-- @usage Frame:SetClampRectInsets(50, -50, -50, 50)
		------------------------------------

		-- SetClampRectInsets
        ["SetClampRectInsets"] = function(self, ...)
            return self.__UI:SetClampRectInsets(...)
        end,

		------------------------------------
		--- Sets the 3D depth of the frame (for stereoscopic 3D configurations)
		-- @name Frame:SetDepth
		-- @class function
		-- @param depth Apparent 3D depth of this frame relative to that of its parent frame
		-- @usage Frame:SetDepth(10)
		------------------------------------

		-- SetDepth
        ["SetDepth"] = function(self, ...)
            return self.__UI:SetDepth(...)
        end,

		------------------------------------
		--- Set the level of this frame (determines which of overlapping frames shows on top).
		-- @name Frame:SetFrameLevel
		-- @class function
		-- @param level Layering level of the frame relative to others in its frameStrata
		-- @usage Frame:SetFrameLevel(10)
		------------------------------------

		-- SetFrameLevel
        ["SetFrameLevel"] = function(self, ...)
            return self.__UI:SetFrameLevel(...)
        end,

		------------------------------------
		--- Set the strata of this frame.
		-- @name Frame:SetFrameStrata
		-- @class function
		-- @param strata Token identifying the strata in which the frame should be layered
		-- @usage Frame:SetFrameStrata("TOOLTIP")
		------------------------------------

		-- SetFrameStrata
        ["SetFrameStrata"] = function(self, ...)
            return self.__UI:SetFrameStrata(...)
        end,

		------------------------------------
		--- Set the inset distances for the frame's hit rectangle
		-- @name Frame:SetHitRectInsets
		-- @class function
		-- @param left Distance from the left edge of the frame to the left edge of its mouse-interactive area (in pixels)
		-- @param right Distance from the right edge of the frame to the right edge of its mouse-interactive area (in pixels)
		-- @param top Distance from the top edge of the frame to the top edge of its mouse-interactive area (in pixels)
		-- @param bottom Distance from the bottom edge of the frame to the bottom edge of its mouse-interactive area (in pixels)
		-- @usage Frame:SetHitRectInsets(10, -10, -10, 10)
		------------------------------------

		-- SetHitRectInsets
        ["SetHitRectInsets"] = function(self, ...)
            return self.__UI:SetHitRectInsets(...)
        end,

		------------------------------------
		--- Set the ID of this frame.
		-- @name Frame:SetID
		-- @class function
		-- @param id A numeric identifier for the frame
		-- @usage Frame:SetID(1)
		------------------------------------

		-- SetID
        ["SetID"] = function(self, ...)
            return self.__UI:SetID(...)
        end,

		------------------------------------
		--- Set the maximum dimensions this frame can be resized to.
		-- @name Frame:SetMaxResize
		-- @class function
		-- @param maxWidth Maximum width of the frame (in pixels), or 0  for no limit
		-- @param maxHeight Maximum height of the frame (in pixels), or 0  for no limit
		-- @usage Frame:SetMaxResize(400, 300)
		------------------------------------

		-- SetMaxResize
        ["SetMaxResize"] = function(self, ...)
            return self.__UI:SetMaxResize(...)
        end,

		------------------------------------
		--- Set the minimum dimensions this frame can be resized to.
		-- @name Frame:SetMinResize
		-- @class function
		-- @param minWidth Minimum width of the frame (in pixels), or 0  for no limit
		-- @param minHeight Minimum height of the frame (in pixels), or 0  for no limit
		-- @usage Frame:SetMinResize(200, 300)
		------------------------------------

		-- SetMinResize
        ["SetMinResize"] = function(self, ...)
            return self.__UI:SetMinResize(...)
        end,

		------------------------------------
		--- Set whether the frame can be moved.
		-- @name Frame:SetMovable
		-- @class function
		-- @param enable True to allow the frame to be moved by the user; false to disable
		-- @usage Frame:SetMovable(true)
		------------------------------------

		-- SetMovable
        ["SetMovable"] = function(self, enabled)
			if not self.InDesignMode then
				self.__UI:SetMovable(enabled)
			else
				self.__Movable = (enabled and true) or false
			end
        end,

		------------------------------------
		--- Set whether the frame can be resized.
		-- @name Frame:SetResizable
		-- @class function
		-- @param enable True to allow the frame to be resized by the user; false to disable
		-- @usage Frame:SetResizable(true)
		------------------------------------

		-- SetResizable
        ["SetResizable"] = function(self, enabled)
			if not self.InDesignMode then
				self.__UI:SetResizable(enabled)
			else
				self.__Resizable = (enabled and true) or false
			end
        end,

		------------------------------------
		--- Set the scale factor of this frame relative to its parent.
		-- @name Frame:SetScale
		-- @class function
		-- @param scale Scale factor for the frame relative to its parent
		-- @usage Frame:SetScale(1.5)
		------------------------------------

		-- SetScale
        ["SetScale"] = function(self, ...)
            return self.__UI:SetScale(...)
        end,

		------------------------------------
		--- Set whether the frame should raise itself when clicked
		-- @name Frame:SetToplevel
		-- @class function
		-- @param enable True to cause the frame to automatically come to the front when clicked; false otherwise
		-- @usage Frame:SetToplevel(true)
		------------------------------------

		-- SetToplevel
        ["SetToplevel"] = function(self, ...)
            return self.__UI:SetToplevel(...)
        end,

		------------------------------------
		--- Set whether the frame has been relocated by the user (and will thus be saved in the layout cache). Not supported in IGAS GUI Sytem now.
		-- @name Frame:SetUserPlaced
		-- @class function
		-- @param enable True to enable automatic saving and restoration of the frame's position and dimensions; false to disable
		-- @usage Frame:SetUserPlaced(true)
		------------------------------------

		-- SetUserPlaced
        ["SetUserPlaced"] = function(self, ...)
            return self.__UI:SetUserPlaced(...)
        end,

		------------------------------------
		--- Determine if this frame has been relocated by the user.
		-- @name Frame:IsUserPlaced
		-- @class function
		-- @return 1 if the frame is flagged for automatic saving and restoration of position and dimensions; otherwise nil
		-- @usage Frame:IsUserPlaced()
		------------------------------------

		-- IsUserPlaced
        ["IsUserPlaced"] = function(self, ...)
            return self.__UI:IsUserPlaced(...)
        end,

		------------------------------------
		--- Start moving this frame.
		-- @name Frame:StartMoving
		-- @class function
		-- @usage Frame:StartMoving()
		------------------------------------

		-- StartMoving
        ["StartMoving"] = function(self, ...)
            return self.__UI:StartMoving(...)
        end,

		------------------------------------
		--- Start sizing this frame using the specified anchor point.
		-- @name Frame:StartSizing
		-- @class function
		-- @param point the specified anchor point
		-- @usage Frame:StartSizing("TOPLEFT")
		------------------------------------

		-- StartSizing
        ["StartSizing"] = function(self, ...)
            return self.__UI:StartSizing(...)
        end,

		------------------------------------
		--- Stop moving and/or sizing this frame.
		-- @name Frame:StopMovingOrSizing
		-- @class function
		-- @usage Frame:StopMovingOrSizing()
		------------------------------------

		-- StopMovingOrSizing
        ["StopMovingOrSizing"] = function(self, ...)
            return self.__UI:StopMovingOrSizing(...)
        end,

		------------------------------------
		--- Check whether the event is a system event
		-- @name Frame:IsSystemEvent
		-- @class function
		-- @param event the event's name
		-- @return true if the event is a system event
		-- @usage Frame:IsSystemEvent("PLAYER_LOGIN")
		------------------------------------

		-- IsSystemEvent
		["IsSystemEvent"] = function(self, event)
			if not event or type(event) ~= "string" then
				return false
			end

			return (_SystemEventChkr:IsEventRegistered(event) and true) or false
		end,

		------------------------------------
		--- Returns true if the given event is registered to the frame.
		-- @name Frame:IsEventRegistered
		-- @class function
		-- @param event the event's name
		-- @return true if the event is registered to the frame
		-- @usage Frame:IsEventRegistered("PLAYER_LOGIN")
		------------------------------------

		-- IsEventRegistered
        ["IsEventRegistered"] = function(self, event)
			if self:IsSystemEvent(event) then
				return self.__UI:IsEventRegistered(event)
			else
				-- Custom Event
				return IGAS:IsCustomEventRegistered(self, event)
			end
        end,

		------------------------------------
		--- Register this frame to receive all events (For debugging purposes only!)
		-- @name Frame:RegisterAllEvents
		-- @class function
		-- @usage Frame:RegisterAllEvents()
		------------------------------------

		-- RegisterAllEvents
        ["RegisterAllEvents"] = function(self, ...)
            return self.__UI:RegisterAllEvents(...)
        end,

		------------------------------------
		--- Indicate that this frame should be notified when event occurs.
		-- @name Frame:RegisterEvent
		-- @class function
		-- @param event the event's name, can be a custom event
		-- @usage Frame:RegisterEvent("IGAS_LOADED")
		------------------------------------

		-- RegisterEvent
        ["RegisterEvent"] = function(self, event)
			if self.InDesignMode then
				self.__RegisterEventList = self.__RegisterEventList or {}
				self.__RegisterEventList[event] = true
				return
			end
			if self:IsSystemEvent(event) then
				return self.__UI:RegisterEvent(event)
			else
				return IGAS:RegisterCustomEvent(self, event)
			end
        end,

		------------------------------------
		--- Indicate that this frame should no longer be notified when any events occur.
		-- @name Frame:UnregisterAllEvents
		-- @class function
		-- @usage Frame:UnregisterAllEvents()
		------------------------------------

		-- UnregisterAllEvents
        ["UnregisterAllEvents"] = function(self)
			self.__UI:UnregisterAllEvents()
            return IGAS:UnregisterAllCustomEvents(self)
        end,

		------------------------------------
		--- Indicate that this frame should no longer be notified when event occurs.
		-- @name Frame:UnregisterEvent
		-- @class function
		-- @param event the event's name
		-- @usage Frame:UnregisterEvent("IGAS_LOADED")
		------------------------------------

		-- UnregisterEvent
        ["UnregisterEvent"] = function(self, event)
			if self.InDesignMode then
				self.__RegisterEventList = self.__RegisterEventList or {}
				self.__RegisterEventList[event] = nil
			end
			if self:IsSystemEvent(event) then
				return self.__UI:UnregisterEvent(event)
			else
				return IGAS:UnregisterCustomEvent(self, event)
			end
        end,

		------------------------------------
		--- Fire an Event, can only fire custom event, fire system event will do nothing
		-- @name Frame:FireEvent
		-- @class function
		-- @param event the event's name
		-- @param ... Optional,the event's parameters
		-- @usage Frame:FireEvent("IGAS_LOADED", IGAS)
		------------------------------------

		-- FireEvent
		["FireEvent"] = function(self, event, ...)
			return IGAS:FireCustomEvent(self, event, ...)
		end,
	}

	-- Property
	local _Property = {
		-- KeyboardEnabled
		["KeyboardEnabled"] = {
			["Set"] = function(self, enabled)
				self:EnableKeyboard(enabled)
			end,

			["Get"] = function(self)
				return self:IsKeyboardEnabled()
			end,

			["Type"] = "boolean",
		},

		["MouseEnabled"] = {
			["Set"] = function(self, enabled)
				self:EnableMouse(enabled)
			end,

			["Get"] = function(self)
				return self:IsMouseEnabled()
			end,

			["Type"] = "boolean",
		},

		["Movable"] = {
			["Set"] = function(self, enabled)
				self:SetMovable(enabled)
			end,

			["Get"] = function(self)
				return self:IsMovable()
			end,

			["Type"] = "boolean",
		},

		["Resizable"] = {
			["Set"] = function(self, enabled)
				self:SetResizable(enabled)
			end,

			["Get"] = function(self)
				return self:IsResizable()
			end,

			["Type"] = "boolean",
		},
		
		["MouseWheelEnabled"] = {
			["Set"] = function(self, enabled)
				self:EnableMouseWheel(enabled)
			end,

			["Get"] = function(self)
				return (self:IsMouseWheelEnabled() and true) or false
			end,

			["Type"] = "boolean",
		},

		["Backdrop"] = {
			["Set"] = function(self, backdropTable)
				self:SetBackdrop(backdropTable)
			end,

			["Get"] = function(self)
				return self:GetBackdrop()
			end,

			["Type"] = "Backdrop",
		},

		["BackdropBorderColor"] = {
			["Set"] = function(self, colorTable)
				self.__UI:SetBackdropBorderColor(colorTable.red, colorTable.green, colorTable.blue, colorTable.alpha)
			end,

			["Get"] = function(self)
				local colorTable = {}
				colorTable.red, colorTable.green, colorTable.blue, colorTable.alpha = self.__UI:GetBackdropBorderColor()
				return colorTable
			end,

			["Type"] = "Color",
		},

		["BackdropColor"] = {
			["Set"] = function(self, colorTable)
				self.__UI:SetBackdropColor(colorTable.red, colorTable.green, colorTable.blue, colorTable.alpha)
			end,

			["Get"] = function(self)
				local colorTable = {}
				colorTable.red, colorTable.green, colorTable.blue, colorTable.alpha = self.__UI:GetBackdropColor()
				return colorTable
			end,

			["Type"] = "Color",
		},

		["ClampedToScreen"] = {
			["Set"] = function(self, enabled)
				self.__UI:SetClampedToScreen(enabled)
			end,

			["Get"] = function(self)
				return (self.__UI:IsClampedToScreen() and true) or false
			end,

			["Type"] = "boolean",
		},

		["ClampRectInsets"] = {
			["Set"] = function(self, RectInset)
				self.__UI:SetClampRectInsets(RectInset.left, RectInset.right, RectInset.top, RectInset.bottom)
			end,

			["Get"] = function(self)
				local RectInset = {}
				RectInset.left, RectInset.right, RectInset.top, RectInset.bottom = self.__UI:GetClampRectInsets()
				return RectInset
			end,

			["Type"] = "RectInset",
		},

		["FrameLevel"] = {
			["Set"] = function(self, level)
				self.__UI:SetFrameLevel(level)
			end,

			["Get"] = function(self)
				return self.__UI:GetFrameLevel()
			end,

			["Type"] = "number",
		},

		["FrameStrata"] = {
			["Set"] = function(self, strata)
				self.__UI:SetFrameStrata(strata)
			end,

			["Get"] = function(self)
				return self.__UI:GetFrameStrata()
			end,

			["Type"] = "FrameStrata",
		},

		["HitRectInsets"] = {
			["Set"] = function(self, RectInset)
				self.__UI:SetHitRectInsets(RectInset.left, RectInset.right, RectInset.top, RectInset.bottom)
			end,

			["Get"] = function(self)
				local RectInset = {}
				RectInset.left, RectInset.right, RectInset.top, RectInset.bottom = self.__UI:GetHitRectInsets()
				return RectInset
			end,

			["Type"] = "RectInset",
		},

		["ID"] = {
			["Set"] = function(self, id)
				self.__UI:SetID(id)
			end,

			["Get"] = function(self)
				return self.__UI:GetID()
			end,

			["Type"] = "number",
		},

		["MaxResize"] = {
			["Set"] = function(self, Size)
				self.__UI:SetMaxResize(Size.width, Size.height)
			end,

			["Get"] = function(self)
				local Size = {}
				Size.width, Size.height = self.__UI:GetMaxResize()
				return Size
			end,

			["Type"] = "Size",
		},

		["MinResize"] = {
			["Set"] = function(self, Size)
				self.__UI:SetMinResize(Size.width, Size.height)
			end,

			["Get"] = function(self)
				local Size = {}
				Size.width, Size.height = self.__UI:GetMinResize()
				return Size
			end,

			["Type"] = "Size",
		},

		["Scale"] = {
			["Set"] = function(self, scale)
				self.__UI:SetScale(scale)
			end,

			["Get"] = function(self)
				return self.__UI:GetScale()
			end,

			["Type"] = "number",
		},

		["Toplevel"] = {
			["Set"] = function(self, enabled)
				self.__UI:SetToplevel(enabled)
			end,

			["Get"] = function(self)
				return (self.__UI:IsToplevel() and true) or false
			end,

			["Type"] = "boolean",
		},

		["Depth"] = {
			["Set"] = function(self, depth)
				self.__UI:SetDepth(depth)
			end,

			["Get"] = function(self)
				return self.__UI:GetDepth()
			end,

			["Type"] = "number",
		},

		["DepthIgnored"] = {
			["Set"] = function(self, enabled)
				self.__UI:IgnoreDepth(enabled)
			end,

			["Get"] = function(self)
				return (self.__UI:IsIgnoringDepth() and true) or false
			end,

			["Type"] = "boolean",
		},
	}

    -- New
    local _New = function(_parent,inheritFrame)
        -- New Frame
		return CreateFrame(_WidgetName,nil,IGAS:GetUI(_parent),inheritFrame)
    end

    local _WidgetInfo = {
        ["WidgetName"] = _WidgetName,
        ["Base"] = _Base,
        ["FuncProxy"] = _FuncProxy,
        ["Property"] = _Property,
        ["ScriptType"] = _ScriptType,
        ["New"] = _New,
    }

    IGAS:NewWidget(_WidgetInfo, true, true)
end