﻿-- Author      : Kurapica
-- Create Date : 7/16/2008 15:19
-- ChangeLog   :
--				2010.01.12	Add the default parameters for _New

----------------------------------------------------------------------------------------------------------------------------------------
--- FontStrings are one of the two types of Region that is visible on the screen. It draws a block of text on the screen using the characteristics in an associated FontObject.
-- <br><br>inherit <a href=".\LayeredRegion.html">LayeredRegion</a> For all methods, properties and scriptTypes
-- @name FontString
-- @class table
-- @field Font the font's defined table, contains font path, height and flags' settings
-- @field FontObject the `Font` object
-- @field JustifyH the fontstring's horizontal text alignment style
-- @field JustifyV the fontstring's vertical text alignment style
-- @field ShadowColor the color of the font's text shadow
-- @field ShadowOffset the offset of the fontstring's text shadow from its text
-- @field Spacing the fontstring's amount of spacing between lines
-- @field TextColor the fontstring's default text color
-- @field NonSpaceWrap whether long lines of text will wrap within or between words
-- @field StringHeight the height of the text displayed in the font string
-- @field StringWidth Readonly, the width of the text displayed in the font string
-- @field Text the text to be displayed in the font string
----------------------------------------------------------------------------------------------------------------------------------------
do
	-- Check Version
	local version = 3
	if not IGAS:NewAddon("IGAS.GUI.FontString", version) then
		return
	end

    local CreateFrame = CreateFrame

	local _WidgetName = "FontString"
    local _Base = "LayeredRegion"

    -- FuncProxy
	local _FuncProxy = {
		-- SetParent(parent or "parentName") - Set the parent for this frame
        ["SetParent"] = function(self, parent)
			-- Get frame from name
			if parent and type(parent) == "string" then
				parent = _G[parent]

				if not parent then
					error("No such frame", 2)
				end
			end

			parent = IGAS:GetWrapper(parent)

			-- Check parent
			if parent and (type(parent) ~= "table" or not parent["IsIGASFrame"] or not parent["AddChild"]) then
				error("This object can't add child.", 2)
			end

			-- Set parent
			if parent then
				if self.__Name then
	                -- Check
	                if parent.__Childs[self.__Name] then
	                    if parent.__Childs[self.__Name] ~= self then
	                        error("The parent has a child with the same name")
	                    else
	                        return
	                    end
	                end

	                -- Remove ob from it's old parent
					self:SetParent(nil)

	                -- Add ob to new parent
	                self.__Parent = parent
	                parent.__Childs[self.__Name] = self

					-- SetParent
					if IGAS:GetUI(self) ~= self and IGAS:GetUI(self)["SetParent"] then
						IGAS:GetUI(self):SetParent(IGAS:GetUI(parent))
					end
	            else
	                error("can't add this child", 2)
	            end
			else
				if self.__Name and self.__Parent then
					self.__Parent.__Childs[self.__Name] = nil
				end
				self.__Parent = nil
			end
        end,

        ------------------------------------
		--- Get whether long strings without spaces are wrapped or truncated
		-- @name FontString:CanNonSpaceWrap
		-- @class function
		-- @return 1 if long lines of text will wrap at any character boundary (i.e possibly in the middle of a word); nil  to only wrap at whitespace characters
		-- @usage FontString:CanNonSpaceWrap()
		------------------------------------

		-- CanNonSpaceWrap
		["CanNonSpaceWrap"] = function(self, ...)
			return self.__UI:CanNonSpaceWrap(...)
		end,

        ------------------------------------
		--- Returns the height in pixels of the current string in the current font (without line wrapping)
		-- @name FontString:GetStringHeight
		-- @class function
		-- @return Height of the text currently displayed in the font string (in pixels)
		-- @usage FontString:GetStringHeight()
		------------------------------------

		-- GetStringHeight
		["GetStringHeight"] = function(self, ...)
			return self.__UI:GetStringHeight(...)
		end,

        ------------------------------------
		--- Returns the width in pixels of the current string in the current font (without line wrapping).
		-- @name FontString:GetStringWidth
		-- @class function
		-- @return Width of the text currently displayed in the font string (in pixels)
		-- @usage FontString:GetStringWidth()
		------------------------------------

		-- GetStringWidth
		["GetStringWidth"] = function(self, ...)
			return self.__UI:GetStringWidth(...)
		end,

        ------------------------------------
		--- Get the displayed text.
		-- @name FontString:GetText
		-- @class function
		-- @return the displayed text
		-- @usage FontString:GetText()
		------------------------------------

		-- GetText
		["GetText"] = function(self, ...)
			return self.__UI:GetText(...)
		end,

        ------------------------------------
		--- Create or remove an alpha gradient over the text.
		-- @name FontString:SetAlphaGradient
		-- @class function
		-- @param start Character position in the font string's text at which the gradient should begin (between 0 and string.len(fontString:GetText()) - 6)
		-- @param length Width of the gradient in pixels, or 0  to restore the text to full opacity
		-- @usage FontString:SetAlphaGradient(3, 0)
		------------------------------------

		-- SetAlphaGradient
		["SetAlphaGradient"] = function(self, ...)
			return self.__UI:SetAlphaGradient(...)
		end,

        ------------------------------------
		--- Set the formatted display text.
		-- @name FontString:SetFormattedText
		-- @class function
		-- @param formatString A string containing format specifiers
		-- @param ... A list of values to be included in the formatted string
		-- @usage FontString:SetFormattedText("%s %q", "Hello", "Azeroth!")
		------------------------------------

		-- SetFormattedText
		["SetFormattedText"] = function(self, ...)
			return self.__UI:SetFormattedText(...)
		end,

        ------------------------------------
		--- Set whether long strings without spaces are wrapped or truncated.
		-- @name FontString:SetNonSpaceWrap
		-- @class function
		-- @param enable True to wrap long lines of text at any character boundary,false to only wrap at whitespace characters
		-- @usage FontString:SetNonSpaceWrap(true)
		------------------------------------

		-- SetNonSpaceWrap
		["SetNonSpaceWrap"] = function(self, ...)
			return self.__UI:SetNonSpaceWrap(...)
		end,

        ------------------------------------
		--- Set the displayed text.
		-- @name FontString:SetText
		-- @class function
		-- @param text the text to be displayed
		-- @usage FontString:SetText("Hello")
		------------------------------------

		-- SetText
		["SetText"] = function(self, ...)
			return self.__UI:SetText(...)
		end,

        ------------------------------------
		--- Set the height of the text by scaling graphics
		-- @name FontString:SetTextHeight
		-- @class function
		-- @param height Height (point size) to which the text should be scaled
		-- @usage FontString:SetTextHeight(24)
		------------------------------------

		-- SetTextHeight
		["SetTextHeight"] = function(self, ...)
			return self.__UI:SetTextHeight(...)
		end,

        ------------------------------------
		--- Returns the font instance's basic font properties
		-- @name FontInstance:GetFont
		-- @class function
		-- @return filename Path to a font file
		-- @return fontHeight Height (point size) of the font to be displayed (in pixels)
		-- @return flags Additional properties for the font specified by one or more (separated by commas) of the following tokens: MONOCHROME, OUTLINE, THICKOUTLINE
		-- @usage FontInstance:GetFont()
		------------------------------------

		-- GetFont
        ["GetFont"] = function(self)
            return self.__UI:GetFont()
        end,

		------------------------------------
		--- Return the 'parent' Font object, or nil if none.
		-- @name FontInstance:GetFontObject
		-- @class function
		-- @return the 'parent' Font object, or nil if none
		-- @usage FontInstance:GetFontObject()
		------------------------------------

		-- GetFontObject
        ["GetFontObject"] = function(self)
            return IGAS:GetWrapper(self.__UI:GetFontObject())
        end,

		------------------------------------
		--- Return the horizontal text justification.
		-- @name FontInstance:GetJustifyH
		-- @class function
		-- @return the font instance's horizontal text alignment style
		-- @usage FontInstance:GetJustifyH()
		------------------------------------

		-- GetJustifyH
        ["GetJustifyH"] = function(self)
            return self.__UI:GetJustifyH()
        end,

		------------------------------------
		--- Return the vertical text justification.
		-- @name FontInstance:GetJustifyV
		-- @class function
		-- @return the font instance's vertical text alignment style
		-- @usage FontInstance:GetJustifyV()
		------------------------------------

		-- GetJustifyV
        ["GetJustifyV"] = function(self)
            return self.__UI:GetJustifyV()
        end,

		------------------------------------
		--- Returns the color of the font's text shadow
		-- @name FontInstance:GetShadowColor
		-- @class function
		-- @return shadowR Red component of the shadow color (0.0 - 1.0)
		-- @return shadowG Green component of the shadow color (0.0 - 1.0)
		-- @return shadowB Blue component of the shadow color (0.0 - 1.0)
		-- @return shadowAlpha Alpha (opacity) of the text's shadow (0.0 = fully transparent, 1.0 = fully opaque)
		-- @usage FontInstance:GetShadowColor()
		------------------------------------

		-- GetShadowColor
        ["GetShadowColor"] = function(self)
            return self.__UI:GetShadowColor()
        end,

		------------------------------------
		--- Returns the offset of the font instance's text shadow from its text
		-- @name FontInstance:GetShadowOffset
		-- @class function
		-- @return xOffset Horizontal distance between the text and its shadow (in pixels)
		-- @return yOffset Vertical distance between the text and its shadow (in pixels)
		-- @usage FontInstance:GetShadowOffset()
		------------------------------------

		-- GetShadowOffset
        ["GetShadowOffset"] = function(self)
            return self.__UI:GetShadowOffset()
        end,

		------------------------------------
		--- Returns the font instance's amount of spacing between lines
		-- @name FontInstance:GetSpacing
		-- @class function
		-- @return spacing Amount of space between lines of text (in pixels)
		-- @usage FontInstance:GetSpacing()
		------------------------------------

		-- GetSpacing
        ["GetSpacing"] = function(self)
            return self.__UI:GetSpacing()
        end,

		------------------------------------
		--- Returns the font instance's default text color
		-- @name FontInstance:GetTextColor
		-- @class function
		-- @return textR Red component of the text color (0.0 - 1.0)
		-- @return textG Green component of the text color (0.0 - 1.0)
		-- @return textB Blue component of the text color (0.0 - 1.0)
		-- @return textAlpha Alpha (opacity) of the text (0.0 = fully transparent, 1.0 = fully opaque)
		-- @usage FontInstance:GetTextColor()
		------------------------------------

		-- GetTextColor
        ["GetTextColor"] = function(self)
            return self.__UI:GetTextColor()
        end,

		------------------------------------
		--- Sets the font to use for text, returns 1 if the path was valid, nil otherwise(no change occurs).
		-- @name FontInstance:SetFont
		-- @class function
		-- @param filename Path to a font file
		-- @param fontHeight Height (point size) of the font to be displayed (in pixels)
		-- @param flags Additional properties for the font specified by one or more (separated by commas) of the following tokens:MONOCHROME,OUTLINE,THICKOUTLINE
		-- @return 1 if filename refers to a valid font file; otherwise nil
		-- @usage FontInstance:SetFont("Fonts\\ARIALN.TTF", 10)
		------------------------------------

		-- SetFont
        ["SetFont"] = function(self, path, height, flags)
            self.__UI:SetFont(path, height, flags)
        end,

		------------------------------------
		--- Sets the 'parent' Font object from which this object inherits properties.
		-- @name FontInstance:SetFontObject
		-- @class function
		-- @param object Reference to a Font object Or Global name of a Font object
		-- @usage FontInstance:SetFontObject(GameFontNormal)
		------------------------------------

		-- SetFontObject
        ["SetFontObject"] = function(self, fontObject)
            self.__UI:SetFontObject(IGAS:GetUI(fontObject))
        end,

		------------------------------------
		--- Sets horizontal text justification ("LEFT","RIGHT", or "CENTER")
		-- @name FontInstance:SetJustifyH
		-- @class function
		-- @param justify Horizontal text alignment style
		-- @usage FontInstance:SetJustifyH("CENTER")
		------------------------------------

		-- SetJustifyH
        ["SetJustifyH"] = function(self, justifyH)
            self.__UI:SetJustifyH(justifyH)
        end,

		------------------------------------
		--- Sets vertical text justification ("TOP","BOTTOM", or "MIDDLE")
		-- @name FontInstance:SetJustifyV
		-- @class function
		-- @param justify Vertical text alignment style
		-- @usage FontInstance:SetJustifyV("BOTTOM")
		------------------------------------

		-- SetJustifyV
        ["SetJustifyV"] = function(self, justifyV)
            self.__UI:SetJustifyV(justifyV)
        end,

		------------------------------------
		--- Sets the text shadow color.
		-- @name FontInstance:SetShadowColor
		-- @class function
		-- @param shadowR Red component of the shadow color (0.0 - 1.0)
		-- @param shadowG Green component of the shadow color (0.0 - 1.0)
		-- @param shadowB Blue component of the shadow color (0.0 - 1.0)
		-- @param shadowAlpha Alpha (opacity) of the text's shadow (0.0 = fully transparent, 1.0 = fully opaque)
		-- @usage FontInstance:SetShadowColor(1, 1, 1, 1)
		------------------------------------

		-- SetShadowColor
        ["SetShadowColor"] = function(self, r,g,b,a)
            self.__UI:SetShadowColor(r,g,b,a)
        end,

		------------------------------------
		--- Sets the text shadow offset.
		-- @name FontInstance:SetShadowOffset
		-- @class function
		-- @param xOffset Horizontal distance between the text and its shadow (in pixels)
		-- @param yOffset Vertical distance between the text and its shadow (in pixels)
		-- @usage FontInstance:SetShadowOffset(5, -5)
		------------------------------------

		-- SetShadowOffset
        ["SetShadowOffset"] = function(self, x,y)
            self.__UI:SetShadowOffset(x,y)
        end,

		------------------------------------
		--- Sets the spacing between lines of text in the object.
		-- @name FontInstance:SetSpacing
		-- @class function
		-- @param spacing the spacing between lines
		-- @usage FontInstance:SetSpacing(3)
		------------------------------------

		-- SetSpacing
        ["SetSpacing"] = function(self, spacing)
            self.__UI:SetSpacing(spacing)
        end,

		------------------------------------
		--- Sets the default text color.
		-- @name FontInstance:SetTextColor
		-- @class function
		-- @param textR Red component of the text color (0.0 - 1.0)
		-- @param textG Green component of the text color (0.0 - 1.0)
		-- @param textB Blue component of the text color (0.0 - 1.0)
		-- @param textAlpha Alpha (opacity) of the text (0.0 = fully transparent, 1.0 = fully opaque)
		-- @usage FontInstance:SetTextColor(1, 1, 1, 0.8)
		------------------------------------

		-- SetTextColor
        ["SetTextColor"] = function(self, r,g,b,a)
            self.__UI:SetTextColor(r,g,b,a)
        end,
    }

	-- Property
	local _Property = {
		-- Font
		["Font"] = {
			["Set"] = function(self, font)
				local flags = ""
				if font.outline then
					if font.outline == "NORMAL" then
						flags = flags.."OUTLINE"
					elseif font.outline == "THICK" then
						flags = flags.."THICKOUTLINE"
					end
				end
				if font.monochrome then
					if flags ~= "" then
						flags = flags..","
					end
					flags = flags.."MONOCHROME"
				end

				self:SetFont(font.path, font.size, flags)
			end,

			["Get"] = function(self)
				local font = {}
				local flags
				font.path, font.size, flags = self:GetFont()
				flags = flags or ""
				if string.find(flags, "THICKOUTLINE") then
					font.outline = "THICK"
				elseif string.find(flags, "OUTLINE") then
					font.outline = "NORMAL"
				end
				if string.find(flags, "MONOCHROME") then
					font.monochrome = true
				end

				return font
			end,

			["Type"] = "FontTable",
		},
		-- FontObject
		["FontObject"] = {
			["Set"] = function(self, fontObject)
				self:SetFontObject(fontObject)
			end,

			["Get"] = function(self)
				return self:GetFontObject()
			end,

			["Type"] = "Font",
		},
		-- JustifyH
		["JustifyH"] = {
			["Set"] = function(self, justifyH)
				self:SetJustifyH(justifyH)
			end,

			["Get"] = function(self)
				return self:GetJustifyH()
			end,

			["Type"] = "JustifyH",
		},
		-- JustifyV
		["JustifyV"] = {
			["Set"] = function(self, justifyV)
				self:SetJustifyV(justifyV)
			end,

			["Get"] = function(self)
				return self:GetJustifyV()
			end,

			["Type"] = "JustifyV",
		},
		-- ShadowColor
		["ShadowColor"] = {
			["Set"] = function(self, color)
				self:SetShadowColor(color.red, color.green, color.blue, color.alpha)
			end,

			["Get"] = function(self)
				local color = {}
				color.red, color.green, color.blue, color.alpha = self:GetShadowColor()

				return color
			end,

			["Type"] = "Color",
		},
		-- ShadowOffset
		["ShadowOffset"] = {
			["Set"] = function(self, offset)
				self:SetShadowOffset(offset.x, offset.y)
			end,

			["Get"] = function(self)
				local offset = {}
				offset.x, offset.y = self:GetShadowOffset()

				return offset
			end,

			["Type"] = "Offset",
		},
		-- Spacing
		["Spacing"] = {
			["Set"] = function(self, spacing)
				self:SetSpacing(spacing)
			end,

			["Get"] = function(self)
				return self:GetSpacing()
			end,

			["Type"] = "number",
		},
		-- TextColor
		["TextColor"] = {
			["Set"] = function(self, color)
				self:SetTextColor(color.red, color.green, color.blue, color.alpha)
			end,

			["Get"] = function(self)
				local color = {}
				color.red, color.green, color.blue, color.alpha = self:GetTextColor()

				return color
			end,

			["Type"] = "Color",
		},
		-- NonSpaceWrap
		["NonSpaceWrap"] = {
			["Set"] = function(self, flag)
				self.__UI:SetNonSpaceWrap(flag)
			end,

			["Get"] = function(self)
				return (self.__UI:CanNonSpaceWrap() and true) or false
			end,

			["Type"] = "boolean",
		},
		-- StringHeight
		["StringHeight"] = {
			["Set"] = function(self, height)
				self.__UI:SetTextHeight(height)
			end,

			["Get"] = function(self)
				return self.__UI:GetStringHeight()
			end,

			["Type"] = "number",
		},
		-- StringWidth
		["StringWidth"] = {
			["Get"] = function(self)
				return self.__UI:GetStringWidth()
			end,

			["Type"] = "number",
		},
		-- Text
		["Text"] = {
			["Set"] = function(self, text)
				self:SetText(text)
			end,

			["Get"] = function(self)
				return self:GetText()
			end,

			["Type"] = "string",
		},
	}

    local _New = function(_parent, layer, inheritsFrom)
		if _parent and _parent["CreateFontString"] then
			return IGAS:GetUI(_parent):CreateFontString(nil, layer or "OVERLAY", inheritsFrom or "GameFontNormal")
		end
    end

    local _WidgetInfo = {
        ["WidgetName"] = _WidgetName,
        ["Base"] = _Base,
        ["FuncProxy"] = _FuncProxy,
        ["Property"] = _Property,
        ["New"] = _New,
    }

    IGAS:NewWidget(_WidgetInfo, true, true)
end