﻿-- Author      : Kurapica
-- Create Date : 7/16/2008 14:19

----------------------------------------------------------------------------------------------------------------------------------------
--- FontFrame is used to combine Frame and FontInstance
-- <br><br>inherit <a href=".\Frame.html">Frame</a> For all methods, properties and scriptTypes
-- @name FontFrame
-- @class table
-- @field Font the font's defined table, contains font path, height and flags' settings
-- @field FontObject the `Font` object
-- @field JustifyH the font instance's horizontal text alignment style
-- @field JustifyV the font instance's vertical text alignment style
-- @field ShadowColor the color of the font's text shadow
-- @field ShadowOffset the offset of the font instance's text shadow from its text
-- @field Spacing the font instance's amount of spacing between lines
-- @field TextColor the font instance's default text color
----------------------------------------------------------------------------------------------------------------------------------------
do
	-- Check Version
	local version = 2
	if not IGAS:NewAddon("IGAS.GUI.FontFrame", version) then
		return
	end

    local CreateFrame = CreateFrame

	local _WidgetName = "FontFrame"
    local _Base = "Frame"

    -- FuncProxy
	local _FuncProxy = {
		------------------------------------
		--- Returns the font instance's basic font properties
		-- @name FontFrame:GetFont
		-- @class function
		-- @return filename Path to a font file
		-- @return fontHeight Height (point size) of the font to be displayed (in pixels)
		-- @return flags Additional properties for the font specified by one or more (separated by commas) of the following tokens: MONOCHROME, OUTLINE, THICKOUTLINE
		-- @usage FontFrame:GetFont()
		------------------------------------

		-- GetFont
        ["GetFont"] = function(self)
            return self.__UI:GetFont()
        end,

		------------------------------------
		--- Return the 'parent' Font object, or nil if none.
		-- @name FontFrame:GetFontObject
		-- @class function
		-- @return the 'parent' Font object, or nil if none
		-- @usage FontFrame:GetFontObject()
		------------------------------------

		-- GetFontObject
        ["GetFontObject"] = function(self)
            return IGAS:GetWrapper(self.__UI:GetFontObject())
        end,

		------------------------------------
		--- Return the horizontal text justification.
		-- @name FontFrame:GetJustifyH
		-- @class function
		-- @return the font instance's horizontal text alignment style
		-- @usage FontFrame:GetJustifyH()
		------------------------------------

		-- GetJustifyH
        ["GetJustifyH"] = function(self)
            return self.__UI:GetJustifyH()
        end,

		------------------------------------
		--- Return the vertical text justification.
		-- @name FontFrame:GetJustifyV
		-- @class function
		-- @return the font instance's vertical text alignment style
		-- @usage FontFrame:GetJustifyV()
		------------------------------------

		-- GetJustifyV
        ["GetJustifyV"] = function(self)
            return self.__UI:GetJustifyV()
        end,

		------------------------------------
		--- Returns the color of the font's text shadow
		-- @name FontFrame:GetShadowColor
		-- @class function
		-- @return shadowR Red component of the shadow color (0.0 - 1.0)
		-- @return shadowG Green component of the shadow color (0.0 - 1.0)
		-- @return shadowB Blue component of the shadow color (0.0 - 1.0)
		-- @return shadowAlpha Alpha (opacity) of the text's shadow (0.0 = fully transparent, 1.0 = fully opaque)
		-- @usage FontFrame:GetShadowColor()
		------------------------------------

		-- GetShadowColor
        ["GetShadowColor"] = function(self)
            return self.__UI:GetShadowColor()
        end,

		------------------------------------
		--- Returns the offset of the font instance's text shadow from its text
		-- @name FontFrame:GetShadowOffset
		-- @class function
		-- @return xOffset Horizontal distance between the text and its shadow (in pixels)
		-- @return yOffset Vertical distance between the text and its shadow (in pixels)
		-- @usage FontFrame:GetShadowOffset()
		------------------------------------

		-- GetShadowOffset
        ["GetShadowOffset"] = function(self)
            return self.__UI:GetShadowOffset()
        end,

		------------------------------------
		--- Returns the font instance's amount of spacing between lines
		-- @name FontFrame:GetSpacing
		-- @class function
		-- @return spacing Amount of space between lines of text (in pixels)
		-- @usage FontFrame:GetSpacing()
		------------------------------------

		-- GetSpacing
        ["GetSpacing"] = function(self)
            return self.__UI:GetSpacing()
        end,

		------------------------------------
		--- Returns the font instance's default text color
		-- @name FontFrame:GetTextColor
		-- @class function
		-- @return textR Red component of the text color (0.0 - 1.0)
		-- @return textG Green component of the text color (0.0 - 1.0)
		-- @return textB Blue component of the text color (0.0 - 1.0)
		-- @return textAlpha Alpha (opacity) of the text (0.0 = fully transparent, 1.0 = fully opaque)
		-- @usage FontFrame:GetTextColor()
		------------------------------------

		-- GetTextColor
        ["GetTextColor"] = function(self)
            return self.__UI:GetTextColor()
        end,

		------------------------------------
		--- Sets the font to use for text, returns 1 if the path was valid, nil otherwise(no change occurs).
		-- @name FontFrame:SetFont
		-- @class function
		-- @param filename Path to a font file
		-- @param fontHeight Height (point size) of the font to be displayed (in pixels)
		-- @param flags Additional properties for the font specified by one or more (separated by commas) of the following tokens:MONOCHROME,OUTLINE,THICKOUTLINE
		-- @return 1 if filename refers to a valid font file; otherwise nil
		-- @usage FontFrame:SetFont("Fonts\\ARIALN.TTF", 10)
		------------------------------------

		-- SetFont
        ["SetFont"] = function(self, path, height, flags)
            self.__UI:SetFont(path, height, flags)
        end,

		------------------------------------
		--- Sets the 'parent' Font object from which this object inherits properties.
		-- @name FontFrame:SetFontObject
		-- @class function
		-- @param object Reference to a Font object Or Global name of a Font object
		-- @usage FontFrame:SetFontObject(GameFontNormal)
		------------------------------------

		-- SetFontObject
        ["SetFontObject"] = function(self, fontObject)
            self.__UI:SetFontObject(IGAS:GetUI(fontObject))
        end,

		------------------------------------
		--- Sets horizontal text justification ("LEFT","RIGHT", or "CENTER")
		-- @name FontFrame:SetJustifyH
		-- @class function
		-- @param justify Horizontal text alignment style
		-- @usage FontFrame:SetJustifyH("CENTER")
		------------------------------------

		-- SetJustifyH
        ["SetJustifyH"] = function(self, justifyH)
            self.__UI:SetJustifyH(justifyH)
        end,

		------------------------------------
		--- Sets vertical text justification ("TOP","BOTTOM", or "MIDDLE")
		-- @name FontFrame:SetJustifyV
		-- @class function
		-- @param justify Vertical text alignment style
		-- @usage FontFrame:SetJustifyV("BOTTOM")
		------------------------------------

		-- SetJustifyV
        ["SetJustifyV"] = function(self, justifyV)
            self.__UI:SetJustifyV(justifyV)
        end,

		------------------------------------
		--- Sets the text shadow color.
		-- @name FontFrame:SetShadowColor
		-- @class function
		-- @param shadowR Red component of the shadow color (0.0 - 1.0)
		-- @param shadowG Green component of the shadow color (0.0 - 1.0)
		-- @param shadowB Blue component of the shadow color (0.0 - 1.0)
		-- @param shadowAlpha Alpha (opacity) of the text's shadow (0.0 = fully transparent, 1.0 = fully opaque)
		-- @usage FontFrame:SetShadowColor(1, 1, 1, 1)
		------------------------------------

		-- SetShadowColor
        ["SetShadowColor"] = function(self, r,g,b,a)
            self.__UI:SetShadowColor(r,g,b,a)
        end,

		------------------------------------
		--- Sets the text shadow offset.
		-- @name FontFrame:SetShadowOffset
		-- @class function
		-- @param xOffset Horizontal distance between the text and its shadow (in pixels)
		-- @param yOffset Vertical distance between the text and its shadow (in pixels)
		-- @usage FontFrame:SetShadowOffset(5, -5)
		------------------------------------

		-- SetShadowOffset
        ["SetShadowOffset"] = function(self, x,y)
            self.__UI:SetShadowOffset(x,y)
        end,

		------------------------------------
		--- Sets the spacing between lines of text in the object.
		-- @name FontFrame:SetSpacing
		-- @class function
		-- @param spacing the spacing between lines
		-- @usage FontFrame:SetSpacing(3)
		------------------------------------

		-- SetSpacing
        ["SetSpacing"] = function(self, spacing)
            self.__UI:SetSpacing(spacing)
        end,

		------------------------------------
		--- Sets the default text color.
		-- @name FontFrame:SetTextColor
		-- @class function
		-- @param textR Red component of the text color (0.0 - 1.0)
		-- @param textG Green component of the text color (0.0 - 1.0)
		-- @param textB Blue component of the text color (0.0 - 1.0)
		-- @param textAlpha Alpha (opacity) of the text (0.0 = fully transparent, 1.0 = fully opaque)
		-- @usage FontFrame:SetTextColor(1, 1, 1, 0.8)
		------------------------------------

		-- SetTextColor
        ["SetTextColor"] = function(self, r,g,b,a)
            self.__UI:SetTextColor(r,g,b,a)
        end,
    }

	-- Property
	local _Property = {
		-- Font
		["Font"] = {
			["Set"] = function(self, font)
				local flags = ""
				if font.outline then
					if font.outline == "NORMAL" then
						flags = flags.."OUTLINE"
					elseif font.outline == "THICK" then
						flags = flags.."THICKOUTLINE"
					end
				end
				if font.monochrome then
					if flags ~= "" then
						flags = flags..","
					end
					flags = flags.."MONOCHROME"
				end

				self:SetFont(font.path, font.size, flags)
			end,

			["Get"] = function(self)
				local font = {}
				local flags
				font.path, font.size, flags = self:GetFont()
				flags = flags or ""
				if string.find(flags, "THICKOUTLINE") then
					font.outline = "THICK"
				elseif string.find(flags, "OUTLINE") then
					font.outline = "NORMAL"
				end
				if string.find(flags, "MONOCHROME") then
					font.monochrome = true
				end

				return font
			end,

			["Type"] = "FontTable",
		},
		-- FontObject
		["FontObject"] = {
			["Set"] = function(self, fontObject)
				self:SetFontObject(fontObject)
			end,

			["Get"] = function(self)
				return self:GetFontObject()
			end,

			["Type"] = "Font",
		},
		-- JustifyH
		["JustifyH"] = {
			["Set"] = function(self, justifyH)
				self:SetJustifyH(justifyH)
			end,

			["Get"] = function(self)
				return self:GetJustifyH()
			end,

			["Type"] = "JustifyH",
		},
		-- JustifyV
		["JustifyV"] = {
			["Set"] = function(self, justifyV)
				self:SetJustifyV(justifyV)
			end,

			["Get"] = function(self)
				return self:GetJustifyV()
			end,

			["Type"] = "JustifyV",
		},
		-- ShadowColor
		["ShadowColor"] = {
			["Set"] = function(self, color)
				self:SetShadowColor(color.red, color,green, color.blue, color.alpha)
			end,

			["Get"] = function(self)
				local color = {}
				color.red, color,green, color.blue, color.alpha = self:GetShadowColor()

				return color
			end,

			["Type"] = "Color",
		},
		-- ShadowOffset
		["ShadowOffset"] = {
			["Set"] = function(self, offset)
				self:SetShadowOffset(offset.x, offset.y)
			end,

			["Get"] = function(self)
				local offset = {}
				offset.x, offset.y = self:GetShadowOffset()

				return offset
			end,

			["Type"] = "Offset",
		},
		-- Spacing
		["Spacing"] = {
			["Set"] = function(self, spacing)
				self:SetSpacing(spacing)
			end,

			["Get"] = function(self)
				return self:GetSpacing()
			end,

			["Type"] = "number",
		},
		-- TextColor
		["TextColor"] = {
			["Set"] = function(self, color)
				self:SetTextColor(color.red, color,green, color.blue, color.alpha)
			end,

			["Get"] = function(self)
				local color = {}
				color.red, color,green, color.blue, color.alpha = self:GetTextColor()

				return color
			end,

			["Type"] = "Color",
		},
	}

    local _WidgetInfo = {
        ["WidgetName"] = _WidgetName,
        ["Base"] = _Base,
        ["FuncProxy"] = _FuncProxy,
        ["Property"] = _Property,
    }

    IGAS:NewWidget(_WidgetInfo, true)
end
