﻿-- Author      : Kurapica
-- Create Date : 7/16/2008 15:01

----------------------------------------------------------------------------------------------------------------------------------------
--- CheckButtons are a specialized form of Button; they maintain an on/off state, which toggles automatically when they are clicked, and additional textures for when they are checked, or checked while disabled.
-- <br><br>inherit <a href=".\Button.html">Button</a> For all methods, properties and scriptTypes
-- @name CheckButton
-- @class table
-- @field Checked true if the checkbutton is checked
-- @field CheckedTexture the texture used when the button is checked
-- @field CheckedTexturePath the texture file used when the button is checked
-- @field DisabledCheckedTexture the texture used when the button is disabled and checked
-- @field DisabledCheckedTexturePath the texture file used when the button is disabled and checked
----------------------------------------------------------------------------------------------------------------------------------------
do
	-- Check Version
	local version = 2
	if not IGAS:NewAddon("IGAS.GUI.CheckButton", version) then
		return
	end

    local CreateFrame = CreateFrame

	local _WidgetName = "CheckButton"
    local _Base = "Button"

	-- ScriptType
	local _ScriptType = {
	}

	-- FuncProxy
	local _FuncProxy = {
		------------------------------------
		--- Get the status of the checkbox.
		-- @name CheckButton:GetChecked
		-- @class function
		-- @return 1 if the button is checked; nil if the button is unchecked
		-- @usage CheckButton:GetChecked()
		------------------------------------

		-- GetChecked
		["GetChecked"] = function(self, ...)
			return self.__UI:GetChecked(...)
		end,

		------------------------------------
		--- Get the texture used for a checked box
		-- @name CheckButton:GetCheckedTexture
		-- @class function
		-- @return the texture used for a checked box
		-- @usage CheckButton:GetCheckedTexture()
		------------------------------------

		-- GetCheckedTexture
		["GetCheckedTexture"] = function(self, ...)
			return IGAS:GetWrapper(self.__UI:GetCheckedTexture(...))
		end,

		------------------------------------
		--- Get the texture used for a disabled checked box
		-- @name CheckButton:GetDisabledCheckedTexture
		-- @class function
		-- @return the texture used for a disabled checked box
		-- @usage CheckButton:GetDisabledCheckedTexture()
		------------------------------------

		-- GetDisabledCheckedTexture
		["GetDisabledCheckedTexture"] = function(self, ...)
			return IGAS:GetWrapper(self.__UI:GetDisabledCheckedTexture(...))
		end,

		------------------------------------
		--- Set the status of the checkbox.
		-- @name CheckButton:SetChecked
		-- @class function
		-- @param enable True to check the button; false to uncheck
		-- @usage CheckButton:SetChecked(true)
		------------------------------------

		-- SetChecked
		["SetChecked"] = function(self, ...)
			return self.__UI:SetChecked(...)
		end,

		------------------------------------
		--- Set the texture to use for a checked box.
		-- @name CheckButton:SetCheckedTexture
		-- @class function
		-- @param texture Reference to an existing Texture object or Path to a texture image file
		-- @usage CheckButton:SetCheckedTexture("Interface\\Buttons\\UI-CheckBox-Check.blp")
		------------------------------------

		-- SetChecked
		["SetCheckedTexture"] = function(self, texture, ...)
			return self.__UI:SetCheckedTexture(IGAS:GetUI(texture), ...)
		end,

		------------------------------------
		--- Set the texture to use for a disabled but checked box.
		-- @name CheckButton:SetDisabledCheckedTexture
		-- @class function
		-- @param texture Reference to an existing Texture object or Path to a texture image file
		-- @usage CheckButton:SetDisabledCheckedTexture("Interface\\Buttons\\UI-CheckBox-Check-Disabled.blp")
		------------------------------------

		-- SetDisabledCheckedTexture
		["SetDisabledCheckedTexture"] = function(self, texture, ...)
			return self.__UI:SetDisabledCheckedTexture(IGAS:GetUI(texture), ...)
		end,

		-- Override
        ["Enable"] = function(self)
            self.__UI:Enable()
            self:SetDisabledCheckedTexture("Interface\\Buttons\\UI-CheckBox-Check.blp")
        end,
        ["Disable"] = function(self)
            self.__UI:Disable()
            self:SetDisabledCheckedTexture("Interface\\Buttons\\UI-CheckBox-Check-Disabled.blp")
        end,
    }

	-- Property
	local _Property = {
        -- Style
        ["Style"] = {
			["Get"] = function(self)
				return nil
			end,
		},
		
        -- Checked
        ["Checked"] = {
			["Set"] = function(self, value)
				self.__UI:SetChecked(value)
			end,

			["Get"] = function(self)
				return (self.__UI:GetChecked() and true) or false
			end,

			["Type"] = "boolean",
		},

        -- CheckedTexture
        ["CheckedTexture"] = {
			["Set"] = function(self, texture)
				self:SetCheckedTexture(texture)
			end,

			["Get"] = function(self)
				return self:GetCheckedTexture()
			end,

			["Type"] = "Texture",
		},

        -- CheckedTexturePath
        ["CheckedTexturePath"] = {
			["Set"] = function(self, texture)
				self:SetCheckedTexture(texture)
			end,

			["Get"] = function(self)
				return self:GetCheckedTexture() and self:GetCheckedTexture().TexturePath
			end,

			["Type"] = "TexturePath",
		},
		
        -- DisabledCheckedTexture
        ["DisabledCheckedTexture"] = {
			["Set"] = function(self, texture)
				self:SetDisabledCheckedTexture(texture)
			end,

			["Get"] = function(self)
				return self:GetDisabledCheckedTexture()
			end,

			["Type"] = "Texture",
		},
		
        -- DisabledCheckedTexturePath
        ["DisabledCheckedTexturePath"] = {
			["Set"] = function(self, texture)
				self:SetDisabledCheckedTexture(texture)
			end,

			["Get"] = function(self)
				return self:GetDisabledCheckedTexture() and self:GetDisabledCheckedTexture().TexturePath
			end,

			["Type"] = "TexturePath",
		},
    }

    -- New
    local _New = function(_parent,inheritFrame)
        -- New Frame
		local frame = CreateFrame(_WidgetName,nil,IGAS:GetUI(_parent),inheritFrame)

        frame:SetNormalTexture("Interface\\Buttons\\UI-CheckBox-Up.blp")
        frame:SetPushedTexture("Interface\\Buttons\\UI-CheckBox-Down.blp")
        frame:SetHighlightTexture("Interface\\Buttons\\UI-CheckBox-Highlight.blp", "ADD")
        frame:SetCheckedTexture("Interface\\Buttons\\UI-CheckBox-Check.blp")
        frame:SetDisabledCheckedTexture("Interface\\Buttons\\UI-CheckBox-Check.blp")
        --frame:SetDisabledCheckedTexture("Interface\\Buttons\\UI-CheckBox-Check-Disabled.blp")

        frame:SetWidth(24)
		frame:SetHeight(24)

        return frame
    end

    local _WidgetInfo = {
        ["WidgetName"] = _WidgetName,
        ["Base"] = _Base,
        ["FuncProxy"] = _FuncProxy,
        ["Property"] = _Property,
        ["ScriptType"] = _ScriptType,
        ["New"] = _New,
    }

    IGAS:NewWidget(_WidgetInfo, true, true)
end