﻿-- Author      : Kurapica
-- Create Date : 7/12/2008 14:38

----------------------------------------------------------------------------------------------------------------------------------------
--- Button is the primary means for users to control the game and their characters.
-- <br><br>inherit <a href=".\Frame.html">Frame</a> For all methods, properties and scriptTypes
-- @name Button
-- @class table
-- @field Style the style of the button:NONE, NORMAL, GRAY, CLOSE, PLUS, MINUS
-- @field Enabled true if the button is enabled
-- @field ButtonState the button's current state: DISABLED, NORMAL, PUSHED
-- @field Locked true if the button is locked
-- @field HighlightLocked true if the button's highlight state is locked
-- @field DisabledTexture the texture used when the button is disabled
-- @field DisabledTexturePath the texture file used when the button is disabled
-- @field HighlightTexture the texture used when the button is highlighted
-- @field HighlightTexturePath the texture file used when the button is highlighted
-- @field NormalTexture the texture used for the button's normal state
-- @field NormalTexturePath the texture file used for the button's normal state
-- @field PushedTexture the texture used when the button is pushed
-- @field PushedTexturePath the texture file used when the button is pushed
-- @field FontString the `FontString` object used for the button's label text
-- @field DisabledFontObject the font object used for the button's disabled state
-- @field HighlightFontObject the font object used when the button is highlighted
-- @field NormalFontObject the font object used for the button's normal state
-- @field PushedTextOffset the offset for moving the button's label text when pushed
-- @field Text the text displayed as the button's label
----------------------------------------------------------------------------------------------------------------------------------------
do
	-- Check Version
	local version = 2
	if not IGAS:NewAddon("IGAS.GUI.Button", version) then
		return
	end

    local CreateFrame = CreateFrame

	local _WidgetName = "Button"
    local _Base = "Frame"

    -- Button Template
    local TEMPLATE_NONE = "NONE"
    local TEMPLATE_NORMAL = "NORMAL"
    local TEMPLATE_GRAY = "GRAY"
    local TEMPLATE_CLOSEBTN = "CLOSE"
	local TEMPLATE_ADD = "PLUS"
	local TEMPLATE_MINUS = "MINUS"

    -- Define Block
	local _StyleEnum = IGAS:NewEnum("ButtonStyle", {
        TEMPLATE_NONE,
        TEMPLATE_NORMAL,
        TEMPLATE_GRAY,
        TEMPLATE_CLOSEBTN,
		TEMPLATE_ADD,
		TEMPLATE_MINUS,
    })

	local _StyleType = IGAS:NewPropertyType("ButtonStyle", {
		["Type"] = "enum",
		["EnumType"] = _StyleEnum,
	})

    -- Scripts
    local function OnClick_Close(self)
		if not self.Parent.InDesignMode then
			self.Parent.Visible = false
		end
    end

	-- ScriptType
	local _ScriptType = {
		------------------------------------
		--- ScriptType, Run when the button is clicked
		-- @name Button:OnClick
		-- @class function
		-- @param button Name of the mouse button responsible for the click action:Button4, Button5, LeftButton, MiddleButton, RightButton
		-- @param down True for a mouse button down action; false for button up or other actions
		-- @usage function Button:OnClick(button, down)<br>
		--    -- do someting<br>
		-- end
		------------------------------------
		["OnClick"] = true,

		------------------------------------
		--- ScriptType, Run when the button is double-clicked
		-- @name Button:OnDoubleClick
		-- @class function
		-- @param button Name of the mouse button responsible for the click action:Button4, Button5, LeftButton, MiddleButton, RightButton
		-- @usage function Button:OnDoubleClick(button)<br>
		--    -- do someting<br>
		-- end
		------------------------------------
		["OnDoubleClick"] = true,

		------------------------------------
		--- ScriptType, Run immediately following the button's `OnClick` handler with the same arguments
		-- @name Button:PostClick
		-- @class function
		-- @param button Name of the mouse button responsible for the click action:Button4, Button5, LeftButton, MiddleButton, RightButton
		-- @param down True for a mouse button down action; false for button up or other actions
		-- @usage function Button:PostClick(button, down)<br>
		--    -- do someting<br>
		-- end
		------------------------------------
		["PostClick"] = true,

		------------------------------------
		--- ScriptType, Run immediately before the button's `OnClick` handler with the same arguments
		-- @name Button:PreClick
		-- @class function
		-- @param button Name of the mouse button responsible for the click action:Button4, Button5, LeftButton, MiddleButton, RightButton
		-- @param down True for a mouse button down action; false for button up or other actions
		-- @usage function Button:PreClick(button, down)<br>
		--    -- do someting<br>
		-- end
		------------------------------------
		["PreClick"] = true,
	}

	-- FuncProxy
	local _FuncProxy = {
		------------------------------------
		--- Execute the click action of the button.
		-- @name Button:Click
		-- @class function
		-- @usage Button:Click()
		------------------------------------

		-- Click
        ["Click"] = function(self, ...)
            return self.__UI:Click(...)
        end,

		------------------------------------
		--- Disable the Button so that it cannot be clicked.
		-- @name Button:Disable
		-- @class function
		-- @usage Button:Disable()
		------------------------------------

		-- Click
        ["Disable"] = function(self, ...)
            return self.__UI:Disable(...)
        end,

		------------------------------------
		--- Enable to the Button so that it may be clicked.
		-- @name Button:Enable
		-- @class function
		-- @usage Button:Enable()
		------------------------------------

		-- Enable
        ["Enable"] = function(self, ...)
            return self.__UI:Enable(...)
        end,

		------------------------------------
		--- Return the current state ("PUSHED","NORMAL") of the Button.
		-- @name Button:GetButtonState
		-- @class function
		-- @return the current state
		-- @usage Button:GetButtonState()
		------------------------------------

		-- GetButtonState
        ["GetButtonState"] = function(self, ...)
            return self.__UI:GetButtonState(...)
        end,

		------------------------------------
		--- Return the font object for the Button when disabled
		-- @name Button:GetDisabledFontObject
		-- @class function
		-- @return the font object for the Button when disabled
		-- @usage Button:GetDisabledFontObject()
		------------------------------------

		-- GetDisabledFontObject
        ["GetDisabledFontObject"] = function(self)
            return IGAS:GetWrapper(self.__UI:GetDisabledFontObject())
        end,

		------------------------------------
		--- Get the texture for this button when disabled
		-- @name Button:GetDisabledTexture
		-- @class function
		-- @return the texture for this button when disabled
		-- @usage Button:GetDisabledTexture()
		------------------------------------

		-- GetDisabledTexture
        ["GetDisabledTexture"] = function(self, ...)
            return IGAS:GetWrapper(self.__UI:GetDisabledTexture(...))
        end,

		------------------------------------
		--- Get this button's label FontString
		-- @name Button:GetFontString
		-- @class function
		-- @return this button's label FontString
		-- @usage Button:GetFontString()
		------------------------------------

		-- GetFontString
        ["GetFontString"] = function(self, ...)
            return IGAS:GetWrapper(self.__UI:GetFontString(...))
        end,

		------------------------------------
		--- Return the font object for the Button when highlighted
		-- @name Button:GetHighlightFontObject
		-- @class function
		-- @return the font object for the Button when highlighted
		-- @usage Button:GetHighlightFontObject()
		------------------------------------

		-- GetHighlightFontObject
        ["GetHighlightFontObject"] = function(self)
            return IGAS:GetWrapper(self.__UI:GetHighlightFontObject())
        end,

		------------------------------------
		--- Get the texture for this button when highlighted
		-- @name Button:GetHighlightTexture
		-- @class function
		-- @return the texture for this button when highlighted
		-- @usage Button:GetHighlightTexture()
		------------------------------------

		-- GetHighlightTexture
        ["GetHighlightTexture"] = function(self, ...)
            return IGAS:GetWrapper(self.__UI:GetHighlightTexture(...))
        end,

		------------------------------------
		--- Get the normal texture for this button
		-- @name Button:GetNormalTexture
		-- @class function
		-- @return the normal texture for this button
		-- @usage Button:GetNormalTexture()
		------------------------------------

		-- GetNormalTexture
        ["GetNormalTexture"] = function(self, ...)
            return IGAS:GetWrapper(self.__UI:GetNormalTexture(...))
        end,

		------------------------------------
		--- Get the Normal Font Object of the button
		-- @name Button:GetNormalFontObject
		-- @class function
		-- @return the Normal Font Object of the button
		-- @usage Button:GetNormalFontObject()
		------------------------------------

		-- GetNormalFontObject
        ["GetNormalFontObject"] = function(self)
            return IGAS:GetWrapper(self.__UI:GetNormalFontObject())
        end,

		------------------------------------
		--- Get the text offset when this button is pushed (x, y)
		-- @name Button:GetPushedTextOffset
		-- @class function
		-- @return the text offset when this button is pushed
		-- @usage Button:GetPushedTextOffset()
		------------------------------------

		-- GetPushedTextOffset
        ["GetPushedTextOffset"] = function(self, ...)
            return self.__UI:GetPushedTextOffset(...)
        end,

		------------------------------------
		--- Get the texture for this button when pushe
		-- @name Button:GetPushedTexture
		-- @class function
		-- @return the texture for this button when pushe
		-- @usage Button:GetPushedTexture()
		------------------------------------

		-- GetPushedTexture
        ["GetPushedTexture"] = function(self, ...)
            return IGAS:GetWrapper(self.__UI:GetPushedTexture(...))
        end,

		------------------------------------
		--- Get the text label for the Button.
		-- @name Button:GetText
		-- @class function
		-- @return the text label for the Button
		-- @usage Button:GetText()
		------------------------------------

		-- GetText
        ["GetText"] = function(self, ...)
            return self.__UI:GetText(...)
        end,

		------------------------------------
		--- Get the height of the Button's text.
		-- @name Button:GetTextHeight
		-- @class function
		-- @return the height of the Button's text.
		-- @usage Button:GetTextHeight()
		------------------------------------

		-- GetTextHeight
        ["GetTextHeight"] = function(self, ...)
            return self.__UI:GetTextHeight(...)
        end,

		------------------------------------
		--- Get the width of the Button's text.
		-- @name Button:GetTextWidth
		-- @class function
		-- @return the width of the Button's text
		-- @usage Button:GetTextWidth()
		------------------------------------

		-- GetTextWidth
        ["GetTextWidth"] = function(self, ...)
            return self.__UI:GetTextWidth(...)
        end,

		------------------------------------
		--- Determine whether the Button is enabled.
		-- @name Button:IsEnabled
		-- @class function
		-- @return true if the button is enabled
		-- @usage Button:IsEnabled()
		------------------------------------

		-- IsEnabled
        ["IsEnabled"] = function(self)
            return (self.__UI:IsEnabled() and true) or false
        end,

		------------------------------------
		--- Set the Button to always be drawn highlighted.
		-- @name Button:LockHighlight
		-- @class function
		-- @usage Button:LockHighlight()
		------------------------------------

		-- LockHighlight
        ["LockHighlight"] = function(self, ...)
            return self.__UI:LockHighlight(...)
        end,

		------------------------------------
		--- Specify which mouse button up/down actions cause this button to receive an OnClick notification
		-- @name Button:RegisterForClicks
		-- @class function
		-- @param clickType Button4Down,Button4Up,Button5Down,Button5Up,LeftButtonDown,LeftButtonUp,MiddleButtonDown,MiddleButtonUp,RightButtonDown,RightButtonUp,AnyDown,AnyUp
		-- @param clickType, ... Optional,other type to registered
		-- @usage Button:RegisterForClicks("AnyDown", "AnyUp")
		------------------------------------

		-- RegisterForClicks
        ["RegisterForClicks"] = function(self, ...)
            return self.__UI:RegisterForClicks(...)
        end,

		------------------------------------
		--- Set the state of the Button ("PUSHED", "NORMAL") and whether it is locked.
		-- @name Button:SetButtonState
		-- @class function
		-- @param state state for the button: "PUSHED", "NORMAL"
		-- @param locked Optional,Locks the button in the given state
		-- @usage Button:SetButtonState("PUSHED", true)
		------------------------------------

		-- SetButtonState
        ["SetButtonState"] = function(self, state, locked)
            if locked ~= nil then
                self.__Locked = locked
            end

            self.__UI:SetButtonState(state, self.__Locked)
        end,

		------------------------------------
		--- Set the font object for settings when disabled
		-- @name Button:SetDisabledFontObject
		-- @class function
		-- @param font Optional,Reference to a Font object to be used when the button is disabled
		-- @usage Button:SetDisabledFontObject(GameFontNormal)
		------------------------------------

		-- SetDisabledFontObject
        ["SetDisabledFontObject"] = function(self, font)
            return self.__UI:SetDisabledFontObject(IGAS:GetUI(font))
        end,

		------------------------------------
		--- Set the disabled texture for the Button
		-- @name Button:SetDisabledTexture
		-- @class function
		-- @param texture Reference to an existing Texture object or Path to a texture image file
		-- @usage Button:SetDisabledTexture("Interface\\Buttons\\UI-Panel-Button-Disabled")
		------------------------------------

		-- SetDisabledTexture
        ["SetDisabledTexture"] = function(self, texture, ...)
            return self.__UI:SetDisabledTexture(IGAS:GetUI(texture), ...)
        end,

		------------------------------------
		--- Set the button's label FontString
		-- @name Button:SetFontString
		-- @class function
		-- @param fontString the `FontString` object used for the button's label text
		-- @usage Button:SetFontString(IGAS:NewFrame("FontString", "MyFontString", Button))
		------------------------------------

		-- SetFontString
        ["SetFontString"] = function(self, fontString)
            return self.__UI:SetFontString(IGAS:GetUI(fontString))
        end,

		------------------------------------
		--- Set the formatted text label for the Button.
		-- @name Button:SetFormattedText
		-- @class function
		-- @param formatstring A string containing format specifiers
		-- @param ... Optional,A list of values to be included in the formatted string
		-- @usage Button:SetFormattedText("%s %q", "Hello", "Azeroth!")
		------------------------------------

		-- SetFormattedText
        ["SetFormattedText"] = function(self, ...)
            return self.__UI:SetFormattedText(...)
        end,

		------------------------------------
		--- Set the font object for settings when highlighted
		-- @name Button:SetHighlightFontObject
		-- @class function
		-- @param font Optional,Reference to a Font object to be used when the button is disabled
		-- @usage Button:SetHighlightFontObject(GameFontHighlight)
		------------------------------------

		-- SetHighlightFontObject
        ["SetHighlightFontObject"] = function(self, font)
            return self.__UI:SetHighlightFontObject(IGAS:GetUI(font))
        end,

		------------------------------------
		--- Set the highlight texture for the Button
		-- @name Button:SetHighlightTexture
		-- @class function
		-- @param texture Reference to an existing Texture object or Path to a texture image file
		-- @param alphaMode Blend mode for the texture; defaults to ADD if omitted
		-- @usage Button:SetHighlightTexture("Interface\\Buttons\\UI-Panel-Button-Highlight")
		------------------------------------

		-- SetHighlightTexture
        ["SetHighlightTexture"] = function(self, texture, ...)
            return self.__UI:SetHighlightTexture(IGAS:GetUI(texture), ...)
        end,

		------------------------------------
		--- Set the normal texture for the Button
		-- @name Button:SetNormalTexture
		-- @class function
		-- @param texture Reference to an existing Texture object or Path to a texture image file
		-- @usage Button:SetNormalTexture("Interface\\Buttons\\UI-Panel-Button-Up")
		------------------------------------

		-- SetNormalTexture
        ["SetNormalTexture"] = function(self, texture, ...)
            return self.__UI:SetNormalTexture(IGAS:GetUI(texture), ...)
        end,

		------------------------------------
		--- Set the normal font for the Button
		-- @name Button:SetNormalFontObject
		-- @class function
		-- @param font Reference to a Font object to be used in the button's normal state
		-- @usage Button:SetNormalFontObject(GameFontNormal)
		------------------------------------

		-- SetNormalFontObject
        ["SetNormalFontObject"] = function(self, font)
            return self.__UI:SetNormalFontObject(IGAS:GetUI(font))
        end,

		------------------------------------
		--- Set the text offset for this button when pushed
		-- @name Button:SetPushedTextOffset
		-- @class function
		-- @param x Horizontal offset for the text (in pixels; values increasing to the right)
		-- @param y Vertical offset for the text (in pixels; values increasing upward)
		-- @usage Button:SetPushedTextOffset(1, 1)
		------------------------------------

		-- SetPushedTextOffset
        ["SetPushedTextOffset"] = function(self, ...)
            return self.__UI:SetPushedTextOffset(...)
        end,

		------------------------------------
		--- Set the text offset for this button when pushed
		-- @name Button:SetPushedTexture
		-- @class function
		-- @param texture Reference to an existing Texture object or Path to a texture image file
		-- @usage Button:SetPushedTexture("Interface\\Buttons\\UI-Panel-Button-Down")
		------------------------------------

		-- SetPushedTexture
        ["SetPushedTexture"] = function(self, texture, ...)
            return self.__UI:SetPushedTexture(IGAS:GetUI(texture), ...)
        end,

		------------------------------------
		--- Set the text label for the Button.
		-- @name Button:SetText
		-- @class function
		-- @param text Text to be displayed as the button's label
		-- @usage Button:SetText("Hello World")
		------------------------------------

		-- SetText
        ["SetText"] = function(self, ...)
            return self.__UI:SetText(...)
        end,

		------------------------------------
		--- Set the Button to not always be drawn highlighted.
		-- @name Button:UnlockHighlight
		-- @class function
		-- @usage Button:UnlockHighlight()
		------------------------------------

		-- UnlockHighlight
        ["UnlockHighlight"] = function(self, ...)
            return self.__UI:UnlockHighlight(...)
        end,

		------------------------------------
		--- Set the Button's Style
		-- @name Button:SetStyle
		-- @class function
		-- @param style the button's style, NONE, NORMAL, GRAY, CLOSE, PLUS, MINUS
		-- @usage Button:SetStyle("NORMAL")
		------------------------------------

		-- SetStyle
        ["SetStyle"] = function(self, style)
            local t

            -- Check Style
            if not style or type(style) ~= "string" then
                return
            end

            if (not _StyleEnum[style]) or style == self.__Style then
                return
            end

            self.__Style = style

            -- Change Style
            if style == TEMPLATE_NORMAL then
                --- Font
                self:SetNormalFontObject(GameFontNormal)
                self:SetDisabledFontObject(GameFontDisable)
                self:SetHighlightFontObject(GameFontHighlight)

                --- Texture
                self:SetNormalTexture("Interface\\Buttons\\UI-Panel-Button-Up")
                t = self:GetNormalTexture()
                t:SetTexCoord(0,0.625,0,0.6875)

                self:SetPushedTexture("Interface\\Buttons\\UI-Panel-Button-Down")
                t = self:GetPushedTexture()
                t:SetTexCoord(0,0.625,0,0.6875)

                self:SetDisabledTexture("Interface\\Buttons\\UI-Panel-Button-Disabled")
                t = self:GetDisabledTexture()
                t:SetTexCoord(0,0.625,0,0.6875)

                self:SetHighlightTexture("Interface\\Buttons\\UI-Panel-Button-Highlight")
                t = self:GetHighlightTexture()
                t:SetTexCoord(0,0.625,0,0.6875)
            elseif style == TEMPLATE_GRAY then
                --- Font
                self:SetNormalFontObject(GameFontHighlight)
                self:SetDisabledFontObject(GameFontDisable)
                self:SetHighlightFontObject(GameFontHighlight)

                --- Texture
                self:SetNormalTexture("Interface\\Buttons\\UI-Panel-Button-Disabled")
                t = self:GetNormalTexture()
                t:SetTexCoord(0,0.625,0,0.6875)

                self:SetPushedTexture("Interface\\Buttons\\UI-Panel-Button-Disabled-Down")
                t = self:GetPushedTexture()
                t:SetTexCoord(0,0.625,0,0.6875)

                self:SetDisabledTexture("Interface\\Buttons\\UI-Panel-Button-Disabled")
                t = self:GetDisabledTexture()
                t:SetTexCoord(0,0.625,0,0.6875)

                self:SetHighlightTexture("Interface\\Buttons\\UI-Panel-Button-Highlight")
                t = self:GetHighlightTexture()
                t:SetTexCoord(0,0.625,0,0.6875)
            elseif style == TEMPLATE_CLOSEBTN then
                --- Texture
                self:SetNormalTexture("Interface\\Buttons\\UI-Panel-MinimizeButton-Up")

                self:SetPushedTexture("Interface\\Buttons\\UI-Panel-MinimizeButton-Down")

                self:SetHighlightTexture("Interface\\Buttons\\UI-Panel-MinimizeButton-Highlight")
                t = self:GetHighlightTexture()
                t:SetBlendMode("ADD")

                self.Width = 32
                self.Height = 32
                self.OnClick = OnClick_Close
			elseif style == TEMPLATE_ADD then
				self:SetNormalTexture("Interface\\Buttons\\UI-PlusButton-UP")
				self:SetPushedTexture("Interface\\Buttons\\UI-PlusButton-DOWN")
				self:SetHighlightTexture("Interface\\Buttons\\UI-PlusButton-Hilight")
				t = self:GetHighlightTexture()
				t:SetBlendMode("ADD")
				self.Width = 24
				self.Height = 24
			elseif style == TEMPLATE_MINUS then
				self:SetNormalTexture("Interface\\Buttons\\UI-MinusButton-UP")
				self:SetPushedTexture("Interface\\Buttons\\UI-MinusButton-DOWN")
				self:SetHighlightTexture("Interface\\Buttons\\UI-PlusButton-Hilight")
				t = self:GetHighlightTexture()
				t:SetBlendMode("ADD")
				self.Width = 24
				self.Height = 24
            end
        end,

		------------------------------------
		--- Get the Button's Style
		-- @name Button:GetStyle
		-- @class function
		-- @return the button's style, NONE, NORMAL, GRAY, CLOSE, PLUS, MINUS
		-- @usage Button:GetStyle()
		------------------------------------

		-- GetStyle
        ["GetStyle"] = function(self)
            return self.__Style or TEMPLATE_NONE
        end,
    }

	-- Property
	local _Property = {
        -- Style
        ["Style"] = {
			["Set"] = function(self, style)
                self:SetStyle(style)
			end,

			["Get"] = function(self)
				return self:GetStyle()
			end,

			["Type"] = _StyleType,
		},

        -- Enabled
        ["Enabled"] = {
			["Set"] = function(self, enabled)
                if enabled then
				    self:Enable()
                else
				    self:Disable()
                end
			end,

			["Get"] = function(self)
				return self:IsEnabled()
			end,

			["Type"] = "boolean",
		},

        -- ButtonState
        ["ButtonState"] = {
			["Set"] = function(self, state)
                self:SetButtonState(state, self.__Locked)
			end,

			["Get"] = function(self)
				return self:GetButtonState()
			end,

			["Type"] = "ButtonState",
		},

        -- Locked
        ["Locked"] = {
			["Set"] = function(self, enabled)
                self.__Locked = enabled
                self:SetButtonState(self.__UI:GetButtonState(), enabled)
			end,

			["Get"] = function(self)
				return (self.__Locked and true) or false
			end,

			["Type"] = "boolean",
		},

        -- DisabledFontObject
        ["DisabledFontObject"] = {
			["Set"] = function(self, font)
                self:SetDisabledFontObject(font)
			end,

			["Get"] = function(self)
				return self:GetDisabledFontObject()
			end,

			["Type"] = "Font",
		},

     	-- DisabledTexture
     	["DisabledTexture"] = {
			["Set"] = function(self, texture)
				self:SetDisabledTexture(texture)
			end,

			["Get"] = function(self)
				return self:GetDisabledTexture()
			end,

			["Type"] = "Texture",
     	},

     	-- DisabledTexturePath
     	["DisabledTexturePath"] = {
			["Set"] = function(self, texture)
				self:SetDisabledTexture(texture)
			end,

			["Get"] = function(self)
				return self:GetDisabledTexture() and self:GetDisabledTexture().TexturePath
			end,

			["Type"] = "TexturePath",
     	},
		
     	-- HighlightTexture
     	["HighlightTexture"] = {
			["Set"] = function(self, texture)
				self:SetHighlightTexture(texture)
			end,

			["Get"] = function(self)
				return self:GetHighlightTexture()
			end,

			["Type"] = "Texture",
     	},

     	-- HighlightTexturePath
     	["HighlightTexturePath"] = {
			["Set"] = function(self, texture)
				self:SetHighlightTexture(texture)
			end,

			["Get"] = function(self)
				return self:GetHighlightTexture() and self:GetHighlightTexture().TexturePath
			end,

			["Type"] = "TexturePath",
     	},
		
     	-- NormalTexture
     	["NormalTexture"] = {
			["Set"] = function(self, texture)
				self:SetNormalTexture(texture)
			end,

			["Get"] = function(self)
				return self:GetNormalTexture()
			end,

			["Type"] = "Texture",
     	},

     	-- NormalTexturePath
     	["NormalTexturePath"] = {
			["Set"] = function(self, texture)
				self:SetNormalTexture(texture)
			end,

			["Get"] = function(self)
				return self:GetNormalTexture() and self:GetNormalTexture().TexturePath
			end,

			["Type"] = "TexturePath",
     	},
		
     	-- PushedTexture
     	["PushedTexture"] = {
			["Set"] = function(self, texture)
				self:SetPushedTexture(texture)
			end,

			["Get"] = function(self)
				return self:GetPushedTexture()
			end,

			["Type"] = "Texture",
     	},

     	-- PushedTexturePath
     	["PushedTexturePath"] = {
			["Set"] = function(self, texture)
				self:SetPushedTexture(texture)
			end,

			["Get"] = function(self)
				return self:GetPushedTexture() and self:GetPushedTexture().TexturePath
			end,

			["Type"] = "TexturePath",
     	},
		
		--- FontString
		["FontString"] = {
			["Set"] = function(self, label)
				self:SetFontString(label)
			end,

			["Get"] = function(self)
				return self:GetFontString()
			end,

			["Type"] = "FontString",
		},

		--- HighlightFontObject
		["HighlightFontObject"] = {
			["Set"] = function(self, fontObject)
				self:SetHighlightFontObject(fontObject)
			end,

			["Get"] = function(self)
				return self:GetHighlightFontObject()
			end,

			["Type"] = "Font",
		},

		--- NormalFontObject
		["NormalFontObject"] = {
			["Set"] = function(self, fontObject)
				self:SetNormalFontObject(fontObject)
			end,

			["Get"] = function(self)
				return self:GetNormalFontObject()
			end,

			["Type"] = "Font",
		},

		--- PushedTextOffset
		["PushedTextOffset"] = {
			["Set"] = function(self, offset)
				self:SetPushedTextOffset(offset.x, offset.y)
			end,

			["Get"] = function(self)
				local offset = {}
				offset.x, offset.y = self:GetPushedTextOffset()

				return offset
			end,

			["Type"] = "Offset",
		},

		--- Text
		["Text"] = {
			["Set"] = function(self, text)
				self:SetText(text)
			end,

			["Get"] = function(self)
				return self:GetText()
			end,

			["Type"] = "string",
		},

        -- HighlightLocked
        ["HighlightLocked"] = {
			["Set"] = function(self, locked)
                if locked then
				    self.__UI:LockHighlight()
				    self.__HighlightLocked = true
                else
				    self.__UI:UnlockHighlight()
				    self.__HighlightLocked = false
                end
			end,

			["Get"] = function(self)
				return (self.__HighlightLocked and true) or false
			end,

			["Type"] = "boolean",
		},
    }

    -- New
    local _New = function(_parent, ...)
        -- New Frame
		return CreateFrame(_WidgetName,nil,IGAS:GetUI(_parent), ...)
    end

    local _WidgetInfo = {
        ["WidgetName"] = _WidgetName,
        ["Base"] = _Base,
        ["FuncProxy"] = _FuncProxy,
        ["Property"] = _Property,
        ["ScriptType"] = _ScriptType,
        ["New"] = _New,
    }

    IGAS:NewWidget(_WidgetInfo, true, true)
end