﻿-- Author      : Kurapica
-- Create Date : 8/18/2009

----------------------------------------------------------------------------------------------------------------------------------------
--- An AnimationGroup is how various animations are actually applied to a region; this is how different behaviors can be run in sequence or in parallel with each other, automatically. When you pause an AnimationGroup, it tracks which of its child animations were playing and how far advanced they were, and resumes them from that point.<br>
-- An Animation in a group has an order from 1 to 100, which determines when it plays; once all animations with order 1 have completed, including any delays, the AnimationGroup starts all animations with order 2.<br>
-- An AnimationGroup can also be set to loop, either repeating from the beginning or playing backward back to the beginning. An AnimationGroup has an OnLoop handler that allows you to call your own code back whenever a loop completes. The :Finish() method stops the animation after the current loop has completed, rather than immediately.<br>
-- <br><br>inherit <a href=".\UIObject.html">UIObject</a> For all methods, properties and scriptTypes
-- @name AnimationGroup
-- @class table
-- @field Looping Looping type for the animation group: BOUNCE , NONE  , REPEAT
-- @field LoopState the current loop state of the group: FORWARD , NONE , REVERSE
----------------------------------------------------------------------------------------------------------------------------------------
do
	-- Check Version
	local version = 2
	if not IGAS:NewAddon("IGAS.GUI.AnimationGroup", version) then
		return
	end

    local CreateFrame = CreateFrame

	local _WidgetName = "AnimationGroup"
    local _Base = "UIObject"

	-- Enum
	local _LoopEnum = IGAS:NewEnum("LoopType", {"NONE", "REPEAT", "BOUNCE"})
	local _LoopState = IGAS:NewEnum("LoopState", {"NONE", "FORWARD", "REVERSE"})

	-- Property
	local _LoopEnumType = IGAS:NewPropertyType("LoopType", {
			["Type"] = "enum",
			["EnumType"] = _LoopEnum,
		})
	local _LoopStateType = IGAS:NewPropertyType("LoopState", {
			["Type"] = "enum",
			["EnumType"] = _LoopState,
		})

	-- ScriptType
	local _ScriptType = {
		------------------------------------
		--- ScriptType, Run whenever an event fires for which the frame is registered
		-- @name AnimationGroup:OnEvent
		-- @class function
		-- @param event the event's name
		-- @param ... the event's parameters
		-- @usage function AnimationGroup:OnEvent(event, ...)<br>
		--    -- do someting<br>
		-- end
		------------------------------------
		["OnEvent"] = true,

		------------------------------------
		--- ScriptType, Run when the animation (or animation group) finishes animating
		-- @name AnimationGroup:OnFinished
		-- @class function
		-- @param requested True if animation finished because of a call to AnimationGroup:Finish(); false otherwise
		-- @usage function AnimationGroup:OnFinished(requested)<br>
		--    -- do someting<br>
		-- end
		------------------------------------
		["OnFinished"] = true,

		------------------------------------
		--- ScriptType, Run when the frame is created
		-- @name AnimationGroup:OnLoad
		-- @class function
		-- @usage function AnimationGroup:OnLoad()<br>
		--    -- do someting<br>
		-- end
		------------------------------------
		["OnLoad"] = true,

		------------------------------------
		--- ScriptType, Run when the animation group's loop state changes
		-- @name AnimationGroup:OnLoop
		-- @class function
		-- @param loopState the animation group's loop state
		-- @usage function AnimationGroup:OnLoop(loopState)<br>
		--    -- do someting<br>
		-- end
		------------------------------------
		["OnLoop"] = true,

		------------------------------------
		--- ScriptType, Run when the animation (or animation group) is paused
		-- @name AnimationGroup:OnPause
		-- @class function
		-- @usage function AnimationGroup:OnPause()<br>
		--    -- do someting<br>
		-- end
		------------------------------------
		["OnPause"] = true,

		------------------------------------
		--- ScriptType, Run when the animation (or animation group) begins to play
		-- @name AnimationGroup:OnPlay
		-- @class function
		-- @usage function AnimationGroup:OnPlay()<br>
		--    -- do someting<br>
		-- end
		------------------------------------
		["OnPlay"] = true,

		------------------------------------
		--- ScriptType, Run when the animation (or animation group) is stopped
		-- @name AnimationGroup:OnStop
		-- @class function
		-- @param requested True if the animation was stopped due to a call to the animation's or group's :Stop() method; false if the animation was stopped for other reasons
		-- @usage function AnimationGroup:OnStop(requested)<br>
		--    -- do someting<br>
		-- end
		------------------------------------
		["OnStop"] = true,

		------------------------------------
		--- ScriptType, Run each time the screen is drawn by the game engine
		-- @name AnimationGroup:OnUpdate
		-- @class function
		-- @param elapsed Number of seconds since the OnUpdate handlers were last run (likely a fraction of a second)
		-- @usage function AnimationGroup:OnUpdate(elapsed)<br>
		--    -- do someting<br>
		-- end
		------------------------------------
		["OnUpdate"] = true,
	}

	-- FuncProxy
	local _FuncProxy = {
		------------------------------------
		--- Start playing the animations in this group.
		-- @name AnimationGroup:Play
		-- @class function
		-- @usage AnimationGroup:Play()
		------------------------------------

		-- Play
		["Play"] = function(self)
			return self.__UI:Play()
		end,

		------------------------------------
		--- Pause the animations in this group.
		-- @name AnimationGroup:Pause
		-- @class function
		-- @usage AnimationGroup:Pause()
		------------------------------------

		-- Pause
		["Pause"] = function(self)
			return self.__UI:Pause()
		end,

		------------------------------------
		--- Stop all animations in this group.
		-- @name AnimationGroup:Stop
		-- @class function
		-- @usage AnimationGroup:Stop()
		------------------------------------

		-- Stop
		["Stop"] = function(self)
			return self.__UI:Stop()
		end,

		------------------------------------
		--- Notify this group to stop playing once the current loop cycle is done. Does nothing if this group is not playing.
		-- @name AnimationGroup:Finish
		-- @class function
		-- @usage AnimationGroup:Finish()
		------------------------------------

		-- Finish
		["Finish"] = function(self)
			return self.__UI:Finish()
		end,

		------------------------------------
		--- Returns the progress of this animation as a unit value [0,1].
		-- @name AnimationGroup:GetProgress
		-- @class function
		-- @return the progress of this animation as a unit value
		-- @usage AnimationGroup:GetProgress()
		------------------------------------

		-- GetProgress
		["GetProgress"] = function(self)
			return self.__UI:GetProgress()
		end,

		------------------------------------
		--- Returns true if the group has finished playing.
		-- @name AnimationGroup:IsDone
		-- @class function
		-- @return true if the group has finished playing
		-- @usage AnimationGroup:IsDone()
		------------------------------------

		-- IsDone
		["IsDone"] = function(self)
			return self.__UI:IsDone()
		end,

		------------------------------------
		--- Returns true if the group is playing.
		-- @name AnimationGroup:IsPlaying
		-- @class function
		-- @return true if the group is playing
		-- @usage AnimationGroup:IsPlaying()
		------------------------------------

		-- IsPlaying
		["IsPlaying"] = function(self)
			return self.__UI:IsPlaying()
		end,

		------------------------------------
		--- Returns true if the group is paused.
		-- @name AnimationGroup:IsPaused
		-- @class function
		-- @return true if the group is pause
		-- @usage AnimationGroup:IsPaused()
		------------------------------------

		-- IsPaused
		["IsPaused"] = function(self)
			return self.__UI:IsPaused()
		end,

		------------------------------------
		--- Gets the total duration across all child Animations that the group will take to complete one loop cycle.
		-- @name AnimationGroup:GetDuration
		-- @class function
		-- @return the total duration across all child Animations that the group will take to complete one loop cycle
		-- @usage AnimationGroup:GetDuration()
		------------------------------------

		-- GetDuration
		["GetDuration"] = function(self)
			return self.__UI:GetDuration()
		end,

		------------------------------------
		--- Sets the type of looping for the group. Input is [NONE, REPEAT, or BOUNCE].
		-- @name AnimationGroup:SetLooping
		-- @class function
		-- @param loopType  the type of looping
		-- @usage AnimationGroup:SetLooping("REPEAT")
		------------------------------------

		-- SetLooping
		["SetLooping"] = function(self, loopType)
			return self.__UI:SetLooping(loopType)
		end,

		------------------------------------
		--- Gets the type of looping for the group.
		-- @name AnimationGroup:GetLooping
		-- @class function
		-- @return the type of looping
		-- @usage AnimationGroup:GetLooping()
		------------------------------------

		-- GetLooping
		["GetLooping"] = function(self)
			return self.__UI:GetLooping()
		end,

		------------------------------------
		--- Gets the current loop state of the group. Output is [NONE, FORWARD, or REVERSE].
		-- @name AnimationGroup:GetLoopState
		-- @class function
		-- @return the current loop state
		-- @usage AnimationGroup:GetLoopState()
		------------------------------------

		-- GetLoopState
		["GetLoopState"] = function(self)
			return self.__UI:GetLoopState()
		end,

		------------------------------------
		--- Create and return an Animation as a child of this group.
		-- @name AnimationGroup:CreateAnimation
		-- @class function
		-- @param animationType Type of Animation object to be created
		-- @param name Optional,name to use for the new animation
		-- @param inheritsFrom Optional,A template from which to inherit
		-- @return an Animation that a child of this group
		-- @usage AnimationGroup:CreateAnimation("Alpha")
		------------------------------------

		-- CreateAnimation
		["CreateAnimation"] = function(self, animationType, name, inheritsFrom)
			return IGAS:NewFrame(animationType, name, self, inheritsFrom)
		end,
	}

	-- Property
	local _Property = {
        -- Looping
        ["Looping"] = {
			["Set"] = function(self, loopType)
				if _LoopEnum[loopType] then
					self:SetLooping(loopType)
				end
			end,

			["Get"] = function(self)
				return self:GetLooping()
			end,

			["Type"] = _LoopEnumType,
		},
        -- LoopState
        ["LoopState"] = {
			["Get"] = function(self)
				return self:GetLoopState()
			end,

			["Type"] = _LoopStateType,
		},
	}

    -- New
    local _New = function(_parent,inheritsFrom)
		if _parent and _parent["CreateAnimationGroup"] then
			return IGAS:GetUI(_parent):CreateAnimationGroup(nil, inheritsFrom)
		end
    end

    local _WidgetInfo = {
        ["WidgetName"] = _WidgetName,
        ["Base"] = _Base,
        ["FuncProxy"] = _FuncProxy,
        ["Property"] = _Property,
        ["ScriptType"] = _ScriptType,
        ["New"] = _New,
    }

    IGAS:NewWidget(_WidgetInfo, true, true)
end