﻿-- Author      : Kurapica
-- Create Date : 8/18/2009

----------------------------------------------------------------------------------------------------------------------------------------
--- Animations are used to change presentations or other characteristics of a frame or other region over time. The Animation object will take over the work of calling code over time, or when it is done, and tracks how close the animation is to completion.<br>
-- The Animation type doesn't create any visual effects by itself, but it does provide an OnUpdate handler that you can use to support specialized time-sensitive behaviors that aren't provided by the transformations descended from Animations. In addition to tracking the passage of time through an elapsed argument, you can query the animation's progress as a 0-1 fraction to determine how you should set your behavior.<br>
-- You can also change how the elapsed time corresponds to the progress by changing the smoothing, which creates acceleration or deceleration, or by adding a delay to the beginning or end of the animation.<br>
-- You can also use an Animation as a timer, by setting the Animation's OnFinished script to trigger a callback and setting the duration to the desired time.<br>
-- <br><br>inherit <a href=".\UIObject.html">UIObject</a> For all methods, properties and scriptTypes
-- @name Animation
-- @class table
-- @field StartDelay Amount of time the animation delays before its progress begins (in seconds)
-- @field EndDelay Time for the animation to delay after finishing (in seconds)
-- @field Duration Time for the animation to progress from start to finish (in seconds)
-- @field MaxFramerate Maximum number of times per second that the animation will update its progress
-- @field Order Position at which the animation will play relative to others in its group (between 0 and 100)
-- @field Smoothing Type of smoothing for the animation, IN, IN_OUT, NONE, OUT
----------------------------------------------------------------------------------------------------------------------------------------
do
	-- Check Version
	local version = 2
	if not IGAS:NewAddon("IGAS.GUI.Animation", version) then
		return
	end

    local CreateFrame = CreateFrame

	local _WidgetName = "Animation"
    local _Base = "UIObject"

	-- Enum
	local _SmoothingEnum = IGAS:NewEnum("SmoothingType", {"IN", "OUT", "IN_OUT"})

	-- Property
	local _SmoothingType = IGAS:NewPropertyType("SmoothingType", {
			["Type"] = "enum",
			["EnumType"] = _SmoothingEnum,
		})

	IGAS:NewPropertyType("Origin", {
		["Type"] = "table",
		["Members"] = {
			["point"] = {
				["Type"] = "enum",
				["EnumType"] = IGAS:GetEnum("FramePoint"),
			},
			["x"] = {
				["Type"] = "number",
			},
			["y"] = {
				["Type"] = "number",
			},
		},
	})

	-- ScriptType
	local _ScriptType = {
		------------------------------------
		--- ScriptType, Run whenever an event fires for which the frame is registered
		-- @name Animation:OnEvent
		-- @class function
		-- @param event the event's name
		-- @param ... the event's parameters
		-- @usage function Animation:OnEvent(event, ...)<br>
		--    -- do someting<br>
		-- end
		------------------------------------
		["OnEvent"] = true,

		------------------------------------
		--- ScriptType, Run when the animation (or animation group) finishes animating
		-- @name Animation:OnFinished
		-- @class function
		-- @param requested True if animation finished because of a call to AnimationGroup:Finish(); false otherwise
		-- @usage function Animation:OnFinished(requested)<br>
		--    -- do someting<br>
		-- end
		------------------------------------
		["OnFinished"] = true,

		------------------------------------
		--- ScriptType, Run when the frame is created
		-- @name Animation:OnLoad
		-- @class function
		-- @usage function Animation:OnLoad()<br>
		--    -- do someting<br>
		-- end
		------------------------------------
		["OnLoad"] = true,

		------------------------------------
		--- ScriptType, Run when the animation (or animation group) is paused
		-- @name Animation:OnPause
		-- @class function
		-- @usage function Animation:OnPause()<br>
		--    -- do someting<br>
		-- end
		------------------------------------
		["OnPause"] = true,

		------------------------------------
		--- ScriptType, Run when the animation (or animation group) begins to play
		-- @name Animation:OnPlay
		-- @class function
		-- @usage function Animation:OnPlay()<br>
		--    -- do someting<br>
		-- end
		------------------------------------
		["OnPlay"] = true,

		------------------------------------
		--- ScriptType, Run when the animation (or animation group) is stopped
		-- @name Animation:OnStop
		-- @class function
		-- @param requested True if the animation was stopped due to a call to the animation's or group's :Stop() method; false if the animation was stopped for other reasons
		-- @usage function Animation:OnStop(requested)<br>
		--    -- do someting<br>
		-- end
		------------------------------------
		["OnStop"] = true,

		------------------------------------
		--- ScriptType, Run each time the screen is drawn by the game engine
		-- @name Animation:OnUpdate
		-- @class function
		-- @param elapsed Number of seconds since the OnUpdate handlers were last run (likely a fraction of a second)
		-- @usage function Animation:OnUpdate(elapsed)<br>
		--    -- do someting<br>
		-- end
		------------------------------------
		["OnUpdate"] = true,
	}

	-- FuncProxy
	local _FuncProxy = {
		------------------------------------
		--- Play the animation.
		-- @name Animation:Play
		-- @class function
		-- @usage Animation:Play()
		------------------------------------

		-- Play
		["Play"] = function(self)
			return self.__UI:Play()
		end,

		------------------------------------
		--- Pause the animation.
		-- @name Animation:Pause
		-- @class function
		-- @usage Animation:Pause()
		------------------------------------

		-- Pause
		["Pause"] = function(self)
			return self.__UI:Pause()
		end,

		------------------------------------
		--- Stop the animation.
		-- @name Animation:Stop
		-- @class function
		-- @usage Animation:Stop()
		------------------------------------

		-- Stop
		["Stop"] = function(self)
			return self.__UI:Stop()
		end,

		------------------------------------
		--- Returns true if the animation has finished playing.
		-- @name Animation:IsDone
		-- @class function
		-- @return true if the animation has finished playing.
		-- @usage Animation:IsDone()
		------------------------------------

		-- IsDone
		["IsDone"] = function(self)
			return self.__UI:IsDone()
		end,

		------------------------------------
		--- Returns true if the animation is playing.
		-- @name Animation:IsPlaying
		-- @class function
		-- @return true if the animation is playing.
		-- @usage Animation:IsPlaying()
		------------------------------------

		-- IsPlaying
		["IsPlaying"] = function(self)
			return self.__UI:IsPlaying()
		end,

		------------------------------------
		--- Returns true if the animation is paused.
		-- @name Animation:IsPaused
		-- @class function
		-- @return true if the animation is paused.
		-- @usage Animation:IsPaused()
		------------------------------------

		-- IsPaused
		["IsPaused"] = function(self)
			return self.__UI:IsPaused()
		end,

		------------------------------------
		--- Returns true if the animation is stopped.
		-- @name Animation:IsStopped
		-- @class function
		-- @return true if the animation is stopped.
		-- @usage Animation:IsStopped()
		------------------------------------

		-- IsStopped
		["IsStopped"] = function(self)
			return self.__UI:IsStopped()
		end,

		------------------------------------
		--- Returns true if the animation is in the middle of a start or end delay.
		-- @name Animation:IsDelaying
		-- @class function
		-- @return true if the animation is in the middle of a start or end delay.
		-- @usage Animation:IsDelaying()
		------------------------------------

		-- IsDelaying
		["IsDelaying"] = function(self)
			return self.__UI:IsDelaying()
		end,

		------------------------------------
		--- Gets the amount of time in seconds that the animation has been playing for.
		-- @name Animation:GetElapsed
		-- @class function
		-- @return the amount of time in seconds that the animation has been playing for
		-- @usage Animation:GetElapsed()
		------------------------------------

		-- GetElapsed
		["GetElapsed"] = function(self)
			return self.__UI:GetElapsed()
		end,

		------------------------------------
		--- Set the number of seconds that the animation delays before it starts to progress.
		-- @name Animation:SetStartDelay
		-- @class function
		-- @param delaySec the number of seconds that the animation delays before it starts to progress
		-- @usage Animation:SetStartDelay(2)
		------------------------------------

		-- SetStartDelay
		["SetStartDelay"] = function(self, delaySec)
			return self.__UI:SetStartDelay(delaySec)
		end,

		------------------------------------
		--- Get the number of seconds that the animation delays before it starts to progress.
		-- @name Animation:GetStartDelay
		-- @class function
		-- @return the number of seconds that the animation delays before it starts to progress
		-- @usage Animation:GetStartDelay()
		------------------------------------

		-- GetStartDelay
		["GetStartDelay"] = function(self)
			return self.__UI:GetStartDelay()
		end,

		------------------------------------
		--- Set the number of seconds the animation delays after finishing.
		-- @name Animation:SetEndDelay
		-- @class function
		-- @param delaySec the number of seconds the animation delays after finishing
		-- @usage Animation:SetEndDelay(2)
		------------------------------------

		-- SetEndDelay
		["SetEndDelay"] = function(self, delaySec)
			return self.__UI:SetEndDelay(delaySec)
		end,

		------------------------------------
		--- Get the number of seconds the animation delays after finishing.
		-- @name Animation:GetEndDelay
		-- @class function
		-- @return the number of seconds the animation delays after finishing
		-- @usage Animation:GetEndDelay()
		------------------------------------

		-- GetEndDelay
		["GetEndDelay"] = function(self)
			return self.__UI:GetEndDelay()
		end,

		------------------------------------
		--- Set the number of seconds it takes for the animation to progress from start to finish.
		-- @name Animation:SetDuration
		-- @class function
		-- @param durationSec the number of seconds it takes for the animation to progress from start to finish
		-- @usage Animation:SetDuration(2)
		------------------------------------

		-- SetDuration
		["SetDuration"] = function(self, durationSec)
			return self.__UI:SetDuration(durationSec)
		end,

		------------------------------------
		--- Get the number of seconds it takes for the animation to progress from start to finish.
		-- @name Animation:GetDuration
		-- @class function
		-- @return the number of seconds it takes for the animation to progress from start to finish
		-- @usage Animation:GetDuration()
		------------------------------------

		-- GetDuration
		["GetDuration"] = function(self)
			return self.__UI:GetDuration()
		end,

		------------------------------------
		--- Returns the progress of the animation as a unit value [0,1]. Ignores start and end delay.
		-- @name Animation:GetProgress
		-- @class function
		-- @return the progress of the animation as a unit value [0,1]. Ignores start and end delay
		-- @usage Animation:GetProgress()
		------------------------------------

		-- GetProgress
		["GetProgress"] = function(self)
			return self.__UI:GetProgress()
		end,

		------------------------------------
		--- Returns a smoothed, [0,1] progress value for the animation.
		-- @name Animation:GetSmoothProgress
		-- @class function
		-- @return a smoothed, [0,1] progress value for the animation
		-- @usage Animation:GetSmoothProgress()
		------------------------------------

		-- GetSmoothProgress
		["GetSmoothProgress"] = function(self)
			return self.__UI:GetSmoothProgress()
		end,

		------------------------------------
		--- Returns the progress of the animation combined with its start and end delay.
		-- @name Animation:GetProgressWithDelay
		-- @class function
		-- @return the progress of the animation combined with its start and end delay
		-- @usage Animation:GetProgressWithDelay()
		------------------------------------

		-- GetProgressWithDelay
		["GetProgressWithDelay"] = function(self)
			return self.__UI:GetProgressWithDelay()
		end,

		------------------------------------
		--- Sets the maximum frames per second that the animation will update its progress.
		-- @name Animation:SetMaxFramerate
		-- @class function
		-- @param framerate the maximum frames per second that the animation will update its progress
		-- @usage Animation:SetMaxFramerate(30)
		------------------------------------

		-- SetMaxFramerate
		["SetMaxFramerate"] = function(self, framerate)
			return self.__UI:SetMaxFramerate(framerate)
		end,

		------------------------------------
		--- Gets the maximum frames per second that the animation will update its progress.
		-- @name Animation:GetMaxFramerate
		-- @class function
		-- @return the maximum frames per second that the animation will update its progress
		-- @usage Animation:GetMaxFramerate()
		------------------------------------

		-- GetMaxFramerate
		["GetMaxFramerate"] = function(self)
			return self.__UI:GetMaxFramerate()
		end,

		------------------------------------
		--- Sets the order that the animation plays within its parent group. Range is [1,100].
		-- @name Animation:SetOrder
		-- @class function
		-- @param order the order that the animation plays within its parent group
		-- @usage Animation:SetOrder(30)
		------------------------------------

		-- SetOrder
		["SetOrder"] = function(self, order)
			return self.__UI:SetOrder(order)
		end,

		------------------------------------
		--- Gets the order of the animation within its parent group.
		-- @name Animation:GetOrder
		-- @class function
		-- @return the order of the animation within its parent group
		-- @usage Animation:GetOrder()
		------------------------------------

		-- GetOrder
		["GetOrder"] = function(self)
			return self.__UI:GetOrder()
		end,

		------------------------------------
		--- Sets the smoothing type for the animation. Input is [IN,OUT, or IN_OUT].
		-- @name Animation:SetSmoothing
		-- @class function
		-- @param smoothType the smoothing type for the animation
		-- @usage Animation:SetSmoothing("IN")
		------------------------------------

		-- SetSmoothing
		["SetSmoothing"] = function(self, smoothType)
			return self.__UI:SetSmoothing(smoothType)
		end,

		------------------------------------
		--- Gets the smoothing type for the animation.
		-- @name Animation:GetSmoothing
		-- @class function
		-- @return the smoothing type for the animation
		-- @usage Animation:GetSmoothing()
		------------------------------------

		-- GetSmoothing
		["GetSmoothing"] = function(self)
			return self.__UI:GetSmoothing()
		end,

		------------------------------------
		--- Gets the Region object that the animation operates on. The region object is this Animation's parent's parent (the AnimationGroup's parent).
		-- @name Animation:GetRegionParent
		-- @class function
		-- @return the Region object that the animation operates on. The region object is this Animation's parent's parent (the AnimationGroup's parent).
		-- @usage Animation:GetRegionParent()
		------------------------------------

		-- GetRegionParent
		["GetRegionParent"] = function(self)
			return IGAS:GetWrapper(self.__UI:GetRegionParent())
		end,
	}

	-- Property
	local _Property = {
        -- StartDelay
        ["StartDelay"] = {
			["Set"] = function(self, delaySec)
				self:SetStartDelay(delaySec)
			end,

			["Get"] = function(self)
				return self:GetStartDelay()
			end,

			["Type"] = "number",
		},
        -- EndDelay
        ["EndDelay"] = {
			["Set"] = function(self, delaySec)
				self:SetEndDelay(delaySec)
			end,

			["Get"] = function(self)
				return self:GetEndDelay()
			end,

			["Type"] = "number",
		},
        -- Duration
        ["Duration"] = {
			["Set"] = function(self, durationSec)
				self:SetDuration(durationSec)
			end,

			["Get"] = function(self)
				return self:GetDuration()
			end,

			["Type"] = "number",
		},
        -- MaxFramerate
        ["MaxFramerate"] = {
			["Set"] = function(self, framerate)
				self:SetMaxFramerate(framerate)
			end,

			["Get"] = function(self)
				return self:GetMaxFramerate()
			end,

			["Type"] = "number",
		},
        -- Order
        ["Order"] = {
			["Set"] = function(self, order)
				self:SetOrder(order)
			end,

			["Get"] = function(self)
				return self:GetOrder()
			end,

			["Type"] = "number",
		},
        -- Smoothing
        ["Smoothing"] = {
			["Set"] = function(self, smoothType)
				if _SmoothingEnum[smoothType] then
					self:SetSmoothing(smoothType)
				end
			end,

			["Get"] = function(self)
				return self:GetSmoothing()
			end,

			["Type"] = _SmoothingType,
		},
	}

    local _WidgetInfo = {
        ["WidgetName"] = _WidgetName,
        ["Base"] = _Base,
        ["FuncProxy"] = _FuncProxy,
        ["Property"] = _Property,
        ["ScriptType"] = _ScriptType,
    }

    IGAS:NewWidget(_WidgetInfo, true)
end