﻿-- Author      : Kurapica
-- Create Date : 6/12/2008 1:11:28 AM

do
	local version = 2

	if not IGAS:NewAddon("IGAS.GUI.PropertyType", version) then
		return
	end

	local wipe = table.wipe
	local _Widget = IGAS.GUI.Widget
	local assert = assert

	_PropertyType = _PropertyType or {}

	-- namespace
	IGAS.GUI.PropertyType = _PropertyType

	-- Register API

	------------------------------------
	-- create or get a new property type
	-- @name IGAS:NewPropertyType
	-- @class function
	-- @param enumName the name of the property
	-- @param enums a table contains the define of property
	-- @return the property type
	-- @usage IGAS:NewPropertyType("Style", {["Type"] = "enum",	["EnumType"] = IGAS:GetEnum("Style"),})
	------------------------------------
	IGAS:RegisterFunction("NewPropertyType", function(self, propName, propSet)
		if propName and (_Widget[propName] or (type(propName) == "table" and propName.WidgetName)) then
			propName = _Widget[propName] or propName

			_PropertyType[propName] = _PropertyType[propName] or {}

			_PropertyType[propName]["Type"] = "widget"
			_PropertyType[propName]["Widget"] = propName
			_PropertyType[propName]["Check"] = propSet.Check or function(frame)
				assert(not frame or (type(frame) == "table" and frame.IsObjectType and frame:IsObjectType(propName)), "The value must be a "..propName.WidgetName..".")

				return frame
			end
		elseif propName and type(propName) == "string" and propName ~= "" and propSet and type(propSet) == "table" and next(propSet) then
			_PropertyType[propName] = _PropertyType[propName] or {}

			wipe(_PropertyType[propName])

			for k, v in pairs(propSet) do
				_PropertyType[propName][k] = v
			end
		end
		return (propName and _PropertyType[propName]) or nil
	end)

	------------------------------------
	-- get a property type
	-- @name IGAS:GetPropertyType
	-- @class function
	-- @param enumName the name of the property
	-- @return the property type
	-- @usage IGAS:GetPropertyType("Style")
	------------------------------------
	IGAS:RegisterFunction("GetPropertyType", function(self, propName)
		if type(propName) == "string" then
			-- Check if the string end with '?', means this value can be nil, so no need go on checking.
			-- and of course convert it to normal propertyType name if value is not nil.
			if strfind(propName, "?$") then
				propName = strmatch(propName, "^[_%w]+")
			end
		end
		if propName and (_Widget[propName] or (type(propName) == "table" and propName.WidgetName)) then
			-- Check if the name is a widget name
			propName = _Widget[propName] or propName

			if not _PropertyType[propName] or _PropertyType[propName].Type ~= "widget" then
				_PropertyType[propName] = {
					["Type"] = "widget",
					["Widget"] = propName,
					["Check"] = function(frame)
						assert(not frame or (type(frame) == "table" and frame.IsObjectType and frame:IsObjectType(propName)), "The value must be a "..propName.WidgetName..".")

						return frame
					end,
				}
			end
		elseif propName and type(propName) == "table" and (propName.Type or propName.Check) then
			-- Check if the name is a property
			return propName
		end

		return (propName and _PropertyType[propName]) or nil
	end)

 	------------------------------------------------------
	-- Default Property Types
	------------------------------------------------------
	-- any
	IGAS:NewPropertyType("any", {
		["Check"] = function(value)
			return value
		end,
	})
	-- number
	IGAS:NewPropertyType("number", {
		["Type"] = "number",
		["Check"] = function(num)
			assert(num and type(num) == "number", "The value must be a number.")

			return num
		end,
	})
	-- string
	IGAS:NewPropertyType("string", {
		["Type"] = "string",
		["Check"] = function(str)
			str = (str == nil and "") or str

			assert(type(str) == "string", "The value must be a string.")

			return str
		end,
	})
	-- boolean
	IGAS:NewPropertyType("boolean", {
		["Type"] = "boolean",
		["Check"] = function(flag)
			flag = (flag and true) or false
			return flag
		end,
	})
	-- 1nil
	IGAS:NewPropertyType("1nil", {
		["Type"] = "number",
		["Check"] = function(flag)
			flag = (flag and 1) or nil
			return flag
		end,
	})
	-- function
	IGAS:NewPropertyType("function", {
		["Type"] = "function",
		["Check"] = function(func)
			assert(func == nil or type(func) == "function", "The value must be a function or nil.")

			return func
		end,
	})
	-- table
	IGAS:NewPropertyType("table", {
		["Type"] = "table",
		["Check"] = function(tbl)
			assert(tbl == nil or type(tbl) == "table", "The value must be a table or nil.")

			return tbl
		end,
	})
	-- TexturePath
	IGAS:NewPropertyType("TexturePath", {
		["Type"] = "string",
		["Check"] = function(path)
			assert(path == nil or type(path) == "string", "The value must be a string or nil.")

			return path
		end,
		["NotLocal"] = true,
	})
	-- Unit
	IGAS:NewPropertyType("Unit", {
		["Type"] = "string",
		["Check"] = function(unit)
			assert(unit == nil or type(unit) == "string", "The value must be a string or nil.")

			return unit
		end,
	})
	-- Alpha
	IGAS:NewPropertyType("Alpha", {
		["Type"] = "number",
		["Check"] = function(alpha)
			assert(alpha and type(alpha) == "number", "The value must be a number.")

			if alpha < 0 then alpha = 0 end
			
			if alpha > 1 then alpha = 1 end
			
			return alpha
		end,
	})
	-- Font
	IGAS:NewPropertyType("FontTable", {
		["Type"] = "table",
		["Check"] = function(font)
			assert(font and type(font) == "table", "The value must be a table.")

			return font
		end,
		["Members"] = {
			["path"] = {
				["Type"] = "string",
				["Check"] = function(path)
					assert(path and type(path) == "string", "The Font.path must be a string.")

					return path
				end,
			},
			["size"] = {
				["Type"] = "number",
				["Check"] = function(size)
					assert(size and type(size) == "number" and size > 0, "The Font.size must be greater than 0.")

					return size
				end,
			},
			["outline"] = {
				["Type"] = "enum",
				["EnumType"] = IGAS:GetEnum("OutLineType"),
				["Check"] = function(enum)
					assert(enum and IGAS:GetEnum("OutLineType")[enum], "The Font.outline must be an enumeration value.")

					return enum
				end,
			},
			["monochrome"] = "boolean",
		},
	})
	-- JustifyH
	IGAS:NewPropertyType("JustifyH", {
		["Type"] = "enum",
		["EnumType"] = IGAS:GetEnum("JustifyHType"),
	})
	-- JustifyV
	IGAS:NewPropertyType("JustifyV", {
		["Type"] = "enum",
		["EnumType"] = IGAS:GetEnum("JustifyVType"),
	})
	-- FrameStrata
	IGAS:NewPropertyType("FrameStrata", {
		["Type"] = "enum",
		["EnumType"] = IGAS:GetEnum("FrameStrata"),
	})
	-- Color
	local defaultColor = {}
	IGAS:NewPropertyType("Color", {
		["Type"] = "table",
		["Members"] = {
			["red"] = {
				["Type"] = "number",
				["Check"] = function(red)
					assert(red == nil or (type(red) == "number" and red >= 0 and red <= 1), "The Color.red must between 0 and 1.")

					return red
				end,
			},
			["green"] = {
				["Type"] = "number",
				["Check"] = function(green)
					assert(green  == nil or (type(green) == "number" and green >= 0 and green <= 1), "The Color.green must between 0 and 1.")

					return green
				end,
			},
			["blue"] = {
				["Type"] = "number",
				["Check"] = function(blue)
					assert(blue  == nil or (type(blue) == "number" and blue >= 0 and blue <= 1), "The Color.blue must between 0 and 1.")

					return blue
				end,
			},
			["alpha"] = {
				["Type"] = "number",
				["Check"] = function(alpha)
					assert(alpha == nil or (type(alpha) == "number" and alpha >= 0 and alpha <= 1), "The Color.alpha must between 0 and 1, can be nil.")

					return alpha
				end,
			},
		},
		["Check"] = function(color)
			if not color then
				color = defaultColor
			end
			
			assert(color and type(color) == "table", "The color value must be a table.")

			return color
		end,
	})
	-- Offset
	IGAS:NewPropertyType("Offset", {
		["Type"] = "table",
		["Check"] = function(off)
			assert(off and type(off) == "table", "The offset value must be a table.")

			return off
		end,
		["Members"] = {
			["x"] = {
				["Type"] = "number",
				["Check"] = function(x)
					assert(x and type(x) == "number", "The Offset.x must be a number.")

					return x
				end,
			},
			["y"] = {
				["Type"] = "number",
				["Check"] = function(y)
					assert(y and type(y) == "number", "The Offset.y must be a number.")

					return y
				end,
			},
		},
	})
	-- DrawLayer
	IGAS:NewPropertyType("DrawLayer", {
		["Type"] = "enum",
		["EnumType"] = IGAS:GetEnum("DrawLayer"),
	})
	-- AlphaMode
	IGAS:NewPropertyType("AlphaMode", {
		["Type"] = "enum",
		["EnumType"] = IGAS:GetEnum("AlphaMode"),
	})
	-- ButtonState
	IGAS:NewPropertyType("ButtonState", {
		["Type"] = "enum",
		["EnumType"] = IGAS:GetEnum("ButtonState"),
	})
	-- ButtonState
	IGAS:NewPropertyType("InsertMode", {
		["Type"] = "enum",
		["EnumType"] = IGAS:GetEnum("InsertMode"),
	})
	-- Position
	IGAS:NewPropertyType("Position", {
		["Type"] = "table",
		["Check"] = function(pos)
			assert(pos and type(pos) == "table", "The position value must be a table.")

			return pos
		end,
		["Members"] = {
			["x"] = {
				["Type"] = "number",
				["Check"] = function(x)
					assert(x and type(x) == "number", "The Position.x must be a number.")

					return x
				end,
			},
			["y"] = {
				["Type"] = "number",
				["Check"] = function(y)
					assert(y and type(y) == "number", "The Position.y must be a number.")

					return y
				end,
			},
			["z"] = {
				["Type"] = "number",
				["Check"] = function(z)
					assert(z and type(z) == "number", "The Position.z must be a number.")

					return z
				end,
			},
		},
	})
	-- Light
	IGAS:NewPropertyType("Light", {
		["Type"] = "table",
		["Check"] = function(lgt)
			assert(lgt and type(lgt) == "table", "The light value must be a table.")

			return lgt
		end,
		["Members"] = {
			["enabled"] = "1nil",
			["omni"] = {
				["Type"] = "number",
				["Check"] = function(omni)
					assert(omni == 1 or omni == 0, "The Light.omni must be 1 or 0.")

					return omni
				end,
			},
			["dirX"] = {
				["Type"] = "number",
				["Check"] = function(dirX)
					assert(dirX and type(dirX) == "number", "The Light.dirX must be a number.")

					return dirX
				end,
			},
			["dirY"] = {
				["Type"] = "number",
				["Check"] = function(dirY)
					assert(dirY and type(dirY) == "number", "The Light.dirY must be a number.")

					return dirY
				end,
			},
			["dirZ"] = {
				["Type"] = "number",
				["Check"] = function(dirZ)
					assert(dirZ and type(dirZ) == "number", "The Light.dirZ must be a number.")

					return dirZ
				end,
			},
			["ambIntensity"] = {
				["Type"] = "number",
				["Check"] = function(ambIntensity)
					assert(ambIntensity and type(ambIntensity) == "number", "The Light.ambIntensity must be a number.")

					return ambIntensity
				end,
			},
			["ambR"] = {
				["Type"] = "number",
				["Check"] = function(ambR)
					assert(ambR == nil or type(ambR) == "number", "The Light.ambR must be a number.")

					return ambR
				end,
			},
			["ambG"] = {
				["Type"] = "number",
				["Check"] = function(ambG)
					assert(ambG == nil or type(ambG) == "number", "The Light.ambG must be a number.")

					return ambG
				end,
			},
			["ambB"] = {
				["Type"] = "number",
				["Check"] = function(ambB)
					assert(ambB == nil or type(ambB) == "number", "The Light.ambB must be a number.")

					return ambB
				end,
			},
			["dirIntensity"] = {
				["Type"] = "number",
				["Check"] = function(dirIntensity)
					assert(dirIntensity == nil or type(dirIntensity) == "number", "The Light.dirIntensity must be a number.")

					return dirIntensity
				end,
			},
			["dirR"] = {
				["Type"] = "number",
				["Check"] = function(dirR)
					assert(dirR == nil or type(dirR) == "number", "The Light.dirR must be a number.")

					return dirR
				end,
			},
			["dirG"] = {
				["Type"] = "number",
				["Check"] = function(dirG)
					assert(dirG == nil or type(dirG) == "number", "The Light.dirG must be a number.")

					return dirG
				end,
			},
			["dirB"] = {
				["Type"] = "number",
				["Check"] = function(dirB)
					assert(dirB == nil or type(dirB) == "number", "The Light.dirB must be a number.")

					return dirB
				end,
			},
		},
	})
	-- MinMaxValue
	IGAS:NewPropertyType("MinMaxValue", {
		["Type"] = "table",
		["Check"] = function(mm)
			assert(mm and type(mm) == "table", "The minmax value must be a table.")

			assert(mm.min and mm.max and type(mm.min) == "number" and type(mm.max) == "number" and mm.min <= mm.max, "The MinMaxValue.max must be no less than MinMaxValue.min.")

			return mm
		end,
		["Members"] = {
			["min"] = "number",
			["max"] = "number",
		},
	})
	-- RectInset
	IGAS:NewPropertyType("RectInset", {
		["Type"] = "table",
		["Check"] = function(rct)
			assert(rct and type(rct) == "table", "The rectInset value must be a table.")

			return rct
		end,
		["Members"] = {
			["left"] = {
				["Type"] = "number",
				["Check"] = function(left)
					assert(left and type(left) == "number", "The RectInset.left must be a number.")

					return left
				end,
			},
			["right"] = {
				["Type"] = "number",
				["Check"] = function(right)
					assert(right and type(right) == "number", "The RectInset.right must be a number.")

					return right
				end,
			},
			["top"] = {
				["Type"] = "number",
				["Check"] = function(top)
					assert(top and type(top) == "number", "The RectInset.top must be a number.")

					return top
				end,
			},
			["bottom"] = {
				["Type"] = "number",
				["Check"] = function(bottom)
					assert(bottom and type(bottom) == "number", "The RectInset.bottom must be a number.")

					return bottom
				end,
			},
		},
	})
	-- Size
	IGAS:NewPropertyType("Size", {
		["Type"] = "table",
		["Check"] = function(size)
			assert(size and type(size) == "table", "The size value must be a table.")

			return size
		end,
		["Members"] = {
			["width"] = {
				["Type"] = "number",
				["Check"] = function(width)
					assert(width and type(width) == "number" and width > 0, "The Size.width must be no less than 0.")

					return width
				end,
			},
			["height"] = {
				["Type"] = "number",
				["Check"] = function(height)
					assert(height and type(height) == "number" and height > 0, "The Size.height must be no less than 0.")

					return height
				end,
			},
		},
	})
	-- Backdrop
	IGAS:NewPropertyType("Backdrop", {
		["Type"] = "table",
		["Check"] = function(back)
			assert(back == nil or type(back) == "table", "The backdrop value must be a table or nil.")

			return back
		end,
		["Members"] = {
			["bgFile"] = {
				["Type"] = "string",
				["Check"] = function(path)
					assert(path == nil or type(path) == "string", "The Backdrop.bgFile must be a string or nil.")

					return path
				end,
			},
			["edgeFile"] = {
				["Type"] = "string",
				["Check"] = function(path)
					assert(path == nil or type(path) == "string", "The Backdrop.edgeFile must be a string or nil.")

					return path
				end,
			},
			["tile"] = "boolean",
			["tileSize"] = {
				["Type"] = "number",
				["Check"] = function(size)
					assert(size == nil or type(size) == "number", "The Backdrop.tileSize must be a number or nil.")

					return size
				end,
			},
			["edgeSize"] = {
				["Type"] = "number",
				["Check"] = function(size)
					assert(size == nil or type(size) == "number", "The Backdrop.edgeSize must be a number or nil.")

					return size
				end,
			},
			["insets"] = {
				["Type"] = "table",
				["Check"] = function(insets)
					assert(insets == nil or type(insets) == "table", "The Backdrop.insets must be a table or nil.")

					return insets
				end,
				["Members"] = {
					["left"] = {
						["Type"] = "number",
						["Check"] = function(left)
							assert(left and type(left) == "number", "The Backdrop.insets.left must be a number.")

							return left
						end,
					},
					["right"] = {
						["Type"] = "number",
						["Check"] = function(right)
							assert(right and type(right) == "number", "The Backdrop.insets.right must be a number.")

							return right
						end,
					},
					["top"] = {
						["Type"] = "number",
						["Check"] = function(top)
							assert(top and type(top) == "number", "The Backdrop.insets.top must be a number.")

							return top
						end,
					},
					["bottom"] = {
						["Type"] = "number",
						["Check"] = function(bottom)
							assert(bottom and type(bottom) == "number", "The Backdrop.insets.bottom must be a number.")

							return bottom
						end,
					},
				},
			},
		},
	})
	-- Orientation
	IGAS:NewPropertyType("Orientation", {
		["Type"] = "enum",
		["EnumType"] = IGAS:GetEnum("Orientation"),
	})
	-- MiniMapPosition
	IGAS:NewPropertyType("MiniMapPosition", {
		["Type"] = "table",
		["Check"] = function(pos)
			assert(pos or type(pos) == "table", "The MiniMapPosition value must be a table.")

			return pos
		end,
		["Members"] = {
			["radius"] = {
				["Type"] = "number",
				["Check"] = function(radius)
					assert(radius and type(radius) == "number" and radius > 0, "The MiniMapPosition.radius must be greater than 0.")

					return radius
				end,
			},
			["rounding"] = {
				["Type"] = "number",
				["Check"] = function(rounding)
					assert(rounding and type(rounding) == "number", "The MiniMapPosition.rounding must be a number.")

					return rounding
				end,
			},
			["angel"] = {
				["Type"] = "number",
				["Check"] = function(angel)
					assert(angel and type(angel) == "number" and angel >= 0 and angel <= 360, "The MiniMapPosition.angel must between 0 and 360.")

					return angel
				end,
			},
		},
	})
	-- Direction
	IGAS:NewPropertyType("Anchor", {
		["Type"] = "enum",
		["EnumType"] = IGAS:GetEnum("Anchor"),
	})
	-- Origin
	IGAS:NewPropertyType("Origin", {
		["Type"] = "table",
		["Check"] = function(origin)
			assert(origin and type(origin) == "table", "The Origin value must be a table.")

			return origin
		end,
		["Members"] = {
			["point"] = {
				["Type"] = "enum",
				["EnumType"] = IGAS:GetEnum("FramePoint"),
			},
			["x"] = {
				["Type"] = "number",
				["Check"] = function(x)
					assert(x and type(x) == "number", "The Origin.x must be a number.")

					return x
				end,
			},
			["y"] = {
				["Type"] = "number",
				["Check"] = function(y)
					assert(y and type(y) == "number", "The Origin.y must be a number.")

					return y
				end,
			},
		},
	})
end