﻿-- Author		: Kurapica
-- Create Date	: 2010/05/22
-- ChangeLog	:
--					2010/06/03 Property Check System added.

do
	local version = 3

	if not IGAS:NewAddon("IGAS.GUI.Core", version) then
		return
	end

	------------------------------------------------------
	-- Core Functions
	------------------------------------------------------
    local error = error
    local rawset = rawset
	local rawget = rawget
	local strtrim = strtrim
	local strmatch = strmatch
    local type = type
	local pcall = pcall
    local geterrorhandler = geterrorhandler
    local setmetatable = setmetatable
	local wipe = table.wipe

	local _Widget = IGAS.GUI.Widget
	
	IGAS_CORE_TESTFRAME = IGAS_CORE_TESTFRAME or CreateFrame("Frame")

	local _InheritTable = {
		-- Keep these three, it's base for this gui system
		["FuncProxy"] = true,
		["Property"] = true,
		["ScriptType"] = true,
	}

	local _UsedTable = {
		["__Childs"] = true,
		["__StackScripts"] = true,
	}

 	------------------------------------------------------
	-- Special Script Handlers
	------------------------------------------------------
	-- OnEvent
	local function OnEvent(self, _event, ...)
		if self[_event] then
			return self[_event](self, ...)
		end
	end

 	------------------------------------------------------
	-- Property Check Function
	------------------------------------------------------
	local _ChkErr

	local function CheckPropType(propType, value)
		if type(propType) == "string" then
			-- Check if the string end with '?', means this value can be nil, so no need go on checking.
			-- and of course convert it to normal propertyType name if value is not nil.
			if strfind(propType, "?$") then
				if value == nil then
					return value
				end
				
				propType = strmatch(propType, "^[_%w]+")
			end
		end
		
		if type(propType) == "string" or (type(propType) == "table" and propType.WidgetName) then
			-- Mean it's name of the propertyType or it's a widget to be convert to propertyType, no propertyType would has a field 'WidgetName'
			propType = IGAS:GetPropertyType(propType)
		end
		
		if propType and type(propType) == "table" then
			-- Now checking property
			if propType.Check and type(propType.Check) == "function" then
				-- do checcking
				value = propType.Check(value)				
			elseif propType.Type == "enum" then
				-- enum checking is easy to do, no need left it to authors
				assert(propType.EnumType and type(propType.EnumType) == "table", "The property don't have a correct enumeration type.")
				assert(propType.EnumType[value], "The value is not a correct enumeration value.")
				
				return value
			end
			
			if propType.Type == "table" and value then
				assert(type(value) == "table", "The value must be a table.")
				
				if propType.Members then
					for subName, subProp in pairs(propType.Members) do
						value[subName] = CheckPropType(subProp, value[subName])
					end
				end
			end
		end

		return value
	end

	local function CheckProperty(property, value)
		if property then
			if property.Check and type(property.Check) == "function" then
				-- if the property has it's own checking function, just do it
				_ChkErr, value =  pcall(property.Check, value)
				if not _ChkErr then
					error(strmatch(value or "", "[^:]*$"), 3)
				end
			elseif property.Type then
				-- Check value with the property's type
				_ChkErr, value =  pcall(CheckPropType, property.Type, value)
				if not _ChkErr then
					error(strmatch(value or "", "[^:]*$"), 3)
				end
			end
		end
		return value
	end

 	------------------------------------------------------
	-- MetaTables
	------------------------------------------------------
	-- metatable for frame
	_MetaFrame = _MetaFrame or {}

	-- "index": The indexing access table[key].
	_MetaFrame.__index = function(_table, _key)
		local _Object = _table.__ObjectType

		-- Special key: __Childs, __StackScripts
		if _UsedTable[_key] then
			rawset(_table, _key, {})
			return rawget(_table, _key)
		end

		-- Property Get
		if _Object.Property and _Object.Property[_key] then
			if _Object.Property[_key]["Get"] then
				return _Object.Property[_key]["Get"](_table)
			else
				error(_key.." is write-only.",2)
			end
		end

		-- Frame Func
		if _Object.FuncProxy and _Object.FuncProxy[_key] then
			return _Object.FuncProxy[_key]
		end

		-- Child
		if _table.__Childs[_key] then
			return _table.__Childs[_key]
		end

		-- Scripts
		if _table:HasScript(_key) then
			return _table:GetScript(_key)
		end

		-- Custom index metametods
		if _Object.__index then
			return _Object.__index(_table, _key)
		end
	end

	-- "newindex": The indexing assignment table[key] = value.
	_MetaFrame.__newindex = function(_table, _key, _value)
		local _Object = _table.__ObjectType

		-- Special key: __Childs
		if _UsedTable[_key] and type(_value) ~= "table" then
			error(_key.." must be a table.", 2)
		end

		-- Property Set
		if _Object.Property and _Object.Property[_key] then
			if _Object.Property[_key]["Set"] then
				return _Object.Property[_key]["Set"](_table, CheckProperty(_Object.Property[_key], _value))
			else
				error(_key.." is read-only.", 2)
			end
		end

		-- Child
		if _table.__Childs[_key] then
			if _table.__Childs[_key] == _value then
				return
			else
				error(_key.." is a child-widget, can't be used in other way.", 2)
			end
		end

		-- Scripts
		if _table:HasScript(_key) then
			_table:SetScript(_key, _value)
			return
		end

		-- Custom newindex metametods
		if _Object.__newindex then
			return _Object.__newindex(_table, _key, _value)
		end

		rawset(_table,_key,_value)			-- Other key can be set as usual
	end

	-- "call": called when Lua calls a value.
	_MetaFrame.__call = function(op1, ...)
		local _Object = op1.__ObjectType

		-- Custom call metametods
		if _Object and _Object.__call then
			return _Object.__call(op1, ...)
		else
			error("attempt to use a table as function", 2)
		end
	end

 	------------------------------------------------------
	-- Base Functions
	------------------------------------------------------
    local Wrapper = function(object, objectType)
        local _UI, _Wrapper, _Type, _objectType

		if object == UIParent then
			Log(2, "The UIParent is wrappered.")
		end
		
        -- Set _UI, _Wrapper
        if object.__UI or object.__Wrapper then
            if object.__UI then
                _UI = object.__UI
                _Wrapper = object
            elseif object.__Wrapper then
                _UI = object
                _Wrapper = object.__Wrapper
            end
        else
            _UI = object
            _Wrapper = {}
        end

        -- Check objectType
        _Type = nil

        _objectType = objectType or (_Wrapper["GetObjectType"] and _Wrapper:GetObjectType()) or (_UI["GetObjectType"] and _UI:GetObjectType()) or "UIObject"

        if _objectType then
			if type(_objectType) == "string" and _Widget[_objectType] then
				_Type = _Widget[_objectType]
			elseif type(_objectType) == "table" and _objectType["FuncProxy"] then
				_Type = _objectType
			end
		end

        if not _Type then
        	_Type = _Widget["UIObject"]
       	end

        _UI.__Wrapper = _Wrapper
        _Wrapper.__UI = _UI

        _Wrapper.__ObjectType = _Type
		_Wrapper.__StackScripts = _Wrapper.__StackScripts or {}

        setmetatable(_Wrapper, _MetaFrame)

		-- Script
		-- Special for OnEvent, Others is no need to set.
		if _Wrapper:HasScript("OnEvent") then
			_Wrapper:StackScript("OnEvent", OnEvent, true)
		end

        return _Wrapper
    end

	local GetUI = function(ob)
	    return ob and (ob.__UI or ob)
	end

	local GetWrapper = function(ob)
		if not ob or type(ob) ~= "table" or not ob.GetObjectType then
			return ob or nil
		elseif ob.__Wrapper then
            return ob.__Wrapper
		elseif not ob["IsIGASFrame"] then
			return Wrapper(ob)
        else
            return ob
		end
	end

 	------------------------------------------------------
	-- Argument Convert
	------------------------------------------------------
    local function GetRet(arrV, i, maxI)
        if i < maxI then
            return arrV[i], GetRet(arrV, i+1, maxI)
        else
            return arrV[i]
        end
    end

	local function GetArgs(...)
        local ret = {...}
        local e = nil
		local t

		e, t = next(ret, e)
		while(e) do
			if t and type(t) == "table" then
				ret[e] = GetUI(t)
			end
			e, t = next(ret, e)
		end

		return GetRet(ret, 1, select('#', ...))
    end

    local function GetVals(...)
        local ret = {...}
        local e = nil
		local t

		e, t = next(ret, e)
		while(e) do
			if t and type(t) == "table" then
				ret[e] = GetWrapper(t)
			end
			e, t = next(ret, e)
		end

		return GetRet(ret, 1, select('#', ...))
    end

	------------------------------------------------------
	-- Constructor
	------------------------------------------------------
    --- Name Creator
	local function NewName(_typeN, _parent)
		local i = 1
		local name = _typeN["WidgetName"]
		local parentF = _parent or UIParent

		if not name or name == "" then
			name = "Widget"
		end

		parentF = GetWrapper(parentF)

		while true do
			if parentF:GetChild(name..i) then
				i = i + 1
			else
				break
			end
		end

		return name..i
	end

    ------------------------------------------------------
	-- Global Function
	------------------------------------------------------
	local function Constructor(frameType,frameName,parentFrame,...)
		local frame

		---------------------------------------------- Wrapper Parent ---------------------------------------------
		parentFrame = GetWrapper(parentFrame or UIParent)

		---------------------------------------------- Check -------------------------------------------------------
		---- Check FrameType
		if not frameType then
			error("There must be a frame's type be set.", 2)
		elseif type(frameType) == "string" then
			if _Widget[frameType] then
				frameType = _Widget[frameType]
			else
				error("This frameType ["..frameType.."] is not exist.", 2)
			end
		elseif type(frameType) == "table" then
            if not frameType["FuncProxy"] then
				error("This frameType is invalid.", 2)
            end
		else
			error("This frameType is invalid.", 2)
		end

		if not frameType.New then
            error("This frameType ["..frameType.WidgetName.."] is abstracted.", 2)
		end

		---- Check Parent
		if not parentFrame["AddChild"] then
			error("Can't add child to the object ["..(parentFrame.Name or tostring(parentFrame)).."].", 2)
		end

		---- Check FrameName
        if frameName and type(frameName) ~= "string" then
            error("name must be string.", 2)
        end

        if frameName and frameName ~= "" then
            -- Check the parent's childs
            if parentFrame["GetChild"] and parentFrame:GetChild(frameName) then
            	if parentFrame:GetChild(frameName)["GetObjectType"] and parentFrame:GetChild(frameName):GetObjectType() == frameType then
            		return parentFrame:GetChild(frameName)
            	else
            		error("this name ["..frameName.."] is used", 2)
            	end
            end
        else
        	frameName = NewName(frameType, parentFrame)
        end

		----------------------------------------------------------------- CreateFrame ---------------------------------------------
		if frameName then
			-- New Frame
			frame = frameType.New(parentFrame,...)

			if not frame then
				return nil
			end

	      	frame = Wrapper(frame, frameType)

            frame.Name = frameName
            parentFrame:AddChild(frame)

			return frame
		else
			return nil
		end
	end

	--[[
		CreateFrame
		Args:
		     frameType		string or FrameType,like "Frame" or IGAS.GUI.Frame
		     frameName		the object's name
		     parentFrame	the object's parent
	--]]
	local function CreateFrame(frameType,frameName,parentFrame,...)
		local status, frame = pcall(Constructor, frameType, frameName, parentFrame, ...)

		if not status then
			--errorhandler(frame)
			error(frame, 2)
			frame = nil
		end

		return frame
	end

	--[[
		New Widget, Used to create Widget's Constructor
		Args:
			 _WidgetInfo				Table		Contains the information about the widget
				 {
					WidgetName				String		Widget Name
					Base                    			String      	Base Widget
					ScriptType	            			Table		ScriptType, used for design
					FuncProxy       				Table		FuncProxy
					Property		       	 	Table		Property
					New					Function	Constructor
				}
			isPublic					Boolean	true if the widget is public and will be stored in IGAS.GUI.Widget namespace.
									Table		if isPublic is a table, and it's a widget, it'll be override.
			isSystem					Boolean	true if the widget is the system widget. and need to scan methods.
			_Parent
	--]]
	local function NewWidget(_WidgetInfo, isPublic, isSystem, _Parent)
		-- Check the WidgetInfo
		if not _WidgetInfo or type(_WidgetInfo) ~= "table" then
			error("Widget's information is needed, and must be a table.", 2)
		end

		if not _WidgetInfo.WidgetName or type(_WidgetInfo.WidgetName) ~= "string" or strtrim(_WidgetInfo.WidgetName) == "" then
			error("Widget's Name must be a string", 2)
		end

		for key in pairs(_InheritTable) do
			if _WidgetInfo[key] and type(_WidgetInfo[key]) ~= "table" then
				error("Widget's "..key.." must be a table.", 2)
			end
		end

		if _WidgetInfo.New and type(_WidgetInfo.New) ~= "function" then
			error("Widget's New must be a function.", 2)
		end

		-- Check the base
	    local _Base = _WidgetInfo.Base
		local _NewWidget

		-- Inherit from base
        if _Base then
            if type(_Base) == "string" then
                if _Widget[_Base] then
                    _Base = _Widget[_Base]
                else
                    _Base = _Widget["UIObject"]
                end
            elseif type(_Base) == "table" then
                if not _Base["FuncProxy"] then
                    _Base = _Widget["UIObject"]
                end
            else
                _Base = _Widget["UIObject"]
            end
		end

		-- Check the isPublic
		if isPublic and type(isPublic) == "table" then
			_NewWidget = isPublic

			-- Make sure nothing would control the below settings
			setmetatable(_NewWidget, nil)

			_NewWidget.New = _WidgetInfo.New

			-- Rewrite infomations
			for key in pairs(_InheritTable) do
				if _InheritTable[key] then
					_NewWidget[key] = _NewWidget[key] or {}
					wipe(_NewWidget[key])
					for i, v in pairs(_WidgetInfo[key] or {}) do
						_NewWidget[key][i] = v
					end
				end
			end
		elseif isPublic == true and _Widget[_WidgetInfo.WidgetName] then
			-- if the widget already exists, just rewrite the table, not replace it.
			_NewWidget = _Widget[_WidgetInfo.WidgetName]

			-- Make sure nothing would control the below settings
			setmetatable(_NewWidget, nil)

			_NewWidget.New = _WidgetInfo.New

			-- Rewrite infomations
			for key in pairs(_InheritTable) do
				if _InheritTable[key] then
					_NewWidget[key] = _NewWidget[key] or {}
					wipe(_NewWidget[key])
					for i, v in pairs(_WidgetInfo[key] or {}) do
						_NewWidget[key][i] = v
					end
				end
			end
		else
			-- create new widget
			_NewWidget = {
				["WidgetName"] = _WidgetInfo.WidgetName,
				["New"] = _WidgetInfo.New,
			}
			-- Write infomations
			for key in pairs(_InheritTable) do
				if _InheritTable[key] then
					_NewWidget[key] = _WidgetInfo[key] or {}
				end
			end
		end

		if _Base and (not _NewWidget.Base or _NewWidget.Base ~= _Base) then
			-- Set metatable
			for key in pairs(_InheritTable) do
				if _InheritTable[key] then
					setmetatable(_NewWidget[key], {__index = _Base[key]})
				end
			end
			_NewWidget.Base = _Base
		elseif not _Base and _NewWidget.Base then
			for key in pairs(_InheritTable) do
				if _InheritTable[key] then
					setmetatable(_NewWidget[key], nil)
				end
			end
			_NewWidget.Base = nil
		end

		-- The other parameters will be convert to widget directly.
		for i in pairs(_NewWidget) do
			if _InheritTable[i] == nil and i ~= "New" and i ~= "Base" then
				_NewWidget[i] = nil
			end
		end
		for i, v in pairs(_WidgetInfo) do
			if not _NewWidget[i] and _InheritTable[i] == nil then
				_NewWidget[i] = v
			end
		end

		if _NewWidget and isPublic == true then
			-- Register to system widget
			_Widget[_WidgetInfo.WidgetName] = _NewWidget
		end

		if _NewWidget and isSystem then
			-- Scan new functions
			local status, frame = pcall(Constructor, _NewWidget, nil, _Parent or IGAS_CORE_TESTFRAME)

			if not status then
				frame = nil
			end

			if frame then
				local _oriMeta = GetUI(frame) and getmetatable(GetUI(frame))
				if _oriMeta and _oriMeta.__index and type(_oriMeta.__index) == "table" then
					_oriMeta = _oriMeta.__index

					for _f in pairs(_oriMeta) do
						if not _NewWidget["FuncProxy"][_f] and type(_oriMeta[_f]) == "function" then
							_NewWidget["FuncProxy"][_f] = function(self, ...)
								return GetVals(self.__UI[_f](self.__UI, GetArgs(...)))
							end
							Log(2, "[".._NewWidget.WidgetName.."]"..tostring(_f).." is added.")
						end
					end

					-- Debug part
					for _f in pairs(_NewWidget["FuncProxy"]) do
						if not _oriMeta[_f] then
							Log(2, "[".._NewWidget.WidgetName.."]"..tostring(_f).." not found.")
						end
					end
				end

				frame:Dispose()
			end
		end

        return _NewWidget
	end

    ------------------------------------------------------
	-- GUI Function
	------------------------------------------------------
    IGAS.GUI.CreateFrame = CreateFrame

	-- Register Function

	------------------------------------
	--- Create or get a frame, can use IGAS.GUI.Create(frameType, frameName, parentFrame, ...) instead
	-- @name IGAS:NewFrame
	-- @class function
	-- @param frameType the widget defined in IGAS' GUI lib, can be a string or a widget object
	-- @param frameName the frame's name, must be unique in parentFrame's child, this name won't set to _G
	-- @param parentFrame the parent where the frame is created, UIParent if not setted
	-- @return the frame that created or existed
	-- @usage IGAS:NewFrame("Form", "HelloWorldMain")
	------------------------------------
	IGAS:RegisterFunction("NewFrame", function(self, frameType, frameName, parentFrame, ...)
		local status, frame = pcall(Constructor, frameType, frameName, parentFrame, ...)

		if not status then
			--errorhandler(frame)
			error(frame, 2)
			frame = nil
		end

		return frame
	end)

	------------------------------------
	--- Create a custom widget
	-- @name IGAS:NewWidget
	-- @class function
	-- @param ... See wiki for more infomations
	-- @return the widget that created
	-- @usage See wiki for more infomations
	------------------------------------
	IGAS:RegisterFunction("NewWidget", function(self, ...)
		return NewWidget(...)
	end)

	------------------------------------
	--- Get the true frame of a IGAS frame
	-- @name IGAS:GetUI
	-- @class function
	-- @param frame the IGAS frame
	-- @return the true frame of the IGAS frame
	-- @usage IGAS:GetUI(MyFrame1)
	------------------------------------
	IGAS:RegisterFunction("GetUI", function(self, ...)
		return GetUI(...)
	end)

	------------------------------------
	--- Get the IGAS frame of a frame
	-- @name IGAS:GetWrapper
	-- @class function
	-- @param frame the frame
	-- @return the IGAS frame of the frame
	-- @usage IGAS:GetWrapper(UIParent)
	------------------------------------
	IGAS:RegisterFunction("GetWrapper", function(self, ...)
		return GetWrapper(...)
	end)
end