﻿-- Author      : Kurapica
-- Create Date : 04/17/2009
-- ChangeLog   :
--				2010/01/23 When VARIABLES_LOADED, the SavedVariables in addon's namespace will be replaced with the _G one.
--				2010/01/26 Add AddSlashCmd function.
--				2010/01/29 Make Addon can be rewrited.
--				2010/01/29 Add module system.
--				2010/02/06 Remove or rename some function that the wow system used.
--				2010/02/24 Add EventSystem to Module.
--				2010/02/25 Auto add global functions and tables to addon's environment when used, auto update frame to igas's frame
--				2010/05/20 Check Version method is changed.GetAddon, GetModule methods are added.
--				2010/06/16 New slashcmd system.

----------------------------------------------------------------------------------------------------------------------------------------
--- Addon is an class type for addons in IGAS.The addon class supply some methods, to help authors to manager their addons easily.
-- an addon is also a private scope using to contains it's own functions and variables.
-- addons can access _G's functions and variables, and making an image in itself at the same time.When it access a frame in _G, it'll automatically convert it to an IGAS' frame into the addon' scope.
-- @name Addon
-- @class table
-- @field _Addon the addon self, readonly
-- @field _Name the addon's name, readonly
-- @field _Version the addon's version
-- @field _AutoWrapper whether the addon would auto making wrappers for frames defined in _G when the addon access it.
----------------------------------------------------------------------------------------------------------------------------------------

----------------------------------------------------------------------------------------------------------------------------------------
--- Module is an class type for module in IGAS.a module is contained in an addon or another module.
-- a module is also a private scope.It can access all resources defined in it's container.
-- @name Module
-- @class table
-- @field _M the module self, readonly
-- @field _Name the module's name, readonly
-- @field _Version the module's version
----------------------------------------------------------------------------------------------------------------------------------------
do
	-- Check Version
	local version = 9

	------------------------------------------------------
	-- Version Check
	------------------------------------------------------
	local _G = _G
	local rawset = rawset
	local rawget = rawget
	local setfenv = setfenv

	IGAS.__ADDON_ENV = IGAS.__ADDON_ENV or setmetatable({}, {
		__index = function(self, key)
			if _G[key] then
				rawset(self, key, _G[key])

				return rawget(self, key)
			end
		end,
	})
	setfenv(1, IGAS.__ADDON_ENV)

	if IGAS_ADDON_VERSION and IGAS_ADDON_VERSION >= version then
		return
	end
	IGAS_ADDON_VERSION = version

	------------------------------------------------------
	-- Main Trunk
	------------------------------------------------------
	local Log = IGAS:NewLogger("IGAS")

	-- MetaWeak
	local _metaWeak = {__mode = "k"}

	-- Error Handler
	local function errorhandler(err)
		return geterrorhandler()(err)
	end

	-- Data Storage
	IGAS_ADDON_EVENTCTRL = IGAS_ADDON_EVENTCTRL or CreateFrame("Frame")
	IGAS.Addon = IGAS.Addon or {}

	_EventDistribution =  _EventDistribution or {}
	_Disabled = _Disabled or setmetatable({}, _metaWeak)
	_SavedVariables = _SavedVariables or {}
	_Modules =  _Modules or setmetatable({}, {__mode="kv"})
	_Names = _Names or setmetatable({}, _metaWeak)
	_Versions = _Versions or setmetatable({}, _metaWeak)
	_NoAutoWrapper = _NoAutoWrapper or setmetatable({}, _metaWeak)
	_metaModule = _metaModule or {}
	_metaAddon = _metaAddon or {}
	_SlashFuncs = _SlashFuncs or {}

    local _RegisterAddon = IGAS.Addon
	local _EventCtrl = IGAS_ADDON_EVENTCTRL

	-- Infomations
	local _ScriptHandler = {
		------------------------------------
		--- ScriptType, Fired when the addon is enabled.
		-- @name Addon:OnEnable
		-- @class function
		-- @return nil
		-- @usage function Addon:OnEnable()<br>
		--    -- do someting<br>
		-- end
		------------------------------------
		["OnEnable"] = true,

		------------------------------------
		--- ScriptType, Fired when the addon is diabled.
		-- @name Addon:OnDisable
		-- @class function
		-- @return nil
		-- @usage function Addon:OnDisable()<br>
		--    -- do someting<br>
		-- end
		------------------------------------
		["OnDisable"] = true,

		------------------------------------
		--- ScriptType, Fired when the addon is disposing.
		-- @name Addon:OnDispose
		-- @class function
		-- @return nil
		-- @usage function Addon:OnDispose()<br>
		--    -- do someting<br>
		-- end
		------------------------------------
		["OnDispose"] = true,

		------------------------------------
		--- ScriptType, Fired when the slash cmds are triggered.
		-- @name Addon:OnSlashCmd
		-- @class function
		-- @param msg the message that inputed with the slashcmd
		-- @return nil
		-- @usage function Addon:OnSlashCmd(msg, ...)<br>
		--    -- do someting<br>
		-- end
		------------------------------------
		["OnSlashCmd"] = true,
	}

	local _UsedTable = {
		["__StackScripts"] = true,
		["__Meta"] = true,
	}

	local _WeakTable = {
		["__Frames"] = true,
	}

	local _UsedEvent = {
		["VARIABLES_LOADED"] = true,
	}

	--  EventHandler
	_EventCtrl:Hide()

	function _EventCtrl:AddEvent(event)
		self:RegisterEvent(event)

		if not self:IsEventRegistered(event) then
			IGAS:RegisterCustomEvent(self, event)
		end
	end

	function _EventCtrl:RemoveEvent(event)
		if not _UsedEvent[event] then
			if self:IsEventRegistered(event) then
				self:UnregisterEvent(event)
			else
				IGAS:UnregisterCustomEvent(self, event)
			end
		end
	end

	function _EventCtrl:Fire(sc, ...)
		if self[sc] and type(self[sc]) == "function" then
			return self[sc](self, ...)
		end
	end

	function _EventCtrl:OnEvent(event, ...)
		local chk, ret

		if self[event] and type(self[event]) == "function" then
			chk, ret = pcall(self[event], self, ...)
			if not chk then
				errorhandler(ret)
			end
		end

		if _EventDistribution[event] then
			for v in pairs(_EventDistribution[event]) do
				if not _Disabled[v] and v[event] and type(v[event]) == "function" then
					chk, ret = pcall(v[event], v, ...)
					if not chk then
						errorhandler(ret)
					end
				end
			end
		end
	end

	_EventCtrl:SetScript("OnEvent", _EventCtrl.OnEvent)

	-- Special Events
	for event in pairs(_UsedEvent) do
		_EventCtrl:AddEvent(event)
	end

	function _EventCtrl:VARIABLES_LOADED()
		for addon, sv in pairs(_SavedVariables) do
			for svn in pairs(sv) do
				_G[svn] = _G[svn] or {}
				addon[svn] = _G[svn]
			end
		end
		wipe(_SavedVariables)
		_EventCtrl._VariablesLoaded = true
	end

	-- SlashCmd Operation
	local function GetSlashCmd(name)
		_SlashFuncs[name] = _SlashFuncs[name] or function(...)
			if _RegisterAddon[name] then
				_RegisterAddon[name]:Fire("OnSlashCmd", ...)
			end
		end
		return _SlashFuncs[name]
	end

	-- Create Addon Or Module
	local function CreateNameSpace(parent, ...)
		local name

		if select('#', ...) == 0 then
			return
		end

		for i = 1, select('#', ...) do
			name = select(i, ...)

			if parent == _RegisterAddon then
				if not rawget(parent, name) then
					rawset(parent, name, setmetatable({}, _metaAddon))
					_Names[parent[name]] = name
				elseif type(rawget(parent, name)) ~= "table" or getmetatable(rawget(parent, name)) ~= _metaAddon then
					error("There is another object with that name, Addon can't be created.", 2)
				end
			else
				if not rawget(parent, name) then
					rawset(parent, name, setmetatable({}, _metaModule))
					_Modules[parent[name]] = parent
					_Names[parent[name]] = name
				elseif type(rawget(parent, name)) ~= "table" or getmetatable(rawget(parent, name)) ~= _metaModule then
					error("There is another object with that name, Module can't be created", 2)
				end
			end

			parent = parent[name]
		end

		return parent
	end

	-- Get Addon or Module
	local function GetNameSpace(parent, ...)
		local name

		if select('#', ...) == 0 then
			return
		end

		for i = 1, select('#', ...) do
			name = select(i, ...)

			if parent == _RegisterAddon then
				if not rawget(parent, name) or type(rawget(parent, name)) ~= "table" or getmetatable(rawget(parent, name)) ~= _metaAddon then
					return nil
				end
			else
				if not rawget(parent, name) or type(rawget(parent, name)) ~= "table" or getmetatable(rawget(parent, name)) ~= _metaModule then
					return nil
				end
			end

			parent = rawget(parent, name)
		end

		return parent
	end

	local _FuncModule = {
		------------------------------------
		--- This is using to dispose a module, and set the enviroment to it's container if needed.
		-- @name Module:Dispose
		-- @class function
		-- @return nil
		-- @usage Module:Dispose()
		------------------------------------

		-- Dispose
		["Dispose"] = function(self)
			self:UnregisterAllEvents()

			for _, mdl in pairs(self) do
				if type(mdl) == "table" and getmetatable(mdl) == _metaModule then
					mdl:Dispose()
				end
			end

			if self._Name and _Modules[self] and _Modules[self][self._Name] == self then
				_Modules[self][self._Name] = nil
			end

			if getfenv(2) == self then
				if _Modules[self] then
					setfenv(2, _Modules[self])
				else
					setfenv(2, _G)
				end
			end

			_Modules[self] = nil
			_Names[self] = nil

			setmetatable(self, nil)

			wipe(self)
		end,

		------------------------------------
	    --- Create or get a child-module of this module.And set the environment to the child-module
		-- @name Module:NewModule
		-- @class function
		-- @param name the child-module's name,can be "Mdl1.SubMdl1.SubSubMdl1", use dot to concat
		-- @param version Optional,used for version control, if there is another version, and the old version is equal or great than this one, return nil, else return the child-module
		-- @return the child-module
		-- @usage Module:NewModule("Mdl_MyFrame1")
		------------------------------------

		-- NewModule
		["NewModule"] = function(self, name, version)
			if name and type(name) == "string" and strtrim(name) ~= "" then
				if version and type(version) ~= "number" then
					error("The version must be a number", 2)
				end

				local module = CreateNameSpace(self, strsplit(".", name))

				if version and _Versions[module] and _Versions[module] >= version then
					return
				elseif version then
					_Versions[module] = version
				end

				setfenv(2, module)

				return module
			end
		end,

		------------------------------------
	    --- Get the child-module of this module.
		-- @name Module:GetModule
		-- @class function
		-- @param name the child-module's name,can be "Mdl1.SubMdl1.SubSubMdl1", use dot to concat
		-- @return the child-module
		-- @usage Module:GetModule("Mdl_MyFrame1")
		------------------------------------

		-- GetModule
		["GetModule"] = function(self, name)
			if name and type(name) == "string" and strtrim(name) ~= "" then
				local module = GetNameSpace(self, strsplit(".", name))

				return module
			end
		end,

		------------------------------------
	    --- Returns true if the given event is registered to the module.
		-- @name Module:IsEventRegistered
		-- @class function
		-- @param event the event's name, the event can be a custom event.
		-- @return true if the given event is registered to the module.
		-- @usage Module:IsEventRegistered("PLAYER_LOGIN")
		------------------------------------

		-- IsEventRegistered
        ["IsEventRegistered"] = function(self, event)
			if event and type(event) == "string" and event ~= "" then
				-- Custom Event
				return (_EventDistribution[event] and _EventDistribution[event][self] and true) or false
			else
				return false
			end
        end,

		------------------------------------
	    --- Indicate that this module should be notified when event occurs.
		-- @name Module:RegisterEvent
		-- @class function
		-- @param event the event's name, the event can be a custom event.
		-- @return nil
		-- @usage Module:RegisterEvent("PLAYER_LOGIN")
		------------------------------------

		-- RegisterEvent
        ["RegisterEvent"] = function(self, event)
			if event and type(event) == "string" and event ~= "" then
				_EventDistribution[event] = _EventDistribution[event] or setmetatable({}, _metaWeak)

				_EventDistribution[event][self] = true

				return _EventCtrl:AddEvent(event)
			end
        end,

		------------------------------------
	    --- Indicate that this module should no longer be notified when event occurs.
		-- @name Module:UnregisterEvent
		-- @class function
		-- @param event the event's name, the event can be a custom event.
		-- @return nil
		-- @usage Module:UnregisterEvent("PLAYER_LOGIN")
		------------------------------------

		-- UnregisterEvent
        ["UnregisterEvent"] = function(self, event)
			if event and type(event) == "string" and event ~= "" then
				if _EventDistribution[event] then
					_EventDistribution[event][self] = nil
					if not next(_EventDistribution[event]) then
						_EventCtrl:RemoveEvent(event)
					end
				end
			end
        end,

		------------------------------------
	    --- Indicate that this module should no longer be notified when all event occurs.
		-- @name Module:UnregisterAllEvents
		-- @class function
		-- @return nil
		-- @usage Module:UnregisterAllEvents()
		------------------------------------

		-- UnregisterAllEvents
		["UnregisterAllEvents"] = function(self)
			for event, list in pairs(_EventDistribution) do
				if list[self] then
					list[self] = nil
					if not next(list) then
						_EventCtrl:RemoveEvent(event)
					end
				end
			end
		end,

		------------------------------------
	    --- Fire Event, can only be used to fire custom event, fire system event will do nothing
		-- @name Module:FireEvent
		-- @class function
		-- @param event the event's name, the event MUST be a custom event.
		-- @param ... the parameters of the event
		-- @return nil
		-- @usage Module:FireEvent("MY_CUSTOM_EVENT_1", "First Arg", "Sceond arg")
		------------------------------------

		-- FireEvent
		["FireEvent"] = function(self, event, ...)
			if not event or type(event) ~= "string" or event == "" then
				return
			end

			return IGAS:FireCustomEvent(_EventCtrl, event, ...)
		end,
	}

	local _FuncAddon = {
		------------------------------------
	    --- Add slash commands to an addon
		-- @name Addon:AddSlashCmd
		-- @class function
		-- @param slash1 the slash commands, a string started with "/", if not started with "/", will auto add it.
		-- @param slash2, ... Optional,other slash commands
		-- @return nil
		-- @usage Addon:AddSlashCmd("/hw", hw2")
		------------------------------------

		-- AddSlashCmd
		["AddSlashCmd"] = function(self, ...)
			if select('#', ...) == 0 then
				return nil
			end

			local AddName = _Names[self]

			local slash = {}
			local sl
			for i = 1, select('#', ...) do
				sl = select(i, ...)
				if sl and type(sl) == "string" then
					if strmatch(sl, "^/%S+") then
						slash[strmatch(sl, "^/%S+")] = true
					elseif strmatch(sl, "^%S+") then
						slash["/"..strmatch(sl, "^%S+")] = true
					end
				end
			end

			if not next(slash) then
				error("There must be some slash commands", 2)
			end

			local index = 1

			while _G["SLASH_"..AddName..index] do
				slash[_G["SLASH_"..AddName..index]] = nil
				index = index + 1
			end

			for cmd in pairs(slash) do
				_G["SLASH_"..AddName..index] = cmd
				index = index + 1
			end

			if not _G["SlashCmdList"][AddName] then
				_G["SlashCmdList"][AddName] = GetSlashCmd(AddName)
			end
		end,

		------------------------------------
	    --- Add SaveVariables to the addon
		-- @name Addon:AddSavedVariable
		-- @class function
		-- @param name the savevariable's name
		-- @return the savevariable, no need to receive it, the name passed in this function will be the identify.
		-- @usage Addon:AddSavedVariable("HelloWorld_SaveV")
		------------------------------------

		-- AddSavedVariable
		["AddSavedVariable"] = function(self, name)
			if name and type(name) == "string" and strtrim(name) ~= "" then
				name = strtrim(name)
				if not _EventCtrl._VariablesLoaded then
					_SavedVariables[self] = _SavedVariables[self] or {}
					_SavedVariables[self][name] = true
				end

				_G[name] = _G[name] or {}
				self[name] = _G[name]
				return self[name]
			else
				error("The SavedVariable's name must be a string", 2)
			end
		end,

		------------------------------------
	    --- Returns true if the given event is registered to the addon.
		-- @name Addon:IsEventRegistered
		-- @class function
		-- @param event the event's name, the event can be a custom event.
		-- @return true if the given event is registered to the addon.
		-- @usage Addon:IsEventRegistered("PLAYER_LOGIN")
		------------------------------------

		-- IsEventRegistered
        ["IsEventRegistered"] = function(self, event)
			if event and type(event) == "string" and event ~= "" then
				-- Custom Event
				return (_EventDistribution[event] and _EventDistribution[event][self] and true) or false
			else
				return false
			end
        end,

		------------------------------------
	    ---  Indicate that this addon should be notified when event occurs.
		-- @name Addon:RegisterEvent
		-- @class function
		-- @param event the event's name, the event can be a custom event.
		-- @return nil
		-- @usage Addon:RegisterEvent("PLAYER_LOGIN")
		------------------------------------

		-- RegisterEvent
        ["RegisterEvent"] = function(self, event)
			if event and type(event) == "string" and event ~= "" then
				_EventDistribution[event] = _EventDistribution[event] or setmetatable({}, _metaWeak)

				_EventDistribution[event][self] = true

				return _EventCtrl:AddEvent(event)
			end
        end,

		------------------------------------
	    ---  Indicate that this addon should no longer be notified when event occurs.
		-- @name Addon:UnregisterEvent
		-- @class function
		-- @param event the event's name, the event can be a custom event.
		-- @return nil
		-- @usage Addon:UnregisterEvent("PLAYER_LOGIN")
		------------------------------------

		-- UnregisterEvent
        ["UnregisterEvent"] = function(self, event)
			if event and type(event) == "string" and event ~= "" then
				if _EventDistribution[event] then
					_EventDistribution[event][self] = nil
					if not next(_EventDistribution[event]) then
						_EventCtrl:RemoveEvent(event)
					end
				end
			end
        end,

		------------------------------------
	    ---  Indicate that this addon should no longer be notified when all event occurs.
		-- @name Addon:UnregisterAllEvents
		-- @class function
		-- @return nil
		-- @usage Addon:UnregisterAllEvents()
		------------------------------------

		-- UnregisterAllEvents
		["UnregisterAllEvents"] = function(self)
			for event, list in pairs(_EventDistribution) do
				if list[self] then
					list[self] = nil
					if not next(list) then
						_EventCtrl:RemoveEvent(event)
					end
				end
			end
		end,

		------------------------------------
	    --- Fire Event, can only be used to fire custom event, fire system event will do nothing
		-- @name Addon:FireEvent
		-- @class function
		-- @param event the event's name, the event MUST be a custom event.
		-- @param ... the parameters of the event
		-- @return nil
		-- @usage Addon:FireEvent("MY_CUSTOM_EVENT_1", "First Arg", "Sceond arg")
		------------------------------------

		-- FireEvent
		["FireEvent"] = function(self, event, ...)
			if not event or type(event) ~= "string" or event == "" then
				return
			end

			return IGAS:FireCustomEvent(_EventCtrl, event, ...)
		end,

		------------------------------------
	    --- Fire Script, trigger the script's handle
		-- @name Addon:Fire
		-- @class function
		-- @param scripType the addon's script's type
		-- @param ... the parameters
		-- @return nil
		-- @usage Addon:Fire("OnEnable")
		------------------------------------

		-- Fire
		["Fire"] = function(self, sc, ...)
			local chk = false
			local ret = false

			if not sc then return end

			if self.__StackScripts[sc] then
				for i = 1, getn(self.__StackScripts[sc]) do
					chk, ret = pcall(self.__StackScripts[sc][i], self, ...)
					if not chk then
						return errorhandler(ret)
					end
					if not self.__StackScripts then
						-- means it's disposed
						ret = true
					end
					if ret then
						break
					end
				end

				if not ret and self.__StackScripts[sc][0] then
					chk, ret = pcall(self.__StackScripts[sc][0], self, ...)
					if not chk then
						return errorhandler(ret)
					end
				end
			elseif self[sc] and type(self[sc]) == "function" then
				chk, ret = pcall(self[sc], self, ...)
				if not chk then
					return errorhandler(ret)
				end
			end
		end,

		------------------------------------
	    --- Returns true if the given scriptType is supported for the addon.
		-- @name Addon:HasScript
		-- @class function
		-- @param scriptType the script's type, must be a string
		-- @return true if the given scriptType is supported for the addon.
		-- @usage Addon:HasScript("OnEnable")    -- return true
		------------------------------------

		-- HasScript
		["HasScript"] = function(self, handler)
			if not handler or type(handler) ~= "string" then
				return false
			end
			return (_ScriptHandler[handler] and true) or false
		end,

		------------------------------------
	    --- Returns the addon's script handler by the given name
		-- @name Addon:GetScript
		-- @class function
		-- @param scriptType the script's type, must be a string
		-- @return the addon's script handler by the given name
		-- @usage Addon:GetScript("OnEnable")
		------------------------------------

		-- GetScript
		["GetScript"] = function(self, handler)
			if not handler or type(handler) ~= "string" then
				return nil
			end
			return (self.__StackScripts[handler] and self.__StackScripts[handler][0]) or nil
		end,

		------------------------------------
	    --- Set the addon's script handler by the given name
		-- @name Addon:SetScript
		-- @class function
		-- @param scriptType the script's type
		-- @param func the handler
		-- @return nil
		-- @usage Addon:SetScript("OnEnable", function() print("HI") end)
		------------------------------------

		-- SetScript
		["SetScript"] = function(self, handler, func)
			if not handler or type(handler) ~= "string" then
				error("The handler can't be use", 2)
			end
			if not self:HasScript(handler) then
				error("The handler is not exist.", 2)
			end
			if func and type(func) ~= "function" then
				error("This is a script handler, must be a function", 2)
			end

			if not self.__StackScripts[handler] or type(self.__StackScripts[handler]) ~= "table" then
				self.__StackScripts[handler] = {}
			end

			rawset(self.__StackScripts[handler], 0, func)
		end,

		------------------------------------
	    --- Stack the addon's script handler by the given name. This wouldn't remove the prev handler
		-- @name Addon:StackScript
		-- @class function
		-- @param scriptType the script's type
		-- @param func the handler
		-- @param flag if true clear all prev stacked handlers.
		-- @return nil
		-- @usage Addon:StackScript("OnEnable", function() print("HI") end)
		------------------------------------

		-- StackScript
		["StackScript"] = function(self, handler, func, flag)
			if type(handler) ~= "string" then
				error("The handler's name must be a string", 2)
			end
			if not self:HasScript(handler) then
				error("The handler is not exist.", 2)
			end
			if func and type(func) ~= "function" then
				error("The handler must be a function", 2)
			end

			if not self.__StackScripts[handler] or type(self.__StackScripts[handler]) ~= "table" then
				self.__StackScripts[handler] = {}
			end

			if flag then
				-- Clear this handler's stack
				for i = getn(self.__StackScripts[handler]), 1, -1 do
					tremove(self.__StackScripts[handler], i)
				end
			end

			-- Check if it exist
			for i = getn(self.__StackScripts[handler]), 1, -1 do
				if self.__StackScripts[handler][i] == func then
					return
				end
			end

			tinsert(self.__StackScripts[handler], func)
		end,

		------------------------------------
	    --- Undo Stack the addon's script handler by the given name.
		-- @name Addon:UnStackScript
		-- @class function
		-- @param scriptType the script's type
		-- @param func the handler
		-- @return nil
		-- @usage Addon:UnStackScript("OnEnable", function() print("HI") end)
		------------------------------------

		-- UnStackScript
		["UnStackScript"] = function(self, handler, func)
			if type(handler) ~= "string" then
				error("The handler's name must be a string", 2)
			end
			if not self:HasScript(handler) then
				error("The handler is not exist.", 2)
			end
			if not func then
				error("Don't use nil for unstack function.", 2)
			elseif func and type(func) ~= "function" then
				error("The unstack function must be a function", 2)
			end

			if not self.__StackScripts[handler] or type(self.__StackScripts[handler]) ~= "table" then
				return
			end

			-- Clear this handler's stack
			for i = getn(self.__StackScripts[handler]), 1, -1 do
				if self.__StackScripts[handler][i] == func then
					tremove(self.__StackScripts[handler], i)
					return
				end
			end
		end,

		------------------------------------
	    --- Get the addon's information by the given name.
		-- @name Addon:GetMetadata
		-- @class function
		-- @param field the informations' name
		-- @return infomations stored in the addon
		-- @usage Addon:GetMetadata("Author")    -- return "Kurapica"
		------------------------------------

		-- GetMetadata
		["GetMetadata"] = function(self, field)
			return self.__Meta[field] or GetAddOnMetadata(self.__Meta["Name"], field)
		end,

		------------------------------------
	    --- Set the addon's information by the given name.
		-- @name Addon:SetMetadata
		-- @class function
		-- @param field the informations' name
		-- @return infomations stored in the addon
		-- @usage Addon:SetMetadata("Author", "Kurapica")
		------------------------------------

		-- SetMetadata
		["SetMetadata"] = function(self, field, value)
			if field and type(field) == "string" then
				self.__Meta[field] = tostring(value)
			end
		end,

		------------------------------------
	    --- Enable the Addon.
		-- @name Addon:Enable
		-- @class function
		-- @return nil
		-- @usage Addon:Enable()
		------------------------------------

		-- Enable
		["Enable"] = function(self)
			_Disabled[v] = nil
			return self:Fire("OnEnable")
		end,

		------------------------------------
	    --- Enable the Addon.The addon won't receive events.
		-- @name Addon:Disable
		-- @class function
		-- @return nil
		-- @usage Addon:Disable()
		------------------------------------

		-- Disable
		["Disable"] = function(self)
			_Disabled[v] = true
			return self:Fire("OnDisable")
		end,

		------------------------------------
	    --- Check if the addon is enabled
		-- @name Addon:IsEnabled
		-- @class function
		-- @return true if the addon is enabled
		-- @usage Addon:IsEnabled()
		------------------------------------

		-- IsEnabled
		["IsEnabled"] = function(self)
			return (not _Disabled[v]) or false
		end,

		------------------------------------
	    --- Add a frame to addon.
		-- @name Addon:AddFrame
		-- @class function
		-- @param frame the user-defined frame that needed to be add to the addon, must be the parent of other user-defined frames, will be auto-diposed when addon is dispoing.Also make the frame can be access in the addon's scope by it's name.
		-- @return nil
		-- @usage Addon:AddFrame(MyFrame1)
		------------------------------------

		-- AddFrame
		["AddFrame"] = function(self, frame)
			if frame and type(frame) == "table" and frame.Name then
				self.__Frames[frame] = true
				self[frame.Name] = frame
			end
		end,

		------------------------------------
	    --- Remove a frame from addon.
		-- @name Addon:RemoveFrame
		-- @class function
		-- @param frame the user-defined frame that needed to be removed to the addon, and the frame will be disposed.
		-- @return nil
		-- @usage Addon:RemoveFrame(MyFrame1)
		------------------------------------

		-- RemoveFrame
		["RemoveFrame"] = function(self, frame)
			if frame and type(frame) == "table" and frame.Name then
				self.__Frames[frame] = nil
				if self[frame.Name] == frame then
					self[frame.Name] = nil
				end
				frame:Dispose()
			end
		end,

		------------------------------------
	    --- Dispose the addon.
		-- @name Addon:Dispose
		-- @class function
		-- @return nil
		-- @usage Addon:Dispose()
		------------------------------------

		-- Dispose
		["Dispose"] = function(self)
			self:Fire("OnDispose")

			for fr in pairs(self.__Frames) do
				if fr.Dispose then
					fr:Dispose()
				end
			end

			self:UnregisterAllEvents()

			for _, mdl in pairs(self) do
				if type(mdl) == "table" and getmetatable(mdl) == _metaModule then
					mdl:Dispose()
				end
			end

			for i, v in pairs(_RegisterAddon) do
				if v == self then
					_RegisterAddon[i] = nil
					break
				end
			end

			if getfenv(2) == self then
				setfenv(2, _G)
			end

			_SavedVariables[self] = nil
			_Names[self] = nil

			setmetatable(self, nil)

			wipe(self)
		end,

		------------------------------------
	    --- Create or get a child-module of this addon.And set the environment to the child-module
		-- @name Addon:NewModule
		-- @class function
		-- @param name the child-module's name,can be "Mdl1.SubMdl1.SubSubMdl1", use dot to concat
		-- @param version Optional,used for version control, if there is another version, and the old version is equal or great than this one, return nil, else return the child-module
		-- @return the child-module
		-- @usage Addon:NewModule("Mdl_MyFrame1")
		------------------------------------

		-- NewModule
		["NewModule"] = function(self, name, version)
			if name and type(name) == "string" and strtrim(name) ~= "" then
				if version and type(version) ~= "number" then
					error("The version must be a number", 2)
				end

				local module = CreateNameSpace(self, strsplit(".", name))

				if version and _Versions[module] and _Versions[module] >= version then
					return
				elseif version then
					_Versions[module] = version
				end

				setfenv(2, module)

				return module
			end
		end,

		------------------------------------
	    --- Get the child-module of this addon.
		-- @name Addon:GetModule
		-- @class function
		-- @param name the child-module's name,can be "Mdl1.SubMdl1.SubSubMdl1", use dot to concat
		-- @return the child-module
		-- @usage Addon:GetModule("Mdl_MyFrame1")
		------------------------------------

		-- GetModule
		["GetModule"] = function(self, name)
			if name and type(name) == "string" and strtrim(name) ~= "" then
				local module = GetNameSpace(self, strsplit(".", name))

				return module
			end
		end,
	}

	local _PropModule = {
		["_M"] = {
			["Get"] = function(self)
				return self
			end,
		},

		["_Name"] = {
			["Get"] = function(self)
				return _Names[self]
			end,
		},

		["_Version"] = {
			["Get"] = function(self)
				return _Versions[self] or 0
			end,

			["Set"] = function(self, version)
				if version and type(version) == "number" and version > (_Versions[self] or 0) then
					_Versions[self] = version
				else
					error("The version must be a number and must be bigger than the previous version.", 2)
				end
			end,
		},
	}

	local _PropAddon = {
		["_Addon"] = {
			["Get"] = function(self)
				return self
			end,
		},

		["_Name"] = {
			["Get"] = function(self)
				return _Names[self]
			end,
		},

		["_Version"] = {
			["Get"] = function(self)
				return _Versions[self] or 0
			end,

			["Set"] = function(self, version)
				if version and type(version) == "number" and version > (_Versions[self] or 0) then
					_Versions[self] = version
				else
					error("The version must be a number and must be bigger than the previous version.", 2)
				end
			end,
		},

		["_AutoWrapper"] = {
			["Get"] = function(self)
				return not _NoAutoWrapper[self]
			end,

			["Set"] = function(self, auto)
				_NoAutoWrapper[self] = not auto
			end,
		},
	}

	-- Metatable Define
	_metaModule.__index = function(_table, _key)
		-- Property Get
		if _PropModule[_key] then
			if _PropModule[_key]["Get"] then
				return _PropModule[_key]["Get"](_table)
			else
				error(_key.." is write-only.",2)
			end
		end

		if _FuncModule[_key] then
			return _FuncModule[_key]
		end

		if _Modules[_table] then
			return _Modules[_table][_key]
		end

		return _G[_key]
	end

    _metaAddon.__index = function(_table, _key)
		-- Special key for table
		if _WeakTable[_key] then
			rawset(_table, _key, setmetatable({}, _metaWeak))
			return rawget(_table, _key)
		end

		if _UsedTable[_key] then
			rawset(_table, _key, {})
			return rawget(_table, _key)
		end

		-- Property Get
		if _PropAddon[_key] then
			if _PropAddon[_key]["Get"] then
				return _PropAddon[_key]["Get"](_table)
			else
				error(_key.." is write-only.",2)
			end
		end

		if _FuncAddon[_key] then
			return _FuncAddon[_key]
		end

		-- Scripts
		if _table.__StackScripts[_key] then
			return rawget(_table.__StackScripts[_key], 0)
		end

		if _G[_key] then
			if type(_G[_key]) == "function" or type(_G[_key]) == "table" then
				if (not _NoAutoWrapper[_table]) and IGAS.GetWrapper then
					rawset(_table, _key, IGAS:GetWrapper(_G[_key]))
				else
					rawset(_table, _key, _G[_key])
				end
				return rawget(_table, _key)
			end

			return _G[_key]
		end
	end

	_metaModule.__newindex = function(_table, _key, _value)
		-- Property Set
		if _PropModule[_key] then
			if _PropModule[_key]["Set"] then
				return _PropModule[_key]["Set"](_table,_value)
			else
				error(_key.." is read-only.", 2)
			end
		end

		rawset(_table,_key,_value)			-- Other key can be set as usual
	end

	_metaAddon.__newindex = function(_table, _key, _value)
		-- Property Set
		if _PropAddon[_key] then
			if _PropAddon[_key]["Set"] then
				return _PropAddon[_key]["Set"](_table,_value)
			else
				error(_key.." is read-only.", 2)
			end
		end

		-- Scripts
		if _ScriptHandler[_key] then
			if _value and type(_value) ~= "function" then
				error("The script handler must be a function", 2)
			end

			if not _table.__StackScripts[_key] or type(_table.__StackScripts[_key]) ~= "table" then
				rawset(_table.__StackScripts, _key, {})
			end

			rawset(_table.__StackScripts[_key], 0, _value)

			return
		end

		rawset(_table,_key,_value)			-- Other key can be set as usual
	end

	------------------------------------
	--- Create or Get an addon by name, then change the environment to the addon if not set setEnv to false
	-- @name IGAS:NewAddon
	-- @class function
	-- @param addonName	the addon's name, must be unique, also can be a serie tokens seperate by ".",the first token is addon' name, the second token will be a child-module to the addon, the third token will create the child-module to the second, and go on.
	-- @param version Optional,used for version control, if there is another version, and the old version is equal or great than this one, return nil, else return the child-module
	-- @param info Optional,Optional,must be a table, contains informations such as "author", "version" .etc
	-- @return addon that a table contains the addon's scope, no need to receive it if you don't set setEnv to false, after call this function, you can using "_Addon" to access the addon
	-- @usage local addon = IGAS:NewAddon("HelloWorld", 1, {Author="Kurapica", Version="1.0", Note="Hello World Example", Interface="30010", Title="Hello World"})
	-- @usage IGAS:NewAddon("HelloWorld")
	------------------------------------
	IGAS:RegisterFunction("NewAddon", function(self, addonName, version, info)
		if addonName and type(addonName) == "string" and strtrim(addonName) ~= "" then
			if version and type(version) ~= "number" then
				error("The version must be a number", 2)
			end

			Log(1, "[NewAddon]"..addonName..((version and (" [Version]"..version)) or ""))

			local namespace = CreateNameSpace(_RegisterAddon, strsplit(".", addonName))

			if version and _Versions[namespace] and _Versions[namespace] >= version then
				return
			elseif version then
				_Versions[namespace] = version
			end

			-- Get addon
			if info and type(info) == "table" and next(info) then
				local addon = namespace

				while _Modules[addon] do
					addon = _Modules[addon]
				end

				for k, v in pairs(info) do
					if type(k) == "string" then
						addon.__Meta[k] = tostring(v)
					end
				end
			end

			setfenv(2, namespace)

			return namespace
		end
	end)

	------------------------------------
	--- Get an addon by name
	-- @name IGAS:GetAddon
	-- @class function
	-- @param addonName	the addon's name, must be unique, also can be a serie tokens seperate by ".",the first token is addon' name, the second token will be a child-module to the addon, the third token will create the child-module to the second, and go on.
	-- @return addon that a table contains the addon's scope, no need to receive it if you don't set setEnv to false, after call this function, you can using "_Addon" to access the addon
	-- @usage local addon = IGAS:GetAddon("HelloWorld")
	-- @usage IGAS:GetAddon("HelloWorld")
	------------------------------------
	IGAS:RegisterFunction("GetAddon", function(self, addonName)
		if addonName and type(addonName) == "string" and strtrim(addonName) ~= "" then
			return GetNameSpace(_RegisterAddon, strsplit(".", addonName))
		end
	end)

	------------------------------------
	--- Check if an addon is existed with the given name
	-- @name IGAS:IsAddonExist
	-- @class function
	-- @param addonName the addon's name that you query
	-- @return result a boolean flag, true if addon with that name is existed, else false
	-- @usage IGAS:IsAddonExist("HelloWorld")
	------------------------------------
	IGAS:RegisterFunction("IsAddonExist", function(self, addonName)
		return (_RegisterAddon[addonName] and true) or false
	end)
end