--[[
ClosetGnome, a World of Warcraft addon to manage item sets.
Modified in WoW 3.1 to just be a LDB plugin, handle keybindings and display the quips.
Copyright (C) 2010 Rabbit.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
]]

local ClosetGnome = CreateFrame("Frame")
_G.ClosetGnome = ClosetGnome
-- Just so older closetgnome macros will still work for people.
function ClosetGnome:WearSet(s) EquipmentManager_EquipSet(s) end
ClosetGnome:SetScript("OnEvent", function(self, event, ...)
	self[event](self, ...)
end)

local db = nil
local L = LibStub("AceLocale-3.0"):GetLocale("ClosetGnome")
local icon = LibStub("LibDBIcon-1.0", true)
local ldb = LibStub:GetLibrary("LibDataBroker-1.1", true)
local defaultIcon = "Interface\\Icons\\INV_Chest_Cloth_17"
local menuNeedsUpdate = nil

local CGLDB = nil
if ldb then
	CGLDB = ldb:NewDataObject("ClosetGnome", {
		type = "data source",
		text = "ClosetGnome",
		icon = defaultIcon,
	})
end

-------------------------------------------------------------------------------
-- Keybindings
--

local cgKeyFrame = CreateFrame("Button", "ClosetGnomeKeyFrame", UIParent)
cgKeyFrame:SetScript("OnMouseUp", function(self, set) EquipmentManager_EquipSet(set) end)

-- Verify that the given equipment set actually exists.
-- The Blizzard code does not (currently) invoke DeleteEquipmentSet with
-- the sets numeric IDs, it always uses the names.
local function setExists(set)
	local exists = GetEquipmentSetInfoByName(set)
	if not exists or type(exists) ~= "string" or exists:trim() == "" then return false end
	return true
end

local function clearBinding(set)
	if not set or not db.keybindings[set] or not setExists(set) then return end
	SetOverrideBinding(cgKeyFrame, false, db.keybindings[set])
	db.keybindings[set] = nil
	menuNeedsUpdate = true
end

local function registerBinding(combo, set)
	if not combo or not set or not setExists(set) then return end
	if db.keybindings[set] then
		SetOverrideBinding(cgKeyFrame, false, db.keybindings[set])
	end
	db.keybindings[set] = combo
	SetOverrideBindingClick(cgKeyFrame, false, combo, "ClosetGnomeKeyFrame", set)
	menuNeedsUpdate = true
end

-------------------------------------------------------------------------------
-- Config UI
--

local updateKeybindingButtons = nil
do
	local frame = CreateFrame("Frame", nil, InterfaceOptionsFramePanelContainer)
	frame.name = "ClosetGnome"
	frame:Hide()

	local buttons = {}
	local label = "%s: %s"
	local function updateButton(button)
		local set = button.set
		if db.keybindings[set] then
			button:SetNormalFontObject("GameFontHighlight")
			button:SetText(label:format(set, db.keybindings[set]))
		else
			button:SetNormalFontObject("GameFontNormal")
			button:SetText(label:format(set, NOT_BOUND))
		end
	end

	local function clearButton(button)
		if not button:IsKeyboardEnabled() then return end
		button:EnableKeyboard(false)
		button:UnlockHighlight()
	end

	local function onClick(self, b)
		if b ~= "LeftButton" then return end
		for i, button in next, buttons do
			if button ~= self then
				clearButton(button)
			end
		end
		if self:IsKeyboardEnabled() then
			clearButton(self)
		else
			self:EnableKeyboard(true)
			self:LockHighlight()
		end
	end

	local ignoreKeys = {
		UNKNOWN = true,
		LSHIFT = true, LCTRL = true, LALT = true,
		RSHIFT = true, RCTRL = true, RALT = true,
	}
	local function onKeyDown(self, key)
		if self:IsKeyboardEnabled() then
			local keyPressed = key
			if keyPressed == "ESCAPE" then
				keyPressed = nil
			else
				if ignoreKeys[keyPressed] then return end
				if IsShiftKeyDown() then
					keyPressed = "SHIFT-"..keyPressed
				end
				if IsControlKeyDown() then
					keyPressed = "CTRL-"..keyPressed
				end
				if IsAltKeyDown() then
					keyPressed = "ALT-"..keyPressed
				end
			end
			self:EnableKeyboard(false)
			self:UnlockHighlight()

			if not keyPressed or keyPressed == "" then
				clearBinding(self.set)
			else
				registerBinding(keyPressed, self.set)
			end
			updateButton(self)
		end
	end

	local function createScrollButtons(parent, num)
		for i = 1, num do
			local button = CreateFrame("Button", "CGScrollButton" .. i, parent, "UIPanelButtonTemplate2")
			button:SetScript("OnClick", onClick)
			button:SetScript("OnKeyDown", onKeyDown)
			button:SetHeight(25)
			button.tooltipText = L["Keybinding"]
			button.newbieText = L["Click and press a key combination to bind it to this equipment set. Escape clears an existing binding."]
			if i == 1 then
				button:SetPoint("TOPLEFT", parent, 12, -8)
			else
				button:SetPoint("TOPLEFT", buttons[i - 1], "BOTTOMLEFT", 0, -2)
				button:SetPoint("TOPRIGHT", buttons[i - 1], "BOTTOMRIGHT", 0, -2)
			end
			buttons[i] = button
		end
	end

	local function newCheckbox(label, onClick)
		local check = CreateFrame("CheckButton", nil, frame)
		check:SetWidth(26)
		check:SetHeight(26)
		check:SetHitRectInsets(0, -100, 0, 0)
		check:SetNormalTexture("Interface\\Buttons\\UI-CheckBox-Up")
		check:SetPushedTexture("Interface\\Buttons\\UI-CheckBox-Down")
		check:SetHighlightTexture("Interface\\Buttons\\UI-CheckBox-Highlight")
		check:SetCheckedTexture("Interface\\Buttons\\UI-CheckBox-Check")
		check:SetScript("OnClick", function(self)
			PlaySound(self:GetChecked() and "igMainMenuOptionCheckBoxOn" or "igMainMenuOptionCheckBoxOff")
			onClick(self, self:GetChecked() and true or false)
		end)
		local fs = check:CreateFontString(nil, "ARTWORK", "GameFontHighlight")
		fs:SetPoint("LEFT", check, "RIGHT", 0, 1)
		fs:SetPoint("RIGHT", frame)
		fs:SetJustifyH("LEFT")
		fs:SetText(label)
		fs:SetWidth(200)
		return check
	end

	frame:SetScript("OnShow", function(frame)
		local title = frame:CreateFontString(nil, "ARTWORK", "GameFontNormalLarge")
		title:SetPoint("TOPLEFT", 16, -16)
		title:SetText("ClosetGnome")
		local subtitle = frame:CreateFontString(nil, "ARTWORK", "GameFontHighlight")
		subtitle:SetPoint("TOPLEFT", title, "BOTTOMLEFT", 0, -8)
		subtitle:SetWidth(frame:GetWidth() - 24)
		subtitle:SetJustifyH("LEFT")
		subtitle:SetJustifyV("TOP")
		subtitle:SetText(L["closetgnome_description"])

		local quips = newCheckbox(
			L["Quips"],
			function(self, value) ClosetGnome.db.profile.quips = value end)
		quips:SetChecked(ClosetGnome.db.profile.quips)
		quips:SetPoint("TOPLEFT", subtitle, "BOTTOMLEFT", -2, -8)

		local minimap = nil
		if icon and CGLDB and icon:IsRegistered("ClosetGnome") then
			minimap = newCheckbox(
				L["Show minimap icon"],
				function(self, value)
					local hide = not value
					ClosetGnome.db.profile.minimap.hide = hide
					if hide then
						icon:Hide("ClosetGnome")
					else
						icon:Show("ClosetGnome")
					end
				end)
			minimap:SetChecked(not ClosetGnome.db.profile.minimap.hide)
			minimap:SetPoint("TOPLEFT", quips, "BOTTOMLEFT", 0, -4)
		end

		local bindings = CreateFrame("Frame", "CGKeyBinds", frame)
		bindings:SetPoint("TOPLEFT", minimap or quips, "BOTTOMLEFT", 0, -8)
		bindings:SetPoint("BOTTOMRIGHT", frame, "BOTTOMRIGHT", -16, 12)
		bindings:SetBackdrop({
			bgFile = "Interface\\Tooltips\\UI-Tooltip-Background",
			edgeFile = "Interface\\Tooltips\\UI-Tooltip-Border",
			edgeSize = 16,
			insets = {left = 5, right = 5, top = 5, bottom = 5}
		})
		local tdc = TOOLTIP_DEFAULT_COLOR
		local tdbc = TOOLTIP_DEFAULT_BACKGROUND_COLOR
		bindings:SetBackdropColor(tdbc.r, tdbc.g, tdbc.b)
		bindings:SetBackdropBorderColor(tdc.r, tdc.g, tdc.b, 0.5)

		local displayNum = math.floor(bindings:GetHeight() / 25) - 1
		createScrollButtons(bindings, displayNum)

		local scroll = CreateFrame("ScrollFrame", "CGKBScroll", bindings, "FauxScrollFrameTemplateLight")
		scroll:SetPoint("TOPLEFT", bindings, "TOPLEFT", 5, -6)
		scroll:SetPoint("BOTTOMRIGHT", bindings, "BOTTOMRIGHT", -28, 4)

		local tmp = {}
		local function update()
			wipe(tmp)
			for i = 1, GetNumEquipmentSets() do
				local name = GetEquipmentSetInfo(i)
				tmp[#tmp + 1] = name
			end
			local offset = FauxScrollFrame_GetOffset(scroll)
			local index = nil
			for i = 1, displayNum do
				index = offset + i
				if index <= #tmp then
					buttons[i].set = tmp[index]
					updateButton(buttons[i])
					buttons[i]:Show()
				else
					buttons[i]:Hide()
				end
				clearButton(buttons[i])
			end
			if FauxScrollFrame_Update(scroll, #tmp, displayNum, 25) then
				buttons[1]:SetPoint("TOPRIGHT", bindings, -29, -7)
			else
				buttons[1]:SetPoint("TOPRIGHT", bindings, -12, -7)
			end
		end
		scroll:SetScript("OnVerticalScroll", function(self, offset)
			FauxScrollFrame_OnVerticalScroll(self, offset, 25, update)
		end)

		update()

		updateKeybindingButtons = update
		frame:SetScript("OnShow", update)
	end)
	InterfaceOptions_AddCategory(frame)
end

local NUM_QUIPS = L["num_quips"]
local function getQuip(input)
	local rand = 1
	repeat rand = math.random(1, NUM_QUIPS) until L[rand] ~= nil
	return "|cffeda55f"..L[rand]:format("|cffd9d919"..input.."|cffeda55f").."|r"
end

--------------------------------------------------------------------------------
-- Utility
--

-- Returns the name of the currently equipped set, or nil if none matchs.
-- If one set is included in another set, the set with the higher number of registered slots is returned.
local function guessEquippedSet()
	local currentSlotCount, currentName = 0, nil
	for i = 1, GetNumEquipmentSets() do
		local name = GetEquipmentSetInfo(i)
		local locations = GetEquipmentSetLocations(name)
		if locations then
			local slotCount, slotMatches = 0, 0
			for slot = INVSLOT_FIRST_EQUIPPED, INVSLOT_LAST_EQUIPPED do
				local location = locations[slot]
				if location then
					slotCount = slotCount + 1
					if location == EQUIPMENT_SET_EMPTY_SLOT then
						if not GetInventoryItemID("player", slot) then
							slotMatches = slotMatches + 1
						end
					elseif location == EQUIPMENT_SET_IGNORED_SLOT then
						slotMatches = slotMatches + 1
					elseif location ~= EQUIPMENT_SET_ITEM_MISSING then
						local player, bank, bags, actualSlot = EquipmentManager_UnpackLocation(location)
						if player and not bank and not bags and slot == actualSlot then
							slotMatches = slotMatches + 1
						end
					end
				end
			end
			if slotMatches == slotCount and slotCount > currentSlotCount then
				currentName, currentSlotCount = name, slotCount
			end
		end
	end
	return currentName
end

local function openOptions()
	InterfaceOptionsFrame_OpenToCategory("ClosetGnome")
end

--------------------------------------------------------------------------------
-- Events
--

function ClosetGnome:VARIABLES_LOADED()
	SetCVar("equipmentManager", 1)
	GearManagerToggleButton:Show()
end

function ClosetGnome:ADDON_LOADED(addon)
	if addon:lower() ~= "closetgnome" then return end

	self.db = LibStub("AceDB-3.0"):New("ClosetGnomeDB", {
		profile = {
			quips = true,
			minimap = {
				hide = false,
			},
		},
		char = {
			keybindings = {},
		},
	}, true)
	db = self.db.char

	SlashCmdList["CLOSETGNOME"] = openOptions
	SLASH_CLOSETGNOME1 = "/closetgnome"
	SLASH_CLOSETGNOME2 = "/cg"

	-- We don't even remove keybindings when a player
	-- deletes a set. If he later re-adds a set with the same
	-- name, chances are real good he wants the same keybinding
	-- anyway.
	--hooksecurefunc("DeleteEquipmentSet", clearBinding)

	if icon and CGLDB then
		icon:Register("ClosetGnome", CGLDB, self.db.profile.minimap)
	end

	-- AddonLoader support
	if IsLoggedIn() then self:PLAYER_LOGIN() end
end

function ClosetGnome:EQUIPMENT_SETS_CHANGED()
	menuNeedsUpdate = true
	if updateKeybindingButtons then updateKeybindingButtons() end
end

function ClosetGnome:PLAYER_LOGIN()
	-- Create the button frames for keybindings
	-- Hopefully the Blizzard equipment set information is available at this point.
	for k, v in pairs(db.keybindings) do
		registerBinding(v, k)
	end

	local name = guessEquippedSet()
	if name then self:EQUIPMENT_SWAP_FINISHED(true, name, true) end
	self:EQUIPMENT_SETS_CHANGED()
end

function ClosetGnome:EQUIPMENT_SWAP_FINISHED(success, name, silent)
	if not success then return end
	if CGLDB then
		if name then
			CGLDB.text = name
			local icon = GetEquipmentSetInfoByName(name)
			-- Work around a strange Blizzard API inconsistency where
			-- GetEquipmentSetInfoByName only returns the actual icon name, NOT
			-- prepended by Interface\\Icons\\.
			if icon and icon:sub(1, 9) ~= "Interface" then icon = "Interface\\Icons\\" .. icon end
			CGLDB.icon = icon or defaultIcon
		else
			CGLDB.text = "ClosetGnome"
			CGLDB.icon = defaultIcon
		end
	end

	if not silent and name and ClosetGnome.db.profile.quips then
		DEFAULT_CHAT_FRAME:AddMessage(getQuip(name))
	end
end

--------------------------------------------------------------------------------
-- LDB Plugin
--

if CGLDB then
local popupFrame = CreateFrame("Frame", "ClosetGnomeMenu", UIParent, "UIDropDownMenuTemplate")
local menu = {}

local function menuSorter(a, b)
	return a.text > b.text
end

local function updateMenu()
	menuNeedsUpdate = nil
	menu = wipe(menu)
	for i = 1, GetNumEquipmentSets() do
		local name, icon = GetEquipmentSetInfo(i)
		local k = name
		if db.keybindings[k] then
			name = name .. " |cff999999(" .. db.keybindings[k] .. ")|r"
		end
		menu[#menu + 1] = {
			text = name,
			tooltipTitle = k,
			tooltipText = string.format("|cffeda55fClick|r to equip or |cffeda55fCtrl-Click|r to delete %s.", k),
			func = function()
				if IsControlKeyDown() then
					DeleteEquipmentSet(k)
				else
					EquipmentManager_EquipSet(k)
				end
			end,
			icon = icon or nil,
		}
	end
	table.sort(menu, menuSorter)
end

function CGLDB.OnClick(self, button)
	if button == "RightButton" then
		openOptions()
	else
		if menuNeedsUpdate then updateMenu() end
		EasyMenu(menu, popupFrame, self, 20, 4, "MENU")
	end
end

function CGLDB.OnTooltipShow(tt)
	tt:AddLine("ClosetGnome")
	tt:AddLine(L.ldbTooltip, 0.2, 1, 0.2, 1)
end

end

--------------------------------------------------------------------------------
-- Register events
--

ClosetGnome:RegisterEvent("ADDON_LOADED")
ClosetGnome:RegisterEvent("EQUIPMENT_SETS_CHANGED")
ClosetGnome:RegisterEvent("PLAYER_LOGIN")
ClosetGnome:RegisterEvent("VARIABLES_LOADED")
ClosetGnome:RegisterEvent("EQUIPMENT_SWAP_FINISHED")

