﻿CharMap = {};

local LDB = LibStub and LibStub:GetLibrary("LibDataBroker-1.1", true);

local lower_alpha = {
  "à", "á", "â", "ã", "ä", "å", "æ", "ç",
  "è", "é", "ê", "ë", "ì", "í", "î", "ï",
  "ð", "ñ", "ò", "ó", "ô", "õ", "ö", "ø",
  "ß", "ù", "ú", "û", "ü", "ý", "þ", "ÿ", 
}

local upper_alpha = {
  "À", "Á", "Â", "Ã", "Ä", "Å", "Æ", "Ç",
  "È", "É", "Ê", "Ë", "Ì", "Í", "Î", "Ï",
  "Ð", "Ñ", "Ò", "Ó", "Ô", "Õ", "Ö", "Ø",
   "", "Ù", "Ú", "Û", "Ü", "Ý", "Þ",  ""
};

local symbol_1 = {
  "¡", "¿", "£", "¢", "¥", "¦", "§", "¶",
  "©", "®", "™", "€", "·", "±", "º", "°",
  "¹", "²", "³", "ª", "µ", "‡", "†", "∞",
  "∑", "∏", "√", "∩", "∫", "≈", "≠", "≡",
   "", "¼", "½", "¾", "∂", "≤", "≥", ""
};

local symbol_2 = {
  "•", "…", "‰", "‹", "›", "«", "»", "¬",
  "←", "↑", "→", "↓", "↔", "↕", "♪", "♫",
  "♂", "♀", "◊", "♥", "♦", "♣", "♠", "☼",
  "☻", "∆", "∟", "⌂", "⌠", "⌡", "▪", "▫",
   "", "☺", "▬", "●", "◘", "◙", "◦", ""
};

local drawing_1 = {
  "┌", "┬", "┐", "─", "═", "╔", "╦", "╗",
  "├", "┼", "┤", "│", "║", "╠", "╬", "╣",
  "└", "┴", "┘",  "",  "", "╚", "╩", "╝",
  "░", "▒", "▓", "█", "►", "◄", "▲", "▼",
};

local drawing_2 = {
  "╒", "╤", "╕",  "",  "", "╓", "╥", "╖",
  "╞", "╪", "╡",  "",  "", "╟", "╫", "╢", 
  "╘", "╧", "╛",  "",  "", "╙", "╨", "╜", 
   "", "▀", "▄", "▌", "▐", "■", "□", ""
};

local full = {
  "à", "á", "â", "ã", "ä", "å", "æ", "ç",  "", "•", "…", "‰", "‹", "›", "«", "»", "¬",  "", "┌", "┬", "┐", "╔", "╦", "╗",
  "è", "é", "ê", "ë", "ì", "í", "î", "ï",  "", "©", "®", "™", "€", "·", "±", "º", "°",  "", "├", "┼", "┤", "╠", "╬", "╣",
  "ð", "ñ", "ò", "ó", "ô", "õ", "ö", "ø",  "", "¹", "²", "³", "ª", "µ", "‡", "†", "∞",  "", "└", "┴", "┘", "╚", "╩", "╝",
  "ß", "ù", "ú", "û", "ü", "ý", "þ", "ÿ",  "", "∑", "∏", "√", "∩", "∫", "≈", "≠", "≡",  "", "╒", "╤", "╕", "╓", "╥", "╖",
   "",  "",  "",  "",  "",  "",  "",  "",  "",  "", "¼", "½", "¾", "∂", "≤", "≥", "" ,  "", "╞", "╪", "╡", "╟", "╫", "╢",
  "À", "Á", "Â", "Ã", "Ä", "Å", "Æ", "Ç",  "", "•", "…", "‰", "‹", "›", "«", "»", "¬",  "", "╘", "╧", "╛", "╙", "╨", "╜", 
  "È", "É", "Ê", "Ë", "Ì", "Í", "Î", "Ï",  "", "←", "↑", "→", "↓", "↔", "↕", "♪", "♫",  "", "■", "─", "│", "║", "═", "□", 
  "Ð", "Ñ", "Ò", "Ó", "Ô", "Õ", "Ö", "Ø",  "", "♂", "♀", "◊", "♥", "♦", "♣", "♠", "☼",  "", "░", "▒", "▓", "█", "►", "◄",
   "", "Ù", "Ú", "Û", "Ü", "Ý", "Þ",  "",  "", "☻", "∆", "∟", "⌂", "⌠", "⌡", "▪", "▫",  "", "▀", "▄", "▌", "▐", "▲", "▼", 
   "",  "",  "",  "",  "",  "",  "",  "",  "",  "", "☺", "▬", "●", "◘", "◙", "◦",  "",  "",  "",  "",  "",  "",  "",  ""
}

SLASH_CHARACTERMAP1, SLASH_CHARACTERMAP2, SLASH_CHARACTERMAP3 = "/charactermap", "/charmap", "/cm";
function SlashCmdList.CHARACTERMAP(msg)
  local function print_pick_usage()
    local pick = CharacterMapSaved.pick;
    print("|cFF8080FFpick|r - Pick a single layout (disables modifiers)");
    print("  "..(((pick == 0 or not pick) and "|cFF00FF00") or "|cFFFF0000").."none|r or "..(((pick == 0 or not pick) and "|cFF00FF00") or "|cFFFF0000").."0|r - User modifier keys");
    for i,v in ipairs(CharacterMapFrame.sub_frames) do
      print("  "..((pick == i and "|cFF00FF00") or "|cFFFF0000")..i.."|r - "..v.name);
    end
  end
  if(msg == "anchor") then
    if(CharacterMapSaved.anchor) then
      CharacterMapSaved.anchor = false;
      print("|cFF8080FFCharacter Map|r can be moved freely");
    else
      CharacterMapSaved.anchor = true;
      CharMap:Anchor();
      print("|cFF8080FFCharacter Map|r will anchor to text fields");
    end
  elseif(msg == "auto") then
    if(CharacterMapSaved.auto) then
      CharacterMapSaved.auto = false;
      print("|cFF8080FFCharacter Map|r will no longer auto-show and hide");
    else
      CharacterMapSaved.auto = true;
      print("|cFF8080FFCharacter Map|r will auto-show and hide");
    end
  elseif(msg == "hide") then
    CharacterMapFrame:Hide();
  elseif(msg == "pick" or string.sub(msg, 1, 5) == "pick ") then
    local submsg = string.sub(msg, 6);
    local num = tonumber(submsg);
    if(num and (num == 0 or CharacterMapFrame.sub_frames[num])) then
      CharacterMapSaved.pick = num;
    else
      print_pick_usage();
    end
  elseif(msg == "show") then
    CharacterMapFrame:Show();
  else
    print("|cFF8080FFCharacter Map:|r Usage");
    print("|cFF00FF00show|r / |cFFFF0000hide|r - Show or Hide the |cFF8080FFCharacter Map|r");
    print("|cFF8080FFanchor|r - ("..((CharacterMapSaved.anchor and "|cFF00FF00Enabled|r") or "|cFFFF0000Disabled")..") Anchor |cFF8080FFCharacter Map|r to text fields");
    print("|cFF8080FFauto|r - ("..((CharacterMapSaved.auto and "|cFF00FF00Enabled|r") or "|cFFFF0000Disabled")..") Auto-show and hide |cFF8080FFCharacter Map|r");
    print_pick_usage();
  end
end

local function add_character_button(text, parent)
  local temp = CreateFrame("BUTTON", "CharacterMapFrameButton"..text, parent);
  temp:SetBackdrop({
    bgFile = "Interface/Tooltips/UI-Tooltip-Background",
    edgeFile = "Interface/Tooltips/UI-Tooltip-Border",
    tile = true, tileSize = 16, edgeSize = 16, 
    insets = {
      left = 4,
      right = 4,
      top = 4,
      bottom = 4
    }
  });
  temp:SetBackdropColor(0, 0, 0, 1);
  temp:SetFrameStrata("FULLSCREEN_DIALOG");
  temp:SetWidth(25);
  temp:SetHeight(25);
  temp:SetHighlightTexture("Interface\\BUTTONS\\ButtonHilight-SquareQuickslot.blp");
  temp:SetNormalFontObject("ChatFontNormal");
  temp:SetPushedTextOffset(0, 0);
  temp:SetPushedTexture("Interface\\BUTTONS\\UI-Common-MouseHilight.blp", 0.5);

  temp:SetText(text);
  temp:SetScript("OnClick", function() CharMap:AddText(parent, text); end);
  
  return temp;
end

local function create_keyboard(globalname, name, anchor, width, height)
  local f = CreateFrame("MOVIEFRAME", globalname, CharacterMapFrame);
  f:SetFrameStrata("FULLSCREEN_DIALOG");
  f:SetWidth(width);
  f:SetHeight(height);
  f:SetPoint(anchor, CharacterMapFrame, anchor);
  f:SetFrameLevel(CharacterMapFrame:GetFrameLevel());
  
  f.name = name;
  f.title = CreateFrame("FRAME", globalname.."Title", f);
  f.title:SetFrameStrata("FULLSCREEN_DIALOG");
  f.title:SetHeight(25);
  f.title:SetBackdrop({
    bgFile = "Interface/Tooltips/UI-Tooltip-Background", 
    edgeFile = "Interface/Tooltips/UI-Tooltip-Border", 
    tile = true, tileSize = 16, edgeSize = 16, 
    insets = {
      left = 4,
      right = 4,
      top = 4,
      bottom = 4
    }
  });
  f.title:SetBackdropColor(0, 0, 0, 1);
  f.title:SetPoint("BOTTOM", f, "TOP");
  f.title:EnableMouse(true);
  f.title:SetScript("OnMouseDown", function() CharacterMapFrame:StartMoving(); CharacterMapSaved.anchor = false; end);
  f.title:SetScript("OnMouseUp", function() CharacterMapFrame:StopMovingOrSizing() end);
  
  f.text = f.title:CreateFontString(nil, "OVERLAY", "GameFontNormal");
  f.text:SetText("|cFF8080FFCharMap:|cFFFFFFFF "..name);
  f.text:SetPoint("TOP", f.title, "TOP", 8, -6);
  f.text:Show();
  
  f.title:SetWidth(width);

  return f;
end

local function layout_buttons(charlist, f, layout_limit)
  if not(f.layout) then
    f.layout = {x = 10, y = -10}
  end
  for i,v in pairs(charlist) do
    if(f.layout.x > (layout_limit - 27)) then
      f.layout.y = f.layout.y - 27;
      f.layout.x = 10;
    end
    
    local temp = add_character_button(v, f);
    temp:SetPoint("TOPLEFT", f, "TOPLEFT", f.layout.x, f.layout.y);
    if(v == "") then
      temp:Hide();
    end

    f.layout.x = f.layout.x + 27;
  end
end

local scriptFrame = CreateFrame("FRAME");
scriptFrame:RegisterEvent("VARIABLES_LOADED")
scriptFrame:SetScript("OnEvent", function(self, event)
  if not(CharacterMapSaved) then
    CharacterMapSaved = {
      anchor = true,
      auto = true,
      pick = 0
    };
  end
end);
do
  local editbox;
  --This can't go in CharacterMapFrame's OnUpdate because then it wouldn't be able to auto-show itself
  --(OnUpdate only fires when the frame is already visible)
  scriptFrame:SetScript("OnUpdate", function(self, event)
    --Auto-anchor and Auto-show and hide if necessary
    local visible = CharacterMapFrame:IsVisible();
    local focus = GetCurrentKeyBoardFocus();
    if(focus ~= editbox) then
      if(CharacterMapSaved.anchor) then
        CharMap:Anchor();
      end
      
      if(CharacterMapSaved.auto) then
        if(focus and not visible) then
          CharacterMapFrame:Show();
        end
        if(visible and not focus) then
          CharacterMapFrame:Hide();
        end
      end
    end
    editbox = focus;
  end);
end

function CharacterMapFrame_OnLoad(self)
  print("|cFF8080FFCharacter Map|r Loaded");
  self:SetBackdrop({
      bgFile = "Interface/Tooltips/UI-Tooltip-Background",
      edgeFile = "Interface/Tooltips/UI-Tooltip-Border",
      tile = true, tileSize = 16, edgeSize = 16, 
      insets = {
        left = 4,
        right = 4,
        top = 4,
        bottom = 4
      }
    });
  self:SetBackdropColor(0, 0, 0, 1);
  self:SetFrameStrata("FULLSCREEN_DIALOG");
    
  local close = CreateFrame("Button", "CharacterMapFrameCloseButton", self, "UIPanelButtonTemplate2");
  close:SetText("Close");
  close:SetScript("OnClick", function() self:Hide(); end);
  close:SetWidth(105);
  close:SetPoint("BOTTOMRIGHT", self, "BOTTOMRIGHT", -10, 10);
  self.close = close;

  local edit = CreateFrame("Editbox", "CharacterMapFrameEditBox", self, "InputBoxTemplate")
  edit:SetAutoFocus(false);
  edit:SetFontObject(ChatFontNormal);
  edit:SetScript("OnEnterPressed", function() self.edit:ClearFocus() end);
  edit:SetTextInsets(0, 0, 3, 3);
  edit:SetMaxLetters(32);
  edit:SetPoint("BOTTOMLEFT", self, "BOTTOMLEFT", 17, 10);
  edit:SetPoint("RIGHT", self.close, "LEFT");
  edit:SetHeight(25);
  self.edit = edit;

  self:SetScript("OnHide", function() self.edit:SetText("") end);
  local anchor = CreateFrame("Button", "CharacterMapFrameAnchorButton", self);
  anchor:SetPoint("TOPLEFT", self, "TOPLEFT", 4, 20);
  anchor:SetWidth(16);
  anchor:SetHeight(16);
  anchor:SetHighlightTexture("Interface\\BUTTONS\\UI-Panel-MinimizeButton-Highlight.blp");
  anchor.texture = anchor:CreateTexture();
  function anchor.set_on_texture()
    anchor.texture:SetTexture("Interface\\GLUES\\CharacterSelect\\Glues-AddOn-Icons.blp");
    anchor.texture:SetTexCoord(0, 0.25, 0, 1);
    anchor.texture:SetAllPoints(anchor);
    anchor:SetNormalTexture(anchor.texture);
  end
  function anchor.set_off_texture()
    anchor.texture:SetTexture("Interface\\GLUES\\CharacterSelect\\Glues-AddOn-Icons.blp");
    anchor.texture:SetTexCoord(0.25, 0.5, 0, 1);
    anchor.texture:SetAllPoints(anchor);
    anchor:SetNormalTexture(anchor.texture);
  end
  anchor:SetScript("OnClick", function()
    if(CharacterMapSaved.anchor) then
      CharacterMapSaved.anchor = false;
    else
      CharacterMapSaved.anchor = true;
      CharMap:Anchor();
    end
    if(GameTooltip:GetOwner() == anchor) then
      anchor.draw_tooltip();
    end
  end);
  anchor.state = false;
  anchor:set_off_texture();
  anchor:SetScript("OnUpdate", function(button)
    if(anchor.state ~= CharacterMapSaved.anchor) then
      anchor.state = CharacterMapSaved.anchor;
      if(anchor.state) then
        anchor.set_on_texture();
      else
        anchor.set_off_texture();
      end
    end
  end);
  function anchor.draw_tooltip()
    GameTooltip:ClearLines();
    GameTooltip:AddDoubleLine("Anchor", ((CharacterMapSaved.anchor and "|cFF00FF00Enabled|r") or "|cFFFF0000Disabled"));
    GameTooltip:AddLine("Toggle whether |cFF8080FFCharacter Map|r anchors to text fields", 1, 1, 1, 1);
    GameTooltip:Show()
  end
  anchor:SetScript("OnEnter", function(self)
    GameTooltip:SetOwner(self, "ANCHOR_TOPLEFT");
    anchor.draw_tooltip();
  end);
	anchor:SetScript("OnLeave", function(self)
    GameTooltip:Hide()
  end);
  self.anchor = anchor;
  
  local auto = CreateFrame("Button", "CharacterMapFrameAutoButton", self);
  auto:SetPoint("LEFT", self.anchor, "RIGHT");
  auto:SetWidth(16);
  auto:SetHeight(16);
  auto:SetHighlightTexture("Interface\\BUTTONS\\UI-Panel-MinimizeButton-Highlight.blp");
  function auto.set_on_texture(button)
    button:SetNormalTexture("Interface\\Buttons\\UI-GuildButton-OfficerNote-Up.blp");
  end
  function auto.set_off_texture(button)
    button:SetNormalTexture("Interface\\Buttons\\UI-GuildButton-OfficerNote-Disabled.blp");
  end
  auto:SetScript("OnClick", function()
    if(CharacterMapSaved.auto) then
      CharacterMapSaved.auto = false;
    else
      CharacterMapSaved.auto = true;
    end
    if(GameTooltip:GetOwner() == auto) then
      auto.draw_tooltip();
    end
  end);
  auto.state = false;
  auto:set_off_texture();
  auto:SetScript("OnUpdate", function()
    if(auto.state ~= CharacterMapSaved.auto) then
      auto.state = CharacterMapSaved.auto;
      if(auto.state) then
        auto:set_on_texture();
      else
        auto:set_off_texture();
      end
    end
  end);
  function auto.draw_tooltip()
    GameTooltip:ClearLines();
    GameTooltip:AddDoubleLine("Auto-Show", ((CharacterMapSaved.auto and "|cFF00FF00Enabled|r") or "|cFFFF0000Disabled"));
    GameTooltip:AddLine("Toggle whether |cFF8080FFCharacter Map|r automatically shows and hides", 1, 1, 1, 1);
    GameTooltip:Show()
  end
  auto:SetScript("OnEnter", function(self)
    GameTooltip:SetOwner(self, "ANCHOR_TOPLEFT");
    auto.draw_tooltip();
  end);
	auto:SetScript("OnLeave", function(self)
    GameTooltip:Hide()
  end);
  self.auto = auto;
  
  local pick = CreateFrame("Button", "CharacterMapFramePickButton", self);
  pick:SetPoint("TOPRIGHT", self, "TOPRIGHT", -4, 20);
  pick:SetWidth(16);
  pick:SetHeight(16);
  pick:SetFrameStrata("TOOLTIP");
  local texture = pick:CreateTexture(nil, "OVERLAY");
  texture:SetTexture("Interface\\RAIDFRAME\\UI-RAIDFRAME-ARROW.BLP");
  texture:SetTexCoord(.25, .75, .25, .75);
  texture:SetAllPoints(pick);
  pick:SetNormalTexture(texture);
  pick:SetHighlightTexture("Interface\\BUTTONS\\UI-Panel-MinimizeButton-Highlight.blp");
  function pick.draw_tooltip()
    GameTooltip:ClearLines();
    local pick = CharacterMapSaved.pick;
    picked_name = ((pick == 0 or not pick) and "|cFF8080FFModifiers") or "|cFFFFFFFF"..CharacterMapFrame.sub_frames[pick].name;
    GameTooltip:AddDoubleLine("Pick Layout", picked_name);
    GameTooltip:AddLine("Pick a single layout to show, or revert to using modifier keys", 1, 1, 1, 1);
    GameTooltip:Show()
  end
  pick:SetScript("OnEnter", function(self)
    GameTooltip:SetOwner(self, "ANCHOR_TOPLEFT");
    pick.draw_tooltip();
  end);
	pick:SetScript("OnLeave", function(self)
    GameTooltip:Hide()
  end);
  self.pick = pick;
  
  self.sub_frame = 0;
  self.sub_frames = {};
  self.sub_frames[1] = create_keyboard("CharacterMapLowerFrame", "Lower Case", "BOTTOM", 234, 151);
  layout_buttons(lower_alpha, self.sub_frames[1], 234);
  self.sub_frames[2] = create_keyboard("CharacterMapUpperFrame", "Upper Case", "BOTTOM", 234, 151);
  layout_buttons(upper_alpha, self.sub_frames[2], 234);
  self.sub_frames[3] = create_keyboard("CharacterMapSymbol1Frame", "Symbol Set 1", "BOTTOM", 234, 178);
  layout_buttons(symbol_1, self.sub_frames[3], 234);
  self.sub_frames[4] = create_keyboard("CharacterMapSymbol2Frame", "Symbol Set 2", "BOTTOM", 234, 178);
  layout_buttons(symbol_2, self.sub_frames[4], 234);
  self.sub_frames[5] = create_keyboard("CharacterMapDrawing1Frame", "Drawing Set 1", "BOTTOM", 234, 151);
  layout_buttons(drawing_1, self.sub_frames[5], 234);
  self.sub_frames[6] = create_keyboard("CharacterMapDrawing2Frame", "Drawing Set 2", "BOTTOM", 234, 151);
  layout_buttons(drawing_2, self.sub_frames[6], 234);
  self.sub_frames[7] = create_keyboard("CharacterMapFullFrame", "All Sets", "BOTTOM", 664, 313);
  layout_buttons(full, self.sub_frames[7], 664);
  
  local PickMenuFrame = CreateFrame("FRAME", "CharacterMapPickMenuFrame", nil, "UIDropDownMenuTemplate");
  local pick_menu = {
    {
      text = "|cFF8080FFUse modifier keys",
      func = function() CharacterMapSaved.pick = 0 end,
      checked = function() return CharacterMapSaved.pick == 0 or not CharacterMapSaved.pick end
    },
    {
      text = " ",
      disabled = true
    }
  };
  for i,v in ipairs(self.sub_frames) do
    local option = {
      text = v.name,
      func = function() CharacterMapSaved.pick = i end,
      checked = function() return CharacterMapSaved.pick == i end
    }
    tinsert(pick_menu, option);
  end
  
  self.pick:SetScript("OnClick", function()
    EasyMenu(pick_menu, PickMenuFrame, self.pick, 20, 8, "MENU");
  end);
end

function CharacterMapFrame_OnUpdate(self)
  local modifier = CharacterMapSaved.pick;
  if(modifier == 0 or not modifier) then
    if(IsControlKeyDown()) then
      if(IsShiftKeyDown()) then
        if(IsAltKeyDown()) then
          modifier = 7
        else
          modifier = 4;
        end
      else
        modifier = 3;
      end
    elseif(IsAltKeyDown()) then
      if(IsShiftKeyDown()) then
        modifier = 6;
      else
        modifier = 5;
      end
    elseif(IsShiftKeyDown()) then
      modifier = 2;
    else
      modifier = 1;
    end
  end
  if(self.sub_frame ~= modifier) then
    self.sub_frame = modifier;
    for i,v in ipairs(self.sub_frames) do
      if(i == self.sub_frame) then
        v:Show();
      else
        v:Hide();
      end
    end
    CharacterMapFrame:SetWidth(CharacterMapFrame.sub_frames[modifier]:GetWidth());
    CharacterMapFrame:SetHeight(CharacterMapFrame.sub_frames[modifier]:GetHeight());
  end
end

function CharacterMapFrame_OnShow(self)
  if(CharacterMapSaved.anchor) then
    CharMap:Anchor();
  end
end

if(LDB) then
  local CharMapLDB = LDB:NewDataObject("Character Map", {
    type = "launcher",
    text = "CharMap",
    icon = "Interface\\GossipFrame\\DailyActiveQuestIcon",

    OnTooltipShow = function(self)
      local alt = "|cFFFF4040Alt|r";
      local control = "|cFF40FF40Control|r";
      local shift = "|cFF8080FFShift|r";
      local right = "|cFFFFFFFFRight|r";
      local click = "|cFFFFFFFFClick|r";
      self:AddLine("|cFF8080FFCharacter Map");
      self:AddDoubleLine(click, "Toggle |cFF8080FFCharacter Map|r");
      self:AddLine(" ");
      self:AddDoubleLine(right.."-"..click, "Open to |cFFFFFFFFLower Case");
      self:AddDoubleLine(shift.."-"..click, "Open to |cFFFFFFFFUpper Case");
      self:AddDoubleLine(control.."-"..click, "Open to |cFFFFFFFFSymbol Set 1");
      self:AddDoubleLine(control.."-"..shift.."-"..click, "Open to |cFFFFFFFFSymbol Set 2");
      self:AddDoubleLine(alt.."-"..click, "Open to |cFFFFFFFFDrawing Set 1");
      self:AddDoubleLine(alt.."-"..shift.."-"..click, "Open to |cFFFFFFFFDrawing Set 2");
      self:AddDoubleLine(alt.."-"..control.."-"..shift.."-"..click, "Open to |cFFFFFFFFFull (all characters)");
    end,

    OnClick = function(self, button)
      local modifier = 0;
      if(IsControlKeyDown()) then
        if(IsShiftKeyDown()) then
          if(IsAltKeyDown()) then
            modifier = 7
          else
            modifier = 4;
          end
        else
          modifier = 3;
        end
      elseif(IsAltKeyDown()) then
        if(IsShiftKeyDown()) then
          modifier = 6;
        else
          modifier = 5;
        end
      elseif(IsShiftKeyDown()) then
        modifier = 2;
      else
        if(button == "RightButton") then
          modifier = 1;
        else
          modifier = 0;
        end
      end
      if(modifier == CharacterMapSaved.pick or modifier == 0) then
        CharMap:Toggle();
      else
        CharacterMapSaved.pick = modifier;
        CharacterMapFrame:Show();
        if not(GetCurrentKeyBoardFocus()) then
          CharacterMapFrameEditBox:SetFocus();
        end
      end
    end
  });
end

function CharMap:Toggle()
  if(CharacterMapFrame:IsVisible()) then
    CharacterMapFrame:Hide();
  else
    CharacterMapFrame:Show();
    if not(GetCurrentKeyBoardFocus()) then
      CharacterMapFrameEditBox:SetFocus();
    end
  end
end

function CharMap:Anchor()
  local editbox = GetCurrentKeyBoardFocus();
  if(editbox and editbox ~= CharacterMapFrameEditBox) then
    local width = CharacterMapFrame:GetWidth();
    local height = CharacterMapFrame:GetHeight();
    local anchor_x = editbox:GetLeft();
    local anchor_y = editbox:GetTop();
    
    local y_offset = (anchor_y < height and 0) or -25;
    local in_scroll_frame = editbox:GetParent() and editbox:GetParent():GetObjectType() == "ScrollFrame";
    local x_offset = (anchor_x < width and ((in_scroll_frame and 25) or 5)) or -5;
    local anchor_self = ((anchor_y < height and "BOTTOM") or "TOP")..((anchor_x < width and "LEFT") or "RIGHT");
    local anchor_other = ((anchor_y < height and "BOTTOM") or "TOP")..((anchor_x < width and "RIGHT") or "LEFT");
    
    CharacterMapFrame:ClearAllPoints();
    CharacterMapFrame:SetPoint(anchor_self, editbox, anchor_other, x_offset, y_offset);
  end
end

function CharMap:AddText(f, text)
  local editbox = GetCurrentKeyBoardFocus();
  if(editbox) then
    editbox:Insert(text);
  else
    CharacterMapFrame.edit:Insert(text);
  end
end
