--[[

	BagSaver addon. 
	Primary functions are auto-selling unwanted items
	to a vendor, finding the least valuable item in 
	your bags to discard and more.
	Author: OneWingedAngel

--]]


--[[ Globals --]]
BagSaverConfig = {}

--[[ Main addon frame creation and main local vars ]]--
local BagSaver = CreateFrame("Frame", "BagSaverFrame", UIParent)

local ScanningTooltip = CreateFrame( "GameTooltip", "ScanningTooltipFrame" )
ScanningTooltip:SetOwner( WorldFrame, "ANCHOR_NONE" )
ScanningTooltip:AddFontStrings(
	ScanningTooltip:CreateFontString( "$parentTextLeft1", nil, "GameTooltipText" ),
	ScanningTooltip:CreateFontString( "$parentTextRight1", nil, "GameTooltipText" ) )

local DisposableItemFrames = {}
local UnusableEquipment = {}
local NonPrimaryArmor = {}
local DefaultConfig = {}
local RegisteredEvents = {}
BagSaverFrame:SetScript("OnEvent", function (self, event, ...) if (RegisteredEvents[event]) then return RegisteredEvents[event](self, event, ...) end end)

--[[ Event Handlers --]]

function RegisteredEvents:ADDON_LOADED(event, addon, ...)
	if (addon == "BagSaver") then
		SLASH_BAGSAVER1, SLASH_BAGSAVER2 = '/bagsaver', '/bs'
		SlashCmdList["BAGSAVER"] = function (msg, editbox)
			BagSaver.SlashCmdHandler(msg, editbox)	
		end

		--Use metatable to implement defaults, see http://www.lua.org/pil/13.4.1.html
		setmetatable(BagSaverConfig, {__index = DefaultConfig})

		BagSaver:CreateConfigMenu()
		--for quality=0,1 do
		--	DisposableItemFrames[quality] = BagSaver.CreateDisposableItemFrame(quality)
		--end
		print("BagSaver " .. GetAddOnMetadata("BagSaver","Version") .. " Loaded")
	end
end

function RegisteredEvents:MERCHANT_SHOW()
	if BagSaver.GetConfigValue("autoSell") then
		local itemsToSell, numItemsToSell = BagSaver.BagSearch(BagSaver.IsJunkItem)

		if numItemsToSell ~= 0 then
			if BagSaver.GetConfigValue("autoSellPrompt") then
				BagSaver.PromptToSellItemsToVendor(itemsToSell)
			else
				BagSaver.SellItemsToVendor(itemsToSell)
			end
		end
	end
end

function RegisteredEvents:UI_ERROR_MESSAGE(event,message)
	--print("UI ERROR MESSAGE caught, message is " .. message)
	if message == INVENTORY_FULL then
		if BagSaver.GetConfigValue("findDisposableItems") then
			if BagSaver.IsDisposableItemFrameShown() == false then
				local disposableItems, numDisposableItems = BagSaver.BagSearch(BagSaver.IsLessValuable)
				BagSaver.ShowDisposableItems(disposableItems, numDisposableItems)
			else
				--print("DisposableItems frame already shown, skipping")
			end
		end
	end
end

function RegisteredEvents:BAG_UPDATE(event,bagId)
	if BagSaver.IsDisposableItemFrameShown() and BagSaver.GetNumFreeBagSlots() > 0 then
		for i,frame in pairs(DisposableItemFrames) do
			frame:Hide()
		end
	end
end

for k, v in pairs(RegisteredEvents) do
	BagSaver:RegisterEvent(k)
end

--[[ Addon functions --]]
function BagSaver.SellItemsToVendor(items)
	local totalProfit = 0
	for itemQuality=0,7 do
		for i,item in pairs(items[itemQuality]) do
			--Check to make sure the items haven't moved since we looked
			local newTexture, newCount, newLocked, newQuality, newReadable, newLootable, newItemLink = GetContainerItemInfo(item["bag"], item["slot"]) 
			if (item["link"] == newItemLink) then
				if item["count"] == 1 then
					print("BagSaver auto-selling: " .. newItemLink .. " for: " .. BagSaver.GetMoneyString(item.value))
					totalProfit = totalProfit + item.value
				else
					print("BagSaver auto-selling: " .. newCount .. "x " .. newItemLink .. " for: " .. BagSaver.GetMoneyString(item.value * item.count))
					totalProfit = totalProfit + (item.value * item.count)
				end
				UseContainerItem(item["bag"],item["slot"])
			else
				print(item["link"] .. " has moved! Not attempting to sell " .. newItemLink)
			end
		end
	end
	print("Total profit from BagSaver auto-sell: " .. BagSaver.GetMoneyString(totalProfit))
end

function BagSaver.PromptToSellItemsToVendor(items)
	local itemLinks = {}
	local totalSellValue = 0;
	for itemQuality=0,7 do
		for i,item in pairs(items[itemQuality]) do
			if item["count"] == 1 then
				table.insert(itemLinks,item["link"] .. " for: " .. BagSaver.GetMoneyString(item.value))
				totalSellValue = totalSellValue + item.value
			else
				table.insert(itemLinks,item["count"] .. "x " .. item["link"] .. " for: " .. BagSaver.GetMoneyString(item.value * item.count))
				totalSellValue = totalSellValue + (item.value * item.count)
			end
		end
	end

	local dialogText = "Sell the following items?\n"
	StaticPopupDialogs["BAGSAVER_AUTOSELL_PROMPT"] = {
		text = dialogText .. table.concat(itemLinks, "\n") .. "\n Total: " .. BagSaver.GetMoneyString(totalSellValue),
		button1 = "Yes",
		button2 = "No",
		OnAccept = function() BagSaver.SellItemsToVendor(items) end,
		timeout = 0,
		whileDead = 1,
		hideOnEscape = 1,
		showAlert = 1
	}
	StaticPopup_Show("BAGSAVER_AUTOSELL_PROMPT")
end

function BagSaver.ShowDisposableItems(items, numItems)
	if DisposableItemFrames[numItems] == nil then
		DisposableItemFrames[numItems] = BagSaver.CreateDisposableItemFrame(numItems)
	end
	local itemFrame = DisposableItemFrames[numItems]
	local itemsToDisplay = {}

	for quality=0,7 do
		if #items[quality] > 0 then
			for i,item in ipairs(items[quality]) do
				tinsert(itemsToDisplay,item)
			end
		end
	end
	for i,item in ipairs(itemsToDisplay) do
		BagSaver.SetItemButtonInfo(itemFrame.itemButtons[i],item)
	end
	itemFrame:Show()
end

function BagSaver.BagSearch(addDesiredItem)
	local itemsFound = {}
	for itemQuality=0,7 do
		itemsFound[itemQuality] = {}
	end
	local numItemsFound = 0
	for bag = 0,NUM_BAG_SLOTS do
		for slot = 1,GetContainerNumSlots(bag) do
			local id = GetContainerItemID(bag,slot)
			if id then --We're not looking at an empty slot
				currentItem = BagSaver.BuildItemTable(bag,slot,id)
				if addDesiredItem(currentItem, itemsFound) then
					numItemsFound = numItemsFound + 1
				end
			end
		end
	end
	return itemsFound, numItemsFound
end

function BagSaver.IsJunkItem(item, itemTable)
	local playerClassLocal, playerClass = UnitClass("player")
	local playerLevel = UnitLevel("player")
	if item.value > 0 then --We don't care about items we can't sell
		if (BagSaver.GetConfigValue("autoSellGrayItems")) then
			if item.quality == 0 then
				tinsert(itemTable[item.quality],item)				
				return true
			end
		end
		if BagSaver.GetConfigValue("autoSellUnusableBoundItems") then
			if (BagSaver.ItemIsSoulbound(item.bag,item.slot) and 
			    UnusableEquipment[playerClass][item.class] ~= nil and 
			    UnusableEquipment[playerClass][item.class][item.subClass] ~= nil) then --item is soulbound and could never be equipped by the player's class
				tinsert(itemTable[item.quality],item)				
				return true
			end
		end
		if BagSaver.GetConfigValue("autoSellNonPrimaryBoundArmor") then
			if BagSaver.ItemIsSoulbound(item.bag,item.slot) then
				if item.equipSlot ~= "INVTYPE_CLOAK" then --cloaks are always considered cloth
					if NonPrimaryArmor[playerClass][item.subClass] ~= nil then --item is of an armor type below the primary type of the player's class
						tinsert(itemTable[item.quality],item)				
						return true
					elseif playerLevel >= 40 and NonPrimaryArmor[playerClass]["Post40"][item.subClass] ~= nil then --item is of an armor type below the primary type of the player's class after level 40
						tinsert(itemTable[item.quality],item)				
						return true
					end
				end
			end
		end

	end
	return false
end

function BagSaver.IsLessValuable(item, itemTable)
	if item.value > 0 then --Items with no value are usually quest items, conjured, etc.
		if item.quality == 0 or item.quality == 1 then --Only check on gray and white items
			if #itemTable[item.quality] == 0 then --if this is our first item with value
				tinsert(itemTable[item.quality],item)				
				return true
			else 
				if item.value * item.count < itemTable[item.quality][1]["value"] * itemTable[item.quality][1]["count"] then --if current item value is less than stored item value
					itemTable[item.quality][1] = item
				end
			end
		end
	end
	return false
end

function BagSaver.BuildItemTable(bag,slot,id)
	local texture, count, locked, _, readable, lootable, link = GetContainerItemInfo(bag,slot)
	--local link = GetContainerItemLink(bag,slot)
	local name, _, quality, iLevel, reqLevel, class, subClass, maxStack, equipSlot, _, value = GetItemInfo(id)
	local itemTable = { bag = bag, 
			      slot = slot, 
			      value = value, 
			      count = count, 
			      texture = texture, 
			      name = name, 
			      link = link, 
			      quality = quality, 
			      color = { GetItemQualityColor(quality) }, 
			      class = class, 
			      subClass = subClass, 
			      equipSlot = equipSlot }

	return itemTable
end

function BagSaver.ItemIsSoulbound(bag,slot)
	ScanningTooltip:ClearLines()
	ScanningTooltip:SetBagItem(bag,slot)

	--print(getglobal("ScanningTooltipFrameTextLeft2"):GetText())

	if (getglobal("ScanningTooltipFrameTextLeft2"):GetText() == ITEM_SOULBOUND) then
		return true
	end
	return false
end

function BagSaver.GetNumFreeBagSlots()
	local totalFreeSlots = 0
	for bag = 0,NUM_BAG_SLOTS do
		local freeSlots, bagType = GetContainerNumFreeSlots(bag)
		if bagType == 0 then
			totalFreeSlots = totalFreeSlots + freeSlots
		end
	end
	return totalFreeSlots
end

function BagSaver.IsDisposableItemFrameShown()
	for i,frame in pairs(DisposableItemFrames) do
		if frame:IsShown() then
			return true
		end
	end

	return false
end

function BagSaver.SlashCmdHandler(msg, editbox)
	--print("command is " .. msg .. "\n")
	if (string.lower(msg) == "config") then
		InterfaceOptionsFrame_OpenToCategory("BagSaver")
	elseif (string.lower(msg) == "dumpconfig") then
		print("With defaults")
		for k,v in pairs(DefaultConfig) do
			print(k,BagSaver.GetConfigValue(k))
		end
		print("Direct table")
		for k,v in pairs(BagSaverConfig) do
			print(k,v)
		end
	elseif (string.lower(msg) == "dumpequip") then
		BagSaver.DumpEquipmentTable()
	elseif (string.lower(msg) == "dumparmor") then
		BagSaver.DumpNonPrimaryArmorTable()
	elseif (string.lower(msg) == "reset") then
		BagSaver.SetConfigToDefaults()
	elseif (string.lower(msg) == "perf") then
		BagSaver.PrintPerformanceData()
	else
		BagSaver.ShowHelp()
	end
end

function BagSaver.ShowHelp()
	print("Slash commands (/bagsaver or /bs):")
	print(" /bs config:	Open addon config menu (also found in Addon tab in Blizzard's Interface menu)")
	print(" /bs reset:	Resets your config to defaults")
end

function BagSaver.DumpEquipmentTable()
	for className, localizedClassName in pairs(LOCALIZED_CLASS_NAMES_MALE) do
		print(className)
		for itemType,v in pairs(UnusableEquipment[className]) do
			print(" " .. itemType)
			for itemSubType,v in pairs(UnusableEquipment[className][itemType]) do
				print("  " .. itemSubType,v)
			end
		end
	end
end

function BagSaver.DumpNonPrimaryArmorTable()
	for className, localizedClassName in pairs(LOCALIZED_CLASS_NAMES_MALE) do
		print(className)
		print(" Always:")
		for armorType,v in pairs(NonPrimaryArmor[className]) do
			if (armorType ~= "Post40") then
				print("  " .. armorType)
			end
		end
		print(" Post level 40:")
		for armorType,v in pairs(NonPrimaryArmor[className]["Post40"]) do
			if (armorType ~= "Post40") then
				print("  " .. armorType)
			end
		end

	end
end

function BagSaver.PrintPerformanceData()
	UpdateAddOnMemoryUsage()
	local mem = GetAddOnMemoryUsage("BagSaver")
	print("BagSaver is currently using " .. mem .. " kbytes of memory")
	--collectgarbage(collect)
	--UpdateAddOnMemoryUsage()
	--local mem = GetAddOnMemoryUsage("BagSaver")
	--print("BagSaver is currently using " .. mem .. " kbytes of memory after garbage collection")

end

function BagSaver.DumpItem(item)
	for k,v in pairs(item) do
		print(k,v)
	end
end

--[[ Configuration methods --]]
function BagSaver.GetConfigValue(key)
	return BagSaverConfig[key]
end

function BagSaver.SetConfigValue(key, value)
	if (DefaultConfig[key] == value) then
		BagSaverConfig[key] = nil
	else 
		BagSaverConfig[key] = value
	end
end

function BagSaver.SetConfigToDefaults()
	print("Resetting config to defaults")
	BagSaverConfig = {}
	setmetatable(BagSaverConfig, {__index = DefaultConfig})
end

function BagSaver.GetMoneyString(money)
	local gold = floor(money / 10000)
	local silver = floor((money - gold * 10000) / 100)
	local copper = mod(money, 100)
	if gold > 0 then
		return format(GOLD_AMOUNT_TEXTURE.." "..SILVER_AMOUNT_TEXTURE.." "..COPPER_AMOUNT_TEXTURE, gold, 0, 0, silver, 0, 0, copper, 0, 0)
	elseif silver > 0 then
		return format(SILVER_AMOUNT_TEXTURE.." "..COPPER_AMOUNT_TEXTURE, silver, 0, 0, copper, 0, 0)
	else
		return format(COPPER_AMOUNT_TEXTURE, copper, 0, 0)
	end
end

--[[ "Global" Vars --]]

DefaultConfig = {
	autoSell = true,
	autoSellPrompt = true,
	autoSellGrayItems = true,
	autoSellUnusableBoundItems = false,
	autoSellNonPrimaryBoundArmor = false,
	findDisposableItems = true,
}

--Define local vars with localized name of each weapon and armor type

local WEAPON,ARMOR,CONTAINER,CONSUMABLE,GLYPH,TRADEGOODS,PROJECTILE,QUIVER,RECIPE,GEM,MISCELLANEOUS,QUEST = GetAuctionItemClasses()
local ONEHANDAXES,TWOHANDAXES,BOWS,GUNS,ONEHANDMACES,TWOHANDMACES,POLEARMS,ONEHANDSWORDS,TWOHANDSWORDS,STAVES,FISTWEAPONS,WEAPONMISC,DAGGERS,THROWN,CROSSBOWS,WANDS,FISHINGPOLES = GetAuctionItemSubClasses(1)
local ARMORMISC,CLOTH,LEATHER,MAIL,PLATE,SHIELDS,LIBRAMS,IDOLS,TOTEMS,SIGILS = GetAuctionItemSubClasses(2)
--print(WEAPON,ARMOR,CONTAINER,CONSUMABLE,GLYPH,TRADEGOODS,PROJECTILE,QUIVER,RECIPE,GEM,MISCELLANEOUS,QUEST)
--print(ONEHANDAXES,TWOHANDAXES,BOWS,GUNS,ONEHANDMACES,TWOHANDMACES,POLEARMS,ONEHANDSWORDS,TWOHANDSWORDS,STAVES,FISTWEAPONS,WEAPONMISC,DAGGERS,THROWN,CROSSBOWS,WANDS,FISHINGPOLES)
--print(ARMORMISC,CLOTH,LEATHER,MAIL,PLATE,SHIELDS,LIBRAMS,IDOLS,TOTEMS,SIGILS)

--Initialize our table structure for each class
--UnusableEquipment["CLASS"] = {}
--UnusableEquipment["CLASS"][WEAPON] = {}
--UnusableEquipment["CLASS"][ARMOR] = {}
for className, localizedClassName in pairs(LOCALIZED_CLASS_NAMES_MALE) do
	UnusableEquipment[className] = {}
	UnusableEquipment[className][WEAPON] = {}
	UnusableEquipment[className][ARMOR] = {}
end

--Reference for all equipment and armor types
--[[
UnusableEquipment["CLASS"][WEAPON][BOWS] = 1
UnusableEquipment["CLASS"][WEAPON][CROSSBOWS] = 1
UnusableEquipment["CLASS"][WEAPON][DAGGERS] = 1
UnusableEquipment["CLASS"][WEAPON][FISTWEAPONS] = 1
UnusableEquipment["CLASS"][WEAPON][GUNS] = 1
UnusableEquipment["CLASS"][WEAPON][ONEHANDAXES] = 1
UnusableEquipment["CLASS"][WEAPON][ONEHANDMACES] = 1
UnusableEquipment["CLASS"][WEAPON][ONEHANDSWORDS] = 1
UnusableEquipment["CLASS"][WEAPON][POLEARMS] = 1
UnusableEquipment["CLASS"][WEAPON][STAVES] = 1
UnusableEquipment["CLASS"][WEAPON][THROWN] = 1
UnusableEquipment["CLASS"][WEAPON][TWOHANDAXES] = 1
UnusableEquipment["CLASS"][WEAPON][TWOHANDMACES] = 1
UnusableEquipment["CLASS"][WEAPON][TWOHANDSWORDS] = 1
UnusableEquipment["CLASS"][WEAPON][WANDS] = 1
UnusableEquipment["CLASS"][ARMOR][CLOTH] = 1
UnusableEquipment["CLASS"][ARMOR][LEATHER] = 1
UnusableEquipment["CLASS"][ARMOR][MAIL] = 1
UnusableEquipment["CLASS"][ARMOR][PLATE] = 1
UnusableEquipment["CLASS"][ARMOR][SHIELDS] = 1
UnusableEquipment["CLASS"][ARMOR][LIBRAMS] = 1
UnusableEquipment["CLASS"][ARMOR][IDOLS] = 1
UnusableEquipment["CLASS"][ARMOR][TOTEMS] = 1
UnusableEquipment["CLASS"][ARMOR][SIGILS] = 1
--]]

UnusableEquipment["DEATHKNIGHT"][WEAPON][BOWS] = 1
UnusableEquipment["DEATHKNIGHT"][WEAPON][CROSSBOWS] = 1
UnusableEquipment["DEATHKNIGHT"][WEAPON][DAGGERS] = 1
UnusableEquipment["DEATHKNIGHT"][WEAPON][FISTWEAPONS] = 1
UnusableEquipment["DEATHKNIGHT"][WEAPON][GUNS] = 1
UnusableEquipment["DEATHKNIGHT"][WEAPON][STAVES] = 1
UnusableEquipment["DEATHKNIGHT"][WEAPON][THROWN] = 1
UnusableEquipment["DEATHKNIGHT"][WEAPON][WANDS] = 1
UnusableEquipment["DEATHKNIGHT"][ARMOR][SHIELDS] = 1
UnusableEquipment["DEATHKNIGHT"][ARMOR][LIBRAMS] = 1
UnusableEquipment["DEATHKNIGHT"][ARMOR][IDOLS] = 1
UnusableEquipment["DEATHKNIGHT"][ARMOR][TOTEMS] = 1

UnusableEquipment["DRUID"][WEAPON][BOWS] = 1
UnusableEquipment["DRUID"][WEAPON][CROSSBOWS] = 1
UnusableEquipment["DRUID"][WEAPON][GUNS] = 1
UnusableEquipment["DRUID"][WEAPON][ONEHANDAXES] = 1
UnusableEquipment["DRUID"][WEAPON][ONEHANDSWORDS] = 1
UnusableEquipment["DRUID"][WEAPON][THROWN] = 1
UnusableEquipment["DRUID"][WEAPON][TWOHANDAXES] = 1
UnusableEquipment["DRUID"][WEAPON][TWOHANDSWORDS] = 1
UnusableEquipment["DRUID"][WEAPON][WANDS] = 1
UnusableEquipment["DRUID"][ARMOR][MAIL] = 1
UnusableEquipment["DRUID"][ARMOR][PLATE] = 1
UnusableEquipment["DRUID"][ARMOR][SHIELDS] = 1
UnusableEquipment["DRUID"][ARMOR][LIBRAMS] = 1
UnusableEquipment["DRUID"][ARMOR][TOTEMS] = 1
UnusableEquipment["DRUID"][ARMOR][SIGILS] = 1

UnusableEquipment["HUNTER"][WEAPON][ONEHANDMACES] = 1
UnusableEquipment["HUNTER"][WEAPON][TWOHANDMACES] = 1
UnusableEquipment["HUNTER"][WEAPON][WANDS] = 1
UnusableEquipment["HUNTER"][ARMOR][PLATE] = 1
UnusableEquipment["HUNTER"][ARMOR][SHIELDS] = 1
UnusableEquipment["HUNTER"][ARMOR][LIBRAMS] = 1
UnusableEquipment["HUNTER"][ARMOR][IDOLS] = 1
UnusableEquipment["HUNTER"][ARMOR][TOTEMS] = 1
UnusableEquipment["HUNTER"][ARMOR][SIGILS] = 1

UnusableEquipment["MAGE"][WEAPON][BOWS] = 1
UnusableEquipment["MAGE"][WEAPON][CROSSBOWS] = 1
UnusableEquipment["MAGE"][WEAPON][FISTWEAPONS] = 1
UnusableEquipment["MAGE"][WEAPON][GUNS] = 1
UnusableEquipment["MAGE"][WEAPON][ONEHANDAXES] = 1
UnusableEquipment["MAGE"][WEAPON][ONEHANDMACES] = 1
UnusableEquipment["MAGE"][WEAPON][POLEARMS] = 1
UnusableEquipment["MAGE"][WEAPON][THROWN] = 1
UnusableEquipment["MAGE"][WEAPON][TWOHANDAXES] = 1
UnusableEquipment["MAGE"][WEAPON][TWOHANDMACES] = 1
UnusableEquipment["MAGE"][WEAPON][TWOHANDSWORDS] = 1
UnusableEquipment["MAGE"][ARMOR][LEATHER] = 1
UnusableEquipment["MAGE"][ARMOR][MAIL] = 1
UnusableEquipment["MAGE"][ARMOR][PLATE] = 1
UnusableEquipment["MAGE"][ARMOR][SHIELDS] = 1
UnusableEquipment["MAGE"][ARMOR][LIBRAMS] = 1
UnusableEquipment["MAGE"][ARMOR][IDOLS] = 1
UnusableEquipment["MAGE"][ARMOR][TOTEMS] = 1
UnusableEquipment["MAGE"][ARMOR][SIGILS] = 1

UnusableEquipment["PALADIN"][WEAPON][BOWS] = 1
UnusableEquipment["PALADIN"][WEAPON][CROSSBOWS] = 1
UnusableEquipment["PALADIN"][WEAPON][DAGGERS] = 1
UnusableEquipment["PALADIN"][WEAPON][FISTWEAPONS] = 1
UnusableEquipment["PALADIN"][WEAPON][GUNS] = 1
UnusableEquipment["PALADIN"][WEAPON][STAVES] = 1
UnusableEquipment["PALADIN"][WEAPON][THROWN] = 1
UnusableEquipment["PALADIN"][WEAPON][WANDS] = 1
UnusableEquipment["PALADIN"][ARMOR][IDOLS] = 1
UnusableEquipment["PALADIN"][ARMOR][TOTEMS] = 1
UnusableEquipment["PALADIN"][ARMOR][SIGILS] = 1

UnusableEquipment["PRIEST"][WEAPON][BOWS] = 1
UnusableEquipment["PRIEST"][WEAPON][CROSSBOWS] = 1
UnusableEquipment["PRIEST"][WEAPON][FISTWEAPONS] = 1
UnusableEquipment["PRIEST"][WEAPON][GUNS] = 1
UnusableEquipment["PRIEST"][WEAPON][ONEHANDAXES] = 1
UnusableEquipment["PRIEST"][WEAPON][ONEHANDSWORDS] = 1
UnusableEquipment["PRIEST"][WEAPON][POLEARMS] = 1
UnusableEquipment["PRIEST"][WEAPON][THROWN] = 1
UnusableEquipment["PRIEST"][WEAPON][TWOHANDAXES] = 1
UnusableEquipment["PRIEST"][WEAPON][TWOHANDMACES] = 1
UnusableEquipment["PRIEST"][WEAPON][TWOHANDSWORDS] = 1
UnusableEquipment["PRIEST"][ARMOR][LEATHER] = 1
UnusableEquipment["PRIEST"][ARMOR][MAIL] = 1
UnusableEquipment["PRIEST"][ARMOR][PLATE] = 1
UnusableEquipment["PRIEST"][ARMOR][SHIELDS] = 1
UnusableEquipment["PRIEST"][ARMOR][LIBRAMS] = 1
UnusableEquipment["PRIEST"][ARMOR][IDOLS] = 1
UnusableEquipment["PRIEST"][ARMOR][TOTEMS] = 1
UnusableEquipment["PRIEST"][ARMOR][SIGILS] = 1

UnusableEquipment["ROGUE"][WEAPON][POLEARMS] = 1
UnusableEquipment["ROGUE"][WEAPON][STAVES] = 1
UnusableEquipment["ROGUE"][WEAPON][TWOHANDAXES] = 1
UnusableEquipment["ROGUE"][WEAPON][TWOHANDMACES] = 1
UnusableEquipment["ROGUE"][WEAPON][TWOHANDSWORDS] = 1
UnusableEquipment["ROGUE"][WEAPON][WANDS] = 1
UnusableEquipment["ROGUE"][ARMOR][MAIL] = 1
UnusableEquipment["ROGUE"][ARMOR][PLATE] = 1
UnusableEquipment["ROGUE"][ARMOR][SHIELDS] = 1
UnusableEquipment["ROGUE"][ARMOR][LIBRAMS] = 1
UnusableEquipment["ROGUE"][ARMOR][IDOLS] = 1
UnusableEquipment["ROGUE"][ARMOR][TOTEMS] = 1
UnusableEquipment["ROGUE"][ARMOR][SIGILS] = 1

UnusableEquipment["SHAMAN"][WEAPON][BOWS] = 1
UnusableEquipment["SHAMAN"][WEAPON][CROSSBOWS] = 1
UnusableEquipment["SHAMAN"][WEAPON][GUNS] = 1
UnusableEquipment["SHAMAN"][WEAPON][ONEHANDSWORDS] = 1
UnusableEquipment["SHAMAN"][WEAPON][POLEARMS] = 1
UnusableEquipment["SHAMAN"][WEAPON][THROWN] = 1
UnusableEquipment["SHAMAN"][WEAPON][TWOHANDSWORDS] = 1
UnusableEquipment["SHAMAN"][WEAPON][WANDS] = 1
UnusableEquipment["SHAMAN"][ARMOR][PLATE] = 1
UnusableEquipment["SHAMAN"][ARMOR][LIBRAMS] = 1
UnusableEquipment["SHAMAN"][ARMOR][IDOLS] = 1
UnusableEquipment["SHAMAN"][ARMOR][SIGILS] = 1

UnusableEquipment["WARLOCK"][WEAPON][BOWS] = 1
UnusableEquipment["WARLOCK"][WEAPON][CROSSBOWS] = 1
UnusableEquipment["WARLOCK"][WEAPON][FISTWEAPONS] = 1
UnusableEquipment["WARLOCK"][WEAPON][GUNS] = 1
UnusableEquipment["WARLOCK"][WEAPON][ONEHANDAXES] = 1
UnusableEquipment["WARLOCK"][WEAPON][ONEHANDMACES] = 1
UnusableEquipment["WARLOCK"][WEAPON][POLEARMS] = 1
UnusableEquipment["WARLOCK"][WEAPON][THROWN] = 1
UnusableEquipment["WARLOCK"][WEAPON][TWOHANDAXES] = 1
UnusableEquipment["WARLOCK"][WEAPON][TWOHANDMACES] = 1
UnusableEquipment["WARLOCK"][WEAPON][TWOHANDSWORDS] = 1
UnusableEquipment["WARLOCK"][ARMOR][LEATHER] = 1
UnusableEquipment["WARLOCK"][ARMOR][MAIL] = 1
UnusableEquipment["WARLOCK"][ARMOR][PLATE] = 1
UnusableEquipment["WARLOCK"][ARMOR][SHIELDS] = 1
UnusableEquipment["WARLOCK"][ARMOR][LIBRAMS] = 1
UnusableEquipment["WARLOCK"][ARMOR][IDOLS] = 1
UnusableEquipment["WARLOCK"][ARMOR][TOTEMS] = 1
UnusableEquipment["WARLOCK"][ARMOR][SIGILS] = 1

UnusableEquipment["WARRIOR"][WEAPON][WANDS] = 1
UnusableEquipment["WARRIOR"][ARMOR][LIBRAMS] = 1
UnusableEquipment["WARRIOR"][ARMOR][IDOLS] = 1
UnusableEquipment["WARRIOR"][ARMOR][TOTEMS] = 1
UnusableEquipment["WARRIOR"][ARMOR][SIGILS] = 1

--NonPrimaryArmor["CLASS"] = {}
--NonPrimaryArmor["CLASS"]["Pre40"] = {}
--NonPrimaryArmor["CLASS"]["Post40"] = {}
for className, localizedClassName in pairs(LOCALIZED_CLASS_NAMES_MALE) do
	NonPrimaryArmor[className] = {}
	NonPrimaryArmor[className]["Post40"] = {}
end
NonPrimaryArmor["DEATHKNIGHT"][CLOTH] = 1
NonPrimaryArmor["DEATHKNIGHT"][LEATHER] = 1
NonPrimaryArmor["DEATHKNIGHT"][MAIL] = 1

NonPrimaryArmor["DRUID"][CLOTH] = 1

NonPrimaryArmor["HUNTER"][CLOTH] = 1
NonPrimaryArmor["HUNTER"]["Post40"][LEATHER] = 1

NonPrimaryArmor["PALADIN"][CLOTH] = 1
NonPrimaryArmor["PALADIN"][LEATHER] = 1
NonPrimaryArmor["PALADIN"]["Post40"][MAIL] = 1

NonPrimaryArmor["ROGUE"][CLOTH] = 1

NonPrimaryArmor["SHAMAN"][CLOTH] = 1
NonPrimaryArmor["SHAMAN"]["Post40"][LEATHER] = 1

NonPrimaryArmor["WARRIOR"][CLOTH] = 1
NonPrimaryArmor["WARRIOR"][LEATHER] = 1
NonPrimaryArmor["WARRIOR"]["Post40"][MAIL] = 1

--[[ Config UI --]]
function BagSaver:CreateConfigMenu()
	self.name = "BagSaver"
	self.okay = function (self) return end
	self.cancel = function (self) return end

	local title, notes = select(2, GetAddOnInfo('BagSaver'))
	local text = self:CreateFontString(nil, 'ARTWORK', 'GameFontNormalLarge')
	text:SetPoint('TOPLEFT', 16, -16)
	text:SetText(title)

	local desc = self:CreateFontString(nil, 'ARTWORK', 'GameFontHighlightSmall')
	desc:SetHeight(32)
	desc:SetPoint('TOPLEFT', text, 'BOTTOMLEFT', 0, -8)
	desc:SetPoint('RIGHT', self, -32, 0)
	desc:SetNonSpaceWrap(true)
	desc:SetJustifyH('LEFT')
	desc:SetJustifyV('TOP')
	desc:SetText(notes)

	local autoSell = self:CreateCheckButton("Auto-Sell", self,"Enable automatic selling of \"junk\" to vendors", "autoSell", 'InterfaceOptionsCheckButtonTemplate')
	autoSell:SetPoint('TOPLEFT', 10, -75)

	local autoSellPrompt = self:CreateCheckButton("Prompt before Auto-Sell", self,"Prompt before automatically selling anything to a vendor", "autoSellPrompt", 'InterfaceOptionsSmallCheckButtonTemplate')
	autoSellPrompt:SetPoint('TOPLEFT', 150, -75)
	BagSaver:SetupDependentControl(autoSell,autoSellPrompt)

	local autoSellGrayItems = self:CreateCheckButton("Gray Items", self,"Consider gray items for Auto-Sale", "autoSellGrayItems", 'InterfaceOptionsCheckButtonTemplate')
	autoSellGrayItems:SetPoint('TOPLEFT', 20, -100)
	BagSaver:SetupDependentControl(autoSell,autoSellGrayItems)

	local autoSellUnusableBoundItems = self:CreateCheckButton("Unusable Soulbound Items", self,"Consider Soulbound items that you could *never* equip for Auto-Sale", "autoSellUnusableBoundItems", 'InterfaceOptionsCheckButtonTemplate')
	autoSellUnusableBoundItems:SetPoint('TOPLEFT', 20, -125)
	BagSaver:SetupDependentControl(autoSell,autoSellUnusableBoundItems)

	local autoSellNonPrimaryBoundArmor = self:CreateCheckButton("Non-Primary Soulbound Armor", self,"Consider soulbound armor below your primary type (cloth, leather, mail, plate) for Auto-Sale", "autoSellNonPrimaryBoundArmor", 'InterfaceOptionsCheckButtonTemplate')
	autoSellNonPrimaryBoundArmor:SetPoint('TOPLEFT', 20, -150)
	BagSaver:SetupDependentControl(autoSell,autoSellNonPrimaryBoundArmor)

	local findDisposableItems = self:CreateCheckButton("Find Discardable Items", self,"Find items to discard when bags are full", "findDisposableItems", 'InterfaceOptionsCheckButtonTemplate')
	findDisposableItems:SetPoint('TOPLEFT', 10, -185)

	InterfaceOptions_AddCategory(self)
end

function BagSaver.CreateDisposableItemFrame(numItems)
	local DisposableItemFrame = CreateFrame("Frame","DisposableItemFrame"..numItems,UIParent)
	tinsert(UISpecialFrames,DisposableItemFrame:GetName())
	DisposableItemFrame:SetBackdrop({bgFile = "Interface\\DialogFrame\\UI-DialogBox-Background", edgeFile = "Interface\\DialogFrame\\UI-DialogBox-Border", tile = true, tileSize = 32, edgeSize = 32, insets = { left = 11, right = 12, top = 12, bottom = 11 } } );
	DisposableItemFrame:SetPoint("TOP",0,-133)
	DisposableItemFrame:SetWidth(320)
	DisposableItemFrame:SetHeight(70 + (37 * numItems))
	DisposableItemFrame:SetFrameStrata("DIALOG")
	local DisposableItemsText = DisposableItemFrame:CreateFontString(DisposableItemFrame:GetName().."Text", 'ARTWORK', 'GameFontNormal')
	DisposableItemsText:SetPoint('TOP', 0, -16)
	DisposableItemsText:SetWidth(275)
	DisposableItemsText:SetJustifyH("CENTER")
	DisposableItemsText:SetText("These are your least valuable stacks.\nClick and discard one to make room.")
	local DisposableItemCloseButton = CreateFrame("Button","DisposableItemsCloseButton",DisposableItemFrame,"UIPanelCloseButton")
	DisposableItemCloseButton:SetPoint("TOPRIGHT",0,0)
	DisposableItemFrame.itemButtons = {}
	for i=1,numItems do
		local itemButton = BagSaver:CreateItemButton("DisposableItemButton"..numItems.."-"..i,DisposableItemFrame)
		itemButton:SetPoint("TOP",DisposableItemsText:GetName(),"BOTTOM",-90,-10-(37*(i-1)))
		tinsert(DisposableItemFrame.itemButtons,itemButton)
	end

	return DisposableItemFrame
end

function BagSaver:CreateCheckButton(name, parent, tooltipText, configKey, template)
	local button = CreateFrame('CheckButton', parent:GetName() .. name, parent, template)
	getglobal(button:GetName() .. 'Text'):SetText(name)
	button:SetChecked(BagSaver.GetConfigValue(configKey))
	button:SetScript('OnClick', function(self)
		if self:GetChecked() then
			BagSaver.SetConfigValue(configKey, true)
		else 
			BagSaver.SetConfigValue(configKey, false)
		end
		if ( self.dependentControls ) then
			if ( self:GetChecked() ) then
				for _, control in pairs(self.dependentControls) do
					control:Enable()
				end
			else
				for _, control in pairs(self.dependentControls) do
					control:Disable()
				end
			end
		end
	end)
	button:SetScript('OnEnter', function(self)
		GameTooltip:SetOwner(self, "ANCHOR_RIGHT")
		GameTooltip:SetText(tooltipText, nil, nil, nil, 1, 1)
		GameTooltip:Show() end)
	button:SetScript('OnLeave', function(self) GameTooltip:Hide() end)

	return button
end

function BagSaver:CreateItemButton(name, parent, item)
	local itemButton = CreateFrame("Button",name,parent,"ItemButtonTemplate")
	local itemTexture = itemButton:CreateTexture(itemButton:GetName().."Texture","ARTWORK")
	itemTexture:SetTexture("Interface\\QuestFrame\\UI-QuestItemNameFrame")
	itemTexture:SetWidth(140)
	itemTexture:SetHeight(62)
	itemTexture:SetPoint("LEFT",30,0)
	local itemFont = itemButton:CreateFontString(itemButton:GetName().."Text","ARTWORK","GameFontNormal")
	itemFont:SetWidth(103)
	itemFont:SetHeight(38)
	itemFont:SetJustifyH("LEFT")
	itemFont:SetPoint("LEFT",itemButton:GetName(),"RIGHT",8,0)
	local itemExtraFont = itemButton:CreateFontString(itemButton:GetName().."ExtraText","ARTWORK","GameFontNormal")
	itemExtraFont:SetWidth(100)
	itemExtraFont:SetHeight(38)
	itemExtraFont:SetJustifyH("LEFT")
	itemExtraFont:SetPoint("LEFT",itemFont:GetName(),"RIGHT",12,0)

	itemButton:Show()
	itemButton:SetHeight(37)
	itemButton:SetWidth(37)

	return itemButton
end

function BagSaver.SetItemButtonInfo(itemButton, item)
	if ( item and type(item) == "table" ) then
		_G[itemButton:GetName()].link = item.link
		_G[itemButton:GetName().."IconTexture"]:SetTexture(item.texture);
		local nameText = _G[itemButton:GetName().."Text"];
		nameText:SetTextColor(unpack(item.color or {1, 1, 1, 1}));
		nameText:SetText(item.name);
		local nameExtraText = _G[itemButton:GetName().."ExtraText"];
		nameExtraText:SetTextColor(unpack({1, 1, 1, 1}));
		nameExtraText:SetText("Total Value:\n " .. BagSaver.GetMoneyString(item.count * item.value));
		if ( item.count and item.count > 1 ) then
			_G[itemButton:GetName().."Count"]:SetText(item.count);
			_G[itemButton:GetName().."Count"]:Show();
		else
			_G[itemButton:GetName().."Count"]:Hide();
		end

		itemButton:SetScript("OnEnter", function(self) if ( self.link ) then GameTooltip:SetOwner(self, "ANCHOR_RIGHT") GameTooltip:SetHyperlink(self.link) end end)
		itemButton:SetScript("OnLeave", function(self) GameTooltip:Hide() end)
		itemButton:SetScript("OnClick", function(self,button,down) 
						if GetContainerItemLink(item.bag,item.slot) == item.link then 
							PickupContainerItem(item.bag, item.slot) 
						else 
							print("Inventory has changed, tried to use wrong item!") 
							itemButton:GetParent():Hide() 
						end 
						end )
	end
end

function BagSaver:SetupDependentControl (dependency, control)
	if ( not dependency ) then
		return
	end

	assert(control)

	dependency.dependentControls = dependency.dependentControls or {}
	tinsert(dependency.dependentControls, control)

	if ( control.type ~= CONTROLTYPE_DROPDOWN ) then
		control.Disable = function (self) getmetatable(self).__index.Disable(self) _G[self:GetName().."Text"]:SetTextColor(GRAY_FONT_COLOR.r, GRAY_FONT_COLOR.g, GRAY_FONT_COLOR.b) end
		control.Enable = function (self) getmetatable(self).__index.Enable(self) _G[self:GetName().."Text"]:SetTextColor(HIGHLIGHT_FONT_COLOR.r, HIGHLIGHT_FONT_COLOR.g, HIGHLIGHT_FONT_COLOR.b) end
		if dependency:GetChecked() then
			control:Enable()
		else
			control:Disable()
		end
	else
		control.Disable = function (self) UIDropDownMenu_DisableDropDown(self) end
		control.Enable = function (self) UIDropDownMenu_EnableDropDown(self) end
	end

end