-- define the shop size in % of map size (this influence the shop-ID and sign-ID)
local AMC_ShopSize = {
	["TheExodar"]      = 2.0,
	["Darnassis"]      = 2.0,
	["Ironforge"]      = 1.6,
	["Stormwind"]      = 1.0,
	["SilvermoonCity"] = 3.0,
	["Undercity"]      = 1.6,
	["ThunderBluff"]   = 1.4,
	["Ogrimmar"]       = 1.2,
	["ShattrathCity"]  = 1.4,
	["Dalaran1_"]      = 2.0,
	["Dalaran2_"]      = 3.0,
}

-- all the saved variables
AtlasMajorCities_DB = {};
AtlasMajorCities_Continent = {};
AtlasMajorCities_City = {};
AtlasMajorCities_Zone = {};
AtlasMajorCities_Shop = {};
AtlasMajorCities_Title = {};
AtlasMajorCities_NPC = {};
AtlasMajorCities_Comment = {};

-- sub-commands
local AMC_DeleteString   = AtlasMajorCities_loc["subDeleteString"];
local AMC_NoPosString    = AtlasMajorCities_loc["subNoPosString"];
local AMC_DelNoPosString = AtlasMajorCities_loc["subDelNoPosString"];

-- global identifiers
local AMC_MapName = "";
local AMC_Zone = "";
local AMC_Color = "";
local AMC_PosX = 0.0;
local AMC_PosY = 0.0;
local AMC_ShopID = "";
local AMC_ToolTipText = "";
local AMC_ToolTipSign = "";
local AMC_ShopList = {};

-- global strings
local AMC_ContinentText = "";
local AMC_CityText = "";
local AMC_ZoneText = "";
local AMC_ShopLabel = "";
local AMC_ShopText = "";
local AMC_NPCComment = "";

-- scan mode switch
local AMC_ScanMode = 0;


-- ************************************************
-- *** Start definition of some basic functions ***
-- ************************************************

-- check that the displayed tooltip is originating from a sign at the world frame
function AtlasMajorCities_CheckSignToolTip()
	if ( (GetMouseFocus():GetName() == "WorldFrame") and GameTooltip:IsVisible() ) then
		if ( (GameTooltip:NumLines() == 1) and (not GameTooltipTextRight1:GetText()) ) then
			local tcolor = { GameTooltipTextLeft1:GetTextColor() };
			if ( math.floor((tcolor[1]+tcolor[2]+tcolor[3])*100.0) == 182 ) then
				local bcolor = { GameTooltip:GetBackdropColor() };
				if ( math.floor((bcolor[1]+bcolor[2]+bcolor[3])*100.0) == 53 ) then
					return true;
				end
			end
		end
	end
	return false;
end

-- define the sort order for zone, and shop
function AtlasMajorCities_CompareEntry(a, b)
	local sa, sb;
	if ( a.Name ) then sa = a.Name; end
	if ( b.Name ) then sb = b.Name; end
	if ( a.ID ) then sa = a.ID; end
	if ( b.ID ) then sb = b.ID; end
	if ( a.Label ) then sa = a.Label; end
	if ( b.Label ) then sb = b.Label; end

	-- set high number if ID="000000" (sort no position shop to bottom)
	if ( sa == "000000" ) then sa = "999999"; end
	if ( sb == "000000" ) then sb = "999999"; end

	-- check if both variables are defined
	if ( not sa and not sb ) then sa = "0"; sb = "1";
	elseif ( not sa )        then sa = "0";
	elseif ( not sb )        then sb = "0";
	end

	-- change variables, that strings are sorted to the top
	na = tonumber(sa); nb = tonumber(sb);
	if ( na and nb ) then sa = na; sb = nb;
		elseif ( na )    then sb = "."..sb;
	elseif ( nb )    then sa = "."..sa;
	end

	return sa < sb;
end

-- set some variables for the actual map: AMC_MapName, AMC_PosX, AMC_PosY
local function FAMC_GetMapInfos()
	SetMapToCurrentZone();

	-- get the map name
	AMC_MapName = GetMapInfo();
	local dungeonLevel = GetCurrentMapDungeonLevel();
	if ( dungeonLevel > 0 ) then
		AMC_MapName = AMC_MapName..dungeonLevel.."_";
	else
		if ( AMC_MapName == "Dalaran" ) then
			AMC_MapName = "Dalaran1_";
		end
	end

	-- get player position
	local posX, posY = GetPlayerMapPosition("player");
	AMC_PosX = math.floor(posX * 1000.0 + 0.5) / 10.0;
	AMC_PosY = math.floor(posY * 1000.0 + 0.5) / 10.0;
end

-- get the list for the zone AMC_Zone at the map AMC_MapName from DB
local function FAMC_GetZoneList()
	local alist = AtlasMajorCities_DB[AMC_MapName];
	if ( alist ) then
		alist = AtlasMajorCities_DB[AMC_MapName]["Areas"];
	end
	local zlist;
	if ( alist ) then
		local index = next(alist);
		while ( index and alist[index] ) do
			if ( alist[index].Name == AMC_Zone ) then
				zlist = alist[index]; 
			end
			index = next(alist, index);
		end
	end
	return zlist;
end

-- compute ID string from position
local function FAMC_GetPositionID(size)
	local xpos = math.floor(AMC_PosX / size + 0.5) * math.floor(size * 10.0);
	local ypos = math.floor(AMC_PosY / size + 0.5) * math.floor(size * 10.0);
	return string.format("%03i%03i", xpos, ypos);
end

-- extract the NPC name and title from the tooltip
local function FAMC_GetNPCtext()
	local text1 = getglobal("GameTooltipTextLeft1"):GetText();
	local text2 = ", "..getglobal("GameTooltipTextLeft2"):GetText();
	local text3 = "";
	if ( string.find(text2,LEVEL) == 3 ) then
		text2 = "";
	else
		text3 = " "..getglobal("GameTooltipTextLeft3"):GetText();
		if ( string.find(text3,LEVEL) == 2 ) then
			text3 = "";
		end
	end
	return text1..text2..text3;
end

-- show help
local function FAMC_ShowHelp()
	if ( not AtlasMajorCities_VariablesLoaded ) then return; end

	DEFAULT_CHAT_FRAME:AddMessage(AtlasMajorCities_loc["Help01"], .9, .0, .9);
	DEFAULT_CHAT_FRAME:AddMessage(AtlasMajorCities_loc["Help02"], .9, .0, .9);
	DEFAULT_CHAT_FRAME:AddMessage(AtlasMajorCities_loc["Help03"], .9, .0, .9);
	DEFAULT_CHAT_FRAME:AddMessage(AtlasMajorCities_loc["Help04"], .9, .0, .9);
	DEFAULT_CHAT_FRAME:AddMessage(AtlasMajorCities_loc["Help05"], .9, .0, .9);
	DEFAULT_CHAT_FRAME:AddMessage(AtlasMajorCities_loc["Help06"], .9, .0, .9);
	DEFAULT_CHAT_FRAME:AddMessage(AtlasMajorCities_loc["Help07"], .9, .0, .9);
	DEFAULT_CHAT_FRAME:AddMessage(AtlasMajorCities_loc["Help08"], .9, .0, .9);
	DEFAULT_CHAT_FRAME:AddMessage(AtlasMajorCities_loc["Help09"], .9, .0, .9);
	DEFAULT_CHAT_FRAME:AddMessage(AtlasMajorCities_loc["Help10"], .9, .0, .9);
	DEFAULT_CHAT_FRAME:AddMessage(AtlasMajorCities_loc["Help11"], .9, .0, .9);
	DEFAULT_CHAT_FRAME:AddMessage(AtlasMajorCities_loc["Help12"], .9, .0, .9);
	DEFAULT_CHAT_FRAME:AddMessage(AtlasMajorCities_loc["Help13"], .9, .0, .9);
	DEFAULT_CHAT_FRAME:AddMessage(AtlasMajorCities_loc["Help14"], .9, .0, .9);
	DEFAULT_CHAT_FRAME:AddMessage(AtlasMajorCities_loc["Help15"], .9, .0, .9);
	DEFAULT_CHAT_FRAME:AddMessage(AtlasMajorCities_loc["Help16"], .9, .0, .9);
	DEFAULT_CHAT_FRAME:AddMessage(AtlasMajorCities_loc["Help17"], .9, .0, .9);
	DEFAULT_CHAT_FRAME:AddMessage(AtlasMajorCities_loc["Help18"], .9, .0, .9);
	DEFAULT_CHAT_FRAME:AddMessage(AtlasMajorCities_loc["Help19"], .9, .0, .9);
	DEFAULT_CHAT_FRAME:AddMessage(AtlasMajorCities_loc["Help20"], .9, .0, .9);
	DEFAULT_CHAT_FRAME:AddMessage(AtlasMajorCities_loc["Help21"], .9, .0, .9);
	DEFAULT_CHAT_FRAME:AddMessage(AtlasMajorCities_loc["Help22"], .9, .0, .9);
	DEFAULT_CHAT_FRAME:AddMessage(AtlasMajorCities_loc["Help23"], .9, .0, .9);
	DEFAULT_CHAT_FRAME:AddMessage(AtlasMajorCities_loc["Help24"], .9, .0, .9);
	DEFAULT_CHAT_FRAME:AddMessage(AtlasMajorCities_loc["Help25"], .9, .0, .9);
	DEFAULT_CHAT_FRAME:AddMessage(AtlasMajorCities_loc["Help26"], .9, .0, .9);
	DEFAULT_CHAT_FRAME:AddMessage(AtlasMajorCities_loc["Help27"], .9, .0, .9);
end


-- *******************************************
-- *** Define some popup windows for input ***
-- *******************************************

StaticPopupDialogs["AtlasMajorCities_ZoneManually"] = {
	text = TEXT(AtlasMajorCities_loc["dialogZoneTitle"]),
	button1 = TEXT(ACCEPT),
	button2 = TEXT(CANCEL),
	hasEditBox = 1,
	maxLetters = 100,
	OnShow = function(self)
		self.editBox:SetText("");
		self.editBox:SetFocus();
	end,
	OnAccept = function(self)
		AMC_ZoneText = self.editBox:GetText();
		AtlasMajorCities_SetZoneDB();
	end,
	EditBoxOnEnterPressed = function(self)
		local parent = self:GetParent();
		AMC_ZoneText = parent.editBox:GetText();
		AtlasMajorCities_SetZoneDB();
		parent:Hide();
	end,
	timeout = 0,
	exclusive = 1,
	whileDead = 1,
	hideOnEscape = 1
};

StaticPopupDialogs["AtlasMajorCities_LabelManually"] = {
	text = TEXT(AtlasMajorCities_loc["dialogShopLabel"]),
	button1 = TEXT(ACCEPT),
	button2 = TEXT(CANCEL),
	hasEditBox = 1,
	maxLetters = 100,
	OnShow = function(self)
		local text = "";
		if ( AMC_ShopList.Label ) then text = AMC_ShopList.Label; end
		self.editBox:SetText(text);
		self.editBox:SetFocus();
	end,
	OnAccept = function(self)
		AMC_ShopLabel = self.editBox:GetText();
		AtlasMajorCities_SetLabelDB();
	end,
	EditBoxOnEnterPressed = function(self)
		local parent = self:GetParent();
		AMC_ShopLabel = parent.editBox:GetText();
		AtlasMajorCities_SetLabelDB();
		parent:Hide();
	end,
	timeout = 0,
	exclusive = 1,
	whileDead = 1,
	hideOnEscape = 1
};

StaticPopupDialogs["AtlasMajorCities_TitleManually"] = {
	text = TEXT(AtlasMajorCities_loc["dialogShopTitle"]),
	button1 = TEXT(ACCEPT),
	button2 = TEXT(CANCEL),
	hasEditBox = 1,
	maxLetters = 100,
	OnShow = function(self)
		local SignToolTip = AtlasMajorCities_CheckSignToolTip();
		if ( SignToolTip and (AMC_ShopText == "") ) then
			AMC_ShopText = getglobal("GameTooltipTextLeft1"):GetText();
		end
		if ( (GetMinimapZoneText() ~= GetZoneText()) and (GetMinimapZoneText() ~= AMC_ZoneText) and (AMC_ShopText == "") ) then
			AMC_ShopText = GetMinimapZoneText();
		end
		self.editBox:SetText(AMC_ShopText);
		self.editBox:SetFocus();
	end,
	OnAccept = function(self)
		AMC_ShopText = self.editBox:GetText();
		AtlasMajorCities_SetTitleDB();
	end,
	EditBoxOnEnterPressed = function(self)
		local parent = self:GetParent();
		AMC_ShopText = parent.editBox:GetText();
		AtlasMajorCities_SetTitleDB();
		parent:Hide();
	end,
	timeout = 0,
	exclusive = 1,
	whileDead = 1,
	hideOnEscape = 1
};

StaticPopupDialogs["AtlasMajorCities_CommentManually"] = {
	text = TEXT(AtlasMajorCities_loc["dialogNPCComment"]),
	button1 = TEXT(ACCEPT),
	button2 = TEXT(CANCEL),
	hasEditBox = 1,
	maxLetters = 100,
	OnShow = function(self)
		self.editBox:SetText(AMC_NPCComment);
		self.editBox:SetFocus();
	end,
	OnAccept = function(self)
		AMC_NPCComment = self.editBox:GetText();
		AtlasMajorCities_SetCommentMSG();
	end,
	EditBoxOnEnterPressed = function(self)
		local parent = self:GetParent();
		AMC_NPCComment = parent.editBox:GetText();
		AtlasMajorCities_SetCommentMSG();
		parent:Hide();
	end,
	timeout = 0,
	exclusive = 1,
	whileDead = 1,
	hideOnEscape = 1
};


-- ******************************************
-- *** Start definition of main functions ***
-- ******************************************

-- activate the requested scan mode or reload the database
function AtlasMajorCities_SetScan(msg)
	if ( not AtlasMajorCities_VariablesLoaded ) then return; end

	if ( msg == "help" ) then
		FAMC_ShowHelp();
		return;
	end

	-- show label at its default positions and activate label positioning at the map
	if ( msg == "movelabel" ) then
		if ( not AtlasMajorCities_UserDB ) then
			DEFAULT_CHAT_FRAME:AddMessage(AtlasMajorCities_loc["msgUserOnly"], .9, .0, .0);
			return;
		end
		AtlasMajorCities_LabelAtPos = -1;
		AtlasMajorCities_InsertData();
		DEFAULT_CHAT_FRAME:AddMessage(AtlasMajorCities_loc["msgOnMove"], .9, .0, .9);
		return;
	end

	-- show label at its default position
	if ( msg == "defaultlabel" ) then
		AtlasMajorCities_LabelAtPos = 0;
		AtlasMajorCities_InsertData();
		return;
	end

	-- show label at shop position
	if ( msg == "shoplabel" ) then
		AtlasMajorCities_LabelAtPos = 1;
		AtlasMajorCities_InsertData();
		return;
	end

	-- show label at shop sign position
	if ( msg == "signlabel" ) then
		AtlasMajorCities_LabelAtPos = 2;
		AtlasMajorCities_InsertData();
		return;
	end

	-- reload DB in Atlas
	if ( msg == "reload" ) then
		AtlasMajorCities_LabelAtPos = 0;
		AtlasMajorCities_UserDB = true;
		AtlasMajorCities_InsertData();
		return;
	end

	-- load original city data like Atlas_RegisterPlugin
	if ( msg == "revert" ) then
		AtlasMajorCities_LabelAtPos = 0;
		AtlasMajorCities_UserDB = false;
		AtlasMajorCities_InsertData();
		return;
	end

	-- copy internal db to user data (Attention! This overwrite all user data.)
	if ( msg == "copydbtouser" ) then
		if ( AtlasMajorCities_DB0 ) then
			AtlasMajorCities_DB        = AtlasMajorCities_DB0;
			AtlasMajorCities_Continent = AtlasMajorCities_Continent0;
			AtlasMajorCities_City      = AtlasMajorCities_City0;
			AtlasMajorCities_Zone      = AtlasMajorCities_Zone0;
			AtlasMajorCities_Shop      = AtlasMajorCities_Shop0;
			AtlasMajorCities_Title     = AtlasMajorCities_Title0;
			AtlasMajorCities_NPC       = AtlasMajorCities_NPC0;
			AtlasMajorCities_Comment   = AtlasMajorCities_Comment0;
			DEFAULT_CHAT_FRAME:AddMessage(AtlasMajorCities_loc["msgDBcopied"], .9, .0, .0);
		else
			DEFAULT_CHAT_FRAME:AddMessage(AtlasMajorCities_loc["msgnoDB"], .9, .0, .0);
		end
		return;
	end

	-- switch off the san mode
	if ( msg == "off" ) then
		AMC_Zone = nil;
		AMC_ShopID = nil;
		AMC_ShopList = nil;
		AMC_ScanMode = 0;
		DEFAULT_CHAT_FRAME:AddMessage(AtlasMajorCities_loc["msgOff"], .9, .0, .9);
		return;
	end

	-- set the scan mode
	AMC_ToolTipText = "";
	if ( msg == "pos" ) then
		AMC_ScanMode = 1;
		DEFAULT_CHAT_FRAME:AddMessage(AtlasMajorCities_loc["msgOnPos"], .9, .0, .9);
	elseif ( msg == "all" ) then
		AMC_ScanMode = 2;
		DEFAULT_CHAT_FRAME:AddMessage(AtlasMajorCities_loc["msgOnAll"], .9, .0, .9);
	elseif ( msg == "on" ) then
		AMC_ScanMode = 3;
		DEFAULT_CHAT_FRAME:AddMessage(AtlasMajorCities_loc["msgOn"], .9, .0, .9);
	elseif ( msg == "sign" ) then
		AMC_Zone = nil;
		AMC_ShopID = nil;
		AMC_ShopList = nil;
		AMC_ScanMode = 4;
		DEFAULT_CHAT_FRAME:AddMessage(AtlasMajorCities_loc["msgOnSign"], .9, .0, .9);
	elseif ( msg == "npc" ) then
		AMC_Zone = nil;
		AMC_ShopID = nil;
		AMC_ShopList = nil;
		AMC_ScanMode = 5;
		DEFAULT_CHAT_FRAME:AddMessage(AtlasMajorCities_loc["msgOnNPC"], .9, .0, .9);
	else
		FAMC_ShowHelp();
	end
end

-- get the zone of the city by coordinates
-- set: AMC_MapName, AMC_Zone, AMC_Color, AMC_PosX, AMC_PosY
-- return: zlist
local function FAMC_GetZone(set)
	-- set AMC_PosX, AMC_PosY, and AMC_MapName
	FAMC_GetMapInfos();

	-- search for the zone name by coordinates
	AMC_Zone = nil;
	AMC_Color = nil;
	local zonelist = AtlasMajorCities_ZoneList[AMC_MapName];
	if ( zonelist ) then
		local x0 = AMC_PosX;
		local y0 = AMC_PosY;
		for idx = 1, table.getn(zonelist) do
			local x1 = zonelist[idx]["x1"];
			local x2 = zonelist[idx]["x2"];
			local y1 = zonelist[idx]["y1"];
			local y2 = zonelist[idx]["y2"];
			if ( (x0 >= x1) and (x0 <= x2) and (y0 >= y1) and (y0 <= y2) ) then
				AMC_Zone = zonelist[idx]["Zone"];
				AMC_Color = zonelist[idx]["Color"];
			end
		end
	end

	-- error if no zones defined yet for this map
	if ( not AMC_Zone ) then
		local error = AtlasMajorCities_loc["errNoZoneDef"]..AMC_MapName;
		DEFAULT_CHAT_FRAME:AddMessage(error, .9, .0, .0);
		return;
	end

	-- find this zone list in the DB
	local zlist = FAMC_GetZoneList();

	-- error if no zone list in DB and option not set
	if ( not set and not zlist ) then
		local error = AtlasMajorCities_loc["errNoZoneList"];
		DEFAULT_CHAT_FRAME:AddMessage(error, .9, .0, .0);
		AMC_Zone = nil;
		AMC_Color = nil;
	end

	return zlist;
end

-- get the continent, city, and zone title
function AtlasMajorCities_SetZone(msg)
	if ( not AtlasMajorCities_VariablesLoaded ) then return; end
	if ( (AMC_ScanMode == 0) or (AMC_ScanMode > 3) ) then return; end

	-- get zone, color, and position
	FAMC_GetZone(true);
	if ( not AMC_Zone ) then return; end

	-- get continent
	local continentID = GetCurrentMapContinent();
	local continentNames = { GetMapContinents() };
	for key, val in pairs(continentNames) do
		if ( key == continentID ) then AMC_ContinentText = val; end
	end

	-- get the name of the city
	CityName = GetZoneText();
	if ( GetNumDungeonMapLevels() > 1 ) then
		SetMapToCurrentZone();
		local level = GetCurrentMapDungeonLevel();
		local mapname = strupper(GetMapInfo());
		CityName = _G["DUNGEON_FLOOR_"..mapname..level];
	end

	-- get faction of city
	local CityFaction, CityRace;
	local pvpType, isFFA, faction = GetZonePVPInfo();
	if ( pvpType == "friendly" or pvpType == "hostile" ) then
		CityFaction = faction;
	elseif ( pvpType == "sanctuary" ) then
		CityFaction = FACTION_STANDING_LABEL4; -- "Neutral" faction standing
	end

	-- get race of city (from a player target belonging to this city)
	CityRace = UnitRace("target");

	-- create city text
	if ( CityFaction and CityRace) then
		AMC_CityText = CityName.." ("..CityFaction.." - "..CityRace..")";
	elseif ( CityFaction) then
		AMC_CityText = CityName.." ("..CityFaction..")";
	else
		AMC_CityText = CityName;
	end

	-- get (MinimapZoneText) and set zone title
	if ( msg and msg ~= "" ) then
		AMC_ZoneText = msg;
		if ( msg == AMC_DeleteString ) then AMC_ZoneText = nil; end
		AtlasMajorCities_SetZoneDB();
	else
		AMC_ZoneText = GetMinimapZoneText();
		if ( AMC_ZoneText == GetZoneText() ) then
			StaticPopup_Show("AtlasMajorCities_ZoneManually");
		else
			AtlasMajorCities_SetZoneDB();
		end
	end
end

-- set DB entry for city and zone
function AtlasMajorCities_SetZoneDB()
	-- create city entry in DB
	if ( not AtlasMajorCities_DB[AMC_MapName] ) then
		if ( AMC_ScanMode < 3 ) then
			AtlasMajorCities_DB[AMC_MapName] = {};
			AtlasMajorCities_DB[AMC_MapName]["Areas"] = {};
		else
			local error = AtlasMajorCities_loc["errNoCityDef"]..AMC_MapName;
			DEFAULT_CHAT_FRAME:AddMessage(error, .9, .0, .0);
			return;
		end
	end

	-- check if zone already exist
	local alist = AtlasMajorCities_DB[AMC_MapName]["Areas"];
	local zlist;
	local index = next(alist);
	while ( index and alist[index] ) do
		if ( alist[index].Name == AMC_Zone ) then
			zlist = alist[index];
		end
		index = next(alist, index);
	end

	-- create zone entry in DB
	if ( not zlist ) then
		if ( AMC_ScanMode < 3 ) then
			zlist = {};
			zlist.Name = AMC_Zone;
			zlist.Color = AMC_Color;
			table.insert(alist, zlist)
			table.sort(alist, AtlasMajorCities_CompareEntry);
		else
			local error = AtlasMajorCities_loc["errNoZoneList"];
			DEFAULT_CHAT_FRAME:AddMessage(error, .9, .0, .0);
			return;
		end
	end

	-- set continent, city, and zone name
	AtlasMajorCities_Continent[AMC_MapName] = AMC_ContinentText;
	AtlasMajorCities_City[AMC_MapName] = AMC_CityText;
	AtlasMajorCities_Zone[AMC_MapName.."-"..AMC_Zone] = AMC_ZoneText;

	-- show status msg
	local text;
	if ( AMC_ZoneText ) then
		text = "*** "..AMC_ContinentText.." - "..AMC_CityText.." ("..AMC_MapName..") - "
		text = text..AMC_ZoneText.." ("..AMC_Zone..") - "..AMC_Color;	
	else
		text = AtlasMajorCities_loc["msgDelZone"];
		AMC_ZoneText = "";
	end
	DEFAULT_CHAT_FRAME:AddMessage(text, .9, .0, .9);
end

-- set shop assignment
function AtlasMajorCities_AssignShop(msg)
	if ( not AtlasMajorCities_VariablesLoaded ) then return; end
	if ( (AMC_ScanMode == 0) or (AMC_ScanMode > 3) ) then return; end

	-- get zone list, zone, color, and position
	local zone = AMC_Zone;
	local map = AMC_MapName;
	local zlist = FAMC_GetZone();
	if ( not AMC_Zone ) then
		if ( (msg == "check") ) then
			AMC_Zone = zone;
			AMC_MapName = map;
		end
		return;
	end

	-- set position to zero to assign shop with no position
	if ( msg == AMC_NoPosString ) then
		AMC_PosX = 0.0;
		AMC_PosY = 0.0;
	end

	-- get shop ID from position
	sid = FAMC_GetPositionID(AMC_ShopSize[AMC_MapName]);

	-- get shop list from shop ID
	local slist;
	index = next(zlist);
	while ( index and zlist[index] ) do
		if ( zlist[index].ID == sid ) then
			slist = zlist[index];
		end
		index = next(zlist, index);
	end

	-- if in checked mode and no shop at this position
	if ( not slist and (msg == "check") ) then return; end

	-- assign shop list
	if ( not slist ) then
		AMC_ShopID = nil;
		AMC_ShopList = nil;
	else
		AMC_ShopList = slist;
		AMC_ShopID = sid;
	end

	-- show status msg
	if ( not slist ) then
		local error = AtlasMajorCities_loc["errNoShopList"];
		DEFAULT_CHAT_FRAME:AddMessage(error, .9, .0, .0);
	else
		local sign = AtlasMajorCities_Shop[slist.sID];
		if ( not sign ) then sign = ""; end
		local title = AtlasMajorCities_Title[AMC_MapName.."-"..AMC_Zone.."-"..AMC_ShopID];
		if ( not title ) then title = ""; end
		local text = AtlasMajorCities_loc["msgIsShop"]..AMC_ShopID.." - "..sign.."."..title;
		DEFAULT_CHAT_FRAME:AddMessage(text, .9, .0, .9);
	end
end

-- set shop-position
function AtlasMajorCities_SetPosition(msg)
	if ( not AtlasMajorCities_VariablesLoaded ) then return; end
	if ( (AMC_ScanMode == 0) or (AMC_ScanMode > 2) ) then return; end

	-- get zone list, zone, color, and position
	local zlist = FAMC_GetZone();
	if ( not AMC_Zone ) then return; end

	-- set position to zero to assign shop with no position
	if ( (msg == AMC_NoPosString) or (msg == AMC_DelNoPosString) ) then
		AMC_PosX = 0.0;
		AMC_PosY = 0.0;
	end

	-- get shop ID from position
	AMC_ShopID = FAMC_GetPositionID(AMC_ShopSize[AMC_MapName]);

	-- get shop list from shop ID and set position or remove shop
	local oldtitle = "";
	local slist;
	index = next(zlist);
	while ( index and zlist[index] ) do
		if ( zlist[index].ID == AMC_ShopID ) then
			if ( (msg == AMC_DeleteString) or (msg == AMC_DelNoPosString) ) then
				-- remove all NPCs at this shop from the NPC and comment list
				slist = zlist[index];
				for key, value in pairs(slist) do
					if ( type(key) == "number" ) then
						local value0 = "NPC"..tostring(value);
						AtlasMajorCities_NPC[value0] = nil;
						AtlasMajorCities_Comment[value0] = nil;
					end
				end
				-- remove the title of this shop from the shop sign and title list
				if ( AtlasMajorCities_Shop[slist.sID] ) then
					oldtitle = AtlasMajorCities_Shop[slist.sID];
					AtlasMajorCities_Shop[slist.sID] = nil;
				end
				if ( AtlasMajorCities_Title[AMC_MapName.."-"..AMC_Zone.."-"..AMC_ShopID] ) then
					oldtitle = oldtitle.."."..AtlasMajorCities_Title[AMC_MapName.."-"..AMC_Zone.."-"..AMC_ShopID]..".";
					AtlasMajorCities_Title[AMC_MapName.."-"..AMC_Zone.."-"..AMC_ShopID] = nil;
				end
				-- finally remove the shop
				slist = nil;
				table.remove(zlist, index);
				index = index - 1;
			else
				slist = zlist[index];
				if ( msg ~= AMC_NoPosString ) then
					slist.PosX = AMC_PosX;
					slist.PosY = AMC_PosY;
				end
			end
		end
		if ( (type(index) == "number") and (index == 0) ) then
			index = next(zlist);
		else
			index = next(zlist, index);
		end
	end

	-- create a new shop and set its position
	local setlabel = false;
	if ( not slist and (msg ~= AMC_DeleteString) and (msg ~= AMC_DelNoPosString) ) then
		slist = {};
		slist.ID = AMC_ShopID;
		if ( msg ~= AMC_NoPosString ) then
			slist.PosX = AMC_PosX;
			slist.PosY = AMC_PosY;
		end
		table.insert(zlist, slist)
		table.sort(zlist, AtlasMajorCities_CompareEntry);
		-- set the label
		if ( msg ~= AMC_NoPosString ) then
			setlabel = true;
		end
	end

	-- assign the shop
	if ( not slist ) then
		AMC_ShopList = nil;
	else
		if ( msg == AMC_NoPosString ) then
			AtlasMajorCities_AssignShop(AMC_NoPosString);
		else
			AtlasMajorCities_AssignShop();
		end
	end

	-- show status msg
	local text;
	if ( not slist ) then
		text = AtlasMajorCities_loc["msgDelShop"].." - "..oldtitle.." - "..AMC_PosX..", "..AMC_PosY;
	else
		text = AtlasMajorCities_loc["msgCreatedShop"]..AMC_ShopID.." - "..AMC_MapName.." - "..AMC_PosX..", "..AMC_PosY;
	end
	DEFAULT_CHAT_FRAME:AddMessage(text, .9, .0, .9);

	-- set the label
	if ( setlabel ) then AtlasMajorCities_SetLabel(); end
end

-- get shop label
function AtlasMajorCities_SetLabel(msg)
	if ( not AtlasMajorCities_VariablesLoaded ) then return; end
	if ( (AMC_ScanMode == 0) or (AMC_ScanMode > 2) ) then return; end

	if ( not AMC_ShopList ) then
		local error = AtlasMajorCities_loc["errNoAssignment"];
		DEFAULT_CHAT_FRAME:AddMessage(error, .9, .0, .0);
		return;
	end

	if ( msg and msg ~= "" ) then
		AMC_ShopLabel = msg;
		if ( msg == AMC_DeleteString ) then AMC_ShopLabel = nil; end
		AtlasMajorCities_SetLabelDB();
	else
		StaticPopup_Show("AtlasMajorCities_LabelManually");
	end
end

-- set DB entry for shop label
function AtlasMajorCities_SetLabelDB()
	if ( AMC_ShopLabel and (AMC_ShopLabel == "") ) then AMC_ShopLabel = nil; end

	AMC_ShopList.Label = AMC_ShopLabel;

	-- get zone list
	local zlist = FAMC_GetZoneList()

	-- resort zone list after label change
	table.sort(zlist, AtlasMajorCities_CompareEntry);

	-- show status msg
	local text;
	if ( AMC_ShopLabel ) then
		text = AtlasMajorCities_loc["msgIsLabel"]..AMC_ShopLabel;
	else
		text = AtlasMajorCities_loc["msgDelLabel"];
		AMC_ShopLabel = "";
	end
	DEFAULT_CHAT_FRAME:AddMessage(text, .9, .0, .9);
end

-- create a list entry for a shop sign at actual position
function AtlasMajorCities_SetSign(msg)
	if ( not AtlasMajorCities_VariablesLoaded ) then return; end
	if ( (AMC_ScanMode == 0) or (AMC_ScanMode > 2) ) then return; end

	if ( not AMC_ShopList ) then
		local error = AtlasMajorCities_loc["errNoAssignment"];
		DEFAULT_CHAT_FRAME:AddMessage(error, .9, .0, .0);
		return;
	end

	-- get player position
	SetMapToCurrentZone();
	local posX, posY = GetPlayerMapPosition("player");
	AMC_PosX = math.floor(posX * 1000.0 + 0.5) / 10.0;
	AMC_PosY = math.floor(posY * 1000.0 + 0.5) / 10.0;

	local signID;
	if ( AMC_ShopList.sID ) then
		signID = AMC_ShopList.sID;
	else
		-- get shop sign ID from position
		signID = AMC_MapName..FAMC_GetPositionID(AMC_ShopSize[AMC_MapName]/2.0);
	end

	-- check if there is no sign yet
	local oldsign;
	if ( AtlasMajorCities_Shop[signID] ) then
		oldsign = AtlasMajorCities_Shop[signID];
	else
		AMC_ShopList.sID = signID;
		AtlasMajorCities_Shop[signID] = "-";
	end

	-- show status msg
	if ( oldsign ) then
		local error = AtlasMajorCities_loc["errOldShopSign"]..oldsign;
		DEFAULT_CHAT_FRAME:AddMessage(error, .9, .0, .0);
	else
		text = AtlasMajorCities_loc["msgCreatedSign"]..signID;
		DEFAULT_CHAT_FRAME:AddMessage(text, .9, .0, .9);
	end
end

-- get shop comment
function AtlasMajorCities_SetTitle(msg)
	if ( not AtlasMajorCities_VariablesLoaded ) then return; end
	if ( (AMC_ScanMode == 0) or (AMC_ScanMode > 3) ) then return; end

	-- if there is a shop at this position, assign its shop list
	AtlasMajorCities_AssignShop("check");

	-- check that shop list is assigned
	if ( not AMC_ShopList ) then
		local error = AtlasMajorCities_loc["errNoAssignment"];
		DEFAULT_CHAT_FRAME:AddMessage(error, .9, .0, .0);
		return;
	end

	-- set shop comment
	if ( msg and msg ~= "" ) then
		AMC_ShopText = msg;
		if ( msg == AMC_DeleteString ) then AMC_ShopText = nil; end
		AtlasMajorCities_SetTitleDB();
	else
		AMC_ShopText = AtlasMajorCities_Title[AMC_MapName.."-"..AMC_Zone.."-"..AMC_ShopID];
		if ( not AMC_ShopText ) then AMC_ShopText = ""; end
		StaticPopup_Show("AtlasMajorCities_TitleManually");
	end
end

-- set DB entry for shop comment
function AtlasMajorCities_SetTitleDB()
	if ( AMC_ShopText and (AMC_ShopText == "") ) then AMC_ShopText = nil; end

	-- write shop comment in DB
	AtlasMajorCities_Title[AMC_MapName.."-"..AMC_Zone.."-"..AMC_ShopID] = AMC_ShopText;

	-- show status msg
	local text;
	if ( AMC_ShopText ) then
		text = AtlasMajorCities_loc["msgIsShopComment"]..AMC_ShopText;
	else
		text = AtlasMajorCities_loc["msgDelTitle"];
		AMC_ShopText = "";
	end
	DEFAULT_CHAT_FRAME:AddMessage(text, .9, .0, .9);
end

-- add comment to the target NPC
function AtlasMajorCities_SetComment(msg)
	if ( not AtlasMajorCities_VariablesLoaded ) then return; end

	AMC_NPCComment = "";
	local exist;
	if ( TargetFrame:IsVisible() ) then
		-- check the type of target
		local guid = UnitGUID("target");
		local typ = tonumber(guid:sub(5,5), 16) % 8;
		if ( typ == 3 ) then
			-- get the NPC ID
			local npcid = tonumber(string.sub(guid,9,12), 16);
			local npcid0 = "NPC"..tostring(npcid);
			exist = AtlasMajorCities_NPC[npcid0];
			if ( AtlasMajorCities_Comment[npcid0] ) then
				AMC_NPCComment = AtlasMajorCities_Comment[npcid0];
			end
		end
	end

	if ( not exist ) then
		local error = AtlasMajorCities_loc["errNoNPC"];
		DEFAULT_CHAT_FRAME:AddMessage(error, .9, .0, .0);
		return;
	end

	-- set NPC comment
	if ( msg and msg ~= "" ) then
		AMC_NPCComment = msg;
		AtlasMajorCities_SetCommentMSG();
	else
		StaticPopup_Show("AtlasMajorCities_CommentManually");
	end
end

-- show status msg for NPC comment
function AtlasMajorCities_SetCommentMSG(msg)
	if ( AMC_NPCComment and (AMC_NPCComment == "") ) then AMC_NPCComment = AMC_DeleteString; end

	-- get npcid
	local guid = UnitGUID("target");
	local typ = tonumber(guid:sub(5,5), 16) % 8;
	local npcid = tonumber(string.sub(guid,9,12), 16);
	local npcid0 = "NPC"..tostring(npcid);

	-- add comment to NPC name
	if ( AMC_NPCComment == "delete" ) then
		AtlasMajorCities_Comment[npcid0] = nil;
	else
		AtlasMajorCities_Comment[npcid0] = AMC_NPCComment;
	end

	-- get NPC
	local npc = AtlasMajorCities_NPC[npcid0];

	-- show status msg
	local msg;
	if ( AMC_NPCComment ~= AMC_DeleteString ) then
		msg = AtlasMajorCities_loc["msgIsNPCComment"]..npc.." ("..AMC_NPCComment..")";
	else
		msg = AtlasMajorCities_loc["msgDelComment"];
	end
	DEFAULT_CHAT_FRAME:AddMessage(msg, .9, .0, .9);
end

-- set NPC name and assign to shop
local function FAMC_SetNPC(text, npcid)
	if (not text or not npcid) then return; end
	local npcid0 = "NPC"..tostring(npcid);

	-- add NPC to shop list
	if ( not AtlasMajorCities_NPC[npcid0] ) then
		if ( AMC_ScanMode < 3 ) then
			table.insert(AMC_ShopList, npcid);
			table.sort(AMC_ShopList);
		else
			local error = AtlasMajorCities_loc["errNoNPCDef"];
			DEFAULT_CHAT_FRAME:AddMessage(error, .9, .0, .0);
			return;
		end
	end

	-- set NPC name in DB
	AtlasMajorCities_NPC[npcid0] = text;

	-- get comment
	local comment = AtlasMajorCities_Comment[npcid0];
	if ( not comment ) then comment = "";
			   else comment = " ("..comment..")";
	end

	-- show status msg
	local msg = "#"..npcid.." - "..text..comment.." - "..AMC_PosX..", "..AMC_PosY;
	DEFAULT_CHAT_FRAME:AddMessage(msg, .9, .0, .9);
end

-- get shop sign text and set DB entry
local function FAMC_SetShopSign(SignText)
	-- get actual viewing  direction (degree)
	local angle = math.deg(GetPlayerFacing());

	-- get mouse position relativ to screen center (= viewing direction)
	local xmouse = GetCursorPosition() - UIParent:GetCenter() * UIParent:GetEffectiveScale();

	-- get mouse angle (scale = 0.069 degree per xmouse)
	local amouse = angle - xmouse * 0.069;

	-- set AMC_PosX, AMC_PosY, and AMC_MapName
	FAMC_GetMapInfos();

	-- save actual player position
	local Xpos = AMC_PosX;
	local Ypos = AMC_PosY;

	-- get pre-defined field size of this map
	local size = AMC_ShopSize[AMC_MapName] / 2.0;

	-- scan  field IDs in direction of the mouse pointer for a sign (up to 4x size)
	local oldtext = "";
	local found;
	for i = 0, 20 do
		if ( not found ) then
			AMC_PosX = math.floor((Xpos - math.sin(math.rad(amouse)) * size / 5.0 * i) * 10.0 + 0.5) / 10.0;
			AMC_PosY = math.floor((Ypos - math.cos(math.rad(amouse)) * size / 5.0 * i) * 10.0 + 0.5) / 10.0;
			local signID = AMC_MapName..FAMC_GetPositionID(size);
			-- change sign text
			if ( AtlasMajorCities_Shop[signID] ) then
				oldtext = AtlasMajorCities_Shop[signID];
				AtlasMajorCities_Shop[signID] = SignText;
				found = signID;
			end
		end
	end

	-- show status msg
	if ( found ) then
		local msg = "#"..found.." - "..SignText.." (old: "..oldtext..") - "..AMC_PosX..", "..AMC_PosY;
		DEFAULT_CHAT_FRAME:AddMessage(msg, .9, .0, .9);
	else
		local error = AtlasMajorCities_loc["errNoShopSign"];
		DEFAULT_CHAT_FRAME:AddMessage(error, .9, .0, .0);
	end
end

-- define a scan interval
local AMC_UpdateInterval = 1.0;
local AMC_TimeSinceLastUpdate = 0;

-- scan for NPCs and shop signs
function AtlasMajorCities_OnUpdate(self, elapsed)
	if ( not AtlasMajorCities_VariablesLoaded ) then return; end
	if ( AMC_ScanMode < 2 ) then return; end
	if ( not AMC_ShopList and (AMC_ScanMode < 4) ) then return; end

	-- fix the camera to the player
	if ( AMC_ScanMode < 5 ) then SetView(1); end

	AMC_TimeSinceLastUpdate = AMC_TimeSinceLastUpdate + elapsed;
	if (AMC_TimeSinceLastUpdate < AMC_UpdateInterval) then return; end

	-- scan shop sign
	if ( IsControlKeyDown() and IsShiftKeyDown() and (AMC_ScanMode < 5) ) then
		local SignToolTip = AtlasMajorCities_CheckSignToolTip();
		if ( SignToolTip ) then
			local SignText = getglobal("GameTooltipTextLeft1"):GetText();
			if ( SignText ~= AMC_ToolTipSign ) then
				AMC_ToolTipSign = SignText;
				-- set shop sign if there is one
				FAMC_SetShopSign(SignText);
			end
		end
	else
		AMC_ToolTipSign = "";
	end

	-- scan NPC
	if ( TargetFrame:IsVisible() and MouseIsOver(TargetFrame) and GameTooltip:IsVisible() ) then
		-- check the type of target
		local guid = UnitGUID("target");
		local typ = tonumber(guid:sub(5,5), 16) % 8;
		if ( (typ == 3) and (GameTooltip:NumLines() >= 2) ) then
			-- get NPC name and titlefrom tooltip of NPC target
			local text = FAMC_GetNPCtext();
			if ( text ~= AMC_ToolTipText ) then
				AMC_ToolTipText = text;
				-- get the NPC ID
				local npcid = tonumber(string.sub(guid,9,12), 16);
				-- add NPC to shop list
				FAMC_SetNPC(text, npcid);
			end
		end
	end

	AMC_TimeSinceLastUpdate = 0;
end
