-- assign internal city names to old DB city entries
local AMC_myDataKeys = {
	["TheExodar"]      = "EX",
	["Darnassis"]      = "DN",
	["Ironforge"]      = "IF",
	["Stormwind"]      = "SW",
	["SilvermoonCity"] = "SM",
	["Undercity"]      = "UC",
	["ThunderBluff"]   = "TB",
	["Ogrimmar"]       = "OG",
	["ShattrathCity"]  = "SR",
	["Dalaran1_"]      = "DL",
	["Dalaran2_"]      = "DLS",
}

-- the left and right skip together must be 33.4
local AMC_XMapCoordSkip = {
	["TheExodar1"] = 19.7,
	["TheExodar2"] = 13.7,
	["Darnassis1"] = 14.7,
	["Darnassis2"] = 18.7,
	["Ironforge1"] = 14.4,
	["Ironforge2"] = 19.0,
	["Stormwind1"] = 22.0,
	["Stormwind2"] = 11.4,
	["SilvermoonCity1"] = 33.4,
	["SilvermoonCity2"] = 0.0,
	["Undercity1"] = 33.4,
	["Undercity2"] = 0.0,
	["ThunderBluff1"] = 17.4,
	["ThunderBluff2"] = 16.0,
	["Ogrimmar1"] = 16.9,
	["Ogrimmar2"] = 16.5,
	["ShattrathCity1"] = 15.0,
	["ShattrathCity2"] = 18.4,
	["Dalaran1_1"] = 12.4,
	["Dalaran1_2"] = 21.0,
	["Dalaran2_1"] = 10.4,
	["Dalaran2_2"] = 23.0,
}

-- define the colors assigned in AtlasMajorCities_zones.lua
local AMC_Colors = {
	["WHITE"]  = "|cffffffff",
	["GREY"]   = "|cff999999",
	["BLACK"]  = "|c00000000",
	["BLUE"]   = "|cff3333ff",
	["GREEN"]  = "|cff00cc00",
	["RED"]    = "|cffcc0000",
	["YELLOW"] = "|cffcccc00",
	["CYAN"]   = "|cff00bbbb",
	["PINK"]   = "|cffcc00cc",
	["ORANGE"] = "|cffbb6600",
	["PURPLE"] = "|cff9900ff",
}

-- flag for addon database (db0) or user database (db)
AtlasMajorCities_UserDB = false;

-- flag at which position the labels are shown (-1 - moving labels ; 0 - label pos. ; 1 - shop pos. ; 2 - shop sign pos.)
AtlasMajorCities_LabelAtPos = 0;

-- flag that AtlasRefresh() was called from AtlasMajorCities_InsertData()
-- -> prevent endless loop with nested call from AtlasMajorCities_Atlas_Refresh_Hook()
local AMC_InsertData_Run = false;

-- collect all actual map labels
AtlasMajorCities_LabelPos = {};

-- save hook to the Atlas function Atlas_Refresh()
local AMC_Atlas_Refresh_Orig = nil;

-- global identifiers
AtlasMajorCities_VariablesLoaded = false;

-- key bindings
BINDING_HEADER_AMC_TITLE = "AtlasMajorCities Enhanced";
BINDING_NAME_AMC_ZONE    = AtlasMajorCities_loc["BIND_ZONE"];
BINDING_NAME_AMC_CREATE  = AtlasMajorCities_loc["BIND_CREATE"];
BINDING_NAME_AMC_ASSIGN  = AtlasMajorCities_loc["BIND_ASSIGN"];
BINDING_NAME_AMC_LABEL   = AtlasMajorCities_loc["BIND_LABEL"];
BINDING_NAME_AMC_SIGN    = AtlasMajorCities_loc["BIND_SIGN"];
BINDING_NAME_AMC_TITLE   = AtlasMajorCities_loc["BIND_TITLE"];
BINDING_NAME_AMC_COMMENT = AtlasMajorCities_loc["BIND_COMMENT"];

-- insert a line break if line is longer than maxstrlen
local function FAMC_ListFormat(myData)
	local maxstrlen = 45;
	-- city
	for Key1, Val1 in pairs(myData) do
		if ( Key1 ~= "Search" ) then
			-- zone
			for Key2, Val2 in pairs(myData[Key1]["Areas"]) do
				-- shop
				for Key3, Val3 in pairs(myData[Key1]["Areas"][Key2]) do
					if ( (Key3 ~= "Name") and (Key3 ~= "Color") ) then
						local tabpos = 0;
						-- NPCs and shop title
						for Key4, Val4 in pairs(myData[Key1]["Areas"][Key2][Key3]) do
							tabpos = tabpos + 1;
							if ( Key4 ~= "Label" ) then
								local list = myData[Key1]["Areas"][Key2][Key3];
								local length = string.len(Val4);
								-- split up long lines and insert rest as new indented line
								if ( length > maxstrlen ) then
									local index = maxstrlen - string.find(string.reverse(string.sub(Val4, 1, maxstrlen))," ");
									list[Key4] = string.sub(Val4, 1, index);
									Val4 = "    "..string.sub(Val4, index+2);
									table.insert(list, tabpos+1, Val4);
								end
							end
						end
					end
				end
			end
		end
	end
end

-- create a copy of AtlasMajorCities_DB to insert in AtlasMajorCities_myData
function AtlasMajorCities_InsertData(init)
	-- ATTENTION: Atlas_Refresh() is called by AtlasMajorCities_InsertData()
	--            -> AMC_InsertData_Run prevent endless loop
	AMC_InsertData_Run = true;

	local init = ( not AtlasMajorCities_UserDB );
	local myData = {};

	-- reset positions of all map labels
	AtlasMajorCities_LabelPos = {};

	-- assign city list
	local database;
	if ( init ) then database = AtlasMajorCities_DB0;
		    else database = AtlasMajorCities_DB; end
	for lkey, lval in pairs(database) do
		local list = {};
		local city = lkey;
		if ( init ) then list.ZoneName = { AtlasMajorCities_City0[city] };
			    else list.ZoneName = { AtlasMajorCities_City[city] }; end
		if ( init ) then list.Location = { AtlasMajorCities_Continent0[city] };
			    else list.Location = { AtlasMajorCities_Continent[city] }; end
		list.Areas = {};
		local oalist = list["Areas"];
		local ialist = database[city]["Areas"];

		-- assign zone list
		for akey, aval in pairs(ialist) do
			local ozlist = {};
			local izlist = ialist[akey];
			local zone = izlist.Name;
			if ( init ) then ozlist.Name  = "=- "..AtlasMajorCities_Zone0[city.."-"..zone].." -=";
				    else ozlist.Name  = "=- "..AtlasMajorCities_Zone[city.."-"..zone].." -="; end
			ozlist.Color = AMC_Colors[izlist.Color];

			-- assign shop list
			for zkey, zval in pairs(izlist) do
				if ( type(zkey) == "number" ) then
					local oslist = {};
					local islist = izlist[zkey];
					local shop = islist.ID;

					-- add label
					if ( islist.Label ) then
						oslist.Label = islist.Label;
						-- add label with position to map overlay table
						local xpos, ypos;
						if ( AtlasMajorCities_LabelAtPos <= 0 ) then
							xpos = islist.PosX;
							ypos = islist.PosY;
						elseif ( (AtlasMajorCities_LabelAtPos == 1) and (shop ~= "000000") ) then
							xpos = tonumber(string.sub(shop,1,3))/10.0;
							ypos = tonumber(string.sub(shop,4,6))/10.0;
						elseif ( (AtlasMajorCities_LabelAtPos == 2) and islist.sID ) then
							local sid = islist.sID;
							sid = string.sub(sid,string.len(sid)-5);
							xpos = tonumber(string.sub(sid,1,3))/10.0;
							ypos = tonumber(string.sub(sid,4,6))/10.0;
						end
						if ( xpos and ypos ) then
							local tab = {l=ozlist.Color..oslist.Label, c=city, x=xpos, y=ypos, id=shop};
							table.insert(AtlasMajorCities_LabelPos, tab);
						end
					else
						oslist.Label = "-";
					end

					-- insert shop title
					local sign, title;
					if ( init ) then sign = AtlasMajorCities_Shop0[islist.sID];
						    else sign = AtlasMajorCities_Shop[islist.sID]; end
					if ( init ) then title = AtlasMajorCities_Title0[city.."-"..zone.."-"..shop];
						    else title = AtlasMajorCities_Title[city.."-"..zone.."-"..shop]; end
					if ( sign and title ) then title = "* "..sign.." ."..title..".*";
					elseif ( sign )       then title = "* "..sign.." *";
					elseif ( title )      then title = "*."..title..".*";
					end
					if ( title ) then table.insert(oslist, title); end

					-- assign NPCs
					for skey, sval in pairs(islist) do
						if ( type(skey) == "number" ) then
							local entry;
							local sval0 = "NPC"..tostring(sval);
							if ( init ) then entry = AtlasMajorCities_Comment0[sval0];
								    else entry = AtlasMajorCities_Comment[sval0]; end
							if ( entry ) then entry = " ("..entry..")"; else entry = ""; end
							if ( init ) then entry = AtlasMajorCities_NPC0[sval0]..entry;
								    else entry = AtlasMajorCities_NPC[sval0]..entry; end
							table.insert(oslist, entry);
						end
					end
					if ( table.getn(oslist) > 0 ) then table.insert(ozlist, oslist); end
				end
			end
			table.insert(oalist, ozlist);
		end

		-- check if city is in original AtlasMajorCities_myData list
		local myDataKey = AMC_myDataKeys[city];
		if ( AtlasMajorCities_myData[myDataKey] ) then myData[myDataKey] = list; end
	end

	-- insert a line breaks
	FAMC_ListFormat(myData);

	-- load city data like Atlas_RegisterPlugin
	for k,v in pairs(myData) do
		AtlasMaps[k] = v;
	end
	Atlas_PopulateDropdowns();
	Atlas_Refresh();
end

-- load the DB, format it, and register with Atlas
local function FAMC_OnLoad(self)
	SlashCmdList["AtlasMajorCitiesHelp"] = AtlasMajorCities_SetScan;
	SLASH_AtlasMajorCitiesHelp1 = "/amc";

	SlashCmdList["AtlasMajorCitiesZone"] = AtlasMajorCities_SetZone;
	SLASH_AtlasMajorCitiesZone1 = "/amcz";

	SlashCmdList["AtlasMajorCitiesPosition"] = AtlasMajorCities_SetPosition;
	SLASH_AtlasMajorCitiesPosition1 = "/amcp";

	SlashCmdList["AtlasMajorCitiesAssign"] = AtlasMajorCities_AssignShop;
	SLASH_AtlasMajorCitiesAssign1 = "/amca";

	SlashCmdList["AtlasMajorCitiesLabel"] = AtlasMajorCities_SetLabel;
	SLASH_AtlasMajorCitiesLabel1 = "/amcl";

	SlashCmdList["AtlasMajorCitiesScan"] = AtlasMajorCities_SetSign;
	SLASH_AtlasMajorCitiesScan1 = "/amcs";

	SlashCmdList["AtlasMajorCitiesTitle"] = AtlasMajorCities_SetTitle;
	SLASH_AtlasMajorCitiesTitle1 = "/amct";

	SlashCmdList["AtlasMajorCitiesComment"] = AtlasMajorCities_SetComment;
	SLASH_AtlasMajorCitiesComment1 = "/amcc";

	-- add basic DB to Atlas
	Atlas_RegisterPlugin("AtlasMajorCitiesEnhanced", AtlasMajorCities_myCategory, AtlasMajorCities_myData);

	-- hook to the Atlas function Atlas_Refresh() to set the map labels
	AMC_Atlas_Refresh_Orig = Atlas_Refresh;
	Atlas_Refresh = AtlasMajorCities_Atlas_Refresh_Hook;

	-- load whole db
	AtlasMajorCities_InsertData();

	-- show status msg if addon loaded
	local info = ChatTypeInfo["SYSTEM"];
	if ( DEFAULT_CHAT_FRAME ) then
		local msg = "AtlasMajorCities loaded";
		DEFAULT_CHAT_FRAME:AddMessage(msg, info.r, info.g, info.b, info.id);
	end

	AtlasMajorCities_VariablesLoaded = true;
end

-- must use event VARIABLES_LOADED, because AtlasLoot replace Atlas_Refresh()
-- because AtlasLoot load later, event ADDON_LOADED is to early
local function FAMC_OnEvent(self, event)
	if ( (event == "VARIABLES_LOADED") ) then
		self:UnregisterEvent("VARIABLES_LOADED");
		self:SetScript("OnUpdate", AtlasMajorCities_OnUpdate);
		FAMC_OnLoad(self);
	end
end

local AtlasMajorCities_Frame = CreateFrame("Frame","AtlasMajorCities_Frame");
AtlasMajorCities_Frame:RegisterEvent("VARIABLES_LOADED");
AtlasMajorCities_Frame:SetScript("OnEvent", FAMC_OnEvent);

-- upend some functionality to the Atlas-function Atlas_Refresh()
local AMC_NumFrames = 0;
local AMC_ShownCity = "";
function AtlasMajorCities_Atlas_Refresh_Hook()
	-- call original function first
	AMC_Atlas_Refresh_Orig();
	local zoneID = ATLAS_DROPDOWNS[AtlasOptions.AtlasType][AtlasOptions.AtlasZone];

	local city;
	for key, val in pairs(AMC_myDataKeys) do
		if ( val == zoneID ) then city = key; end
	end

	-- refresh display of labels (maybe the labels are moved)
	-- ATTENTION: Atlas_Refresh() is called by AtlasMajorCities_InsertData()
	--            -> AMC_InsertData_Run prevent endless loop
	if ( city and AtlasMajorCities_UserDB and not AMC_InsertData_Run ) then
		AtlasMajorCities_InsertData();
		return;
	end
	AMC_InsertData_Run = false;

	-- use internal maps -> empty provided map
	if ( city ) then
		AtlasMap:SetTexture("");

		-- tiles size scaled to 512 pixel map hight (AtlasMap size) -> 769 pixel width of whole map
		-- original size: 256x256 (233x155) = 1001x667 -> 197x197 (178x118) = 769x512
		local tsize = 197;
		-- tile size with removed right border
		local tsizex = 178;
		-- tile size with removed lower border
		local tsizey = 118;

		-- number of pixel to skip in x direction
		local xskip1 = math.floor(AMC_XMapCoordSkip[city.."1"] / 100.0 * 769 + 0.5);
		local xskip2 = math.floor(AMC_XMapCoordSkip[city.."2"] / 100.0 * 769 + 0.5);

		for idx = 1, 12 do
			local tex = getglobal("AMC_Atlas_Map_"..idx);
			if ( not tex ) then tex = AtlasFrame:CreateTexture("AMC_Atlas_Map_"..idx,"BACKGROUND"); end

			-- set the tile size, include xskip
			if ( idx == 1 or idx == 5 or idx == 9 ) then
				if ( tsize < xskip1 ) then tex:SetWidth(1);
						      else tex:SetWidth(tsize - xskip1);
				end
			end
			if ( idx == 2 or idx == 6 or idx == 10 ) then
				if ( tsize < xskip1 ) then tex:SetWidth(tsize + tsize - xskip1);
						      else tex:SetWidth(tsize);
				end
			end
			if ( idx == 3 or idx == 7 or idx == 11 ) then
				if ( tsizex < xskip2 ) then tex:SetWidth(tsize + tsizex - xskip2);
						       else tex:SetWidth(tsize);
				end
			end
			if ( idx == 4 or idx == 8 or idx == 12 ) then
				if ( tsizex < xskip2 ) then tex:SetWidth(1);
						       else tex:SetWidth(tsizex - xskip2);
				end
			end
			if ( idx > 8 ) then tex:SetHeight(tsizey);
				       else tex:SetHeight(tsize);
			end

			-- set position of tile, include xskip
			local int, flt = math.modf((idx - 0.97) / 4.0); flt = math.floor(flt * 4.0 + 0.5)
			xpos = flt * tsize - xskip1;
			if ( xpos < 0 ) then xpos = 0; end
			ypos = int * tsize;
			tex:SetPoint("TOPLEFT", "AtlasFrame", "TOPLEFT", 18 + xpos, -84 - ypos);

			-- set region for tiles, include xskip and border
			local xstart = 0.0;
			local xend = 1.0;
			local yend = 1.0;
			if ( idx == 1 or idx == 5 or idx == 9 ) then
				if ( tsize < xskip1 ) then xstart = 0.0;
						      else xstart = xskip1 / tsize;
				end
			end
			if ( idx == 2 or idx == 6 or idx == 10 ) then
				if ( tsize < xskip1 ) then xstart = (xskip1 - tsize) / tsize;
						      else xstart = 0.0;
				end
			end
			if ( idx == 3 or idx == 7 or idx == 11 ) then
				if ( tsizex < xskip2 ) then xend = (tsize + tsizex - xskip2) / tsize;
						       else xend = 1.0;
				end
			end
			if ( idx == 4 or idx == 8 or idx == 12 ) then
				if ( tsizex < xskip2 ) then xend = 0.0;
						       else xend = (tsizex - xskip2) / tsize;
				end
			end
			if ( idx > 8 ) then yend = tsizey / tsize; end
			tex:SetTexCoord(xstart, xend, 0.0, yend);

			-- insert image
			local length = string.len(city);
			if ( string.sub(city,length) == "_" ) then
				city0 = string.sub(city,1,length-2);
			else
				city0 = city;
			end
			tex:SetTexture("Interface\\WorldMap\\"..city0.."\\"..city..idx);
		end
	else
		-- clear internal maps -> use provided map
		for idx = 1, 12 do
			local tex = getglobal("AMC_Atlas_Map_"..idx);
			if ( tex ) then tex:ClearAllPoints(); tex:SetTexture(""); end
		end
	end

	-- remove labels and images of old map
	if ( AMC_NumFrames > 0 ) then
		local maplabel;
		for idx = 1, AMC_NumFrames do
			maplabel = getglobal("AMC_Atlas_Frame"..idx);
			maplabel:ClearAllPoints();

			maplabel = getglobal("AMC_Atlas_Image"..idx);
			if ( maplabel ) then maplabel:SetTexture(""); end

			maplabel = getglobal("AMC_Atlas_Label"..idx);
			if ( maplabel ) then maplabel:SetText(""); end
		end
		AMC_NumFrames = 0;
		AMC_ShownCity = "";
	end

	if ( not city ) then return; end

	-- set labels for actual map
	for key, val in pairs(AtlasMajorCities_LabelPos) do
		if ( zoneID == AMC_myDataKeys[val.c] ) then
			AMC_ShownCity = val.c;
			local label = val.l;
			AMC_NumFrames = AMC_NumFrames + 1;

			-- get or create next font or image object at the map
			local mapframe = getglobal("AMC_Atlas_Frame"..AMC_NumFrames);
			if ( not mapframe ) then
				mapframe = CreateFrame("FRAME", "AMC_Atlas_Frame"..AMC_NumFrames, AtlasFrame);
			end
			local maplabel;
			if ( string.sub(label,11) == "@" ) then  -- (the label start behind the color string)
				maplabel = getglobal("AMC_Atlas_Image"..AMC_NumFrames);
				if ( not maplabel ) then
					maplabel = mapframe:CreateTexture("AMC_Atlas_Image"..AMC_NumFrames, "LOW")
					maplabel:SetWidth(15);
					maplabel:SetHeight(15);
					maplabel:SetPoint("CENTER", mapframe, "CENTER", 0, 0);
				end
			else
				maplabel = getglobal("AMC_Atlas_Label"..AMC_NumFrames);
				if ( not maplabel ) then
					maplabel = mapframe:CreateFontString("AMC_Atlas_Label"..AMC_NumFrames, "MEDIUM", "SystemFont_Outline");
					maplabel:SetPoint("CENTER", mapframe, "CENTER", 0, 0);
				end
			end

			-- compute and set position of frame object at the map
			local x1 = AMC_XMapCoordSkip[AMC_ShownCity.."1"];
			local x2 = AMC_XMapCoordSkip[AMC_ShownCity.."2"];
			local xpos = (val.x - x1) / (100.0 - x1 - x2);
			local ypos = val.y / 100.0;
			local wmap = AtlasMap:GetWidth();
			local hmap = AtlasMap:GetHeight();
			local xoff = math.floor(wmap*xpos+0.5);
			local yoff = math.floor(hmap*(-ypos)+0.5);
			mapframe:SetPoint("CENTER", AtlasMap, "TOPLEFT", xoff, yoff);

			-- set letter box icon or label text and frame size
			if ( string.sub(label,11) == "@" ) then  -- (the label start behind the color string)
				maplabel:SetTexture("Interface\\Minimap\\Tracking\\Mailbox");
				mapframe:SetWidth(15);
				mapframe:SetHeight(15);
			else
				maplabel:SetText(label);
				mapframe:SetWidth(maplabel:GetWidth());
				mapframe:SetHeight(maplabel:GetHeight());

				mapframe:SetID(tonumber(val.id));
				mapframe:EnableMouse(true);
				mapframe:SetScript("OnEnter",AtlasMajorCities_Label_ShowTT);
				mapframe:SetScript("OnLeave",AtlasMajorCities_Label_HideTT);

				if ( AtlasMajorCities_LabelAtPos == -1 ) then
					mapframe:SetMovable(true);
					mapframe:SetScript("OnMouseDown", function(self,button)
						if ( (button == "LeftButton") and not self.isMoving ) then
							local x, y;
							_, _, _, x, y = self:GetPoint(1);
							self.x1 = math.floor(x+0.5); self.y1 = math.floor(y+0.5);
							self:StartMoving();
							_, _, _, x, y = self:GetPoint(1);
							self.x2 = math.floor(x+0.5); self.y2 = math.floor(y+0.5);
							self.isMoving = true;
						end
					end)
					mapframe:SetScript("OnMouseUp", function(self,button)
						if ( (button == "LeftButton") and self.isMoving ) then
							local x, y;
							_, _, _, x, y = self:GetPoint(1);
							self.x3 = math.floor(x+0.5); self.y3 = math.floor(y+0.5);
							self:StopMovingOrSizing();
							self:SetUserPlaced(false);
							self.isMoving = false;
							AtlasMajorCities_Label_SetNewPos(self);
							self:ClearAllPoints();
							self:SetPoint("CENTER", AtlasMap, "TOPLEFT", self.x1, self.y1);
						end
					end)
					mapframe:SetScript("OnHide", function(self,button)
						if ( self.isMoving ) then
							self:StopMovingOrSizing();
							self:SetUserPlaced(false);
							self.isMoving = false;
							self:ClearAllPoints();
							self:SetPoint("CENTER", AtlasMap, "TOPLEFT", self.x1, self.y1);
						end
					end)
				else
					mapframe:SetMovable(false);
					mapframe:SetScript("OnMouseDown",nil);
					mapframe:SetScript("OnMouseUp",nil);
					mapframe:SetScript("OnHide",nil)
				end
			end
		end
	end
end

-- write the new label position to the UserDB
function AtlasMajorCities_Label_SetNewPos(self)
	-- work only for UserDB
	if ( not AtlasMajorCities_UserDB or (AtlasMajorCities_LabelAtPos ~= -1) ) then return; end

	-- inside the image ?
	local newx = self.x1+self.x3-self.x2;
	local newy = self.y1+self.y3-self.y2;
	if (newx<0 or newx>512 or newy>0 or newy<-512) then return; end

	local ID = self:GetID();

	-- find the right database entry
	for lkey, lval in pairs(AtlasMajorCities_DB) do
		if ( lkey == AMC_ShownCity ) then
			local city = lkey;
			local ialist = AtlasMajorCities_DB[city]["Areas"];
			-- get zone list
			for akey, aval in pairs(ialist) do
				local izlist = ialist[akey];
				-- get shop list
				for zkey, zval in pairs(izlist) do
					if ( type(zkey) == "number" ) then
						local islist = izlist[zkey];
						local shop = islist.ID;

						-- check if we found the right shop by its ID
						if ( ID == tonumber(shop) ) then
							-- shop position
							local sx = tonumber(string.sub(shop,1,3))/10.0;
							local sy = tonumber(string.sub(shop,4,6))/10.0;
							-- new label position
							local x1 = AMC_XMapCoordSkip[AMC_ShownCity.."1"];
							local x2 = AMC_XMapCoordSkip[AMC_ShownCity.."2"];
							local wmap = AtlasMap:GetWidth();
							local hmap = AtlasMap:GetHeight();
							local px = math.floor((newx / wmap * (100.0 - x1 - x2) + x1) * 10.0) / 10.0;
							local py = math.floor((-newy / hmap * 100.0) * 10.0) / 10.0;
							-- allow label a max. distance of 5% of map size to its shop position
							-- (consider the different scale of the x-axis -> 1/3 larger then y-axis)
							if ( (math.abs(sx-px) < 5.0/0.66) and (math.abs(sy-py) < 5.0) ) then
								islist.PosX = px;
								islist.PosY = py;
								self.x1 = newx;
								self.y1 = newy;
							end
							return;
						end
					end
				end
			end
		end
	end
end

function AtlasMajorCities_Label_ShowTT(self)
	local ID = self:GetID();
	local init = ( not AtlasMajorCities_UserDB );

	-- find the right database entry
	local database;
	if ( init ) then database = AtlasMajorCities_DB0;
		    else database = AtlasMajorCities_DB; end
	for lkey, lval in pairs(database) do
		if ( lkey == AMC_ShownCity ) then
			local city = lkey;
			local ialist = database[city]["Areas"];

			-- get zone list
			for akey, aval in pairs(ialist) do
				local izlist = ialist[akey];
				local TT_zone;
				local zone = izlist.Name;
				if ( init ) then TT_zone  = "=- "..AtlasMajorCities_Zone0[city.."-"..zone].." -=";
					    else TT_zone  = "=- "..AtlasMajorCities_Zone[city.."-"..zone].." -="; end
				local TT_color = AMC_Colors[izlist.Color];

				-- get shop list
				for zkey, zval in pairs(izlist) do
					if ( type(zkey) == "number" ) then
						local islist = izlist[zkey];
						local shop = islist.ID;

						-- check if we found the right shop be its ID
						if ( ID == tonumber(shop) ) then
							local showTT = false;

							-- get shop title
							local TT_shop;
							local sign, title;
							if ( init ) then sign = AtlasMajorCities_Shop0[islist.sID];
								    else sign = AtlasMajorCities_Shop[islist.sID]; end
							if ( init ) then title = AtlasMajorCities_Title0[city.."-"..zone.."-"..shop];
								    else title = AtlasMajorCities_Title[city.."-"..zone.."-"..shop]; end
							if ( sign and title ) then TT_shop = "* "..sign.." ."..title..".*";
							elseif ( sign )       then TT_shop = "* "..sign.." *";
							elseif ( title )      then TT_shop = "*."..title..".*";
							end
							if ( TT_shop ) then
								showTT = true;
								GameTooltip:SetOwner(self, "ANCHOR_BOTTOMRIGHT");
								if ( TT_zone ) then GameTooltip:SetText(TT_color..TT_zone); end
								GameTooltip:AddLine(TT_color..TT_shop);
							end

							-- get NPCs
							for skey, sval in pairs(islist) do
								if ( type(skey) == "number" ) then
									local TT_npc;
									local sval0 = "NPC"..tostring(sval);
									if ( init ) then TT_npc = AtlasMajorCities_Comment0[sval0];
										    else TT_npc = AtlasMajorCities_Comment[sval0]; end
									if ( TT_npc ) then TT_npc = " ("..TT_npc..")"; else TT_npc = ""; end
									if ( init ) then TT_npc = AtlasMajorCities_NPC0[sval0]..TT_npc;
										    else TT_npc = AtlasMajorCities_NPC[sval0]..TT_npc; end
									if ( not showTT ) then
										showTT = true;
										GameTooltip:SetOwner(self, "ANCHOR_BOTTOMRIGHT");
										if ( TT_zone ) then GameTooltip:SetText(TT_color..TT_zone); end
									end
									GameTooltip:AddLine(TT_color..TT_npc);
								end
							end

							if ( showTT ) then
								GameTooltip:SetScale(0.8);

								-- give tooltip an opaque background
								local TT_tex = getglobal("AMC_GameTooltip_BG");
								if ( not TT_tex ) then
									TT_tex = GameTooltip:CreateTexture("AMC_GameTooltip_BG", "BACKGROUND");
								end
								TT_tex:SetTexture([[Interface\ChatFrame\ChatFrameBackground]]);
								TT_tex:SetPoint("TOPLEFT",GameTooltip,"TOPLEFT",4,-5);
								TT_tex:SetPoint("BOTTOMRIGHT",GameTooltip,"BOTTOMRIGHT",-4,5);
								TT_tex:SetVertexColor(0.0, 0.0, 0.0);

								GameTooltip:Show();
							end

							return;
						end
					end
				end
			end
		end
	end
end

function AtlasMajorCities_Label_HideTT(self)
	GameTooltip:Hide();
	local TT_tex = getglobal("AMC_GameTooltip_BG");
	if ( TT_tex ) then
		TT_tex:SetTexture(nil);
		TT_tex:ClearAllPoints();
	end
	GameTooltip:SetScale(1.0);
end
