--------------------------------------------------
-- ArekosGratz, Author: Areko                   --
--------------------------------------------------
local ADDONNAME, ArekosGratz = ...

LibStub("AceAddon-3.0"):NewAddon(ArekosGratz, ADDONNAME, "AceConsole-3.0", "AceEvent-3.0", "AceTimer-3.0");

local L = LibStub("AceLocale-3.0"):GetLocale("ArekosGratz")

-- GLOBALS: LibStub
local _G = _G

local string, strjoin, tostring = string, strjoin, tostring
local type, select, gsub, random, bit = type, select, gsub, random, bit
local tinsert, pairs, ipairs, unpack, next, setmetatable, wipe, rawget, rawset = tinsert, pairs, ipairs, unpack, next, setmetatable, wipe, rawget, rawset

local lower = string.utf8lower or string.lower

local NEAR_TYPES = {
    NONE    = 0,    -- L["NONE"]    = "None"
    SAY     = 1,    -- L["SAY"]     = "Say"
    WHISPER = 2,    -- L["WHISPER"] = "Whisper"
}
local NEAR_TYPES_VALUES = {}
for name, value in pairs(NEAR_TYPES) do
    NEAR_TYPES_VALUES[value] = L[name]
end
local MESSAGE_FLAGS = {
    NONE        = 0,
    NEAR        = 1,
    GROUP       = 2,
    GUILD       = 4,
    INSTANCE    = 8,
}

local messageDefaults = {
    exists = false,
    isIntern = false,
    removeRealmName = false,
    capitalizeFirstChar = false,
    onlyFor = {
        -- [#] = "name-realm" -> Playername
        -- [#] = "Name#1234" -> Battle.net Tag
    },
    notFor = {
        -- same as onlyFor
    },
}
local defaults = {
    global = {
        gzTriggersImported = false,
        gzTriggers = {
            gz = true,
            gw = true,
            gg = true,
            gzi = true,
            grats = true,
            gratz = true,
            salute = true,
            congrat = true,
            goncrats = true,
            dingrats = true,
            gradulate = true,
            condolence = true,
            coragulate = true,
            congrations = true,
            gratulation = true,
            congrazzles = true,
            congratsory = true,
            ["glückwunsch"] = true,
            felicitation = true,
            concragulate = true,
            congradulate = true,
            comendations = true,
            congraduation = true,
            confagulations = true,
            congratulation = true,
            congratulating = true,
            congratulations = true,
            concrapulations = true,
        },
        scheduleTimer = {
            ['*'] = 0.5,
            sendMessageDelay = 0.2
        },
    },
    profile = {
        protectAFK = true,
        congratulations = {
            guild = false,
            party = false,
            ignoreInstanceGroup = false,
            near = NEAR_TYPES.NONE,
            guildInOtherChannel = false,
            minWaitTime = 3,
            maxWaitTime = 5,
        },
        thanks = {
            guild = false,
            party = false,
            instance = false,
            near = NEAR_TYPES.NONE,
            minWaitTime = 15,
            maxWaitTime = 20,
            maxExtraThanksTrys = 1,
        },
        gratzKeywords = nil,
        thxKeywords = nil,
        keywordsImported = false,
        messageTHX = {
            ['**'] = messageDefaults,
        },
        messageGZ = {
            ['**'] = messageDefaults,
        },
    }   
}

local get, del
do -- TablePool
    local tablePool = setmetatable({},{__mode="k"})
    get = function()
        local t = next(tablePool)
        if t then
            tablePool[t] = nil
            return t
        else
            return {}
        end
    end
    del = function(t)
        setmetatable(t, nil)    -- removes metatable
        wipe(t)
        tablePool[t] = true
    end
end

local HOMEREALM = _G.GetRealmName():gsub("%s+", "")
ArekosGratz.thanksPendingList = {}
ArekosGratz.gratzPendingList = {}

function ArekosGratz:OnInitialize()
    self.db = LibStub("AceDB-3.0"):New("ArekosGratzDB", defaults, true)

    self.db.RegisterCallback(self, "OnProfileChanged", "RefreshDB")
    self.db.RegisterCallback(self, "OnDatabaseShutdown", "RefreshDB")

    if not self.db.global.gzTriggersImported then
        local gratzList = {
            global = {
                triggers = nil,
            },
        }
        local gratzDB = LibStub("AceDB-3.0"):New("GratzDB", gratzList, true)
        if type(gratzDB.global.triggers) == "table" then
            for _, trigger in pairs(gratzDB.global.triggers) do
                self.db.global.gzTriggers[ lower(trigger) ] = true
            end
        end
        self.db.global.gzTriggersImported = true
    end

    -- We call this for import of old settings on profile level
    self:RefreshDB("OnProfileChanged", self.db, self.db:GetCurrentProfile() )

    self:RegisterChatCommand("arekosgratz", "ChatCommand")
    self:RegisterChatCommand("ag", "ChatCommand")

    self:RegisterConfig()

    self.playername = _G.UnitName("player").."-".._G.GetRealmName():gsub("%s", "")
    self.playernameLower = lower(self.playername)
    self:GuildUpdate()    
end

function ArekosGratz:RefreshDB(event, db, profile)
    if event == "OnProfileChanged" then
        if not self.db.profile.keywordsImported then
            --[===[@debug@
            self:Print("Start import of old settings: ", profile or self.db:GetCurrentProfile() )
            --@end-debug@]===]
            if type(self.db.profile.gratzKeywords) == "table" then
                for key, data in pairs(self.db.profile.gratzKeywords) do
                    key = key:trim()
                    self.db.profile.messageGZ[key].exists = true
                    for id, value in ipairs(data.onlyFor) do
                        self.db.profile.messageGZ[key].onlyFor[id] = lower( self:UniformPlayerName( tostring(value) ) )
                    end
                    for id, value in ipairs(data.notFor) do
                        self.db.profile.messageGZ[key].notFor[id] = lower( self:UniformPlayerName( tostring(value) ) )
                    end
                end
            end
            if type(self.db.profile.thxKeywords) == "table" then
                for key, data in pairs(self.db.profile.thxKeywords) do
                    key = key:trim()
                    self.db.profile.messageTHX[key].exists = true
                    for id, value in ipairs(data.onlyFor) do
                        self.db.profile.messageTHX[key].onlyFor[id] = lower( self:UniformPlayerName( tostring(value) ) )
                    end
                    for id, value in ipairs(data.notFor) do
                        self.db.profile.messageTHX[key].notFor[id] = lower( self:UniformPlayerName( tostring(value) ) )
                    end
                end
            end
            self.db.profile.keywordsImported = true
        end
    elseif event == "OnDatabaseShutdown" then
        -- clean up unused triggers
        for key, inUse in pairs(self.db.global.gzTriggers) do
            if not inUse and not defaults.global.gzTriggers[key] then
                self.db.global.gzTriggers[key] = nil
            end
        end
    end
end

function ArekosGratz:OnEnable()
    self:GuildUpdate()
    self:RegisterEvent("PLAYER_GUILD_UPDATE", "GuildUpdate")
    self:RegisterEvent("GUILD_ROSTER_UPDATE", "GuildUpdate")

    self:RegisterEvent("ACHIEVEMENT_EARNED")
    self:RegisterEvent("CHAT_MSG_ACHIEVEMENT", "ACHIEVEMENT_EARNED_BY_OHTER_PLAYER")
    self:RegisterEvent("CHAT_MSG_GUILD_ACHIEVEMENT", "ACHIEVEMENT_EARNED_BY_OHTER_PLAYER")
end

function ArekosGratz:OnDisable()
end

function ArekosGratz:GuildUpdate(event)
    local guild, _, _, realm = _G.GetGuildInfo("player")
    self.guildname = (type(guild)=="string" and guild:len() > 0 ) and self:UniformGuildName(guild, realm or nil) or ""
    self.guildnameLower = lower(self.guildname)
end

function ArekosGratz:ChatCommand(input)
    local a1, lastPos = self:GetArgs(input, 1)

    if not a1 or a1:len() < 1 or a1 == "op" or a1 == "options" then
        _G.InterfaceOptionsFrame_OpenToCategory(L["Areko's Gratz"])
        _G.InterfaceOptionsFrame_OpenToCategory(L["Areko's Gratz"]) -- Note: remove this when not any more needed
    else
        if a1 == "config" then
            LibStub("AceConfigCmd-3.0").HandleCommand(self, "ag config", ADDONNAME.."OptionsMain", input:sub(lastPos) )
        else
            self:Print(L["Usage: /ag <command> or /arekosgratz <command>"])
            self:Print(L["/ag op|options    - Open Options"])
            self:Print(L["/ag config        - Change settings"])
        end
    end
end

--[[
ACHIEVEMENT_EARNED, a1=achievementid
CHAT_MSG_ACHIEVEMENT, a1=message,a2=playername
CHAT_MSG_GUILD_ACHIEVEMENT, a1=message,a2=playername,a12=GUID


CHAT_MSG_GUILD, a1=message,a2=user
CHAT_MSG_PARTY, a1=message,a2=user
CHAT_MSG_PARTY_LEADER, a1=message,a2=user
CHAT_MSG_RAID, a1=message,a2=user
CHAT_MSG_RAID_LEADER, a1=message,a2=user
CHAT_MSG_SAY, a1=message,a2=user
CHAT_MSG_WHISPER, a1=message,a2=user
]]

function ArekosGratz:IsInMyGuild(name)
    -- TODO: maybe add a better way to test this
    return _G.UnitIsInMyGuild(name)
end

function ArekosGratz:IsInMyGroup(name)
    -- TODO: check for better way
    return _G.UnitInParty(name) or _G.UnitInRaid(name)
end

function ArekosGratz:IsMy(name)
    local playername = lower( self:UniformPlayerName(name) )

    return playername == self.playernameLower
end

function ArekosGratz:IsMyGuild(guild)
    local guildname = lower( self:UniformGuildName(guild) )

    return guildname == self.guildnameLower
end

function ArekosGratz:IAmAFK()
    return _G.UnitIsAFK("player")
end

function ArekosGratz:GetGroupChannel(ignoreInstanceGroup)
    if ignoreInstanceGroup then
        return _G.IsInRaid(_G.LE_PARTY_CATEGORY_HOME) and "RAID" or _G.IsInGroup(_G.LE_PARTY_CATEGORY_HOME) and "GROUP" or ""
    else
        return _G.IsInGroup(_G.LE_PARTY_CATEGORY_INSTANCE) and "INSTANCE_CHAT" or ( _G.IsInRaid() and "RAID" or "GROUP" )
    end
end

local nearOrGroupFlag = MESSAGE_FLAGS.GROUP + MESSAGE_FLAGS.NEAR
function ArekosGratz:ACHIEVEMENT_EARNED_BY_OHTER_PLAYER(event, message, playername)
    if self:IAmAFK() and self.db.profile.protectAFK then return end
    if type(playername) ~= "string" or playername:len() < 1 then return end
    local guildname = playername
    playername = lower( self:UniformPlayerName( playername ) )
    if self:IsMy(playername) or self:IsMyGuild(guildname) then return end

    if event == "CHAT_MSG_ACHIEVEMENT" then
        if not self.db.profile.congratulations.guildInOtherChannel and self:IsInMyGuild(playername) then
            return
        end
        
        if self:IsInMyGroup(playername) then
            if self.db.profile.congratulations.party then
                self.gratzPendingList[playername] = bit.bor(MESSAGE_FLAGS.GROUP, (self.gratzPendingList[playername] or MESSAGE_FLAGS.NONE) )
                
                if self.TimeLeft(self.gratzTimerNearGroup) <= 0 then
                    self.gratzTimerNearGroup = self:ScheduleTimer("GratzForAchievement", random(self.db.profile.congratulations.minWaitTime, self.db.profile.congratulations.maxWaitTime), nearOrGroupFlag )
                end
            end
        else
            if self.db.profile.congratulations.near ~= NEAR_TYPES.NONE then
                self.gratzPendingList[playername] = bit.bor(MESSAGE_FLAGS.NEAR, (self.gratzPendingList[playername] or MESSAGE_FLAGS.NONE) )
                
                if self.TimeLeft(self.gratzTimerNearGroup) <= 0 then
                    self.gratzTimerNearGroup = self:ScheduleTimer("GratzForAchievement", random(self.db.profile.congratulations.minWaitTime, self.db.profile.congratulations.maxWaitTime), nearOrGroupFlag )
                end 
            end
        end
    elseif event == "CHAT_MSG_GUILD_ACHIEVEMENT" then
        if self.db.profile.congratulations.guild then
            self.gratzPendingList[playername] = bit.bor(MESSAGE_FLAGS.GUILD, (self.gratzPendingList[playername] or MESSAGE_FLAGS.NONE) )
            
            if self.TimeLeft(self.gratzTimerGuild) <= 0 then
                self.gratzTimerGuild = self:ScheduleTimer("GratzForAchievement", random(self.db.profile.congratulations.minWaitTime, self.db.profile.congratulations.maxWaitTime), MESSAGE_FLAGS.GUILD )
            end
        end
    end
end

function ArekosGratz:GratzForAchievement(flag)
    --[===[@debug@
    local start = _G.GetTime()
    --@end-debug@]===]
    if bit.band(flag, MESSAGE_FLAGS.GUILD) ~= 0 then
        local guild = get()
        local found = 0

        for name, flags in pairs(self.gratzPendingList) do
            if bit.band(flags, MESSAGE_FLAGS.GUILD) ~= 0 then
                guild[name] = true
                found = found + 1
                self.gratzPendingList[name] = bit.bxor(flags, MESSAGE_FLAGS.GUILD)
            end
        end

        if found > 0 then
            local messages = self:GetMessages(guild, "messageGZ", found)
            if type(messages) == "table" then
                local i = 0
                for _, message in pairs(messages) do
                    if i == 0 then
                        self:SendChatMessage(messages, "GUILD")
                    else
                        self:ScheduleTimer("SendChatMessage", self.global.scheduleTimer.sendMessageDelay*i, message, "GUILD" )
                    end
                    i = i + 1
                end
                del(messages)
            elseif type(messages) == "string" then
                self:SendChatMessage(messages, "GUILD")
            end
        end
        del(guild)
    end
    if bit.band(flag, nearOrGroupFlag) ~= 0 then
        local group = get()
        local groupFound = 0
        local near = get()
        local nearFound = 0

        for name, flags in pairs(self.gratzPendingList) do
            if bit.band(flags, MESSAGE_FLAGS.GROUP) ~= 0 then
                group[name] = true
                groupFound = groupFound + 1
                self.gratzPendingList[name] = bit.bxor(flags, MESSAGE_FLAGS.GROUP)
            end
            if bit.band(flags, MESSAGE_FLAGS.NEAR) ~= 0 then
                near[name] = true
                nearFound = nearFound + 1
                self.gratzPendingList[name] = bit.bxor(flags, MESSAGE_FLAGS.NEAR)
            end
        end

        if groupFound > 0 then
            local channel = self:GetGroupChannel(self.db.profile.congratulations.ignoreInstanceGroup)

            if type(channel) == "string" and channel:len() > 1 then
                local messages = self:GetMessages(group, "messageGZ", groupFound)

                if type(messages) == "table" then
                    local i = 0
                    for _, message in pairs(messages) do
                        if i == 0 then
                            self:SendChatMessage(messages, channel)
                        else
                            self:ScheduleTimer("SendChatMessage", self.global.scheduleTimer.sendMessageDelay*i, message, channel)
                        end
                        i = i + 1
                    end
                    del(messages)
                elseif type(messages) == "string" then
                    self:SendChatMessage(messages, channel)
                end
            end
        end
        if nearFound > 0 then
            if self.db.profile.thanks.near == NEAR_TYPES.WHISPER then
                for name in pairs(near) do
                    local nameTable = get()
                    nameTable[name] = true
                    local messages = self:GetMessages(nameTable, "messageGZ", nearFound)
                    if type(messages) == "table" then
                        local message = messages[random(1, #messages)]

                        self:SendChatMessage(message, "WHISPER", name)
                        del(messages)
                    elseif type(messages) == "string" then
                        self:SendChatMessage(messages, "WHISPER", name)
                    end
                    del(nameTable)
                end
            elseif self.db.profile.thanks.near == NEAR_TYPES.SAY then
                local messages = self:GetMessages(near, "messageGZ", nearFound)
                if type(messages) == "table" then
                    local i = 0
                    for _, message in pairs(messages) do
                        if i == 0 then
                            self:SendChatMessage(messages, "SAY")
                        else
                            self:ScheduleTimer("SendChatMessage", self.global.scheduleTimer.sendMessageDelay*i, message, "SAY")
                        end
                        i = i + 1
                    end
                    del(messages)
                elseif type(messages) == "string" then
                    self:SendChatMessage(messages, "SAY")
                end
            end
        end

        del(group)
        del(near)
    end

    if self:TimeLeft(self.gratzTimerGuild) <= 0 and self:TimeLeft(self.gratzTimerNearGroup) <= 0 then
        wipe(self.gratzPendingList)
    end
    --[===[@debug@
    self:Printf("GratzForAchievement: %f", (_G.GetTime() - start) )
    --@end-debug@]===]
end

function ArekosGratz:ACHIEVEMENT_EARNED(event, achievementID)
    if self:IAmAFK() and self.db.profile.protectAFK then return end
    local isGuildAchievement, wasEarnedByMe, earnedBy = select(12, _G.GetAchievementInfo(achievementID) )
    --[===[@debug@
    self:Printf("wasEarnedByMe: %s, isGuildAchievement: %s, earnedBy: %s", tostring(wasEarnedByMe), tostring(isGuildAchievement), tostring(earnedBy))
    --@end-debug@]===]
    if wasEarnedByMe and not isGuildAchievement then -- NOTE: test when wasEarnedByMe is true when on a twink and not first time to get the achievement
        self:ListeningForCongratulations()
    end
end

function ArekosGratz:ListeningForCongratulations()
    if self:TimeLeft(self.thanksTimer) <= 0 then
        self.thanksTimer = self:ScheduleTimer("ThanksForCongratulations", random(self.db.profile.thanks.minWaitTime, self.db.profile.thanks.maxWaitTime), 0 )
    end
    if self.db.profile.thanks.guild then
        self:RegisterEvent("CHAT_MSG_GUILD", "CheckForCongratulations", MESSAGE_FLAGS.GUILD)
    end
    if self.db.profile.thanks.party then
        self:RegisterEvent("CHAT_MSG_PARTY", "CheckForCongratulations", MESSAGE_FLAGS.GROUP)
        self:RegisterEvent("CHAT_MSG_PARTY_LEADER", "CheckForCongratulations", MESSAGE_FLAGS.GROUP)
        self:RegisterEvent("CHAT_MSG_RAID", "CheckForCongratulations", MESSAGE_FLAGS.GROUP)
        self:RegisterEvent("CHAT_MSG_RAID_LEADER", "CheckForCongratulations", MESSAGE_FLAGS.GROUP)
    end
    if self.db.profile.thanks.instance then
        self:RegisterEvent("CHAT_MSG_INSTANCE_CHAT", "CheckForCongratulations", MESSAGE_FLAGS.INSTANCE)
        self:RegisterEvent("CHAT_MSG_INSTANCE_CHAT_LEADER", "CheckForCongratulations", MESSAGE_FLAGS.INSTANCE)
    end
    if self.db.profile.thanks.near ~= NEAR_TYPES.NONE then
        self:RegisterEvent("CHAT_MSG_SAY", "CheckForCongratulations", MESSAGE_FLAGS.NEAR)
        self:RegisterEvent("CHAT_MSG_WHISPER", "CheckForCongratulations", MESSAGE_FLAGS.NEAR)
    end
end
function ArekosGratz:UnregisterCongratulationsEvents()
    self:UnregisterEvent("CHAT_MSG_GUILD")
    self:UnregisterEvent("CHAT_MSG_PARTY")
    self:UnregisterEvent("CHAT_MSG_PARTY_LEADER")
    self:UnregisterEvent("CHAT_MSG_RAID")
    self:UnregisterEvent("CHAT_MSG_RAID_LEADER")
    self:UnregisterEvent("CHAT_MSG_SAY")
    self:UnregisterEvent("CHAT_MSG_WHISPER")
    self:UnregisterEvent("CHAT_MSG_INSTANCE_CHAT")
    self:UnregisterEvent("CHAT_MSG_INSTANCE_CHAT_LEADER")
end

do
    local resultCache = setmetatable({}, {
        __mode = "k",
        __index = function(t,k)
            local found = false
            for trigger, inUse in pairs(ArekosGratz.db.global.gzTriggers) do
                if inUse then
                    if (" "..k.." "):find("%s"..trigger.."%s") then
                        found = true
                        break
                    end
                end
            end
            --[===[@debug@
            ArekosGratz:Printf("Result of %q: %s", k, tostring(found) )
            --@end-debug@]===]
            rawset(t,k,found)
            return found
        end,
    })

    --local allFlags = MESSAGE_FLAGS.NONE
    --for _, flag in pairs(MESSAGE_FLAGS) do
    --    allFlags = bit.bor(allFlags, flag)
    --end

    function ArekosGratz:CheckForCongratulations(flag, event, message, playername) -- ...)
        if type(playername) ~= "string" or playername:len() < 1 then return end
        local guildname = playername
        playername = lower( self:UniformPlayerName( playername ) )
        if self:IsMy(playername) or self:IsMyGuild(guildname) then return end
        --[===[@debug@
        self:Printf("Check player: %s", playername)
        --@end-debug@]===]
        if resultCache[ lower(message) ] then
            --[===[@debug@
            self:Printf("Add flag: %s", flag)
            --@end-debug@]===]            
            self.thanksPendingList[playername] = bit.bor( (self.thanksPendingList[playername] or MESSAGE_FLAGS.NONE), flag )
        end
    end

    --@alpha@
    function ArekosGratz:TestResultCache(text)
        return resultCache[ lower(text) ]
    end
    --@end-alpha@
end

function ArekosGratz:ThanksForCongratulations(count)
    --[===[@debug@
    local start = _G.GetTime()
    --@end-debug@]===]
    local guild = get()
    local guildFound = 0
    local group = get()
    local groupFound = 0
    local instance = get()
    local instanceFound = 0
    local near  = get()
    local nearFound = 0
    
    for name, flags in pairs(self.thanksPendingList) do
        if bit.band(flags, MESSAGE_FLAGS.GUILD) ~= 0 then
            guild[name] = true
            guildFound = guildFound + 1
        end
        if bit.band(flags, MESSAGE_FLAGS.GROUP) ~= 0 then
            group[name] = true
            groupFound = groupFound + 1
        end
        if bit.band(flags, MESSAGE_FLAGS.INSTANCE) ~= 0 then
            instance[name] = true
            instanceFound = instanceFound + 1
        end
        if bit.band(flags, MESSAGE_FLAGS.NEAR) ~= 0 then
            near[name] = true
            nearFound = nearFound + 1
        end
    end
    if ( guildFound + groupFound + nearFound + instanceFound ) == 0 and (count or 0) < self.db.profile.thanks.maxExtraThanksTrys then
        --[===[@debug@
        self:Print("Retry: ", count or 0 )
        --@end-debug@]===]
        if self:TimeLeft(self.thanksTimer) <= 0 then
            --[===[@debug@
            self:Print("Start timer")
            --@end-debug@]===]
            self.thanksTimer = self:ScheduleTimer("ThanksForCongratulations", random(self.db.profile.thanks.minWaitTime, self.db.profile.thanks.maxWaitTime), ( (count or 0) + 1 ) )
        end
        return
    else
        self:UnregisterCongratulationsEvents()
        wipe(self.thanksPendingList)
    end

    if guildFound > 0 then
        local messages = self:GetMessages(guild, "messageTHX", guildFound)
        if type(messages) == "table" then
            local i = 0
            for _, message in pairs(messages) do
                if i == 0 then
                    self:SendChatMessage(messages, "GUILD")
                else
                    self:ScheduleTimer("SendChatMessage", self.global.scheduleTimer.sendMessageDelay*i, message, "GUILD" )
                end
                i = i + 1
            end
            del(messages)
        elseif type(messages) == "string" then
            self:SendChatMessage(messages, "GUILD")
        end
    end
    if groupFound > 0 then
        local channel = self:GetGroupChannel(true)

        if type(channel) == "string" and channel:len() > 1 then
            local messages = self:GetMessages(group, "messageTHX", groupFound)

            if type(messages) == "table" then
                local i = 0
                for _, message in pairs(messages) do
                    if i == 0 then
                        self:SendChatMessage(messages, channel)
                    else
                        self:ScheduleTimer("SendChatMessage", self.global.scheduleTimer.sendMessageDelay*i, message, channel)
                    end
                    i = i + 1
                end
                del(messages)
            elseif type(messages) == "string" then
                self:SendChatMessage(messages, channel)
            end
        end
    end
    if instanceFound > 0 then
        local channel = "INSTANCE_CHAT"
        local messages = self:GetMessages(instance, "messageTHX", instanceFound)
        if type(messages) == "table" then
            local i = 0
            for _, message in pairs(messages) do
                if i == 0 then
                    self:SendChatMessage(messages, channel)
                else
                    self:ScheduleTimer("SendChatMessage", self.global.scheduleTimer.sendMessageDelay*i, message, channel)
                end
                i = i + 1
            end
            del(messages)
        elseif type(messages) == "string" then
            self:SendChatMessage(messages, channel)
        end
    end
    if nearFound > 0 then
        if self.db.profile.thanks.near == NEAR_TYPES.WHISPER then
            for name in pairs(near) do
                local nameTable = get()
                nameTable[name] = true
                local messages = self:GetMessages(nameTable, "messageTHX", nearFound)
                if type(messages) == "table" then
                    local message = messages[random(1, #messages)]

                    self:SendChatMessage(message, "WHISPER", name)
                    del(messages)
                elseif type(messages) == "string" then
                    self:SendChatMessage(messages, "WHISPER", name)
                end
                del(nameTable)
            end
        elseif self.db.profile.thanks.near == NEAR_TYPES.SAY then
            local messages = self:GetMessages(near, "messageTHX", nearFound)
            if type(messages) == "table" then
                local i = 0
                for _, message in pairs(messages) do
                    if i == 0 then
                        self:SendChatMessage(messages, "SAY")
                    else
                        self:ScheduleTimer("SendChatMessage", self.global.scheduleTimer.sendMessageDelay*i, message, "SAY")
                    end
                    i = i + 1
                end
                del(messages)
            elseif type(messages) == "string" then
                self:SendChatMessage(messages, "SAY")
            end
        end
    end

    del(guild)
    del(group)
    del(instance)
    del(near)
    --[===[@debug@
    self:Printf("ThanksForCongratulations: %f", (_G.GetTime() - start) )
    --@end-debug@]===]
end

do-- GetMessage
    local function getUnusedNames(names, unusedNames)
        local count = 0
        unusedNames = unusedNames or get()
        
        for name, notUsed in pairs(names) do
            if notUsed then
                count = count + 1
                unusedNames[name] = true
            end
        end

        return unusedNames, count
    end
    
    local function replaceNames(message, names, scope)
        if message:find("#n") and type(names) == "table" then
            if ArekosGratz.db.profile[scope][message].removeRealmName then
                -- TODO: removing realm name
            end
            if ArekosGratz.db.profile[scope][message].capitalizeFirstChar then
                -- TODO: capitalize first char
            end
            if #names > 0 then
                return message:gsub("#n", strjoin(", ", unpack(names) ))
            else
                local namelist
                for name in pairs(names) do
                    if not namelist then
                        namelist = name
                    else
                        namelist = namelist .. ", " .. name
                    end
                end
                return message:gsub("#n", namelist)
            end
        else
            return message
        end
    end

    function ArekosGratz:GetMessages(names, scope, foundCount)
        local forAll = get()
        local forAllFiltered = get()
        
        local filtered  = get()
        local filteredCount, filteredMessage = 0, nil


        for message, data in pairs(self.db.profile[scope]) do
            if #data.notFor == 0 and #data.onlyFor == 0 then
                tinsert(forAll, message)
                --[===[@debug@
                self:Printf("Add to forAll: %q", message)
                --@end-debug@]===]
            else
                local blocked = false
                
                if #data.notFor > 0 then
                    for _, rule in ipairs(data.notFor) do
                        if names[rule] then
                            blocked = true
                        else
                            if type(filtered[message]) ~= "table" then
                                filtered[message] = get()
                            end
                            tinsert(filtered[message], rule)
                        end
                    end
                end
                if #data.onlyFor > 0 then
                    for _, rule in ipairs(data.onlyFor) do
                        if names[rule] then
                            if type(filtered[message]) ~= "table" then
                                filtered[message] = get()
                            end
                            tinsert(filtered[message], rule)
                        else
                            blocked = true
                        end
                    end
                end

                --[===[@debug@
                self:Printf("Results for %q", message)
                --@end-debug@]===]

                if type(filtered[message]) == "table" and #filtered[message] > filteredCount then
                    filteredCount = #filtered[message]
                    filteredMessage = message
                    --[===[@debug@
                    self:Printf("Update filtered max: %d", filteredCount)
                    --@end-debug@]===]
                end

                if not blocked or ( type(filtered[message]) == "table" and #filtered[message] >= foundCount ) then
                    tinsert(forAllFiltered, message)
                    --[===[@debug@
                    self:Printf("Add to forAllFiltered: %q", message)
                    --@end-debug@]===]
                end
            end
        end

        if #forAll > 0 or #forAllFiltered > 0 then
            for key in pairs(filtered) do
                del(filtered[key])
                filtered[key] = nil
            end
            del(filtered)
            local message
            if #forAllFiltered > 0 then
                message = forAllFiltered[ random(1,#forAllFiltered) ]
            else
                message = forAll[ random(1,#forAll) ]
            end
            del(forAll)
            del(forAllFiltered)

            return replaceNames(message, names, scope)
        else
            del(forAll)
            del(forAllFiltered)
                    
            if filteredMessage then
                local messages = get()

                tinsert(messages, replaceNames(filteredMessage, filtered[filteredMessage], scope) )

                for i, name  in ipairs(filtered[filteredMessage]) do
                    names[name] = false
                end
                del(filtered[filteredMessage])
                filtered[filteredMessage] = nil

                local unusedNames, unusedNamesCount = getUnusedNames(names)
                while true do -- breaks out when we have no message remaining or none unused names
                    local matchCount = 0
                    local matchMessage
                    for message, data in pairs(filtered) do
                        local count = 0
                        for i, name in ipairs(data) do
                            if unusedNames[name] then
                                count = count + 1
                            end
                        end
                        if count > matchCount then
                            matchCount = count
                            matchMessage = message

                            if count >= unusedNamesCount then
                                break
                            end 
                        end
                    end

                    if matchMessage then
                        local usedNames = get()

                        for i, name in ipairs(filtered[matchMessage]) do
                            if unusedNames[name] then
                                unusedNames[name] = false
                                tinsert(usedNames, name)
                            end
                        end

                        tinsert(messages, replaceNames(matchMessage, usedNames, scope) )
                        del(usedNames)

                        if matchCount >= unusedNamesCount then
                            break
                        end

                        del(filtered[matchMessage])
                        filtered[matchMessage] = nil
                        if not next(filtered) then
                            break
                        end

                        local _unusedNames
                        _unusedNames, unusedNamesCount = getUnusedNames(unusedNames)
                        del(unusedNames)
                        unusedNames = _unusedNames
                    else
                        break
                    end
                end
                del(unusedNames)


                for key in pairs(filtered) do
                    del(filtered[key])
                    filtered[key] = nil
                end
                del(filtered)
                return messages
            else -- No result, clean up and return a empty table
                --[===[@debug@
                self:Printf("No result found for %s", scope)
                --@end-debug@]===]
                
                for key in pairs(filtered) do
                    del(filtered[key])
                    filtered[key] = nil
                end
                del(filtered)

                return get()
            end
        end
    end
end

function ArekosGratz:UniformPlayerName(playername, realmOverride)
    playername = playername:gsub("%s+", "")
    local name, realm = ("-"):split( playername )
    
    if type(realmOverride) == "string" and realmOverride:len() > 0 then
        realm = realmOverride:gsub("%s+", "")
    elseif type(realm) ~= "string" or realm:len() < 1 then
        realm = HOMEREALM
    end

    return name.."-"..realm, name, realm
end
function ArekosGratz:UniformGuildName(guildname, realmOverride)
    local guild, realm = ("-"):split(guildname)

    if type(realmOverride) == "string" and realmOverride:trim():len() > 0 then
        realm = realmOverride:gsub("%s+", "")
    else
        if type(realm) ~= "string" then
                realm = HOMEREALM
        else
            realm = realm:gsub("%s+", "")
            if realm:len() < 1 then
                realm = HOMEREALM
            end
        end
    end

    return guild.."-"..realm, guild, realm   
end

function ArekosGratz:SendChatMessage(message, channel, to)
    if self:IAmAFK() and self.db.profile.protectAFK then return end
    if type(message) ~= "string" or message:len() < 1 or type(channel) ~= "string" or channel:len() < 1 then return end

    if to then
        _G.SendChatMessage(message, channel, nil, to)
    else
        _G.SendChatMessage(message, channel)
    end
end


local optionsHandler = setmetatable({ cache = {} },{
    __index = function(t, k)
        if k == "db" then
            return ArekosGratz.db
        else
            return rawget(t,k)
        end
    end
})

local function getInternCacheKey(info)
    local key = ""
    if type(info) == "table" then
        for i=1, #info do
            key = key..( info[i] or "root"..i )
        end
    else
        key = tostring(info)
    end
    return key
end
function optionsHandler:SetInternCache(info, value)
    local key = getInternCacheKey(info)
    self.cache[key] = value
end
function optionsHandler:GetInternCache(info, defaultValue)
    local defaultValue = defaultValue == nil and "" or defaultValue
    local key = getInternCacheKey(info)
    local value = self.cache[key]

    if value == nil then
        value = defaultValue
    end
    return value
end
function optionsHandler:SetCache(info, value)
    self:SetInternCache(info, value)
end
function optionsHandler:GetCache(info)
    return self:GetInternCache(info, "")
end

function optionsHandler:Set(info, value)
    local scope = info.arg or "profile"
    local name = info[#info]
    
    self.db[scope][name] = value
end
function optionsHandler:Get(info)
    local scope = info.arg or "profile"
    local name = info[#info]

    return self.db[scope][name]
end
function optionsHandler:IsIgnoreInstanceGroupDisabled(info)
    return not self.db.profile.congratulations.party
end
function optionsHandler:SetCongratulations(info, value)
    local name = info[#info]

    self.db.profile.congratulations[name] = value
end
function optionsHandler:GetCongratulations(info)
    local name = info[#info]

    return self.db.profile.congratulations[name]
end
function optionsHandler:SetThanks(info, value)
    local name = info[#info]

    self.db.profile.thanks[name] = value
end
function optionsHandler:GetThanks(info)
    local name = info[#info]

    return self.db.profile.thanks[name]
end
function optionsHandler:SetMinWaitTime(info, value)
    local scope
    if type(info.arg) == "string" and info.arg:len() > 0 then
        scope = info.arg
    else
        return
    end

    if self.db.profile[scope].maxWaitTime > value then
        self.db.profile[scope].minWaitTime = value
    end
end
function optionsHandler:SetMaxWaitTime(info, value)
    local scope
    if type(info.arg) == "string" and info.arg:len() > 0 then
        scope = info.arg
    else
        return
    end

    if self.db.profile[scope].minWaitTime < value then
        self.db.profile[scope].maxWaitTime = value
    end
end

local ID_KEY = "MK:"
local ID_KEY_LENGTH = ID_KEY:len()

function optionsHandler:ValidateAddMessage(info, value, scope)
    local scope = type(scope) == "string" and scope or info[#info-1] or info.arg
    local name = info[#info]

    if name:sub(-6) == "Button" then
        info[#info] = name:sub(0,-7)
        value = self:GetInternCache(info)
        info[#info] = name
    end

    local result = true

    if value:trim():len() < 1 then
        result = L["Message is empty."]
    elseif self.db.profile[scope][value].exists then
        result = L["Exists already."]
    end

    if info.uiType == "dialog" and type(result) == "string" then
        ArekosGratz:Printf("%s: %s", info.option.name or L["Input"], result)
    end
    return result
end

function optionsHandler:AddMessage(info, value)
    local scope = info[#info-1] or info.arg
    local name = info[#info]

    local validated = self:ValidateAddMessage(info, value)
    if type(validated) == "string" or not validated then
        return
    end

    if name:sub(-6) == "Button" then
        info[#info] = name:sub(0,-7)
        value = self:GetInternCache(info)
        self:SetInternCache(info, "")
        info[#info] = name
    end

    self.db.profile[scope][value:trim()].exists = true
end
function optionsHandler:DeleteMessage(info, value)
    local scope = info[#info-2] or info.arg
    local key = info[#info-1]:sub(ID_KEY_LENGTH+1)

    self.db.profile[scope][key] = nil
    
    info.option = nil
end
function optionsHandler:HasNoneNameReplace(info)
    local key = info[#info-1]:sub(ID_KEY_LENGTH+1)

    local cache = self:GetInternCache(info, 1)
    if cache == 1 then
        if key:find("#n") then
            cache = false
        else
            cache = true
        end
        self:SetInternCache(info, cache)
    end
    return cache
end
function optionsHandler:SetMessage(info, value)
    local scope = info.arg or info[#info-2]
    local key = info[#info-1]:sub(ID_KEY_LENGTH+1)
    local name = info[#info]

    self.db.profile[scope][key][name] = value
end
function optionsHandler:GetMessage(info)
    local scope = info.arg or info[#info-2]
    local key = info[#info-1]:sub(ID_KEY_LENGTH+1)
    local name = info[#info]

    return self.db.profile[scope][key][name]
end
function optionsHandler:GetMessageKey(info)
    return self:GetInternCache(info, false) or info[#info-1]:sub(ID_KEY_LENGTH+1)
end
function optionsHandler:RenameMessage(info, value)
    local scope = info[#info-2] or info.arg
    local key = info[#info-1]:sub(ID_KEY_LENGTH+1)
    local name = info[#info]

    local validated = self:ValidateAddMessage(info, value, scope)
    if type(validated) == "string" or not validated then
        return
    end
    
    if name:sub(-6) == "Button" then
        info[#info] = name:sub(0,-7)
        value = self:GetInternCache(info, key)
        info[#info] = name
    end

    self.db.profile[scope][value:trim()] = self.db.profile[scope][key]
    self.db.profile[scope][key] = nil
end

function optionsHandler:SetMessageFilter(info, value)
    local scope = info[#info-2] or info.arg
    local key = info[#info-1]:sub(ID_KEY_LENGTH+1)
    local name = info[#info]

    local values = get()
    for v in value:gmatch("[^\n]+") do
        v = v:gsub("%s+", "")

        if v:len() > 1 then
            if v:find("#") then
                local name, id = ("#"):split(v)

                if type(id) == "string" and id:len() == 4 and name:len() > 1 then
                    tinsert(values, lower(name.."#"..id) )
                end
            else
                local name = ArekosGratz:UniformPlayerName(v)
                tinsert(values, lower(name) )
            end
        end
    end

    del(self.db.profile[scope][key][name])
    self.db.profile[scope][key][name] = values
    self:SetInternCache(info, false)
end
function optionsHandler:GetMessageFilter(info)
    local cache = self:GetInternCache(info, false)
    if not cache then
        local scope = info.arg or info[#info-2]
        local key = info[#info-1]:sub(ID_KEY_LENGTH+1)
        local name = info[#info]

        local data = self.db.profile[scope][key][name]
        cache = strjoin("\n", unpack(data))
        self:SetInternCache(info, cache)
    end

    return cache
end

function optionsHandler:SetTrigger(info, value)
    local triggers = self.db.global.gzTriggers

    for trigger, inUse in pairs(triggers) do
        if inUse then
            triggers[trigger] = false
        end
    end
    for trigger in value:gmatch("[^\n]+") do
        triggers[lower( trigger:trim() )] = true
    end
    self:SetInternCache(info, false)
end
function optionsHandler:GetTrigger(info)
    local cache = self:GetInternCache(info, false)
    if not cache then
        local data = get()

        for trigger, inUse in pairs(self.db.global.gzTriggers) do
            if inUse then
                tinsert(data, trigger)
            end
        end

        cache = strjoin("\n", unpack(data) )
        del(data)
        self:SetInternCache(info, cache)
    end

    return cache
end

local configs = {
    type    = "group",
    name    = L["Areko's Gratz"],
    handler = optionsHandler,
    set     = "Set",
    get     = "Get",
    args = {
        main = {
            type    = "group",
            name    = L["Areko's Gratz"],
            handler = optionsHandler,
            set     = "Set",
            get     = "Get",
            args = {
                protectAFK = {
                    type    = "toggle",
                    name    = L["Protect AFK-Status"],
                    desc    = L["No congratulation while AFK"],
                    order   = 1,
                },
                congratulations = {
                    type    = "group",
                    name    = L["Congratulate"],
                    desc    = L["Settings for congratulations"],
                    order   = 2,
                    set     = "SetCongratulations",
                    get     = "GetCongratulations",
                    inline  = true,
                    args    = {
                        guild = {
                            type    = "toggle",
                            name    = L["Guild"],
                            desc    = L["Congratulations to the guild channel"],
                            order   = 1,
                            arg     = "congratulations",
                        },
                        party = {
                            type    = "toggle",
                            name    = L["Party"],
                            desc    = L["Congratulations to the party channel"],
                            order   = 2,
                        },
                        ignoreInstanceGroup = {
                            type        = "toggle",
                            name        = L["Ignore Instance"],
                            desc        = L["Don't congratulate to instance groups"],
                            disabled    = "IsIgnoreInstanceGroupDisabled",
                            order       = 2.5,
                        },
                        near = {
                            type    = "select",
                            name    = L["Congratulations to near players"],
                            values  = NEAR_TYPES_VALUES,
                            style   = "radio",
                            order   = 3
                        },
                        minWaitTime = {
                            type    = "range",
                            name    = L["Min. wait time to congratulation"],
                            min     = 0,
                            max     = 60,
                            step    = 1,
                            bigStep = 1,
                            arg     = "congratulations",
                            set     = "SetMinWaitTime",
                            order   = 4,
                            width   = "double",
                        },
                        maxWaitTime = {
                            type    = "range",
                            name    = L["Max. wait time to congratulation"],
                            min     = 0,
                            max     = 60,
                            step    = 1,
                            bigStep = 1,
                            arg     = "congratulations",
                            set     = "SetMaxWaitTime",
                            order   = 5,
                            width   = "double",
                        },
                        guildInOtherChannel = {
                            type    = "toggle",
                            name    = L["Congratulate guildmembers over other channels"],
                            desc    = L["Congratulations to guildmembers who in our party or near your in other channels as guildchannel"],
                            order   = 6,
                            width   = "full"
                        },
                    },
                },
                thanks = {
                    type    = "group",
                    name    = L["Thanks"],
                    desc    = L["Settings for saying thanks"],
                    set     = "SetThanks",
                    get     = "GetThanks",
                    inline  = true,
                    order   = 3,
                    args    = {
                        guild = {
                            type    = "toggle",
                            name    = L["Guild"],
                            desc    = L["Saying thanks to the guild channel"],
                            order   = 8,
                        },
                        party = {
                            type    = "toggle",
                            name    = L["Party"],
                            desc    = L["Saying thanks to the party channel"],
                            order   = 8,
                        },                        
                        instance = {
                            type    = "toggle",
                            name    = L["Instance"],
                            desc    = L["Saying thanks to the instance group"],
                            order   = 8.5,
                        },
                        near = {
                            type    = "select",
                            name    = L["Saying thanks near players"],
                            values  = NEAR_TYPES_VALUES,
                            style   = "radio",
                            order   = 9,
                        },
                        minWaitTime = {
                            type    = "range",
                            name    = L["Min. wait time to thanks"],
                            min     = 0,
                            max     = 60,
                            step    = 1,
                            bigStep = 1,
                            arg     = "thanks",
                            set     = "SetMinWaitTime",
                            order   = 10,
                            width   = "double",
                        },
                        maxWaitTime = {
                            type    = "range",
                            name    = L["Max. wait time to thanks"],
                            min     = 0,
                            max     = 60,
                            step    = 1,
                            bigStep = 1,
                            arg     = "thanks",
                            set     = "SetMaxWaitTime",
                            order   = 11,
                            width   = "double",
                        },
                        maxExtraThanksTrys = {
                            type    = "range",
                            name    = L["Max. times to retry to thanks for congratulations"],
                            min     = 0,
                            max     = 3,
                            bigStep = 1,
                            arg     = "thanks",
                            order   = 12,
                        },
                    },
                },
            },
        },
        messageGZ = {
            type        = "group",
            name        = L["Congratulate"],
            childGroups = "tree",
            handler     = optionsHandler,
            set         = "Set",
            get         = "Get",
            args = {
                AddMessageGZ = {
                    name        = L["Congratulations message"],
                    desc        = L["Use #n for playername"],
                    type        = "input",
                    arg         = "messageGZ",
                    set         = "SetCache",
                    get         = "GetCache",
                    order       = 1,
                },
                AddMessageGZButton = {
                    name        = L["Add message"],
                    type        = "execute",
                    arg         = "messageGZ",
                    validate    = "ValidateAddMessage",
                    func        = "AddMessage",
                    order       = 1.1,
                },                
            },
        },
        messageTHX = {
            type        = "group",
            name        = L["Thanks"],
            childGroups = "tree",
            handler     = optionsHandler,
            set         = "Set",
            get         = "Get",
            args = {
                AddMessageTHX = {
                    name        = L["Thanks message"],
                    desc        = L["Use #n for playername"],
                    type        = "input",
                    arg         = "messageTHX",
                    set         = "SetCache",
                    get         = "GetCache",
                    order       = 1,
                },
                AddMessageTHXButton = {
                    name        = L["Add message"],
                    type        = "execute",
                    arg         = "messageTHX",
                    validate    = "ValidateAddMessage",
                    func        = "AddMessage",
                    order       = 1.1,
                },                
            },
        },
        trigger = {
            type = "group",
            name = L["Thanks Triggers"],
            arg  = "global",
            handler = optionsHandler,
            set  = "SetTrigger",
            get  = "GetTrigger",
            args = {
                list = {
                    name    = L["Congratulations Keywords"],
                    desc    = L["List of triggers for thanks\nOne word per line"],
                    type    = "input",
                    multiline = 20,
                    width   = "full",
                    order   = 1
                    },
                },
        },
    },
}

do
    local filterDescription = L["One player per line.\n\nName-Realm: Match on playername\nName#0000: Battle.net Tag(NYI)"]
    function ArekosGratz:GetMessageOptionsTable(uiType, uiName, appName, scope)
        local message = configs.args[scope].args

        for key, data in pairs(self.db.profile[scope]) do
            if not data.isIntern then
                local configKey = ID_KEY..key

                message[configKey] = message[configKey] or {}
                message[configKey].name    = key
                message[configKey].type    = "group"
                message[configKey].set     = "SetMessage"
                message[configKey].get     = "GetMessage"
                message[configKey].arg     = scope
                message[configKey].args    = message[configKey].args or {}

                message[configKey].args.rename = message[configKey].args.rename or {}
                message[configKey].args.rename.name    = L["Rename message"]
                message[configKey].args.rename.desc    = L["Use #n for playername"]
                message[configKey].args.rename.type    = "input"
                message[configKey].args.rename.set     = "SetCache"
                message[configKey].args.rename.get     = "GetMessageKey"
                message[configKey].args.rename.arg     = scope
                message[configKey].args.rename.order   = 1

                message[configKey].args.renameButton = message[configKey].args.renameButton or {}
                message[configKey].args.renameButton.name      = L["Rename message"]
                message[configKey].args.renameButton.type      = "execute"
                message[configKey].args.renameButton.validate  = "ValidateAddMessage"
                message[configKey].args.renameButton.func      = "RenameMessage"
                message[configKey].args.renameButton.arg       = scope
                message[configKey].args.renameButton.order     = 2

                message[configKey].args.removeRealmName = message[configKey].args.removeRealmName or {}
                message[configKey].args.removeRealmName.name      = L["Remove Realm"]
                message[configKey].args.removeRealmName.desc      = L["Remove Realm from the playername"]
                message[configKey].args.removeRealmName.type      = "toggle"
                message[configKey].args.removeRealmName.arg       = scope
                message[configKey].args.removeRealmName.hidden    = "HasNoneNameReplace"
                message[configKey].args.removeRealmName.order     = 2.1
                
                message[configKey].args.capitalizeFirstChar = message[configKey].args.capitalizeFirstChar or {}
                message[configKey].args.capitalizeFirstChar.name    = L["Capitalize Name"]
                message[configKey].args.capitalizeFirstChar.desc    = L["Capitalize the first char from the playername"]
                message[configKey].args.capitalizeFirstChar.type    = "toggle"
                message[configKey].args.capitalizeFirstChar.arg     = scope
                message[configKey].args.capitalizeFirstChar.hidden  = "HasNoneNameReplace"
                message[configKey].args.capitalizeFirstChar.order   = 2.2

                message[configKey].args.onlyFor = message[configKey].args.onlyFor or {}
                message[configKey].args.onlyFor.name       = L["Only for:"]
                message[configKey].args.onlyFor.desc       = filterDescription
                message[configKey].args.onlyFor.type       = "input"
                message[configKey].args.onlyFor.multiline  = 9
                message[configKey].args.onlyFor.set        = "SetMessageFilter"
                message[configKey].args.onlyFor.get        = "GetMessageFilter"
                message[configKey].args.onlyFor.width      = "double"
                message[configKey].args.onlyFor.arg        = scope
                message[configKey].args.onlyFor.order      = 3

                message[configKey].args.notFor = message[configKey].args.notFor or {}
                message[configKey].args.notFor.name        = L["Not for:"]
                message[configKey].args.notFor.desc        = filterDescription
                message[configKey].args.notFor.type        = "input"
                message[configKey].args.notFor.multiline   = 9
                message[configKey].args.notFor.set         = "SetMessageFilter"
                message[configKey].args.notFor.get         = "GetMessageFilter"
                message[configKey].args.notFor.width       = "double"
                message[configKey].args.notFor.arg         = scope
                message[configKey].args.notFor.order       = 4

                message[configKey].args.deleteButton = message[configKey].args.deleteButton or {}
                message[configKey].args.deleteButton.name           = L["Delete"]
                message[configKey].args.deleteButton.type           = "execute"
                message[configKey].args.deleteButton.func           = "DeleteMessage"
                message[configKey].args.deleteButton.confirm        = true
                message[configKey].args.deleteButton.confirmText    = L["Delete %q message?"]:format(key)
                message[configKey].args.deleteButton.arg            = scope
                message[configKey].args.deleteButton.order          = 5
            end
        end

        for key in pairs(message) do
            if key:sub(0,ID_KEY_LENGTH) == ID_KEY then
                if self.db.profile[scope] and not self.db.profile[scope][key:sub(ID_KEY_LENGTH+1)].exists then
                    message[key] = nil
                end     
            end
        end

        return configs.args[scope]
    end
    function ArekosGratz.GetMessageGZOptionsTable(uiType, uiName, appName)
        return ArekosGratz:GetMessageOptionsTable(uiType, uiName, appName, "messageGZ")
    end
    function ArekosGratz.GetMessageTHXOptionsTable(uiType, uiName, appName)
        return ArekosGratz:GetMessageOptionsTable(uiType, uiName, appName, "messageTHX")
    end
end

function ArekosGratz:RegisterConfig()
    local registry = LibStub("AceConfigRegistry-3.0")
    local dialog = LibStub("AceConfigDialog-3.0")
    local skip = true
    --[===[@debug@
    skip = false
    --@end-debug@]===]

    registry:RegisterOptionsTable(ADDONNAME.."OptionsMain", configs.args.main, skip)
    registry:RegisterOptionsTable(ADDONNAME.."OptionsMessageGZ", self.GetMessageGZOptionsTable, skip)
    registry:RegisterOptionsTable(ADDONNAME.."OptionsMessageTHX", self.GetMessageTHXOptionsTable, skip)
    registry:RegisterOptionsTable(ADDONNAME.."OptionsTrigger", configs.args.trigger, skip)
    
    local profiles = LibStub("AceDBOptions-3.0"):GetOptionsTable(self.db)
    registry:RegisterOptionsTable(ADDONNAME.."Profile",  profiles)


    dialog:AddToBlizOptions(ADDONNAME.."OptionsMain", L["Areko's Gratz"], nil)
    dialog:AddToBlizOptions(ADDONNAME.."OptionsMessageGZ", L["Congratulate"], L["Areko's Gratz"])
    dialog:AddToBlizOptions(ADDONNAME.."OptionsMessageTHX", L["Thanks"], L["Areko's Gratz"])
    dialog:AddToBlizOptions(ADDONNAME.."OptionsTrigger", L["Thanks Triggers"], L["Areko's Gratz"])

    --L["Profile"]
    dialog:AddToBlizOptions(ADDONNAME.."Profile", profiles.name, L["Areko's Gratz"])
end


