--[[ ---------------------------------------------------------------------------

Acheron: death reports

Acheron is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

Acheron is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with Acheron.	If not, see <http://www.gnu.org/licenses/>.

----------------------------------------------------------------------------- ]]

--[[ ---------------------------------------------------------------------------
	 Addon and libraries
----------------------------------------------------------------------------- ]]

local L = LibStub("AceLocale-3.0"):GetLocale("Acheron")
local GUI = LibStub("AceGUI-3.0")


--[[ ---------------------------------------------------------------------------
	 Create the main display frame
----------------------------------------------------------------------------- ]]
function Acheron:CreateFrame()

	if self.frame then return end
	
	local f = GUI:Create("Frame")
	f:SetTitle("Acheron")
	f:SetStatusText("")
	f:SetLayout("Flow")
	f:SetWidth(700)
	f:SetHeight(650)
	f.width = "fill"
	
	local g1 = GUI:Create("InlineGroup")
	g1:SetTitle(L["Filter"])
	g1:SetLayout("Flow")
	g1.width = "fill"
	
	local ds = GUI:Create("Dropdown")
	ds:SetLabel(L["Show"])
	ds:SetWidth(125)
	ds:SetList(Acheron:GetAvailableReports())
	ds:SetCallback("OnValueChanged", function(widget,event,value) 
 		Acheron:ShowDeathReports(value)
	end)
	g1:AddChild(ds)
	
	local s = GUI:Create("Slider")
	s:SetLabel(L["Time to Show"])
	s:SetWidth(150)
	s:SetSliderValues(1, 60, 1)
	s:SetValue(Acheron:GetProfileParam("reporttime"))
	s:SetCallback("OnValueChanged", function(widget,event,value)
		Acheron:SetProfileParam("reporttime", value)
		local status = Acheron.frameReports.status or Acheron.frameReports.localstatus
		if status and status.selected then
			Acheron:PopulateEntries(strsplit(":", status.selected))
		end
	end)
	g1:AddChild(s)
	
	local s = GUI:Create("Slider")
	s:SetLabel(L["Amount to Show >"])
	s:SetWidth(150)
	s:SetSliderValues(0, 1000, 10)
	s:SetValue(Acheron:GetProfileParam("reportthreshold"))
	s:SetCallback("OnValueChanged", function(widget,event,value)
		Acheron:SetProfileParam("reportthreshold", value)
		local status = Acheron.frameReports.status or Acheron.frameReports.localstatus
		if status and status.selected then
			Acheron:PopulateEntries(strsplit(":", status.selected))
		end
	end)
	g1:AddChild(s)
	
	local cbd = GUI:Create("CheckBox")
	cbd:SetLabel(L["Damage"])
	cbd:SetWidth(100)
	cbd:SetValue(Acheron:GetProfileParam("showdamage"))
	cbd:SetCallback("OnValueChanged", function(widget,event,value)
		Acheron:SetProfileParam("showdamage", value)
		local status = Acheron.frameReports.status or Acheron.frameReports.localstatus
		if status and status.selected then
			Acheron:PopulateEntries(strsplit(":", status.selected))
		end
	end)
	g1:AddChild(cbd)
	
	local cbh = GUI:Create("CheckBox")
	cbh:SetLabel(L["Healing"])
	cbh:SetWidth(100)
	cbh:SetValue(Acheron:GetProfileParam("showhealing"))
	cbh:SetCallback("OnValueChanged", function(widget,event,value)
		Acheron:SetProfileParam("showhealing", value)
		local status = Acheron.frameReports.status or Acheron.frameReports.localstatus
		if status and status.selected then
			Acheron:PopulateEntries(strsplit(":", status.selected))
		end
	end)
	g1:AddChild(cbh)
	
	local cbb = GUI:Create("CheckBox")
	cbb:SetLabel(L["Buffs"])
	cbb:SetWidth(100)
	cbb:SetValue(Acheron:GetProfileParam("showbuff"))
	cbb:SetCallback("OnValueChanged", function(widget,event,value)
		Acheron:SetProfileParam("showbuff", value)
		local status = Acheron.frameReports.status or Acheron.frameReports.localstatus
		if status and status.selected then
			Acheron:PopulateEntries(strsplit(":", status.selected))
		end
	end)
	g1:AddChild(cbb)
	
	local cbdb = GUI:Create("CheckBox")
	cbdb:SetLabel(L["Debuffs"])
	cbdb:SetWidth(100)
	cbdb:SetValue(Acheron:GetProfileParam("showdebuff"))
	cbdb:SetCallback("OnValueChanged", function(widget,event,value)
		Acheron:SetProfileParam("showdebuff", value)
		local status = Acheron.frameReports.status or Acheron.frameReports.localstatus
		if status and status.selected then
			Acheron:PopulateEntries(strsplit(":", status.selected))
		end
	end)
	g1:AddChild(cbdb)
	
	f:AddChild(g1)
	
	local g2 = GUI:Create("InlineGroup")
	g2:SetTitle(L["Report"])
	g2:SetLayout("Flow")
	g2.width = "fill"
	
	local d = GUI:Create("Dropdown")
	d:SetLabel(L["Report To"])
	d:SetWidth(125)
	d:SetList(Acheron:GetAcheronChannels())
	d:SetValue(Acheron:GetProfileParam("reportchannel"))
	d:SetCallback("OnValueChanged", function(widget,event,value) 
		Acheron:SetProfileParam("reportchannel", value)
		if value == L["whisper"] then
			Acheron.whisperEditBox.frame:Show()
		else
			Acheron.whisperEditBox.frame:Hide()
		end
	end)
	g2:AddChild(d)
	
	local wt = GUI:Create("EditBox")
	wt:SetLabel(L["Whisper To"])
	wt:SetWidth(150)
	wt:SetCallback("OnEnterPressed", function(widget,event,value) Acheron.whisperTarget = value end)
	g2:AddChild(wt)
	
	local cba = GUI:Create("CheckBox")
	cba:SetLabel(L["Absolute Health"])
	cba:SetWidth(125)
	cba:SetValue(Acheron:GetProfileParam("abshealth"))
	cba:SetCallback("OnValueChanged", function(widget,event,value)
		Acheron:SetProfileParam("abshealth", value)
		local status = Acheron.frameReports.status or Acheron.frameReports.localstatus
		if status and status.selected then
			Acheron:PopulateEntries(strsplit(":", status.selected))
		end
	end)
	g2:AddChild(cba)
	
	f:AddChild(g2)
	
	local bu = GUI:Create("Button")
	bu:SetText(L["Clear All"])
	bu:SetWidth(100)
	bu:SetCallback("OnClick", function(widget,event,value) 
 		Acheron:ClearReports()
 		Acheron:ShowDeathReports()
	end)
	f:AddChild(bu)
	
	local t = GUI:Create("TreeGroup")
	t:SetLayout("Fill")
	t:SetCallback("OnClick", function(_, _, value) Acheron:PopulateEntries(strsplit(":", value)) end)
	t.width = "fill"
	t.height = "fill"
	f:AddChild(t)
	
	f:Hide()

	self.frame = f
	self.frameReports = t
	self.channels = d
	self.availableReports = ds
	self.whisperEditBox = wt
	
end


--[[ ---------------------------------------------------------------------------
	 Formulate the table of channels Acheron may use to report, filtering out
	 server channels
----------------------------------------------------------------------------- ]]
function Acheron:GetAcheronChannels()

	local acheronChannels = {}
	acheronChannels[L["say"]] = L["say"]
	acheronChannels[L["party"]] = L["party"]
	acheronChannels[L["raid"]] = L["raid"]
	acheronChannels[L["guild"]] = L["guild"]
	acheronChannels[L["officer"]] = L["officer"]
	acheronChannels[L["whisper"]] = L["whisper"]
	
	local serverChanTable = {}
	local serverChannels = {EnumerateServerChannels()}
	
	for idx, chan in ipairs(serverChannels) do
		serverChanTable[chan] = true
	end
	
	local channels={GetChannelList()}

	for i=1,(#channels)/2 do
		if not serverChanTable[channels[i*2]] then
			acheronChannels[tostring(channels[i*2-1])] = channels[i*2]
		end
	end
	
	return acheronChannels
	
end


--[[ ---------------------------------------------------------------------------
	 Formulate the table of reports Acheron has
----------------------------------------------------------------------------- ]]
function Acheron:GetAvailableReports()

	local acheronReports = {}

	for id,entry in pairs(Acheron.combatLogs) do
		acheronReports[entry.name] = entry.name
	end
	
	return acheronReports

end


--[[ ---------------------------------------------------------------------------
	 Display reports for a person
----------------------------------------------------------------------------- ]]
function Acheron:ShowDeathReports(name)

	if not name then name = _G[UIDROPDOWNMENU_INIT_MENU].name end
	local id = UnitGUID(tostring(name))
	local lastReportNum = nil
	
	if Acheron.combatLogs[id] and #(Acheron.combatLogs[id]) > 1 then
		lastReportNum = #(Acheron.combatLogs[id]) - 1
	end
	
	if Acheron.debug then Acheron:debug("Showing death report for %s (%s) - #%s", tostring(name), tostring(id), tostring(lastReportNum)) end
	
	Acheron.channels:SetList(Acheron:GetAcheronChannels())
	Acheron.availableReports:SetList(Acheron:GetAvailableReports())
	
	if id then
		Acheron.availableReports:SetValue(name)
		Acheron:PopulateReports(name)
	elseif Acheron.frameReports then
		Acheron.frameReports:SetTree({})
		Acheron.availableReports:SetValue(nil)
		Acheron.frame:SetStatusText(nil)
		if Acheron.frameReports.status and Acheron.frameReports.status.selected then
			Acheron.frameReports.status.selected = nil
		end
		if Acheron.frameReports.localstatus and Acheron.frameReports.localstatus.selected then
			Acheron.frameReports.localstatus.selected = nil
		end
	end
	
	if lastReportNum then
		Acheron:PopulateEntries(name, lastReportNum)
		Acheron.frameReports:SelectByValue(name..":"..lastReportNum)
	elseif Acheron.frameReports then
		Acheron.frameReports:ReleaseChildren()
	end
	
	Acheron.frame:Show()
	
	if Acheron:GetProfileParam("reportchannel") == L["whisper"] then
		Acheron.whisperEditBox.frame:Show()
	else
		Acheron.whisperEditBox.frame:Hide()
	end
	
end


--[[ ---------------------------------------------------------------------------
	 Populate the main display frames with the report data for name
----------------------------------------------------------------------------- ]]
function Acheron:PopulateReports(name)

	self.frame:SetStatusText(name)

	local id = UnitGUID(tostring(name))
	if not self.combatLogs[id] then return end
	
	local t = {}

	for i = 1, #(self.combatLogs[id])-1 do

		local lastEntry = self:GetLastDamageEntry(self.combatLogs[id][i])
		local report = {}
		report.value = name..":"..i
		report.text = format("[%s] %s - %s", date("%H:%M:%S", lastEntry.timeStamp), lastEntry.source or "", lastEntry.spell or "")
		
		tinsert(t, report)
		
	end
	
	self.frameReports:SetTree(t)
	
end


--[[ ---------------------------------------------------------------------------
	 Find the last entry that caused damage
----------------------------------------------------------------------------- ]]
function Acheron:GetLastDamageEntry(report)

	local lastEntry = nil
	
	for i = (report.last - 1), 1, -1 do
		if report.log[i] and report.log[i].amount < 0 then
			lastEntry = report.log[i]
			break
		end
	end
	
	if not lastEntry then
		lastEntry = report.log[report.last]
	end
	
	return lastEntry

end


--[[ ---------------------------------------------------------------------------
	 Populate the main display frames with the death report for the entry
----------------------------------------------------------------------------- ]]
function Acheron:PopulateEntries(name, reportNum)

	self.frameReports:ReleaseChildren()

	local id = UnitGUID(tostring(name))
	reportNum = tonumber(reportNum)
	
	local reportSet = self:FilterReport(id, reportNum, self:GetProfileParam("reporttime"))
	local lastTimeStamp = reportSet[#reportSet].timeStamp
	
	local sf = GUI:Create("ScrollFrame")
	sf:SetLayout("Flow")
	
	for i,entry in ipairs(reportSet) do

		local label = GUI:Create("Label")
		label.width = "fill"
		local p, h, f = label.label:GetFont()
		label.label:SetFont(p, self:GetProfileParam("fontsize"), f)
		label.frame:EnableMouse(true)
		label.frame.highlight = label.frame:CreateTexture("Highlight", "OVERLAY")
		label.frame.highlight:SetTexture(0.4,0.4,0,0.5)
		label.frame.highlight:SetAllPoints()
		label.frame.highlight:SetBlendMode("ADD")
		label.frame.highlight:Hide()
		
		label.frame:SetScript("OnEnter", function()
			GameTooltip_SetDefaultAnchor(GameTooltip, UIParent)
          	GameTooltip:ClearLines()
          	
          	if label.userdata.entry then
          		if label.userdata.entry.isCrit then GameTooltip:AddLine(format("|cffffffff%s|r", L["Critical"])) end
          		if label.userdata.entry.isCrush then GameTooltip:AddLine(format("|cffffffff%s|r", L["Crushing"])) end
          		GameTooltip:AddLine(format("|cffffffff%d/%d (%s)|r",
          							label.userdata.entry.curHealth,
          							label.userdata.entry.maxHealth,
          							label.userdata.entry.pctHealth))
          		if label.userdata.entry.msg then GameTooltip:AddLine(format("|cffffffff%s|r", label.userdata.entry.msg)) end
          	end
          	
            GameTooltip:AddLine(L["Click to report from this point"])
            
            GameTooltip:Show()
			this.highlight:Show() 
		end)
		
		label.frame:SetScript("OnLeave", function()
			GameTooltip:Hide()
			this.highlight:Hide()
		end)
		
		label.frame:SetScript("OnMouseDown", function(_, button)
			if button == "LeftButton" then
				Acheron:Report(name, reportNum, label.userdata.seconds)
			end
		end)

		label:SetText(self:EntryToString(entry, name, lastTimeStamp))
		label.userdata.entry = entry
		label.userdata.seconds = lastTimeStamp - entry.timeStamp
		
		sf:AddChild(label)

	end
	
	self.frameReports:AddChild(sf)

end


--[[ ---------------------------------------------------------------------------
	 Report a combat log
----------------------------------------------------------------------------- ]]
function Acheron:Report(name, reportNum, seconds)

	-- validate name
	local id = UnitGUID(tostring(name))
	
	if not self.combatLogs[id] then
		return
	else
		name = self.combatLogs[id].name
	end
	
	-- get report number
	if not self.combatLogs[id][tonumber(reportNum)+1] or tonumber(reportNum) < 1 then
		return
	else
		reportNum = tonumber(reportNum)
	end
	
	-- when and where to report
	seconds = tonumber(seconds)
	channel = self:GetProfileParam("reportchannel")
	
	-- validate channel destination
	if channel == L["whisper"] then
		if not self.whisperTarget then
			if not UnitExists("target") then self:DoPrint(nil, L["Acheron: No whisper target"]) return end
			if not UnitIsPlayer("target") then self:DoPrint(nil, L["Acheron: Whisper target is not a player"]) return end
		end
	elseif channel == L["party"] then
		if GetNumPartyMembers() == 0 then self:DoPrint(nil, L["Acheron: You are not in a party"]) return end
	elseif channel == L["raid"] then
		if GetNumRaidMembers() == 0 then self:DoPrint(nil, L["Acheron: You are not in a raid"]) return end
	elseif channel == L["guild"] or channel == L["officer"] then
		if not IsInGuild() then self:DoPrint(nil, L["Acheron: You are not in a guild"]) return end
	elseif channel ~= L["say"]  then
		local chanNum = GetChannelName(channel);
		if not chanNum or chanNum == 0 then self:DoPrint(nil, L["Acheron: No such channel: %s"], channel) return end
	end
	
	-- finally report
	local reportSet = self:FilterReport(id, reportNum, seconds)
	
	if next(reportSet) then
		self:DoPrint(channel, format(L["Acheron: Death report for %s:"], name))
	end
	
	local lastTimeStamp = reportSet[#reportSet].timeStamp
	
	for i,entry in ipairs(reportSet) do
		self:DoPrint(channel, self:EntryToString(entry, name, lastTimeStamp))
	end

end


--[[ ---------------------------------------------------------------------------
	 Return the set of report data for output based on current filter settings
----------------------------------------------------------------------------- ]]
function Acheron:FilterReport(id, reportNum, seconds)

	local reportSet = {}

	local report = self.combatLogs[id][reportNum]
	local lastTimeStamp = report.log[report.last].timeStamp
	
	local threshold = self:GetProfileParam("reportthreshold")
	local showDmg = self:GetProfileParam("showdamage")
	local showHeal = self:GetProfileParam("showhealing")
	local showBuff = self:GetProfileParam("showbuff")
	local showDebuff = self:GetProfileParam("showdebuff")

	for i = report.first, report.last do
	
		local entry = report.log[i]
		
		if ((entry.timeStamp >= (lastTimeStamp - seconds)) and
			((entry.eventType == "HEAL" and showHeal and entry.amount >= threshold) or
			 (entry.eventType == "DAMAGE" and showDmg and entry.amount <= -threshold) or
			 (entry.eventType == "BUFF" and showBuff) or
			 (entry.eventType == "DEBUFF" and showDebuff) or
			 (entry.eventType == "DEATH")))
		then
			tinsert(reportSet, entry)
		end
		
	end
	
	return reportSet

end


--[[ ---------------------------------------------------------------------------
	 Formats a death report entry into a displayable string
----------------------------------------------------------------------------- ]]
function Acheron:EntryToString(entry, name, lastTimeStamp)

	local showHealth = self:GetProfileParam("abshealth")

	-- timestamp
	local formatStr = "%7.3f "
	local printArgs = {entry.timeStamp - lastTimeStamp}
	
	-- if not the end, include the current approx health
	if entry.eventType ~= "DEATH" then
		formatStr = formatStr.." (%s) "
		if showHealth then
			tinsert(printArgs, tostring(entry.curHealth))
		else
			tinsert(printArgs, entry.pctHealth)
		end
	end
		
	-- either add the special action or add the health damage/heal
	if ((entry.eventType == "HEAL" or entry.eventType == "DAMAGE") and (not entry.action)) then
		local red = 255
		local green = 255
	
		if entry.eventType == "HEAL" then
			red = 0
		else
			green = 0
		end
	
		formatStr = formatStr.."|cff%02X%02X00%+7d%s|r"
		tinsert(printArgs, red)
		tinsert(printArgs, green)
		tinsert(printArgs, entry.amount)
		tinsert(printArgs, (entry.isCrit and "!") or (entry.isCrush and "*") or " ")
	elseif entry.eventType == "DAMAGE" and entry.action then
		formatStr = formatStr.."|cffaaaaaa%s|r"
		tinsert(printArgs, entry.action)
	elseif entry.action then
		formatStr = formatStr.."%s"
		tinsert(printArgs, entry.action)
	end
			
	-- if there's a source for the action, add it
	if entry.source then
		local classColor = nil
		if entry.sourceClass then
			classColor = RAID_CLASS_COLORS[entry.sourceClass]
		end
		
		if classColor then
			formatStr = formatStr.." [|cff%02x%02x%02x%s|r"
			tinsert(printArgs, classColor.r*255)
			tinsert(printArgs, classColor.g*255)
			tinsert(printArgs, classColor.b*255)
			tinsert(printArgs, entry.source)
		else
			formatStr = formatStr.." [%s"
			tinsert(printArgs, entry.source)
		end
	end
			
	-- if there's a spell for the action, add it
	if entry.spell then
		formatStr = formatStr.." - %s]"
		tinsert(printArgs, entry.spell)
	elseif entry.source then
		formatStr = formatStr.."]"
	end

	return format(formatStr, unpack(printArgs))

end


--[[ ---------------------------------------------------------------------------
	 Determine where to do text output
----------------------------------------------------------------------------- ]]
function Acheron:DoPrint(loc, str)

    if loc then
    
    	local target = nil
    	
    	if loc == L["whisper"] then
    		target = self.whisperTarget or UnitName("target")
    		loc = "whisper"
    	elseif loc == L["say"] then
    		loc = "say"
    	elseif loc == L["party"] then
    		loc = "party"
    	elseif loc == L["raid"] then
    		loc = "raid"
    	elseif loc == L["guild"] then
    		loc = "guild"    		
    	elseif loc == L["officer"] then
    		loc = "officer"
    	elseif tonumber(loc) then
    		target = tonumber(loc)
    	    loc = "channel"
    	end    	
    	
    	-- strip color tags out
    	str = gsub(str, "|c%x%x%x%x%x%x%x%x(.-)|r", "%1");
    	
    	SendChatMessage(str, loc, nil, target)
    	
    else
    
    	DEFAULT_CHAT_FRAME:AddMessage(str)
    	
    end

end
