--[[ ---------------------------------------------------------------------------

Acheron: death reports

Acheron is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

Acheron is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with Acheron.	If not, see <http://www.gnu.org/licenses/>.

----------------------------------------------------------------------------- ]]

--[[ ---------------------------------------------------------------------------
	 Addon and libraries
----------------------------------------------------------------------------- ]]
Acheron = nil

if LibStub("LibLogger-1.0", true) then
    Acheron = LibStub("AceAddon-3.0"):NewAddon("Acheron", "AceEvent-3.0", "AceBucket-3.0", "AceTimer-3.0", "AceConsole-3.0", "AceHook-3.0", "LibLogger-1.0")
else
	Acheron = LibStub("AceAddon-3.0"):NewAddon("Acheron", "AceEvent-3.0", "AceBucket-3.0", "AceTimer-3.0", "AceConsole-3.0", "AceHook-3.0")
end

local L = LibStub("AceLocale-3.0"):GetLocale("Acheron")
local GUI = LibStub("AceGUI-3.0")
local QH = LibStub("LibQuickHealth-1.0")
local LDB = LibStub:GetLibrary("LibDataBroker-1.1")

local menuTypes= {"SELF", "PARTY", "RAID_PLAYER"}


--[[ ---------------------------------------------------------------------------
	 Ace3 initialization
----------------------------------------------------------------------------- ]]
function Acheron:OnInitialize()

	-- Initialize persistent addon variables
	self.combatLogs = {}
	self.updateUnitsBucket = nil
	self.combatLogSettings = nil
	
	self.currentAuraList = nil
	
	self.frame = nil
	self.frameReports = nil
	self.channels = nil
	self.availableReports = nil
	self.whisperEditBox = nil
	self.whisperTarget = nil
	
	-- Set up saved variables and config options
	self.db = LibStub("AceDB-3.0"):New("AcheronDB", self.defaults, "Default")

	self.db.RegisterCallback(self, "OnProfileChanged", "OnProfileChanged")
	self.db.RegisterCallback(self, "OnProfileCopied", "OnProfileChanged")
	self.db.RegisterCallback(self, "OnProfileDeleted","OnProfileChanged")
	self.db.RegisterCallback(self, "OnProfileReset", "OnProfileChanged")
	
	self.options.args.profile = LibStub("AceDBOptions-3.0"):GetOptionsTable(self.db)
	LibStub("AceConfig-3.0"):RegisterOptionsTable("Acheron", self.options, {"acheron"})
	self.optionsFrame = LibStub("AceConfigDialog-3.0"):AddToBlizOptions("Acheron", "Acheron")
	
	-- Logging
	if LibStub("LibLogger-1.0", true) then
		self:SetLogLevel(self.db.profile.logLevel)
		self:SetPerformanceMode(true)
	end
	
	-- Set up the data broker object
	self.dobj = LDB:NewDataObject("Acheron", {
		type = "launcher",
		icon = "Interface\\Icons\\Ability_Creature_Cursed_03",
    	OnClick = function(frame, button)
    		Acheron:ShowDeathReports()
        end
	})
	
	-- Create the display frame
	self:CreateFrame()
	self:DisplayWhiteList()
	self:DisplayBlackList()
	
end


--[[ ---------------------------------------------------------------------------
	 Ace3 enable addon
----------------------------------------------------------------------------- ]]
function Acheron:OnEnable()

		self:DoEnable()

end


--[[ ---------------------------------------------------------------------------
	 Enables addon 
----------------------------------------------------------------------------- ]]
function Acheron:DoEnable()

	if not self:GetProfileParam("enable") then return end
	if self.debug then self:debug("Addon enabled") end
	
	-- Register events
	self.updateUnitsBucket = self:RegisterBucketEvent({"PLAYER_ENTERING_WORLD", "PARTY_MEMBERS_CHANGED", "RAID_ROSTER_UPDATE"}, 1, "ScanRaidParty");
	self:RegisterEvent("COMBAT_LOG_EVENT_UNFILTERED")
	
	-- Register callbacks
	QH.RegisterCallback(self, "HealthUpdated", function(event, GUID, newHealth) end)
	
	-- Inject unit frame right-click menu option
	UnitPopupButtons["SHOW_DEATH_REPORTS"] = {
		text = L["Show Acheron Death Reports"],
		dist = 0,
		func = self.ShowDeathReports
	}
	
	for i = 1, #menuTypes do
		tinsert(UnitPopupMenus[menuTypes[i]], #UnitPopupMenus[menuTypes[i]]-1, "SHOW_DEATH_REPORTS")
	end

	self:SecureHook("UnitPopup_ShowMenu")

end


--[[ ---------------------------------------------------------------------------
	 Ace3 disable addon
----------------------------------------------------------------------------- ]]
function Acheron:OnDisable()

	self:DoDisable()

end


--[[ ---------------------------------------------------------------------------
	 Disables addon
----------------------------------------------------------------------------- ]]
function Acheron:DoDisable()

	if self.debug then self:debug("Addon disabled") end

	-- Unregister events
	self:UnregisterAllEvents()
	self:UnregisterBucket(self.updateUnitsBucket)

	-- Unregister callbacks
	QH.UnregisterCallback(self, "HealthUpdated")
	
	-- Remove unit frame right-click menu option
	for j = 1, #menuTypes do
		local t = menuTypes[j]
		for i = 1, #UnitPopupMenus[t] do
			if UnitPopupMenus[t][i] == "SHOW_DEATH_REPORTS" then
				tremove(UnitPopupMenus[t], i)
				break
			end
		end
	end

	self:UnhookAll()
	
end


--[[ ---------------------------------------------------------------------------
	 Hook function for injecting unit frame right-click menu option
----------------------------------------------------------------------------- ]]
function Acheron:UnitPopup_ShowMenu(dropdownMenu, which, unit, name, userData, ...)
	for i=1, UIDROPDOWNMENU_MAXBUTTONS do
		button = _G["DropDownList"..UIDROPDOWNMENU_MENU_LEVEL.."Button"..i];
		if button.value == "SHOW_DEATH_REPORTS" then
		    button.func = UnitPopupButtons["SHOW_DEATH_REPORTS"].func
		end
	end
end


--[[ ---------------------------------------------------------------------------
	 Iterate through party/raid to see what we need to track
----------------------------------------------------------------------------- ]]
function Acheron:ScanRaidParty()

	if self.debug then self:debug("Scanning raid/party") end

	local newCombatLogs = {}
	
	local groupType = "party"
	local groupSize = GetNumPartyMembers()

	if GetNumRaidMembers() > 0 then
		groupType = "raid"
		groupSize = GetNumRaidMembers()
	end
	
	local i = 0
	local unit = "player"
		
	while i <= groupSize do
	
		if UnitExists(unit) then
			
			local id = UnitGUID(unit)
			
			if self.combatLogs[id] then
				newCombatLogs[id] = self.combatLogs[id]
			else
				newCombatLogs[id] = {name = UnitName(unit)}
				tinsert(newCombatLogs[id], {first = 1, last = 0, auras = {}, log = {}})
			end
			
		end
	
		i = i + 1
		unit = groupType..i
	
	end
	
	self.combatLogs = newCombatLogs
	
end


--[[ ---------------------------------------------------------------------------
	 Handle combat log events
----------------------------------------------------------------------------- ]]
function Acheron:COMBAT_LOG_EVENT_UNFILTERED(event, timeStamp, eventType, srcGUID, srcName, srcFlags, dstGUID, dstName, dstFlags, ...)
  
	-- Stop if this is a combat log event we don't care about
	if not dstGUID then return end						  
	if not self.combatLogs[dstGUID] then return end
	
	if eventType == "UNIT_DIED" then
		self:HandleDeathEvent(timeStamp, dstGUID)
	elseif eventType == "SPELL_AURA_APPLIED" or
		   eventType == "SPELL_AURA_REMOVED" or
		   eventType == "SPELL_AURA_APPLIED_DOSE" or
		   eventType == "SPELL_AURA_REMOVED_DOSE"
	then
		self:HandleBuffEvent(event, timeStamp, eventType, srcGUID, srcName, srcFlags, dstGUID, dstName, dstFlags, ...)
	else
		self:HandleHealthEvent(event, timeStamp, eventType, srcGUID, srcName, srcFlags, dstGUID, dstName, dstFlags, ...)
	end

end


--[[ ---------------------------------------------------------------------------
	 Handle death
----------------------------------------------------------------------------- ]]
function Acheron:HandleDeathEvent(timeStamp, dstGUID)

	-- If nothing happened before the death, forget it (Spirit of Redemption)
	local latestLog = self.combatLogs[dstGUID][#(self.combatLogs[dstGUID])]
	if latestLog.last == 0 then return end

	self:TrackEvent("DEATH", timeStamp, nil, dstGUID, 0, L["Death"])
	tinsert(self.combatLogs[dstGUID], {first = 1, last = 0, auras = {}, log = {}})
		
	-- Remove older logs if threshold is set
	if self:GetProfileParam("numreports") > 0 and (#(self.combatLogs[dstGUID]) - 1) > self:GetProfileParam("numreports") then
		tremove(self.combatLogs[dstGUID], 1)
	end

end


--[[ ---------------------------------------------------------------------------
	 Handle combat events that affect buffs (damage or healing)
----------------------------------------------------------------------------- ]]
function Acheron:HandleBuffEvent(event, timeStamp, eventType, srcGUID, srcName, srcFlags, dstGUID, dstName, dstFlags, ...)

	local spellId = select(1, ...)
	local spell = select(2, ...)
	local auraType = select(4,...)
	local debuffCount = select(5,...) or 1

	if not spell then
		return
	elseif ((self:GetProfileParam("enablewhitelist") and not self.db.profile.aurawhitelist[spell]) or
			(self:GetProfileParam("enableblacklist") and self.db.profile.aurablacklist[spell]))
	then
		return
	end
	
	local action = spell
	
	if self.trace then self:trace("%s: %s x %d", eventType, spell, debuffCount) end
								
	local latestLog = self.combatLogs[dstGUID][#(self.combatLogs[dstGUID])]
	
	if eventType == "SPELL_AURA_REMOVED" then
		latestLog.auras[spell] = nil
		action = "-" .. action
	else
		latestLog.auras[spell] = debuffCount
		action = "+" .. action
	end
	
	if debuffCount > 1 then
		action = action .. " ("..debuffCount..")"
	end

	local msg = CombatLog_OnEvent(Blizzard_CombatLog_CurrentSettings, timeStamp, eventType, srcGUID, srcName, srcFlags, dstGUID, dstName, dstFlags, ...)
	
	self:TrackEvent(auraType, timeStamp, msg, dstGUID, 0, action)
	
end

--[[ ---------------------------------------------------------------------------
	 Handle combat events that affect health (damage or healing)
----------------------------------------------------------------------------- ]]
function Acheron:HandleHealthEvent(event, timeStamp, eventType, srcGUID, srcName, srcFlags, dstGUID, dstName, dstFlags, ...)
 	
	-- Initialize values		  
	local amount = nil
	local action = nil
	local missType = nil
	local spell = select(2, ...)
	local source = srcName
	local isCrit = false
	local isCrush = false
	local trackType = "DAMAGE"

	-- Extract the useful data	
	local prefix, suffix, special = strsplit("_", eventType)
		
	if prefix == "SPELL" then
		if suffix == "HEAL" or special == "HEAL" then
			amount = select(4, ...)
			isCrit = select(5, ...)
		elseif suffix == "DAMAGE" or special == "DAMAGE" then
			amount = 0 - select(4, ...)
			isCrit = select(9, ...)
		elseif suffix == "MISSED" or special == "MISSED" then
			missType =  select(4, ...)
		end
	elseif suffix == "DAMAGE" then
		if prefix == "SWING" then
			amount = 0 - select(1, ...)
			spell = L["Melee"]
			isCrit = select(6, ...)
			isCrush = select(8, ...)
		elseif prefix == "RANGE" then
			amount = 0 - select(4, ...)
			isCrit = select(9, ...)
		elseif prefix == "ENVIRONMENTAL" then
			amount = 0 - select(2, ...)
			spell = select(1, ...)
			source = L["Environment"]
		end
	elseif prefix == "DAMAGE" and special ~= "MISSED" then
		amount = 0 - select(4, ...)
	elseif suffix == "MISSED" then
		if prefix == "RANGE" then
			missType =  select(4, ...)
		else
			missType =  select(1, ...)
		end
	end
	
	-- Convert miss type to string
	if missType then
		if missType == "DODGE" then amount, action = 0, L["Dodge"] end
		if missType == "PARRY" then amount, action = 0, L["Parry"] end
		if missType == "MISS" then amount, action = 0, L["Miss"] end
		if missType == "RESIST" then amount, action = 0, L["Resist"] end
	end
	
	if not amount and not action then return end
	if amount > 0 then trackType = "HEAL" end
	
	local msg = CombatLog_OnEvent(Blizzard_CombatLog_CurrentSettings, timeStamp, eventType, srcGUID, srcName, srcFlags, dstGUID, dstName, dstFlags, ...)
	
	-- Track the event
	self:TrackEvent(trackType, timeStamp, msg, dstGUID, amount, action, source, spell, isCrit, isCrush)
	
end


--[[ ---------------------------------------------------------------------------
	 Track a combat log event
----------------------------------------------------------------------------- ]]
function Acheron:TrackEvent(eventType, timeStamp, msg, dstGUID, amount, action, source, spell, isCrit, isCrush)

	local dstName = self.combatLogs[dstGUID].name

	if self.trace then self:trace("%s: %s %s %s %s %s %s", eventType, tostring(timeStamp), tostring(dstName),
								tostring(amount), tostring(action), tostring(source), tostring(spell)) end
	
	local latestLog = self.combatLogs[dstGUID][#(self.combatLogs[dstGUID])]								
								
	-- Compose the entry for this event
	local sourceClass = nil
	if source and UnitIsPlayer(source) then
		_, sourceClass = UnitClass(source)
	end
	
	if not amount then amount = 0 end
	if not source then source = spell end
	local curHealth = QH:UnitHealth(dstName)
	local maxHealth = UnitHealthMax(dstName)
	local pctHealth = ((curHealth/maxHealth) * 100)
	if pctHealth < 0 then pctHealth = "--%" else pctHealth = format("%3d%%", pctHealth) end
	
	local entry = {
		eventType = eventType,
		timeStamp = timeStamp,
		msg = msg,
		curHealth = curHealth,
		maxHealth = maxHealth,
		pctHealth = pctHealth,
		amount = amount,
		action = action,
		source = source,
		sourceClass = sourceClass,
		spell = spell,
		isCrit = isCrit,
		isCrush = isCrush,
	}
		
	-- Add the entry to the latest log
	latestLog.last = latestLog.last + 1
	latestLog.log[latestLog.last] = entry
	
	-- Remove any entries that are older than our threshold
	for i = latestLog.first, latestLog.last do
	
		if (latestLog.log[i].timeStamp >= (timeStamp - self:GetProfileParam("history"))) then
			latestLog.first = i;
			break
		else
			latestLog.log[i] = nil
		end
	
	end

end


--[[ ---------------------------------------------------------------------------
	 Clear all reports
----------------------------------------------------------------------------- ]]
function Acheron:ClearReports()

	self.combatLogs = {}
	self:ScanRaidParty()

end