-- ReaperHPBar.lua
-- All-in-one: main bar + options panel + plugin API + profiles + slash commands
local addonName, ns = ...

-- Saved variables (declared in TOC: SavedVariables: ReaperHP_Settings, ReaperHP_Profiles)
ReaperHP_Settings = ReaperHP_Settings or {}
ReaperHP_Profiles = ReaperHP_Profiles or {}

-- Minimal defaults
local defaults = {
    width = 200,
    height = 6,
    showNumbers = true,
    autoColor = true,
    classGradient = false,
    smoothing = 12,       -- higher = snappier
    glowAlpha = 0.25,
    showAbsorb = true,
    showIncomingHeal = true,
    locked = false,
    anchor = {"TOP", "PlayerFrame", "BOTTOM", 0, -5},
    barColor = {0,1,0},
}

-- Ensure defaults in settings
for k,v in pairs(defaults) do
    if ReaperHP_Settings[k] == nil then ReaperHP_Settings[k] = v end
end

local settings = ReaperHP_Settings
ns.settings = settings
ns.profiles = ReaperHP_Profiles

-- Plugin storage
ns.plugins = ns.plugins or {}
function ns.RegisterPlugin(name, tbl)
    if not name or not tbl then return end
    ns.plugins[name] = tbl
    if tbl.OnLoad and ns.Bar then pcall(tbl.OnLoad, tbl, ns.Bar, ns) end
end
function ns.UpdatePlugins(bar, elapsed)
    for n,p in pairs(ns.plugins) do
        if p.OnUpdate then
            p.OnUpdate(p, bar, ns, elapsed)
        end
    end
end

-- Utility: get class color fallback
local function GetPlayerClassColor()
    local _, class = UnitClass("player")
    if class and RAID_CLASS_COLORS and RAID_CLASS_COLORS[class] then
        local c = RAID_CLASS_COLORS[class]
        return c.r, c.g, c.b
    end
    return 0, 1, 0
end

-- Create main frame (keeps the exact API shape you provided)
local addon = CreateFrame("Frame", "ReaperHPBarFrame", UIParent)
addon:SetSize(settings.width, settings.height)
-- anchor from settings
if settings.anchor then
    pcall(function()
        addon:SetPoint(unpack(settings.anchor))
    end)
else
    addon:SetPoint(unpack(defaults.anchor))
end

-- Movable
addon:SetMovable(true)
addon:EnableMouse(true)
addon:RegisterForDrag("LeftButton")
addon:SetScript("OnDragStart", function(self)
    if not settings.locked then self:StartMoving() end
end)
addon:SetScript("OnDragStop", function(self)
    self:StopMovingOrSizing()
    -- save anchor
    local point, relativeTo, relPoint, x, y = self:GetPoint()
    -- store simplified anchor: point, relative name (PlayerFrame), relPoint, x, y
    ReaperHP_Settings.anchor = { point or "TOP", "PlayerFrame", relPoint or "BOTTOM", x or 0, y or -5 }
end)

-- Background
local barBg = addon:CreateTexture(nil, "BACKGROUND")
barBg:SetAllPoints()
barBg:SetColorTexture(0,0,0,0.5)

-- Foreground HP bar (StatusBar like behavior but texture-based to match original)
local hpBar = addon:CreateTexture(nil, "ARTWORK")
hpBar:SetPoint("LEFT", addon, "LEFT", 0, 0)
hpBar:SetHeight(settings.height)
hpBar:SetColorTexture(unpack(settings.barColor))
hpBar:SetWidth(settings.width) -- current visual width; we will animate it

-- Absorb overlay (on top of hpBar)
local absorbTex = addon:CreateTexture(nil, "ARTWORK")
absorbTex:SetPoint("LEFT", hpBar, "RIGHT", 0, 0)
absorbTex:SetHeight(settings.height)
absorbTex:SetColorTexture(0.6,0.6,1,0.8) -- light blue for absorbs
absorbTex:SetWidth(0)

-- Incoming heal overlay (on top of hpBar, behind absorb)
local healTex = addon:CreateTexture(nil, "ARTWORK")
healTex:SetPoint("LEFT", hpBar, "RIGHT", 0, 0)
healTex:SetHeight(settings.height)
healTex:SetColorTexture(0,1,0,0.45)
healTex:SetWidth(0)

-- Glow behind
local glow = addon:CreateTexture(nil, "BACKGROUND")
glow:SetPoint("LEFT", addon, "LEFT", -6, 0)
glow:SetHeight(settings.height + 12)
glow:SetWidth(settings.width + 12)
glow:SetColorTexture(1,1,1, settings.glowAlpha)
glow:SetBlendMode("ADD")

-- HP Text
local hpText = addon:CreateFontString(nil, "OVERLAY", "GameFontHighlightSmall")
hpText:SetPoint("CENTER", addon, "CENTER", 0, 0)

-- Expose pieces for plugins/options
addon.hpBar = hpBar
addon.barBg = barBg
addon.hpText = hpText
addon.absorb = absorbTex
addon.heal = healTex
addon.glow = glow

ns.Bar = addon

-- Internal smoothing state
local currentWidth = settings.width
local targetWidth = settings.width
local targetR, targetG, targetB = unpack(settings.barColor)
local curR, curG, curB = targetR, targetG, targetB

-- Helper: clamp
local function clamp(v, a, b) if v < a then return a elseif v > b then return b else return v end end

-- Helper: tries to use available APIs for absorbs/incoming heals, fallback to zero
local function GetAbsorbAndIncoming(unit)
    local absorb = 0
    local incoming = 0
    -- Try UnitGetTotalAbsorbs (may not exist on older clients). Use pcall to be safe.
    if UnitIsUnit and UnitIsUnit(unit, "player") then
        -- attempt different calls
        if (type(UnitGetTotalAbsorbs) == "function") then
            pcall(function() absorb = UnitGetTotalAbsorbs(unit) or 0 end)
        end
        if (type(UnitGetIncomingHeals) == "function") then
            pcall(function() incoming = UnitGetIncomingHeals(unit) or 0 end)
        end
        -- older clients sometimes don't have those functions; attempt to use UnitGetTotalAbsorbs via secure path fails silently
    end
    if not absorb then absorb = 0 end
    if not incoming then incoming = 0 end
    return absorb, incoming
end

-- Color helper for auto coloring (smooth transition: green > yellow > red)
local function ColorForPercent(p)
    if p >= 0.6 then
        -- interpolates from yellow (1,1,0) -> green (0,1,0) as p goes 0.6..1
        local t = (p - 0.6) / 0.4
        return (1 - t) * 1, 1, 0
    elseif p >= 0.3 then
        -- 0.3..0.6 : red(1,0,0) -> yellow(1,1,0)
        local t = (p - 0.3) / 0.3
        return 1, t, 0
    else
        return 1, 0, 0
    end
end

-- Main update function (calculates target widths/colors)
local function Recalculate()
    local hp = UnitHealth("player") or 0
    local maxhp = UnitHealthMax("player") or 1
    local pct = maxhp > 0 and (hp / maxhp) or 0

    -- main target width in pixels
    targetWidth = clamp(pct * settings.width, 0, settings.width)

    -- class gradient override
    if settings.classGradient then
        targetR, targetG, targetB = GetPlayerClassColor()
    else
        if settings.autoColor then
            targetR, targetG, targetB = ColorForPercent(pct)
        else
            targetR, targetG, targetB = unpack(settings.barColor)
        end
    end

    -- absorbs and incoming heals
    local absorb, incoming = 0, 0
    pcall(function()
        absorb, incoming = GetAbsorbAndIncoming("player")
        if not absorb then absorb = 0 end
        if not incoming then incoming = 0 end
    end)

    -- Bound the overlays: they cannot exceed remaining space to full width
    local missing = math.max(0, settings.width - (pct * settings.width))
    local healPixels = clamp((incoming / maxhp) * settings.width, 0, missing)
    local absorbPixels = clamp((absorb / maxhp) * settings.width, 0, missing - healPixels)

    -- store desired overlay widths
    addon._desired = addon._desired or {}
    addon._desired.hp = targetWidth
    addon._desired.heal = healPixels
    addon._desired.absorb = absorbPixels
    addon._desired.r = targetR
    addon._desired.g = targetG
    addon._desired.b = targetB
end

-- OnUpdate loop: do smooth animation and color blending
addon:SetScript("OnUpdate", function(self, elapsed)
    -- ensure recalc occasionally (every frame is fine)
    Recalculate()

    local speed = math.max(1, settings.smoothing)
    -- lerp width
    currentWidth = currentWidth + (addon._desired.hp - currentWidth) * math.min(elapsed * speed, 1)
    hpBar:SetWidth(currentWidth)

    -- heal overlay: position at right of current fill
    local healW = (addon._desired.heal or 0)
    local absorbW = (addon._desired.absorb or 0)
    healTex:SetPoint("LEFT", hpBar, "RIGHT", 0, 0)
    healTex:SetWidth(healW)
    absorbTex:SetPoint("LEFT", hpBar, "RIGHT", healW, 0)
    absorbTex:SetWidth(absorbW)

    -- color lerp
    curR = curR + ((addon._desired.r or curR) - curR) * math.min(elapsed * speed * 0.6, 1)
    curG = curG + ((addon._desired.g or curG) - curG) * math.min(elapsed * speed * 0.6, 1)
    curB = curB + ((addon._desired.b or curB) - curB) * math.min(elapsed * speed * 0.6, 1)
    hpBar:SetColorTexture(curR, curG, curB, 1)

    -- glow follow
    glow:SetWidth(currentWidth + 12)
    glow:SetHeight(settings.height + 12)
    glow:SetColorTexture(curR, curG, curB, settings.glowAlpha)

    -- text update (raw cur/max)
    local cur = UnitHealth("player") or 0
    local max = UnitHealthMax("player") or 0
    if settings.showNumbers then
        hpText:SetText(string.format("%d / %d", cur, max))
        hpText:Show()
    else
        hpText:Hide()
    end

    -- plugins hook
    ns.UpdatePlugins(addon, elapsed)
end)

-- Events to trigger recalc more often (also saves)
addon:RegisterEvent("PLAYER_ENTERING_WORLD")
addon:RegisterEvent("UNIT_HEALTH")
addon:RegisterEvent("UNIT_MAXHEALTH")
addon:RegisterEvent("PLAYER_REGEN_ENABLED")
addon:RegisterEvent("PLAYER_REGEN_DISABLED")
addon:SetScript("OnEvent", function(self, event, arg1)
    if event == "UNIT_HEALTH" and arg1 ~= "player" then return end
    Recalculate()
end)

-- Public: apply settings to immediate UI
function ns.ApplySettings()
    addon:SetSize(settings.width, settings.height)
    hpBar:SetHeight(settings.height)
    hpBar:SetWidth(currentWidth)
    healTex:SetHeight(settings.height)
    absorbTex:SetHeight(settings.height)
    glow:SetHeight(settings.height + 12)
    glow:SetWidth(currentWidth + 12)
    hpText:SetFont("Fonts\\FRIZQT__.TTF", 10, "OUTLINE")
    -- color setting
    if not settings.autoColor and not settings.classGradient then
        hpBar:SetColorTexture(unpack(settings.barColor))
        glow:SetColorTexture(unpack(settings.barColor), settings.glowAlpha)
    end
end

-- Profiles: save/load simple snapshot (only keys in defaults)
function ns.SaveProfile(name)
    if not name or name == "" then return false, "invalid name" end
    ReaperHP_Profiles[name] = {}
    for k,_ in pairs(defaults) do ReaperHP_Profiles[name][k] = settings[k] end
    return true
end
function ns.LoadProfile(name)
    local p = ReaperHP_Profiles[name]
    if not p then return false, "no such profile" end
    for k,v in pairs(p) do settings[k] = v end
    ns.ApplySettings()
    return true
end

-- Simple Interface Options panel (safe)
local function CreateOptionsPanel()
    if _G["ReaperHPBarOptions"] then return end
    local panel = CreateFrame("Frame", "ReaperHPBarOptions", UIParent)
    panel.name = "Reaper HP Bar"
    InterfaceOptions_AddCategory(panel)

    local title = panel:CreateFontString(nil, "OVERLAY", "GameFontNormalLarge")
    title:SetPoint("TOPLEFT", 16, -16)
    title:SetText("Reaper HP Bar Options")

    local ypos = -40
    local function MakeSlider(label, key, min, max, step)
        local name = "RHP_Slider_"..key
        local s = CreateFrame("Slider", name, panel, "OptionsSliderTemplate")
        s:SetPoint("TOPLEFT", 16, ypos)
        s:SetMinMaxValues(min, max)
        s:SetValueStep(step)
        s:SetWidth(260)
        s:SetValue(settings[key])
        _G[name.."Low"]:SetText(tostring(min))
        _G[name.."High"]:SetText(tostring(max))
        _G[name.."Text"]:SetText(label)
        s:SetScript("OnValueChanged", function(self, val)
            if step >= 1 then val = math.floor(val) end
            settings[key] = val
            ns.ApplySettings()
        end)
        ypos = ypos - 60
        return s
    end

    MakeSlider("Bar Width", "width", 80, 600, 1)
    MakeSlider("Bar Height", "height", 4, 40, 1)
    MakeSlider("Smoothing (higher = faster)", "smoothing", 2, 30, 1)
    MakeSlider("Glow Alpha", "glowAlpha", 0, 1, 0.01)

    -- checkboxes
    local function MakeCheck(label, key)
        local c = CreateFrame("CheckButton", nil, panel, "InterfaceOptionsCheckButtonTemplate")
        c:SetPoint("TOPLEFT", 16, ypos)
        _G[c:GetName().."Text"]:SetText(label)
        c:SetChecked(settings[key])
        c:SetScript("OnClick", function(self)
            settings[key] = self:GetChecked()
            ns.ApplySettings()
        end)
        ypos = ypos - 28
        return c
    end

    MakeCheck("Show HP Numbers (current / max)", "showNumbers")
    MakeCheck("Auto color (green → yellow → red)", "autoColor")
    MakeCheck("Class-colored gradient", "classGradient")
    MakeCheck("Show Absorb Overlay (if available)", "showAbsorb")
    MakeCheck("Show Incoming Heal Overlay (if available)", "showIncomingHeal")

    -- profiles UI (small)
    local profLabel = panel:CreateFontString(nil, "OVERLAY", "GameFontNormal")
    profLabel:SetPoint("TOPLEFT", 320, -40)
    profLabel:SetText("Profiles")

    local profBox = CreateFrame("EditBox", "RHP_ProfName", panel, "InputBoxTemplate")
    profBox:SetSize(140, 22)
    profBox:SetPoint("TOPLEFT", profLabel, "BOTTOMLEFT", 0, -6)
    profBox:SetAutoFocus(false)
    profBox:SetText("MyProfile")

    local saveBtn = CreateFrame("Button", nil, panel, "UIPanelButtonTemplate")
    saveBtn:SetPoint("LEFT", profBox, "RIGHT", 8, 0)
    saveBtn:SetSize(80,22)
    saveBtn:SetText("Save")
    saveBtn:SetScript("OnClick", function()
        local nm = profBox:GetText()
        local ok,err = ns.SaveProfile(nm)
        if ok then print("ReaperHPBar: saved profile: "..nm) else print("ReaperHPBar: save failed: "..tostring(err)) end
    end)

    local loadBtn = CreateFrame("Button", nil, panel, "UIPanelButtonTemplate")
    loadBtn:SetPoint("TOPLEFT", profBox, "BOTTOMLEFT", 0, -6)
    loadBtn:SetSize(80,22)
    loadBtn:SetText("Load")
    loadBtn:SetScript("OnClick", function()
        local nm = profBox:GetText()
        local ok,err = ns.LoadProfile(nm)
        if ok then print("ReaperHPBar: loaded profile: "..nm) else print("ReaperHPBar: load failed: "..tostring(err)) end
    end)
end

-- Create options panel (deferred to ensure InterfaceOptions code exists)
local loader = CreateFrame("Frame")
loader:RegisterEvent("PLAYER_LOGIN")
loader:SetScript("OnEvent", function()
    CreateOptionsPanel()
    ns.ApplySettings()
end)

-- Slash commands
SLASH_REAPERHP1 = "/rhp"
SlashCmdList["REAPERHP"] = function(msg)
    local cmd, rest = msg:match("^(%S*)%s*(.-)$")
    cmd = cmd:lower()
    if cmd == "lock" then
        settings.locked = true
        print("ReaperHPBar: locked.")
    elseif cmd == "unlock" or cmd == "move" then
        settings.locked = false
        print("ReaperHPBar: unlocked (drag to move).")
    elseif cmd == "save" and rest ~= "" then
        local ok = ns.SaveProfile(rest)
        print(ok and ("Saved profile "..rest) or ("Failed to save "..rest))
    elseif cmd == "load" and rest ~= "" then
        local ok,err = ns.LoadProfile(rest)
        print(ok and ("Loaded profile "..rest) or ("Failed to load "..rest.." : "..tostring(err)))
    elseif cmd == "list" then
        print("ReaperHPBar profiles:")
        for k,v in pairs(ReaperHP_Profiles) do print(" - "..k) end
    elseif cmd == "reset" then
        for k,v in pairs(defaults) do settings[k] = v end
        ns.ApplySettings()
        print("ReaperHPBar: reset to defaults.")
    elseif cmd == "options" or cmd == "" then
        -- open Interface Options to this panel
        InterfaceOptionsFrame_OpenToCategory("Reaper HP Bar")
    else
        print("ReaperHPBar commands: lock, unlock, move, save <name>, load <name>, list, reset, options")
    end
end

-- Ensure apply settings at load
ns.ApplySettings()
