-- filterboost.lua
-- Accelerates TSM2 “Preparing Filters…” by caching and short-circuiting filters.

local PB = _G.TSMPerfBoost or {}
PB.filter = PB.filter or {}

-------------------------------------------
-- 1. VERY FAST itemString → itemID
-------------------------------------------
local function QuickItemID(link)
    if not link then return nil end
    -- Extract: |cffXXXXXX|Hitem:12345:...
    local itemID = link:match("item:(%d+)")
    return tonumber(itemID)
end

-------------------------------------------
-- 2. Local iteminfo cache (LRU style)
-------------------------------------------
local itemCache = {}
function PB.filter.GetItemInfoFast(itemID)
    local c = itemCache[itemID]
    if c then return unpack(c) end

    local name, link, quality, ilvl, req, class, subclass, maxStack, equipSlot, texture, vendorPrice =
        GetItemInfo(itemID)

    itemCache[itemID] = { name, link, quality, ilvl, req, class, subclass, maxStack, equipSlot, texture, vendorPrice }
    return name, link, quality, ilvl, req, class, subclass, maxStack, equipSlot, texture, vendorPrice
end

-------------------------------------------
-- 3. Memoized filter validator
-------------------------------------------
local filterCache = {}
function PB.filter.ValidateFilter(filter)
    if filterCache[filter] ~= nil then
        return filterCache[filter]
    end

    -- Call the real validator
    local ok = true
    local validator = TSMAPI and TSMAPI.Auction and TSMAPI.Auction.ValidateFilter
    if validator then
        ok = validator(filter)
    end

    filterCache[filter] = ok
    return ok
end

-------------------------------------------
-- 4. Override TSM’s filter builder with fast version
-------------------------------------------
local function FastBuildFilterList(scanObj)
    local list = scanObj.filterList
    if not list then return end

    -- If filterList unchanged → skip rebuild entirely
    if PB.filter._lastFilter == list then
        scanObj.preparedFilters = PB.filter._cachedBuiltFilters
        return
    end

    PB.filter._lastFilter = list

    -- Build quickly
    local out = {}
    for _, filter in ipairs(list) do
        if PB.filter.ValidateFilter(filter) then
            out[#out+1] = filter
        end
    end

    PB.filter._cachedBuiltFilters = out
    scanObj.preparedFilters = out
end

-------------------------------------------
-- 5. Hook into TSM scan
-------------------------------------------
local function HookTSMFilters()
    local tsm = _G.TSM or _G.TradeSkillMaster
    if not tsm or not tsm.Scan then return end

    if not PB.filter._hooked then
        hooksecurefunc(tsm.Scan, "StartScan", function(scanObj)
            FastBuildFilterList(scanObj)
        end)
        PB.filter._hooked = true
        print("|cFF50C0FFTSMPerfBoost|r: Filter accelerator active")
    end
end

-- On init
HookTSMFilters()
